<?php
session_start();

// Verificación de sesión y roles
if (empty($_SESSION['username']) || ($_SESSION['rol'] !== 'administrador' && $_SESSION['rol'] !== 'coordinador1' && $_SESSION['rol'] !== 'coordinador2')) {
    header("Location: ../login.php");
    exit();
}

// Configuración para UTF-8
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');

include '../includes/db.php';

// Configurar conexión para usar UTF-8
$conn->set_charset("utf8mb4");

// Obtener parámetros de filtro
$filtros = [
    'postgrado_id' => $_GET['postgrado_id'] ?? '',
    'escolaridad_id' => $_GET['escolaridad_id'] ?? '',
    'cuatrimestre' => $_GET['cuatrimestre'] ?? '',
    'estado_nota' => $_GET['estado_nota'] ?? 'todas'
];

// Construir consulta rápida (solo primeros 20 registros)
$where_conditions = [];
$params = [];
$types = '';

if (!empty($filtros['postgrado_id'])) {
    $where_conditions[] = "p.id = ?";
    $params[] = $filtros['postgrado_id'];
    $types .= 'i';
}

if (!empty($filtros['escolaridad_id'])) {
    $where_conditions[] = "i.escolaridad_id = ?";
    $params[] = $filtros['escolaridad_id'];
    $types .= 'i';
}

if (!empty($filtros['cuatrimestre'])) {
    $where_conditions[] = "al.nombre LIKE ?";
    $params[] = '%-' . $filtros['cuatrimestre'];
    $types .= 's';
}

if ($filtros['estado_nota'] === 'pendientes') {
    $where_conditions[] = "i.nota_definitiva IS NULL";
} elseif ($filtros['estado_nota'] === 'cargadas') {
    $where_conditions[] = "i.nota_definitiva IS NOT NULL";
}

$where_clause = '';
if (!empty($where_conditions)) {
    $where_clause = "WHERE " . implode(' AND ', $where_conditions);
}

$sql = "SELECT 
            e.nombres, e.apellidos, e.cedula, p.nombre as postgrado_nombre,
            d.codigo, d.nombre as asignatura_nombre, d.credito,
            i.nota_definitiva, i.fecha_modificacion, al.nombre as año_lectivo,
            esc.nombre as escolaridad, u.username as modificado_por
        FROM inscripciones i
        JOIN estudiantes e ON i.estudiante_id = e.id
        JOIN denominaciones d ON i.denominacion_id = d.id
        JOIN postgrados p ON d.postgrado_id = p.id
        JOIN años_lectivos al ON i.año_lectivo_id = al.id
        JOIN escolaridades esc ON i.escolaridad_id = esc.id
        LEFT JOIN usuarios u ON i.usuario_modificacion = u.id
        $where_clause
        ORDER BY e.apellidos, e.nombres
        LIMIT 20";

$resultados = [];
if (!empty($params) || $where_clause === '') {
    $stmt = $conn->prepare($sql);
    if ($stmt) {
        if (!empty($params)) {
            $stmt->bind_param($types, ...$params);
        }
        $stmt->execute();
        $result = $stmt->get_result();
        $resultados = $result->fetch_all(MYSQLI_ASSOC);
        $stmt->close();
    }
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Vista Previa Rápida - Reporte de Notas</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        body { background-color: #f8f9fa; padding: 20px; }
        .excel-preview { background-color: white; border: 2px solid #28a745; }
        .excel-header { background-color: #28a745; color: white; }
        .excel-row:nth-child(even) { background-color: #f8f9fa; }
        .excel-row:hover { background-color: #e9ecef; }
        .preview-info { background-color: #d1ecf1; border-left: 4px solid #17a2b8; }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="card border-success">
            <div class="card-header bg-success text-white d-flex justify-content-between align-items-center">
                <h5 class="mb-0">
                    <i class="bi bi-eye"></i> Vista Previa Rápida - Formato Excel
                </h5>
                <span class="badge bg-light text-dark">Mostrando primeros 20 registros</span>
            </div>
            <div class="card-body p-0">
                <!-- Información de Filtros -->
                <div class="p-3 preview-info">
                    <div class="row">
                        <div class="col-md-12">
                            <strong>Filtros aplicados:</strong>
                            <?php
                            $filtros_texto = [];
                            if (!empty($filtros['postgrado_id'])) {
                                $sql_postgrado = "SELECT nombre FROM postgrados WHERE id = ?";
                                $stmt = $conn->prepare($sql_postgrado);
                                $stmt->bind_param("i", $filtros['postgrado_id']);
                                $stmt->execute();
                                $postgrado = $stmt->get_result()->fetch_assoc();
                                $filtros_texto[] = 'Postgrado: ' . ($postgrado['nombre'] ?? 'N/A');
                                $stmt->close();
                            }
                            
                            if (!empty($filtros['escolaridad_id'])) {
                                $sql_escolaridad = "SELECT nombre FROM escolaridades WHERE id = ?";
                                $stmt = $conn->prepare($sql_escolaridad);
                                $stmt->bind_param("i", $filtros['escolaridad_id']);
                                $stmt->execute();
                                $escolaridad = $stmt->get_result()->fetch_assoc();
                                $filtros_texto[] = 'Escolaridad: ' . ($escolaridad['nombre'] ?? 'N/A');
                                $stmt->close();
                            }
                            
                            if (!empty($filtros['cuatrimestre'])) {
                                $filtros_texto[] = 'Cuatrimestre: ' . $filtros['cuatrimestre'];
                            }
                            
                            if (!empty($filtros['estado_nota'])) {
                                $estado_texto = [
                                    'todas' => 'Todas las notas',
                                    'pendientes' => 'Solo notas pendientes',
                                    'cargadas' => 'Solo notas cargadas'
                                ];
                                $filtros_texto[] = 'Estado: ' . $estado_texto[$filtros['estado_nota']];
                            }
                            
                            echo implode(' | ', $filtros_texto);
                            ?>
                        </div>
                    </div>
                </div>
                
                <!-- Tabla de Previsualización -->
                <div class="table-responsive">
                    <table class="table table-bordered mb-0 excel-preview">
                        <thead>
                            <tr class="excel-header">
                                <th>Estudiante</th>
                                <th>Cédula</th>
                                <th>Postgrado</th>
                                <th>Asignatura</th>
                                <th>Crédito</th>
                                <th>Nota</th>
                                <th>Año Lectivo</th>
                                <th>Escolaridad</th>
                                <th>Fecha Modificación</th>
                                <th>Modificado Por</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($resultados as $row): ?>
                                <tr class="excel-row">
                                    <td><?= htmlspecialchars($row['apellidos'] . ', ' . $row['nombres']) ?></td>
                                    <td><?= htmlspecialchars($row['cedula']) ?></td>
                                    <td><?= htmlspecialchars($row['postgrado_nombre']) ?></td>
                                    <td><?= htmlspecialchars($row['codigo'] . ' - ' . $row['asignatura_nombre']) ?></td>
                                    <td class="text-center"><?= htmlspecialchars($row['credito']) ?></td>
                                    <td class="text-center">
                                        <?php if ($row['nota_definitiva']): ?>
                                            <span class="badge bg-success"><?= htmlspecialchars($row['nota_definitiva']) ?></span>
                                        <?php else: ?>
                                            <span class="badge bg-warning">PENDIENTE</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <span class="badge bg-info"><?= htmlspecialchars($row['año_lectivo']) ?></span>
                                    </td>
                                    <td><?= htmlspecialchars($row['escolaridad']) ?></td>
                                    <td>
                                        <small class="text-muted">
                                            <?= $row['fecha_modificacion'] ? date('d/m/Y H:i', strtotime($row['fecha_modificacion'])) : 'N/A' ?>
                                        </small>
                                    </td>
                                    <td><?= htmlspecialchars($row['modificado_por'] ?? 'N/A') ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            <div class="card-footer text-center">
                <a href="reportes_cuatrimestral.php?<?= http_build_query($filtros) ?>&export=excel" 
                   class="btn btn-success">
                    <i class="bi bi-download"></i> Descargar Excel Completo
                </a>
                <a href="reportes_cuatrimestral.php?<?= http_build_query($filtros) ?>" 
                   class="btn btn-primary">
                    <i class="bi bi-arrow-left"></i> Volver al Reporte Principal
                </a>
                <button onclick="window.print()" class="btn btn-secondary">
                    <i class="bi bi-printer"></i> Imprimir Vista Previa
                </button>
            </div>
        </div>
        
        <div class="mt-3 text-center text-muted">
            <small>
                <i class="bi bi-info-circle"></i> 
                Esta es una vista previa limitada a 20 registros. 
                Use el botón "Descargar Excel Completo" para obtener todos los resultados.
            </small>
        </div>
    </div>
</body>
</html>

<?php $conn->close(); ?>