<?php
session_start();

// Configurar encoding UTF-8
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');

if (!isset($_SESSION['username'])) {
    header("Location: ../login.php");
    exit();
}

include '../includes/db.php';

// Configurar conexión para usar UTF-8
$conn->set_charset("utf8mb4");
$conn->query("SET NAMES 'utf8mb4'");
$conn->query("SET CHARACTER SET utf8mb4");
$conn->query("SET COLLATION_CONNECTION = 'utf8mb4_unicode_ci'");

if ($conn->connect_error) {
    die("Error de conexión: " . $conn->connect_error);
}

// Habilitar reportes de error para depuración
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

// Obtener todos los postgrados
$sql_postgrados = "SELECT p.id, p.nombre 
                  FROM postgrados p
                  ORDER BY p.nombre";
$result_postgrados = $conn->query($sql_postgrados);
$postgrados = $result_postgrados->fetch_all(MYSQLI_ASSOC);

// Obtener años lectivos disponibles
$sql_anios = "SELECT DISTINCT nombre FROM anios_lectivos ORDER BY nombre DESC";
$result_anios = $conn->query($sql_anios);
$anios_lectivos = $result_anios->fetch_all(MYSQLI_ASSOC);

// Obtener escolaridades disponibles
$sql_escolaridades = "SELECT DISTINCT nombre FROM escolaridades ORDER BY nombre";
$result_escolaridades = $conn->query($sql_escolaridades);
$escolaridades = $result_escolaridades->fetch_all(MYSQLI_ASSOC);

// Definir cuatrimestres
$cuatrimestres = [
    '01C' => '1º Cuatrimestre',
    '02C' => '2º Cuatrimestre', 
    '03C' => '3º Cuatrimestre'
];

// Procesar filtros
$filtros = [
    'postgrado_id' => isset($_GET['postgrado_id']) ? intval($_GET['postgrado_id']) : '',
    'anio_lectivo' => isset($_GET['anio_lectivo']) ? $conn->real_escape_string($_GET['anio_lectivo']) : '',
    'escolaridad' => isset($_GET['escolaridad']) ? $conn->real_escape_string($_GET['escolaridad']) : '',
    'cuatrimestre' => isset($_GET['cuatrimestre']) ? $conn->real_escape_string($_GET['cuatrimestre']) : ''
];

// Construir consulta para el reporte
$sql_report = "SELECT 
                e.id as estudiante_id,
                e.nombres,
                e.apellidos,
                e.cedula_venezolana,
                e.cedula_extranjera,
                e.pasaporte,
                e.tipo_documento,
                p.nombre as postgrado_nombre,
                d.codigo,
                d.nombre as denominacion_nombre,
                d.periodo,
                d.credito,
                d.tipo,
                i.nota_definitiva,
                i.estado as estado_inscripcion,
                al.nombre as anio_lectivo_nombre,
                esc.nombre as escolaridad_nombre,
                i.fecha_modificacion,
                u_mod.nombres as modificado_por_nombres,
                u_mod.apellidos as modificado_por_apellidos
               FROM estudiantes e
               JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
               JOIN postgrados p ON ep.postgrado_id = p.id
               JOIN inscripciones i ON e.id = i.estudiante_id
               JOIN denominaciones d ON i.denominacion_id = d.id
               JOIN anios_lectivos al ON i.año_lectivo_id = al.id
               JOIN escolaridades esc ON i.escolaridad_id = esc.id
               LEFT JOIN usuarios u_mod ON i.usuario_modificacion = u_mod.id
               WHERE 1=1";

$params = [];
$types = "";

// Aplicar filtros
if (!empty($filtros['postgrado_id'])) {
    $sql_report .= " AND p.id = ?";
    $params[] = $filtros['postgrado_id'];
    $types .= "i";
}

if (!empty($filtros['anio_lectivo'])) {
    $sql_report .= " AND al.nombre LIKE ?";
    $params[] = $filtros['anio_lectivo'] . '%';
    $types .= "s";
}

if (!empty($filtros['escolaridad'])) {
    $sql_report .= " AND esc.nombre = ?";
    $params[] = $filtros['escolaridad'];
    $types .= "s";
}

if (!empty($filtros['cuatrimestre'])) {
    $sql_report .= " AND al.nombre LIKE ?";
    $params[] = '%' . $filtros['cuatrimestre'];
    $types .= "s";
}

$sql_report .= " ORDER BY p.nombre, e.apellidos, e.nombres, d.periodo, d.codigo";

$stmt_report = $conn->prepare($sql_report);
if (!empty($params)) {
    $stmt_report->bind_param($types, ...$params);
}
$stmt_report->execute();
$result_report = $stmt_report->get_result();
$report_data = $result_report->fetch_all(MYSQLI_ASSOC);

// Procesar generación de Excel (HTML table method)
if (isset($_GET['generar_excel'])) {
    header('Content-Type: application/vnd.ms-excel; charset=utf-8');
    header('Content-Disposition: attachment; filename="reporte_notas_' . date('Y-m-d_H-i-s') . '.xls"');
    header('Cache-Control: max-age=0');
    header('Expires: Mon, 26 Jul 1997 05:00:00 GMT');
    header('Last-Modified: ' . gmdate('D, d M Y H:i:s') . ' GMT');
    header('Cache-Control: cache, must-revalidate');
    header('Pragma: public');
    
    echo '<!DOCTYPE html>
    <html lang="es">
    <head>
        <meta http-equiv="Content-Type" content="text/html; charset=utf-8">
        <style>
            body { font-family: Arial, sans-serif; font-size: 12px; }
            table { border-collapse: collapse; width: 100%; }
            th, td { border: 1px solid #000000; padding: 5px; text-align: left; }
            th { background-color: #F2F2F2; font-weight: bold; }
            .titulo { font-size: 16px; font-weight: bold; text-align: center; }
            .subtitulo { font-size: 14px; text-align: center; }
        </style>
    </head>
    <body>
        <table>
            <tr>
                <th colspan="15" class="titulo">REPORTE DE NOTAS</th>
            </tr>
            <tr>
                <th colspan="15" class="subtitulo">Generado: ' . date('d/m/Y H:i:s') . '</th>
            </tr>
            <tr>
                <th>Postgrado</th>
                <th>Estudiante</th>
                <th>Documento</th>
                <th>Tipo Doc</th>
                <th>Código</th>
                <th>Denominación</th>
                <th>Periodo</th>
                <th>Créditos</th>
                <th>Tipo</th>
                <th>Nota</th>
                <th>Estado</th>
                <th>Año Lectivo</th>
                <th>Escolaridad</th>
                <th>Modificado Por</th>
                <th>Fecha Modificación</th>
            </tr>';
    
    foreach ($report_data as $data) {
        $documento = '';
        switch($data['tipo_documento']) {
            case 'V': $documento = $data['cedula_venezolana']; break;
            case 'E': $documento = $data['cedula_extranjera']; break;
            case 'P': $documento = $data['pasaporte']; break;
        }
        
        echo '<tr>
                <td>' . htmlspecialchars($data['postgrado_nombre']) . '</td>
                <td>' . htmlspecialchars($data['apellidos'] . ', ' . $data['nombres']) . '</td>
                <td>' . htmlspecialchars($documento) . '</td>
                <td>' . htmlspecialchars($data['tipo_documento']) . '</td>
                <td>' . htmlspecialchars($data['codigo']) . '</td>
                <td>' . htmlspecialchars($data['denominacion_nombre']) . '</td>
                <td>' . htmlspecialchars($data['periodo']) . '</td>
                <td>' . htmlspecialchars($data['credito']) . '</td>
                <td>' . htmlspecialchars($data['tipo']) . '</td>
                <td>' . htmlspecialchars($data['nota_definitiva'] ?? '') . '</td>
                <td>' . htmlspecialchars($data['estado_inscripcion']) . '</td>
                <td>' . htmlspecialchars($data['anio_lectivo_nombre']) . '</td>
                <td>' . htmlspecialchars($data['escolaridad_nombre']) . '</td>
                <td>' . htmlspecialchars($data['modificado_por_nombres'] . ' ' . $data['modificado_por_apellidos']) . '</td>
                <td>' . htmlspecialchars($data['fecha_modificacion']) . '</td>
            </tr>';
    }
    
    echo '</table>
    </body>
    </html>';
    exit;
}

// Procesar generación de PDF con mPDF (que ya tienes funcionando)
if (isset($_GET['generar_pdf'])) {
    $mpdfPath = __DIR__ . '/../vendor/autoload.php';
    if (!file_exists($mpdfPath)) {
        die("ERROR: mPDF no encontrado. Ejecute: composer require mpdf/mpdf");
    }
    require_once $mpdfPath;
    
    try {
        $mpdf = new \Mpdf\Mpdf([
            'mode' => 'utf-8',
            'format' => 'A4-L',
            'margin_left' => 10,
            'margin_right' => 10,
            'margin_top' => 15,
            'margin_bottom' => 15,
            'default_font_size' => 8,
            'default_font' => 'arial',
            'tempDir' => __DIR__ . '/tmp'
        ]);
        
        $mpdf->SetTitle('Reporte de Notas');
        $mpdf->SetAuthor('Sistema Postgrado');
        
        $html = '<!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <style>
                body { font-family: Arial, sans-serif; font-size: 8px; }
                table { width: 100%; border-collapse: collapse; }
                th, td { border: 1px solid #000; padding: 3px; text-align: left; }
                th { background-color: #f2f2f2; font-weight: bold; }
                .title { text-align: center; font-size: 12px; font-weight: bold; margin-bottom: 10px; }
                .subtitle { text-align: center; font-size: 10px; margin-bottom: 10px; }
            </style>
        </head>
        <body>
            <div class="title">REPORTE DE NOTAS</div>
            <div class="subtitle">Generado: ' . date('d/m/Y H:i:s') . '</div>';
        
        // Filtros aplicados
        $filtros_text = "Filtros aplicados: ";
        $filtros_arr = [];
        if (!empty($filtros['postgrado_id'])) {
            $postgrado_nombre = '';
            foreach ($postgrados as $p) {
                if ($p['id'] == $filtros['postgrado_id']) {
                    $postgrado_nombre = $p['nombre'];
                    break;
                }
            }
            $filtros_arr[] = "Postgrado: " . $postgrado_nombre;
        }
        if (!empty($filtros['anio_lectivo'])) $filtros_arr[] = "Año: " . $filtros['anio_lectivo'];
        if (!empty($filtros['escolaridad'])) $filtros_arr[] = "Escolaridad: " . $filtros['escolaridad'];
        if (!empty($filtros['cuatrimestre'])) $filtros_arr[] = "Cuatrimestre: " . $cuatrimestres[$filtros['cuatrimestre']];
        
        if (!empty($filtros_arr)) {
            $html .= '<div style="margin-bottom: 10px; font-size: 7px;">' . $filtros_text . implode(', ', $filtros_arr) . '</div>';
        }
        
        $html .= '<table>
                <thead>
                    <tr>
                        <th>Postgrado</th>
                        <th>Estudiante</th>
                        <th>Documento</th>
                        <th>Código</th>
                        <th>Denominación</th>
                        <th>Periodo</th>
                        <th>Nota</th>
                        <th>Estado</th>
                        <th>Año Lectivo</th>
                    </tr>
                </thead>
                <tbody>';
        
        foreach ($report_data as $data) {
            $documento = '';
            switch($data['tipo_documento']) {
                case 'V': $documento = $data['cedula_venezolana']; break;
                case 'E': $documento = $data['cedula_extranjera']; break;
                case 'P': $documento = $data['pasaporte']; break;
            }
            
            $html .= '<tr>
                    <td>' . htmlspecialchars(mb_substr($data['postgrado_nombre'], 0, 20)) . '</td>
                    <td>' . htmlspecialchars(mb_substr($data['apellidos'] . ', ' . $data['nombres'], 0, 25)) . '</td>
                    <td>' . htmlspecialchars($documento) . '</td>
                    <td>' . htmlspecialchars($data['codigo']) . '</td>
                    <td>' . htmlspecialchars(mb_substr($data['denominacion_nombre'], 0, 30)) . '</td>
                    <td>' . htmlspecialchars($data['periodo']) . '</td>
                    <td>' . htmlspecialchars($data['nota_definitiva'] ?? '') . '</td>
                    <td>' . htmlspecialchars($data['estado_inscripcion']) . '</td>
                    <td>' . htmlspecialchars($data['anio_lectivo_nombre']) . '</td>
                </tr>';
        }
        
        $html .= '</tbody>
            </table>
            <div style="margin-top: 10px; font-size: 7px; text-align: center;">
                Total de registros: ' . count($report_data) . '
            </div>
        </body>
        </html>';
        
        $mpdf->WriteHTML($html);
        $filename = "reporte_notas_" . date('Y-m-d_H-i-s') . ".pdf";
        $mpdf->Output($filename, \Mpdf\Output\Destination::DOWNLOAD);
        exit;
        
    } catch (\Mpdf\MpdfException $e) {
        die("ERROR al generar PDF: " . $e->getMessage());
    }
}

$page_title = "Reportes de Notas - Administrador";
include '../includes/header.php';
?>

<div class="container-fluid px-3 px-md-4">
    <!-- Botón para volver -->
    <div class="mb-3 mb-md-4">
        <a href="index.php" class="btn btn-secondary btn-sm">
            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-arrow-left" viewBox="0 0 16 16">
                <path fill-rule="evenodd" d="M15 8a.5.5 0 0 0-.5-.5H2.707l3.147-3.146a.5.5 0 1 0-.708-.708l-4 4a.5.5 0 0 0 0 .708l4 4a.5.5 0 0 0 .708-.708L2.707 8.5H14.5A.5.5 0 0 0 15 8z"/>
            </svg>
            Volver al Panel
        </a>
    </div>

    <div class="card mb-4">
        <div class="card-header bg-primary text-white py-2 py-md-3">
            <h4 class="mb-0">Generar Reportes de Notas por Cuatrimestre</h4>
        </div>
        <div class="card-body p-2 p-md-3">
            <form method="get" class="mb-0">
                <div class="row g-2 g-md-3">
                    <!-- Filtro de postgrado -->
                    <div class="col-12 col-md-4">
                        <label for="postgrado_id" class="form-label">Postgrado</label>
                        <select name="postgrado_id" id="postgrado_id" class="form-select form-select-sm">
                            <option value="">Todos los postgrados</option>
                            <?php foreach ($postgrados as $p): ?>
                                <option value="<?php echo htmlspecialchars($p['id']); ?>" 
                                    <?php echo ($p['id'] == $filtros['postgrado_id']) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($p['nombre']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <!-- Filtro de año lectivo -->
                    <div class="col-12 col-md-3">
                        <label for="anio_lectivo" class="form-label">Año Lectivo</label>
                        <select name="anio_lectivo" id="anio_lectivo" class="form-select form-select-sm">
                            <option value="">Todos los años</option>
                            <?php foreach ($anios_lectivos as $al): ?>
                                <option value="<?php echo htmlspecialchars($al['nombre']); ?>" 
                                    <?php echo ($al['nombre'] == $filtros['anio_lectivo']) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($al['nombre']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <!-- Filtro de escolaridad -->
                    <div class="col-12 col-md-3">
                        <label for="escolaridad" class="form-label">Escolaridad</label>
                        <select name="escolaridad" id="escolaridad" class="form-select form-select-sm">
                            <option value="">Todas las escolaridades</option>
                            <?php foreach ($escolaridades as $esc): ?>
                                <option value="<?php echo htmlspecialchars($esc['nombre']); ?>" 
                                    <?php echo ($esc['nombre'] == $filtros['escolaridad']) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($esc['nombre']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <!-- Filtro de cuatrimestre -->
                    <div class="col-12 col-md-2">
                        <label for="cuatrimestre" class="form-label">Cuatrimestre</label>
                        <select name="cuatrimestre" id="cuatrimestre" class="form-select form-select-sm">
                            <option value="">Todos</option>
                            <?php foreach ($cuatrimestres as $key => $value): ?>
                                <option value="<?php echo htmlspecialchars($key); ?>" 
                                    <?php echo ($key == $filtros['cuatrimestre']) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($value); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <!-- Botones de acción -->
                    <div class="col-12">
                        <div class="d-flex flex-wrap gap-2 mt-2">
                            <button type="submit" class="btn btn-primary btn-sm flex-fill">
                                <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" fill="currentColor" class="bi bi-search me-1" viewBox="0 0 16 16">
                                    <path d="M11.742 10.344a6.5 6.5 0 1 0-1.397 1.398h-.001c.03.04.062.078.098.115l3.85 3.85a1 1 0 0 0 1.415-1.414l-3.85-3.85a1.007 1.007 0 0 0-.115-.1zM12 6.5a5.5 5.5 0 1 1-11 0 5.5 5.5 0 0 1 11 0z"/>
                                </svg>
                                Generar Reporte
                            </button>
                            <a href="reportes_notas.php" class="btn btn-secondary btn-sm flex-fill">
                                <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" fill="currentColor" class="bi bi-arrow-clockwise me-1" viewBox="0 0 16 16">
                                    <path fill-rule="evenodd" d="M8 3a5 5 0 1 0 4.546 2.914.5.5 0 0 1 .908-.417A6 6 0 1 1 8 2v1z"/>
                                    <path d="M8 4.466V.534a.25.25 0 0 1 .41-.192l2.36 1.966c.12.1.12.284 0 .384L8.41 4.658A.25.25 0 0 1 8 4.466z"/>
                                </svg>
                                Limpiar Filtros
                            </a>
                        </div>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <?php if (!empty($report_data)): ?>
        <div class="card">
            <div class="card-header bg-success text-white py-2 py-md-3 d-flex justify-content-between align-items-center">
                <h4 class="mb-0">Resultados del Reporte</h4>
                <div class="d-flex gap-2">
                    <a href="?<?php echo http_build_query(array_merge($_GET, ['generar_excel' => 1])); ?>" 
                       class="btn btn-light btn-sm">
                        <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" fill="currentColor" class="bi bi-file-earmark-excel me-1" viewBox="0 0 16 16">
                            <path d="M5.884 6.68a.5.5 0 1 0-.768.64L7.349 10l-2.233 2.68a.5.5 0 0 0 .768.64L8 10.781l2.116 2.54a.5.5 0 0 0 .768-.641L8.651 10l2.233-2.68a.5.5 0 0 0-.768-.64L8 9.219l-2.116-2.54z"/>
                            <path d="M14 14V4.5L9.5 0H4a2 2 0 0 0-2 2v12a2 2 0 0 0 2 2h8a2 2 0 0 0 2-2zM9.5 3A1.5 1.5 0 0 0 11 4.5h2V14a1 1 0 0 1-1 1H4a1 1 0 0 1-1-1V2a1 1 0 0 1 1-1h5.5v2z"/>
                        </svg>
                        Exportar Excel
                    </a>
                    <a href="?<?php echo http_build_query(array_merge($_GET, ['generar_pdf' => 1])); ?>" 
                       class="btn btn-light btn-sm">
                        <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" fill="currentColor" class="bi bi-file-earmark-pdf me-1" viewBox="0 0 16 16">
                            <path d="M14 14V4.5L9.5 0H4a2 2 0 0 0-2 2v12a2 2 0 0 0 2 2h8a2 2 0 0 0 2-2zM9.5 3A1.5 1.5 0 0 0 11 4.5h2V14a1 1 0 0 1-1 1H4a1 1 0 0 1-1-1V2a1 1 0 0 1 1-1h5.5v2z"/>
                            <path d="M4.603 14.087a.81.81 0 0 1-.438-.42c-.195-.388-.13-.776.08-1.102.198-.307.526-.568.897-.787a7.68 7.68 0 0 1 1.482-.645 19.697 19.697 0 0 0 1.062-2.227 7.269 7.269 0 0 1-.43-1.295c-.086-.4-.119-.796-.046-1.136.075-.354.274-.672.65-.823.192-.077.4-.12.602-.077a.7.7 0 0 1 .477.365c.088.164.12.356.127.538.007.188-.012.396-.047.614-.084.51-.27 1.134-.52 1.734a17.431 17.431 0 0 0 .975 1.137l.01.012c.118.13.24.254.367.372.067.06.135.12.204.17.21.155.46.278.736.372.292.1.63.163 1.005.195.377.032.78.032 1.194.032.136 0 .273 0 .406-.004.06-.002.118-.005.176-.008.57-.035 1.095-.127 1.553-.285.455-.156.85-.384 1.172-.674.16-.144.304-.3.432-.456l.003-.005c.114-.16.22-.325.317-.495.1-.176.19-.357.27-.542.08-.186.15-.376.21-.568.058-.19.105-.382.142-.576.036-.194.062-.39.078-.586.015-.197.02-.394.014-.591a3.104 3.104 0 0 0-.049-.562c-.035-.186-.087-.367-.155-.542a2.028 2.028 0 0 0-.383-.522 1.94 1.94 0 0 0-.584-.408 2.618 2.618 0 0 0-.875-.268 6.834 6.834 0 0 0-1.172-.083c-.38.008-.75.05-1.1.127-.346.077-.667.193-.957.345-.285.15-.532.337-.735.558-.2.22-.354.475-.456.758-.1.283-.147.59-.137.91a2.12 2.12 0 0 0 .073.613c.053.188.14.363.256.516a1.3 1.3 0 0 0 .464.367c.183.082.39.128.61.135.222.007.446-.025.664-.077.216-.052.424-.13.62-.234.196-.103.38-.23.546-.38.166-.15.312-.322.437-.515.125-.192.228-.403.308-.63.08-.226.136-.467.166-.72.03-.253.034-.516.012-.788a7.577 7.577 0 0 0-2.3-.67 9.796 9.796 0 0 0-1.098-.13 16.045 16.045 0 0 0-1.327-.064c-.45.008-.884.038-1.3.091-.415.053-.81.13-1.182.232-.37.102-.712.232-1.026.39-.313.157-.594.343-.84.556-.246.213-.454.453-.624.72-.17.268-.3.565-.388.89-.088.325-.133.68-.133 1.06 0 .377.045.75.135 1.11.09.358.223.694.4 1 .174.307.392.584.653.83.26.246.563.457.907.632.345.175.73.317 1.152.424.424.107.885.18 1.38.22.497.04 1.018.06 1.56.06.543 0 1.064-.02 1.56-.06.496-.04.957-.113 1.38-.22.422-.107.807-.25 1.152-.424.344-.175.647-.386.907-.632.26-.246.48-.523.653-.83.177-.306.31-.642.4-1 .09-.36.135-.733.135-1.11 0-.38-.045-.735-.133-1.06a3.616 3.616 0 0 0-.388-.89 3.206 3.206 0 0 0-.624-.72 3.01 3.01 0 0 0-.84-.556 4.29 4.29 0 0 0-1.026-.39 7.72 7.72 0 0 0-1.182-.232 8.964 8.964 0 0 0-1.3-.091 16.05 16.05 0 0 0-1.327.064 9.8 9.8 0 0 0-1.098.13 7.58 7.58 0 0 0-2.3.67c-.222.072-.438.162-.646.27-.208.107-.406.232-.593.375-.187.143-.362.303-.524.48-.162.177-.31.37-.443.58-.133.21-.25.437-.35.68-.1.243-.182.5-.245.772-.064.272-.096.558-.096.858 0 .299.032.585.096.857.063.272.145.53.245.772.1.243.217.47.35.68.133.21.28.403.443.58.163.177.337.337.524.48.187.143.385.268.593.375.208.108.424.198.646.27.57.186 1.18.279 1.82.279.64 0 1.25-.093 1.82-.279.222-.072.438-.162.646-.27.208-.107.406-.232.593-.375.187-.143.362-.303.524-.48.162-.177.31-.37.443-.58.133-.21.25-.437.35-.68.1-.243.182-.5.245-.772.064-.272.096-.558.096-.857 0-.3-.032-.586-.096-.858a4.48 4.48 0 0 0-.245-.772 4.198 4.198 0 0 0-.35-.68 4.387 4.387 0 0 0-.443-.58 4.08 4.08 0 0 0-.524-.48 3.88 3.88 0 0 0-.593-.375 3.67 3.67 0 0 0-.646-.27 7.123 7.123 0 0 0-1.82-.279c-.64 0-1.25.093-1.82.279z"/>
                        </svg>
                        Exportar PDF
                    </a>
                </div>
            </div>
            <div class="card-body p-2 p-md-3">
                <div class="alert alert-info mb-3">
                    <strong>Total de registros:</strong> <?php echo count($report_data); ?> | 
                    <strong>Filtros aplicados:</strong> 
                    <?php
                    $filtros_aplicados = [];
                    if (!empty($filtros['postgrado_id'])) {
                        $postgrado_nombre = '';
                        foreach ($postgrados as $p) {
                            if ($p['id'] == $filtros['postgrado_id']) {
                                $postgrado_nombre = $p['nombre'];
                                break;
                            }
                        }
                        $filtros_aplicados[] = "Postgrado: " . $postgrado_nombre;
                    }
                    if (!empty($filtros['anio_lectivo'])) $filtros_aplicados[] = "Año: " . $filtros['anio_lectivo'];
                    if (!empty($filtros['escolaridad'])) $filtros_aplicados[] = "Escolaridad: " . $filtros['escolaridad'];
                    if (!empty($filtros['cuatrimestre'])) $filtros_aplicados[] = "Cuatrimestre: " . $cuatrimestres[$filtros['cuatrimestre']];
                    
                    echo implode(' | ', $filtros_aplicados);
                    ?>
                </div>

                <div class="table-responsive-container">
                    <div class="table-responsive">
                        <table class="table table-striped table-hover table-sm">
                            <thead>
                                <tr>
                                    <th>Postgrado</th>
                                    <th>Estudiante</th>
                                    <th>Documento</th>
                                    <th>Código</th>
                                    <th>Denominación</th>
                                    <th>Periodo</th>
                                    <th>Créditos</th>
                                    <th>Tipo</th>
                                    <th>Nota</th>
                                    <th>Estado</th>
                                    <th>Año Lectivo</th>
                                    <th>Escolaridad</th>
                                    <th>Modificado Por</th>
                                    <th>Fecha Modificación</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($report_data as $data): 
                                    $documento = '';
                                    switch($data['tipo_documento']) {
                                        case 'V': $documento = $data['cedula_venezolana']; break;
                                        case 'E': $documento = $data['cedula_extranjera']; break;
                                        case 'P': $documento = $data['pasaporte']; break;
                                    }
                                ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($data['postgrado_nombre']); ?></td>
                                        <td>
                                            <strong><?php echo htmlspecialchars($data['apellidos']); ?></strong>, 
                                            <?php echo htmlspecialchars($data['nombres']); ?>
                                        </td>
                                        <td>
                                            <small class="text-muted"><?php echo $data['tipo_documento']; ?>:</small><br>
                                            <?php echo htmlspecialchars($documento); ?>
                                        </td>
                                        <td><?php echo htmlspecialchars($data['codigo']); ?></td>
                                        <td><?php echo htmlspecialchars($data['denominacion_nombre']); ?></td>
                                        <td><?php echo htmlspecialchars($data['periodo']); ?></td>
                                        <td><?php echo htmlspecialchars($data['credito']); ?></td>
                                        <td>
                                            <span class="badge bg-<?php echo $data['tipo'] == 'puntaje' ? 'primary' : 'info'; ?>">
                                                <?php echo htmlspecialchars($data['tipo']); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php if (!empty($data['nota_definitiva'])): ?>
                                                <span class="badge bg-<?php echo ($data['tipo'] == 'puntaje' && floatval($data['nota_definitiva']) >= 10) ? 'success' : 'danger'; ?>">
                                                    <?php echo htmlspecialchars($data['nota_definitiva']); ?>
                                                </span>
                                            <?php else: ?>
                                                <span class="badge bg-warning">Sin nota</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <span class="badge bg-<?php echo $data['estado_inscripcion'] == 'finalizada' ? 'success' : 'secondary'; ?>">
                                                <?php echo htmlspecialchars(ucfirst($data['estado_inscripcion'])); ?>
                                            </span>
                                        </td>
                                        <td><?php echo htmlspecialchars($data['anio_lectivo_nombre']); ?></td>
                                        <td><?php echo htmlspecialchars($data['escolaridad_nombre']); ?></td>
                                        <td>
                                            <?php if (!empty($data['modificado_por_nombres'])): ?>
                                                <?php echo htmlspecialchars($data['modificado_por_nombres'] . ' ' . $data['modificado_por_apellidos']); ?>
                                            <?php else: ?>
                                                <span class="text-muted">N/A</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if (!empty($data['fecha_modificacion'])): ?>
                                                <?php echo date('d/m/Y H:i', strtotime($data['fecha_modificacion'])); ?>
                                            <?php else: ?>
                                                <span class="text-muted">N/A</span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    <?php elseif (!empty($_GET)): ?>
        <div class="card">
            <div class="card-body text-center py-4">
                <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" fill="currentColor" class="bi bi-search text-muted mb-2" viewBox="0 0 16 16">
                    <path d="M11.742 10.344a6.5 6.5 0 1 0-1.397 1.398h-.001c.03.04.062.078.098.115l3.85 3.85a1 1 0 0 0 1.415-1.414l-3.85-3.85a1.007 1.007 0 0 0-.115-.1zM12 6.5a5.5 5.5 0 1 1-11 0 5.5 5.5 0 0 1 11 0z"/>
                </svg>
                <h5 class="text-muted">No se encontraron datos</h5>
                <p class="text-muted">No hay registros que coincidan con los criterios de búsqueda.</p>
            </div>
        </div>
    <?php endif; ?>
</div>

<style>
.table-responsive-container {
    position: relative;
    width: 100%;
    overflow-x: auto;
    -webkit-overflow-scrolling: touch;
    border: 1px solid #dee2e6;
    border-radius: 6px;
}
.table-responsive {
    min-width: 1200px;
    margin-bottom: 0;
}
</style>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>

<?php
include '../includes/footer.php';
$conn->close();
?>