<?php
// generar_reporte.php - VERSIÓN FINAL CORREGIDA
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');

if (!isset($_SESSION['username'])) {
    header("Location: ../login.php");
    exit();
}

require_once '../includes/db.php';

// Validación de parámetros
if(!isset($_GET['postgrado_id']) || !isset($_GET['promocion_id'])) {
    die("Error: Parámetros postgrado_id y promocion_id son requeridos");
}

$postgrado_id = intval($_GET['postgrado_id']);
$promocion_id = intval($_GET['promocion_id']);
$preview = isset($_GET['preview']) ? true : false;

if($postgrado_id <= 0 || $promocion_id <= 0) {
    die("Error: Los parámetros deben ser números positivos");
}

// Configurar encoding UTF-8 para la conexión ANTES de cualquier consulta
$conn->set_charset("utf8mb4");

// 1. Obtener información del postgrado y promoción
$query_postgrado = "SELECT p.*, pl.nombre as plan_nombre
                   FROM postgrados p
                   LEFT JOIN planes pl ON p.plan_id = pl.id
                   WHERE p.id = ?";
$stmt = $conn->prepare($query_postgrado);
$stmt->bind_param("i", $postgrado_id);
$stmt->execute();
$postgrado = $stmt->get_result()->fetch_assoc();
$stmt->close();

if(!$postgrado) die("No se encontró el postgrado especificado");

$query_promocion = "SELECT nombre FROM promociones WHERE id = ?";
$stmt = $conn->prepare($query_promocion);
$stmt->bind_param("i", $promocion_id);
$stmt->execute();
$promocion = $stmt->get_result()->fetch_assoc();
$stmt->close();

if(!$promocion) die("No se encontró la promoción especificada");

$postgrado['promocion_nombre'] = $promocion['nombre'];

// 2. Obtener estudiantes graduados
$query = "SELECT e.*
          FROM estudiantes e
          JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
          WHERE ep.postgrado_id = ? AND ep.promocion_id = ? AND ep.estado = 'graduado'
          ORDER BY e.apellidos, e.nombres";
$stmt = $conn->prepare($query);
$stmt->bind_param("ii", $postgrado_id, $promocion_id);
$stmt->execute();
$estudiantes = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

if(empty($estudiantes)) {
    echo "<h2>No hay estudiantes graduados</h2>";
    echo "<p>No se encontraron estudiantes graduados para:</p>";
    echo "<ul><li>Postgrado: " . htmlspecialchars($postgrado['nombre']) . "</li>";
    echo "<li>Promoción: " . htmlspecialchars($promocion['nombre']) . "</li></ul>";
    echo "<p><a href='seleccion_reporte.php'>Volver a intentar</a></p>";
    $conn->close();
    exit();
}

// 3. Obtener denominaciones (asignaturas)
$query = "SELECT d.* FROM denominaciones d WHERE d.postgrado_id = ? ORDER BY d.periodo, d.codigo";
$stmt = $conn->prepare($query);
$stmt->bind_param("i", $postgrado_id);
$stmt->execute();
$denominaciones = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

if(empty($denominaciones)) die("No hay asignaturas definidas para este postgrado");

// 4. Obtener notas de los estudiantes con años lectivos - CONSULTA CORREGIDA
$notas = array();
$estudiantes_ids = array_column($estudiantes, 'id');

if(empty($estudiantes_ids)) {
    die("No hay IDs de estudiantes para buscar notas");
}

$placeholders = implode(',', array_fill(0, count($estudiantes_ids), '?'));

// CONSULTA CORREGIDA - Sin referencia a d.año_lectivo que no existe
$query = "SELECT i.*, d.id as denominacion_id, d.codigo, d.nombre as asignatura, 
                 d.credito, d.tipo, d.periodo, 
                 al.nombre as año_lectivo
          FROM inscripciones i
          JOIN denominaciones d ON i.denominacion_id = d.id
          LEFT JOIN años_lectivos al ON i.año_lectivo_id = al.id
          WHERE d.postgrado_id = ? AND i.estudiante_id IN ($placeholders)
          ORDER BY d.periodo, d.codigo";

$stmt = $conn->prepare($query);
if (!$stmt) {
    die("Error en la preparación de la consulta: " . $conn->error);
}

$params = array_merge([$postgrado_id], $estudiantes_ids);
$types = str_repeat('i', count($params));
$stmt->bind_param($types, ...$params);

if(!$stmt->execute()) {
    die("Error al ejecutar la consulta: " . $stmt->error);
}

$resultados = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

foreach($resultados as $row) {
    $notas[$row['estudiante_id']][] = $row;
}

// 5. Calcular promedios
$promedios = array();
foreach($estudiantes as $estudiante) {
    $suma_simple = $suma_ponderada = $contador = $creditos_total = 0;
    
    if(isset($notas[$estudiante['id']])) {
        foreach($notas[$estudiante['id']] as $nota) {
            if(is_numeric($nota['nota_definitiva'])) {
                $suma_simple += $nota['nota_definitiva'];
                $suma_ponderada += $nota['nota_definitiva'] * $nota['credito'];
                $creditos_total += $nota['credito'];
                $contador++;
            }
        }
    }
    
    $promedios[$estudiante['id']] = array(
        'simple' => $contador > 0 ? $suma_simple / $contador : 0,
        'ponderado' => $creditos_total > 0 ? $suma_ponderada / $creditos_total : 0
    );
}

// 6. Obtener año lectivo general
$query_años = "SELECT nombre FROM años_lectivos LIMIT 1";
$result_años = $conn->query($query_años);
$año_lectivo_general = $result_años ? $result_años->fetch_assoc()['nombre'] : '2023-2024';

// 7. Pasar todas las variables necesarias
if($preview) {
    // Para previsualización
    require_once 'reporte_preview.php';
} else {
    // Para Excel
    require_once 'reporte_excel.php';
    $conn->close();
}
?>