<?php
ini_set('display_errors', 1);
error_reporting(E_ALL);

// Centralized authentication and session check
$allowed_roles = ['estudiante'];
require_once __DIR__ . '/../includes/auth_check.php';
include __DIR__ . '/../includes/db.php';

// Verificar que la conexión use UTF-8
$conn->set_charset("utf8mb4");

$username = $_SESSION['username']; // Cédula del estudiante

// Obtener todos los datos del estudiante - MÉTODO MEJORADO CON MÚLTIPLES INTENTOS
$student_data = null;
$estudiante_id = null;

// INTENTO 1: Buscar por estudiante_id en la tabla usuarios
$stmt_student = $conn->prepare("SELECT e.* FROM estudiantes e 
                               INNER JOIN usuarios u ON u.estudiante_id = e.id 
                               WHERE u.username = ?");
if ($stmt_student) {
    $stmt_student->bind_param("s", $username);
    $stmt_student->execute();
    $result_student = $stmt_student->get_result();
    
    if ($result_student->num_rows === 1) {
        $student_data = $result_student->fetch_assoc();
        $estudiante_id = $student_data['id'];
    }
    $stmt_student->close();
}

// INTENTO 2: Si no se encontró, buscar directamente por cédula
if (!$student_data) {
    $stmt_student = $conn->prepare("SELECT * FROM estudiantes WHERE cedula = ?");
    if ($stmt_student) {
        $stmt_student->bind_param("s", $username);
        $stmt_student->execute();
        $result_student = $stmt_student->get_result();
        
        if ($result_student->num_rows === 1) {
            $student_data = $result_student->fetch_assoc();
            $estudiante_id = $student_data['id'];
            
            // Actualizar la relación en la tabla usuarios si se encontró por cédula
            $stmt_update = $conn->prepare("UPDATE usuarios SET estudiante_id = ? WHERE username = ?");
            $stmt_update->bind_param("is", $estudiante_id, $username);
            $stmt_update->execute();
            $stmt_update->close();
        }
        $stmt_student->close();
    }
}

// INTENTO 3: Buscar por otros documentos si aún no se encontró
if (!$student_data) {
    $stmt_student = $conn->prepare("SELECT * FROM estudiantes 
                                   WHERE cedula_venezolana = ? 
                                   OR cedula_extranjera = ? 
                                   OR pasaporte = ?");
    if ($stmt_student) {
        $stmt_student->bind_param("sss", $username, $username, $username);
        $stmt_student->execute();
        $result_student = $stmt_student->get_result();
        
        if ($result_student->num_rows === 1) {
            $student_data = $result_student->fetch_assoc();
            $estudiante_id = $student_data['id'];
            
            // Actualizar la relación en la tabla usuarios
            $stmt_update = $conn->prepare("UPDATE usuarios SET estudiante_id = ? WHERE username = ?");
            $stmt_update->bind_param("is", $estudiante_id, $username);
            $stmt_update->execute();
            $stmt_update->close();
        }
        $stmt_student->close();
    }
}

// Si después de todos los intentos no se encontró el estudiante
if (!$student_data) {
    // Mostrar página de error detallada
    echo "<!DOCTYPE html>
    <html lang='es'>
    <head>
        <meta charset='UTF-8'>
        <meta name='viewport' content='width=device-width, initial-scale=1.0'>
        <title>Error - Sistema de Postgrado</title>
        <link href='https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css' rel='stylesheet'>
        <style>
            .error-container {
                max-width: 600px;
                margin: 50px auto;
                padding: 20px;
            }
            .debug-info {
                background: #f8f9fa;
                border-left: 4px solid #dc3545;
                padding: 15px;
                margin: 15px 0;
                font-family: monospace;
                font-size: 0.9em;
            }
        </style>
    </head>
    <body>
        <div class='container error-container'>
            <div class='alert alert-danger'>
                <h4><i class='bi bi-exclamation-triangle'></i> Error: Datos del estudiante no encontrados</h4>
                <p class='mb-3'>No se pudieron cargar los datos de su perfil de estudiante.</p>
                
                <div class='debug-info'>
                    <strong>Información para diagnóstico:</strong><br>
                    Usuario en sesión: <strong>" . htmlspecialchars($username) . "</strong><br>
                    ID de usuario: <strong>" . ($_SESSION['user_id'] ?? 'N/A') . "</strong><br>
                    Rol: <strong>" . ($_SESSION['rol'] ?? 'N/A') . "</strong>
                </div>

                <p><strong>Posibles causas:</strong></p>
                <ul>
                    <li>No existe un registro en la tabla 'estudiantes' con su cédula/documento</li>
                    <li>El campo 'estudiante_id' en su usuario no está correctamente relacionado</li>
                    <li>Su cédula en 'usuarios.username' no coincide con 'estudiantes.cedula'</li>
                    <li>Hay un problema temporal con la base de datos</li>
                </ul>
                
                <p class='mb-0'><strong>Solución:</strong> Por favor contacte al administrador del sistema para resolver este problema.</p>
            </div>
            
            <div class='text-center'>
                <a href='../includes/logout.php' class='btn btn-warning me-2'>Cerrar Sesión</a>
                <a href='javascript:location.reload()' class='btn btn-secondary'>Reintentar</a>
            </div>
            
            <div class='mt-4 text-center'>
                <small class='text-muted'>
                    Si el problema persiste, informe al administrador con la información de diagnóstico mostrada arriba.
                </small>
            </div>
        </div>
        
        <script>
            // Intentar recargar automáticamente después de 10 segundos
            setTimeout(function() {
                location.reload();
            }, 10000);
        </script>
    </body>
    </html>";
    exit();
}

// Obtener los postgrados del estudiante
$sql_postgrados = "SELECT 
                        p.id as postgrado_id,
                        p.nombre as postgrado_nombre,
                        ep.estado,
                        ep.fecha_inscripcion,
                        pr.nombre as promocion_nombre
                   FROM estudiante_postgrado ep
                   JOIN postgrados p ON ep.postgrado_id = p.id
                   LEFT JOIN promociones pr ON ep.promocion_id = pr.id
                   WHERE ep.estudiante_id = ?
                   ORDER BY ep.fecha_inscripcion DESC";

$stmt_postgrados = $conn->prepare($sql_postgrados);
$stmt_postgrados->bind_param("i", $estudiante_id);
$stmt_postgrados->execute();
$postgrados_result = $stmt_postgrados->get_result();
$postgrados = $postgrados_result->fetch_all(MYSQLI_ASSOC);
$stmt_postgrados->close();

// Para cada postgrado, obtener las notas con información del período
foreach ($postgrados as $key => $postgrado) {
    // Consulta modificada para incluir el período de la denominación
    $sql_notas = "SELECT 
                     d.nombre as materia,
                     d.periodo as periodo,
                     d.tipo as tipo_evaluacion,
                     d.credito as credito,
                     i.nota_definitiva,
                     al.nombre as anio_lectivo
                FROM inscripciones i
                JOIN denominaciones d ON i.denominacion_id = d.id
                LEFT JOIN anios_lectivos al ON i.año_lectivo_id = al.id
                WHERE i.estudiante_id = ? AND d.postgrado_id = ?
                ORDER BY d.periodo, d.codigo";

    $stmt_notas = $conn->prepare($sql_notas);
    if ($stmt_notas === false) {
        die("Error en la preparación de la consulta de notas: " . $conn->error);
    }
    $stmt_notas->bind_param("ii", $estudiante_id, $postgrado['postgrado_id']);
    
    if (!$stmt_notas->execute()) {
        die("Error al ejecutar la consulta de notas: " . $stmt_notas->error);
    }
    
    $notas_result = $stmt_notas->get_result();
    $postgrados[$key]['notas'] = $notas_result->fetch_all(MYSQLI_ASSOC);
    $stmt_notas->close();
}

$page_title = "Panel de Estudiante";
include __DIR__ . '/../includes/header.php';
?>

<div class="container mt-4">
    <!-- Encabezado con información del estudiante -->
    <div class="mb-4">
        <h1>Bienvenido, <?= htmlspecialchars($student_data['nombres'] ?? '') ?></h1>
        <p class="text-muted mb-0">Cédula: <?= htmlspecialchars($student_data['cedula'] ?? '') ?></p>
    </div>

    <!-- Información Personal -->
    <div class="card mb-4">
        <div class="card-header bg-primary text-white d-flex justify-content-between align-items-center">
            <h4 class="mb-0">Mi Información Personal</h4>
            <span class="badge bg-light text-primary">Estudiante</span>
        </div>
        <div class="card-body">
            <div class="row">
                <div class="col-md-6">
                    <p><strong>Nombre Completo:</strong> <?= htmlspecialchars(($student_data['nombres'] ?? '') . ' ' . ($student_data['apellidos'] ?? '')) ?></p>
                    <p><strong>Cédula:</strong> <?= htmlspecialchars($student_data['cedula'] ?? '') ?></p>
                    <p><strong>Tipo de Documento:</strong> 
                        <?php
                        $tipo_doc = $student_data['tipo_documento'] ?? 'V';
                        switch($tipo_doc) {
                            case 'V': echo 'Cédula Venezolana'; break;
                            case 'E': echo 'Cédula Extranjera'; break;
                            case 'P': echo 'Pasaporte'; break;
                            default: echo htmlspecialchars($tipo_doc);
                        }
                        ?>
                    </p>
                </div>
                <div class="col-md-6">
                    <p><strong>Correo Electrónico:</strong> 
                        <?= !empty($student_data['correo']) ? htmlspecialchars($student_data['correo']) : '<span class="text-muted">No registrado</span>' ?>
                    </p>
                    <p><strong>Teléfono:</strong> 
                        <?= !empty($student_data['telefono']) ? htmlspecialchars($student_data['telefono']) : '<span class="text-muted">No registrado</span>' ?>
                    </p>
                    <p><strong>Celular:</strong> 
                        <?= !empty($student_data['celular']) ? htmlspecialchars($student_data['celular']) : '<span class="text-muted">No registrado</span>' ?>
                    </p>
                </div>
            </div>
            
            <?php if (!empty($student_data['direccion'])): ?>
            <div class="row mt-2">
                <div class="col-12">
                    <p><strong>Dirección:</strong> <?= htmlspecialchars($student_data['direccion']) ?></p>
                </div>
            </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Postgrados y Notas -->
    <div class="card">
        <div class="card-header bg-primary text-white">
            <h4 class="mb-0">Mis Postgrados y Calificaciones</h4>
        </div>
        <div class="card-body">
            <?php if (empty($postgrados)): ?>
                <div class="alert alert-info">
                    <div class="d-flex">
                        <div class="me-3">
                            <i class="bi bi-info-circle fs-4"></i>
                        </div>
                        <div>
                            <h5>No estás inscrito en ningún postgrado</h5>
                            <p class="mb-0">Actualmente no tienes postgrados activos. Si crees que esto es un error, por favor contacta con la secretaría de postgrado.</p>
                        </div>
                    </div>
                </div>
            <?php else: ?>
                <!-- Navegación por pestañas para postgrados -->
                <ul class="nav nav-tabs" id="postgradosTab" role="tablist">
                    <?php foreach ($postgrados as $index => $postgrado): ?>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link <?= $index === 0 ? 'active' : '' ?>" 
                                    id="tab-<?= $index ?>" 
                                    data-bs-toggle="tab" 
                                    data-bs-target="#content-<?= $index ?>" 
                                    type="button" 
                                    role="tab" 
                                    aria-controls="content-<?= $index ?>" 
                                    aria-selected="<?= $index === 0 ? 'true' : 'false' ?>">
                                <strong><?= htmlspecialchars($postgrado['postgrado_nombre']) ?></strong>
                                <span class="badge bg-<?= $postgrado['estado'] === 'activo' ? 'success' : ($postgrado['estado'] === 'graduado' ? 'info' : 'warning') ?> ms-1">
                                    <?= htmlspecialchars(ucfirst($postgrado['estado'])) ?>
                                </span>
                            </button>
                        </li>
                    <?php endforeach; ?>
                </ul>

                <!-- Contenido de las pestañas -->
                <div class="tab-content p-3 border border-top-0" id="postgradosTabContent">
                    <?php foreach ($postgrados as $index => $postgrado): ?>
                        <div class="tab-pane fade <?= $index === 0 ? 'show active' : '' ?>" 
                             id="content-<?= $index ?>" 
                             role="tabpanel" 
                             aria-labelledby="tab-<?= $index ?>">
                            
                            <!-- Información del postgrado -->
                            <div class="row mb-4">
                                <div class="col-md-6">
                                    <p><strong>Programa de Postgrado:</strong> <?= htmlspecialchars($postgrado['postgrado_nombre']) ?></p>
                                    <p><strong>Promoción:</strong> <?= !empty($postgrado['promocion_nombre']) ? htmlspecialchars($postgrado['promocion_nombre']) : '<span class="text-muted">No asignada</span>' ?></p>
                                </div>
                                <div class="col-md-6">
                                    <p><strong>Estado:</strong> 
                                        <span class="badge bg-<?= $postgrado['estado'] === 'activo' ? 'success' : ($postgrado['estado'] === 'graduado' ? 'info' : 'warning') ?>">
                                            <?= htmlspecialchars(ucfirst($postgrado['estado'])) ?>
                                        </span>
                                    </p>
                                    <?php 
                                    $fecha_inscripcion = DateTime::createFromFormat('Y-m-d', $postgrado['fecha_inscripcion']);
                                    ?>
                                    <p><strong>Fecha de Inscripción:</strong> <?= $fecha_inscripcion ? $fecha_inscripcion->format('d/m/Y') : htmlspecialchars($postgrado['fecha_inscripcion']) ?></p>
                                </div>
                            </div>

                            <!-- Notas del postgrado -->
                            <?php if (empty($postgrado['notas'])): ?>
                                <div class="alert alert-warning">
                                    <div class="d-flex">
                                        <div class="me-3">
                                            <i class="bi bi-exclamation-triangle fs-4"></i>
                                        </div>
                                        <div>
                                            <h5>Sin calificaciones registradas</h5>
                                            <p class="mb-0">Aún no hay calificaciones registradas para este postgrado. Las calificaciones se actualizan periódicamente.</p>
                                        </div>
                                    </div>
                                </div>
                            <?php else: ?>
                                <!-- Cálculos generales del postgrado -->
                                <?php
                                $total_asignaturas_postgrado = count($postgrado['notas']);
                                $asignaturas_aprobadas_postgrado = 0;
                                $asignaturas_reprobadas_postgrado = 0;
                                $total_creditos_postgrado = 0;
                                $nota_simple_postgrado = 0;
                                $nota_ponderada_postgrado = 0;
                                $asignaturas_puntaje_aprobadas_postgrado = 0;
                                $asignaturas_sin_nota_postgrado = 0;
                                
                                foreach ($postgrado['notas'] as $nota) {
                                    $nota_valor = $nota['nota_definitiva'] ?? '';
                                    $creditos = intval($nota['credito'] ?? 0);
                                    $es_numerica = is_numeric($nota_valor);
                                    $aprobada = false;
                                    $reprobada = false;
                                    
                                    // Contar asignaturas sin nota
                                    if (empty($nota_valor)) {
                                        $asignaturas_sin_nota_postgrado++;
                                    } else {
                                        if ($nota['tipo_evaluacion'] == 'Aprobado_Reprobado') {
                                            $aprobada = (strtoupper($nota_valor) == 'APROBADO');
                                            $reprobada = !$aprobada;
                                        } elseif ($es_numerica) {
                                            $nota_num = floatval($nota_valor);
                                            $aprobada = ($nota_num >= 10);
                                            $reprobada = ($nota_num < 10);
                                            
                                            // Cálculos para promedios (solo asignaturas de tipo puntaje aprobadas)
                                            if ($aprobada) {
                                                $nota_simple_postgrado += $nota_num;
                                                $nota_ponderada_postgrado += ($nota_num * $creditos);
                                                $total_creditos_postgrado += $creditos;
                                                $asignaturas_puntaje_aprobadas_postgrado++;
                                            }
                                        }
                                    }
                                    
                                    if ($aprobada) {
                                        $asignaturas_aprobadas_postgrado++;
                                    }
                                    if ($reprobada) {
                                        $asignaturas_reprobadas_postgrado++;
                                    }
                                }
                                
                                $promedio_simple_postgrado = $asignaturas_puntaje_aprobadas_postgrado > 0 ? 
                                                            $nota_simple_postgrado / $asignaturas_puntaje_aprobadas_postgrado : 0;
                                $promedio_ponderado_postgrado = $total_creditos_postgrado > 0 ? 
                                                               $nota_ponderada_postgrado / $total_creditos_postgrado : 0;
                                ?>
                                
                                <!-- Resumen general del postgrado -->
                                <div class="row mb-4">
                                    <div class="col-md-12">
                                        <div class="card bg-light border-primary">
                                            <div class="card-header bg-primary text-white">
                                                <h5 class="mb-0">Resumen General del Postgrado</h5>
                                            </div>
                                            <div class="card-body">
                                                <div class="row text-center">
                                                    <div class="col-md-2 mb-3">
                                                        <small class="text-muted d-block">Total Materias</small>
                                                        <div class="fw-bold fs-4 text-primary"><?= $total_asignaturas_postgrado ?></div>
                                                    </div>
                                                    <div class="col-md-2 mb-3">
                                                        <small class="text-muted d-block">Aprobadas</small>
                                                        <div class="fw-bold fs-4 text-success"><?= $asignaturas_aprobadas_postgrado ?></div>
                                                    </div>
                                                    <div class="col-md-2 mb-3">
                                                        <small class="text-muted d-block">Reprobadas</small>
                                                        <div class="fw-bold fs-4 text-danger"><?= $asignaturas_reprobadas_postgrado ?></div>
                                                    </div>
                                                    <div class="col-md-2 mb-3">
                                                        <small class="text-muted d-block">Sin Calificar</small>
                                                        <div class="fw-bold fs-4 text-secondary"><?= $asignaturas_sin_nota_postgrado ?></div>
                                                    </div>
                                                    <div class="col-md-2 mb-3">
                                                        <small class="text-muted d-block">Promedio Simple</small>
                                                        <div class="fw-bold fs-4 <?= $promedio_simple_postgrado >= 15 ? 'text-success' : ($promedio_simple_postgrado >= 10 ? 'text-warning' : 'text-danger') ?>">
                                                            <?= number_format($promedio_simple_postgrado, 2) ?>
                                                        </div>
                                                    </div>
                                                    <div class="col-md-2 mb-3">
                                                        <small class="text-muted d-block">Promedio Ponderado</small>
                                                        <div class="fw-bold fs-4 <?= $promedio_ponderado_postgrado >= 15 ? 'text-success' : ($promedio_ponderado_postgrado >= 10 ? 'text-warning' : 'text-danger') ?>">
                                                            <?= number_format($promedio_ponderado_postgrado, 2) ?>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <!-- Agrupar notas por período -->
                                <?php 
                                $notas_por_periodo = [];
                                foreach ($postgrado['notas'] as $nota) {
                                    $periodo = $nota['periodo'] ?? 0;
                                    $notas_por_periodo[$periodo][] = $nota;
                                }
                                // Ordenar por período
                                ksort($notas_por_periodo);
                                ?>

                                <!-- Navegación por pestañas para períodos -->
                                <?php if (count($notas_por_periodo) > 1): ?>
                                    <ul class="nav nav-pills mb-3" id="periodosTab-<?= $index ?>" role="tablist">
                                        <?php $periodo_index = 0; ?>
                                        <?php foreach ($notas_por_periodo as $periodo_num => $notas_del_periodo): ?>
                                            <li class="nav-item" role="presentation">
                                                <button class="nav-link <?= $periodo_index === 0 ? 'active' : '' ?>" 
                                                        id="periodo-tab-<?= $index ?>-<?= $periodo_index ?>" 
                                                        data-bs-toggle="pill" 
                                                        data-bs-target="#periodo-content-<?= $index ?>-<?= $periodo_index ?>" 
                                                        type="button" 
                                                        role="tab" 
                                                        aria-controls="periodo-content-<?= $index ?>-<?= $periodo_index ?>" 
                                                        aria-selected="<?= $periodo_index === 0 ? 'true' : 'false' ?>">
                                                    Período <?= $periodo_num ?>
                                                    <span class="badge bg-secondary ms-1"><?= count($notas_del_periodo) ?></span>
                                                </button>
                                            </li>
                                            <?php $periodo_index++; ?>
                                        <?php endforeach; ?>
                                    </ul>
                                <?php endif; ?>

                                <!-- Contenido de períodos -->
                                <div class="<?= count($notas_por_periodo) > 1 ? 'tab-content' : '' ?>">
                                    <?php $periodo_index = 0; ?>
                                    <?php foreach ($notas_por_periodo as $periodo_num => $notas_del_periodo): ?>
                                        <div class="<?= count($notas_por_periodo) > 1 ? 'tab-pane fade' . ($periodo_index === 0 ? ' show active' : '') : '' ?>" 
                                             id="<?= count($notas_por_periodo) > 1 ? 'periodo-content-' . $index . '-' . $periodo_index : '' ?>" 
                                             role="tabpanel" 
                                             aria-labelledby="<?= count($notas_por_periodo) > 1 ? 'periodo-tab-' . $index . '-' . $periodo_index : '' ?>">
                                            
                                            <?php if (count($notas_por_periodo) > 1): ?>
                                                <h5 class="mb-3 text-primary">Período <?= $periodo_num ?></h5>
                                            <?php else: ?>
                                                <h5 class="mb-3 text-primary">Todas las Materias</h5>
                                            <?php endif; ?>
                                            
                                            <!-- Cálculos para el resumen del período -->
                                            <?php
                                            $total_creditos_periodo = 0;
                                            $nota_simple_periodo = 0;
                                            $nota_ponderada_periodo = 0;
                                            $asignaturas_puntaje_aprobadas_periodo = 0;
                                            $total_asignaturas_periodo = count($notas_del_periodo);
                                            $asignaturas_aprobadas_periodo = 0;
                                            $asignaturas_reprobadas_periodo = 0;
                                            $asignaturas_sin_nota_periodo = 0;
                                            
                                            foreach ($notas_del_periodo as $nota) {
                                                $nota_valor = $nota['nota_definitiva'] ?? '';
                                                $creditos = intval($nota['credito'] ?? 0);
                                                $es_numerica = is_numeric($nota_valor);
                                                $aprobada = false;
                                                $reprobada = false;
                                                $tiene_nota = !empty($nota_valor);
                                                
                                                // Contar asignaturas sin nota
                                                if (!$tiene_nota) {
                                                    $asignaturas_sin_nota_periodo++;
                                                } else {
                                                    if ($nota['tipo_evaluacion'] == 'Aprobado_Reprobado') {
                                                        $aprobada = (strtoupper($nota_valor) == 'APROBADO');
                                                        $reprobada = !$aprobada;
                                                    } elseif ($es_numerica) {
                                                        $nota_num = floatval($nota_valor);
                                                        $aprobada = ($nota_num >= 10);
                                                        $reprobada = ($nota_num < 10);
                                                        
                                                        // Cálculos para promedios (solo asignaturas de tipo puntaje aprobadas)
                                                        if ($aprobada) {
                                                            $nota_simple_periodo += $nota_num;
                                                            $nota_ponderada_periodo += ($nota_num * $creditos);
                                                            $total_creditos_periodo += $creditos;
                                                            $asignaturas_puntaje_aprobadas_periodo++;
                                                        }
                                                    }
                                                }
                                                
                                                if ($aprobada) {
                                                    $asignaturas_aprobadas_periodo++;
                                                }
                                                if ($reprobada) {
                                                    $asignaturas_reprobadas_periodo++;
                                                }
                                            }
                                            
                                            $promedio_simple_periodo = $asignaturas_puntaje_aprobadas_periodo > 0 ? 
                                                                      $nota_simple_periodo / $asignaturas_puntaje_aprobadas_periodo : 0;
                                            $promedio_ponderado_periodo = $total_creditos_periodo > 0 ? 
                                                                         $nota_ponderada_periodo / $total_creditos_periodo : 0;
                                            ?>
                                            
                                            <!-- Resumen del período -->
                                            <div class="row mb-4">
                                                <div class="col-md-12">
                                                    <div class="card bg-light">
                                                        <div class="card-header bg-secondary text-white">
                                                            <h6 class="mb-0">Resumen del Período <?= $periodo_num ?></h6>
                                                        </div>
                                                        <div class="card-body">
                                                            <div class="row text-center">
                                                                <div class="col-md-2 mb-2">
                                                                    <small class="text-muted d-block">Total Materias</small>
                                                                    <div class="fw-bold fs-5"><?= $total_asignaturas_periodo ?></div>
                                                                </div>
                                                                <div class="col-md-2 mb-2">
                                                                    <small class="text-muted d-block">Aprobadas</small>
                                                                    <div class="fw-bold fs-5 text-success"><?= $asignaturas_aprobadas_periodo ?></div>
                                                                </div>
                                                                <div class="col-md-2 mb-2">
                                                                    <small class="text-muted d-block">Reprobadas</small>
                                                                    <div class="fw-bold fs-5 text-danger"><?= $asignaturas_reprobadas_periodo ?></div>
                                                                </div>
                                                                <div class="col-md-2 mb-2">
                                                                    <small class="text-muted d-block">Sin Calificar</small>
                                                                    <div class="fw-bold fs-5 text-secondary"><?= $asignaturas_sin_nota_periodo ?></div>
                                                                </div>
                                                                <div class="col-md-2 mb-2">
                                                                    <small class="text-muted d-block">Promedio Simple</small>
                                                                    <div class="fw-bold fs-5 <?= $promedio_simple_periodo >= 15 ? 'text-success' : ($promedio_simple_periodo >= 10 ? 'text-warning' : 'text-danger') ?>">
                                                                        <?= number_format($promedio_simple_periodo, 2) ?>
                                                                    </div>
                                                                </div>
                                                                <div class="col-md-2 mb-2">
                                                                    <small class="text-muted d-block">Promedio Ponderado</small>
                                                                    <div class="fw-bold fs-5 <?= $promedio_ponderado_periodo >= 15 ? 'text-success' : ($promedio_ponderado_periodo >= 10 ? 'text-warning' : 'text-danger') ?>">
                                                                        <?= number_format($promedio_ponderado_periodo, 2) ?>
                                                                    </div>
                                                                </div>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>

                                            <!-- Tabla de materias -->
                                            <div class="table-responsive">
                                                <table class="table table-striped table-hover table-bordered">
                                                    <thead class="table-dark">
                                                        <tr>
                                                            <th width="50%">Materia</th>
                                                            <th width="15%" class="text-center">Año Lectivo</th>
                                                            <th width="15%" class="text-center">Créditos</th>
                                                            <th width="20%" class="text-center">Calificación</th>
                                                        </tr>
                                                    </thead>
                                                    <tbody>
                                                        <?php 
                                                        foreach ($notas_del_periodo as $nota): 
                                                            $nota_valor = $nota['nota_definitiva'] ?? '';
                                                            $creditos = intval($nota['credito'] ?? 0);
                                                            $es_numerica = is_numeric($nota_valor);
                                                            $clase_nota = '';
                                                            $aprobada = false;
                                                            $reprobada = false;
                                                            $tiene_nota = !empty($nota_valor);
                                                            
                                                            if ($tiene_nota) {
                                                                if ($nota['tipo_evaluacion'] == 'Aprobado_Reprobado') {
                                                                    $aprobada = (strtoupper($nota_valor) == 'APROBADO');
                                                                    $reprobada = !$aprobada;
                                                                    $clase_nota = $aprobada ? 'text-success' : 'text-danger';
                                                                } elseif ($es_numerica) {
                                                                    $nota_num = floatval($nota_valor);
                                                                    $aprobada = ($nota_num >= 10);
                                                                    $reprobada = ($nota_num < 10);
                                                                    if ($aprobada) {
                                                                        $clase_nota = 'text-success';
                                                                    } elseif ($nota_num >= 5) {
                                                                        $clase_nota = 'text-warning';
                                                                    } else {
                                                                        $clase_nota = 'text-danger';
                                                                    }
                                                                }
                                                            } else {
                                                                $clase_nota = 'text-secondary';
                                                            }
                                                            
                                                            // Determinar texto a mostrar
                                                            $texto_nota = 'N/A';
                                                            if ($tiene_nota) {
                                                                if ($nota['tipo_evaluacion'] == 'Aprobado_Reprobado') {
                                                                    $texto_nota = htmlspecialchars($nota_valor);
                                                                } else {
                                                                    $texto_nota = htmlspecialchars($nota_valor);
                                                                }
                                                            }
                                                        ?>
                                                            <tr>
                                                                <td>
                                                                    <strong><?= htmlspecialchars($nota['materia'] ?? '') ?></strong>
                                                                </td>
                                                                <td class="text-center">
                                                                    <?= !empty($nota['anio_lectivo']) ? htmlspecialchars($nota['anio_lectivo']) : '<span class="text-muted">N/A</span>' ?>
                                                                </td>
                                                                <td class="text-center">
                                                                    <?= $creditos > 0 ? $creditos : '<span class="text-muted">0</span>' ?>
                                                                </td>
                                                                <td class="text-center">
                                                                    <span class="badge <?= $clase_nota ?> bg-light border <?= $clase_nota ?> fs-6">
                                                                        <?= $texto_nota ?>
                                                                    </span>
                                                                </td>
                                                            </tr>
                                                        <?php endforeach; ?>
                                                    </tbody>
                                                </table>
                                            </div>
                                        </div>
                                        <?php $periodo_index++; ?>
                                    <?php endforeach; ?>
                                </div>
                            <?php endif; ?>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<?php
include __DIR__ . '/../includes/footer.php';
?>