<?php
// Centralized authentication and session check
$allowed_roles = ['coordinador2'];
require_once __DIR__ . '/../includes/auth_check.php';

if ($conn->connect_error) {
    die("Error de conexión: " . $conn->connect_error);
}

mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

// CSRF Token
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

$coordinador_id = $_SESSION['user_id'];

// Obtener postgrados asignados al coordinador2
$sql_postgrados = "SELECT DISTINCT p.id, p.nombre 
                  FROM postgrados p
                  JOIN coordinador_postgrado cp ON p.id = cp.postgrado_id
                  WHERE cp.coordinador_id = ?
                  ORDER BY p.nombre";
$stmt_postgrados = $conn->prepare($sql_postgrados);
$stmt_postgrados->bind_param("i", $coordinador_id);
$stmt_postgrados->execute();
$postgrados = $stmt_postgrados->get_result()->fetch_all(MYSQLI_ASSOC);

if (empty($postgrados)) {
    die("No tienes postgrados asignados. Contacta al administrador.");
}

// Obtener el postgrado seleccionado
$postgrado_seleccionado = isset($_GET['postgrado_id']) ? intval($_GET['postgrado_id']) : $postgrados[0]['id'];

// Verificar permiso para el postgrado seleccionado
$postgrado_permitido = false;
foreach ($postgrados as $p) {
    if ($p['id'] == $postgrado_seleccionado) {
        $postgrado_permitido = true;
        break;
    }
}

if (!$postgrado_permitido) {
    $postgrado_seleccionado = $postgrados[0]['id'];
    header("Location: cargar_notas.php?postgrado_id=".$postgrado_seleccionado);
    exit();
}

// Variables para filtros
$cedula = isset($_GET['cedula']) ? $conn->real_escape_string($_GET['cedula']) : '';
$anio_lectivo = isset($_GET['anio_lectivo']) ? $conn->real_escape_string($_GET['anio_lectivo']) : '';
$escolaridad = isset($_GET['escolaridad']) ? $conn->real_escape_string($_GET['escolaridad']) : '';

// Obtener años lectivos y escolaridades disponibles
$sql_filtros = "SELECT DISTINCT al.id as anio_lectivo_id, al.nombre as anio_lectivo, 
               esc.id as escolaridad_id, esc.nombre as escolaridad 
               FROM inscripciones i
               JOIN estudiante_postgrado ep ON i.estudiante_id = ep.estudiante_id
               JOIN años_lectivos al ON i.año_lectivo_id = al.id
               JOIN escolaridades esc ON i.escolaridad_id = esc.id
               WHERE ep.postgrado_id = ?
               ORDER BY al.nombre DESC, esc.nombre";
$stmt_filtros = $conn->prepare($sql_filtros);
$stmt_filtros->bind_param("i", $postgrado_seleccionado);
$stmt_filtros->execute();
$result_filtros = $stmt_filtros->get_result();

$anios_lectivos = [];
$escolaridades = [];
while ($filtro = $result_filtros->fetch_assoc()) {
    if (!in_array($filtro['anio_lectivo'], $anios_lectivos)) {
        $anios_lectivos[] = $filtro['anio_lectivo'];
    }
    if (!in_array($filtro['escolaridad'], $escolaridades)) {
        $escolaridades[] = $filtro['escolaridad'];
    }
}

// Obtener estudiantes con filtros
$sql_estudiantes = "SELECT DISTINCT e.id, e.cedula, e.nombres, e.apellidos, 
                   GROUP_CONCAT(DISTINCT al.nombre SEPARATOR ', ') as anio_lectivo, 
                   GROUP_CONCAT(DISTINCT esc.nombre SEPARATOR ', ') as escolaridad,
                   ep.estado as estado_postgrado
                   FROM estudiantes e
                   JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                   JOIN inscripciones i ON e.id = i.estudiante_id
                   JOIN años_lectivos al ON i.año_lectivo_id = al.id
                   JOIN escolaridades esc ON i.escolaridad_id = esc.id
                   WHERE ep.postgrado_id = ?";
$params = [$postgrado_seleccionado];
$types = "i";

if (!empty($cedula)) {
    $sql_estudiantes .= " AND e.cedula LIKE ?";
    $params[] = "%$cedula%";
    $types .= "s";
}
if (!empty($anio_lectivo)) {
    $sql_estudiantes .= " AND al.nombre = ?";
    $params[] = $anio_lectivo;
    $types .= "s";
}
if (!empty($escolaridad)) {
    $sql_estudiantes .= " AND esc.nombre = ?";
    $params[] = $escolaridad;
    $types .= "s";
}

$sql_estudiantes .= " GROUP BY e.id ORDER BY e.apellidos, e.nombres LIMIT 100";
$stmt_estudiantes = $conn->prepare($sql_estudiantes);

if (count($params) > 0) {
    $stmt_estudiantes->bind_param($types, ...$params);
}

$stmt_estudiantes->execute();
$estudiantes = $stmt_estudiantes->get_result()->fetch_all(MYSQLI_ASSOC);

$page_title = "Cargar Notas - Coordinador";
include '../includes/header.php';
?>

<div class="container">
    <div class="mb-4">
        <a href="index.php" class="btn btn-secondary">
            <i class="bi bi-arrow-left"></i> Volver al Panel
        </a>
    </div>

    <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success"><?= htmlspecialchars($_SESSION['success']) ?></div>
        <?php unset($_SESSION['success']); ?>
    <?php endif; ?>
    
    <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger"><?= htmlspecialchars($_SESSION['error']) ?></div>
        <?php unset($_SESSION['error']); ?>
    <?php endif; ?>

    <div class="card border-0 shadow-sm">
        <div class="card-body p-0">
            <h2 class="card-title p-3 mb-0 bg-light border-bottom">Cargar Notas de Estudiantes</h2>

            <div class="p-3">
                <form method="get" id="postgrado-form" class="mb-4">
                    <label for="postgrado_id" class="form-label">Seleccionar Postgrado:</label>
                    <select name="postgrado_id" id="postgrado_id" class="form-select" onchange="this.form.submit()">
                        <?php foreach ($postgrados as $p): ?>
                            <option value="<?= htmlspecialchars($p['id']) ?>" <?= $postgrado_seleccionado == $p['id'] ? 'selected' : '' ?>>
                                <?= htmlspecialchars($p['nombre']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </form>

                <form method="get" class="mb-4">
                    <input type="hidden" name="postgrado_id" value="<?= htmlspecialchars($postgrado_seleccionado) ?>">
                    <div class="row g-3">
                        <div class="col-md-4">
                            <label for="cedula" class="form-label">Cédula:</label>
                            <input type="text" id="cedula" name="cedula" class="form-control" 
                                   value="<?= htmlspecialchars($cedula) ?>" placeholder="Buscar por cédula">
                        </div>
                        
                        <?php if(!empty($anios_lectivos)): ?>
                        <div class="col-md-4">
                            <label for="anio_lectivo" class="form-label">Año Lectivo:</label>
                            <select id="anio_lectivo" name="anio_lectivo" class="form-select">
                                <option value="">Todos</option>
                                <?php foreach ($anios_lectivos as $al): ?>
                                    <option value="<?= htmlspecialchars($al) ?>" <?= $al == $anio_lectivo ? 'selected' : '' ?>>
                                        <?= htmlspecialchars($al) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <?php endif; ?>
                        
                        <?php if(!empty($escolaridades)): ?>
                        <div class="col-md-4">
                            <label for="escolaridad" class="form-label">Escolaridad:</label>
                            <select id="escolaridad" name="escolaridad" class="form-select">
                                <option value="">Todas</option>
                                <?php foreach ($escolaridades as $esc): ?>
                                    <option value="<?= htmlspecialchars($esc) ?>" <?= $esc == $escolaridad ? 'selected' : '' ?>>
                                        <?= htmlspecialchars($esc) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <?php endif; ?>
                        
                        <div class="col-md-12">
                            <button type="submit" class="btn btn-primary">Aplicar Filtros</button>
                            <a href="cargar_notas.php?postgrado_id=<?= htmlspecialchars($postgrado_seleccionado) ?>" 
                               class="btn btn-secondary">Limpiar</a>
                        </div>
                    </div>
                </form>
            </div>

            <div class="p-3 border-top">
                <h3 class="mb-3">Listado de Estudiantes</h3>
                <?php if (!empty($estudiantes)): ?>
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead class="table-light">
                                <tr>
                                    <th>Cédula</th>
                                    <th>Nombres</th>
                                    <th>Apellidos</th>
                                    <th>Años Lectivos</th>
                                    <th>Escolaridades</th>
                                    <th>Estado Postgrado</th>
                                    <th>Acciones</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($estudiantes as $est): ?>
                                    <tr>
                                        <td><?= htmlspecialchars($est['cedula']) ?></td>
                                        <td><?= htmlspecialchars($est['nombres']) ?></td>
                                        <td><?= htmlspecialchars($est['apellidos']) ?></td>
                                        <td><?= htmlspecialchars($est['anio_lectivo']) ?></td>
                                        <td><?= htmlspecialchars($est['escolaridad']) ?></td>
                                        <td>
                                            <span class="estado-<?= strtolower(htmlspecialchars($est['estado_postgrado'])) ?>">
                                                <?= htmlspecialchars(ucfirst($est['estado_postgrado'])) ?>
                                            </span>
                                        </td>
                                        <td>
                                            <a href="cargar_notas_estudiante.php?estudiante_id=<?= htmlspecialchars($est['id']) ?>&postgrado_id=<?= htmlspecialchars($postgrado_seleccionado) ?>" 
                                               class="btn btn-sm btn-primary">Cargar Notas</a>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <div class="alert alert-info">No se encontraron estudiantes para los filtros aplicados.</div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<style>
    .estado-activo { color: #28a745; font-weight: bold; }
    .estado-suspendido, .estado-reprobado { color: #dc3545; font-weight: bold; }
    .estado-graduado { color: #17a2b8; font-weight: bold; }
    .estado-retirado { color: #6c757d; font-weight: bold; }
    .estado-aprobado { color: #28a745; font-weight: bold; }
</style>

<?php
include '../includes/footer.php';
$conn->close();
?>