<?php
// 1. Configuración de sesión
require_once __DIR__ . '/../includes/session_config.php';

// 2. Verificación de auth
$allowed_roles = ['coordinador1'];
require_once __DIR__ . '/../includes/auth_check.php';

// 3. Configurar título de página
$page_title = "Cargar Notas - Estudiante";

// 4. Configurar encoding UTF-8
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');

// 5. Lógica específica de la página
include '../includes/db.php';

$conn->set_charset("utf8mb4");
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

$estudiante_id = isset($_GET['estudiante_id']) ? intval($_GET['estudiante_id']) : 0;
$postgrado_seleccionado = isset($_GET['postgrado_id']) ? intval($_GET['postgrado_id']) : 0;
$coordinador_id = $_SESSION['user_id'];

// VERIFICAR PERMISOS DEL COORDINADOR SOBRE EL POSTGRADO
$sql_permiso = "SELECT COUNT(*) as count FROM coordinador_postgrado WHERE coordinador_id = ? AND postgrado_id = ?";
$stmt_permiso = $conn->prepare($sql_permiso);
$stmt_permiso->bind_param("ii", $coordinador_id, $postgrado_seleccionado);
$stmt_permiso->execute();
if ($stmt_permiso->get_result()->fetch_assoc()['count'] == 0) {
    $_SESSION['error'] = "No tiene permisos para gestionar este postgrado.";
    header("Location: cargar_notas.php");
    exit();
}

// Obtener información del estudiante
$sql_estudiante_info = "SELECT e.*, ep.estado as estado_postgrado, p.nombre as postgrado_nombre
                       FROM estudiantes e
                       JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                       JOIN postgrados p ON ep.postgrado_id = p.id
                       WHERE e.id = ? AND ep.postgrado_id = ?";
$stmt_estudiante_info = $conn->prepare($sql_estudiante_info);
$stmt_estudiante_info->bind_param("ii", $estudiante_id, $postgrado_seleccionado);
$stmt_estudiante_info->execute();
$estudiante_info = $stmt_estudiante_info->get_result()->fetch_assoc();

if (!$estudiante_info) {
    $_SESSION['error'] = "No se encontró al estudiante o no pertenece al postgrado seleccionado.";
    header("Location: cargar_notas.php?postgrado_id=".$postgrado_seleccionado);
    exit();
}

// FUNCIÓN PARA OBTENER EL NÚMERO DE DOCUMENTO SEGÚN EL TIPO
function obtener_numero_documento($tipo_documento, $cedula_venezolana, $cedula_extranjera, $pasaporte) {
    switch ($tipo_documento) {
        case 'V': return $cedula_venezolana;
        case 'E': return $cedula_extranjera;
        case 'P': return $pasaporte;
        default: return $cedula_venezolana;
    }
}

// FUNCIÓN PARA OBTENER EL TIPO DE DOCUMENTO EN TEXTO
function obtener_tipo_documento_texto($tipo_documento) {
    switch ($tipo_documento) {
        case 'V': return 'Cédula Venezolana';
        case 'E': return 'Cédula Extranjera';
        case 'P': return 'Pasaporte';
        default: return 'Cédula Venezolana';
    }
}

// Obtener el número de documento del estudiante
$numero_documento = obtener_numero_documento(
    $estudiante_info['tipo_documento'], 
    $estudiante_info['cedula_venezolana'], 
    $estudiante_info['cedula_extranjera'], 
    $estudiante_info['pasaporte']
);
$tipo_documento_texto = obtener_tipo_documento_texto($estudiante_info['tipo_documento']);

// Verificar si existe la columna veces_modificada
$sql_check_columns = "SHOW COLUMNS FROM inscripciones LIKE 'veces_modificada'";
$result = $conn->query($sql_check_columns);
$has_veces_modificada = ($result->num_rows > 0);

// Obtener denominaciones y notas con información de usuario que modificó
$sql_denominaciones = "SELECT d.id, d.nombre, d.codigo, d.tipo, d.periodo, d.credito,
                      i.id as id_inscripcion, i.nota_definitiva, i.estado,
                      i.fecha_creacion, i.fecha_modificacion,
                      al.nombre as anio_lectivo_nombre,
                      esc.nombre as escolaridad_nombre,
                      u_mod.nombres as modificado_por_nombres,
                      u_mod.apellidos as modificado_por_apellidos,
                      u_mod.username as modificado_por_username";
                      
if ($has_veces_modificada) {
    $sql_denominaciones .= ", i.veces_modificada";
} else {
    $sql_denominaciones .= ", i.veces_modificado as veces_modificada";
}

$sql_denominaciones .= " FROM denominaciones d
                      JOIN inscripciones i ON d.id = i.denominacion_id
                      JOIN anios_lectivos al ON i.año_lectivo_id = al.id
                      JOIN escolaridades esc ON i.escolaridad_id = esc.id
                      LEFT JOIN usuarios u_mod ON i.usuario_modificacion = u_mod.id
                      WHERE i.estudiante_id = ? AND d.postgrado_id = ?
                      ORDER BY d.periodo, d.codigo";

$stmt_denominaciones = $conn->prepare($sql_denominaciones);
$stmt_denominaciones->bind_param("ii", $estudiante_id, $postgrado_seleccionado);
$stmt_denominaciones->execute();
$denominaciones = $stmt_denominaciones->get_result()->fetch_all(MYSQLI_ASSOC);

$periodos = [];
$totales = [
    'total_materias' => 0, 
    'total_materias_puntaje' => 0, 
    'total_materias_cualitativas' => 0,
    'aprobadas_puntaje' => 0,
    'aprobadas_cualitativas' => 0, 
    'sin_nota_puntaje' => 0, 
    'sin_nota_cualitativas' => 0,
    'reprobadas_puntaje' => 0, 
    'reprobadas_cualitativas' => 0,
    'promedio_simple' => 0,
    'promedio_ponderado' => 0,
    'suma_notas_simples' => 0,
    'suma_notas_ponderadas' => 0,
    'suma_creditos_aprobados' => 0
];

// Array para almacenar promedios por periodo
$promedios_por_periodo = array();

foreach ($denominaciones as $denominacion) {
    $periodo = $denominacion['periodo'];
    if (!isset($periodos[$periodo])) {
        $periodos[$periodo] = array();
    }
    $periodos[$periodo][] = $denominacion;
    
    // Inicializar promedios por periodo si no existen
    if (!isset($promedios_por_periodo[$periodo])) {
        $promedios_por_periodo[$periodo] = array(
            'suma_notas_simples' => 0,
            'suma_notas_ponderadas' => 0,
            'suma_creditos_aprobados' => 0,
            'aprobadas_puntaje' => 0,
            'sin_nota_puntaje' => 0,
            'reprobadas_puntaje' => 0,
            'promedio_simple' => 0,
            'promedio_ponderado' => 0
        );
    }
    
    $totales['total_materias']++;
    
    if ($denominacion['tipo'] == 'puntaje') {
        $totales['total_materias_puntaje']++;
        $credito = intval($denominacion['credito']);
        
        if (empty($denominacion['nota_definitiva'])) {
            $totales['sin_nota_puntaje']++;
            $promedios_por_periodo[$periodo]['sin_nota_puntaje']++;
        } else {
            $nota = floatval($denominacion['nota_definitiva']);
            if ($nota >= 10) {
                $totales['suma_notas_simples'] += $nota;
                $totales['suma_notas_ponderadas'] += ($nota * $credito);
                $totales['suma_creditos_aprobados'] += $credito;
                $totales['aprobadas_puntaje']++;
                
                // Acumular para promedios por periodo
                $promedios_por_periodo[$periodo]['suma_notas_simples'] += $nota;
                $promedios_por_periodo[$periodo]['suma_notas_ponderadas'] += ($nota * $credito);
                $promedios_por_periodo[$periodo]['suma_creditos_aprobados'] += $credito;
                $promedios_por_periodo[$periodo]['aprobadas_puntaje']++;
            } else {
                $totales['reprobadas_puntaje']++;
                $promedios_por_periodo[$periodo]['reprobadas_puntaje']++;
            }
        }
    } else {
        $totales['total_materias_cualitativas']++;
        
        if (empty($denominacion['nota_definitiva'])) {
            $totales['sin_nota_cualitativas']++;
        } else {
            $nota_cualitativa = trim($denominacion['nota_definitiva']);
            if (strtoupper($nota_cualitativa) == 'APROBADO') {
                $totales['aprobadas_cualitativas']++;
            } else {
                $totales['reprobadas_cualitativas']++;
            }
        }
    }
}

// Calcular promedios generales (solo para materias de puntaje)
if ($totales['aprobadas_puntaje'] > 0) {
    $totales['promedio_simple'] = $totales['suma_notas_simples'] / $totales['aprobadas_puntaje'];
}
if ($totales['suma_creditos_aprobados'] > 0) {
    $totales['promedio_ponderado'] = $totales['suma_notas_ponderadas'] / $totales['suma_creditos_aprobados'];
}

// Calcular promedios por periodo (solo para materias de puntaje)
foreach ($promedios_por_periodo as $periodo => $datos) {
    if ($datos['aprobadas_puntaje'] > 0) {
        $promedios_por_periodo[$periodo]['promedio_simple'] = $datos['suma_notas_simples'] / $datos['aprobadas_puntaje'];
        if ($datos['suma_creditos_aprobados'] > 0) {
            $promedios_por_periodo[$periodo]['promedio_ponderado'] = $datos['suma_notas_ponderadas'] / $datos['suma_creditos_aprobados'];
        }
    }
}

// LÓGICA DE GUARDADO DE NOTAS CON RESTRICCIONES CORREGIDAS
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['guardar_notas'])) {
    // Validar token CSRF
    if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        $_SESSION['error'] = "Token de seguridad inválido.";
        header("Location: " . $_SERVER['PHP_SELF'] . "?" . http_build_query($_GET));
        exit();
    }

    $conn->begin_transaction();
    try {
        $periodo_activo = $_POST['periodo_activo'];
        $notas_guardadas = 0;
        $notas_bloqueadas = 0;
        $notas_sin_cambio = 0;

        foreach ($_POST['denominaciones'] as $inscripcion_id => $datos) {
            $inscripcion_id = intval($inscripcion_id);
            
            // Obtener estado actual de la nota desde la BD para seguridad
            $sql_nota_actual = "SELECT i.nota_definitiva, i.fecha_creacion, d.periodo, d.tipo";
            if ($has_veces_modificada) {
                $sql_nota_actual .= ", i.veces_modificada";
            } else {
                $sql_nota_actual .= ", i.veces_modificado as veces_modificada";
            }
            $sql_nota_actual .= " FROM inscripciones i
                                JOIN denominaciones d ON i.denominacion_id = d.id
                                WHERE i.id = ? AND i.estudiante_id = ?";
            
            $stmt_nota_actual = $conn->prepare($sql_nota_actual);
            $stmt_nota_actual->bind_param("ii", $inscripcion_id, $estudiante_id);
            $stmt_nota_actual->execute();
            $nota_db = $stmt_nota_actual->get_result()->fetch_assoc();

            if (!$nota_db || $nota_db['periodo'] != $periodo_activo) {
                continue; // Ignorar si no pertenece al periodo activo o no existe
            }

            $nota_nueva_str = trim($datos['nota']);
            $nota_actual_str = $nota_db['nota_definitiva'] ? trim($nota_db['nota_definitiva']) : '';

            // Detectar si hay un cambio real
            $hay_cambio = false;
            if ($nota_db['tipo'] == 'puntaje') {
                $nota_nueva_float = $nota_nueva_str === '' ? null : floatval($nota_nueva_str);
                $nota_actual_float = $nota_actual_str === '' ? null : floatval($nota_actual_str);
                
                if ($nota_nueva_float === null && $nota_actual_float !== null) {
                    $hay_cambio = true;
                } elseif ($nota_nueva_float !== null && $nota_actual_float === null) {
                    $hay_cambio = true;
                } elseif ($nota_nueva_float !== null && $nota_actual_float !== null) {
                    if (abs($nota_nueva_float - $nota_actual_float) > 0.001) {
                        $hay_cambio = true;
                    }
                }
            } else {
                if ($nota_nueva_str !== $nota_actual_str) {
                    $hay_cambio = true;
                }
            }

            if (!$hay_cambio) {
                $notas_sin_cambio++;
                continue;
            }

            // LÓGICA DE RESTRICCIÓN CORREGIDA - PERMITE HASTA 2 MODIFICACIONES
            $es_modificacion = !is_null($nota_db['fecha_creacion']);
            if ($es_modificacion && $has_veces_modificada) {
                $fecha_creacion = new DateTime($nota_db['fecha_creacion']);
                $un_mes_despues = $fecha_creacion->add(new DateInterval('P1M'));
                $ahora = new DateTime();

                // CORRECCIÓN: Permitir hasta 2 modificaciones (>= 2 bloquea)
                if ($nota_db['veces_modificada'] >= 2) {
                    $notas_bloqueadas++;
                    continue; // Ya se modificó 2 veces, no se puede cambiar más
                }
                if ($ahora > $un_mes_despues) {
                    $notas_bloqueadas++;
                    continue; // Pasó más de un mes, no se puede cambiar
                }
            }

            // Si pasa la validación, proceder a guardar
            $fecha_actual = date('Y-m-d H:i:s');
            $usuario_id = $_SESSION['user_id'];

            if ($nota_db['tipo'] == 'puntaje') {
                $nota_nueva = $nota_nueva_str === '' ? null : floatval($nota_nueva_str);
                if ($nota_nueva !== null && ($nota_nueva < 0 || $nota_nueva > 20)) {
                    throw new Exception("Las notas deben estar entre 0 y 20.");
                }
                $estado_inscripcion = ($nota_nueva !== null && $nota_nueva >= 10) ? 'finalizada' : 'activa';
                
                // Construir query dinámicamente basado en columnas disponibles
                $sql_update = "UPDATE inscripciones SET nota_definitiva = ?, estado = ?, fecha_modificacion = ?, usuario_modificacion = ?";
                if ($has_veces_modificada) {
                    $sql_update .= ", veces_modificada = veces_modificada + 1";
                } else {
                    $sql_update .= ", veces_modificado = veces_modificado + 1";
                }
                $sql_update .= " WHERE id = ?";
                
                $stmt_update = $conn->prepare($sql_update);
                $stmt_update->bind_param("dssii", $nota_nueva, $estado_inscripcion, $fecha_actual, $usuario_id, $inscripcion_id);
            } else { // Cualitativa
                $nota_nueva = $nota_nueva_str === '' ? null : $nota_nueva_str;
                if ($nota_nueva !== null && !in_array($nota_nueva, ['Aprobado', 'Reprobado'])) {
                    throw new Exception("Valor no permitido para nota cualitativa.");
                }
                $estado_inscripcion = ($nota_nueva === 'Aprobado') ? 'finalizada' : 'activa';

                $sql_update = "UPDATE inscripciones SET nota_definitiva = ?, estado = ?, fecha_modificacion = ?, usuario_modificacion = ?";
                if ($has_veces_modificada) {
                    $sql_update .= ", veces_modificada = veces_modificada + 1";
                } else {
                    $sql_update .= ", veces_modificado = veces_modificado + 1";
                }
                $sql_update .= " WHERE id = ?";
                
                $stmt_update = $conn->prepare($sql_update);
                $stmt_update->bind_param("sssii", $nota_nueva, $estado_inscripcion, $fecha_actual, $usuario_id, $inscripcion_id);
            }
            
            $stmt_update->execute();
            $notas_guardadas++;
        }

        $conn->commit();

        $mensaje = "Proceso finalizado. ";
        if ($notas_guardadas > 0) $mensaje .= "$notas_guardadas nota(s) guardada(s) correctamente. ";
        if ($notas_bloqueadas > 0) $mensaje .= "$notas_bloqueadas nota(s) no se pudieron modificar por estar bloqueadas. ";
        if ($notas_sin_cambio > 0 && $notas_guardadas == 0 && $notas_bloqueadas == 0) $mensaje = "No se detectaron cambios para guardar.";
        
        $_SESSION['success'] = $mensaje;

    } catch (Exception $e) {
        $conn->rollback();
        $_SESSION['error'] = "Error al actualizar: " . $e->getMessage();
    }
    header("Location: " . $_SERVER['PHP_SELF'] . "?" . http_build_query($_GET));
    exit();
}

// Generar token CSRF
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// 6. Incluir header
include '../includes/header.php';
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        .estado-activo { color: #28a745; font-weight: bold; }
        .estado-suspendido, .estado-reprobado, .promedio-bajo { color: #dc3545; font-weight: bold; }
        .estado-graduado, .estado-finalizada { color: #17a2b8; font-weight: bold; }
        .estado-retirado { color: #6c757d; font-weight: bold; }
        .estado-aprobado, .promedio-alto { color: #28a745; font-weight: bold; }
        .estado-pendiente { color: #ffc107; font-weight: bold; }
        
        .tab-content {
            display: none;
        }
        .tab-content.show {
            display: block;
        }
        .tab-button {
            margin-right: 5px;
            margin-bottom: 5px;
        }
        .tab-button.active {
            background-color: #0d6efd;
            color: white;
        }
        .nota-input {
            width: 90px;
            text-align: center;
            font-weight: bold;
        }
        .tabs-container {
            gap: 5px;
        }
        .badge {
            font-size: 0.75rem;
        }
        .card {
            box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
            border: 1px solid rgba(0, 0, 0, 0.125);
        }
        .table th {
            background-color: #f8f9fa;
            border-bottom: 2px solid #dee2e6;
        }
        .resumen-card {
            height: 100%;
            transition: transform 0.2s;
        }
        .resumen-card:hover {
            transform: translateY(-2px);
        }
        .sin-nota {
            background-color: #fff3cd !important;
        }
        .anio-lectivo-badge {
            font-size: 0.7rem;
        }
        .resumen-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            margin-bottom: 1rem;
        }
        .resumen-item {
            background: #f8f9fa;
            border-radius: 0.375rem;
            padding: 1rem;
            text-align: center;
            border-left: 4px solid #0d6efd;
        }
        .resumen-valor {
            font-size: 1.5rem;
            font-weight: bold;
            margin-bottom: 0.5rem;
        }
        .resumen-label {
            font-size: 0.875rem;
            color: #6c757d;
        }
        .table-responsive {
            margin-bottom: 1rem;
        }
        .select-cualitativo {
            width: 120px;
        }
        .form-control:focus {
            border-color: #86b7fe;
            box-shadow: 0 0 0 0.25rem rgba(13, 110, 253, 0.25);
        }
        .is-invalid {
            border-color: #dc3545;
            box-shadow: 0 0 0 0.25rem rgba(220, 53, 69, 0.25);
        }
        .changed {
            background-color: #e7f3ff !important;
            border: 2px solid #0d6efd !important;
        }
        .nota-bloqueada {
            background-color: #f8d7da !important;
            color: #721c24 !important;
        }
        .nota-modificable {
            background-color: #d1ecf1 !important;
            color: #0c5460 !important;
        }
        .modificaciones-info {
            font-size: 0.75rem;
            margin-top: 2px;
        }
    </style>
</head>
<body>
    <div class="container-fluid px-3 px-md-4 mt-4">
        <div class="mb-3 mb-md-4">
            <a href="cargar_notas.php?postgrado_id=<?php echo $postgrado_seleccionado; ?>" class="btn btn-secondary btn-sm">
                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-arrow-left" viewBox="0 0 16 16">
                    <path fill-rule="evenodd" d="M15 8a.5.5 0 0 0-.5-.5H2.707l3.147-3.146a.5.5 0 1 0-.708-.708l-4 4a.5.5 0 0 0 0 .708l4 4a.5.5 0 0 0 .708-.708L2.707 8.5H14.5A.5.5 0 0 0 15 8z"/>
                </svg>
                Volver al Listado
            </a>
        </div>

        <?php if (isset($_SESSION['success'])): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <?= $_SESSION['success'] ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
            <?php unset($_SESSION['success']); ?>
        <?php endif; ?>
        
        <?php if (isset($_SESSION['error'])): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <?= $_SESSION['error'] ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
            <?php unset($_SESSION['error']); ?>
        <?php endif; ?>

        <div class="card border-0 shadow-sm">
            <div class="card-body p-0">
                <div class="p-3 bg-light border-bottom">
                    <h3 class="mb-0">Notas para: <?php echo htmlspecialchars($estudiante_info['nombres'] . ' ' . $estudiante_info['apellidos']); ?></h3>
                    <p class="mb-0">
                        <strong><?php echo htmlspecialchars($tipo_documento_texto); ?>:</strong> <?php echo htmlspecialchars($numero_documento); ?>
                    </p>
                    <p class="mb-0">Postgrado: <strong><?php echo htmlspecialchars($estudiante_info['postgrado_nombre']); ?></strong></p>
                    <p class="mb-0">Estado del Postgrado: 
                        <span class="estado-<?php echo strtolower(htmlspecialchars($estudiante_info['estado_postgrado'])); ?>">
                            <?php echo htmlspecialchars(ucfirst($estudiante_info['estado_postgrado'])); ?>
                        </span>
                    </p>
                </div>

                <?php if (!empty($denominaciones)): ?>
                    <div class="p-3 border-bottom">
                        <h4 class="mb-3">Resumen Académico</h4>
                        
                        <div class="resumen-grid">
                            <div class="resumen-item">
                                <div class="resumen-valor"><?php echo $totales['total_materias']; ?></div>
                                <div class="resumen-label">Total Asignaturas</div>
                            </div>
                            <div class="resumen-item">
                                <div class="resumen-valor text-warning"><?php echo $totales['sin_nota_puntaje'] + $totales['sin_nota_cualitativas']; ?></div>
                                <div class="resumen-label">Sin Nota</div>
                                <small class="text-muted">
                                    P: <?php echo $totales['sin_nota_puntaje']; ?> | 
                                    C: <?php echo $totales['sin_nota_cualitativas']; ?>
                                </small>
                            </div>
                            <div class="resumen-item">
                                <div class="resumen-valor text-success"><?php echo $totales['aprobadas_puntaje'] + $totales['aprobadas_cualitativas']; ?></div>
                                <div class="resumen-label">Aprobadas</div>
                                <small class="text-muted">
                                    P: <?php echo $totales['aprobadas_puntaje']; ?> | 
                                    C: <?php echo $totales['aprobadas_cualitativas']; ?>
                                </small>
                            </div>
                            <div class="resumen-item">
                                <div class="resumen-valor text-danger"><?php echo $totales['reprobadas_puntaje'] + $totales['reprobadas_cualitativas']; ?></div>
                                <div class="resumen-label">Reprobadas</div>
                                <small class="text-muted">
                                    P: <?php echo $totales['reprobadas_puntaje']; ?> | 
                                    C: <?php echo $totales['reprobadas_cualitativas']; ?>
                                </small>
                            </div>
                            <div class="resumen-item">
                                <div class="resumen-valor <?php echo ($totales['promedio_simple'] < 15 && $totales['promedio_simple'] > 0) ? 'text-danger' : 'text-primary'; ?>">
                                    <?php echo number_format($totales['promedio_simple'], 2); ?>
                                </div>
                                <div class="resumen-label">Promedio Simple</div>
                            </div>
                            <div class="resumen-item">
                                <div class="resumen-valor <?php echo ($totales['promedio_ponderado'] < 15 && $totales['promedio_ponderado'] > 0) ? 'text-danger' : 'text-primary'; ?>">
                                    <?php echo number_format($totales['promedio_ponderado'], 2); ?>
                                </div>
                                <div class="resumen-label">Promedio Ponderado</div>
                            </div>
                            <div class="resumen-item">
                                <div class="resumen-valor text-info"><?php echo $totales['suma_creditos_aprobados']; ?></div>
                                <div class="resumen-label">Créditos Aprobados</div>
                            </div>
                        </div>

                        <?php if (!$has_veces_modificada): ?>
                            <div class="alert alert-warning mt-3">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-exclamation-triangle me-2" viewBox="0 0 16 16">
                                    <path d="M7.938 2.016A.13.13 0 0 1 8.002 2a.13.13 0 0 1 .063.016.146.146 0 0 1 .054.057l6.857 11.667c.036.06.035.124.002.183a.163.163 0 0 1-.054.06.116.116 0 0 1-.066.017H1.146a.115.115 0 0 1-.066-.017.163.163 0 0 1-.054-.06.176.176 0 0 1 .002-.183L7.884 2.073a.147.147 0 0 1 .054-.057zm1.044-.45a1.13 1.13 0 0 0-1.96 0L.165 13.233c-.457.778.091 1.767.98 1.767h13.713c.889 0 1.438-.99.98-1.767L8.982 1.566z"/>
                                    <path d="M7.002 12a1 1 0 1 1 2 0 1 1 0 0 1-2 0zM7.1 5.995a.905.905 0 1 1 1.8 0l-.35 3.507a.552.552 0 0 1-1.1 0L7.1 5.995z"/>
                                </svg>
                                <strong>Nota:</strong> El sistema de restricciones de modificación no está completamente configurado. 
                                Contacte al administrador.
                            </div>
                        <?php else: ?>
                            <div class="alert alert-info mt-3">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-info-circle me-2" viewBox="0 0 16 16">
                                    <path d="M8 15A7 7 0 1 1 8 1a7 7 0 0 1 0 14zm0 1A8 8 0 1 0 8 0a8 8 0 0 0 0 16z"/>
                                    <path d="m8.93 6.588-2.29.287-.082.38.45.083c.294.07.352.176.288.469l-.738 3.468c-.194.897.105 1.319.808 1.319.545 0 1.178-.252 1.465-.598l.088-.416c-.2.176-.492.246-.686.246-.275 0-.375-.193-.304-.533L8.93 6.588zM9 4.5a1 1 0 1 1-2 0 1 1 0 0 1 2 0z"/>
                                </svg>
                                <strong>Información:</strong> Las notas pueden modificarse hasta 2 veces como máximo dentro del primer mes desde su creación.
                            </div>
                        <?php endif; ?>
                    </div>

                    <div class="p-3">
                        <form method="post" id="form-notas">
                            <input type="hidden" name="estudiante_id" value="<?php echo $estudiante_id; ?>">
                            <input type="hidden" name="postgrado_id" value="<?php echo $postgrado_seleccionado; ?>">
                            <input type="hidden" name="guardar_notas" value="1">
                            <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                            <input type="hidden" name="periodo_activo" id="periodo-activo" value="<?php echo isset(array_keys($periodos)[0]) ? array_keys($periodos)[0] : ''; ?>">

                            <div class="d-flex flex-wrap tabs-container mb-3" id="periodos-tabs">
                                <?php 
                                $periodos_keys = array_keys($periodos);
                                $tabs_per_row = 6;
                                $rows = ceil(count($periodos_keys) / $tabs_per_row);
                                
                                for ($row = 0; $row < $rows; $row++): ?>
                                    <div class="d-flex w-100 mb-2">
                                        <?php 
                                        $start = $row * $tabs_per_row;
                                        $end = min(($row + 1) * $tabs_per_row, count($periodos_keys));
                                        
                                        for ($i = $start; $i < $end; $i++): 
                                            $periodo_nombre = $periodos_keys[$i];
                                            $sin_nota_periodo = isset($promedios_por_periodo[$periodo_nombre]['sin_nota_puntaje']) ? $promedios_por_periodo[$periodo_nombre]['sin_nota_puntaje'] : 0;
                                        ?>
                                            <button type="button" class="btn btn-outline-primary tab-button position-relative <?php echo ($i === 0) ? 'active' : ''; ?>" 
                                                    data-target="tab-<?php echo $i; ?>" 
                                                    data-periodo="<?php echo htmlspecialchars($periodo_nombre); ?>">
                                                Periodo <?php echo htmlspecialchars($periodo_nombre); ?>
                                                <?php if ($sin_nota_periodo > 0): ?>
                                                    <span class="position-absolute top-0 start-100 translate-middle badge rounded-pill bg-warning">
                                                        <?php echo $sin_nota_periodo; ?>
                                                    </span>
                                                <?php endif; ?>
                                            </button>
                                        <?php endfor; ?>
                                    </div>
                                <?php endfor; ?>
                            </div>

                            <?php 
                            $periodo_index = 0;
                            foreach ($periodos as $periodo_nombre => $denom_list): 
                                $promedio_simple_periodo = isset($promedios_por_periodo[$periodo_nombre]['promedio_simple']) ? number_format($promedios_por_periodo[$periodo_nombre]['promedio_simple'], 2) : 'N/A';
                                $promedio_ponderado_periodo = isset($promedios_por_periodo[$periodo_nombre]['promedio_ponderado']) ? number_format($promedios_por_periodo[$periodo_nombre]['promedio_ponderado'], 2) : 'N/A';
                                $sin_nota_periodo = isset($promedios_por_periodo[$periodo_nombre]['sin_nota_puntaje']) ? $promedios_por_periodo[$periodo_nombre]['sin_nota_puntaje'] : 0;
                            ?>
                                <div id="tab-<?php echo $periodo_index; ?>" class="tab-content <?php echo ($periodo_index === 0) ? 'show active' : ''; ?>">
                                    <div class="d-flex justify-content-between align-items-center mb-3">
                                        <h5>Periodo: <?php echo htmlspecialchars($periodo_nombre); ?></h5>
                                        <div>
                                            <?php if ($sin_nota_periodo > 0): ?>
                                                <span class="badge bg-warning me-2">Sin nota: <?php echo $sin_nota_periodo; ?></span>
                                            <?php endif; ?>
                                            <span class="badge bg-primary me-2">Prom. Simple: <?php echo $promedio_simple_periodo; ?></span>
                                            <span class="badge bg-primary">Prom. Ponderado: <?php echo $promedio_ponderado_periodo; ?></span>
                                        </div>
                                    </div>
                                    
                                    <div class="table-responsive">
                                        <table class="table table-striped table-hover">
                                            <thead class="table-light">
                                                <tr>
                                                    <th>Código</th>
                                                    <th>Denominación</th>
                                                    <th>Crédito</th>
                                                    <th>Tipo</th>
                                                    <th>Escolaridad</th>
                                                    <th>Año Lectivo</th>
                                                    <th>Nota</th>
                                                    <th>Estado</th>
                                                    <th>Última Modificación</th>
                                                    <th>Modificado Por</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach ($denom_list as $denominacion): ?>
                                                    <?php 
                                                    // Lógica para determinar si un campo está bloqueado (solo si la columna existe)
                                                    $is_locked = false;
                                                    $lock_reason = '';
                                                    $modificaciones_restantes = 2;
                                                    $estado_modificacion = 'editabled';
                                                    
                                                    if (!is_null($denominacion['fecha_creacion']) && $has_veces_modificada) {
                                                        $fecha_creacion = new DateTime($denominacion['fecha_creacion']);
                                                        $un_mes_despues = $fecha_creacion->add(new DateInterval('P1M'));
                                                        $ahora = new DateTime();

                                                        // CORRECCIÓN: Permitir hasta 2 modificaciones
                                                        $modificaciones_restantes = 2 - $denominacion['veces_modificada'];
                                                        
                                                        if ($denominacion['veces_modificada'] >= 2) {
                                                            $is_locked = true;
                                                            $lock_reason = 'Esta nota ya fue modificada 2 veces y no puede volver a cambiarse.';
                                                            $estado_modificacion = 'bloqueada';
                                                        } elseif ($ahora > $un_mes_despues) {
                                                            $is_locked = true;
                                                            $lock_reason = 'Ha pasado más de un mes desde su creación. La nota está bloqueada.';
                                                            $estado_modificacion = 'bloqueada';
                                                        } else {
                                                            $estado_modificacion = 'modificable';
                                                        }
                                                    }
                                                    
                                                    $fecha_modificacion = !empty($denominacion['fecha_modificacion']) ? date('d/m/Y H:i', strtotime($denominacion['fecha_modificacion'])) : 'No modificada';
                                                    $modificado_por = !empty($denominacion['modificado_por_nombres']) ? 
                                                        htmlspecialchars($denominacion['modificado_por_nombres'] . ' ' . $denominacion['modificado_por_apellidos']) : 
                                                        (empty($denominacion['nota_definitiva']) ? 'N/A' : 'Sistema');
                                                    $sin_nota = empty($denominacion['nota_definitiva']);
                                                    $tipo_denominacion = ($denominacion['tipo'] == 'puntaje') ? 'Numérica' : 'Cualitativa';
                                                    ?>
                                                    <tr class="<?php echo $sin_nota ? 'sin-nota' : ''; ?> <?php echo $is_locked ? 'nota-bloqueada' : ($estado_modificacion == 'modificable' ? 'nota-modificable' : ''); ?>">
                                                        <td><?php echo htmlspecialchars($denominacion['codigo']); ?></td>
                                                        <td><?php echo htmlspecialchars($denominacion['nombre']); ?></td>
                                                        <td><?php echo htmlspecialchars($denominacion['credito']); ?></td>
                                                        <td>
                                                            <span class="badge <?php echo $denominacion['tipo'] == 'puntaje' ? 'bg-primary' : 'bg-info'; ?>">
                                                                <?php echo $tipo_denominacion; ?>
                                                            </span>
                                                        </td>
                                                        <td>
                                                            <span class="badge bg-secondary"><?php echo htmlspecialchars($denominacion['escolaridad_nombre']); ?></span>
                                                        </td>
                                                        <td>
                                                            <span class="badge bg-dark anio-lectivo-badge">
                                                                <?php echo htmlspecialchars($denominacion['anio_lectivo_nombre']); ?>
                                                            </span>
                                                        </td>
                                                        <td>
                                                            <?php if ($denominacion['tipo'] == 'puntaje'): ?>
                                                                <input type="number" step="0.01" min="0" max="20" 
                                                                       name="denominaciones[<?php echo $denominacion['id_inscripcion']; ?>][nota]" 
                                                                       class="form-control nota-input <?php echo $sin_nota ? 'border-warning' : ''; ?> <?php echo $is_locked ? 'nota-bloqueada' : ''; ?>" 
                                                                       value="<?php echo htmlspecialchars($denominacion['nota_definitiva'] ?? ''); ?>"
                                                                       placeholder="0-20"
                                                                       <?php if ($is_locked) echo 'readonly'; ?>>
                                                                <?php if ($has_veces_modificada && !empty($denominacion['nota_definitiva'])): ?>
                                                                    <div class="modificaciones-info <?php echo $is_locked ? 'text-danger' : 'text-info'; ?>">
                                                                        <?php if ($is_locked): ?>
                                                                            <svg xmlns="http://www.w3.org/2000/svg" width="10" height="10" fill="currentColor" class="bi bi-lock-fill me-1" viewBox="0 0 16 16">
                                                                                <path d="M8 1a2 2 0 0 1 2 2v4H6V3a2 2 0 0 1 2-2zm3 6V3a3 3 0 0 0-6 0v4a2 2 0 0 0-2 2v5a2 2 0 0 0 2 2h6a2 2 0 0 0 2-2V9a2 2 0 0 0-2-2z"/>
                                                                            </svg>
                                                                            Bloqueada
                                                                        <?php else: ?>
                                                                            <svg xmlns="http://www.w3.org/2000/svg" width="10" height="10" fill="currentColor" class="bi bi-pencil-square me-1" viewBox="0 0 16 16">
                                                                                <path d="M15.502 1.94a.5.5 0 0 1 0 .706L14.459 3.69l-2-2L13.502.646a.5.5 0 0 1 .707 0l1.293 1.293zm-1.75 2.456-2-2L4.939 9.21a.5.5 0 0 0-.121.196l-.805 2.414a.25.25 0 0 0 .316.316l2.414-.805a.5.5 0 0 0 .196-.12l6.813-6.814z"/>
                                                                                <path fill-rule="evenodd" d="M1 13.5A1.5 1.5 0 0 0 2.5 15h11a1.5 1.5 0 0 0 1.5-1.5v-6a.5.5 0 0 0-1 0v6a.5.5 0 0 1-.5.5h-11a.5.5 0 0 1-.5-.5v-11a.5.5 0 0 1 .5-.5H9a.5.5 0 0 0 0-1H2.5A1.5 1.5 0 0 0 1 2.5v11z"/>
                                                                            </svg>
                                                                            Modificaciones restantes: <?php echo $modificaciones_restantes; ?>
                                                                        <?php endif; ?>
                                                                    </div>
                                                                <?php endif; ?>
                                                            <?php else: ?>
                                                                <select name="denominaciones[<?php echo $denominacion['id_inscripcion']; ?>][nota]" class="form-control select-cualitativo <?php echo $sin_nota ? 'border-warning' : ''; ?> <?php echo $is_locked ? 'nota-bloqueada' : ''; ?>" <?php if ($is_locked) echo 'disabled'; ?>>
                                                                    <option value="">- Seleccionar -</option>
                                                                    <option value="Aprobado" <?php echo ($denominacion['nota_definitiva'] == 'Aprobado') ? 'selected' : ''; ?>>Aprobado</option>
                                                                    <option value="Reprobado" <?php echo ($denominacion['nota_definitiva'] == 'Reprobado') ? 'selected' : ''; ?>>Reprobado</option>
                                                                </select>
                                                                <?php if ($has_veces_modificada && !empty($denominacion['nota_definitiva'])): ?>
                                                                    <div class="modificaciones-info <?php echo $is_locked ? 'text-danger' : 'text-info'; ?>">
                                                                        <?php if ($is_locked): ?>
                                                                            <svg xmlns="http://www.w3.org/2000/svg" width="10" height="10" fill="currentColor" class="bi bi-lock-fill me-1" viewBox="0 0 16 16">
                                                                                <path d="M8 1a2 2 0 0 1 2 2v4H6V3a2 2 0 0 1 2-2zm3 6V3a3 3 0 0 0-6 0v4a2 2 0 0 0-2 2v5a2 2 0 0 0 2 2h6a2 2 0 0 0 2-2V9a2 2 0 0 0-2-2z"/>
                                                                            </svg>
                                                                            Bloqueada
                                                                        <?php else: ?>
                                                                            <svg xmlns="http://www.w3.org/2000/svg" width="10" height="10" fill="currentColor" class="bi bi-pencil-square me-1" viewBox="0 0 16 16">
                                                                                <path d="M15.502 1.94a.5.5 0 0 1 0 .706L14.459 3.69l-2-2L13.502.646a.5.5 0 0 1 .707 0l1.293 1.293zm-1.75 2.456-2-2L4.939 9.21a.5.5 0 0 0-.121.196l-.805 2.414a.25.25 0 0 0 .316.316l2.414-.805a.5.5 0 0 0 .196-.12l6.813-6.814z"/>
                                                                                <path fill-rule="evenodd" d="M1 13.5A1.5 1.5 0 0 0 2.5 15h11a1.5 1.5 0 0 0 1.5-1.5v-6a.5.5 0 0 0-1 0v6a.5.5 0 0 1-.5.5h-11a.5.5 0 0 1-.5-.5v-11a.5.5 0 0 1 .5-.5H9a.5.5 0 0 0 0-1H2.5A1.5 1.5 0 0 0 1 2.5v11z"/>
                                                                            </svg>
                                                                            Modificaciones restantes: <?php echo $modificaciones_restantes; ?>
                                                                        <?php endif; ?>
                                                                    </div>
                                                                <?php endif; ?>
                                                            <?php endif; ?>
                                                        </td>
                                                        <td>
                                                            <span class="estado-<?php echo strtolower($denominacion['estado']); ?>">
                                                                <?php echo htmlspecialchars(ucfirst($denominacion['estado'])); ?>
                                                            </span>
                                                        </td>
                                                        <td>
                                                            <small class="text-muted">
                                                                <?php echo $fecha_modificacion; ?>
                                                            </small>
                                                        </td>
                                                        <td>
                                                            <small class="text-muted">
                                                                <?php echo $modificado_por; ?>
                                                            </small>
                                                        </td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                                <?php $periodo_index++; ?>
                            <?php endforeach; ?>

                            <div class="mt-4">
                                <button type="submit" class="btn btn-success btn-lg">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-check-circle me-1" viewBox="0 0 16 16">
                                        <path d="M8 15A7 7 0 1 1 8 1a7 7 0 0 1 0 14zm0 1A8 8 0 1 0 8 0a8 8 0 0 0 0 16z"/>
                                        <path d="M10.97 4.97a.235.235 0 0 0-.02.022L7.477 9.417 5.384 7.323a.75.75 0 0 0-1.06 1.06L6.97 11.03a.75.75 0 0 0 1.079-.02l3.992-4.99a.75.75 0 0 0-1.071-1.05z"/>
                                    </svg>
                                    Guardar Cambios
                                </button>
                            </div>
                        </form>
                    </div>
                <?php else: ?>
                    <div class="p-4 text-center">
                        <div class="alert alert-info">
                            <h4>No hay asignaturas inscritas</h4>
                            <p>El estudiante no tiene asignaturas inscritas en este postgrado.</p>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Manejo de pestañas
            const tabButtons = document.querySelectorAll('.tab-button');
            const tabContents = document.querySelectorAll('.tab-content');
            const periodoActivoInput = document.getElementById('periodo-activo');

            tabButtons.forEach(button => {
                button.addEventListener('click', function() {
                    const target = this.getAttribute('data-target');
                    const periodo = this.getAttribute('data-periodo');
                    
                    // Actualizar pestaña activa
                    tabButtons.forEach(btn => btn.classList.remove('active'));
                    this.classList.add('active');
                    
                    // Mostrar contenido correspondiente
                    tabContents.forEach(content => {
                        content.classList.remove('show', 'active');
                    });
                    document.getElementById(target).classList.add('show', 'active');
                    
                    // Actualizar periodo activo en el formulario
                    periodoActivoInput.value = periodo;
                });
            });

            // Validación de notas (0-20) con decimales
            document.querySelectorAll('.nota-input').forEach(input => {
                input.addEventListener('input', function() {
                    if (this.readOnly) return;
                    
                    let value = parseFloat(this.value);
                    if (isNaN(value)) {
                        this.value = '';
                    } else if (value < 0) {
                        this.value = 0;
                    } else if (value > 20) {
                        this.value = 20;
                    }
                });
                
                input.addEventListener('blur', function() {
                    if (this.readOnly) return;
                    
                    if (this.value === '') return;
                    let value = parseFloat(this.value);
                    if (isNaN(value) || value < 0 || value > 20) {
                        this.classList.add('is-invalid');
                    } else {
                        this.classList.remove('is-invalid');
                    }
                });
            });

            // Validación del formulario al enviar
            document.getElementById('form-notas').addEventListener('submit', function(e) {
                let hasErrors = false;
                const currentTab = document.querySelector('.tab-content.show.active');
                const inputs = currentTab.querySelectorAll('.nota-input');
                
                inputs.forEach(input => {
                    if (!input.readOnly && input.value) {
                        const value = parseFloat(input.value);
                        if (isNaN(value) || value < 0 || value > 20) {
                            input.classList.add('is-invalid');
                            hasErrors = true;
                        } else {
                            input.classList.remove('is-invalid');
                        }
                    }
                });
                
                if (hasErrors) {
                    e.preventDefault();
                    alert('Por favor, corrija las notas que están fuera del rango permitido (0-20).');
                }
            });
        });
    </script>
</body>
</html>

<?php
// 7. Incluir footer
include '../includes/footer.php';
$conn->close();
?>