<?php
// Centralized authentication and session check
$allowed_roles = ['administrador'];
require_once __DIR__ . '/../includes/auth_check.php';

// DB Connection
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/document_helper.php';

// Agregar usuario
if (isset($_POST['agregar_usuario'])) {
    $username = trim($_POST['username']);
    $password = $_POST['password'];
    $rol = $_POST['rol'];
    $email = isset($_POST['email']) ? trim($_POST['email']) : null;
    $telefono = isset($_POST['telefono']) ? trim($_POST['telefono']) : null;
    $direccion = isset($_POST['direccion']) ? trim($_POST['direccion']) : null;
    $nombres = isset($_POST['nombres']) ? trim($_POST['nombres']) : null;
    $apellidos = isset($_POST['apellidos']) ? trim($_POST['apellidos']) : null;
    
    // NUEVO: Campos de documento unificados
    $tipo_documento = isset($_POST['tipo_documento']) ? trim($_POST['tipo_documento']) : 'V';
    $numero_documento = isset($_POST['numero_documento']) ? trim($_POST['numero_documento']) : null;
    
    $estudiante_id = null;

    if (empty($username) || empty($password)) {
        $_SESSION['error'] = "Usuario y contraseña son obligatorios";
    } else {
        $stmt = $conn->prepare("SELECT id FROM usuarios WHERE username = ?");
        $stmt->bind_param("s", $username);
        $stmt->execute();
        $stmt->store_result();
        
        if ($stmt->num_rows > 0) {
            $_SESSION['error'] = "El nombre de usuario ya existe";
        } else {
            // Si es estudiante, verificar que exista en la tabla estudiantes
            if ($rol === 'estudiante') {
                $cedula_estudiante = isset($_POST['cedula_estudiante']) ? trim($_POST['cedula_estudiante']) : '';
                
                if (empty($cedula_estudiante)) {
                    $_SESSION['error'] = "Para crear usuario de estudiante, debe verificar la cédula primero";
                    header("Location: gestion_usuarios.php");
                    exit();
                }
                
                // Verificar que el estudiante existe
                $stmt_estudiante = $conn->prepare("SELECT 
                    id, cedula, nombres, apellidos, correo, telefono, direccion,
                    tipo_documento, cedula_venezolana, cedula_extranjera, pasaporte
                    FROM estudiantes 
                    WHERE cedula = ? 
                    OR cedula_venezolana = ? 
                    OR cedula_extranjera = ? 
                    OR pasaporte = ?");
                $stmt_estudiante->bind_param("ssss", $cedula_estudiante, $cedula_estudiante, $cedula_estudiante, $cedula_estudiante);
                $stmt_estudiante->execute();
                $result_estudiante = $stmt_estudiante->get_result();
                
                if ($result_estudiante->num_rows === 0) {
                    $_SESSION['error'] = "No existe un estudiante con ese documento";
                    header("Location: gestion_usuarios.php");
                    exit();
                }
                
                $estudiante_data = $result_estudiante->fetch_assoc();
                $estudiante_id = $estudiante_data['id'];
                
                // Para estudiantes: usar datos de la tabla estudiantes
                $username = $estudiante_data['cedula'];
                $nombres = $estudiante_data['nombres'];
                $apellidos = $estudiante_data['apellidos'];
                $email = $estudiante_data['correo'];
                $telefono = $estudiante_data['telefono'];
                $direccion = $estudiante_data['direccion'];
                
                // CORRECCIÓN: COPIAR SOLO EL DOCUMENTO VERIFICADO
                $tipo_documento = $estudiante_data['tipo_documento'];
                $documento_verificado = $cedula_estudiante;

                // Determinar qué documento se usó para la verificación
                if ($estudiante_data['cedula_venezolana'] === $documento_verificado) {
                    $numero_documento = $estudiante_data['cedula_venezolana'];
                } elseif ($estudiante_data['cedula_extranjera'] === $documento_verificado) {
                    $numero_documento = $estudiante_data['cedula_extranjera'];
                } elseif ($estudiante_data['pasaporte'] === $documento_verificado) {
                    $numero_documento = $estudiante_data['pasaporte'];
                } else {
                    // Fallback: usar el documento principal según el tipo
                    switch($tipo_documento) {
                        case 'V': $numero_documento = $estudiante_data['cedula_venezolana']; break;
                        case 'E': $numero_documento = $estudiante_data['cedula_extranjera']; break;
                        case 'P': $numero_documento = $estudiante_data['pasaporte']; break;
                    }
                }
                
                $stmt_estudiante->close();
                
                // Verificar que no tenga ya un usuario
                $stmt_user_est = $conn->prepare("SELECT id FROM usuarios WHERE estudiante_id = ? OR username = ?");
                $stmt_user_est->bind_param("is", $estudiante_id, $username);
                $stmt_user_est->execute();
                $stmt_user_est->store_result();
                
                if ($stmt_user_est->num_rows > 0) {
                    $_SESSION['error'] = "Este estudiante ya tiene un usuario registrado";
                    header("Location: gestion_usuarios.php");
                    exit();
                }
                $stmt_user_est->close();
            } else {
                // PARA USUARIOS ADMINISTRATIVOS: Validar documento
                if (empty($numero_documento)) {
                    $_SESSION['error'] = "Para usuarios administrativos, debe ingresar un número de documento";
                    header("Location: gestion_usuarios.php");
                    exit();
                }
                
                // Validar que no exista el documento en otros usuarios
                $sql_check_docs = "SELECT id FROM usuarios WHERE 
                                  (cedula_venezolana = ? AND cedula_venezolana IS NOT NULL AND cedula_venezolana != '') OR
                                  (cedula_extranjera = ? AND cedula_extranjera IS NOT NULL AND cedula_extranjera != '') OR
                                  (pasaporte = ? AND pasaporte IS NOT NULL AND pasaporte != '')";
                $stmt_check_docs = $conn->prepare($sql_check_docs);
                $stmt_check_docs->bind_param("sss", $numero_documento, $numero_documento, $numero_documento);
                $stmt_check_docs->execute();
                $stmt_check_docs->store_result();
                
                if ($stmt_check_docs->num_rows > 0) {
                    $_SESSION['error'] = "Ya existe un usuario con el documento ingresado: " . $numero_documento;
                    $stmt_check_docs->close();
                    header("Location: gestion_usuarios.php");
                    exit();
                }
                $stmt_check_docs->close();
            }

            // Preparar campos de documentos para la inserción
            $cedula_venezolana = null;
            $cedula_extranjera = null;
            $pasaporte = null;

            switch($tipo_documento) {
                case 'V': $cedula_venezolana = $numero_documento; break;
                case 'E': $cedula_extranjera = $numero_documento; break;
                case 'P': $pasaporte = $numero_documento; break;
            }

            $hashed_password = password_hash($password, PASSWORD_DEFAULT);
            $estado_cuenta = 'activa';
            
            // Insertar en usuarios
            $stmt_insert = $conn->prepare("INSERT INTO usuarios 
                (username, password, rol, email, telefono, direccion, estado_cuenta, 
                 estudiante_id, nombres, apellidos, tipo_documento, cedula_venezolana, 
                 cedula_extranjera, pasaporte) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
            $stmt_insert->bind_param('sssssssissssss', 
                $username, $hashed_password, $rol, $email, $telefono, $direccion, 
                $estado_cuenta, $estudiante_id, $nombres, $apellidos, $tipo_documento,
                $cedula_venezolana, $cedula_extranjera, $pasaporte);
            
            if ($stmt_insert->execute()) {
                $_SESSION['success'] = "Usuario agregado correctamente";
            } else {
                $_SESSION['error'] = "Error al agregar usuario: " . $stmt_insert->error;
            }
            $stmt_insert->close();
        }
        $stmt->close();
    }
    header("Location: gestion_usuarios.php");
    exit();
}

// Eliminar usuario - CON MANEJO MEJORADO DE RESTRICCIONES DE CLAVE FORÁNEA
if (isset($_GET['eliminar'])) {
    $id = intval($_GET['eliminar']);
    if ($id > 0) {
        // Verificar si el usuario existe
        $stmt_check = $conn->prepare("SELECT id, rol FROM usuarios WHERE id = ?");
        $stmt_check->bind_param("i", $id);
        $stmt_check->execute();
        $result_check = $stmt_check->get_result();
        
        if ($result_check->num_rows === 0) {
            $_SESSION['error'] = "El usuario no existe";
            $stmt_check->close();
            header("Location: gestion_usuarios.php");
            exit();
        }
        
        $user_data = $result_check->fetch_assoc();
        $stmt_check->close();
        
        // Iniciar transacción para manejar todas las operaciones
        $conn->begin_transaction();
        
        try {
            // 1. Eliminar referencias en coordinador_postgrado
            $stmt_coordinador = $conn->prepare("DELETE FROM coordinador_postgrado WHERE coordinador_id = ?");
            $stmt_coordinador->bind_param("i", $id);
            $stmt_coordinador->execute();
            $stmt_coordinador->close();
            
            // 2. Eliminar referencias en coordinador_postgrado_escolaridad
            $stmt_coordinador_esc = $conn->prepare("DELETE FROM coordinador_postgrado_escolaridad WHERE coordinador_id = ?");
            $stmt_coordinador_esc->bind_param("i", $id);
            $stmt_coordinador_esc->execute();
            $stmt_coordinador_esc->close();
            
            // 3. Eliminar referencias en cambios_postgrado (como administrador_id)
            $stmt_cambios = $conn->prepare("DELETE FROM cambios_postgrado WHERE administrador_id = ?");
            $stmt_cambios->bind_param("i", $id);
            $stmt_cambios->execute();
            $stmt_cambios->close();
            
            // 4. Eliminar referencias en inscripciones (usuario_modificacion)
            $stmt_inscripciones = $conn->prepare("UPDATE inscripciones SET usuario_modificacion = NULL WHERE usuario_modificacion = ?");
            $stmt_inscripciones->bind_param("i", $id);
            $stmt_inscripciones->execute();
            $stmt_inscripciones->close();
            
            // 5. Eliminar referencias en retiros_estudiantes (usuario_id y usuario_reactivacion_id)
            $stmt_retiros = $conn->prepare("UPDATE retiros_estudiantes SET usuario_reactivacion_id = NULL WHERE usuario_reactivacion_id = ?");
            $stmt_retiros->bind_param("i", $id);
            $stmt_retiros->execute();
            $stmt_retiros->close();
            
            $stmt_retiros2 = $conn->prepare("DELETE FROM retiros_estudiantes WHERE usuario_id = ?");
            $stmt_retiros2->bind_param("i", $id);
            $stmt_retiros2->execute();
            $stmt_retiros2->close();
            
            // 6. Eliminar tokens de reset de contraseña
            $stmt_tokens = $conn->prepare("DELETE FROM password_reset_tokens WHERE user_id = ?");
            $stmt_tokens->bind_param("i", $id);
            $stmt_tokens->execute();
            $stmt_tokens->close();
            
            // 7. Finalmente eliminar el usuario
            $stmt_delete = $conn->prepare("DELETE FROM usuarios WHERE id = ?");
            $stmt_delete->bind_param("i", $id);
            
            if ($stmt_delete->execute()) {
                $conn->commit();
                $_SESSION['success'] = "Usuario eliminado correctamente";
            } else {
                throw new Exception("Error al eliminar usuario: " . $stmt_delete->error);
            }
            $stmt_delete->close();
            
        } catch (Exception $e) {
            // Revertir transacción en caso de error
            $conn->rollback();
            $_SESSION['error'] = $e->getMessage();
        }
    }
    header("Location: gestion_usuarios.php");
    exit();
}

// CONSULTA MEJORADA - Obtener usuarios con información combinada
$sql = "SELECT 
            u.id, 
            u.username, 
            u.rol, 
            u.email, 
            u.telefono, 
            u.direccion, 
            u.estado_cuenta,
            u.estudiante_id,
            u.nombres as nombres_usuario,
            u.apellidos as apellidos_usuario,
            u.tipo_documento,
            u.cedula_venezolana,
            u.cedula_extranjera,
            u.pasaporte,
            e.nombres as nombres_estudiante, 
            e.apellidos as apellidos_estudiante,
            e.correo as correo_estudiante,
            e.telefono as telefono_estudiante,
            e.direccion as direccion_estudiante
        FROM usuarios u 
        LEFT JOIN estudiantes e ON u.estudiante_id = e.id 
        ORDER BY u.estudiante_id IS NULL, u.rol, u.id ASC";
$result = $conn->query($sql);

$page_title = "Gestión de Usuarios";
include '../includes/header.php';
?>

<div class="container-fluid">
    <div class="mb-4">
        <a href="index.php" class="btn btn-secondary">Volver al Panel</a>
    </div>

    <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger"><?php echo htmlspecialchars($_SESSION['error']); ?></div>
        <?php unset($_SESSION['error']); ?>
    <?php endif; ?>

    <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success"><?php echo htmlspecialchars($_SESSION['success']); ?></div>
        <?php unset($_SESSION['success']); ?>
    <?php endif; ?>

    <div class="card">
        <div class="card-header bg-primary text-white">
            <h3 class="mb-0">
                <a data-bs-toggle="collapse" href="#collapseForm" role="button" aria-expanded="false" aria-controls="collapseForm" class="text-white text-decoration-none">
                    Agregar Nuevo Usuario <i class="bi bi-chevron-down"></i>
                </a>
            </h3>
        </div>
        <div class="collapse" id="collapseForm">
            <div class="card-body">
                <form action="gestion_usuarios.php" method="post" class="row g-3" id="addUserForm">
                    <div class="col-md-3">
                        <label for="rol" class="form-label">Rol:</label>
                        <select class="form-select" id="rol" name="rol" required>
                            <option value="administrador">Administrador</option>
                            <option value="secretaria">Secretaria</option>
                            <option value="coordinador1">Coordinador 1</option>
                            <option value="coordinador2">Coordinador 2</option>
                            <option value="estudiante">Estudiante</option>
                        </select>
                    </div>

                    <div id="student_fields" class="col-md-9" style="display: none;">
                        <label for="cedula_estudiante" class="form-label">Documento del Estudiante:</label>
                        <div class="input-group">
                            <input type="text" class="form-control" id="cedula_estudiante" name="cedula_estudiante" placeholder="Cédula Venezolana, Extranjera o Pasaporte">
                            <button class="btn btn-info" type="button" id="verify_student">Verificar</button>
                        </div>
                        <small id="cedula-message" class="form-text"></small>
                    </div>

                    <!-- NUEVO: Campos de documento unificados y simplificados -->
                    <div id="admin_document_fields" style="display: none;">
                        <div class="col-md-4">
                            <label for="tipo_documento" class="form-label">Tipo de Documento:</label>
                            <select class="form-select" id="tipo_documento" name="tipo_documento" required>
                                <option value="V">Cédula Venezolana</option>
                                <option value="E">Cédula Extranjera</option>
                                <option value="P">Pasaporte</option>
                            </select>
                        </div>
                        <div class="col-md-8">
                            <label for="numero_documento" class="form-label">Número de Documento:</label>
                            <input type="text" class="form-control" id="numero_documento" name="numero_documento" 
                                   placeholder="Ingrese el número de documento" required>
                            <small class="form-text text-muted">
                                Formato esperado: 
                                <span id="format_hint">V-12345678 para cédula venezolana</span>
                            </small>
                        </div>
                    </div>

                    <div class="col-md-3">
                        <label for="username" class="form-label">Nombre de Usuario:</label>
                        <input type="text" class="form-control" id="username" name="username" required>
                        <small class="form-text text-muted">Para estudiantes será la cédula</small>
                    </div>
                    
                    <div class="col-md-3">
                        <label for="password" class="form-label">Contraseña:</label>
                        <input type="password" class="form-control" id="password" name="password" required minlength="8">
                        <small class="form-text text-muted">Mínimo 8 caracteres</small>
                    </div>
                    
                    <!-- Campos de nombres y apellidos para NO estudiantes -->
                    <div class="col-md-3" id="nombres_field">
                        <label for="nombres" class="form-label">Nombres:</label>
                        <input type="text" class="form-control" id="nombres" name="nombres">
                        <small class="form-text text-muted">Para estudiantes se completará automáticamente</small>
                    </div>
                    
                    <div class="col-md-3" id="apellidos_field">
                        <label for="apellidos" class="form-label">Apellidos:</label>
                        <input type="text" class="form-control" id="apellidos" name="apellidos">
                        <small class="form-text text-muted">Para estudiantes se completará automáticamente</small>
                    </div>
                    
                    <div class="col-md-4">
                        <label for="email" class="form-label">Correo Electrónico:</label>
                        <input type="email" class="form-control" id="email" name="email">
                        <small class="form-text text-muted">Para estudiantes se usará el correo registrado</small>
                    </div>
                    
                    <div class="col-md-4">
                        <label for="telefono" class="form-label">Teléfono:</label>
                        <input type="text" class="form-control" id="telefono" name="telefono">
                    </div>
                    
                    <div class="col-md-4">
                        <label for="direccion" class="form-label">Dirección:</label>
                        <input type="text" class="form-control" id="direccion" name="direccion">
                    </div>
                    
                    <div class="col-12">
                        <button type="submit" name="agregar_usuario" class="btn btn-primary">Agregar Usuario</button>
                        <button type="reset" class="btn btn-secondary">Limpiar Formulario</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <div class="card mt-4">
        <div class="card-header bg-info text-white">
            <h3 class="mb-0">Lista de Usuarios</h3>
            <small>Total: <?php echo $result ? $result->num_rows : 0; ?> usuarios</small>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-striped table-hover table-bordered">
                    <thead class="table-dark">
                        <tr>
                            <th>ID</th>
                            <th>Usuario</th>
                            <th>Rol</th>
                            <th>Nombre Completo</th>
                            <th>Correo</th>
                            <th>Teléfono</th>
                            <th>Documento</th>
                            <th>Estado</th>
                            <th>Acciones</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if ($result && $result->num_rows > 0): ?>
                            <?php while ($row = $result->fetch_assoc()): ?>
                                <tr>
                                    <td><?php echo $row['id']; ?></td>
                                    <td>
                                        <strong><?php echo htmlspecialchars($row['username']); ?></strong>
                                        <?php if (!empty($row['estudiante_id'])): ?>
                                            <br><small class="text-muted">Est. ID: <?php echo $row['estudiante_id']; ?></small>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <span class="badge 
                                            <?php 
                                            echo $row['rol'] == 'administrador' ? 'bg-danger' : 
                                                   ($row['rol'] == 'secretaria' ? 'bg-primary' : 
                                                   ($row['rol'] == 'coordinador1' ? 'bg-warning' : 
                                                   ($row['rol'] == 'coordinador2' ? 'bg-info' : 'bg-success'))); 
                                            ?>">
                                            <?php echo htmlspecialchars($row['rol']); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php 
                                        // Para estudiantes: usar datos de la tabla estudiantes
                                        // Para otros roles: usar datos de la tabla usuarios
                                        if ($row['rol'] === 'estudiante' && !empty($row['nombres_estudiante'])) {
                                            echo htmlspecialchars($row['nombres_estudiante'] . ' ' . $row['apellidos_estudiante']);
                                        } elseif (!empty($row['nombres_usuario']) && !empty($row['apellidos_usuario'])) {
                                            echo htmlspecialchars($row['nombres_usuario'] . ' ' . $row['apellidos_usuario']);
                                        } else {
                                            echo '<span class="text-muted">-</span>';
                                        }
                                        ?>
                                    </td>
                                    <td>
                                        <?php 
                                        // Para estudiantes: usar correo de la tabla estudiantes
                                        // Para otros roles: usar email de la tabla usuarios
                                        if ($row['rol'] === 'estudiante' && !empty($row['correo_estudiante'])) {
                                            echo htmlspecialchars($row['correo_estudiante']);
                                        } else {
                                            echo htmlspecialchars($row['email'] ?? '');
                                        }
                                        ?>
                                    </td>
                                    <td>
                                        <?php 
                                        if ($row['rol'] === 'estudiante' && !empty($row['telefono_estudiante'])) {
                                            echo htmlspecialchars($row['telefono_estudiante']);
                                        } else {
                                            echo htmlspecialchars($row['telefono'] ?? '');
                                        }
                                        ?>
                                    </td>
                                    <td>
                                        <div class="documentos-list">
                                            <?php
                                            $documento_mostrar = '';
                                            $tipo_documento = $row['tipo_documento'];
                                            
                                            if (!empty($row['cedula_venezolana'])) {
                                                $documento_mostrar = $row['cedula_venezolana'];
                                                $tipo_documento = 'V';
                                            } elseif (!empty($row['cedula_extranjera'])) {
                                                $documento_mostrar = $row['cedula_extranjera'];
                                                $tipo_documento = 'E';
                                            } elseif (!empty($row['pasaporte'])) {
                                                $documento_mostrar = $row['pasaporte'];
                                                $tipo_documento = 'P';
                                            }
                                            
                                            if (!empty($documento_mostrar)) {
                                                $badge_class = [
                                                    'V' => 'bg-success',
                                                    'E' => 'bg-warning', 
                                                    'P' => 'bg-info'
                                                ];
                                                $tipo_texto = [
                                                    'V' => 'Cédula Venezolana',
                                                    'E' => 'Cédula Extranjera',
                                                    'P' => 'Pasaporte'
                                                ];
                                                
                                                echo '<span class="badge ' . ($badge_class[$tipo_documento] ?? 'bg-secondary') . '">';
                                                echo htmlspecialchars($documento_mostrar);
                                                echo '</span>';
                                                echo '<br><small class="text-muted">' . ($tipo_texto[$tipo_documento] ?? 'Documento') . '</small>';
                                            } else {
                                                echo '<span class="text-muted">Sin documento</span>';
                                            }
                                            ?>
                                        </div>
                                    </td>
                                    <td>
                                        <?php 
                                        $estado = $row['estado_cuenta'] ?? 'activa';
                                        $estado_class = '';
                                        switch($estado) {
                                            case 'activa': $estado_class = 'badge bg-success'; break;
                                            case 'pendiente': $estado_class = 'badge bg-warning'; break;
                                            case 'inactiva': $estado_class = 'badge bg-secondary'; break;
                                            case 'bloqueada': $estado_class = 'badge bg-danger'; break;
                                            default: $estado_class = 'badge bg-info';
                                        }
                                        ?>
                                        <span class="<?php echo $estado_class; ?>">
                                            <?php echo htmlspecialchars($estado); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <div class="btn-group" role="group">
                                            <a href='editar_usuario.php?id=<?php echo $row['id']; ?>' class="btn btn-sm btn-primary" title="Editar usuario">
                                                <i class="bi bi-pencil"></i> Editar
                                            </a>
                                            <a href='gestion_usuarios.php?eliminar=<?php echo $row['id']; ?>' 
                                               class="btn btn-sm btn-danger" 
                                               onclick='return confirm("¿Estás seguro de eliminar este usuario? Esta acción eliminará todas sus asignaciones y referencias en el sistema.")'
                                               title="Eliminar usuario">
                                                <i class="bi bi-trash"></i> Eliminar
                                            </a>
                                        </div>
                                    </td>
                                </tr>
                            <?php endwhile; ?>
                        <?php else: ?>
                            <tr>
                                <td colspan="9" class="text-center py-4">
                                    <?php 
                                    if (!$result) {
                                        echo "<div class='alert alert-danger'>Error en la consulta: " . $conn->error . "</div>";
                                    } else {
                                        echo "<div class='text-muted'>No hay usuarios registrados en el sistema.</div>";
                                    }
                                    ?>
                                </td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const rolSelect = document.getElementById('rol');
    const studentFields = document.getElementById('student_fields');
    const adminDocumentFields = document.getElementById('admin_document_fields');
    const usernameInput = document.getElementById('username');
    const nombresInput = document.getElementById('nombres');
    const apellidosInput = document.getElementById('apellidos');
    const emailInput = document.getElementById('email');
    const telefonoInput = document.getElementById('telefono');
    const direccionInput = document.getElementById('direccion');
    const verifyBtn = document.getElementById('verify_student');
    const cedulaInput = document.getElementById('cedula_estudiante');
    const cedulaMessage = document.getElementById('cedula-message');
    const nombresField = document.getElementById('nombres_field');
    const apellidosField = document.getElementById('apellidos_field');
    const tipoDocumentoSelect = document.getElementById('tipo_documento');
    const numeroDocumentoInput = document.getElementById('numero_documento');
    const formatHint = document.getElementById('format_hint');

    function setFieldsByRole(isStudent) {
        if (isStudent) {
            studentFields.style.display = 'block';
            adminDocumentFields.style.display = 'none';
            nombresField.style.display = 'none';
            apellidosField.style.display = 'none';
            // Limpiar campos cuando se cambia a estudiante
            if (rolSelect.value === 'estudiante') {
                nombresInput.value = '';
                apellidosInput.value = '';
                usernameInput.value = '';
                emailInput.value = '';
                telefonoInput.value = '';
                direccionInput.value = '';
                numeroDocumentoInput.value = '';
            }
        } else {
            studentFields.style.display = 'none';
            adminDocumentFields.style.display = 'block';
            nombresField.style.display = 'block';
            apellidosField.style.display = 'block';
        }
    }

    function updateFormatHint() {
        const tipo = tipoDocumentoSelect.value;
        const hints = {
            'V': 'V-12345678 para cédula venezolana',
            'E': 'E-123456789 para cédula extranjera', 
            'P': 'AB123456 para pasaporte'
        };
        formatHint.textContent = hints[tipo] || 'Ingrese el número de documento';
    }

    rolSelect.addEventListener('change', function() {
        setFieldsByRole(this.value === 'estudiante');
    });

    tipoDocumentoSelect.addEventListener('change', updateFormatHint);

    verifyBtn.addEventListener('click', function() {
        const cedula = cedulaInput.value.trim();
        if (!cedula) {
            cedulaMessage.textContent = 'Por favor, ingrese un documento.';
            cedulaMessage.className = 'form-text text-danger';
            return;
        }

        // Hacer la petición al servidor para verificar el estudiante
        fetch(`../api/verificar_estudiante.php?cedula=${encodeURIComponent(cedula)}`)
            .then(response => {
                if (!response.ok) { 
                    throw new Error('Error en la respuesta del servidor'); 
                }
                return response.json();
            })
            .then(data => {
                if (data.success) {
                    usernameInput.value = data.data.cedula;
                    nombresInput.value = data.data.nombres;
                    apellidosInput.value = data.data.apellidos;
                    emailInput.value = data.data.correo || '';
                    telefonoInput.value = data.data.telefono || '';
                    direccionInput.value = data.data.direccion || '';
                    
                    // Llenar automáticamente el tipo de documento y número
                    tipoDocumentoSelect.value = data.data.tipo_documento;
                    
                    // Determinar qué documento mostrar
                    const documentoBuscado = cedulaInput.value.trim();
                    let documentoMostrar = '';
                    
                    if (data.data.cedula_venezolana === documentoBuscado) {
                        documentoMostrar = data.data.cedula_venezolana || '';
                    } else if (data.data.cedula_extranjera === documentoBuscado) {
                        documentoMostrar = data.data.cedula_extranjera || '';
                    } else if (data.data.pasaporte === documentoBuscado) {
                        documentoMostrar = data.data.pasaporte || '';
                    } else {
                        // Fallback: usar el documento principal
                        switch(data.data.tipo_documento) {
                            case 'V': documentoMostrar = data.data.cedula_venezolana || ''; break;
                            case 'E': documentoMostrar = data.data.cedula_extranjera || ''; break;
                            case 'P': documentoMostrar = data.data.pasaporte || ''; break;
                        }
                    }
                    
                    numeroDocumentoInput.value = documentoMostrar;
                    
                    // Mostrar información del tipo de documento
                    const tipoDoc = data.data.tipo_documento;
                    const tipoTexto = tipoDoc === 'V' ? 'Cédula Venezolana' : 
                                     tipoDoc === 'E' ? 'Cédula Extranjera' : 'Pasaporte';
                    cedulaMessage.textContent = `Estudiante verificado correctamente (${tipoTexto}).`;
                    cedulaMessage.className = 'form-text text-success';
                } else {
                    // Limpiar campos si hay error
                    usernameInput.value = '';
                    nombresInput.value = '';
                    apellidosInput.value = '';
                    emailInput.value = '';
                    telefonoInput.value = '';
                    direccionInput.value = '';
                    numeroDocumentoInput.value = '';
                    cedulaMessage.textContent = data.message || 'Error al verificar estudiante.';
                    cedulaMessage.className = 'form-text text-danger';
                }
            })
            .catch(error => {
                // Limpiar campos si hay error de conexión
                usernameInput.value = '';
                nombresInput.value = '';
                apellidosInput.value = '';
                emailInput.value = '';
                telefonoInput.value = '';
                direccionInput.value = '';
                numeroDocumentoInput.value = '';
                cedulaMessage.textContent = 'Error de conexión: ' + error.message;
                cedulaMessage.className = 'form-text text-danger';
            });
    });

    // Initial state
    setFieldsByRole(rolSelect.value === 'estudiante');
    updateFormatHint();
});
</script>

<style>
.documentos-list .badge {
    font-size: 0.75em;
    margin-bottom: 2px;
}
.table th {
    font-weight: 600;
}
</style>

<?php
include '../includes/footer.php';
$conn->close();
?>