<?php
session_start();

// Configurar encoding UTF-8
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');

if (!isset($_SESSION['username']) || $_SESSION['rol'] != 'administrador') {
    header("Location: ../login.php");
    exit();
}

include '../includes/db.php';

// Configurar conexión para usar UTF-8
$conn->set_charset("utf8mb4");
$conn->query("SET NAMES 'utf8mb4'");
$conn->query("SET CHARACTER SET utf8mb4");
$conn->query("SET COLLATION_CONNECTION = 'utf8mb4_unicode_ci'");

// Endpoint para validación AJAX
if (isset($_GET['ajax']) && $_GET['ajax'] == 'validar_documento') {
    $tipo = $_GET['tipo'];
    $valor = $_GET['valor'];
    
    $existe = false;
    $mensaje = '';
    
    switch($tipo) {
        case 'cedula_venezolana':
            $sql = "SELECT id FROM estudiantes WHERE cedula_venezolana = ?";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("s", $valor);
            $stmt->execute();
            $result = $stmt->get_result();
            $existe = $result->num_rows > 0;
            $mensaje = $existe ? 'Esta cédula venezolana ya está registrada' : '';
            break;
            
        case 'cedula_extranjera':
            $sql = "SELECT id FROM estudiantes WHERE cedula_extranjera = ?";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("s", $valor);
            $stmt->execute();
            $result = $stmt->get_result();
            $existe = $result->num_rows > 0;
            $mensaje = $existe ? 'Esta cédula extranjera ya está registrada' : '';
            break;
            
        case 'pasaporte':
            $sql = "SELECT id FROM estudiantes WHERE pasaporte = ?";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("s", $valor);
            $stmt->execute();
            $result = $stmt->get_result();
            $existe = $result->num_rows > 0;
            $mensaje = $existe ? 'Este pasaporte ya está registrado' : '';
            break;
            
        case 'correo':
            $sql = "SELECT id FROM estudiantes WHERE correo = ?";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("s", $valor);
            $stmt->execute();
            $result = $stmt->get_result();
            $existe = $result->num_rows > 0;
            $mensaje = $existe ? 'Este correo ya está registrado' : '';
            break;
    }
    
    echo json_encode(['existe' => $existe, 'mensaje' => $mensaje]);
    exit();
}

// Guardar estudiante (usando consultas preparadas para seguridad)
if (isset($_POST['guardar_estudiante'])) {
    $tipo_documento = trim($_POST['tipo_documento']);
    $cedula_venezolana = trim($_POST['cedula_venezolana']);
    $cedula_extranjera = trim($_POST['cedula_extranjera']);
    $pasaporte = trim($_POST['pasaporte']);
    $nombres = trim($_POST['nombres']);
    $apellidos = trim($_POST['apellidos']);
    $correo = trim($_POST['correo']);
    $telefono = trim($_POST['telefono']);
    $celular = trim($_POST['celular']);
    $direccion = trim($_POST['direccion']);

    // Determinar qué campo usar como cédula principal (para compatibilidad)
    $cedula_principal = '';
    switch ($tipo_documento) {
        case 'V':
            $cedula_principal = $cedula_venezolana;
            break;
        case 'E':
            $cedula_principal = $cedula_extranjera;
            break;
        case 'P':
            $cedula_principal = $pasaporte;
            break;
    }

    // Verificar si ya existe un estudiante con el mismo documento principal
    $sql_check = "SELECT id FROM estudiantes WHERE cedula = ?";
    $stmt_check = $conn->prepare($sql_check);
    $stmt_check->bind_param("s", $cedula_principal);
    $stmt_check->execute();
    $result_check = $stmt_check->get_result();
    
    if ($result_check->num_rows > 0) {
        $_SESSION['error'] = "Ya existe un estudiante registrado con el documento: " . $cedula_principal;
        header("Location: gestion_estudiantes.php");
        exit();
    }

    // Verificar duplicados por tipo de documento específico
    if ($tipo_documento == 'V' && !empty($cedula_venezolana)) {
        $sql_check_venezolana = "SELECT id FROM estudiantes WHERE cedula_venezolana = ?";
        $stmt_check_venezolana = $conn->prepare($sql_check_venezolana);
        $stmt_check_venezolana->bind_param("s", $cedula_venezolana);
        $stmt_check_venezolana->execute();
        $result_check_venezolana = $stmt_check_venezolana->get_result();
        
        if ($result_check_venezolana->num_rows > 0) {
            $_SESSION['error'] = "Ya existe un estudiante registrado con la cédula venezolana: " . $cedula_venezolana;
            header("Location: gestion_estudiantes.php");
            exit();
        }
    }

    if ($tipo_documento == 'E' && !empty($cedula_extranjera)) {
        $sql_check_extranjera = "SELECT id FROM estudiantes WHERE cedula_extranjera = ?";
        $stmt_check_extranjera = $conn->prepare($sql_check_extranjera);
        $stmt_check_extranjera->bind_param("s", $cedula_extranjera);
        $stmt_check_extranjera->execute();
        $result_check_extranjera = $stmt_check_extranjera->get_result();
        
        if ($result_check_extranjera->num_rows > 0) {
            $_SESSION['error'] = "Ya existe un estudiante registrado con la cédula extranjera: " . $cedula_extranjera;
            header("Location: gestion_estudiantes.php");
            exit();
        }
    }

    if ($tipo_documento == 'P' && !empty($pasaporte)) {
        $sql_check_pasaporte = "SELECT id FROM estudiantes WHERE pasaporte = ?";
        $stmt_check_pasaporte = $conn->prepare($sql_check_pasaporte);
        $stmt_check_pasaporte->bind_param("s", $pasaporte);
        $stmt_check_pasaporte->execute();
        $result_check_pasaporte = $stmt_check_pasaporte->get_result();
        
        if ($result_check_pasaporte->num_rows > 0) {
            $_SESSION['error'] = "Ya existe un estudiante registrado con el pasaporte: " . $pasaporte;
            header("Location: gestion_estudiantes.php");
            exit();
        }
    }

    // Verificar duplicado de correo
    $sql_check_correo = "SELECT id FROM estudiantes WHERE correo = ?";
    $stmt_check_correo = $conn->prepare($sql_check_correo);
    $stmt_check_correo->bind_param("s", $correo);
    $stmt_check_correo->execute();
    $result_check_correo = $stmt_check_correo->get_result();
    
    if ($result_check_correo->num_rows > 0) {
        $_SESSION['error'] = "Ya existe un estudiante registrado con el correo: " . $correo;
        header("Location: gestion_estudiantes.php");
        exit();
    }

    // Si pasa todas las validaciones, insertar el estudiante
    $sql = "INSERT INTO estudiantes (cedula, tipo_documento, cedula_venezolana, cedula_extranjera, pasaporte, nombres, apellidos, correo, telefono, celular, direccion) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("sssssssssss", $cedula_principal, $tipo_documento, $cedula_venezolana, $cedula_extranjera, $pasaporte, $nombres, $apellidos, $correo, $telefono, $celular, $direccion);

    if ($stmt->execute()) {
        $_SESSION['success'] = "Estudiante guardado correctamente.";
    } else {
        $_SESSION['error'] = "Error al guardar el estudiante: " . $conn->error;
    }
    header("Location: gestion_estudiantes.php");
    exit();
}

// Eliminar estudiante (usando consultas preparadas)
if (isset($_GET['eliminar'])) {
    $id = intval($_GET['eliminar']);
    
    // Verificar si el estudiante tiene registros relacionados antes de eliminar
    $sql_check = "SELECT 1 FROM estudiante_postgrado WHERE estudiante_id = ?";
    $stmt_check = $conn->prepare($sql_check);
    $stmt_check->bind_param("i", $id);
    $stmt_check->execute();
    $result_check = $stmt_check->get_result();
    
    if ($result_check->num_rows > 0) {
        $_SESSION['error'] = "No se puede eliminar el estudiante porque tiene postgrados asignados.";
    } else {
        $sql = "DELETE FROM estudiantes WHERE id = ?";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("i", $id);
        
        if ($stmt->execute()) {
            $_SESSION['success'] = "Estudiante eliminado correctamente.";
        } else {
            $_SESSION['error'] = "Error al eliminar el estudiante.";
        }
    }
    header("Location: gestion_estudiantes.php");
    exit();
}

// Procesar búsqueda
$search = isset($_GET['search']) ? trim($_GET['search']) : '';
$where = '';
$params = array();
$types = '';

if (!empty($search)) {
    $where = "WHERE cedula LIKE ? OR nombres LIKE ? OR apellidos LIKE ? OR correo LIKE ? OR cedula_venezolana LIKE ? OR cedula_extranjera LIKE ? OR pasaporte LIKE ?";
    $search_term = "%$search%";
    $types = "sssssss";
    array_push($params, $search_term, $search_term, $search_term, $search_term, $search_term, $search_term, $search_term);
}

// Obtener estudiantes con paginación y búsqueda
$limit = 10;
$page = isset($_GET['page']) ? intval($_GET['page']) : 1;
$offset = ($page - 1) * $limit;

// Consulta para contar total de registros (con búsqueda si aplica)
$sql_count = "SELECT COUNT(*) as total FROM estudiantes $where";
$stmt_count = $conn->prepare($sql_count);

if (!empty($search)) {
    $stmt_count->bind_param($types, ...$params);
}

$stmt_count->execute();
$result_count = $stmt_count->get_result();
$total_rows = $result_count->fetch_assoc()['total'];
$total_pages = ceil($total_rows / $limit);

// Consulta para obtener los datos
$sql = "SELECT *, 
        CASE 
            WHEN tipo_documento = 'V' THEN cedula_venezolana
            WHEN tipo_documento = 'E' THEN cedula_extranjera
            WHEN tipo_documento = 'P' THEN pasaporte
        END as documento_principal
        FROM estudiantes $where ORDER BY apellidos, nombres LIMIT ? OFFSET ?";
$stmt = $conn->prepare($sql);

if (!empty($search)) {
    array_push($params, $limit, $offset);
    $types .= "ii";
    $stmt->bind_param($types, ...$params);
} else {
    $stmt->bind_param("ii", $limit, $offset);
}

$stmt->execute();
$result = $stmt->get_result();

if ($result === false) {
    die("Error en la consulta SQL: " . $conn->error);
}

$page_title = "Gestión de Estudiantes";
include '../includes/header.php';
?>

<div class="container">
    <!-- Botón para volver al panel -->
    <div class="mb-4">
        <a href="index.php" class="btn btn-secondary">
            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-arrow-left" viewBox="0 0 16 16">
                <path fill-rule="evenodd" d="M15 8a.5.5 0 0 0-.5-.5H2.707l3.147-3.146a.5.5 0 1 0-.708-.708l-4 4a.5.5 0 0 0 0 .708l4 4a.5.5 0 0 0 .708-.708L2.707 8.5H14.5A.5.5 0 0 0 15 8z"/>
            </svg>
            Volver al Panel
        </a>
    </div>

    <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success"><?= $_SESSION['success'] ?></div>
        <?php unset($_SESSION['success']); ?>
    <?php endif; ?>

    <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger"><?= $_SESSION['error'] ?></div>
        <?php unset($_SESSION['error']); ?>
    <?php endif; ?>

    <div class="card mb-4">
        <div class="card-header bg-primary text-white">
            <h3 class="mb-0">Agregar Nuevo Estudiante</h3>
        </div>
        <div class="card-body">
            <form method="post" class="row g-3" id="estudianteForm">
                <div class="col-md-3">
                    <label for="tipo_documento" class="form-label">Tipo de Documento:</label>
                    <select class="form-control" id="tipo_documento" name="tipo_documento" required onchange="toggleDocumentFields()">
                        <option value="V">Cédula Venezolana</option>
                        <option value="E">Cédula Extranjera</option>
                        <option value="P">Pasaporte</option>
                    </select>
                </div>
                
                <div class="col-md-3" id="cedula_venezolana_field">
                    <label for="cedula_venezolana" class="form-label">Cédula Venezolana:</label>
                    <div class="input-group">
                        <input type="text" class="form-control" id="cedula_venezolana" name="cedula_venezolana" placeholder="Cédula Venezolana" maxlength="20" onblur="validarDocumento('cedula_venezolana', 'V')">
                        <span class="input-group-text" id="cedula_venezolana_status"></span>
                    </div>
                    <div class="invalid-feedback" id="cedula_venezolana_error"></div>
                </div>
                
                <div class="col-md-3" id="cedula_extranjera_field" style="display: none;">
                    <label for="cedula_extranjera" class="form-label">Cédula Extranjera:</label>
                    <div class="input-group">
                        <input type="text" class="form-control" id="cedula_extranjera" name="cedula_extranjera" placeholder="Cédula Extranjera" maxlength="20" onblur="validarDocumento('cedula_extranjera', 'E')">
                        <span class="input-group-text" id="cedula_extranjera_status"></span>
                    </div>
                    <div class="invalid-feedback" id="cedula_extranjera_error"></div>
                </div>
                
                <div class="col-md-3" id="pasaporte_field" style="display: none;">
                    <label for="pasaporte" class="form-label">Pasaporte:</label>
                    <div class="input-group">
                        <input type="text" class="form-control" id="pasaporte" name="pasaporte" placeholder="Pasaporte" maxlength="20" onblur="validarDocumento('pasaporte', 'P')">
                        <span class="input-group-text" id="pasaporte_status"></span>
                    </div>
                    <div class="invalid-feedback" id="pasaporte_error"></div>
                </div>
                
                <div class="col-md-3">
                    <label for="nombres" class="form-label">Nombres:</label>
                    <input type="text" class="form-control" id="nombres" name="nombres" placeholder="Nombres" required maxlength="100">
                </div>
                
                <div class="col-md-3">
                    <label for="apellidos" class="form-label">Apellidos:</label>
                    <input type="text" class="form-control" id="apellidos" name="apellidos" placeholder="Apellidos" required maxlength="100">
                </div>
                
                <div class="col-md-3">
                    <label for="correo" class="form-label">Correo:</label>
                    <div class="input-group">
                        <input type="email" class="form-control" id="correo" name="correo" placeholder="ejemplo@dominio.com" required maxlength="100" onblur="validarCorreo()">
                        <span class="input-group-text" id="correo_status"></span>
                    </div>
                    <div class="invalid-feedback" id="correo_error"></div>
                </div>
                
                <div class="col-md-3">
                    <label for="telefono" class="form-label">Teléfono:</label>
                    <input type="text" class="form-control" id="telefono" name="telefono" placeholder="Teléfono" maxlength="20">
                </div>
                
                <div class="col-md-3">
                    <label for="celular" class="form-label">Celular:</label>
                    <input type="text" class="form-control" id="celular" name="celular" placeholder="Celular" maxlength="20">
                </div>
                
                <div class="col-md-6">
                    <label for="direccion" class="form-label">Dirección:</label>
                    <input type="text" class="form-control" id="direccion" name="direccion" placeholder="Dirección" maxlength="255">
                </div>
                
                <div class="col-12">
                    <button type="submit" name="guardar_estudiante" class="btn btn-primary" id="submitBtn" disabled>Guardar Estudiante</button>
                    <button type="button" class="btn btn-secondary" onclick="limpiarFormulario()">Limpiar Formulario</button>
                </div>
            </form>
        </div>
    </div>

    <div class="card mb-3">
        <div class="card-header bg-info text-white">
            <h3 class="mb-0">Buscar Estudiantes</h3>
        </div>
        <div class="card-body">
            <form method="get" action="gestion_estudiantes.php" class="row g-3">
                <div class="col-md-10">
                    <input type="text" class="form-control" name="search" placeholder="Buscar por cédula, nombres, apellidos, correo o documentos" value="<?= htmlspecialchars($search) ?>">
                </div>
                <div class="col-md-2">
                    <button type="submit" class="btn btn-primary w-100">Buscar</button>
                    <?php if (!empty($search)): ?>
                        <a href="gestion_estudiantes.php" class="btn btn-secondary w-100 mt-2">Limpiar</a>
                    <?php endif; ?>
                </div>
            </form>
        </div>
    </div>

    <div class="card">
        <div class="card-header bg-info text-white">
            <div class="d-flex justify-content-between align-items-center">
                <h3 class="mb-0">Lista de Estudiantes</h3>
                <span class="badge bg-light text-dark">Total: <?= $total_rows ?></span>
            </div>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-striped table-hover">
                    <thead>
                        <tr>
                            <th>#</th>
                            <th>Tipo Documento</th>
                            <th>Documento</th>
                            <th>Nombres</th>
                            <th>Apellidos</th>
                            <th>Correo</th>
                            <th>Acciones</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if ($result->num_rows > 0): ?>
                            <?php 
                            $contador = ($page - 1) * $limit + 1;
                            while ($row = $result->fetch_assoc()): 
                                $tipo_doc = '';
                                $documento = '';
                                switch ($row['tipo_documento']) {
                                    case 'V':
                                        $tipo_doc = 'Cédula Venezolana';
                                        $documento = $row['cedula_venezolana'];
                                        break;
                                    case 'E':
                                        $tipo_doc = 'Cédula Extranjera';
                                        $documento = $row['cedula_extranjera'];
                                        break;
                                    case 'P':
                                        $tipo_doc = 'Pasaporte';
                                        $documento = $row['pasaporte'];
                                        break;
                                }
                            ?>
                                <tr>
                                    <td><?= $contador ?></td>
                                    <td><?= htmlspecialchars($tipo_doc) ?></td>
                                    <td><?= htmlspecialchars($documento) ?></td>
                                    <td><?= htmlspecialchars($row['nombres']) ?></td>
                                    <td><?= htmlspecialchars($row['apellidos']) ?></td>
                                    <td><?= htmlspecialchars($row['correo']) ?></td>
                                    <td>
                                        <a href='editar_estudiante.php?id=<?= $row['id'] ?>' class="btn btn-sm btn-primary">Editar</a>
                                        <a href='gestion_estudiantes.php?eliminar=<?= $row['id'] ?>' class="btn btn-sm btn-danger" onclick='return confirm("¿Estás seguro de eliminar este estudiante?")'>Eliminar</a>
                                    </td>
                                </tr>
                            <?php 
                            $contador++;
                            endwhile; ?>
                        <?php else: ?>
                            <tr>
                                <td colspan="7" class="text-center">No hay estudiantes registrados <?= !empty($search) ? 'que coincidan con la búsqueda' : '' ?></td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>

            <!-- Paginación -->
            <?php if ($total_pages > 1): ?>
                <nav aria-label="Page navigation">
                    <ul class="pagination justify-content-center mt-4">
                        <?php if ($page > 1): ?>
                            <li class="page-item">
                                <a class="page-link" href="gestion_estudiantes.php?page=<?= $page - 1 ?><?= !empty($search) ? '&search='.urlencode($search) : '' ?>" aria-label="Previous">
                                    <span aria-hidden="true">&laquo;</span>
                                </a>
                            </li>
                        <?php endif; ?>
                        
                        <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                            <li class="page-item <?= $i == $page ? 'active' : '' ?>">
                                <a class="page-link" href="gestion_estudiantes.php?page=<?= $i ?><?= !empty($search) ? '&search='.urlencode($search) : '' ?>"><?= $i ?></a>
                            </li>
                        <?php endfor; ?>
                        
                        <?php if ($page < $total_pages): ?>
                            <li class="page-item">
                                <a class="page-link" href="gestion_estudiantes.php?page=<?= $page + 1 ?><?= !empty($search) ? '&search='.urlencode($search) : '' ?>" aria-label="Next">
                                    <span aria-hidden="true">&raquo;</span>
                                </a>
                            </li>
                        <?php endif; ?>
                    </ul>
                </nav>
            <?php endif; ?>
        </div>
    </div>
</div>

<script>
let validaciones = {
    documento: false,
    correo: false
};

function toggleDocumentFields() {
    const tipoDocumento = document.getElementById('tipo_documento').value;
    
    // Ocultar todos los campos primero
    document.getElementById('cedula_venezolana_field').style.display = 'none';
    document.getElementById('cedula_extranjera_field').style.display = 'none';
    document.getElementById('pasaporte_field').style.display = 'none';
    
    // Limpiar todos los campos y estados
    document.getElementById('cedula_venezolana').value = '';
    document.getElementById('cedula_extranjera').value = '';
    document.getElementById('pasaporte').value = '';
    
    resetStatus('cedula_venezolana');
    resetStatus('cedula_extranjera');
    resetStatus('pasaporte');
    
    // Mostrar solo el campo correspondiente
    switch(tipoDocumento) {
        case 'V':
            document.getElementById('cedula_venezolana_field').style.display = 'block';
            document.getElementById('cedula_venezolana').required = true;
            document.getElementById('cedula_extranjera').required = false;
            document.getElementById('pasaporte').required = false;
            break;
        case 'E':
            document.getElementById('cedula_extranjera_field').style.display = 'block';
            document.getElementById('cedula_venezolana').required = false;
            document.getElementById('cedula_extranjera').required = true;
            document.getElementById('pasaporte').required = false;
            break;
        case 'P':
            document.getElementById('pasaporte_field').style.display = 'block';
            document.getElementById('cedula_venezolana').required = false;
            document.getElementById('cedula_extranjera').required = false;
            document.getElementById('pasaporte').required = true;
            break;
    }
    
    validaciones.documento = false;
    actualizarBoton();
}

function resetStatus(fieldName) {
    const statusElement = document.getElementById(fieldName + '_status');
    const errorElement = document.getElementById(fieldName + '_error');
    const inputElement = document.getElementById(fieldName);
    
    statusElement.innerHTML = '';
    statusElement.className = 'input-group-text';
    errorElement.textContent = '';
    inputElement.classList.remove('is-invalid', 'is-valid');
}

function mostrarEstado(fieldName, esValido, mensaje) {
    const statusElement = document.getElementById(fieldName + '_status');
    const errorElement = document.getElementById(fieldName + '_error');
    const inputElement = document.getElementById(fieldName);
    
    if (esValido) {
        statusElement.innerHTML = '<i class="bi bi-check-circle-fill text-success"></i>';
        statusElement.className = 'input-group-text bg-light';
        inputElement.classList.remove('is-invalid');
        inputElement.classList.add('is-valid');
        errorElement.textContent = '';
    } else {
        statusElement.innerHTML = '<i class="bi bi-x-circle-fill text-danger"></i>';
        statusElement.className = 'input-group-text bg-light';
        inputElement.classList.remove('is-valid');
        inputElement.classList.add('is-invalid');
        errorElement.textContent = mensaje;
    }
}

function validarDocumento(fieldName, tipoDocumento) {
    const valor = document.getElementById(fieldName).value.trim();
    
    if (valor === '') {
        resetStatus(fieldName);
        validaciones.documento = false;
        actualizarBoton();
        return;
    }
    
    // Mostrar loading
    const statusElement = document.getElementById(fieldName + '_status');
    statusElement.innerHTML = '<i class="bi bi-hourglass-split text-warning"></i>';
    
    // Hacer petición AJAX real al servidor
    fetch(`gestion_estudiantes.php?ajax=validar_documento&tipo=${fieldName}&valor=${encodeURIComponent(valor)}`)
        .then(response => response.json())
        .then(data => {
            if (data.existe) {
                mostrarEstado(fieldName, false, data.mensaje);
                validaciones.documento = false;
            } else {
                mostrarEstado(fieldName, true, '');
                validaciones.documento = true;
            }
            actualizarBoton();
        })
        .catch(error => {
            console.error('Error en la validación:', error);
            mostrarEstado(fieldName, false, 'Error al validar documento');
            validaciones.documento = false;
            actualizarBoton();
        });
}

function validarFormatoCorreo(correo) {
    // Expresión regular más completa para validar correos electrónicos
    const emailRegex = /^[a-zA-Z0-9.!#$%&'*+/=?^_`{|}~-]+@[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?(?:\.[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?)*$/;
    
    // Verificar formato básico
    if (!emailRegex.test(correo)) {
        return { valido: false, mensaje: 'Formato de correo electrónico inválido' };
    }
    
    // Verificar que tenga @
    if (correo.indexOf('@') === -1) {
        return { valido: false, mensaje: 'El correo debe contener @' };
    }
    
    // Verificar dominio
    const partes = correo.split('@');
    const dominio = partes[1];
    
    // Lista de extensiones de dominio comunes
    const extensionesValidas = [
        'com', 'net', 'org', 'edu', 'gov', 'mil', 'info', 'biz', 
        've', 'com.ve', 'net.ve', 'org.ve', 'edu.ve', 'gob.ve',
        'co', 'com.co', 'org.co', 'edu.co', 'gov.co',
        'mx', 'com.mx', 'org.mx', 'edu.mx', 'gob.mx',
        'ar', 'com.ar', 'org.ar', 'edu.ar', 'gov.ar',
        'es', 'com.es', 'org.es', 'edu.es', 'gob.es',
        'pe', 'com.pe', 'org.pe', 'edu.pe', 'gob.pe',
        'cl', 'com.cl', 'org.cl', 'edu.cl', 'gob.cl',
        'br', 'com.br', 'org.br', 'edu.br', 'gov.br',
        'uy', 'com.uy', 'org.uy', 'edu.uy', 'gub.uy',
        'py', 'com.py', 'org.py', 'edu.py', 'gov.py',
        'ec', 'com.ec', 'org.ec', 'edu.ec', 'gob.ec'
    ];
    
    // Verificar si el dominio tiene una extensión válida
    const extension = dominio.split('.').pop().toLowerCase();
    if (!extensionesValidas.includes(extension)) {
        return { 
            valido: false, 
            mensaje: 'Extensión de dominio no válida. Use .com, .ve, .com.ve, etc.' 
        };
    }
    
    // Verificar longitud del dominio
    if (dominio.length < 3) {
        return { valido: false, mensaje: 'Dominio de correo demasiado corto' };
    }
    
    // Verificar que no tenga espacios
    if (correo.indexOf(' ') !== -1) {
        return { valido: false, mensaje: 'El correo no puede contener espacios' };
    }
    
    // Verificar que no empiece o termine con punto
    if (correo.startsWith('.') || correo.endsWith('.')) {
        return { valido: false, mensaje: 'El correo no puede empezar o terminar con punto' };
    }
    
    // Verificar que no tenga puntos consecutivos
    if (correo.includes('..')) {
        return { valido: false, mensaje: 'El correo no puede tener puntos consecutivos' };
    }
    
    return { valido: true, mensaje: '' };
}

function validarCorreo() {
    const correo = document.getElementById('correo').value.trim();
    
    if (correo === '') {
        resetStatus('correo');
        validaciones.correo = false;
        actualizarBoton();
        return;
    }
    
    // Primero validar el formato del correo
    const validacionFormato = validarFormatoCorreo(correo);
    if (!validacionFormato.valido) {
        mostrarEstado('correo', false, validacionFormato.mensaje);
        validaciones.correo = false;
        actualizarBoton();
        return;
    }
    
    // Mostrar loading
    const statusElement = document.getElementById('correo_status');
    statusElement.innerHTML = '<i class="bi bi-hourglass-split text-warning"></i>';
    
    // Si el formato es válido, verificar duplicados con AJAX real
    fetch(`gestion_estudiantes.php?ajax=validar_documento&tipo=correo&valor=${encodeURIComponent(correo)}`)
        .then(response => response.json())
        .then(data => {
            if (data.existe) {
                mostrarEstado('correo', false, data.mensaje);
                validaciones.correo = false;
            } else {
                mostrarEstado('correo', true, '');
                validaciones.correo = true;
            }
            actualizarBoton();
        })
        .catch(error => {
            console.error('Error en la validación:', error);
            mostrarEstado('correo', false, 'Error al validar correo');
            validaciones.correo = false;
            actualizarBoton();
        });
}

function actualizarBoton() {
    const tipoDocumento = document.getElementById('tipo_documento').value;
    const documentoValido = validaciones.documento;
    const correoValido = validaciones.correo;
    
    // Verificar que el campo de documento requerido tenga valor
    let documentoConValor = false;
    switch(tipoDocumento) {
        case 'V': documentoConValor = document.getElementById('cedula_venezolana').value.trim() !== ''; break;
        case 'E': documentoConValor = document.getElementById('cedula_extranjera').value.trim() !== ''; break;
        case 'P': documentoConValor = document.getElementById('pasaporte').value.trim() !== ''; break;
    }
    
    const nombresValido = document.getElementById('nombres').value.trim() !== '';
    const apellidosValido = document.getElementById('apellidos').value.trim() !== '';
    
    const habilitar = documentoConValor && documentoValido && correoValido && nombresValido && apellidosValido;
    
    document.getElementById('submitBtn').disabled = !habilitar;
}

function limpiarFormulario() {
    document.getElementById('estudianteForm').reset();
    toggleDocumentFields();
    
    // Resetear todas las validaciones
    validaciones.documento = false;
    validaciones.correo = false;
    
    // Limpiar estados visuales
    resetStatus('cedula_venezolana');
    resetStatus('cedula_extranjera');
    resetStatus('pasaporte');
    resetStatus('correo');
    
    actualizarBoton();
}

// Inicializar campos al cargar la página
document.addEventListener('DOMContentLoaded', function() {
    toggleDocumentFields();
    
    // Agregar event listeners para validación en tiempo real
    document.getElementById('nombres').addEventListener('input', actualizarBoton);
    document.getElementById('apellidos').addEventListener('input', actualizarBoton);
    document.getElementById('cedula_venezolana').addEventListener('input', actualizarBoton);
    document.getElementById('cedula_extranjera').addEventListener('input', actualizarBoton);
    document.getElementById('pasaporte').addEventListener('input', actualizarBoton);
    document.getElementById('correo').addEventListener('input', actualizarBoton);
});
</script>

<!-- Incluir Bootstrap Icons -->
<link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css" rel="stylesheet">
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>

<?php
include '../includes/footer.php';
$conn->close();
?>