<?php
session_start();

// Configurar encoding UTF-8
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');

if (!isset($_SESSION['username'])) {
    header("Location: ../login.php");
    exit();
}

include '../includes/db.php';

// Configurar conexión para usar UTF-8
$conn->set_charset("utf8mb4");
$conn->query("SET NAMES 'utf8mb4'");
$conn->query("SET CHARACTER SET utf8mb4");
$conn->query("SET COLLATION_CONNECTION = 'utf8mb4_unicode_ci'");

// Procesar operaciones CRUD
$mensaje = '';
$tipo_mensaje = '';

// Agregar un nuevo plan
if (isset($_POST['agregar_plan'])) {
    $nombre = trim($_POST['nombre']);
    if (!empty($nombre)) {
        // Verificar si el plan ya existe
        $sql_check = "SELECT id FROM planes WHERE nombre = ?";
        $stmt_check = $conn->prepare($sql_check);
        $stmt_check->bind_param("s", $nombre);
        $stmt_check->execute();
        $result_check = $stmt_check->get_result();
        
        if ($result_check->num_rows > 0) {
            $_SESSION['error'] = "El plan '$nombre' ya existe";
        } else {
            $sql = "INSERT INTO planes (nombre) VALUES (?)";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("s", $nombre);
            if ($stmt->execute()) {
                $_SESSION['success'] = "Plan agregado correctamente";
                header("Location: gestion_datos_fijos.php");
                exit();
            } else {
                $_SESSION['error'] = "Error al agregar plan: " . $conn->error;
            }
        }
    }
}

// Agregar una nueva promoción
if (isset($_POST['agregar_promocion'])) {
    $nombre = trim($_POST['nombre']);
    if (!empty($nombre)) {
        // Verificar si la promoción ya existe
        $sql_check = "SELECT id FROM promociones WHERE nombre = ?";
        $stmt_check = $conn->prepare($sql_check);
        $stmt_check->bind_param("s", $nombre);
        $stmt_check->execute();
        $result_check = $stmt_check->get_result();
        
        if ($result_check->num_rows > 0) {
            $_SESSION['error'] = "La promoción '$nombre' ya existe";
        } else {
            $sql = "INSERT INTO promociones (nombre) VALUES (?)";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("s", $nombre);
            if ($stmt->execute()) {
                $_SESSION['success'] = "Promoción agregada correctamente";
                header("Location: gestion_datos_fijos.php");
                exit();
            } else {
                $_SESSION['error'] = "Error al agregar promoción: " . $conn->error;
            }
        }
    }
}

// Agregar una nueva escolaridad y generar años lectivos automáticamente
if (isset($_POST['agregar_escolaridad'])) {
    $anio_base = trim($_POST['nombre']);
    
    // Validar que el nombre sea un año válido (4 dígitos)
    if (!empty($anio_base) && preg_match('/^\d{4}$/', $anio_base)) {
        // Verificar si la escolaridad ya existe
        $sql_check = "SELECT id FROM escolaridades WHERE nombre = ?";
        $stmt_check = $conn->prepare($sql_check);
        $stmt_check->bind_param("s", $anio_base);
        $stmt_check->execute();
        $result_check = $stmt_check->get_result();
        
        if ($result_check->num_rows > 0) {
            $_SESSION['error'] = "La escolaridad para el año $anio_base ya existe";
        } else {
            $conn->begin_transaction();
            try {
                // Insertar la escolaridad
                $sql_esc = "INSERT INTO escolaridades (nombre) VALUES (?)";
                $stmt_esc = $conn->prepare($sql_esc);
                $stmt_esc->bind_param("s", $anio_base);
                $stmt_esc->execute();
                
                // Generar los años lectivos (trimestres)
                $trimestres = ['01C', '02C', '03C'];
                $errores_anios = [];
                
                foreach ($trimestres as $trimestre) {
                    $nombre_anio_lectivo = "$anio_base-$trimestre";
                    
                    // Verificar si el año lectivo ya existe antes de insertar
                    $sql_check_anio = "SELECT id FROM anios_lectivos WHERE nombre = ?";
                    $stmt_check_anio = $conn->prepare($sql_check_anio);
                    $stmt_check_anio->bind_param("s", $nombre_anio_lectivo);
                    $stmt_check_anio->execute();
                    $result_check_anio = $stmt_check_anio->get_result();
                    
                    if ($result_check_anio->num_rows > 0) {
                        $errores_anios[] = $nombre_anio_lectivo;
                    } else {
                        $sql_anio = "INSERT INTO anios_lectivos (nombre) VALUES (?)";
                        $stmt_anio = $conn->prepare($sql_anio);
                        $stmt_anio->bind_param("s", $nombre_anio_lectivo);
                        $stmt_anio->execute();
                    }
                }
                
                if (!empty($errores_anios)) {
                    // Si hay años lectivos que ya existían, hacer rollback y mostrar error
                    $conn->rollback();
                    $anios_duplicados = implode(', ', $errores_anios);
                    $_SESSION['error'] = "No se pudo crear la escolaridad. Los siguientes años lectivos ya existen: $anios_duplicados";
                } else {
                    $conn->commit();
                    $_SESSION['success'] = "Escolaridad y años lectivos para $anio_base creados correctamente";
                }
            } catch (Exception $e) {
                $conn->rollback();
                $_SESSION['error'] = "Error al crear escolaridad y años lectivos: " . $e->getMessage();
            }
        }
    } else {
        $_SESSION['error'] = "El año debe tener exactamente 4 dígitos (ej: 2025)";
    }
    header("Location: gestion_datos_fijos.php");
    exit();
}

// Agregar un nuevo año lectivo manualmente
if (isset($_POST['agregar_año_lectivo'])) {
    $nombre = trim($_POST['nombre']);
    if (!empty($nombre)) {
        // Verificar si el año lectivo ya existe
        $sql_check = "SELECT id FROM anios_lectivos WHERE nombre = ?";
        $stmt_check = $conn->prepare($sql_check);
        $stmt_check->bind_param("s", $nombre);
        $stmt_check->execute();
        $result_check = $stmt_check->get_result();
        
        if ($result_check->num_rows > 0) {
            $_SESSION['error'] = "El año lectivo '$nombre' ya existe";
        } else {
            $sql = "INSERT INTO anios_lectivos (nombre) VALUES (?)";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("s", $nombre);
            if ($stmt->execute()) {
                $_SESSION['success'] = "Año lectivo agregado correctamente";
                header("Location: gestion_datos_fijos.php");
                exit();
            } else {
                $_SESSION['error'] = "Error al agregar año lectivo: " . $conn->error;
            }
        }
    }
}

// Agregar una nueva sección
if (isset($_POST['agregar_seccion'])) {
    $nombre = trim($_POST['nombre']);
    if (!empty($nombre)) {
        // Verificar si la sección ya existe
        $sql_check = "SELECT id FROM secciones WHERE nombre = ?";
        $stmt_check = $conn->prepare($sql_check);
        $stmt_check->bind_param("s", $nombre);
        $stmt_check->execute();
        $result_check = $stmt_check->get_result();
        
        if ($result_check->num_rows > 0) {
            $_SESSION['error'] = "La sección '$nombre' ya existe";
        } else {
            $sql = "INSERT INTO secciones (nombre) VALUES (?)";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("s", $nombre);
            if ($stmt->execute()) {
                $_SESSION['success'] = "Sección agregada correctamente";
                header("Location: gestion_datos_fijos.php");
                exit();
            } else {
                $_SESSION['error'] = "Error al agregar sección: " . $conn->error;
            }
        }
    }
}

// Eliminar registro
if (isset($_GET['eliminar']) && isset($_GET['tabla']) && isset($_GET['id'])) {
    $tabla = $_GET['tabla'];
    $id = intval($_GET['id']);
    
    // Validar tabla permitida
    $tablas_permitidas = ['planes', 'promociones', 'escolaridades', 'anios_lectivos', 'secciones'];
    if (in_array($tabla, $tablas_permitidas) && $id > 0) {
        try {
            $conn->begin_transaction();
            
            if ($tabla == 'planes') {
                // Primero actualizar las denominaciones que usan este plan
                $sql_update = "UPDATE denominaciones SET plan_id = NULL WHERE plan_id = ?";
                $stmt_update = $conn->prepare($sql_update);
                $stmt_update->bind_param("i", $id);
                $stmt_update->execute();
                
                // Luego eliminar el plan
                $sql = "DELETE FROM planes WHERE id = ?";
            } else if ($tabla == 'escolaridades') {
                // No permitir eliminar escolaridades con años lectivos asociados
                $sql_check = "SELECT COUNT(*) as total FROM anios_lectivos WHERE nombre LIKE CONCAT((SELECT nombre FROM escolaridades WHERE id = ?), '-%')";
                $stmt_check = $conn->prepare($sql_check);
                $stmt_check->bind_param("i", $id);
                $stmt_check->execute();
                $result_check = $stmt_check->get_result();
                $row_check = $result_check->fetch_assoc();
                
                if ($row_check['total'] > 0) {
                    throw new Exception("No se puede eliminar esta escolaridad porque tiene años lectivos asociados");
                }
                
                $sql = "DELETE FROM $tabla WHERE id = ?";
            } else {
                $sql = "DELETE FROM $tabla WHERE id = ?";
            }
            
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("i", $id);
            
            if ($stmt->execute()) {
                $conn->commit();
                $_SESSION['success'] = "Registro eliminado correctamente";
            } else {
                $conn->rollback();
                $_SESSION['error'] = "Error al eliminar registro: " . $conn->error;
            }
        } catch (Exception $e) {
            $conn->rollback();
            $_SESSION['error'] = "Error: " . $e->getMessage();
        }
    } else {
        $_SESSION['error'] = "Operación no permitida";
    }
    header("Location: gestion_datos_fijos.php");
    exit();
}

$page_title = "Gestión de Datos Fijos";
include '../includes/header.php';
?>

<div class="container">
    <!-- Botón Volver al Panel -->
    <div class="mb-4">
        <a href="index.php" class="btn btn-secondary">
            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-arrow-left" viewBox="0 0 16 16">
                <path fill-rule="evenodd" d="M15 8a.5.5 0 0 0-.5-.5H2.707l3.147-3.146a.5.5 0 1 0-.708-.708l-4 4a.5.5 0 0 0 0 .708l4 4a.5.5 0 0 0 .708-.708L2.707 8.5H14.5A.5.5 0 0 0 15 8z"/>
            </svg>
            Volver al Panel
        </a>
    </div>

    <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger"><?= $_SESSION['error'] ?></div>
        <?php unset($_SESSION['error']); ?>
    <?php endif; ?>

    <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success"><?= $_SESSION['success'] ?></div>
        <?php unset($_SESSION['success']); ?>
    <?php endif; ?>

    <div class="card">
        <div class="card-header bg-primary text-white">
            <h3 class="mb-0">Agregar Nuevo Dato</h3>
        </div>
        <div class="card-body">
            <ul class="nav nav-tabs" id="addDataTabs" role="tablist">
                <li class="nav-item" role="presentation">
                    <button class="nav-link active" id="add-plan-tab" data-bs-toggle="tab" data-bs-target="#add-plan" type="button" role="tab">Plan</button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" id="add-promocion-tab" data-bs-toggle="tab" data-bs-target="#add-promocion" type="button" role="tab">Promoción</button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" id="add-escolaridad-tab" data-bs-toggle="tab" data-bs-target="#add-escolaridad" type="button" role="tab">Escolaridad</button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" id="add-anio-lectivo-tab" data-bs-toggle="tab" data-bs-target="#add-anio-lectivo" type="button" role="tab">Año Lectivo</button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" id="add-seccion-tab" data-bs-toggle="tab" data-bs-target="#add-seccion" type="button" role="tab">Sección</button>
                </li>
            </ul>
            <div class="tab-content p-3 border border-top-0 rounded-bottom" id="addDataTabsContent">
                <div class="tab-pane fade show active" id="add-plan" role="tabpanel">
                    <form method="post" class="row g-3 mt-3">
                        <div class="col-md-8">
                            <input type="text" class="form-control" name="nombre" placeholder="Nombre del Plan" required>
                        </div>
                        <div class="col-md-4">
                            <button type="submit" name="agregar_plan" class="btn btn-primary w-100">Agregar Plan</button>
                        </div>
                    </form>
                </div>
                <div class="tab-pane fade" id="add-promocion" role="tabpanel">
                    <form method="post" class="row g-3 mt-3">
                        <div class="col-md-8">
                            <input type="text" class="form-control" name="nombre" placeholder="Nombre de la Promoción" required>
                        </div>
                        <div class="col-md-4">
                            <button type="submit" name="agregar_promocion" class="btn btn-primary w-100">Agregar Promoción</button>
                        </div>
                    </form>
                </div>
                <div class="tab-pane fade" id="add-escolaridad" role="tabpanel">
                    <form method="post" class="row g-3 mt-3">
                        <div class="col-md-8">
                            <input type="text" class="form-control" name="nombre" placeholder="Año (ej: 2025)" pattern="\d{4}" title="Debe ingresar exactamente 4 dígitos (ej: 2025)" required>
                            <small class="form-text text-muted">Ingrese solo el año (ej: 2025). Se crearán automáticamente los trimestres (01C, 02C, 03C)</small>
                        </div>
                        <div class="col-md-4">
                            <button type="submit" name="agregar_escolaridad" class="btn btn-primary w-100">Agregar Escolaridad</button>
                        </div>
                    </form>
                </div>
                <div class="tab-pane fade" id="add-anio-lectivo" role="tabpanel">
                    <form method="post" class="row g-3 mt-3">
                        <div class="col-md-8">
                            <input type="text" class="form-control" name="nombre" placeholder="Nombre del Año Lectivo (ej: 2025-01C)" required>
                        </div>
                        <div class="col-md-4">
                            <button type="submit" name="agregar_año_lectivo" class="btn btn-primary w-100">Agregar Año Lectivo</button>
                        </div>
                    </form>
                </div>
                <div class="tab-pane fade" id="add-seccion" role="tabpanel">
                    <form method="post" class="row g-3 mt-3">
                        <div class="col-md-8">
                            <input type="text" class="form-control" name="nombre" placeholder="Nombre de la Sección" required>
                        </div>
                        <div class="col-md-4">
                            <button type="submit" name="agregar_seccion" class="btn btn-primary w-100">Agregar Sección</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <div class="card mt-4">
        <div class="card-header bg-info text-white">
            <h3 class="mb-0">Datos Registrados</h3>
        </div>
        <div class="card-body">
            <ul class="nav nav-tabs" id="dataTabs" role="tablist">
                <li class="nav-item" role="presentation">
                    <button class="nav-link active" id="planes-tab" data-bs-toggle="tab" data-bs-target="#planes" type="button" role="tab">Planes</button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" id="promociones-tab" data-bs-toggle="tab" data-bs-target="#promociones" type="button" role="tab">Promociones</button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" id="escolaridades-tab" data-bs-toggle="tab" data-bs-target="#escolaridades" type="button" role="tab">Escolaridades</button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" id="anios-tab" data-bs-toggle="tab" data-bs-target="#anios" type="button" role="tab">Años Lectivos</button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" id="secciones-tab" data-bs-toggle="tab" data-bs-target="#secciones" type="button" role="tab">Secciones</button>
                </li>
            </ul>
            <div class="tab-content p-3 border border-top-0 rounded-bottom" id="dataTabsContent">
                <div class="tab-pane fade show active" id="planes" role="tabpanel">
                    <div class="table-responsive">
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Nombre</th>
                                    <th>Acciones</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php
                                $sql = "SELECT * FROM planes";
                                $result = $conn->query($sql);
                                if ($result && $result->num_rows > 0) {
                                    while ($row = $result->fetch_assoc()): ?>
                                        <tr>
                                            <td><?= $row['id'] ?></td>
                                            <td><?= htmlspecialchars($row['nombre']) ?></td>
                                            <td>
                                                <a href="editar_dato.php?tabla=planes&id=<?= $row['id'] ?>" class="btn btn-sm btn-primary">Editar</a>
                                                <a href="gestion_datos_fijos.php?eliminar=1&tabla=planes&id=<?= $row['id'] ?>" class="btn btn-sm btn-danger" onclick="return confirm('¿Está seguro de eliminar este registro?')">Eliminar</a>
                                            </td>
                                        </tr>
                                    <?php endwhile;
                                } else {
                                    echo '<tr><td colspan="3" class="text-center text-muted">No hay planes registrados</td></tr>';
                                }
                                ?>
                            </tbody>
                        </table>
                    </div>
                </div>
                <div class="tab-pane fade" id="promociones" role="tabpanel">
                    <div class="table-responsive">
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Nombre</th>
                                    <th>Acciones</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php
                                $sql = "SELECT * FROM promociones";
                                $result = $conn->query($sql);
                                if ($result && $result->num_rows > 0) {
                                    while ($row = $result->fetch_assoc()): ?>
                                        <tr>
                                            <td><?= $row['id'] ?></td>
                                            <td><?= htmlspecialchars($row['nombre']) ?></td>
                                            <td>
                                                <a href="editar_dato.php?tabla=promociones&id=<?= $row['id'] ?>" class="btn btn-sm btn-primary">Editar</a>
                                                <a href="gestion_datos_fijos.php?eliminar=1&tabla=promociones&id=<?= $row['id'] ?>" class="btn btn-sm btn-danger" onclick="return confirm('¿Está seguro de eliminar este registro?')">Eliminar</a>
                                            </td>
                                        </tr>
                                    <?php endwhile;
                                } else {
                                    echo '<tr><td colspan="3" class="text-center text-muted">No hay promociones registradas</td></tr>';
                                }
                                ?>
                            </tbody>
                        </table>
                    </div>
                </div>
                <div class="tab-pane fade" id="escolaridades" role="tabpanel">
                    <div class="table-responsive">
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Nombre</th>
                                    <th>Acciones</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php
                                $sql = "SELECT * FROM escolaridades";
                                $result = $conn->query($sql);
                                if ($result && $result->num_rows > 0) {
                                    while ($row = $result->fetch_assoc()): ?>
                                        <tr>
                                            <td><?= $row['id'] ?></td>
                                            <td><?= htmlspecialchars($row['nombre']) ?></td>
                                            <td>
                                                <a href="editar_dato.php?tabla=escolaridades&id=<?= $row['id'] ?>" class="btn btn-sm btn-primary">Editar</a>
                                                <a href="gestion_datos_fijos.php?eliminar=1&tabla=escolaridades&id=<?= $row['id'] ?>" class="btn btn-sm btn-danger" onclick="return confirm('¿Está seguro de eliminar este registro? \n\nNOTA: No se puede eliminar si tiene años lectivos asociados')">Eliminar</a>
                                            </td>
                                        </tr>
                                    <?php endwhile;
                                } else {
                                    echo '<tr><td colspan="3" class="text-center text-muted">No hay escolaridades registradas</td></tr>';
                                }
                                ?>
                            </tbody>
                        </table>
                    </div>
                </div>
                <div class="tab-pane fade" id="anios" role="tabpanel">
                    <div class="table-responsive">
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Nombre</th>
                                    <th>Acciones</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php
                                $sql = "SELECT * FROM anios_lectivos ORDER BY nombre DESC";
                                $result = $conn->query($sql);
                                if ($result && $result->num_rows > 0) {
                                    while ($row = $result->fetch_assoc()): ?>
                                        <tr>
                                            <td><?= $row['id'] ?></td>
                                            <td><?= htmlspecialchars($row['nombre']) ?></td>
                                            <td>
                                                <a href="editar_dato.php?tabla=anios_lectivos&id=<?= $row['id'] ?>" class="btn btn-sm btn-primary">Editar</a>
                                                <a href="gestion_datos_fijos.php?eliminar=1&tabla=anios_lectivos&id=<?= $row['id'] ?>" class="btn btn-sm btn-danger" onclick="return confirm('¿Está seguro de eliminar este registro?')">Eliminar</a>
                                            </td>
                                        </tr>
                                    <?php endwhile;
                                } else {
                                    echo '<tr><td colspan="3" class="text-center text-muted">No hay años lectivos registrados</td></tr>';
                                }
                                ?>
                            </tbody>
                        </table>
                    </div>
                </div>
                <div class="tab-pane fade" id="secciones" role="tabpanel">
                    <div class="table-responsive">
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Nombre</th>
                                    <th>Acciones</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php
                                $sql = "SELECT * FROM secciones";
                                $result = $conn->query($sql);
                                if ($result && $result->num_rows > 0) {
                                    while ($row = $result->fetch_assoc()): ?>
                                        <tr>
                                            <td><?= $row['id'] ?></td>
                                            <td><?= htmlspecialchars($row['nombre']) ?></td>
                                            <td>
                                                <a href="editar_dato.php?tabla=secciones&id=<?= $row['id'] ?>" class="btn btn-sm btn-primary">Editar</a>
                                                <a href="gestion_datos_fijos.php?eliminar=1&tabla=secciones&id=<?= $row['id'] ?>" class="btn btn-sm btn-danger" onclick="return confirm('¿Está seguro de eliminar este registro?')">Eliminar</a>
                                            </td>
                                        </tr>
                                    <?php endwhile;
                                } else {
                                    echo '<tr><td colspan="3" class="text-center text-muted">No hay secciones registradas</td></tr>';
                                }
                                ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// Script para asegurar el funcionamiento de las pestañas
document.addEventListener('DOMContentLoaded', function() {
    // Inicializar todas las pestañas
    var tabTriggers = document.querySelectorAll('button[data-bs-toggle="tab"]');
    tabTriggers.forEach(function(tabTriggerEl) {
        tabTriggerEl.addEventListener('click', function (e) {
            e.preventDefault();
            var tab = new bootstrap.Tab(this);
            tab.show();
        });
    });
});
</script>

<?php
include '../includes/footer.php';
$conn->close();
?>