<?php
session_start();

// Configurar encoding UTF-8
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');

if (!isset($_SESSION['username']) || $_SESSION['rol'] != 'administrador') {
    header("Location: ../login.php");
    exit();
}

include '../includes/db.php';

// Configurar conexión para usar UTF-8
$conn->set_charset("utf8mb4");
$conn->query("SET NAMES 'utf8mb4'");
$conn->query("SET CHARACTER SET utf8mb4");
$conn->query("SET COLLATION_CONNECTION = 'utf8mb4_unicode_ci'");

// Verificar si se recibió un ID de inscripción válido
if (!isset($_GET['id'])) {
    die("Error: No se proporcionó ID de inscripción");
}

$inscripcion_id = intval($_GET['id']);

// Obtener datos para los selectores
$sql_escolaridades = "SELECT * FROM escolaridades";
$result_escolaridades = $conn->query($sql_escolaridades);

$sql_anios = "SELECT * FROM años_lectivos";
$result_anios = $conn->query($sql_anios);

$sql_secciones = "SELECT * FROM secciones";
$result_secciones = $conn->query($sql_secciones);

// Obtener los datos de la inscripción
$sql = "SELECT i.*, d.tipo, d.nombre AS denominacion_nombre, 
               e.nombres, e.apellidos, e.cedula,
               p.nombre AS postgrado_nombre
        FROM inscripciones i
        JOIN denominaciones d ON i.denominacion_id = d.id
        JOIN estudiantes e ON i.estudiante_id = e.id
        JOIN postgrados p ON d.postgrado_id = p.id
        WHERE i.id = ?";
$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $inscripcion_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows == 0) {
    die("Error: No se encontró la inscripción con ID $inscripcion_id");
}

$inscripcion = $result->fetch_assoc();

// Procesar el formulario de actualización
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['actualizar'])) {
    $nota_final = $_POST['nota_final'];
    $nota_definitiva = $_POST['nota_definitiva'];
    $estado = $_POST['estado'];
    $escolaridad_id = intval($_POST['escolaridad_id']);
    $anio_lectivo_id = intval($_POST['anio_lectivo_id']);
    $seccion_id = intval($_POST['seccion_id']);
    $fecha_inscripcion = $_POST['fecha_inscripcion'];
    
    // Validaciones básicas
    if (empty($fecha_inscripcion)) {
        $error = "La fecha de inscripción es requerida";
    }
    
    // Validar según el tipo de denominación
    elseif ($inscripcion['tipo'] == 'puntaje') {
        // Validar que las notas sean numéricas y entre 0-20
        if (!empty($nota_final) && (!is_numeric($nota_final) || $nota_final < 0 || $nota_final > 20)) {
            $error = "La nota final debe ser un número entre 0 y 20";
        } elseif (!empty($nota_definitiva) && (!is_numeric($nota_definitiva) || $nota_definitiva < 0 || $nota_definitiva > 20)) {
            $error = "La nota definitiva debe ser un número entre 0 y 20";
        } else {
            // Si la nota final es >=10, marcar como finalizada
            if (!empty($nota_final) && $nota_final >= 10) {
                $estado = 'finalizada';
            }
        }
    } else { // Tipo aprobado/reprobado
        // Validar que sean valores permitidos
        $valores_permitidos = ['aprobado', 'reprobado'];
        if (!empty($nota_final) && !in_array(strtolower($nota_final), $valores_permitidos)) {
            $error = "La nota final debe ser 'aprobado' o 'reprobado'";
        } elseif (!empty($nota_definitiva) && !in_array(strtolower($nota_definitiva), $valores_permitidos)) {
            $error = "La nota definitiva debe ser 'aprobado' o 'reprobado'";
        } else {
            // Si está aprobado, marcar como finalizada
            if (!empty($nota_final) && strtolower($nota_final) == 'aprobado') {
                $estado = 'finalizada';
            }
        }
    }
    
    // Si no hay errores, actualizar la base de datos
    if (!isset($error)) {
        $sql = "UPDATE inscripciones 
                SET nota_final = ?, nota_definitiva = ?, estado = ?,
                    escolaridad_id = ?, año_lectivo_id = ?, seccion_id = ?,
                    fecha_inscripcion = ?
                WHERE id = ?";
        $stmt = $conn->prepare($sql);
        
        if ($inscripcion['tipo'] == 'puntaje') {
            $stmt->bind_param("sssiisss", 
                $nota_final, $nota_definitiva, $estado,
                $escolaridad_id, $anio_lectivo_id, $seccion_id,
                $fecha_inscripcion, $inscripcion_id);
        } else {
            // Convertir a minúsculas para consistencia
            $nota_final = !empty($nota_final) ? strtolower($nota_final) : null;
            $nota_definitiva = !empty($nota_definitiva) ? strtolower($nota_definitiva) : null;
            $stmt->bind_param("sssiisss", 
                $nota_final, $nota_definitiva, $estado,
                $escolaridad_id, $anio_lectivo_id, $seccion_id,
                $fecha_inscripcion, $inscripcion_id);
        }
        
        if ($stmt->execute()) {
            $mensaje = "Inscripción actualizada correctamente";
            // Actualizar los datos mostrados
            $inscripcion['nota_final'] = $nota_final;
            $inscripcion['nota_definitiva'] = $nota_definitiva;
            $inscripcion['estado'] = $estado;
            $inscripcion['escolaridad_id'] = $escolaridad_id;
            $inscripcion['año_lectivo_id'] = $anio_lectivo_id;
            $inscripcion['seccion_id'] = $seccion_id;
            $inscripcion['fecha_inscripcion'] = $fecha_inscripcion;
        } else {
            $error = "Error al actualizar la inscripción: " . $stmt->error;
        }
    }
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Editar Inscripción</title>
    <link rel="stylesheet" href="../css/styles.css">
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 20px;
        }
        .form-container {
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
            border: 1px solid #ddd;
            border-radius: 5px;
            background-color: #f9f9f9;
        }
        .form-group {
            margin-bottom: 15px;
        }
        label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
        }
        input, select {
            width: 100%;
            padding: 8px;
            border: 1px solid #ddd;
            border-radius: 4px;
        }
        .error {
            color: red;
            margin-bottom: 15px;
        }
        .success {
            color: green;
            margin-bottom: 15px;
        }
        .btn {
            padding: 8px 15px;
            background-color: #4CAF50;
            color: white;
            border: none;
            border-radius: 4px;
            cursor: pointer;
        }
        .btn:hover {
            background-color: #45a049;
        }
        .info-box {
            background-color: #f0f0f0;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        .form-row {
            display: flex;
            flex-wrap: wrap;
            margin: 0 -10px;
        }
        .form-col {
            flex: 1;
            padding: 0 10px;
            min-width: 200px;
        }
    </style>
</head>
<body>
    <h1>Editar Inscripción</h1>
    <a href="gestion_inscripciones.php">← Volver a Inscripciones</a>

    <div class="form-container">
        <!-- Mostrar información de la inscripción -->
        <div class="info-box">
            <h3>Información de la Inscripción</h3>
            <p><strong>Estudiante:</strong> <?php echo htmlspecialchars($inscripcion['nombres'] . ' ' . $inscripcion['apellidos']); ?></p>
            <p><strong>Cédula:</strong> <?php echo htmlspecialchars($inscripcion['cedula']); ?></p>
            <p><strong>Postgrado:</strong> <?php echo htmlspecialchars($inscripcion['postgrado_nombre']); ?></p>
            <p><strong>Denominación:</strong> <?php echo htmlspecialchars($inscripcion['denominacion_nombre']); ?></p>
            <p><strong>Tipo:</strong> <?php echo ucfirst($inscripcion['tipo']); ?></p>
            <p><strong>Estado actual:</strong> <?php echo ucfirst($inscripcion['estado']); ?></p>
        </div>

        <!-- Mostrar mensajes de error o éxito -->
        <?php if (isset($error)): ?>
            <div class="error"><?php echo $error; ?></div>
        <?php endif; ?>
        
        <?php if (isset($mensaje)): ?>
            <div class="success"><?php echo $mensaje; ?></div>
        <?php endif; ?>

        <!-- Formulario de edición -->
        <form method="post">
            <div class="form-row">
                <div class="form-col">
                    <div class="form-group">
                        <label for="escolaridad_id">Escolaridad:</label>
                        <select id="escolaridad_id" name="escolaridad_id" required>
                            <?php while($escolaridad = $result_escolaridades->fetch_assoc()): ?>
                                <option value="<?php echo $escolaridad['id']; ?>" 
                                    <?php echo $inscripcion['escolaridad_id'] == $escolaridad['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($escolaridad['nombre']); ?>
                                </option>
                            <?php endwhile; ?>
                        </select>
                    </div>
                </div>
                
                <div class="form-col">
                    <div class="form-group">
                        <label for="anio_lectivo_id">Año Lectivo:</label>
                        <select id="anio_lectivo_id" name="anio_lectivo_id" required>
                            <?php while($anio = $result_anios->fetch_assoc()): ?>
                                <option value="<?php echo $anio['id']; ?>" 
                                    <?php echo $inscripcion['año_lectivo_id'] == $anio['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($anio['nombre']); ?>
                                </option>
                            <?php endwhile; ?>
                        </select>
                    </div>
                </div>
            </div>

            <div class="form-row">
                <div class="form-col">
                    <div class="form-group">
                        <label for="seccion_id">Sección:</label>
                        <select id="seccion_id" name="seccion_id" required>
                            <?php while($seccion = $result_secciones->fetch_assoc()): ?>
                                <option value="<?php echo $seccion['id']; ?>" 
                                    <?php echo $inscripcion['seccion_id'] == $seccion['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($seccion['nombre']); ?>
                                </option>
                            <?php endwhile; ?>
                        </select>
                    </div>
                </div>
                
                <div class="form-col">
                    <div class="form-group">
                        <label for="fecha_inscripcion">Fecha de Inscripción:</label>
                        <input type="date" id="fecha_inscripcion" name="fecha_inscripcion" 
                               value="<?php echo htmlspecialchars($inscripcion['fecha_inscripcion']); ?>" required>
                    </div>
                </div>
            </div>

            <div class="form-row">
                <div class="form-col">
                    <div class="form-group">
                        <label for="nota_final">Nota Final:</label>
                        <?php if ($inscripcion['tipo'] == 'puntaje'): ?>
                            <input type="number" id="nota_final" name="nota_final" 
                                   min="0" max="20" step="0.01"
                                   value="<?php echo htmlspecialchars($inscripcion['nota_final'] ?? ''); ?>">
                        <?php else: ?>
                            <select id="nota_final" name="nota_final">
                                <option value="">-- Seleccione --</option>
                                <option value="aprobado" <?php echo (isset($inscripcion['nota_final']) && strtolower($inscripcion['nota_final']) == 'aprobado') ? 'selected' : ''; ?>>Aprobado</option>
                                <option value="reprobado" <?php echo (isset($inscripcion['nota_final']) && strtolower($inscripcion['nota_final']) == 'reprobado') ? 'selected' : ''; ?>>Reprobado</option>
                            </select>
                        <?php endif; ?>
                    </div>
                </div>
                
                <div class="form-col">
                    <div class="form-group">
                        <label for="nota_definitiva">Nota Definitiva:</label>
                        <?php if ($inscripcion['tipo'] == 'puntaje'): ?>
                            <input type="number" id="nota_definitiva" name="nota_definitiva" 
                                   min="0" max="20" step="0.01"
                                   value="<?php echo htmlspecialchars($inscripcion['nota_definitiva'] ?? ''); ?>">
                        <?php else: ?>
                            <select id="nota_definitiva" name="nota_definitiva">
                                <option value="">-- Seleccione --</option>
                                <option value="aprobado" <?php echo (isset($inscripcion['nota_definitiva']) && strtolower($inscripcion['nota_definitiva']) == 'aprobado') ? 'selected' : ''; ?>>Aprobado</option>
                                <option value="reprobado" <?php echo (isset($inscripcion['nota_definitiva']) && strtolower($inscripcion['nota_definitiva']) == 'reprobado') ? 'selected' : ''; ?>>Reprobado</option>
                            </select>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <div class="form-group">
                <label for="estado">Estado:</label>
                <select id="estado" name="estado" required>
                    <option value="activa" <?php echo $inscripcion['estado'] == 'activa' ? 'selected' : ''; ?>>Activa</option>
                    <option value="finalizada" <?php echo $inscripcion['estado'] == 'finalizada' ? 'selected' : ''; ?>>Finalizada</option>
                </select>
            </div>

            <button type="submit" name="actualizar" class="btn">Actualizar Inscripción</button>
        </form>
    </div>

    <script>
        // Validación del formulario antes de enviar
        document.querySelector('form').addEventListener('submit', function(e) {
            const tipo = '<?php echo $inscripcion["tipo"]; ?>';
            let isValid = true;
            
            if (tipo == 'puntaje') {
                const notaFinal = document.getElementById('nota_final').value;
                const notaDefinitiva = document.getElementById('nota_definitiva').value;
                
                if (notaFinal && (notaFinal < 0 || notaFinal > 20)) {
                    alert('La nota final debe estar entre 0 y 20');
                    isValid = false;
                }
                
                if (notaDefinitiva && (notaDefinitiva < 0 || notaDefinitiva > 20)) {
                    alert('La nota definitiva debe estar entre 0 y 20');
                    isValid = false;
                }
            }
            
            if (!isValid) {
                e.preventDefault();
            }
        });
    </script>
</body>
</html>