<?php
session_start();

if (!isset($_SESSION['username']) || $_SESSION['rol'] != 'administrador') {
    header("Location: ../login.php");
    exit();
}

include '../includes/db.php';

// Obtener el ID del estudiante
$estudiante_id = isset($_GET['estudiante_id']) ? intval($_GET['estudiante_id']) : 0;

if ($estudiante_id <= 0) {
    die("ID de estudiante no válido.");
}

// Obtener detalles del estudiante (consulta corregida eliminando la referencia a promocion_id)
$sql_detalles = "SELECT e.* 
                FROM estudiantes e
                WHERE e.id = ?";
$stmt_detalles = $conn->prepare($sql_detalles);

if (!$stmt_detalles) {
    die("Error en la preparación de la consulta de detalles: " . $conn->error);
}

$stmt_detalles->bind_param("i", $estudiante_id);
$stmt_detalles->execute();
$detalles_estudiante = $stmt_detalles->get_result()->fetch_assoc();

if (!$detalles_estudiante) {
    die("Estudiante no encontrado.");
}

// Obtener historial académico del estudiante agrupado por postgrado
$sql_historial = "SELECT p.nombre AS postgrado, d.nombre AS denominacion, d.credito, d.periodo, 
                         i.nota_definitiva
                  FROM inscripciones i
                  JOIN denominaciones d ON i.denominacion_id = d.id
                  JOIN postgrados p ON d.postgrado_id = p.id
                  WHERE i.estudiante_id = ?
                  ORDER BY p.nombre, d.periodo";
$stmt_historial = $conn->prepare($sql_historial);

if (!$stmt_historial) {
    die("Error en la preparación de la consulta de historial: " . $conn->error);
}

$stmt_historial->bind_param("i", $estudiante_id);
$stmt_historial->execute();
$historial_academico = $stmt_historial->get_result()->fetch_all(MYSQLI_ASSOC);

// Agrupar el historial por postgrado
$historial_por_postgrado = [];
foreach ($historial_academico as $historial) {
    $postgrado = $historial['postgrado'];
    if (!isset($historial_por_postgrado[$postgrado])) {
        $historial_por_postgrado[$postgrado] = [];
    }
    $historial_por_postgrado[$postgrado][] = $historial;
}

// Calcular nota ponderada, nota simple, total de asignaturas y créditos
$nota_ponderada = 0;
$nota_simple = 0;
$total_asignaturas = 0;
$total_creditos = 0;

foreach ($historial_academico as $historial) {
    if (is_numeric($historial['nota_definitiva'])) {
        $nota_simple += $historial['nota_definitiva'];
        $nota_ponderada += ($historial['nota_definitiva'] * $historial['credito']);
        $total_creditos += $historial['credito'];
        $total_asignaturas++;
    }
}

$nota_simple_promedio = $total_asignaturas > 0 ? $nota_simple / $total_asignaturas : 0;
$nota_ponderada_promedio = $total_creditos > 0 ? $nota_ponderada / $total_creditos : 0;
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Detalles del Estudiante</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 20px;
        }
        .details-container {
            margin-top: 20px;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
        }
        th, td {
            border: 1px solid #ddd;
            padding: 8px;
            text-align: left;
        }
        th {
            background-color: #f4f4f4;
        }
        .resumen {
            margin-top: 20px;
            padding: 10px;
            background-color: #f9f9f9;
            border: 1px solid #ddd;
        }
        .tab {
            overflow: hidden;
            border: 1px solid #ccc;
            background-color: #f1f1f1;
        }
        .tab button {
            background-color: inherit;
            float: left;
            border: none;
            outline: none;
            cursor: pointer;
            padding: 10px 16px;
            transition: 0.3s;
        }
        .tab button:hover {
            background-color: #ddd;
        }
        .tab button.active {
            background-color: #ccc;
        }
        .tabcontent {
            display: none;
            padding: 6px 12px;
            border: 1px solid #ccc;
            border-top: none;
        }
    </style>
</head>
<body>
    <h1>Detalles del Estudiante</h1>

    <!-- Enlace para volver al listado -->
    <a href="kardex_estudiantes.php">Volver al Listado</a>

    <div class="details-container">
        <h2>Información Personal</h2>
        <p><strong>Nombre:</strong> <?= htmlspecialchars($detalles_estudiante['nombres']) ?></p>
        <p><strong>Apellidos:</strong> <?= htmlspecialchars($detalles_estudiante['apellidos']) ?></p>
        <p><strong>Cédula:</strong> <?= htmlspecialchars($detalles_estudiante['cedula']) ?></p>
        <p><strong>Correo:</strong> <?= htmlspecialchars($detalles_estudiante['correo']) ?></p>
        <p><strong>Teléfono:</strong> <?= htmlspecialchars($detalles_estudiante['telefono']) ?></p>
        <p><strong>Celular:</strong> <?= htmlspecialchars($detalles_estudiante['celular']) ?></p>
        <p><strong>Dirección:</strong> <?= htmlspecialchars($detalles_estudiante['direccion']) ?></p>
        <p><strong>Graduado:</strong> <?= isset($detalles_estudiante['graduado']) ? ($detalles_estudiante['graduado'] ? 'Sí' : 'No') : 'No' ?></p>
        <?php if (isset($detalles_estudiante['graduado']) && $detalles_estudiante['graduado'] && isset($detalles_estudiante['fecha_grado'])): ?>
            <p><strong>Fecha de Grado:</strong> <?= htmlspecialchars($detalles_estudiante['fecha_grado']) ?></p>
        <?php endif; ?>
    </div>

    <h2>Historial Académico</h2>
    <div class="tab">
        <?php foreach ($historial_por_postgrado as $postgrado => $historial): ?>
            <button class="tablinks" onclick="openTab(event, '<?= $postgrado ?>')"><?= $postgrado ?></button>
        <?php endforeach; ?>
    </div>

    <?php foreach ($historial_por_postgrado as $postgrado => $historial): ?>
        <div id="<?= $postgrado ?>" class="tabcontent">
            <h3><?= $postgrado ?></h3>
            <table>
                <tr>
                    <th>Denominación</th>
                    <th>Créditos</th>
                    <th>Período</th>
                    <th>Nota Definitiva</th>
                </tr>
                <?php foreach ($historial as $materia): ?>
                    <tr>
                        <td><?= htmlspecialchars($materia['denominacion']) ?></td>
                        <td><?= htmlspecialchars($materia['credito']) ?></td>
                        <td><?= htmlspecialchars($materia['periodo']) ?></td>
                        <td><?= htmlspecialchars($materia['nota_definitiva']) ?></td>
                    </tr>
                <?php endforeach; ?>
            </table>
        </div>
    <?php endforeach; ?>

    <!-- Resumen del historial académico -->
    <div class="resumen">
        <h3>Resumen Académico</h3>
        <p><strong>Nota Simple Promedio:</strong> <?= number_format($nota_simple_promedio, 2) ?></p>
        <p><strong>Nota Ponderada Promedio:</strong> <?= number_format($nota_ponderada_promedio, 2) ?></p>
        <p><strong>Total de Asignaturas:</strong> <?= $total_asignaturas ?></p>
        <p><strong>Total de Créditos:</strong> <?= $total_creditos ?></p>
    </div>

    <script>
        function openTab(evt, tabName) {
            var i, tabcontent, tablinks;
            tabcontent = document.getElementsByClassName("tabcontent");
            for (i = 0; i < tabcontent.length; i++) {
                tabcontent[i].style.display = "none";
            }
            tablinks = document.getElementsByClassName("tablinks");
            for (i = 0; i < tablinks.length; i++) {
                tablinks[i].className = tablinks[i].className.replace(" active", "");
            }
            document.getElementById(tabName).style.display = "block";
            evt.currentTarget.className += " active";
        }

        // Mostrar la primera pestaña por defecto
        document.getElementsByClassName("tablinks")[0].click();
    </script>
</body>
</html>