<?php
session_start();

// Verificar sesión y rol de administrador
if (!isset($_SESSION['username']) || $_SESSION['rol'] != 'administrador') {
    header("Location: ../login.php");
    exit();
}

// Configuración para UTF-8
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');

// Verificar e incluir mPDF
$mpdfPath = __DIR__ . '/../vendor/autoload.php';
if (!file_exists($mpdfPath)) {
    die("Error: La biblioteca mPDF no está instalada. Ejecuta 'composer require mpdf/mpdf' para instalarla.");
}
require_once $mpdfPath;

include '../includes/db.php';

// Configurar conexión para usar UTF-8
if ($conn) {
    $conn->set_charset("utf8mb4");
    $conn->query("SET NAMES 'utf8mb4'");
    $conn->query("SET CHARACTER SET utf8mb4");
    $conn->query("SET COLLATION_CONNECTION = 'utf8mb4_unicode_ci'");
}

// Verificar si se proporcionó el estudiante_id y postgrado_id en la URL
if (!isset($_GET['estudiante_id']) || !isset($_GET['postgrado_id']) || !isset($_GET['fecha_constancia'])) {
    die("Error: No se proporcionó el ID del estudiante, del postgrado o la fecha de la constancia.");
}

// Inicializar variables
$estudiante_id = intval($_GET['estudiante_id']);
$postgrado_id = intval($_GET['postgrado_id']);
$fecha_constancia = $_GET['fecha_constancia'];
$detalles_estudiante = [];
$historial_academico = [];
$firmas = [];

// Validar fecha
if (!DateTime::createFromFormat('Y-m-d', $fecha_constancia)) {
    die("ERROR: Fecha inválida");
}

// Función para convertir números a letras (igual que constancia_notas.php)
function numeroALetras($numero) {
    $unidades = ['', 'UNO', 'DOS', 'TRES', 'CUATRO', 'CINCO', 'SEIS', 'SIETE', 'OCHO', 'NUEVE'];
    $decenas = ['', 'DIEZ', 'VEINTE', 'TREINTA', 'CUARENTA', 'CINCUENTA', 'SESENTA', 'SETENTA', 'OCHENTA', 'NOVENTA'];
    $especiales = ['', 'ONCE', 'DOCE', 'TRECE', 'CATORCE', 'QUINCE', 'DIECISEIS', 'DIECISIETE', 'DIECIOCHO', 'DIECINUEVE'];
    
    if ($numero == 0) return 'CERO';
    if ($numero == 10) return 'DIEZ';
    if ($numero == 20) return 'VEINTE';
    
    if ($numero < 10) return $unidades[$numero];
    if ($numero > 10 && $numero < 20) return $especiales[$numero - 10];
    
    $unidad = $numero % 10;
    $decena = floor($numero / 10);
    
    if ($unidad == 0) {
        return $decenas[$decena];
    } else {
        return $decenas[$decena] . ($decena == 2 ? 'I' : '') . $unidades[$unidad];
    }
}

// Función para formatear fecha en formato día/mes/año
function formatearFecha($fecha) {
    if (empty($fecha)) {
        return 'N/A';
    }
    
    try {
        $fechaObj = new DateTime($fecha);
        return $fechaObj->format('d/m/Y');
    } catch (Exception $e) {
        return 'N/A';
    }
}

// Función para determinar el tipo de documento y obtener el valor
function obtenerDocumentoIdentidad($estudiante) {
    $tipo_documento = $estudiante['tipo_documento'] ?? 'V';
    $valor_documento = '';
    $etiqueta_documento = '';
    
    switch ($tipo_documento) {
        case 'V':
            $valor_documento = $estudiante['cedula_venezolana'] ?? $estudiante['cedula'];
            $etiqueta_documento = 'CÉDULA';
            break;
        case 'E':
            $valor_documento = $estudiante['cedula_extranjera'] ?? $estudiante['cedula'];
            $etiqueta_documento = 'CÉDULA';
            break;
        case 'P':
            $valor_documento = $estudiante['pasaporte'] ?? $estudiante['cedula'];
            $etiqueta_documento = 'PASAPORTE';
            break;
        default:
            $valor_documento = $estudiante['cedula'];
            $etiqueta_documento = 'CÉDULA';
    }
    
    return [
        'etiqueta' => $etiqueta_documento,
        'valor' => $valor_documento
    ];
}

if ($estudiante_id > 0 && $postgrado_id > 0) {
    // Obtener detalles del estudiante (incluyendo los nuevos campos de documentos)
    $sql_detalles = "SELECT e.*, p.nombre AS postgrado_nombre, p.id AS postgrado_id,
                    ep.fecha_graduacion, pr.nombre AS promocion
                    FROM estudiantes e
                    JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                    JOIN postgrados p ON ep.postgrado_id = p.id
                    LEFT JOIN promociones pr ON ep.promocion_id = pr.id
                    WHERE e.id = ? AND p.id = ? AND (ep.estado = 'graduado' OR ep.estado = 'activo')
                    LIMIT 1";
    
    $stmt_detalles = $conn->prepare($sql_detalles);
    if ($stmt_detalles) {
        $stmt_detalles->bind_param("ii", $estudiante_id, $postgrado_id);
        $stmt_detalles->execute();
        $result_detalles = $stmt_detalles->get_result();
        $detalles_estudiante = $result_detalles->fetch_assoc() ?: [];
        $stmt_detalles->close();
    }

    if (empty($detalles_estudiante)) {
        die("Error: No se encontró al estudiante o no está activo/graduado en el postgrado seleccionado.");
    }

    // Obtener total de asignaturas inscritas (incluyendo las sin nota)
    $sql_todas_inscripciones = "SELECT COUNT(DISTINCT i.denominacion_id) AS total_inscritas
                               FROM inscripciones i
                               JOIN denominaciones d ON i.denominacion_id = d.id
                               WHERE i.estudiante_id = ?
                               AND d.postgrado_id = ?";
    
    $stmt_todas = $conn->prepare($sql_todas_inscripciones);
    if ($stmt_todas) {
        $stmt_todas->bind_param("ii", $estudiante_id, $postgrado_id);
        $stmt_todas->execute();
        $result_todas = $stmt_todas->get_result();
        $total_inscritas_row = $result_todas->fetch_assoc();
        $total_asignaturas = $total_inscritas_row['total_inscritas'] ?? 0;
        $stmt_todas->close();
    }

    // Obtener historial académico (incluyendo asignaturas sin nota)
    $sql_historial = "SELECT d.codigo AS asignatura, 
                             d.nombre AS denominacion, 
                             d.credito,
                             d.tipo,
                             i.nota_definitiva, 
                             i.nota_final,
                             al.nombre AS año_lectivo
                      FROM inscripciones i
                      INNER JOIN denominaciones d ON i.denominacion_id = d.id
                      LEFT JOIN años_lectivos al ON i.año_lectivo_id = al.id
                      WHERE i.estudiante_id = ? 
                      AND d.postgrado_id = ?
                      ORDER BY d.periodo, d.codigo";

    $stmt_historial = $conn->prepare($sql_historial);
    if ($stmt_historial) {
        $stmt_historial->bind_param("ii", $estudiante_id, $postgrado_id);
        $stmt_historial->execute();
        $result_historial = $stmt_historial->get_result();
        $historial_academico = $result_historial->fetch_all(MYSQLI_ASSOC) ?: [];
        $stmt_historial->close();
    }

    // Obtener asignaturas aprobadas
    $sql_aprobadas = "SELECT COUNT(DISTINCT i.denominacion_id) AS total_aprobadas
                     FROM inscripciones i
                     INNER JOIN denominaciones d ON i.denominacion_id = d.id
                     WHERE i.estudiante_id = ? 
                     AND d.postgrado_id = ?
                     AND i.nota_definitiva IS NOT NULL
                     AND (
                         (d.tipo = 'puntaje' AND CAST(i.nota_definitiva AS DECIMAL(10,2)) >= 10) OR
                         (d.tipo = 'Aprobado_Reprobado' AND UPPER(i.nota_definitiva) = 'APROBADO')
                     )";
    
    $stmt_aprobadas = $conn->prepare($sql_aprobadas);
    $asignaturas_aprobadas = 0;
    if ($stmt_aprobadas) {
        $stmt_aprobadas->bind_param("ii", $estudiante_id, $postgrado_id);
        $stmt_aprobadas->execute();
        $result_aprobadas = $stmt_aprobadas->get_result();
        $row = $result_aprobadas->fetch_assoc();
        $asignaturas_aprobadas = $row['total_aprobadas'] ?? 0;
        $stmt_aprobadas->close();
    }

    // Calcular promedios
    $total_creditos = 0;
    $nota_simple = 0;
    $nota_ponderada = 0;
    $asignaturas_puntaje_aprobadas = 0;

    foreach ($historial_academico as $h) {
        if ($h['tipo'] == 'puntaje' && is_numeric($h['nota_definitiva']) && floatval($h['nota_definitiva']) >= 10) {
            $valor = floatval($h['nota_definitiva']);
            $credito = intval($h['credito']);
            $nota_simple += $valor;
            $nota_ponderada += ($valor * $credito);
            $total_creditos += $credito;
            $asignaturas_puntaje_aprobadas++;
        }
    }

    $nota_simple_promedio = $asignaturas_puntaje_aprobadas > 0 ? $nota_simple / $asignaturas_puntaje_aprobadas : 0;
    $nota_ponderada_promedio = $total_creditos > 0 ? $nota_ponderada / $total_creditos : 0;
}

// Obtener firmas de la misma tabla que constancia_notas.php (firmas_constancia)
$sql_firmas = "SELECT nombre, cargo FROM firmas_constancia ORDER BY id";
$result_firmas = $conn->query($sql_firmas);

if ($result_firmas && $result_firmas->num_rows > 0) {
    $firmas = $result_firmas->fetch_all(MYSQLI_ASSOC);
} else {
    // Firmas por defecto si no hay en la tabla
    $firmas = [
        ['nombre' => 'Prof. MARIO PATIÑO TORRES', 'cargo' => 'DECANO'],
        ['nombre' => 'Prof. JOSÈ RAMÓN GARCÍA RODRÍGUEZ', 'cargo' => 'COORDINADOR DE ESTUDIOS DE POSTGRADO MED'],
        ['nombre' => 'Ing. ZULAY DÍAZ', 'cargo' => 'ÁREA DE POSTGRADO CENTRAL'],
        ['nombre' => 'JESSE LEE HERNÁNDEZ', 'cargo' => 'CONTROL DE ESTUDIOS DE POSTGRADO MED']
    ];
}

// Obtener información del documento de identidad
$documento_info = obtenerDocumentoIdentidad($detalles_estudiante);

try {
    // Configurar PDF en tamaño CARTA
    $mpdf = new \Mpdf\Mpdf([
        'mode' => 'utf-8',
        'format' => 'Letter', // Tamaño carta
        'margin_left' => 10,
        'margin_right' => 10,
        'margin_top' => 15,
        'margin_bottom' => 15,
        'default_font_size' => 10,
        'default_font' => 'arial',
        'tempDir' => __DIR__ . '/tmp'
    ]);

    $mpdf->SetTitle('Constancia de Estudio - ' . $detalles_estudiante['nombres'] . ' ' . $detalles_estudiante['apellidos']);
    $mpdf->SetAuthor('Universidad Central de Venezuela');
    $mpdf->SetCreator('Sistema de Postgrado');

    $html = '<!DOCTYPE html>
    <html lang="es">
    <head>
        <meta charset="UTF-8">
        <title>Constancia de Estudio</title>
        <style>
            body { font-family: Arial, sans-serif; margin: 0; padding: 0; line-height: 1.2; }
            .constancia { width: 100%; padding: 20px; box-sizing: border-box; }
            .header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px; }
            .header h1 { font-size: 14px; margin: 0; text-align: left; line-height: 1.3; }
            .header h2 { font-size: 12px; margin: 0; text-align: right; line-height: 1.3; }
            .datos-estudiante { margin-bottom: 15px; text-align: left; }
            .datos-estudiante p { margin: 3px 0; font-size: 12px; line-height: 1.3; }
            .linea-negra { border-top: 2px solid #000; margin: 15px 0; }
            table { width: 100%; border-collapse: collapse; margin-top: 8px; page-break-inside: avoid; font-size: 12px; line-height: 1.2; }
            th, td { padding: 4px; text-align: left; border: 1px solid #ddd; }
            th { background-color: #f4f4f4; }
            .resumen { margin: 30px 0 0 0; width: 100%; text-align: left; }
            .resumen h3 { text-align: center; margin-bottom: 10px; font-size: 13px; }
            .firmas { margin-top: 100px; text-align: center; }
            .firmas table { width: 100%; border-collapse: collapse; border: none; }
            .firmas td { width: 50%; text-align: center; vertical-align: top; padding: 40px 20px; border: none; } /* Aumentado el padding vertical */
            .firmas p { margin: 0; font-size: 12px; line-height: 1.3; }
            .page-break { page-break-after: always; }
            .avoid-break { page-break-inside: avoid; }
            .resumen-tabla { 
                margin: 50px 0 0 40px;  /* 40px de margen izquierdo */
                width: 65%;
                border: none;
            }
            .resumen-tabla td { 
                border: none; 
                padding: 4px 0px;  /* Padding mínimo para acercar cifras */
                font-size: 12px; 
                vertical-align: top; 
                line-height: 1.1;  /* Reducir interlineado */
            }
            .resumen-label { 
                text-align: left; 
                width: 50%;  /* Más espacio para descripción */
                padding-right: 1px;  /* Espacio mínimo entre descripción y valor */
            }
            .resumen-value { 
                text-align: right; 
                width: 15%;  /* Menos espacio para valores */
                font-weight: bold; 
                padding-left: 1px;  /* Espacio mínimo */
            }
            .espacio-firma { 
                height: 60px; /* Espacio adicional para firmas */
                margin: 5px 0;
            }
        </style>
    </head>
    <body>';

    if ($estudiante_id > 0 && $postgrado_id > 0 && !empty($detalles_estudiante)) {
        $denominaciones_por_pagina = 23;
        $total_paginas_denominaciones = ceil(count($historial_academico) / $denominaciones_por_pagina);
        $total_paginas = $total_paginas_denominaciones + 1;

        // Primero generamos todas las páginas de asignaturas
        for ($pagina = 1; $pagina <= $total_paginas_denominaciones; $pagina++) {
            $html .= '
            <div class="constancia' . ($pagina < $total_paginas_denominaciones ? ' page-break' : '') . '">
                <div class="header">
                    <h1>UNIVERSIDAD CENTRAL DE VENEZUELA<br>OFICINA CENTRAL DE CONTROL DE ESTUDIOS<br>ÁREA DE POSTGRADO</h1>
                    <h2>FECHA: ' . formatearFecha($fecha_constancia) . '<br>Página ' . $pagina . ' de ' . $total_paginas . '</h2>
                </div>

                <div class="datos-estudiante">
                    <h3 style="text-align: center;">*** CONSTANCIA DE ESTUDIO ***</h3>
                    <p><strong>FACULTAD DE MEDICINA</strong></p>
                    <p><strong>POSTGRADO DE MEDICINA</strong></p>
                    <p>&nbsp;</p>
                    <p><strong>ESPECIALISTA EN:</strong> ' . htmlspecialchars($detalles_estudiante['postgrado_nombre'] ?? 'ANESTESIOLOGÍA – HDL') . '</p>
                    <p><strong>ESTUDIANTE:</strong> ' . htmlspecialchars($detalles_estudiante['nombres'] . ' ' . $detalles_estudiante['apellidos']) . '</p>
                    <p><strong>' . $documento_info['etiqueta'] . ':</strong> ' . htmlspecialchars($documento_info['valor']) . '</p>
                </div>

                <div class="linea-negra"></div>

                <table>
                    <thead>
                        <tr>
                            <th>ASIGNATURA</th>
                            <th>DENOMINACIÓN</th>
                            <th>CRÉDITO</th>
                            <th>NOTA DEFINITIVA</th>
                            <th>AÑO LECTIVO</th>
                        </tr>
                    </thead>
                    <tbody>';

            $inicio = ($pagina - 1) * $denominaciones_por_pagina;
            $denominaciones_pagina = array_slice($historial_academico, $inicio, $denominaciones_por_pagina);

            foreach ($denominaciones_pagina as $historial) {
                $nota_display = 'SIN NOTA';
                
                if ($historial['nota_definitiva'] !== null) {
                    if ($historial['tipo'] == 'Aprobado_Reprobado') {
                        $estado = strtoupper($historial['nota_definitiva']);
                        $letra = ($estado == 'APROBADO') ? 'A' : 'R';
                        $nota_display = $letra . ' (' . $estado . ')';
                    } elseif (is_numeric($historial['nota_definitiva'])) {
                        $nota_num = (int)$historial['nota_definitiva'];
                        $nota_display = $nota_num . ' (' . numeroALetras($nota_num) . ')';
                    }
                }
                
                $html .= '
                        <tr>
                            <td>' . htmlspecialchars($historial['asignatura']) . '</td>
                            <td>' . htmlspecialchars($historial['denominacion']) . '</td>
                            <td>' . htmlspecialchars($historial['credito']) . '</td>
                            <td>' . $nota_display . '</td>
                            <td>' . htmlspecialchars($historial['año_lectivo']) . '</td>
                        </tr>';
            }

            $html .= '
                    </tbody>
                </table>
            </div>';
        }

        // Ahora generamos la página final con el resumen y firmas
        $html .= '
        <div class="constancia avoid-break">
            <div class="header">
                <h1>UNIVERSIDAD CENTRAL DE VENEZUELA<br>OFICINA CENTRAL DE CONTROL DE ESTUDIOS<br>ÁREA DE POSTGRADO</h1>
                <h2>FECHA: ' . formatearFecha($fecha_constancia) . '<br>Página ' . $total_paginas . ' de ' . $total_paginas . '</h2>
            </div>

            <div class="datos-estudiante">
                <h3 style="text-align: center;">*** CONSTANCIA DE ESTUDIO ***</h3>
                <p><strong>FACULTAD DE MEDICINA</strong></p>
                <p><strong>POSTGRADO DE MEDICINA</strong></p>
                <p>&nbsp;</p>
                <p><strong>ESPECIALISTA EN:</strong> ' . htmlspecialchars($detalles_estudiante['postgrado_nombre'] ?? 'ANESTESIOLOGÍA – HDL') . '</p>
                <p><strong>ESTUDIANTE:</strong> ' . htmlspecialchars($detalles_estudiante['nombres'] . ' ' . $detalles_estudiante['apellidos']) . '</p>
                <p><strong>' . $documento_info['etiqueta'] . ':</strong> ' . htmlspecialchars($documento_info['valor']) . '</p>
            </div>

            <div class="avoid-break">
                <h3 style="text-align: center; margin-bottom: 20px;">RESUMEN ACADÉMICO</h3>
                <table class="resumen-tabla">
                    <tr>
                        <td class="resumen-label">NRO. DE UNIDADES INSCRITAS</td>
                        <td class="resumen-value">' . $total_asignaturas . '</td>
                    </tr>
                    <tr>
                        <td class="resumen-label">NRO. DE UNIDADES APROBADAS</td>
                        <td class="resumen-value">' . $asignaturas_aprobadas . '</td>
                    </tr>
                    <tr>
                        <td class="resumen-label">NRO. DE UNIDADES APROBADAS POR RECONOC. CRED</td>
                        <td class="resumen-value">0</td>
                    </tr>
                    <tr>
                        <td class="resumen-label">NRO. DE UNIDADES APROBADAS POR ASIG.CRED</td>
                        <td class="resumen-value">0</td>
                    </tr>
                    <tr>
                        <td class="resumen-label">PROMEDIO GENERAL DE NOTAS APROBADAS</td>
                        <td class="resumen-value">' . number_format($nota_simple_promedio, 2) . '</td>
                    </tr>
                    <tr>
                        <td class="resumen-label">PROMEDIO PONDERADO DE NOTAS APROBADAS</td>
                        <td class="resumen-value">' . number_format($nota_ponderada_promedio, 2) . '</td>
                    </tr>
                </table>
            </div>

            <div class="firmas avoid-break">
                <table>';

        foreach (array_chunk($firmas, 2) as $fila) {
            $html .= '<tr>';
            foreach ($fila as $firma) {
                $html .= '
                    <td>
                        <p><strong>' . htmlspecialchars($firma['nombre']) . '</strong></p>
                        <p>' . htmlspecialchars($firma['cargo']) . '</p>
                        <div class="espacio-firma"></div> <!-- Espacio adicional para firmar -->
                    </td>';
            }
            $html .= '</tr>';
        }

        $html .= '
                </table>
            </div>
        </div>';
    }

    $html .= '
    </body>
    </html>';

    $mpdf->WriteHTML($html);
    $filename = 'Constancia_Estudio_' . $documento_info['valor'] . '.pdf';
    $mpdf->Output($filename, \Mpdf\Output\Destination::INLINE);

} catch (\Mpdf\MpdfException $e) {
    die("Error al generar el PDF: " . $e->getMessage());
} finally {
    if ($conn) {
        $conn->close();
    }
}
?>