<?php
session_start();

if (!isset($_SESSION['username']) || $_SESSION['rol'] != 'administrador') {
    header("Location: ../login.php");
    exit();
}

include '../includes/db.php';

// Configurar conexión para UTF-8
$conn->set_charset("utf8mb4");

// Obtener parámetros de filtro
$usuario_id = isset($_GET['usuario_id']) ? intval($_GET['usuario_id']) : 0;
$escolaridad_id = isset($_GET['escolaridad_id']) ? intval($_GET['escolaridad_id']) : 0;
$postgrado_id = isset($_GET['postgrado_id']) ? intval($_GET['postgrado_id']) : 0;
$cuatrimestre = isset($_GET['cuatrimestre']) ? $_GET['cuatrimestre'] : '';
$ver_notas = isset($_GET['ver_notas']) ? $_GET['ver_notas'] : 'todas';
$vista = isset($_GET['vista']) ? $_GET['vista'] : 'resumen';

// Configuración de paginación para escolaridades
$registros_por_pagina = 20;
$pagina_actual = isset($_GET['pagina_esc']) ? (int)$_GET['pagina_esc'] : 1;
$offset_esc = ($pagina_actual - 1) * $registros_por_pagina;

// Obtener años escolares disponibles desde las escolaridades
$sql_escolaridades = "SELECT id, nombre FROM escolaridades ORDER BY nombre DESC";
$escolaridades = $conn->query($sql_escolaridades);

// Definir cuatrimestres del sistema
$cuatrimestres = [
    '01C' => '1º Cuatrimestre (Ene-Abr)',
    '02C' => '2º Cuatrimestre (May-Ago)', 
    '03C' => '3º Cuatrimestre (Sep-Dic)',
    'TODOS' => 'Todos los Cuatrimestres'
];

// Definir opciones para el filtro de notas
$opciones_notas = [
    'todas' => 'Todas las notas',
    'cargadas' => 'Solo notas cargadas',
    'pendientes' => 'Solo notas pendientes'
];

// Obtener listas para filtros
$sql_usuarios = "SELECT u.id, u.username, u.nombres, u.apellidos, u.rol 
                 FROM usuarios u 
                 WHERE u.rol IN ('coordinador1', 'coordinador2', 'administrador')
                 ORDER BY u.rol, u.nombres";
$usuarios = $conn->query($sql_usuarios);

$sql_postgrados = "SELECT DISTINCT p.id, p.nombre 
                   FROM postgrados p
                   WHERE p.estado = 'abierto'
                   ORDER BY p.nombre";
$postgrados = $conn->query($sql_postgrados);

// Obtener el nombre de la escolaridad seleccionada
$escolaridad_nombre = '';
if ($escolaridad_id > 0) {
    $sql_esc_nombre = "SELECT nombre FROM escolaridades WHERE id = ?";
    $stmt_esc = $conn->prepare($sql_esc_nombre);
    $stmt_esc->bind_param("i", $escolaridad_id);
    $stmt_esc->execute();
    $result_esc = $stmt_esc->get_result();
    if ($row_esc = $result_esc->fetch_assoc()) {
        $escolaridad_nombre = $row_esc['nombre'];
    }
}

// FUNCIÓN PARA OBTENER RANGO DE FECHAS DEL CUATRIMESTRE
function obtenerRangoCuatrimestre($cuatrimestre, $anio) {
    switch ($cuatrimestre) {
        case '01C':
            $fecha_inicio = $anio . '-01-01';
            $fecha_fin = $anio . '-04-30';
            break;
        case '02C':
            $fecha_inicio = $anio . '-05-01';
            $fecha_fin = $anio . '-08-31';
            break;
        case '03C':
            $fecha_inicio = $anio . '-09-01';
            $fecha_fin = $anio . '-12-31';
            break;
        default:
            $fecha_inicio = null;
            $fecha_fin = null;
    }
    
    return ['inicio' => $fecha_inicio, 'fin' => $fecha_fin];
}

// Obtener rango de fechas si hay cuatrimestre seleccionado
$rango_fechas = null;
if (!empty($cuatrimestre) && $cuatrimestre != 'TODOS') {
    $anio_filtro = !empty($escolaridad_nombre) ? $escolaridad_nombre : date('Y');
    $rango_fechas = obtenerRangoCuatrimestre($cuatrimestre, $anio_filtro);
}

// CONSULTA PARA RESUMEN GLOBAL
$sql_resumen_global = "SELECT 
                      COUNT(i.id) as total_inscripciones,
                      SUM(CASE WHEN i.nota_definitiva IS NOT NULL AND i.nota_definitiva != '' THEN 1 ELSE 0 END) as notas_cargadas,
                      SUM(CASE WHEN i.nota_definitiva IS NULL OR i.nota_definitiva = '' THEN 1 ELSE 0 END) as notas_pendientes,
                      COUNT(DISTINCT i.usuario_modificacion) as usuarios_activos
                      FROM inscripciones i
                      WHERE 1=1";

$conditions_global = [];
$params_global = [];
$types_global = "";

// Aplicar filtros básicos
if ($usuario_id > 0) {
    $conditions_global[] = " i.usuario_modificacion = ?";
    $params_global[] = $usuario_id;
    $types_global .= "i";
}

if ($escolaridad_id > 0) {
    $conditions_global[] = " i.escolaridad_id = ?";
    $params_global[] = $escolaridad_id;
    $types_global .= "i";
}

if ($postgrado_id > 0) {
    $conditions_global[] = " EXISTS (SELECT 1 FROM denominaciones d WHERE d.id = i.denominacion_id AND d.postgrado_id = ?)";
    $params_global[] = $postgrado_id;
    $types_global .= "i";
}

// Filtro por período académico (año_lectivo)
if (!empty($cuatrimestre) && $cuatrimestre != 'TODOS' && !empty($escolaridad_nombre)) {
    $patron_año_lectivo = $escolaridad_nombre . '-' . $cuatrimestre;
    $conditions_global[] = " EXISTS (SELECT 1 FROM años_lectivos al WHERE al.id = i.año_lectivo_id AND al.nombre = ?)";
    $params_global[] = $patron_año_lectivo;
    $types_global .= "s";
} elseif ($escolaridad_id > 0 && empty($cuatrimestre)) {
    $conditions_global[] = " EXISTS (SELECT 1 FROM años_lectivos al WHERE al.id = i.año_lectivo_id AND al.nombre LIKE ?)";
    $params_global[] = $escolaridad_nombre . '-%';
    $types_global .= "s";
}

// Filtro por tipo de notas
if ($ver_notas == 'cargadas') {
    $conditions_global[] = " i.nota_definitiva IS NOT NULL AND i.nota_definitiva != ''";
} elseif ($ver_notas == 'pendientes') {
    $conditions_global[] = " i.nota_definitiva IS NULL OR i.nota_definitiva = ''";
}

if (!empty($conditions_global)) {
    $sql_resumen_global .= " AND " . implode(" AND ", $conditions_global);
}

$stmt_global = $conn->prepare($sql_resumen_global);
if (!empty($params_global)) {
    $stmt_global->bind_param($types_global, ...$params_global);
}
$stmt_global->execute();
$resumen_global = $stmt_global->get_result()->fetch_assoc();

// Calcular porcentajes
$porcentaje_cargado = $resumen_global['total_inscripciones'] > 0 ? 
    round(($resumen_global['notas_cargadas'] / $resumen_global['total_inscripciones']) * 100, 2) : 0;

$porcentaje_pendiente = $resumen_global['total_inscripciones'] > 0 ? 
    round(($resumen_global['notas_pendientes'] / $resumen_global['total_inscripciones']) * 100, 2) : 0;

// CONSULTA PARA RESUMEN POR ESCOLARIDAD
$sql_resumen_escolaridad = "SELECT 
                            esc.nombre as escolaridad_nombre,
                            al.nombre as periodo_nombre,
                            COUNT(i.id) as total_inscripciones,
                            SUM(CASE WHEN i.nota_definitiva IS NOT NULL AND i.nota_definitiva != '' THEN 1 ELSE 0 END) as notas_cargadas,
                            SUM(CASE WHEN i.nota_definitiva IS NULL OR i.nota_definitiva = '' THEN 1 ELSE 0 END) as notas_pendientes
                            FROM inscripciones i
                            JOIN escolaridades esc ON i.escolaridad_id = esc.id
                            LEFT JOIN años_lectivos al ON i.año_lectivo_id = al.id
                            WHERE 1=1";

$conditions_esc = [];
$params_esc = [];
$types_esc = "";

if ($usuario_id > 0) {
    $conditions_esc[] = " i.usuario_modificacion = ?";
    $params_esc[] = $usuario_id;
    $types_esc .= "i";
}

if ($escolaridad_id > 0) {
    $conditions_esc[] = " i.escolaridad_id = ?";
    $params_esc[] = $escolaridad_id;
    $types_esc .= "i";
}

if ($postgrado_id > 0) {
    $conditions_esc[] = " EXISTS (SELECT 1 FROM denominaciones d WHERE d.id = i.denominacion_id AND d.postgrado_id = ?)";
    $params_esc[] = $postgrado_id;
    $types_esc .= "i";
}

// Filtro por período académico
if (!empty($cuatrimestre) && $cuatrimestre != 'TODOS' && !empty($escolaridad_nombre)) {
    $patron_año_lectivo = $escolaridad_nombre . '-' . $cuatrimestre;
    $conditions_esc[] = " al.nombre = ?";
    $params_esc[] = $patron_año_lectivo;
    $types_esc .= "s";
} elseif ($escolaridad_id > 0 && empty($cuatrimestre)) {
    $conditions_esc[] = " al.nombre LIKE ?";
    $params_esc[] = $escolaridad_nombre . '-%';
    $types_esc .= "s";
}

// Filtro por tipo de notas
if ($ver_notas == 'cargadas') {
    $conditions_esc[] = " i.nota_definitiva IS NOT NULL AND i.nota_definitiva != ''";
} elseif ($ver_notas == 'pendientes') {
    $conditions_esc[] = " i.nota_definitiva IS NULL OR i.nota_definitiva = ''";
}

if (!empty($conditions_esc)) {
    $sql_resumen_escolaridad .= " AND " . implode(" AND ", $conditions_esc);
}

$sql_resumen_escolaridad .= " GROUP BY esc.nombre, al.nombre ORDER BY esc.nombre, al.nombre";

// Consulta para contar total de registros (para paginación)
$sql_count_esc = "SELECT COUNT(*) as total FROM (
                    SELECT esc.nombre as esc_nombre, al.nombre as al_nombre
                    FROM inscripciones i
                    JOIN escolaridades esc ON i.escolaridad_id = esc.id
                    LEFT JOIN años_lectivos al ON i.año_lectivo_id = al.id
                    WHERE 1=1";

if (!empty($conditions_esc)) {
    $sql_count_esc .= " AND " . implode(" AND ", $conditions_esc);
}

$sql_count_esc .= " GROUP BY esc.nombre, al.nombre) as subquery";

$stmt_count_esc = $conn->prepare($sql_count_esc);
if (!empty($params_esc)) {
    $stmt_count_esc->bind_param($types_esc, ...$params_esc);
}
$stmt_count_esc->execute();
$result_count_esc = $stmt_count_esc->get_result();
$total_registros_esc = $result_count_esc->fetch_assoc()['total'];
$total_paginas_esc = ceil($total_registros_esc / $registros_por_pagina);

// Aplicar paginación a la consulta principal
$sql_resumen_escolaridad .= " LIMIT ?, ?";
$params_esc[] = $offset_esc;
$params_esc[] = $registros_por_pagina;
$types_esc .= "ii";

$stmt_esc = $conn->prepare($sql_resumen_escolaridad);
if (!empty($params_esc)) {
    $stmt_esc->bind_param($types_esc, ...$params_esc);
}
$stmt_esc->execute();
$resumen_escolaridades = $stmt_esc->get_result()->fetch_all(MYSQLI_ASSOC);

// CONSULTA ADICIONAL: RESUMEN POR POSTGRADO PARA SABER NOTAS PENDIENTES
$sql_resumen_postgrados = "SELECT 
                          p.id as postgrado_id,
                          p.nombre as postgrado_nombre,
                          COUNT(i.id) as total_inscripciones,
                          SUM(CASE WHEN i.nota_definitiva IS NOT NULL AND i.nota_definitiva != '' THEN 1 ELSE 0 END) as notas_cargadas,
                          SUM(CASE WHEN i.nota_definitiva IS NULL OR i.nota_definitiva = '' THEN 1 ELSE 0 END) as notas_pendientes,
                          GROUP_CONCAT(DISTINCT CONCAT(u.nombres, ' ', u.apellidos) SEPARATOR ', ') as coordinadores_responsables
                          FROM inscripciones i
                          JOIN denominaciones d ON i.denominacion_id = d.id
                          JOIN postgrados p ON d.postgrado_id = p.id
                          LEFT JOIN coordinador_postgrado cp ON p.id = cp.postgrado_id
                          LEFT JOIN usuarios u ON cp.coordinador_id = u.id
                          WHERE 1=1";

$conditions_postgrados = [];
$params_postgrados = [];
$types_postgrados = "";

if ($usuario_id > 0) {
    $conditions_postgrados[] = " i.usuario_modificacion = ?";
    $params_postgrados[] = $usuario_id;
    $types_postgrados .= "i";
}

if ($escolaridad_id > 0) {
    $conditions_postgrados[] = " i.escolaridad_id = ?";
    $params_postgrados[] = $escolaridad_id;
    $types_postgrados .= "i";
}

if ($postgrado_id > 0) {
    $conditions_postgrados[] = " p.id = ?";
    $params_postgrados[] = $postgrado_id;
    $types_postgrados .= "i";
}

// Filtro por período académico
if (!empty($cuatrimestre) && $cuatrimestre != 'TODOS' && !empty($escolaridad_nombre)) {
    $patron_año_lectivo = $escolaridad_nombre . '-' . $cuatrimestre;
    $conditions_postgrados[] = " EXISTS (SELECT 1 FROM años_lectivos al WHERE al.id = i.año_lectivo_id AND al.nombre = ?)";
    $params_postgrados[] = $patron_año_lectivo;
    $types_postgrados .= "s";
} elseif ($escolaridad_id > 0 && empty($cuatrimestre)) {
    $conditions_postgrados[] = " EXISTS (SELECT 1 FROM años_lectivos al WHERE al.id = i.año_lectivo_id AND al.nombre LIKE ?)";
    $params_postgrados[] = $escolaridad_nombre . '-%';
    $types_postgrados .= "s";
}

// Filtro por tipo de notas
if ($ver_notas == 'cargadas') {
    $conditions_postgrados[] = " i.nota_definitiva IS NOT NULL AND i.nota_definitiva != ''";
} elseif ($ver_notas == 'pendientes') {
    $conditions_postgrados[] = " i.nota_definitiva IS NULL OR i.nota_definitiva = ''";
}

if (!empty($conditions_postgrados)) {
    $sql_resumen_postgrados .= " AND " . implode(" AND ", $conditions_postgrados);
}

$sql_resumen_postgrados .= " GROUP BY p.id, p.nombre ORDER BY p.nombre";

$stmt_postgrados = $conn->prepare($sql_resumen_postgrados);
if (!empty($params_postgrados)) {
    $stmt_postgrados->bind_param($types_postgrados, ...$params_postgrados);
}
$stmt_postgrados->execute();
$resumen_postgrados = $stmt_postgrados->get_result()->fetch_all(MYSQLI_ASSOC);

// Consulta para fechas de carga
$sql_fechas_carga = "SELECT 
                    DATE(i.fecha_modificacion) as fecha_carga,
                    COUNT(i.id) as total_cargas,
                    COUNT(DISTINCT i.usuario_modificacion) as usuarios_activos,
                    GROUP_CONCAT(DISTINCT CONCAT(u.nombres, ' ', u.apellidos) SEPARATOR ', ') as responsables,
                    MIN(i.fecha_modificacion) as primera_carga,
                    MAX(i.fecha_modificacion) as ultima_carga
                    FROM inscripciones i
                    JOIN usuarios u ON i.usuario_modificacion = u.id
                    WHERE u.rol IN ('coordinador1', 'coordinador2', 'administrador')";

$conditions_fechas = [];
$params_fechas = [];
$types_fechas = "";

if ($usuario_id > 0) {
    $conditions_fechas[] = " i.usuario_modificacion = ?";
    $params_fechas[] = $usuario_id;
    $types_fechas .= "i";
}

if ($postgrado_id > 0) {
    $conditions_fechas[] = " EXISTS (SELECT 1 FROM denominaciones d WHERE d.id = i.denominacion_id AND d.postgrado_id = ?)";
    $params_fechas[] = $postgrado_id;
    $types_fechas .= "i";
}

if (!empty($rango_fechas)) {
    $conditions_fechas[] = " DATE(i.fecha_modificacion) BETWEEN ? AND ?";
    $params_fechas[] = $rango_fechas['inicio'];
    $params_fechas[] = $rango_fechas['fin'];
    $types_fechas .= "ss";
}

if ($ver_notas == 'cargadas') {
    $conditions_fechas[] = " i.nota_definitiva IS NOT NULL AND i.nota_definitiva != ''";
} elseif ($ver_notas == 'pendientes') {
    $conditions_fechas[] = " i.nota_definitiva IS NULL OR i.nota_definitiva = ''";
}

if (!empty($conditions_fechas)) {
    $sql_fechas_carga .= " AND " . implode(" AND ", $conditions_fechas);
}

$sql_fechas_carga .= " GROUP BY DATE(i.fecha_modificacion) ORDER BY fecha_carga DESC";

if (empty($rango_fechas)) {
    $sql_fechas_carga .= " LIMIT 30";
}

$stmt_fechas = $conn->prepare($sql_fechas_carga);
if (!empty($params_fechas)) {
    $stmt_fechas->bind_param($types_fechas, ...$params_fechas);
}
$stmt_fechas->execute();
$resumen_fechas = $stmt_fechas->get_result()->fetch_all(MYSQLI_ASSOC);

// Consulta para detalle de cargas
$sql_detalle = "SELECT 
                i.id,
                i.fecha_modificacion,
                i.nota_definitiva,
                i.veces_modificada,
                u.nombres as usuario_nombre,
                u.apellidos as usuario_apellido,
                u.rol as usuario_rol,
                e.nombres as estudiante_nombre,
                e.apellidos as estudiante_apellido,
                d.nombre as asignatura_nombre,
                d.codigo as asignatura_codigo,
                d.periodo as periodo_asignatura,
                p.nombre as postgrado_nombre,
                esc.nombre as escolaridad_nombre,
                al.nombre as periodo_nombre
                FROM inscripciones i
                JOIN usuarios u ON i.usuario_modificacion = u.id
                JOIN estudiantes e ON i.estudiante_id = e.id
                JOIN denominaciones d ON i.denominacion_id = d.id
                JOIN postgrados p ON d.postgrado_id = p.id
                JOIN escolaridades esc ON i.escolaridad_id = esc.id
                LEFT JOIN años_lectivos al ON i.año_lectivo_id = al.id
                WHERE u.rol IN ('coordinador1', 'coordinador2', 'administrador')";

$conditions_detalle = [];
$params_detalle = [];
$types_detalle = "";

if ($usuario_id > 0) {
    $conditions_detalle[] = " u.id = ?";
    $params_detalle[] = $usuario_id;
    $types_detalle .= "i";
}

if ($postgrado_id > 0) {
    $conditions_detalle[] = " p.id = ?";
    $params_detalle[] = $postgrado_id;
    $types_detalle .= "i";
}

if (!empty($rango_fechas)) {
    $conditions_detalle[] = " DATE(i.fecha_modificacion) BETWEEN ? AND ?";
    $params_detalle[] = $rango_fechas['inicio'];
    $params_detalle[] = $rango_fechas['fin'];
    $types_detalle .= "ss";
}

if ($ver_notas == 'cargadas') {
    $conditions_detalle[] = " i.nota_definitiva IS NOT NULL AND i.nota_definitiva != ''";
} elseif ($ver_notas == 'pendientes') {
    $conditions_detalle[] = " i.nota_definitiva IS NULL OR i.nota_definitiva = ''";
}

if (!empty($conditions_detalle)) {
    $sql_detalle .= " AND " . implode(" AND ", $conditions_detalle);
}

$sql_detalle .= " ORDER BY i.fecha_modificacion DESC LIMIT 100";

$stmt_detalle = $conn->prepare($sql_detalle);
if (!empty($params_detalle)) {
    $stmt_detalle->bind_param($types_detalle, ...$params_detalle);
}
$stmt_detalle->execute();
$detalle_cargas = $stmt_detalle->get_result()->fetch_all(MYSQLI_ASSOC);

// Preparar filtros aplicados para mostrar en la página
$filtros_aplicados = [];
if ($usuario_id > 0) {
    $usuarios->data_seek(0);
    while ($user = $usuarios->fetch_assoc()) {
        if ($user['id'] == $usuario_id) {
            $filtros_aplicados[] = "Usuario: " . htmlspecialchars($user['nombres'] . ' ' . $user['apellidos']);
            break;
        }
    }
}
if ($escolaridad_id > 0 && !empty($escolaridad_nombre)) {
    $filtros_aplicados[] = "Escolaridad: " . htmlspecialchars($escolaridad_nombre);
}
if ($postgrado_id > 0) {
    $postgrados->data_seek(0);
    while ($post = $postgrados->fetch_assoc()) {
        if ($post['id'] == $postgrado_id) {
            $filtros_aplicados[] = "Postgrado: " . htmlspecialchars($post['nombre']);
            break;
        }
    }
}
if (!empty($cuatrimestre) && $cuatrimestre != 'TODOS') {
    $filtros_aplicados[] = "Cuatrimestre: " . $cuatrimestres[$cuatrimestre];
    if (!empty($rango_fechas)) {
        $filtros_aplicados[] = "Año de carga: " . date('Y', strtotime($rango_fechas['inicio']));
    }
}
$filtros_aplicados[] = "Notas: " . $opciones_notas[$ver_notas];

$page_title = "Monitoreo de Cargas de Notas";
include '../includes/header.php';
?>

<div class="container-fluid">
    <!-- Botón para volver -->
    <div class="mb-3">
        <a href="../admin/index.php" class="btn btn-secondary btn-sm">
            <i class="bi bi-arrow-left"></i> Volver al Panel
        </a>
    </div>

    <div class="card mb-4">
        <div class="card-header bg-primary text-white">
            <h4 class="mb-0"><i class="bi bi-graph-up"></i> Monitoreo de Cargas de Notas</h4>
        </div>
        <div class="card-body">
            <form method="get" class="row g-3">
                <input type="hidden" name="pagina_esc" value="1">
                <input type="hidden" name="vista" value="<?= $vista ?>">
                <div class="col-md-2">
                    <label class="form-label">Usuario:</label>
                    <select name="usuario_id" class="form-select">
                        <option value="0">Todos los usuarios</option>
                        <?php 
                        $usuarios->data_seek(0);
                        while ($usuario = $usuarios->fetch_assoc()): ?>
                            <option value="<?= $usuario['id'] ?>" <?= $usuario_id == $usuario['id'] ? 'selected' : '' ?>>
                                <?= htmlspecialchars($usuario['nombres'] . ' ' . $usuario['apellidos'] . ' (' . $usuario['rol'] . ')') ?>
                            </option>
                        <?php endwhile; ?>
                    </select>
                </div>
                
                <div class="col-md-2">
                    <label class="form-label">Escolaridad (Año):</label>
                    <select name="escolaridad_id" class="form-select">
                        <option value="0">Todas las escolaridades</option>
                        <?php 
                        $escolaridades->data_seek(0);
                        while ($esc = $escolaridades->fetch_assoc()): ?>
                            <option value="<?= $esc['id'] ?>" <?= $escolaridad_id == $esc['id'] ? 'selected' : '' ?>>
                                <?= htmlspecialchars($esc['nombre']) ?>
                            </option>
                        <?php endwhile; ?>
                    </select>
                </div>
                
                <div class="col-md-2">
                    <label class="form-label">Postgrado:</label>
                    <select name="postgrado_id" class="form-select">
                        <option value="0">Todos los postgrados</option>
                        <?php 
                        $postgrados->data_seek(0);
                        while ($post = $postgrados->fetch_assoc()): ?>
                            <option value="<?= $post['id'] ?>" <?= $postgrado_id == $post['id'] ? 'selected' : '' ?>>
                                <?= htmlspecialchars($post['nombre']) ?>
                            </option>
                        <?php endwhile; ?>
                    </select>
                </div>
                
                <div class="col-md-2">
                    <label class="form-label">Cuatrimestre:</label>
                    <select name="cuatrimestre" class="form-select">
                        <?php foreach ($cuatrimestres as $key => $value): ?>
                            <option value="<?= $key ?>" <?= $cuatrimestre == $key ? 'selected' : '' ?>>
                                <?= htmlspecialchars($value) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="col-md-2">
                    <label class="form-label">Ver:</label>
                    <select name="ver_notas" class="form-select">
                        <?php foreach ($opciones_notas as $key => $value): ?>
                            <option value="<?= $key ?>" <?= $ver_notas == $key ? 'selected' : '' ?>>
                                <?= htmlspecialchars($value) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="col-md-2 d-flex align-items-end">
                    <button type="submit" class="btn btn-primary w-100">Filtrar</button>
                </div>
            </form>
            
            <!-- Mostrar filtros aplicados -->
            <?php if (!empty($filtros_aplicados)): ?>
                <div class="mt-3 p-2 bg-light rounded">
                    <strong>Filtros aplicados:</strong> <?= implode(' | ', $filtros_aplicados) ?>
                    <?php if (!empty($cuatrimestre) && $cuatrimestre != 'TODOS' && !empty($rango_fechas)): ?>
                        <br><small class="text-muted">
                            Mostrando notas cargadas durante: <strong><?= $cuatrimestres[$cuatrimestre] ?></strong>
                            <br>Rango de fechas: <strong><?= date('d/m/Y', strtotime($rango_fechas['inicio'])) ?> al <?= date('d/m/Y', strtotime($rango_fechas['fin'])) ?></strong>
                            <?php if (!empty($escolaridad_nombre)): ?>
                                <br>Año de escolaridad para referencia: <strong><?= $escolaridad_nombre ?></strong>
                            <?php endif; ?>
                        </small>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Pestañas de navegación -->
    <ul class="nav nav-tabs mb-4" id="myTab" role="tablist">
        <li class="nav-item" role="presentation">
            <button class="nav-link <?= $vista == 'resumen' ? 'active' : '' ?>" id="resumen-tab" data-bs-toggle="tab" data-bs-target="#resumen" type="button" role="tab" onclick="cambiarVista('resumen')">
                <i class="bi bi-journal-check"></i> Resumen por Escolaridad
            </button>
        </li>
        <li class="nav-item" role="presentation">
            <button class="nav-link <?= $vista == 'fechas' ? 'active' : '' ?>" id="fechas-tab" data-bs-toggle="tab" data-bs-target="#fechas" type="button" role="tab" onclick="cambiarVista('fechas')">
                <i class="bi bi-calendar-check"></i> Fechas de Carga
            </button>
        </li>
        <li class="nav-item" role="presentation">
            <button class="nav-link <?= $vista == 'postgrados' ? 'active' : '' ?>" id="postgrados-tab" data-bs-toggle="tab" data-bs-target="#postgrados" type="button" role="tab" onclick="cambiarVista('postgrados')">
                <i class="bi bi-book"></i> Resumen por Postgrado
            </button>
        </li>
    </ul>

    <div class="tab-content" id="myTabContent">
        <!-- Pestaña de Resumen por Escolaridad -->
        <div class="tab-pane fade <?= $vista == 'resumen' ? 'show active' : '' ?>" id="resumen" role="tabpanel">
            
            <!-- TARJETAS DE RESUMEN GLOBAL -->
            <div class="row mb-4">
                <div class="col-md-3">
                    <div class="card bg-primary text-white">
                        <div class="card-body text-center">
                            <h5 class="card-title">Total Inscripciones</h5>
                            <h2 class="display-4"><?= number_format($resumen_global['total_inscripciones']) ?></h2>
                            <p class="card-text">Registros encontrados</p>
                        </div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="card bg-success text-white">
                        <div class="card-body text-center">
                            <h5 class="card-title">Notas Cargadas</h5>
                            <h2 class="display-4"><?= number_format($resumen_global['notas_cargadas']) ?></h2>
                            <p class="card-text"><?= $porcentaje_cargado ?>% completado</p>
                        </div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="card bg-warning text-dark">
                        <div class="card-body text-center">
                            <h5 class="card-title">Notas Pendientes</h5>
                            <h2 class="display-4"><?= number_format($resumen_global['notas_pendientes']) ?></h2>
                            <p class="card-text"><?= $porcentaje_pendiente ?>% pendiente</p>
                        </div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="card bg-info text-white">
                        <div class="card-body text-center">
                            <h5 class="card-title">Usuarios Activos</h5>
                            <h2 class="display-4"><?= number_format($resumen_global['usuarios_activos']) ?></h2>
                            <p class="card-text">Coordinadores trabajando</p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- BARRA DE PROGRESO GLOBAL -->
            <div class="card mb-4">
                <div class="card-header bg-light">
                    <h6 class="mb-0"><i class="bi bi-bar-chart"></i> Progreso Global de Carga de Notas</h6>
                </div>
                <div class="card-body">
                    <div class="progress" style="height: 30px;">
                        <div class="progress-bar bg-success" style="width: <?= $porcentaje_cargado ?>%">
                            <strong><?= $porcentaje_cargado ?>% Cargadas (<?= number_format($resumen_global['notas_cargadas']) ?>)</strong>
                        </div>
                        <div class="progress-bar bg-warning" style="width: <?= $porcentaje_pendiente ?>%">
                            <strong><?= $porcentaje_pendiente ?>% Pendientes (<?= number_format($resumen_global['notas_pendientes']) ?>)</strong>
                        </div>
                    </div>
                    <div class="mt-2 text-center">
                        <small class="text-muted">
                            Estado general: 
                            <span class="badge bg-<?= $porcentaje_cargado == 100 ? 'success' : ($porcentaje_cargado >= 80 ? 'warning' : 'danger') ?>">
                                <?= $porcentaje_cargado == 100 ? 'COMPLETADO' : ($porcentaje_cargado >= 80 ? 'EN PROCESO AVANZADO' : 'EN PROCESO') ?>
                            </span>
                        </small>
                    </div>
                </div>
            </div>

            <!-- TABLA DETALLADA POR ESCOLARIDAD -->
            <div class="card mb-4">
                <div class="card-header bg-success text-white d-flex justify-content-between align-items-center">
                    <h5 class="mb-0"><i class="bi bi-journal-check"></i> Resumen por Escolaridad y Período</h5>
                    <div class="d-flex gap-2">
                        <span class="badge bg-light text-dark">
                            Página <?= $pagina_actual ?> de <?= $total_paginas_esc ?> 
                            (<?= $total_registros_esc ?> registros)
                        </span>
                        <!-- Botones de exportación -->
                        <?php if (!empty($resumen_escolaridades)): ?>
                            <a href="?<?php echo http_build_query(array_merge($_GET, ['generar_excel' => 1, 'vista' => 'resumen'])); ?>" 
                               class="btn btn-light btn-sm">
                                <i class="bi bi-file-earmark-excel"></i> Excel
                            </a>
                            <a href="?<?php echo http_build_query(array_merge($_GET, ['generar_pdf' => 1, 'vista' => 'resumen'])); ?>" 
                               class="btn btn-light btn-sm">
                                <i class="bi bi-file-earmark-pdf"></i> PDF
                            </a>
                        <?php endif; ?>
                    </div>
                </div>
                <div class="card-body">
                    <?php if (empty($resumen_escolaridades)): ?>
                        <div class="alert alert-info text-center">
                            <i class="bi bi-info-circle"></i> No se encontraron registros con los filtros aplicados.
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-striped table-hover">
                                <thead>
                                    <tr>
                                        <th>Escolaridad</th>
                                        <th>Período</th>
                                        <th>Total Inscripciones</th>
                                        <th>Notas Cargadas</th>
                                        <th>Notas Pendientes</th>
                                        <th>Porcentaje Cargado</th>
                                        <th>Estado</th>
                                        <th>Acciones</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($resumen_escolaridades as $resumen): 
                                        $porcentaje = $resumen['total_inscripciones'] > 0 ? 
                                            round(($resumen['notas_cargadas'] / $resumen['total_inscripciones']) * 100, 2) : 0;
                                    ?>
                                        <tr>
                                            <td>
                                                <strong><?= htmlspecialchars($resumen['escolaridad_nombre']) ?></strong>
                                            </td>
                                            <td>
                                                <span class="badge bg-info"><?= htmlspecialchars($resumen['periodo_nombre']) ?></span>
                                            </td>
                                            <td><span class="badge bg-secondary"><?= $resumen['total_inscripciones'] ?></span></td>
                                            <td><span class="badge bg-success"><?= $resumen['notas_cargadas'] ?></span></td>
                                            <td><span class="badge bg-<?= $resumen['notas_pendientes'] > 0 ? 'warning' : 'success' ?>"><?= $resumen['notas_pendientes'] ?></span></td>
                                            <td>
                                                <div class="progress" style="height: 20px;">
                                                    <div class="progress-bar bg-<?= $porcentaje >= 80 ? 'success' : ($porcentaje >= 50 ? 'warning' : 'danger') ?>" 
                                                         style="width: <?= $porcentaje ?>%">
                                                        <?= $porcentaje ?>%
                                                    </div>
                                                </div>
                                            </td>
                                            <td>
                                                <?php if ($porcentaje == 100): ?>
                                                    <span class="badge bg-success">COMPLETADO</span>
                                                <?php elseif ($porcentaje == 0): ?>
                                                    <span class="badge bg-danger">SIN INICIAR</span>
                                                <?php else: ?>
                                                    <span class="badge bg-warning">EN PROCESO</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <a href="detalle_escolaridad.php?escolaridad=<?= urlencode($resumen['escolaridad_nombre']) ?>&periodo=<?= urlencode($resumen['periodo_nombre']) ?>&usuario_id=<?= $usuario_id ?>&postgrado_id=<?= $postgrado_id ?>&cuatrimestre=<?= $cuatrimestre ?>&ver_notas=<?= $ver_notas ?>" 
                                                   class="btn btn-sm btn-outline-primary">
                                                    <i class="bi bi-list"></i> Ver Detalle
                                                </a>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>

                        <!-- Paginación para escolaridades -->
                        <?php if ($total_paginas_esc > 1): ?>
                            <nav aria-label="Paginación de escolaridades">
                                <ul class="pagination justify-content-center">
                                    <?php if ($pagina_actual > 1): ?>
                                        <li class="page-item">
                                            <a class="page-link" href="?<?php 
                                                echo http_build_query(array_merge($_GET, ['pagina_esc' => $pagina_actual - 1]));
                                            ?>" aria-label="Anterior">
                                                <span aria-hidden="true">&laquo;</span>
                                            </a>
                                        </li>
                                    <?php endif; ?>

                                    <?php 
                                    $inicio = max(1, $pagina_actual - 2);
                                    $fin = min($total_paginas_esc, $pagina_actual + 2);
                                    
                                    for ($i = $inicio; $i <= $fin; $i++): 
                                    ?>
                                        <li class="page-item <?= $i == $pagina_actual ? 'active' : '' ?>">
                                            <a class="page-link" href="?<?php 
                                                echo http_build_query(array_merge($_GET, ['pagina_esc' => $i]));
                                            ?>"><?= $i ?></a>
                                        </li>
                                    <?php endfor; ?>

                                    <?php if ($pagina_actual < $total_paginas_esc): ?>
                                        <li class="page-item">
                                            <a class="page-link" href="?<?php 
                                                echo http_build_query(array_merge($_GET, ['pagina_esc' => $pagina_actual + 1]));
                                            ?>" aria-label="Siguiente">
                                                <span aria-hidden="true">&raquo;</span>
                                            </a>
                                        </li>
                                    <?php endif; ?>
                                </ul>
                            </nav>
                            
                            <div class="text-center text-muted small">
                                Mostrando <?= count($resumen_escolaridades) ?> de <?= $total_registros_esc ?> registros
                            </div>
                        <?php endif; ?>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Pestaña de Fechas de Carga -->
        <div class="tab-pane fade <?= $vista == 'fechas' ? 'show active' : '' ?>" id="fechas" role="tabpanel">
            <div class="card mb-4">
                <div class="card-header bg-info text-white d-flex justify-content-between align-items-center">
                    <h5 class="mb-0"><i class="bi bi-calendar-check"></i> Fechas de Carga por Cuatrimestre</h5>
                    <div class="d-flex gap-2">
                        <span class="badge bg-light text-dark">
                            <?php if (!empty($rango_fechas)): ?>
                                <?= date('d/m/Y', strtotime($rango_fechas['inicio'])) ?> - <?= date('d/m/Y', strtotime($rango_fechas['fin'])) ?>
                            <?php else: ?>
                                Últimos 30 días
                            <?php endif; ?>
                        </span>
                        <!-- Botones de exportación -->
                        <?php if (!empty($resumen_fechas)): ?>
                            <a href="?<?php echo http_build_query(array_merge($_GET, ['generar_excel_fechas' => 1])); ?>" 
                               class="btn btn-light btn-sm">
                                <i class="bi bi-file-earmark-excel"></i> Excel
                            </a>
                            <a href="?<?php echo http_build_query(array_merge($_GET, ['generar_pdf_fechas' => 1])); ?>" 
                               class="btn btn-light btn-sm">
                                <i class="bi bi-file-earmark-pdf"></i> PDF
                            </a>
                        <?php endif; ?>
                    </div>
                </div>
                <div class="card-body">
                    <?php if (empty($resumen_fechas)): ?>
                        <div class="alert alert-info text-center">
                            <i class="bi bi-info-circle"></i> No se encontraron cargas de notas en el período seleccionado.
                            <?php if (!empty($rango_fechas)): ?>
                                <br><small>Rango: <?= date('d/m/Y', strtotime($rango_fechas['inicio'])) ?> al <?= date('d/m/Y', strtotime($rango_fechas['fin'])) ?></small>
                            <?php endif; ?>
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-striped table-hover">
                                <thead>
                                    <tr>
                                        <th>Fecha de Carga</th>
                                        <th>Total Cargas</th>
                                        <th>Usuarios Activos</th>
                                        <th>Responsables</th>
                                        <th>Primera Carga</th>
                                        <th>Última Carga</th>
                                        <th>Actividad</th>
                                        <th>Acciones</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($resumen_fechas as $fecha): 
                                        $nivel_actividad = $fecha['total_cargas'] > 50 ? 'alta' : ($fecha['total_cargas'] > 20 ? 'media' : 'baja');
                                    ?>
                                        <tr>
                                            <td>
                                                <strong><?= date('d/m/Y', strtotime($fecha['fecha_carga'])) ?></strong>
                                                <br>
                                                <small class="text-muted"><?= date('l', strtotime($fecha['fecha_carga'])) ?></small>
                                            </td>
                                            <td>
                                                <span class="badge bg-primary fs-6"><?= $fecha['total_cargas'] ?></span>
                                            </td>
                                            <td>
                                                <span class="badge bg-success"><?= $fecha['usuarios_activos'] ?> usuario(s)</span>
                                            </td>
                                            <td>
                                                <small><?= htmlspecialchars(mb_strimwidth($fecha['responsables'], 0, 50, '...')) ?></small>
                                            </td>
                                            <td>
                                                <span class="badge bg-secondary"><?= date('H:i', strtotime($fecha['primera_carga'])) ?></span>
                                            </td>
                                            <td>
                                                <span class="badge bg-dark"><?= date('H:i', strtotime($fecha['ultima_carga'])) ?></span>
                                            </td>
                                            <td>
                                                <?php if ($nivel_actividad == 'alta'): ?>
                                                    <span class="badge bg-danger">ALTA ACTIVIDAD</span>
                                                <?php elseif ($nivel_actividad == 'media'): ?>
                                                    <span class="badge bg-warning">ACTIVIDAD MEDIA</span>
                                                <?php else: ?>
                                                    <span class="badge bg-success">ACTIVIDAD BAJA</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <a href="detalle_fecha_carga.php?fecha_carga=<?= $fecha['fecha_carga'] ?>&usuario_id=<?= $usuario_id ?>&postgrado_id=<?= $postgrado_id ?>&escolaridad_id=<?= $escolaridad_id ?>&cuatrimestre=<?= $cuatrimestre ?>&ver_notas=<?= $ver_notas ?>" 
                                                   class="btn btn-sm btn-outline-primary">
                                                    <i class="bi bi-list"></i> Ver Detalle
                                                </a>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>

                        <div class="row mt-3">
                            <div class="col-md-4">
                                <div class="card bg-light">
                                    <div class="card-body text-center">
                                        <h6 class="card-title">Total de Días con Actividad</h6>
                                        <h3 class="text-primary"><?= count($resumen_fechas) ?></h3>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="card bg-success text-white">
                                    <div class="card-body text-center">
                                        <h6 class="card-title">Total Cargas Registradas</h6>
                                        <h3><?= array_sum(array_column($resumen_fechas, 'total_cargas')) ?></h3>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="card bg-info text-white">
                                    <div class="card-body text-center">
                                        <h6 class="card-title">Usuarios Únicos</h6>
                                        <h3><?= count(array_unique(explode(', ', implode(', ', array_column($resumen_fechas, 'responsables'))))) ?></h3>
                                    </div>
                                </div>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- NUEVA PESTAÑA: Resumen por Postgrado -->
        <div class="tab-pane fade <?= $vista == 'postgrados' ? 'show active' : '' ?>" id="postgrados" role="tabpanel">
            <div class="card mb-4">
                <div class="card-header bg-warning text-dark d-flex justify-content-between align-items-center">
                    <h5 class="mb-0"><i class="bi bi-book"></i> Resumen por Postgrado y Coordinadores Responsables</h5>
                    <div class="d-flex gap-2">
                        <?php if (!empty($resumen_postgrados)): ?>
                            <a href="?<?php echo http_build_query(array_merge($_GET, ['generar_excel_postgrados' => 1])); ?>" 
                               class="btn btn-light btn-sm">
                                <i class="bi bi-file-earmark-excel"></i> Excel
                            </a>
                            <a href="?<?php echo http_build_query(array_merge($_GET, ['generar_pdf_postgrados' => 1])); ?>" 
                               class="btn btn-light btn-sm">
                                <i class="bi bi-file-earmark-pdf"></i> PDF
                            </a>
                        <?php endif; ?>
                    </div>
                </div>
                <div class="card-body">
                    <?php if (empty($resumen_postgrados)): ?>
                        <div class="alert alert-info text-center">
                            <i class="bi bi-info-circle"></i> No se encontraron registros con los filtros aplicados.
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-striped table-hover">
                                <thead>
                                    <tr>
                                        <th>Postgrado</th>
                                        <th>Coordinadores Responsables</th>
                                        <th>Total Inscripciones</th>
                                        <th>Notas Cargadas</th>
                                        <th>Notas Pendientes</th>
                                        <th>Porcentaje Cargado</th>
                                        <th>Estado</th>
                                        <th>Acciones</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($resumen_postgrados as $postgrado): 
                                        $porcentaje = $postgrado['total_inscripciones'] > 0 ? 
                                            round(($postgrado['notas_cargadas'] / $postgrado['total_inscripciones']) * 100, 2) : 0;
                                        $coordinadores = !empty($postgrado['coordinadores_responsables']) ? 
                                            $postgrado['coordinadores_responsables'] : 'Sin coordinador asignado';
                                    ?>
                                        <tr>
                                            <td>
                                                <strong><?= htmlspecialchars($postgrado['postgrado_nombre']) ?></strong>
                                            </td>
                                            <td>
                                                <small class="text-muted"><?= htmlspecialchars($coordinadores) ?></small>
                                            </td>
                                            <td><span class="badge bg-secondary"><?= $postgrado['total_inscripciones'] ?></span></td>
                                            <td><span class="badge bg-success"><?= $postgrado['notas_cargadas'] ?></span></td>
                                            <td><span class="badge bg-<?= $postgrado['notas_pendientes'] > 0 ? 'warning' : 'success' ?>"><?= $postgrado['notas_pendientes'] ?></span></td>
                                            <td>
                                                <div class="progress" style="height: 20px;">
                                                    <div class="progress-bar bg-<?= $porcentaje >= 80 ? 'success' : ($porcentaje >= 50 ? 'warning' : 'danger') ?>" 
                                                         style="width: <?= $porcentaje ?>%">
                                                        <?= $porcentaje ?>%
                                                    </div>
                                                </div>
                                            </td>
                                            <td>
                                                <?php if ($porcentaje == 100): ?>
                                                    <span class="badge bg-success">COMPLETADO</span>
                                                <?php elseif ($porcentaje == 0): ?>
                                                    <span class="badge bg-danger">SIN INICIAR</span>
                                                <?php else: ?>
                                                    <span class="badge bg-warning">EN PROCESO</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <a href="detalle_postgrado.php?postgrado_id=<?= $postgrado['postgrado_id'] ?>&usuario_id=<?= $usuario_id ?>&escolaridad_id=<?= $escolaridad_id ?>&cuatrimestre=<?= $cuatrimestre ?>&ver_notas=<?= $ver_notas ?>" 
                                                   class="btn btn-sm btn-outline-primary">
                                                    <i class="bi bi-list"></i> Ver Detalle
                                                </a>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>

                        <!-- Información adicional sobre coordinadores1 -->
                        <div class="mt-4 p-3 bg-light rounded">
                            <h6><i class="bi bi-info-circle"></i> Información sobre coordinadores:</h6>
                            <ul class="mb-0">
                                <li><strong>Coordinadores1:</strong> Responsables específicos de cargar las notas de los postgrados asignados</li>
                                <li><strong>Coordinadores2 y Administradores:</strong> Pueden cargar notas de cualquier postgrado</li>
                                <li>Las <span class="badge bg-warning">notas pendientes</span> indican la carga de trabajo pendiente por coordinador</li>
                            </ul>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

    <!-- Detalle de Cargas Recientes -->
    <div class="card">
        <div class="card-header bg-warning text-dark">
            <h5 class="mb-0">
                <i class="bi bi-clock-history"></i> 
                <?= $opciones_notas[$ver_notas] ?> 
                <?php if (!empty($cuatrimestre) && $cuatrimestre != 'TODOS' && !empty($rango_fechas)): ?>
                    durante <?= $cuatrimestres[$cuatrimestre] ?>
                    <br><small>(<?= date('d/m/Y', strtotime($rango_fechas['inicio'])) ?> - <?= date('d/m/Y', strtotime($rango_fechas['fin'])) ?>)</small>
                <?php endif; ?>
                (Últimas 100)
            </h5>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-striped table-hover table-sm">
                    <thead>
                        <tr>
                            <th>Fecha/Hora</th>
                            <th>Usuario</th>
                            <th>Estudiante</th>
                            <th>Asignatura</th>
                            <th>Periodo Asig.</th>
                            <th>Postgrado</th>
                            <th>Escolaridad</th>
                            <th>Período Carga</th>
                            <th>Nota</th>
                            <th>Modificaciones</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($detalle_cargas as $carga): ?>
                            <tr>
                                <td><?= date('d/m/Y H:i', strtotime($carga['fecha_modificacion'])) ?></td>
                                <td>
                                    <small><?= htmlspecialchars($carga['usuario_nombre'] . ' ' . $carga['usuario_apellido']) ?></small>
                                    <br><span class="badge bg-secondary"><?= htmlspecialchars($carga['usuario_rol']) ?></span>
                                </td>
                                <td><?= htmlspecialchars($carga['estudiante_nombre'] . ' ' . $carga['estudiante_apellido']) ?></td>
                                <td>
                                    <small class="text-muted"><?= htmlspecialchars($carga['asignatura_codigo']) ?></small><br>
                                    <?= htmlspecialchars($carga['asignatura_nombre']) ?>
                                </td>
                                <td>
                                    <span class="badge bg-dark"><?= htmlspecialchars($carga['periodo_asignatura']) ?></span>
                                </td>
                                <td><?= htmlspecialchars($carga['postgrado_nombre']) ?></td>
                                <td>
                                    <span class="badge bg-info"><?= htmlspecialchars($carga['escolaridad_nombre']) ?></span>
                                </td>
                                <td>
                                    <small class="text-primary"><?= htmlspecialchars($carga['periodo_nombre']) ?></small>
                                </td>
                                <td>
                                    <span class="badge bg-<?= !empty($carga['nota_definitiva']) ? 'success' : 'warning' ?>">
                                        <?= !empty($carga['nota_definitiva']) ? htmlspecialchars($carga['nota_definitiva']) : 'PENDIENTE' ?>
                                    </span>
                                </td>
                                <td>
                                    <span class="badge bg-<?= $carga['veces_modificada'] > 1 ? 'warning' : 'secondary' ?>">
                                        <?= $carga['veces_modificada'] ?> vez(es)
                                    </span>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<script>
function cambiarVista(vista) {
    const form = document.querySelector('form');
    const vistaInput = document.querySelector('input[name="vista"]');
    if (vistaInput && form) {
        vistaInput.value = vista;
        form.submit();
    }
}
</script>

<?php
include '../includes/footer.php';
$conn->close();
?>