<?php
session_start();

if (!isset($_SESSION['username']) || $_SESSION['rol'] != 'administrador') {
    header("Location: ../login.php");
    exit();
}

include '../includes/db.php';
$conn->set_charset("utf8mb4");

$fecha_carga = isset($_GET['fecha_carga']) ? $_GET['fecha_carga'] : '';
$usuario_id = isset($_GET['usuario_id']) ? intval($_GET['usuario_id']) : 0;
$postgrado_id = isset($_GET['postgrado_id']) ? intval($_GET['postgrado_id']) : 0;

if (empty($fecha_carga)) {
    die("Parámetros incompletos");
}

// Configuración de paginación
$registros_por_pagina = 20;
$pagina_actual = isset($_GET['pagina']) ? (int)$_GET['pagina'] : 1;
$offset = ($pagina_actual - 1) * $registros_por_pagina;

// Consulta para el detalle de la fecha específica (CON PAGINACIÓN)
$sql_detalle = "SELECT 
                i.id,
                i.fecha_modificacion,
                i.nota_definitiva,
                i.veces_modificada,
                u.nombres as usuario_nombre,
                u.apellidos as usuario_apellido,
                u.rol as usuario_rol,
                e.nombres as estudiante_nombre,
                e.apellidos as estudiante_apellido,
                d.nombre as asignatura_nombre,
                d.codigo as asignatura_codigo,
                d.periodo as periodo_asignatura,
                p.nombre as postgrado_nombre,
                esc.nombre as escolaridad_nombre,
                al.nombre as periodo_nombre
                FROM inscripciones i
                JOIN usuarios u ON i.usuario_modificacion = u.id
                JOIN estudiantes e ON i.estudiante_id = e.id
                JOIN denominaciones d ON i.denominacion_id = d.id
                JOIN postgrados p ON d.postgrado_id = p.id
                JOIN escolaridades esc ON i.escolaridad_id = esc.id
                LEFT JOIN años_lectivos al ON i.año_lectivo_id = al.id
                WHERE u.rol IN ('coordinador1', 'coordinador2', 'administrador')
                AND DATE(i.fecha_modificacion) = ?
                AND i.nota_definitiva IS NOT NULL";

$conditions = [];
$params = [$fecha_carga];
$types = "s";

if ($usuario_id > 0) {
    $conditions[] = " u.id = ?";
    $params[] = $usuario_id;
    $types .= "i";
}

if ($postgrado_id > 0) {
    $conditions[] = " p.id = ?";
    $params[] = $postgrado_id;
    $types .= "i";
}

if (!empty($conditions)) {
    $sql_detalle .= " AND " . implode(" AND ", $conditions);
}

$sql_detalle .= " ORDER BY i.fecha_modificacion DESC
                LIMIT ?, ?";

$params[] = $offset;
$params[] = $registros_por_pagina;
$types .= "ii";

$stmt_detalle = $conn->prepare($sql_detalle);
$stmt_detalle->bind_param($types, ...$params);
$stmt_detalle->execute();
$detalle = $stmt_detalle->get_result()->fetch_all(MYSQLI_ASSOC);

// Consulta para contar total de registros
$sql_count = "SELECT COUNT(*) as total
                FROM inscripciones i
                JOIN usuarios u ON i.usuario_modificacion = u.id
                JOIN estudiantes e ON i.estudiante_id = e.id
                JOIN denominaciones d ON i.denominacion_id = d.id
                JOIN postgrados p ON d.postgrado_id = p.id
                JOIN escolaridades esc ON i.escolaridad_id = esc.id
                LEFT JOIN años_lectivos al ON i.año_lectivo_id = al.id
                WHERE u.rol IN ('coordinador1', 'coordinador2', 'administrador')
                AND DATE(i.fecha_modificacion) = ?
                AND i.nota_definitiva IS NOT NULL";

$params_count = [$fecha_carga];
$types_count = "s";

if ($usuario_id > 0) {
    $sql_count .= " AND u.id = ?";
    $params_count[] = $usuario_id;
    $types_count .= "i";
}

if ($postgrado_id > 0) {
    $sql_count .= " AND p.id = ?";
    $params_count[] = $postgrado_id;
    $types_count .= "i";
}

$stmt_count = $conn->prepare($sql_count);
if (!empty($params_count)) {
    $stmt_count->bind_param($types_count, ...$params_count);
}
$stmt_count->execute();
$result_count = $stmt_count->get_result();
$total_registros = $result_count->fetch_assoc()['total'];
$total_paginas = ceil($total_registros / $registros_por_pagina);

// Consulta para estadísticas (sin paginación)
$sql_stats = "SELECT 
                COUNT(i.id) as total_cargas,
                COUNT(DISTINCT i.usuario_modificacion) as usuarios_activos,
                MIN(i.fecha_modificacion) as primera_carga,
                MAX(i.fecha_modificacion) as ultima_carga,
                GROUP_CONCAT(DISTINCT CONCAT(u.nombres, ' ', u.apellidos) SEPARATOR ', ') as responsables
                FROM inscripciones i
                JOIN usuarios u ON i.usuario_modificacion = u.id
                WHERE u.rol IN ('coordinador1', 'coordinador2', 'administrador')
                AND DATE(i.fecha_modificacion) = ?
                AND i.nota_definitiva IS NOT NULL";

$params_stats = [$fecha_carga];
$types_stats = "s";

if ($usuario_id > 0) {
    $sql_stats .= " AND u.id = ?";
    $params_stats[] = $usuario_id;
    $types_stats .= "i";
}

if ($postgrado_id > 0) {
    $sql_stats .= " AND EXISTS (SELECT 1 FROM denominaciones d WHERE d.id = i.denominacion_id AND d.postgrado_id = ?)";
    $params_stats[] = $postgrado_id;
    $types_stats .= "i";
}

$stmt_stats = $conn->prepare($sql_stats);
if (!empty($params_stats)) {
    $stmt_stats->bind_param($types_stats, ...$params_stats);
}
$stmt_stats->execute();
$stats = $stmt_stats->get_result()->fetch_assoc();

$total_cargas = $stats['total_cargas'];
$usuarios_activos = $stats['usuarios_activos'];
$primera_carga = $stats['primera_carga'];
$ultima_carga = $stats['ultima_carga'];
$responsables = $stats['responsables'];

// Procesar generación de Excel
if (isset($_GET['generar_excel'])) {
    header('Content-Type: application/vnd.ms-excel; charset=utf-8');
    header('Content-Disposition: attachment; filename="detalle_fecha_carga_' . date('Y-m-d_H-i-s') . '.xls"');
    header('Cache-Control: max-age=0');
    
    // Obtener todos los datos sin paginación para el Excel
    $sql_excel = "SELECT 
                    i.id,
                    i.fecha_modificacion,
                    i.nota_definitiva,
                    i.veces_modificada,
                    u.nombres as usuario_nombre,
                    u.apellidos as usuario_apellido,
                    u.rol as usuario_rol,
                    e.nombres as estudiante_nombre,
                    e.apellidos as estudiante_apellido,
                    d.nombre as asignatura_nombre,
                    d.codigo as asignatura_codigo,
                    d.periodo as periodo_asignatura,
                    p.nombre as postgrado_nombre,
                    esc.nombre as escolaridad_nombre,
                    al.nombre as periodo_nombre
                    FROM inscripciones i
                    JOIN usuarios u ON i.usuario_modificacion = u.id
                    JOIN estudiantes e ON i.estudiante_id = e.id
                    JOIN denominaciones d ON i.denominacion_id = d.id
                    JOIN postgrados p ON d.postgrado_id = p.id
                    JOIN escolaridades esc ON i.escolaridad_id = esc.id
                    LEFT JOIN años_lectivos al ON i.año_lectivo_id = al.id
                    WHERE u.rol IN ('coordinador1', 'coordinador2', 'administrador')
                    AND DATE(i.fecha_modificacion) = ?
                    AND i.nota_definitiva IS NOT NULL";

    $params_excel = [$fecha_carga];
    $types_excel = "s";

    if ($usuario_id > 0) {
        $sql_excel .= " AND u.id = ?";
        $params_excel[] = $usuario_id;
        $types_excel .= "i";
    }

    if ($postgrado_id > 0) {
        $sql_excel .= " AND p.id = ?";
        $params_excel[] = $postgrado_id;
        $types_excel .= "i";
    }

    $sql_excel .= " ORDER BY i.fecha_modificacion DESC";
    
    $stmt_excel = $conn->prepare($sql_excel);
    if (!empty($params_excel)) {
        $stmt_excel->bind_param($types_excel, ...$params_excel);
    }
    $stmt_excel->execute();
    $detalle_excel = $stmt_excel->get_result()->fetch_all(MYSQLI_ASSOC);
    
    echo '<!DOCTYPE html>
    <html lang="es">
    <head>
        <meta http-equiv="Content-Type" content="text/html; charset=utf-8">
        <style>
            body { font-family: Arial, sans-serif; font-size: 11px; }
            table { border-collapse: collapse; width: 100%; }
            th, td { border: 1px solid #000000; padding: 5px; text-align: left; }
            th { background-color: #F2F2F2; font-weight: bold; }
            .titulo { font-size: 16px; font-weight: bold; text-align: center; }
            .subtitulo { font-size: 14px; text-align: center; }
            .estadisticas { margin-bottom: 15px; padding: 10px; background-color: #f8f9fa; border: 1px solid #ddd; }
            .estadistica-item { display: inline-block; margin-right: 20px; font-weight: bold; }
        </style>
    </head>
    <body>
        <table>
            <tr>
                <th colspan="10" class="titulo">DETALLE DE CARGAS - ' . date('d/m/Y', strtotime($fecha_carga)) . '</th>
            </tr>
            <tr>
                <th colspan="10" class="subtitulo">Generado: ' . date('d/m/Y H:i:s') . '</th>
            </tr>
            <tr>
                <td colspan="10" class="estadisticas">
                    <span class="estadistica-item">Total Cargas: ' . $total_cargas . '</span>
                    <span class="estadistica-item">Usuarios Activos: ' . $usuarios_activos . '</span>
                    <span class="estadistica-item">Primera Carga: ' . date('H:i', strtotime($primera_carga)) . '</span>
                    <span class="estadistica-item">Última Carga: ' . date('H:i', strtotime($ultima_carga)) . '</span>
                </td>
            </tr>
            <tr>
                <th>Hora</th>
                <th>Usuario</th>
                <th>Rol</th>
                <th>Estudiante</th>
                <th>Asignatura</th>
                <th>Código</th>
                <th>Postgrado</th>
                <th>Escolaridad</th>
                <th>Período</th>
                <th>Nota</th>
            </tr>';
    
    foreach ($detalle_excel as $item) {
        echo '<tr>
                <td>' . date('H:i', strtotime($item['fecha_modificacion'])) . '</td>
                <td>' . htmlspecialchars($item['usuario_nombre'] . ' ' . $item['usuario_apellido']) . '</td>
                <td>' . htmlspecialchars($item['usuario_rol']) . '</td>
                <td>' . htmlspecialchars($item['estudiante_nombre'] . ' ' . $item['estudiante_apellido']) . '</td>
                <td>' . htmlspecialchars($item['asignatura_nombre']) . '</td>
                <td>' . htmlspecialchars($item['asignatura_codigo']) . '</td>
                <td>' . htmlspecialchars($item['postgrado_nombre']) . '</td>
                <td>' . htmlspecialchars($item['escolaridad_nombre']) . '</td>
                <td>' . htmlspecialchars($item['periodo_nombre']) . '</td>
                <td>' . htmlspecialchars($item['nota_definitiva']) . '</td>
            </tr>';
    }
    
    echo '</table>
    </body>
    </html>';
    exit;
}

// Procesar generación de PDF
if (isset($_GET['generar_pdf'])) {
    $mpdfPath = __DIR__ . '/../vendor/autoload.php';
    if (!file_exists($mpdfPath)) {
        die("ERROR: mPDF no encontrado. Ejecute: composer require mpdf/mpdf");
    }
    require_once $mpdfPath;
    
    try {
        // Obtener todos los datos sin paginación para el PDF
        $sql_pdf = "SELECT 
                    i.id,
                    i.fecha_modificacion,
                    i.nota_definitiva,
                    i.veces_modificada,
                    u.nombres as usuario_nombre,
                    u.apellidos as usuario_apellido,
                    u.rol as usuario_rol,
                    e.nombres as estudiante_nombre,
                    e.apellidos as estudiante_apellido,
                    d.nombre as asignatura_nombre,
                    d.codigo as asignatura_codigo,
                    d.periodo as periodo_asignatura,
                    p.nombre as postgrado_nombre,
                    esc.nombre as escolaridad_nombre,
                    al.nombre as periodo_nombre
                    FROM inscripciones i
                    JOIN usuarios u ON i.usuario_modificacion = u.id
                    JOIN estudiantes e ON i.estudiante_id = e.id
                    JOIN denominaciones d ON i.denominacion_id = d.id
                    JOIN postgrados p ON d.postgrado_id = p.id
                    JOIN escolaridades esc ON i.escolaridad_id = esc.id
                    LEFT JOIN años_lectivos al ON i.año_lectivo_id = al.id
                    WHERE u.rol IN ('coordinador1', 'coordinador2', 'administrador')
                    AND DATE(i.fecha_modificacion) = ?
                    AND i.nota_definitiva IS NOT NULL";

        $params_pdf = [$fecha_carga];
        $types_pdf = "s";

        if ($usuario_id > 0) {
            $sql_pdf .= " AND u.id = ?";
            $params_pdf[] = $usuario_id;
            $types_pdf .= "i";
        }

        if ($postgrado_id > 0) {
            $sql_pdf .= " AND p.id = ?";
            $params_pdf[] = $postgrado_id;
            $types_pdf .= "i";
        }

        $sql_pdf .= " ORDER BY i.fecha_modificacion DESC";
        
        $stmt_pdf = $conn->prepare($sql_pdf);
        if (!empty($params_pdf)) {
            $stmt_pdf->bind_param($types_pdf, ...$params_pdf);
        }
        $stmt_pdf->execute();
        $detalle_pdf = $stmt_pdf->get_result()->fetch_all(MYSQLI_ASSOC);
        
        $mpdf = new \Mpdf\Mpdf([
            'mode' => 'utf-8',
            'format' => 'A4-L',
            'margin_left' => 10,
            'margin_right' => 10,
            'margin_top' => 15,
            'margin_bottom' => 15,
            'default_font_size' => 8,
            'default_font' => 'arial',
            'tempDir' => __DIR__ . '/tmp'
        ]);
        
        $mpdf->SetTitle('Detalle de Cargas - ' . $fecha_carga);
        $mpdf->SetAuthor('Sistema Postgrado');
        
        $html = '<!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <style>
                body { font-family: Arial, sans-serif; font-size: 7px; }
                table { width: 100%; border-collapse: collapse; margin-bottom: 10px; }
                th, td { border: 1px solid #000; padding: 3px; text-align: left; }
                th { background-color: #f2f2f2; font-weight: bold; }
                .title { text-align: center; font-size: 12px; font-weight: bold; margin-bottom: 5px; }
                .subtitle { text-align: center; font-size: 9px; margin-bottom: 8px; }
                .stats { margin-bottom: 8px; padding: 6px; background-color: #f8f9fa; border: 1px solid #ddd; font-size: 8px; }
                .stat-item { display: inline-block; margin-right: 15px; font-weight: bold; }
            </style>
        </head>
        <body>
            <div class="title">DETALLE DE CARGAS - ' . date('d/m/Y', strtotime($fecha_carga)) . '</div>
            <div class="subtitle">Generado: ' . date('d/m/Y H:i:s') . '</div>
            
            <div class="stats">
                <span class="stat-item">Total Cargas: ' . $total_cargas . '</span>
                <span class="stat-item">Usuarios Activos: ' . $usuarios_activos . '</span>
                <span class="stat-item">Primera Carga: ' . date('H:i', strtotime($primera_carga)) . '</span>
                <span class="stat-item">Última Carga: ' . date('H:i', strtotime($ultima_carga)) . '</span>
            </div>
            
            <table>
                <thead>
                    <tr>
                        <th>Hora</th>
                        <th>Usuario</th>
                        <th>Rol</th>
                        <th>Estudiante</th>
                        <th>Asignatura</th>
                        <th>Código</th>
                        <th>Postgrado</th>
                        <th>Escolaridad</th>
                        <th>Período</th>
                        <th>Nota</th>
                    </tr>
                </thead>
                <tbody>';
        
        foreach ($detalle_pdf as $item) {
            $html .= '<tr>
                    <td>' . date('H:i', strtotime($item['fecha_modificacion'])) . '</td>
                    <td>' . htmlspecialchars($item['usuario_nombre'] . ' ' . $item['usuario_apellido']) . '</td>
                    <td>' . htmlspecialchars($item['usuario_rol']) . '</td>
                    <td>' . htmlspecialchars($item['estudiante_nombre'] . ' ' . $item['estudiante_apellido']) . '</td>
                    <td>' . htmlspecialchars($item['asignatura_nombre']) . '</td>
                    <td>' . htmlspecialchars($item['asignatura_codigo']) . '</td>
                    <td>' . htmlspecialchars($item['postgrado_nombre']) . '</td>
                    <td>' . htmlspecialchars($item['escolaridad_nombre']) . '</td>
                    <td>' . htmlspecialchars($item['periodo_nombre']) . '</td>
                    <td>' . htmlspecialchars($item['nota_definitiva']) . '</td>
                </tr>';
        }
        
        $html .= '</tbody>
            </table>
            <div style="margin-top: 10px; font-size: 6px; text-align: center;">
                Total de registros: ' . count($detalle_pdf) . '
            </div>
        </body>
        </html>';
        
        $mpdf->WriteHTML($html);
        $filename = "detalle_fecha_carga_" . date('Y-m-d_H-i-s') . ".pdf";
        $mpdf->Output($filename, \Mpdf\Output\Destination::DOWNLOAD);
        exit;
        
    } catch (\Mpdf\MpdfException $e) {
        die("ERROR al generar PDF: " . $e->getMessage());
    }
}

$page_title = "Detalle de Cargas - " . date('d/m/Y', strtotime($fecha_carga));
include '../includes/header.php';
?>

<div class="container-fluid">
    <div class="mb-3">
        <a href="monitoreo_cargas.php?vista=fechas&usuario_id=<?= $usuario_id ?>&postgrado_id=<?= $postgrado_id ?>" class="btn btn-secondary btn-sm">
            <i class="bi bi-arrow-left"></i> Volver al Monitoreo
        </a>
    </div>

    <div class="card mb-4">
        <div class="card-header bg-primary text-white d-flex justify-content-between align-items-center">
            <h4 class="mb-0">
                <i class="bi bi-calendar-check"></i> 
                Detalle de Cargas: <?= date('d/m/Y', strtotime($fecha_carga)) ?>
            </h4>
            <div class="d-flex gap-2">
                <span class="badge bg-light text-dark">
                    Página <?= $pagina_actual ?> de <?= $total_paginas ?> 
                    (<?= $total_registros ?> registros)
                </span>
                <!-- Botones de exportación -->
                <?php if (!empty($detalle)): ?>
                    <a href="?<?php echo http_build_query(array_merge($_GET, ['generar_excel' => 1])); ?>" 
                       class="btn btn-success btn-sm">
                        <i class="bi bi-file-earmark-excel"></i> Excel
                    </a>
                    <a href="?<?php echo http_build_query(array_merge($_GET, ['generar_pdf' => 1])); ?>" 
                       class="btn btn-danger btn-sm">
                        <i class="bi bi-file-earmark-pdf"></i> PDF
                    </a>
                <?php endif; ?>
            </div>
        </div>
        <div class="card-body">
            <div class="row">
                <div class="col-md-3">
                    <div class="card bg-light">
                        <div class="card-body text-center">
                            <h5 class="card-title">Total Cargas</h5>
                            <h2 class="text-primary"><?= $total_cargas ?></h2>
                        </div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="card bg-success text-white">
                        <div class="card-body text-center">
                            <h5 class="card-title">Usuarios Activos</h5>
                            <h2><?= $usuarios_activos ?></h2>
                        </div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="card bg-info text-white">
                        <div class="card-body text-center">
                            <h5 class="card-title">Primera Carga</h5>
                            <h4><?= date('H:i', strtotime($primera_carga)) ?></h4>
                        </div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="card bg-warning">
                        <div class="card-body text-center">
                            <h5 class="card-title">Última Carga</h5>
                            <h4><?= date('H:i', strtotime($ultima_carga)) ?></h4>
                        </div>
                    </div>
                </div>
            </div>

            <?php if (!empty($responsables)): ?>
                <div class="mt-3 p-3 bg-light rounded">
                    <strong>Responsables:</strong> <?= htmlspecialchars($responsables) ?>
                </div>
            <?php endif; ?>
            
            <!-- Mensaje si no hay registros -->
            <?php if (empty($detalle) && $total_cargas == 0): ?>
                <div class="alert alert-warning mt-3">
                    <i class="bi bi-exclamation-triangle"></i> 
                    No se encontraron cargas de notas para la fecha <strong><?= date('d/m/Y', strtotime($fecha_carga)) ?></strong>.
                </div>
            <?php endif; ?>
        </div>
    </div>

    <?php if (!empty($detalle)): ?>
    <div class="card">
        <div class="card-header bg-info text-white d-flex justify-content-between align-items-center">
            <h5 class="mb-0"><i class="bi bi-list-check"></i> Detalle de Cargas</h5>
            <div class="d-flex gap-2">
                <a href="?<?php echo http_build_query(array_merge($_GET, ['generar_excel' => 1])); ?>" 
                   class="btn btn-light btn-sm">
                    <i class="bi bi-file-earmark-excel"></i> Excel
                </a>
                <a href="?<?php echo http_build_query(array_merge($_GET, ['generar_pdf' => 1])); ?>" 
                   class="btn btn-light btn-sm">
                    <i class="bi bi-file-earmark-pdf"></i> PDF
                </a>
            </div>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-striped table-hover">
                    <thead>
                        <tr>
                            <th>Hora</th>
                            <th>Usuario</th>
                            <th>Estudiante</th>
                            <th>Asignatura</th>
                            <th>Postgrado</th>
                            <th>Escolaridad</th>
                            <th>Período</th>
                            <th>Nota</th>
                            <th>Modificaciones</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($detalle as $item): ?>
                            <tr>
                                <td>
                                    <span class="badge bg-dark"><?= date('H:i', strtotime($item['fecha_modificacion'])) ?></span>
                                </td>
                                <td>
                                    <small><?= htmlspecialchars($item['usuario_nombre'] . ' ' . $item['usuario_apellido']) ?></small>
                                    <br><span class="badge bg-secondary"><?= htmlspecialchars($item['usuario_rol']) ?></span>
                                </td>
                                <td><?= htmlspecialchars($item['estudiante_nombre'] . ' ' . $item['estudiante_apellido']) ?></td>
                                <td>
                                    <small class="text-muted"><?= htmlspecialchars($item['asignatura_codigo']) ?></small><br>
                                    <?= htmlspecialchars($item['asignatura_nombre']) ?>
                                </td>
                                <td><?= htmlspecialchars($item['postgrado_nombre']) ?></td>
                                <td>
                                    <span class="badge bg-info"><?= htmlspecialchars($item['escolaridad_nombre']) ?></span>
                                </td>
                                <td>
                                    <small class="text-primary"><?= htmlspecialchars($item['periodo_nombre']) ?></small>
                                </td>
                                <td>
                                    <span class="badge bg-success"><?= htmlspecialchars($item['nota_definitiva']) ?></span>
                                </td>
                                <td>
                                    <span class="badge bg-<?= $item['veces_modificada'] > 1 ? 'warning' : 'secondary' ?>">
                                        <?= $item['veces_modificada'] ?> vez(es)
                                    </span>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <!-- Paginación -->
            <?php if ($total_paginas > 1): ?>
                <nav aria-label="Paginación">
                    <ul class="pagination justify-content-center">
                        <?php if ($pagina_actual > 1): ?>
                            <li class="page-item">
                                <a class="page-link" href="?<?php 
                                    echo http_build_query(array_merge($_GET, ['pagina' => $pagina_actual - 1]));
                                ?>" aria-label="Anterior">
                                    <span aria-hidden="true">&laquo;</span>
                                </a>
                            </li>
                        <?php endif; ?>

                        <?php 
                        // Mostrar números de página
                        $inicio = max(1, $pagina_actual - 2);
                        $fin = min($total_paginas, $pagina_actual + 2);
                        
                        for ($i = $inicio; $i <= $fin; $i++): 
                        ?>
                            <li class="page-item <?= $i == $pagina_actual ? 'active' : '' ?>">
                                <a class="page-link" href="?<?php 
                                    echo http_build_query(array_merge($_GET, ['pagina' => $i]));
                                ?>"><?= $i ?></a>
                            </li>
                        <?php endfor; ?>

                        <?php if ($pagina_actual < $total_paginas): ?>
                            <li class="page-item">
                                <a class="page-link" href="?<?php 
                                    echo http_build_query(array_merge($_GET, ['pagina' => $pagina_actual + 1]));
                                ?>" aria-label="Siguiente">
                                    <span aria-hidden="true">&raquo;</span>
                                </a>
                            </li>
                        <?php endif; ?>
                    </ul>
                </nav>
                
                <div class="text-center text-muted small">
                    Mostrando <?= count($detalle) ?> de <?= $total_registros ?> registros
                </div>
            <?php endif; ?>
        </div>
    </div>
    <?php endif; ?>
</div>

<?php
include '../includes/footer.php';
$conn->close();
?>