<?php
// 1. Centralized Authentication Check
require_once __DIR__ . '/includes/auth_check.php';

// 2. Database connection (already included via auth_check -> header, but good practice to ensure)
require_once __DIR__ . '/includes/db.php';

$feedback = [];
$user_id = $_SESSION['user_id'];
$user_rol = $_SESSION['rol'];

// Obtener información completa del usuario desde la base de datos
$user_data = [];
$stmt_user = $conn->prepare("SELECT username, nombres, apellidos, rol, email, telefono, direccion FROM usuarios WHERE id = ?");
$stmt_user->bind_param("i", $user_id);
$stmt_user->execute();
$result_user = $stmt_user->get_result();

if ($result_user->num_rows > 0) {
    $user_data = $result_user->fetch_assoc();
} else {
    $feedback = ['type' => 'danger', 'message' => 'Error: No se pudo cargar la información del usuario.'];
}

// Determinar qué campos pueden ser editados según el rol
$can_edit_profile = ($user_rol === 'administrador');
$show_contact_info = ($user_rol === 'administrador' || $user_rol === 'coordinador2');

// 3. Handle Password Change Request
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['change_password'])) {
    $current_password = $_POST['current_password'];
    $new_password = $_POST['new_password'];
    $confirm_password = $_POST['confirm_password'];

    // Validations
    if (empty($current_password) || empty($new_password) || empty($confirm_password)) {
        $feedback = ['type' => 'danger', 'message' => 'Todos los campos son obligatorios.'];
    } elseif ($new_password !== $confirm_password) {
        $feedback = ['type' => 'danger', 'message' => 'La nueva contraseña y su confirmación no coinciden.'];
    } elseif (strlen($new_password) < 8) {
        $feedback = ['type' => 'danger', 'message' => 'La nueva contraseña debe tener al menos 8 caracteres.'];
    } else {
        // Fetch current password from DB
        $stmt = $conn->prepare("SELECT password FROM usuarios WHERE id = ?");
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $user = $result->fetch_assoc();

        if ($user && password_verify($current_password, $user['password'])) {
            // Current password is correct, proceed to update
            $hashed_new_password = password_hash($new_password, PASSWORD_DEFAULT);
            
            $update_stmt = $conn->prepare("UPDATE usuarios SET password = ? WHERE id = ?");
            $update_stmt->bind_param("si", $hashed_new_password, $user_id);
            
            if ($update_stmt->execute()) {
                $feedback = ['type' => 'success', 'message' => '¡Contraseña actualizada con éxito!'];
                // Limpiar campos de contraseña
                $_POST['current_password'] = $_POST['new_password'] = $_POST['confirm_password'] = '';
            } else {
                $feedback = ['type' => 'danger', 'message' => 'Error al actualizar la contraseña. Por favor, intente de nuevo.'];
            }
        } else {
            // Invalid current password
            $feedback = ['type' => 'danger', 'message' => 'La contraseña actual es incorrecta.'];
        }
    }
}

// 4. Handle Profile Information Update (solo para administrador)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_profile']) && $can_edit_profile) {
    $nombres = trim($_POST['nombres'] ?? '');
    $apellidos = trim($_POST['apellidos'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $telefono = trim($_POST['telefono'] ?? '');
    $direccion = trim($_POST['direccion'] ?? '');

    // Validaciones básicas
    if (empty($nombres) || empty($apellidos)) {
        $feedback = ['type' => 'danger', 'message' => 'Los campos Nombres y Apellidos son obligatorios.'];
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL) && !empty($email)) {
        $feedback = ['type' => 'danger', 'message' => 'El formato del correo electrónico no es válido.'];
    } else {
        // Actualizar información del perfil
        $update_stmt = $conn->prepare("UPDATE usuarios SET nombres = ?, apellidos = ?, email = ?, telefono = ?, direccion = ? WHERE id = ?");
        $update_stmt->bind_param("sssssi", $nombres, $apellidos, $email, $telefono, $direccion, $user_id);
        
        if ($update_stmt->execute()) {
            $feedback = ['type' => 'success', 'message' => '¡Información del perfil actualizada con éxito!'];
            // Actualizar los datos locales
            $user_data['nombres'] = $nombres;
            $user_data['apellidos'] = $apellidos;
            $user_data['email'] = $email;
            $user_data['telefono'] = $telefono;
            $user_data['direccion'] = $direccion;
        } else {
            $feedback = ['type' => 'danger', 'message' => 'Error al actualizar la información del perfil. Por favor, intente de nuevo.'];
        }
    }
}

$page_title = "Mi Perfil";
include __DIR__ . '/includes/header.php';
?>

<div class="container mt-4">
    <h2><i class="fas fa-user-circle me-2"></i>Mi Perfil</h2>
    <hr>

    <?php if (!empty($feedback)): ?>
        <div class="alert alert-<?= htmlspecialchars($feedback['type']) ?> alert-dismissible fade show">
            <?= htmlspecialchars($feedback['message']) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    <?php endif; ?>

    <div class="row">
        <!-- User Information Column -->
        <div class="col-md-6">
            <div class="card">
                <div class="card-header bg-primary text-white">
                    <h5 class="mb-0"><i class="fas fa-user-edit me-2"></i>Información Personal</h5>
                </div>
                <div class="card-body">
                    <?php if ($can_edit_profile): ?>
                        <!-- Formulario editable para administrador -->
                        <form method="POST" action="perfil.php">
                            <div class="mb-3">
                                <label for="nombres" class="form-label fw-bold">Nombres</label>
                                <input type="text" class="form-control" id="nombres" name="nombres" 
                                       value="<?= htmlspecialchars($user_data['nombres'] ?? '') ?>" required>
                            </div>
                            <div class="mb-3">
                                <label for="apellidos" class="form-label fw-bold">Apellidos</label>
                                <input type="text" class="form-control" id="apellidos" name="apellidos" 
                                       value="<?= htmlspecialchars($user_data['apellidos'] ?? '') ?>" required>
                            </div>
                            
                            <?php if ($show_contact_info): ?>
                            <div class="mb-3">
                                <label for="email" class="form-label fw-bold">Correo Electrónico</label>
                                <div class="input-group">
                                    <input type="password" class="form-control sensitive-input" id="email" name="email" 
                                           value="<?= htmlspecialchars($user_data['email'] ?? '') ?>" data-real-type="email">
                                    <button type="button" class="btn btn-outline-secondary toggle-sensitive" data-target="email">
                                        <i class="fas fa-eye"></i>
                                    </button>
                                </div>
                            </div>
                            <div class="mb-3">
                                <label for="telefono" class="form-label fw-bold">Teléfono</label>
                                <div class="input-group">
                                    <input type="password" class="form-control sensitive-input" id="telefono" name="telefono" 
                                           value="<?= htmlspecialchars($user_data['telefono'] ?? '') ?>" data-real-type="text">
                                    <button type="button" class="btn btn-outline-secondary toggle-sensitive" data-target="telefono">
                                        <i class="fas fa-eye"></i>
                                    </button>
                                </div>
                            </div>
                            <div class="mb-3">
                                <label for="direccion" class="form-label fw-bold">Dirección</label>
                                <div class="input-group">
                                    <textarea class="form-control sensitive-textarea" id="direccion" name="direccion" rows="3" 
                                              style="resize: none;" data-real-value="<?= htmlspecialchars($user_data['direccion'] ?? '') ?>">••••••••</textarea>
                                    <button type="button" class="btn btn-outline-secondary toggle-sensitive-textarea align-self-start" data-target="direccion" style="height: fit-content;">
                                        <i class="fas fa-eye"></i>
                                    </button>
                                </div>
                            </div>
                            <?php endif; ?>
                            
                            <button type="submit" name="update_profile" class="btn btn-primary w-100">
                                <i class="fas fa-save me-2"></i>Actualizar Información
                            </button>
                        </form>
                    <?php else: ?>
                        <!-- Información de solo lectura para otros roles -->
                        <div class="mb-3">
                            <label class="form-label fw-bold">Nombres</label>
                            <p class="form-control-plaintext bg-light p-2 rounded"><?= htmlspecialchars($user_data['nombres'] ?? 'No disponible') ?></p>
                        </div>
                        <div class="mb-3">
                            <label class="form-label fw-bold">Apellidos</label>
                            <p class="form-control-plaintext bg-light p-2 rounded"><?= htmlspecialchars($user_data['apellidos'] ?? 'No disponible') ?></p>
                        </div>
                        
                        <?php if ($show_contact_info): ?>
                        <div class="mb-3">
                            <label class="form-label fw-bold">Correo Electrónico</label>
                            <div class="input-group">
                                <p class="form-control-plaintext bg-light p-2 rounded me-2 sensitive-data" 
                                   data-value="<?= htmlspecialchars($user_data['email'] ?? 'No disponible') ?>" 
                                   style="margin-bottom: 0; flex: 1;">
                                    ••••••••
                                </p>
                                <button type="button" class="btn btn-outline-secondary toggle-sensitive-data" data-target="email">
                                    <i class="fas fa-eye"></i>
                                </button>
                            </div>
                        </div>
                        <div class="mb-3">
                            <label class="form-label fw-bold">Teléfono</label>
                            <div class="input-group">
                                <p class="form-control-plaintext bg-light p-2 rounded me-2 sensitive-data" 
                                   data-value="<?= htmlspecialchars($user_data['telefono'] ?? 'No disponible') ?>" 
                                   style="margin-bottom: 0; flex: 1;">
                                    ••••••••
                                </p>
                                <button type="button" class="btn btn-outline-secondary toggle-sensitive-data" data-target="telefono">
                                    <i class="fas fa-eye"></i>
                                </button>
                            </div>
                        </div>
                        <div class="mb-3">
                            <label class="form-label fw-bold">Dirección</label>
                            <div class="input-group">
                                <p class="form-control-plaintext bg-light p-2 rounded me-2 sensitive-data" 
                                   data-value="<?= htmlspecialchars($user_data['direccion'] ?? 'No disponible') ?>" 
                                   style="margin-bottom: 0; flex: 1; min-height: 38px;">
                                    ••••••••
                                </p>
                                <button type="button" class="btn btn-outline-secondary toggle-sensitive-data align-self-start" data-target="direccion" style="height: fit-content;">
                                    <i class="fas fa-eye"></i>
                                </button>
                            </div>
                        </div>
                        <?php endif; ?>
                        
                        <div class="alert alert-info">
                            <i class="fas fa-info-circle me-2"></i>
                            <strong>Información:</strong> Para modificar sus datos personales, contacte al administrador del sistema.
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Change Password Column -->
        <div class="col-md-6">
            <div class="card">
                <div class="card-header bg-info text-white">
                    <h5 class="mb-0"><i class="fas fa-key me-2"></i>Cambiar Contraseña</h5>
                </div>
                <div class="card-body">
                    <form method="POST" action="perfil.php" id="passwordForm">
                        <div class="mb-3">
                            <label for="current_password" class="form-label">Contraseña Actual</label>
                            <div class="input-group">
                                <input type="password" class="form-control" id="current_password" name="current_password" required>
                                <button type="button" class="btn btn-outline-secondary toggle-password" data-target="current_password">
                                    <i class="fas fa-eye"></i>
                                </button>
                            </div>
                            <div class="form-text">Ingrese su contraseña actual para autorizar el cambio.</div>
                        </div>
                        <div class="mb-3">
                            <label for="new_password" class="form-label">Nueva Contraseña</label>
                            <div class="input-group">
                                <input type="password" class="form-control" id="new_password" name="new_password" required minlength="8">
                                <button type="button" class="btn btn-outline-secondary toggle-password" data-target="new_password">
                                    <i class="fas fa-eye"></i>
                                </button>
                            </div>
                            <div class="form-text">Mínimo 8 caracteres. Recomendamos usar letras, números y símbolos.</div>
                        </div>
                        <div class="mb-3">
                            <label for="confirm_password" class="form-label">Confirmar Nueva Contraseña</label>
                            <div class="input-group">
                                <input type="password" class="form-control" id="confirm_password" name="confirm_password" required minlength="8">
                                <button type="button" class="btn btn-outline-secondary toggle-password" data-target="confirm_password">
                                    <i class="fas fa-eye"></i>
                                </button>
                            </div>
                            <div class="form-text">Repita exactamente la nueva contraseña.</div>
                        </div>
                        <button type="submit" name="change_password" class="btn btn-info w-100 text-white">
                            <i class="fas fa-save me-2"></i>Actualizar Contraseña
                        </button>
                    </form>
                </div>
            </div>

            <!-- Información de la Cuenta (solo lectura) -->
            <div class="card mt-4">
                <div class="card-header bg-secondary text-white">
                    <h5 class="mb-0"><i class="fas fa-info-circle me-2"></i>Información de la Cuenta</h5>
                </div>
                <div class="card-body">
                    <div class="mb-3">
                        <label class="form-label fw-bold">Nombre de Usuario</label>
                        <p class="form-control-plaintext bg-light p-2 rounded"><?= htmlspecialchars($user_data['username'] ?? '') ?></p>
                    </div>
                    <div class="mb-3">
                        <label class="form-label fw-bold">Rol</label>
                        <p class="form-control-plaintext">
                            <span class="badge bg-primary"><?= htmlspecialchars(ucfirst($user_data['rol'] ?? '')) ?></span>
                        </p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
.card {
    box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    border: 1px solid rgba(0, 0, 0, 0.125);
    margin-bottom: 1rem;
}
.card-header {
    border-bottom: 1px solid rgba(0, 0, 0, 0.125);
}
.form-control-plaintext {
    min-height: 38px;
    padding: 0.375rem 0.75rem;
}
.toggle-password, .toggle-sensitive, .toggle-sensitive-data, .toggle-sensitive-textarea {
    border-left: 0;
}
.input-group .form-control:focus {
    z-index: 3;
}
.sensitive-data {
    cursor: pointer;
    user-select: none;
}
/* Deshabilitar el redimensionamiento del textarea */
.sensitive-textarea {
    resize: none;
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Validación de contraseñas en tiempo real
    const newPassword = document.getElementById('new_password');
    const confirmPassword = document.getElementById('confirm_password');
    
    function validatePasswords() {
        if (newPassword.value !== confirmPassword.value && confirmPassword.value !== '') {
            confirmPassword.setCustomValidity('Las contraseñas no coinciden');
            confirmPassword.classList.add('is-invalid');
        } else {
            confirmPassword.setCustomValidity('');
            confirmPassword.classList.remove('is-invalid');
        }
    }
    
    if (newPassword && confirmPassword) {
        newPassword.addEventListener('input', validatePasswords);
        confirmPassword.addEventListener('input', validatePasswords);
    }
    
    // Función para mostrar/ocultar contraseña
    function togglePasswordVisibility(inputId) {
        const input = document.getElementById(inputId);
        const icon = input.parentNode.querySelector('i');
        
        if (input.type === 'password') {
            input.type = 'text';
            icon.classList.remove('fa-eye');
            icon.classList.add('fa-eye-slash');
        } else {
            input.type = 'password';
            icon.classList.remove('fa-eye-slash');
            icon.classList.add('fa-eye');
        }
    }
    
    // Función para mostrar/ocultar datos sensibles en inputs
    function toggleSensitiveVisibility(inputId) {
        const input = document.getElementById(inputId);
        const icon = input.parentNode.querySelector('i');
        const realType = input.getAttribute('data-real-type');
        
        if (input.type === 'password') {
            input.type = realType;
            icon.classList.remove('fa-eye');
            icon.classList.add('fa-eye-slash');
        } else {
            input.type = 'password';
            icon.classList.remove('fa-eye-slash');
            icon.classList.add('fa-eye');
        }
    }
    
    // Función para mostrar/ocultar datos sensibles en textarea
    function toggleSensitiveTextarea(textareaId) {
        const textarea = document.getElementById(textareaId);
        const icon = textarea.parentNode.querySelector('i');
        const realValue = textarea.getAttribute('data-real-value');
        
        if (textarea.value === '••••••••') {
            // Mostrar el valor real
            textarea.value = realValue;
            icon.classList.remove('fa-eye');
            icon.classList.add('fa-eye-slash');
            icon.title = 'Ocultar información';
        } else {
            // Ocultar con asteriscos
            textarea.value = '••••••••';
            icon.classList.remove('fa-eye-slash');
            icon.classList.add('fa-eye');
            icon.title = 'Mostrar información';
        }
    }
    
    // Función para mostrar/ocultar datos sensibles en visualización
    function toggleSensitiveDataDisplay(element) {
        const dataElement = element.previousElementSibling;
        const icon = element.querySelector('i');
        
        if (dataElement.textContent === '••••••••') {
            // Mostrar el valor real
            dataElement.textContent = dataElement.getAttribute('data-value');
            icon.classList.remove('fa-eye');
            icon.classList.add('fa-eye-slash');
            element.title = 'Ocultar información';
        } else {
            // Ocultar con asteriscos
            dataElement.textContent = '••••••••';
            icon.classList.remove('fa-eye-slash');
            icon.classList.add('fa-eye');
            element.title = 'Mostrar información';
        }
    }
    
    // Agregar event listeners a los botones de mostrar/ocultar contraseña
    document.querySelectorAll('.toggle-password').forEach(button => {
        button.addEventListener('click', function() {
            const target = this.getAttribute('data-target');
            togglePasswordVisibility(target);
        });
    });
    
    // Agregar event listeners a los botones de mostrar/ocultar datos sensibles en inputs
    document.querySelectorAll('.toggle-sensitive').forEach(button => {
        button.addEventListener('click', function() {
            const target = this.getAttribute('data-target');
            toggleSensitiveVisibility(target);
        });
    });
    
    // Agregar event listeners a los botones de mostrar/ocultar datos sensibles en textarea
    document.querySelectorAll('.toggle-sensitive-textarea').forEach(button => {
        button.addEventListener('click', function() {
            const target = this.getAttribute('data-target');
            toggleSensitiveTextarea(target);
        });
    });
    
    // Agregar event listeners a los botones de mostrar/ocultar datos sensibles en visualización
    document.querySelectorAll('.toggle-sensitive-data').forEach(button => {
        button.addEventListener('click', function() {
            toggleSensitiveDataDisplay(this);
        });
    });
    
    // También permitir hacer clic directamente en el texto para mostrar/ocultar
    document.querySelectorAll('.sensitive-data').forEach(element => {
        element.addEventListener('click', function() {
            const button = this.nextElementSibling;
            if (button && button.classList.contains('toggle-sensitive-data')) {
                toggleSensitiveDataDisplay(button);
            }
        });
    });
    
    // Inicializar campos sensibles como ocultos
    document.querySelectorAll('.sensitive-input').forEach(input => {
        input.type = 'password';
    });
    
    document.querySelectorAll('.sensitive-textarea').forEach(textarea => {
        textarea.value = '••••••••';
    });
    
    // Limpiar campos de contraseña después de enviar el formulario exitosamente
    const passwordForm = document.getElementById('passwordForm');
    if (passwordForm) {
        passwordForm.addEventListener('submit', function() {
            setTimeout(() => {
                if (document.querySelector('.alert-success')) {
                    // Si hay un mensaje de éxito, limpiar los campos
                    document.getElementById('current_password').value = '';
                    document.getElementById('new_password').value = '';
                    document.getElementById('confirm_password').value = '';
                    // Resetear iconos de ojos
                    document.querySelectorAll('.toggle-password i, .toggle-sensitive i, .toggle-sensitive-textarea i').forEach(icon => {
                        icon.classList.remove('fa-eye-slash');
                        icon.classList.add('fa-eye');
                    });
                    // Resetear campos sensibles a estado oculto
                    document.querySelectorAll('.sensitive-input').forEach(input => {
                        input.type = 'password';
                    });
                    document.querySelectorAll('.sensitive-textarea').forEach(textarea => {
                        textarea.value = '••••••••';
                    });
                }
            }, 100);
        });
    }
});
</script>

<?php
include __DIR__ . '/includes/footer.php';
?>