/**
 * session_manager.js - VERSIÓN CORREGIDA
 * Gestiona la expiración de sesión SIN redirecciones a /v2/
 */

document.addEventListener('DOMContentLoaded', () => {
    console.log('🔐 Session Manager iniciado');
    
    // Verificar configuración
    if (typeof sessionConfig === 'undefined') {
        console.error('❌ Session Manager: No hay configuración de sesión');
        return;
    }

    // Si la sesión nunca expira, no hacer nada
    if (sessionConfig.neverExpire == 1 || sessionConfig.neverExpire == '1') {
        console.log('✅ Sesión configurada para no expirar');
        return;
    }

    let sessionTimeout, warningTimeout;
    const warningTime = sessionConfig.warningTime || 300;
    const duration = parseInt(sessionConfig.duration) || 3600;
    const durationInMs = duration * 1000;
    const warningTimeInMs = warningTime * 1000;

    // ✅ CORREGIDO: Usar detección automática de baseUrl
    const baseUrl = detectBaseUrl();
    console.log('📍 Base URL detectada:', baseUrl);

    function detectBaseUrl() {
        // 1. Primero intentar con BASE_URL inyectado desde PHP
        if (typeof BASE_URL !== 'undefined' && BASE_URL) {
            return BASE_URL;
        }
        
        // 2. Detección automática desde la URL actual
        const currentPath = window.location.pathname;
        if (currentPath.includes('/cepmed/')) {
            return '/cepmed';
        }
        if (currentPath.includes('/v23/')) {
            return '/v23';
        }
        if (currentPath.includes('/v2/')) {
            return '/v2';
        }
        
        // 3. Fallback seguro - NUNCA /v2/
        return '/cepmed';
    }

    const startTimers = () => {
        clearTimeout(sessionTimeout);
        clearTimeout(warningTimeout);

        if (durationInMs > warningTimeInMs) {
            warningTimeout = setTimeout(showWarningModal, durationInMs - warningTimeInMs);
            console.log(`⏰ Advertencia en ${(durationInMs - warningTimeInMs)/1000/60} min`);
        }
        
        sessionTimeout = setTimeout(forceLogout, durationInMs);
    };

    // ✅ CORREGIDO: NUNCA usar /v2/ hardcodeado
    const forceLogout = () => {
        console.log('🔒 Sesión expirada - Forzando logout');
        // ✅ SIEMPRE usar baseUrl dinámico
        window.location.href = baseUrl + '/logout.php?reason=session_expired';
    };

    const showWarningModal = () => {
        const modal = document.getElementById('sessionWarningModal');
        if (!modal) {
            console.log('ℹ️ Modal de advertencia no encontrado');
            return;
        }

        const modalInstance = new bootstrap.Modal(modal);
        modalInstance.show();

        let countdown = warningTime;
        const countdownElement = document.getElementById('sessionCountdown');

        const updateCountdown = () => {
            const minutes = Math.floor(countdown / 60);
            const seconds = countdown % 60;
            if (countdownElement) {
                countdownElement.textContent = `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
            }
            countdown--;
            
            if (countdown < 0) {
                clearInterval(countdownInterval);
                modalInstance.hide();
            }
        };

        updateCountdown();
        const countdownInterval = setInterval(updateCountdown, 1000);
    };

    // ✅ CORREGIDO: Usar baseUrl dinámico para keep_alive
    const resetSession = async () => {
        try {
            console.log('🔄 Extendiendo sesión...');
            const response = await fetch(baseUrl + '/api/keep_alive.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                credentials: 'include'
            });

            if (response.ok) {
                const data = await response.json();
                if (data.status === 'success') {
                    console.log('✅ Sesión extendida');
                    startTimers();
                    return;
                }
            }
            console.error('❌ Error en keep_alive:', response.status);
        } catch (error) {
            console.error('❌ Error de red:', error);
        }
        
        // En caso de error, igual reiniciar timers
        startTimers();
    };

    // Event listeners
    let activityTimeout;
    const handleUserActivity = () => {
        clearTimeout(activityTimeout);
        activityTimeout = setTimeout(resetSession, 1000);
    };

    ['click', 'mousemove', 'keypress', 'scroll', 'touchstart'].forEach(event => {
        document.addEventListener(event, handleUserActivity, { passive: true });
    });

    // Botones del modal
    document.getElementById('extendSessionBtn')?.addEventListener('click', () => {
        resetSession();
        bootstrap.Modal.getInstance(document.getElementById('sessionWarningModal'))?.hide();
    });

    document.getElementById('logoutSessionBtn')?.addEventListener('click', forceLogout);

    // Iniciar
    startTimers();
});