<?php
// Centralized authentication and session check
$allowed_roles = ['administrador'];
require_once __DIR__ . '/../includes/auth_check.php';

include '../includes/db.php';

// Configurar conexión para usar UTF-8
$conn->set_charset("utf8mb4");
$conn->query("SET NAMES 'utf8mb4'");
$conn->query("SET CHARACTER SET utf8mb4");
$conn->query("SET COLLATION_CONNECTION = 'utf8mb4_unicode_ci'");

// Configuración de paginación
$postgrados_por_pagina = 25;
$estudiantes_por_pagina = 25; // Paginación para estudiantes

// Paginación para Postgrados con Bajo Promedio
$pagina_postgrados = isset($_GET['pagina_postgrados']) ? (int)$_GET['pagina_postgrados'] : 1;
if ($pagina_postgrados < 1) $pagina_postgrados = 1;

// Paginación para Bajos Promedios
$pagina_bajos_promedios = isset($_GET['pagina_bajos_promedios']) ? (int)$_GET['pagina_bajos_promedios'] : 1;
if ($pagina_bajos_promedios < 1) $pagina_bajos_promedios = 1;

// Paginación para Asignaturas Reprobadas
$pagina_reprobados = isset($_GET['pagina_reprobados']) ? (int)$_GET['pagina_reprobados'] : 1;
if ($pagina_reprobados < 1) $pagina_reprobados = 1;

// Opciones de ordenamiento para postgrados
$orden = isset($_GET['orden']) ? $_GET['orden'] : 'nombre';
$direccion = isset($_GET['dir']) ? $_GET['dir'] : 'asc';

// Validar parámetros de ordenamiento
$ordenes_permitidos = ['nombre', 'estudiantes_activos', 'promedio_simple', 'promedio_ponderado'];
$direcciones_permitidas = ['asc', 'desc'];

if (!in_array($orden, $ordenes_permitidos)) $orden = 'nombre';
if (!in_array($direccion, $direcciones_permitidas)) $direccion = 'asc';

// Consulta para obtener TODOS los postgrados que tienen estudiantes con promedio bajo
$query_postgrados_bajo_promedio = "SELECT p.id, p.nombre
                                  FROM postgrados p
                                  WHERE EXISTS (
                                      SELECT 1 FROM estudiante_postgrado ep 
                                      JOIN estudiantes e ON ep.estudiante_id = e.id
                                      JOIN (
                                          SELECT i.estudiante_id, 
                                                 AVG(CASE 
                                                     WHEN i.nota_definitiva IS NOT NULL AND i.nota_definitiva != 'Aprobado' AND i.nota_definitiva != 'Reprobado'
                                                     AND i.nota_definitiva REGEXP '^[0-9]+(\.[0-9]+)?$'
                                                     THEN CAST(i.nota_definitiva AS DECIMAL(10,2)) 
                                                 END) as promedio_simple,
                                                 SUM(CASE 
                                                     WHEN i.nota_definitiva IS NOT NULL AND i.nota_definitiva != 'Aprobado' AND i.nota_definitiva != 'Reprobado'
                                                     AND i.nota_definitiva REGEXP '^[0-9]+(\.[0-9]+)?$'
                                                     THEN CAST(i.nota_definitiva AS DECIMAL(10,2)) * d.credito
                                                 END) / NULLIF(SUM(CASE 
                                                     WHEN i.nota_definitiva IS NOT NULL AND i.nota_definitiva != 'Aprobado' AND i.nota_definitiva != 'Reprobado'
                                                     AND i.nota_definitiva REGEXP '^[0-9]+(\.[0-9]+)?$'
                                                     THEN d.credito 
                                                 END), 0) as promedio_ponderado
                                          FROM inscripciones i
                                          JOIN denominaciones d ON i.denominacion_id = d.id
                                          WHERE i.estado = 'finalizada' AND d.tipo = 'puntaje'
                                          GROUP BY i.estudiante_id
                                      ) promedios ON e.id = promedios.estudiante_id
                                      WHERE ep.postgrado_id = p.id 
                                      AND ep.estado = 'activo'
                                      AND (promedios.promedio_simple < 15 OR promedios.promedio_ponderado < 15)
                                  )";

$result_postgrados_bajo_promedio = $conn->query($query_postgrados_bajo_promedio);
$total_postgrados_bajo_promedio = $result_postgrados_bajo_promedio ? $result_postgrados_bajo_promedio->num_rows : 0;
$total_paginas_postgrados = ceil($total_postgrados_bajo_promedio / $postgrados_por_pagina);

// Ajustar página actual si es mayor al total
if ($pagina_postgrados > $total_paginas_postgrados && $total_paginas_postgrados > 0) $pagina_postgrados = $total_paginas_postgrados;

// Calcular offset para la paginación de postgrados
$offset_postgrados = ($pagina_postgrados - 1) * $postgrados_por_pagina;

// Consulta principal para postgrados con bajo promedio
$query_postgrados = "SELECT 
                    p.id, 
                    p.nombre, 
                    COUNT(DISTINCT ep.estudiante_id) as estudiantes_activos,
                    COALESCE(ROUND(AVG(CASE 
                        WHEN i.nota_definitiva IS NOT NULL AND i.nota_definitiva != 'Aprobado' AND i.nota_definitiva != 'Reprobado'
                        AND i.nota_definitiva REGEXP '^[0-9]+(\.[0-9]+)?$'
                        THEN CAST(i.nota_definitiva AS DECIMAL(10,2)) 
                    END), 2), 0) as promedio_simple,
                    COALESCE(ROUND(SUM(CASE 
                        WHEN i.nota_definitiva IS NOT NULL AND i.nota_definitiva != 'Aprobado' AND i.nota_definitiva != 'Reprobado'
                        AND i.nota_definitiva REGEXP '^[0-9]+(\.[0-9]+)?$'
                        THEN CAST(i.nota_definitiva AS DECIMAL(10,2)) * d.credito
                    END) / NULLIF(SUM(CASE 
                        WHEN i.nota_definitiva IS NOT NULL AND i.nota_definitiva != 'Aprobado' AND i.nota_definitiva != 'Reprobado'
                        AND i.nota_definitiva REGEXP '^[0-9]+(\.[0-9]+)?$'
                        THEN d.credito 
                    END), 0), 2), 0) as promedio_ponderado,
                    COUNT(DISTINCT CASE 
                        WHEN (promedios_estudiantes.promedio_simple < 15 OR promedios_estudiantes.promedio_ponderado < 15) 
                        THEN ep.estudiante_id 
                    END) as estudiantes_bajo_promedio
                    FROM postgrados p
                    LEFT JOIN estudiante_postgrado ep ON p.id = ep.postgrado_id AND ep.estado = 'activo'
                    LEFT JOIN estudiantes e ON ep.estudiante_id = e.id
                    LEFT JOIN (
                        SELECT i.estudiante_id, 
                               AVG(CASE 
                                   WHEN i.nota_definitiva IS NOT NULL AND i.nota_definitiva != 'Aprobado' AND i.nota_definitiva != 'Reprobado'
                                   AND i.nota_definitiva REGEXP '^[0-9]+(\.[0-9]+)?$'
                                   THEN CAST(i.nota_definitiva AS DECIMAL(10,2)) 
                               END) as promedio_simple,
                               SUM(CASE 
                                   WHEN i.nota_definitiva IS NOT NULL AND i.nota_definitiva != 'Aprobado' AND i.nota_definitiva != 'Reprobado'
                                   AND i.nota_definitiva REGEXP '^[0-9]+(\.[0-9]+)?$'
                                   THEN CAST(i.nota_definitiva AS DECIMAL(10,2)) * d.credito
                               END) / NULLIF(SUM(CASE 
                                   WHEN i.nota_definitiva IS NOT NULL AND i.nota_definitiva != 'Aprobado' AND i.nota_definitiva != 'Reprobado'
                                   AND i.nota_definitiva REGEXP '^[0-9]+(\.[0-9]+)?$'
                                   THEN d.credito 
                               END), 0) as promedio_ponderado
                        FROM inscripciones i
                        JOIN denominaciones d ON i.denominacion_id = d.id
                        WHERE i.estado = 'finalizada' AND d.tipo = 'puntaje'
                        GROUP BY i.estudiante_id
                    ) promedios_estudiantes ON e.id = promedios_estudiantes.estudiante_id
                    LEFT JOIN inscripciones i ON e.id = i.estudiante_id AND i.estado = 'finalizada'
                    LEFT JOIN denominaciones d ON i.denominacion_id = d.id AND d.postgrado_id = p.id AND d.tipo = 'puntaje'
                    WHERE EXISTS (
                        SELECT 1 FROM estudiante_postgrado ep2 
                        JOIN estudiantes e2 ON ep2.estudiante_id = e2.id
                        JOIN (
                            SELECT i2.estudiante_id, 
                                   AVG(CASE 
                                       WHEN i2.nota_definitiva IS NOT NULL AND i2.nota_definitiva != 'Aprobado' AND i2.nota_definitiva != 'Reprobado'
                                       AND i2.nota_definitiva REGEXP '^[0-9]+(\.[0-9]+)?$'
                                       THEN CAST(i2.nota_definitiva AS DECIMAL(10,2)) 
                                   END) as ps,
                                   SUM(CASE 
                                       WHEN i2.nota_definitiva IS NOT NULL AND i2.nota_definitiva != 'Aprobado' AND i2.nota_definitiva != 'Reprobado'
                                       AND i2.nota_definitiva REGEXP '^[0-9]+(\.[0-9]+)?$'
                                       THEN CAST(i2.nota_definitiva AS DECIMAL(10,2)) * d2.credito
                                   END) / NULLIF(SUM(CASE 
                                       WHEN i2.nota_definitiva IS NOT NULL AND i2.nota_definitiva != 'Aprobado' AND i2.nota_definitiva != 'Reprobado'
                                       AND i2.nota_definitiva REGEXP '^[0-9]+(\.[0-9]+)?$'
                                       THEN d2.credito 
                                   END), 0) as pp
                            FROM inscripciones i2
                            JOIN denominaciones d2 ON i2.denominacion_id = d2.id
                            WHERE i2.estado = 'finalizada' AND d2.tipo = 'puntaje'
                            GROUP BY i2.estudiante_id
                        ) proms ON e2.id = proms.estudiante_id
                        WHERE ep2.postgrado_id = p.id 
                        AND ep2.estado = 'activo'
                        AND (proms.ps < 15 OR proms.pp < 15)
                    )
                    GROUP BY p.id, p.nombre
                    HAVING estudiantes_bajo_promedio > 0";

// Construir ORDER BY según los parámetros
switch ($orden) {
    case 'nombre':
        $query_postgrados .= " ORDER BY p.nombre $direccion";
        break;
    case 'estudiantes_activos':
        $query_postgrados .= " ORDER BY estudiantes_activos $direccion, p.nombre ASC";
        break;
    case 'promedio_simple':
        $query_postgrados .= " ORDER BY promedio_simple $direccion, p.nombre ASC";
        break;
    case 'promedio_ponderado':
        $query_postgrados .= " ORDER BY promedio_ponderado $direccion, p.nombre ASC";
        break;
    default:
        $query_postgrados .= " ORDER BY p.nombre ASC";
        break;
}

// Agregar paginación para postgrados
$query_postgrados .= " LIMIT $postgrados_por_pagina OFFSET $offset_postgrados";

$result_postgrados = $conn->query($query_postgrados);

// CONSULTAS PARA ESTUDIANTES CON BAJOS PROMEDIOS (CON PAGINACIÓN)
// Primero obtener el total
$query_total_bajos_promedios = "SELECT COUNT(DISTINCT e.id) as total
                               FROM estudiantes e
                               JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                               JOIN postgrados p ON ep.postgrado_id = p.id
                               LEFT JOIN inscripciones i ON e.id = i.estudiante_id
                               LEFT JOIN denominaciones d ON i.denominacion_id = d.id
                               WHERE ep.estado = 'activo'
                               AND i.estado = 'finalizada'
                               AND d.tipo = 'puntaje'
                               AND i.nota_definitiva IS NOT NULL
                               AND i.nota_definitiva != ''
                               AND i.nota_definitiva REGEXP '^[0-9]+(\.[0-9]+)?$'
                               GROUP BY e.id, p.nombre, ep.id
                               HAVING (
                                   COALESCE(ROUND(AVG(CASE 
                                       WHEN i.nota_definitiva IS NOT NULL AND i.nota_definitiva != 'Aprobado' AND i.nota_definitiva != 'Reprobado'
                                       AND i.nota_definitiva REGEXP '^[0-9]+(\.[0-9]+)?$'
                                       THEN CAST(i.nota_definitiva AS DECIMAL(10,2)) 
                                   END), 2), 0) < 15 
                                   OR 
                                   COALESCE(ROUND(SUM(CASE 
                                       WHEN i.nota_definitiva IS NOT NULL AND i.nota_definitiva != 'Aprobado' AND i.nota_definitiva != 'Reprobado'
                                       AND i.nota_definitiva REGEXP '^[0-9]+(\.[0-9]+)?$'
                                       THEN CAST(i.nota_definitiva AS DECIMAL(10,2)) * d.credito
                                   END) / NULLIF(SUM(CASE 
                                       WHEN i.nota_definitiva IS NOT NULL AND i.nota_definitiva != 'Aprobado' AND i.nota_definitiva != 'Reprobado'
                                       AND i.nota_definitiva REGEXP '^[0-9]+(\.[0-9]+)?$'
                                       THEN d.credito 
                                   END), 0), 2), 0) < 15
                               )";

$result_total_bajos_promedios = $conn->query($query_total_bajos_promedios);
$total_bajos_promedios = $result_total_bajos_promedios ? $result_total_bajos_promedios->num_rows : 0;
$total_paginas_bajos_promedios = ceil($total_bajos_promedios / $estudiantes_por_pagina);

// Ajustar página actual
if ($pagina_bajos_promedios > $total_paginas_bajos_promedios && $total_paginas_bajos_promedios > 0) $pagina_bajos_promedios = $total_paginas_bajos_promedios;

$offset_bajos_promedios = ($pagina_bajos_promedios - 1) * $estudiantes_por_pagina;

// Consulta para estudiantes con bajo promedio (con paginación)
$query_estudiantes_bajo_promedio = "SELECT 
                                   p.nombre as postgrado, 
                                   e.id, 
                                   e.cedula, 
                                   e.nombres, 
                                   e.apellidos,
                                   ep.id as inscripcion_id,
                                   COALESCE(ROUND(AVG(CASE 
                                       WHEN i.nota_definitiva IS NOT NULL AND i.nota_definitiva != 'Aprobado' AND i.nota_definitiva != 'Reprobado'
                                       AND i.nota_definitiva REGEXP '^[0-9]+(\.[0-9]+)?$'
                                       THEN CAST(i.nota_definitiva AS DECIMAL(10,2)) 
                                   END), 2), 0) as promedio_simple,
                                   COALESCE(ROUND(SUM(CASE 
                                       WHEN i.nota_definitiva IS NOT NULL AND i.nota_definitiva != 'Aprobado' AND i.nota_definitiva != 'Reprobado'
                                       AND i.nota_definitiva REGEXP '^[0-9]+(\.[0-9]+)?$'
                                       THEN CAST(i.nota_definitiva AS DECIMAL(10,2)) * d.credito
                                   END) / NULLIF(SUM(CASE 
                                       WHEN i.nota_definitiva IS NOT NULL AND i.nota_definitiva != 'Aprobado' AND i.nota_definitiva != 'Reprobado'
                                       AND i.nota_definitiva REGEXP '^[0-9]+(\.[0-9]+)?$'
                                       THEN d.credito 
                                   END), 0), 2), 0) as promedio_ponderado
                                   FROM estudiantes e
                                   JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                                   JOIN postgrados p ON ep.postgrado_id = p.id
                                   LEFT JOIN inscripciones i ON e.id = i.estudiante_id
                                   LEFT JOIN denominaciones d ON i.denominacion_id = d.id
                                   WHERE ep.estado = 'activo'
                                   AND i.estado = 'finalizada'
                                   AND d.tipo = 'puntaje'
                                   AND i.nota_definitiva IS NOT NULL
                                   AND i.nota_definitiva != ''
                                   AND i.nota_definitiva REGEXP '^[0-9]+(\.[0-9]+)?$'
                                   GROUP BY e.id, p.nombre, ep.id
                                   HAVING (promedio_simple < 15 OR promedio_ponderado < 15)
                                   ORDER BY p.nombre, promedio_simple ASC, promedio_ponderado ASC
                                   LIMIT $estudiantes_por_pagina OFFSET $offset_bajos_promedios";

$result_estudiantes_bajo_promedio = $conn->query($query_estudiantes_bajo_promedio);

// CONSULTAS PARA ESTUDIANTES CON ASIGNATURAS REPROBADAS (CON PAGINACIÓN)
// Primero obtener el total
$query_total_reprobados = "SELECT COUNT(DISTINCT CONCAT(e.id, '-', d.id)) as total
                          FROM estudiantes e
                          JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                          JOIN postgrados p ON ep.postgrado_id = p.id
                          JOIN inscripciones i ON e.id = i.estudiante_id
                          JOIN denominaciones d ON i.denominacion_id = d.id
                          WHERE ep.estado = 'activo'
                          AND i.estado = 'finalizada'
                          AND (
                              (d.tipo = 'puntaje' 
                               AND i.nota_definitiva IS NOT NULL 
                               AND i.nota_definitiva != '' 
                               AND i.nota_definitiva REGEXP '^[0-9]+(\.[0-9]+)?$'
                               AND CAST(i.nota_definitiva AS DECIMAL(10,2)) < 10) 
                              OR (d.tipo = 'Aprobado_Reprobado' AND i.nota_definitiva = 'Reprobado')
                          )";

$result_total_reprobados = $conn->query($query_total_reprobados);
$total_reprobados = $result_total_reprobados ? $result_total_reprobados->fetch_assoc()['total'] : 0;
$total_paginas_reprobados = ceil($total_reprobados / $estudiantes_por_pagina);

// Ajustar página actual
if ($pagina_reprobados > $total_paginas_reprobados && $total_paginas_reprobados > 0) $pagina_reprobados = $total_paginas_reprobados;

$offset_reprobados = ($pagina_reprobados - 1) * $estudiantes_por_pagina;

// Consulta para estudiantes con asignaturas reprobadas (con paginación) - CORREGIDA
$query_estudiantes_reprobados = "SELECT DISTINCT 
                                p.nombre as postgrado, 
                                e.id, 
                                e.cedula, 
                                e.nombres, 
                                e.apellidos,
                                ep.id as inscripcion_id,
                                d.nombre as asignatura,
                                i.nota_definitiva as nota,
                                d.tipo as tipo_evaluacion
                                FROM estudiantes e
                                JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                                JOIN postgrados p ON ep.postgrado_id = p.id
                                JOIN inscripciones i ON e.id = i.estudiante_id
                                JOIN denominaciones d ON i.denominacion_id = d.id
                                WHERE ep.estado = 'activo'
                                AND i.estado = 'finalizada'
                                AND (
                                    (d.tipo = 'puntaje' 
                                     AND i.nota_definitiva IS NOT NULL 
                                     AND i.nota_definitiva != '' 
                                     AND i.nota_definitiva REGEXP '^[0-9]+(\.[0-9]+)?$'
                                     AND CAST(i.nota_definitiva AS DECIMAL(10,2)) < 10) 
                                    OR (d.tipo = 'Aprobado_Reprobado' AND i.nota_definitiva = 'Reprobado')
                                )
                                ORDER BY p.nombre, e.apellidos, e.nombres
                                LIMIT $estudiantes_por_pagina OFFSET $offset_reprobados";

$result_estudiantes_reprobados = $conn->query($query_estudiantes_reprobados);

// CONSULTAS ORIGINALES PARA EL RESUMEN GENERAL
$total_postgrados = 0;
$query_total_postgrados = "SELECT COUNT(*) as total FROM postgrados";
$result_total_postgrados = $conn->query($query_total_postgrados);
if ($result_total_postgrados) {
    $total_postgrados = $result_total_postgrados->fetch_assoc()['total'];
}

$total_activos = 0;
$query_total_activos = "SELECT COUNT(*) as total FROM estudiante_postgrado WHERE estado = 'activo'";
$result_total_activos = $conn->query($query_total_activos);
if ($result_total_activos) {
    $total_activos = $result_total_activos->fetch_assoc()['total'];
}

$total_coordinadores = 0;
$query_total_coordinadores = "SELECT COUNT(*) as total FROM usuarios WHERE rol IN ('coordinador1', 'coordinador2')";
$result_total_coordinadores = $conn->query($query_total_coordinadores);
if ($result_total_coordinadores) {
    $total_coordinadores = $result_total_coordinadores->fetch_assoc()['total'];
}

$total_graduados = 0;
$query_total_graduados = "SELECT COUNT(*) as total FROM estudiante_postgrado WHERE estado = 'graduado'";
$result_total_graduados = $conn->query($query_total_graduados);
if ($result_total_graduados) {
    $total_graduados = $result_total_graduados->fetch_assoc()['total'];
}

$page_title = "Panel de Administrador";
include '../includes/header.php'; 
?>

<div class="container">
    <div class="text-center mb-4">
        <h1 class="display-4">Panel de Administración</h1>
    </div>

    <!-- Resumen General -->
    <div class="card mb-4">
        <div class="card-header bg-info text-white">
            <h5>Resumen General</h5>
        </div>
        <div class="card-body">
            <div class="row">
                <div class="col-md-3">
                    <div class="card text-white bg-primary mb-3 h-100">
                        <div class="card-body text-center">
                            <h5 class="card-title">Total Postgrados</h5>
                            <p class="card-text display-4"><?= $total_postgrados ?></p>
                        </div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="card text-white bg-success mb-3 h-100">
                        <div class="card-body text-center">
                            <h5 class="card-title">Estudiantes Activos</h5>
                            <p class="card-text display-4"><?= $total_activos ?></p>
                        </div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="card text-white bg-warning mb-3 h-100">
                        <div class="card-body text-center">
                            <h5 class="card-title">Coordinadores</h5>
                            <p class="card-text display-4"><?= $total_coordinadores ?></p>
                        </div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="card text-white bg-secondary mb-3 h-100">
                        <div class="card-body text-center">
                            <h5 class="card-title">Estudiantes Graduados</h5>
                            <p class="card-text display-4"><?= $total_graduados ?></p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Estadísticas de Postgrados con Bajo Promedio -->
    <div class="card mb-4">
        <div class="card-header bg-danger text-white d-flex justify-content-between align-items-center">
            <h5>Postgrados con Estudiantes de Bajo Promedio</h5>
            <span class="badge bg-light text-dark"><?= $total_postgrados_bajo_promedio ?> postgrados con promedios < 15</span>
        </div>
        <div class="card-body">
            <!-- Controles de ordenamiento -->
            <div class="row mb-3">
                <div class="col-md-6">
                    <form method="GET" class="row g-2">
                        <input type="hidden" name="pagina_postgrados" value="<?= $pagina_postgrados ?>">
                        <input type="hidden" name="pagina_bajos_promedios" value="<?= $pagina_bajos_promedios ?>">
                        <input type="hidden" name="pagina_reprobados" value="<?= $pagina_reprobados ?>">
                        <div class="col-md-6">
                            <select name="orden" class="form-select" onchange="this.form.submit()">
                                <option value="nombre" <?= $orden == 'nombre' ? 'selected' : '' ?>>Ordenar por Nombre</option>
                                <option value="estudiantes_activos" <?= $orden == 'estudiantes_activos' ? 'selected' : '' ?>>Ordenar por Estudiantes Activos</option>
                                <option value="promedio_simple" <?= $orden == 'promedio_simple' ? 'selected' : '' ?>>Ordenar por Promedio Simple</option>
                                <option value="promedio_ponderado" <?= $orden == 'promedio_ponderado' ? 'selected' : '' ?>>Ordenar por Promedio Ponderado</option>
                            </select>
                        </div>
                        <div class="col-md-6">
                            <select name="dir" class="form-select" onchange="this.form.submit()">
                                <option value="asc" <?= $direccion == 'asc' ? 'selected' : '' ?>>Ascendente</option>
                                <option value="desc" <?= $direccion == 'desc' ? 'selected' : '' ?>>Descendente</option>
                            </select>
                        </div>
                    </form>
                </div>
                <div class="col-md-6 text-end">
                    <?php if ($total_postgrados_bajo_promedio > 0): ?>
                    <span class="text-muted">
                        Mostrando <?= ($offset_postgrados + 1) ?> - <?= min($offset_postgrados + $postgrados_por_pagina, $total_postgrados_bajo_promedio) ?> de <?= $total_postgrados_bajo_promedio ?> postgrados
                    </span>
                    <?php endif; ?>
                </div>
            </div>

            <?php if ($result_postgrados && $result_postgrados->num_rows > 0): ?>
                <div class="table-responsive">
                    <table class="table table-striped table-hover">
                        <thead class="table-dark">
                            <tr>
                                <th>Postgrado</th>
                                <th class="text-center">Estudiantes Activos</th>
                                <th class="text-center">Estudiantes Bajo Promedio</th>
                                <th class="text-center">Prom. Simple</th>
                                <th class="text-center">Prom. Ponderado</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php while($postgrado = $result_postgrados->fetch_assoc()): ?>
                                <?php
                                $activos = (int)$postgrado['estudiantes_activos'];
                                $bajo_promedio = (int)$postgrado['estudiantes_bajo_promedio'];
                                $promedio_simple = (float)$postgrado['promedio_simple'];
                                $promedio_ponderado = (float)$postgrado['promedio_ponderado'];
                                
                                $promedio_simple_class = ($promedio_simple > 0 && $promedio_simple < 15) ? 'text-danger fw-bold' : '';
                                $promedio_ponderado_class = ($promedio_ponderado > 0 && $promedio_ponderado < 15) ? 'text-danger fw-bold' : '';
                                ?>
                                <tr>
                                    <td><?= htmlspecialchars($postgrado['nombre']) ?></td>
                                    <td class="text-center"><?= $activos ?></td>
                                    <td class="text-center text-danger fw-bold"><?= $bajo_promedio ?></td>
                                    <td class="text-center <?= $promedio_simple_class ?>">
                                        <?= $promedio_simple > 0 ? number_format($promedio_simple, 2) : 'N/A' ?>
                                    </td>
                                    <td class="text-center <?= $promedio_ponderado_class ?>">
                                        <?= $promedio_ponderado > 0 ? number_format($promedio_ponderado, 2) : 'N/A' ?>
                                    </td>
                                </tr>
                            <?php endwhile; ?>
                        </tbody>
                    </table>
                </div>
                
                <?php if ($total_paginas_postgrados > 1): ?>
                <nav aria-label="Paginación de postgrados">
                    <ul class="pagination justify-content-center">
                        <li class="page-item <?= $pagina_postgrados == 1 ? 'disabled' : '' ?>">
                            <a class="page-link" href="?pagina_postgrados=<?= $pagina_postgrados - 1 ?>&pagina_bajos_promedios=<?= $pagina_bajos_promedios ?>&pagina_reprobados=<?= $pagina_reprobados ?>&orden=<?= $orden ?>&dir=<?= $direccion ?>" aria-label="Anterior">
                                <span aria-hidden="true">&laquo;</span>
                            </a>
                        </li>
                        
                        <?php
                        $inicio = max(1, $pagina_postgrados - 2);
                        $fin = min($total_paginas_postgrados, $pagina_postgrados + 2);
                        
                        if ($inicio > 1) {
                            echo '<li class="page-item"><a class="page-link" href="?pagina_postgrados=1&pagina_bajos_promedios='.$pagina_bajos_promedios.'&pagina_reprobados='.$pagina_reprobados.'&orden='.$orden.'&dir='.$direccion.'">1</a></li>';
                            if ($inicio > 2) echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                        }
                        
                        for ($i = $inicio; $i <= $fin; $i++): ?>
                            <li class="page-item <?= $i == $pagina_postgrados ? 'active' : '' ?>">
                                <a class="page-link" href="?pagina_postgrados=<?= $i ?>&pagina_bajos_promedios=<?= $pagina_bajos_promedios ?>&pagina_reprobados=<?= $pagina_reprobados ?>&orden=<?= $orden ?>&dir=<?= $direccion ?>"><?= $i ?></a>
                            </li>
                        <?php endfor;
                        
                        if ($fin < $total_paginas_postgrados) {
                            if ($fin < $total_paginas_postgrados - 1) echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                            echo '<li class="page-item"><a class="page-link" href="?pagina_postgrados='.$total_paginas_postgrados.'&pagina_bajos_promedios='.$pagina_bajos_promedios.'&pagina_reprobados='.$pagina_reprobados.'&orden='.$orden.'&dir='.$direccion.'">'.$total_paginas_postgrados.'</a></li>';
                        }
                        ?>
                        
                        <li class="page-item <?= $pagina_postgrados == $total_paginas_postgrados ? 'disabled' : '' ?>">
                            <a class="page-link" href="?pagina_postgrados=<?= $pagina_postgrados + 1 ?>&pagina_bajos_promedios=<?= $pagina_bajos_promedios ?>&pagina_reprobados=<?= $pagina_reprobados ?>&orden=<?= $orden ?>&dir=<?= $direccion ?>" aria-label="Siguiente">
                                <span aria-hidden="true">&raquo;</span>
                            </a>
                        </li>
                    </ul>
                </nav>
                <?php endif; ?>
                
            <?php else: ?>
                <div class="alert alert-success">
                    <h6>¡Excelentes noticias!</h6>
                    <p class="mb-0">No se encontraron postgrados con estudiantes que tengan promedio simple o ponderado por debajo de 15 puntos.</p>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Tarjeta de Alertas Académicas -->
    <div class="card mb-4">
        <div class="card-header bg-warning text-dark">
            <h5>Alertas Académicas</h5>
        </div>
        <div class="card-body">
            <ul class="nav nav-tabs" id="alertasTab" role="tablist">
                <li class="nav-item" role="presentation">
                    <button class="nav-link active" id="bajos-promedios-tab" data-bs-toggle="tab" data-bs-target="#bajos-promedios" type="button" role="tab">
                        Bajos Promedios (<?= $total_bajos_promedios ?>)
                    </button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" id="reprobados-tab" data-bs-toggle="tab" data-bs-target="#reprobados" type="button" role="tab">
                        Asignaturas Reprobadas (<?= $total_reprobados ?>)
                    </button>
                </li>
            </ul>
            <div class="tab-content" id="alertasTabContent">
                <!-- Pestaña de Bajos Promedios -->
                <div class="tab-pane fade show active" id="bajos-promedios" role="tabpanel">
                    <?php if ($result_estudiantes_bajo_promedio && $result_estudiantes_bajo_promedio->num_rows > 0): ?>
                        <div class="table-responsive mt-3">
                            <table class="table table-striped table-hover">
                                <thead>
                                    <tr>
                                        <th>Postgrado</th>
                                        <th>Cédula</th>
                                        <th>Estudiante</th>
                                        <th class="text-center">Prom. Simple</th>
                                        <th class="text-center">Prom. Ponderado</th>
                                        <th class="text-center">Acciones</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php while($estudiante = $result_estudiantes_bajo_promedio->fetch_assoc()): ?>
                                        <?php
                                        $prom_simple = (float)$estudiante['promedio_simple'];
                                        $prom_ponderado = (float)$estudiante['promedio_ponderado'];
                                        $alerta_simple = $prom_simple < 15;
                                        $alerta_ponderado = $prom_ponderado < 15;
                                        ?>
                                        <tr class="<?= ($alerta_simple && $alerta_ponderado) ? 'table-danger' : 'table-warning' ?>">
                                            <td><?= htmlspecialchars($estudiante['postgrado']) ?></td>
                                            <td><?= htmlspecialchars($estudiante['cedula']) ?></td>
                                            <td><?= htmlspecialchars($estudiante['nombres'] . ' ' . $estudiante['apellidos']) ?></td>
                                            <td class="text-center <?= $alerta_simple ? 'text-danger fw-bold' : '' ?>">
                                                <?= number_format($prom_simple, 2) ?>
                                            </td>
                                            <td class="text-center <?= $alerta_ponderado ? 'text-danger fw-bold' : '' ?>">
                                                <?= number_format($prom_ponderado, 2) ?>
                                            </td>
                                            <td class="text-center">
                                                <a href="detalle_inscripcion.php?id=<?= $estudiante['inscripcion_id'] ?>" 
                                                   class="btn btn-sm btn-primary" 
                                                   title="Ver detalles de la inscripción">
                                                    <i class="bi bi-eye"></i> Detalles
                                                </a>
                                                <a href="../retiros/retiro_estudiantes.php?estudiante_id=<?= $estudiante['id'] ?>" 
                                                   class="btn btn-sm btn-danger" 
                                                   title="Retirar estudiante">
                                                    <i class="bi bi-person-x"></i> Retirar
                                                </a>
                                            </td>
                                        </tr>
                                    <?php endwhile; ?>
                                </tbody>
                            </table>
                        </div>
                        
                        <!-- Paginación para Bajos Promedios -->
                        <?php if ($total_paginas_bajos_promedios > 1): ?>
                        <nav aria-label="Paginación de bajos promedios">
                            <ul class="pagination justify-content-center mt-3">
                                <li class="page-item <?= $pagina_bajos_promedios == 1 ? 'disabled' : '' ?>">
                                    <a class="page-link" href="?pagina_postgrados=<?= $pagina_postgrados ?>&pagina_bajos_promedios=<?= $pagina_bajos_promedios - 1 ?>&pagina_reprobados=<?= $pagina_reprobados ?>&orden=<?= $orden ?>&dir=<?= $direccion ?>#bajos-promedios" aria-label="Anterior">
                                        <span aria-hidden="true">&laquo;</span>
                                    </a>
                                </li>
                                
                                <?php
                                $inicio_bp = max(1, $pagina_bajos_promedios - 2);
                                $fin_bp = min($total_paginas_bajos_promedios, $pagina_bajos_promedios + 2);
                                
                                if ($inicio_bp > 1) {
                                    echo '<li class="page-item"><a class="page-link" href="?pagina_postgrados='.$pagina_postgrados.'&pagina_bajos_promedios=1&pagina_reprobados='.$pagina_reprobados.'&orden='.$orden.'&dir='.$direccion.'#bajos-promedios">1</a></li>';
                                    if ($inicio_bp > 2) echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                                }
                                
                                for ($i = $inicio_bp; $i <= $fin_bp; $i++): ?>
                                    <li class="page-item <?= $i == $pagina_bajos_promedios ? 'active' : '' ?>">
                                        <a class="page-link" href="?pagina_postgrados=<?= $pagina_postgrados ?>&pagina_bajos_promedios=<?= $i ?>&pagina_reprobados=<?= $pagina_reprobados ?>&orden=<?= $orden ?>&dir=<?= $direccion ?>#bajos-promedios"><?= $i ?></a>
                                    </li>
                                <?php endfor;
                                
                                if ($fin_bp < $total_paginas_bajos_promedios) {
                                    if ($fin_bp < $total_paginas_bajos_promedios - 1) echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                                    echo '<li class="page-item"><a class="page-link" href="?pagina_postgrados='.$pagina_postgrados.'&pagina_bajos_promedios='.$total_paginas_bajos_promedios.'&pagina_reprobados='.$pagina_reprobados.'&orden='.$orden.'&dir='.$direccion.'#bajos-promedios">'.$total_paginas_bajos_promedios.'</a></li>';
                                }
                                ?>
                                
                                <li class="page-item <?= $pagina_bajos_promedios == $total_paginas_bajos_promedios ? 'disabled' : '' ?>">
                                    <a class="page-link" href="?pagina_postgrados=<?= $pagina_postgrados ?>&pagina_bajos_promedios=<?= $pagina_bajos_promedios + 1 ?>&pagina_reprobados=<?= $pagina_reprobados ?>&orden=<?= $orden ?>&dir=<?= $direccion ?>#bajos-promedios" aria-label="Siguiente">
                                        <span aria-hidden="true">&raquo;</span>
                                    </a>
                                </li>
                            </ul>
                        </nav>
                        <?php endif; ?>
                        
                        <div class="alert alert-info mt-2">
                            <small>Mostrando <?= ($offset_bajos_promedios + 1) ?> - <?= min($offset_bajos_promedios + $estudiantes_por_pagina, $total_bajos_promedios) ?> de <?= $total_bajos_promedios ?> estudiantes con promedio simple o ponderado menor a 15 puntos.</small>
                        </div>
                    <?php else: ?>
                        <div class="alert alert-success mt-3">
                            No se encontraron estudiantes con promedios bajos.
                        </div>
                    <?php endif; ?>
                </div>
                
                <!-- Pestaña de Asignaturas Reprobadas -->
                <div class="tab-pane fade" id="reprobados" role="tabpanel">
                    <?php if ($result_estudiantes_reprobados && $result_estudiantes_reprobados->num_rows > 0): ?>
                        <div class="table-responsive mt-3">
                            <table class="table table-striped table-hover">
                                <thead>
                                    <tr>
                                        <th>Postgrado</th>
                                        <th>Cédula</th>
                                        <th>Estudiante</th>
                                        <th>Asignatura Reprobada</th>
                                        <th class="text-center">Nota</th>
                                        <th class="text-center">Tipo</th>
                                        <th class="text-center">Acciones</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php while($estudiante = $result_estudiantes_reprobados->fetch_assoc()): ?>
                                        <tr class="table-danger">
                                            <td><?= htmlspecialchars($estudiante['postgrado']) ?></td>
                                            <td><?= htmlspecialchars($estudiante['cedula']) ?></td>
                                            <td><?= htmlspecialchars($estudiante['nombres'] . ' ' . $estudiante['apellidos']) ?></td>
                                            <td><?= htmlspecialchars($estudiante['asignatura']) ?></td>
                                            <td class="text-center fw-bold">
                                                <?php 
                                                // CORRECCIÓN: Manejar correctamente ambos tipos de notas
                                                if ($estudiante['tipo_evaluacion'] == 'puntaje') {
                                                    // Para notas numéricas, verificar que sea un número válido
                                                    $nota_numerica = $estudiante['nota'];
                                                    if (is_numeric($nota_numerica)) {
                                                        echo number_format(floatval($nota_numerica), 2);
                                                    } else {
                                                        echo '<span class="text-warning">Inválida</span>';
                                                    }
                                                } else {
                                                    // Para notas cualitativas, mostrar el texto
                                                    echo htmlspecialchars($estudiante['nota']);
                                                }
                                                ?>
                                            </td>
                                            <td class="text-center">
                                                <span class="badge <?= $estudiante['tipo_evaluacion'] == 'puntaje' ? 'bg-primary' : 'bg-info' ?>">
                                                    <?= $estudiante['tipo_evaluacion'] == 'puntaje' ? 'Numérica' : 'Cualitativa' ?>
                                                </span>
                                            </td>
                                            <td class="text-center">
                                                <a href="detalle_inscripcion.php?id=<?= $estudiante['inscripcion_id'] ?>" 
                                                   class="btn btn-sm btn-primary" 
                                                   title="Ver detalles de la inscripción">
                                                    <i class="bi bi-eye"></i> Detalles
                                                </a>
                                                <a href="../retiros/retiro_estudiantes.php?estudiante_id=<?= $estudiante['id'] ?>" 
                                                   class="btn btn-sm btn-danger" 
                                                   title="Retirar estudiante">
                                                    <i class="bi bi-person-x"></i> Retirar
                                                </a>
                                            </td>
                                        </tr>
                                    <?php endwhile; ?>
                                </tbody>
                            </table>
                        </div>
                        
                        <!-- Paginación para Asignaturas Reprobadas -->
                        <?php if ($total_paginas_reprobados > 1): ?>
                        <nav aria-label="Paginación de asignaturas reprobadas">
                            <ul class="pagination justify-content-center mt-3">
                                <li class="page-item <?= $pagina_reprobados == 1 ? 'disabled' : '' ?>">
                                    <a class="page-link" href="?pagina_postgrados=<?= $pagina_postgrados ?>&pagina_bajos_promedios=<?= $pagina_bajos_promedios ?>&pagina_reprobados=<?= $pagina_reprobados - 1 ?>&orden=<?= $orden ?>&dir=<?= $direccion ?>#reprobados" aria-label="Anterior">
                                        <span aria-hidden="true">&laquo;</span>
                                    </a>
                                </li>
                                
                                <?php
                                $inicio_r = max(1, $pagina_reprobados - 2);
                                $fin_r = min($total_paginas_reprobados, $pagina_reprobados + 2);
                                
                                if ($inicio_r > 1) {
                                    echo '<li class="page-item"><a class="page-link" href="?pagina_postgrados='.$pagina_postgrados.'&pagina_bajos_promedios='.$pagina_bajos_promedios.'&pagina_reprobados=1&orden='.$orden.'&dir='.$direccion.'#reprobados">1</a></li>';
                                    if ($inicio_r > 2) echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                                }
                                
                                for ($i = $inicio_r; $i <= $fin_r; $i++): ?>
                                    <li class="page-item <?= $i == $pagina_reprobados ? 'active' : '' ?>">
                                        <a class="page-link" href="?pagina_postgrados=<?= $pagina_postgrados ?>&pagina_bajos_promedios=<?= $pagina_bajos_promedios ?>&pagina_reprobados=<?= $i ?>&orden=<?= $orden ?>&dir=<?= $direccion ?>#reprobados"><?= $i ?></a>
                                    </li>
                                <?php endfor;
                                
                                if ($fin_r < $total_paginas_reprobados) {
                                    if ($fin_r < $total_paginas_reprobados - 1) echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                                    echo '<li class="page-item"><a class="page-link" href="?pagina_postgrados='.$pagina_postgrados.'&pagina_bajos_promedios='.$pagina_bajos_promedios.'&pagina_reprobados='.$total_paginas_reprobados.'&orden='.$orden.'&dir='.$direccion.'#reprobados">'.$total_paginas_reprobados.'</a></li>';
                                }
                                ?>
                                
                                <li class="page-item <?= $pagina_reprobados == $total_paginas_reprobados ? 'disabled' : '' ?>">
                                    <a class="page-link" href="?pagina_postgrados=<?= $pagina_postgrados ?>&pagina_bajos_promedios=<?= $pagina_bajos_promedios ?>&pagina_reprobados=<?= $pagina_reprobados + 1 ?>&orden=<?= $orden ?>&dir=<?= $direccion ?>#reprobados" aria-label="Siguiente">
                                        <span aria-hidden="true">&raquo;</span>
                                    </a>
                                </li>
                            </ul>
                        </nav>
                        <?php endif; ?>
                        
                        <div class="alert alert-info mt-2">
                            <small>Mostrando <?= ($offset_reprobados + 1) ?> - <?= min($offset_reprobados + $estudiantes_por_pagina, $total_reprobados) ?> de <?= $total_reprobados ?> estudiantes con asignaturas reprobadas.</small>
                        </div>
                    <?php else: ?>
                        <div class="alert alert-success mt-3">
                            No se encontraron estudiantes con asignaturas reprobadas.
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
    .fw-bold {
        font-weight: bold;
    }
    .table th {
        background-color: #2c3e50;
        color: white;
    }
    .pagination .page-item.active .page-link {
        background-color: #2c3e50;
        border-color: #2c3e50;
        color: white !important;
    }
    .pagination .page-link {
        color: #2c3e50;
    }
    .pagination .page-link:hover {
        color: #1a252f;
        background-color: #e9ecef;
    }
    .pagination .page-item.disabled .page-link {
        color: #6c757d;
    }
    .btn-sm {
        margin: 1px;
    }
    .nav-tabs .nav-link.active {
        background-color: #ffc107;
        border-color: #ffc107;
        color: #212529;
        font-weight: bold;
    }
    /* Estilo para que todas las cartillas tengan el mismo tamaño */
    .card.h-100 {
        height: 100% !important;
        min-height: 180px;
    }
    .card-body.text-center {
        display: flex;
        flex-direction: column;
        justify-content: center;
        align-items: center;
    }
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    var tabTriggers = document.querySelectorAll('button[data-bs-toggle="tab"]');
    tabTriggers.forEach(function(tabTriggerEl) {
        tabTriggerEl.addEventListener('click', function (e) {
            e.preventDefault();
            var tab = new bootstrap.Tab(this);
            tab.show();
        });
    });
    
    // Mantener la pestaña activa después de la paginación
    const urlParams = new URLSearchParams(window.location.search);
    const hash = window.location.hash;
    if (hash) {
        const activeTab = document.querySelector(hash + '-tab');
        if (activeTab) {
            bootstrap.Tab.getInstance(activeTab).show();
        }
    }
});
</script>

<?php
include '../includes/footer.php';
$conn->close();