<?php
ini_set('display_errors', 1);
error_reporting(E_ALL);

// DB Connection FIRST
require_once __DIR__ . '/../includes/db.php';

// Centralized authentication and session check
$allowed_roles = ['administrador'];
require_once __DIR__ . '/../includes/auth_check.php';

// Verificar conexión
if (!$conn || $conn->connect_error) {
    die("Error de conexión a la base de datos. Contacte al administrador.");
}

$success_message = '';
$error_message = '';

// Manejar el envío del formulario
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    
    // Manejar subida de logo
    if (isset($_FILES['site_logo']) && $_FILES['site_logo']['error'] == 0) {
        $target_dir = __DIR__ . "/../assets/img/";
        if (!file_exists($target_dir)) {
            mkdir($target_dir, 0777, true);
        }
        $file_extension = strtolower(pathinfo($_FILES["site_logo"]["name"], PATHINFO_EXTENSION));
        $new_filename = "logo." . $file_extension;
        $target_file = $target_dir . $new_filename;
        $allowed_types = ['jpg', 'jpeg', 'png', 'gif', 'svg'];

        if (in_array($file_extension, $allowed_types)) {
            if (move_uploaded_file($_FILES["site_logo"]["tmp_name"], $target_file)) {
                // Guardar la ruta relativa en la base de datos
                $logo_path = "assets/img/" . $new_filename;
                $sql_logo = "INSERT INTO app_settings (setting_key, setting_value) VALUES ('site_logo', ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)";
                $stmt_logo = $conn->prepare($sql_logo);
                $stmt_logo->bind_param("s", $logo_path);
                $stmt_logo->execute();
                $stmt_logo->close();
                $success_message = "Logo actualizado correctamente. ";
            } else {
                $error_message = "Error al mover el archivo del logo. ";
            }
        } else {
            $error_message = "Tipo de archivo no permitido para el logo. Use JPG, PNG, GIF o SVG. ";
        }
    } elseif (isset($_FILES['site_logo']) && $_FILES['site_logo']['error'] != UPLOAD_ERR_NO_FILE) {
        $error_message = "Error al subir el logo: " . $_FILES['site_logo']['error'] . ". ";
    }


    try {
        // Guardar estado de registro
        if (isset($_POST['registration_status'])) {
            $status = $_POST['registration_status'] === '1' ? '1' : '0';
            $sql = "INSERT INTO app_settings (setting_key, setting_value) VALUES ('student_registration_enabled', ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("s", $status);
            $stmt->execute();
            $stmt->close();
        }

        // Guardar modo de registro verificado
        if (isset($_POST['verified_only_status'])) {
            $verified_status = $_POST['verified_only_status'] === '1' ? '1' : '0';
            $sql_verified = "INSERT INTO app_settings (setting_key, setting_value) VALUES ('verified_students_only', ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)";
            $stmt_verified = $conn->prepare($sql_verified);
            $stmt_verified->bind_param("s", $verified_status);
            $stmt_verified->execute();
            $stmt_verified->close();
        }

        // Guardar estado de recuperación de contraseña
        if (isset($_POST['password_reset_status'])) {
            $reset_status = $_POST['password_reset_status'] === '1' ? '1' : '0';
            $sql_reset = "INSERT INTO app_settings (setting_key, setting_value) VALUES ('password_reset_enabled', ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)";
            $stmt_reset = $conn->prepare($sql_reset);
            $stmt_reset->bind_param("s", $reset_status);
            $stmt_reset->execute();
            $stmt_reset->close();
        }

        // Guardar configuración de sesión
        if (isset($_POST['session_duration'])) {
            $session_duration = intval($_POST['session_duration']);
            $sql_session = "INSERT INTO app_settings (setting_key, setting_value) VALUES ('session_duration', ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)";
            $stmt_session = $conn->prepare($sql_session);
            $stmt_session->bind_param("i", $session_duration);
            $stmt_session->execute();
            $stmt_session->close();
        }

        // Guardar configuración de sesión nunca expira
        if (isset($_POST['session_never_expire'])) {
            $never_expire = $_POST['session_never_expire'] === '1' ? '1' : '0';
            $sql_never = "INSERT INTO app_settings (setting_key, setting_value) VALUES ('session_never_expire', ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)";
            $stmt_never = $conn->prepare($sql_never);
            $stmt_never->bind_param("s", $never_expire);
            $stmt_never->execute();
            $stmt_never->close();
        }

        // Guardar texto del footer
        if (isset($_POST['footer_text'])) {
            $footer_text = trim($_POST['footer_text']);
            $sql_footer = "INSERT INTO app_settings (setting_key, setting_value) VALUES ('footer_text', ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)";
            $stmt_footer = $conn->prepare($sql_footer);
            $stmt_footer->bind_param("s", $footer_text);
            $stmt_footer->execute();
            $stmt_footer->close();
        }
        
        $success_message .= "La configuración se ha actualizado correctamente.";

        // ACTUALIZAR LA CONFIGURACIÓN EN LA SESIÓN ACTUAL - CORRECCIÓN CRÍTICA
        try {
            $sql_reload = "SELECT setting_key, setting_value FROM app_settings 
                           WHERE setting_key IN ('session_duration', 'session_never_expire')";
            $result_reload = $conn->query($sql_reload);
            
            $app_settings = [
                'session_duration' => 3600,
                'session_never_expire' => '0'
            ];
            
            if ($result_reload && $result_reload->num_rows > 0) {
                while ($row = $result_reload->fetch_assoc()) {
                    $app_settings[$row['setting_key']] = $row['setting_value'];
                }
            }
            
            $_SESSION['app_settings'] = $app_settings;
            
            // DEBUG: Verificar que se actualizó correctamente
            error_log("Configuración de sesión actualizada: " . print_r($_SESSION['app_settings'], true));
            
        } catch (Exception $e) {
            // No interrumpir el flujo si hay error al actualizar la sesión
            error_log("Error al actualizar configuración en sesión: " . $e->getMessage());
        }

    } catch (Exception $e) {
        $error_message .= "Error al actualizar la configuración: " . $e->getMessage();
    }
}

// Obtener el estado actual de las configuraciones
$sql_select = "SELECT setting_key, setting_value FROM app_settings WHERE setting_key IN ('student_registration_enabled', 'verified_students_only', 'password_reset_enabled', 'session_duration', 'session_never_expire', 'site_logo', 'footer_text')";
$result = $conn->query($sql_select);
$settings = [];
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $settings[$row['setting_key']] = $row['setting_value'];
    }
}

$current_status = $settings['student_registration_enabled'] ?? '0';
$verified_only_status = $settings['verified_students_only'] ?? '0';
$password_reset_status = $settings['password_reset_enabled'] ?? '0';
$session_duration = $settings['session_duration'] ?? '3600';
$session_never_expire = $settings['session_never_expire'] ?? '0';
$site_logo = $settings['site_logo'] ?? '';
$footer_text = $settings['footer_text'] ?? '© ' . date('Y') . ' Sistema de Postgrado - Facultad de Medicina UCV. Todos los derechos reservados.';

// Convertir segundos a formato legible
function formatSeconds($seconds) {
    $hours = floor($seconds / 3600);
    $minutes = floor(($seconds % 3600) / 60);
    $secs = $seconds % 60;
    
    if ($hours > 0) {
        return sprintf("%d h %02d min %02d seg", $hours, $minutes, $secs);
    } elseif ($minutes > 0) {
        return sprintf("%d min %02d seg", $minutes, $secs);
    } else {
        return sprintf("%d seg", $secs);
    }
}

$current_duration_formatted = formatSeconds($session_duration);

// NO cerrar la conexión aquí - se cerrará automáticamente al final del script

$page_title = "Gestión de Configuración del Sistema";
include __DIR__ . '/../includes/header.php';
?>

<div class="container">
    <div class="mb-4">
        <a href="index.php" class="btn btn-secondary">Volver al Panel</a>
    </div>

    <h2>Gestión de Configuración del Sistema</h2>
    <p>Controla las funcionalidades disponibles para los usuarios.</p>

    <?php if ($success_message): ?>
        <div class="alert alert-success"><?= htmlspecialchars($success_message) ?></div>
    <?php endif; ?>
    <?php if ($error_message): ?>
        <div class="alert alert-danger"><?= htmlspecialchars($error_message) ?></div>
    <?php endif; ?>

    <!-- Información de Debug (solo para administradores) -->
    <?php if ($_SESSION['rol'] === 'administrador'): ?>
    <div class="alert alert-info mb-4">
        <h6><i class="fas fa-info-circle me-2"></i>Información de Configuración Actual</h6>
        <p class="mb-1"><strong>Sesión configurada para:</strong> 
            <?= ($session_never_expire === '1') ? 'NUNCA EXPIRAR' : "Expirar después de $current_duration_formatted" ?>
        </p>
        <p class="mb-0"><small>Última actividad: <?= isset($_SESSION['LAST_ACTIVITY']) ? date('Y-m-d H:i:s', $_SESSION['LAST_ACTIVITY']) : 'No registrada' ?></small></p>
    </div>
    <?php endif; ?>

    <form method="POST" action="gestion_registro.php" enctype="multipart/form-data">
        <!-- Configuración de Apariencia -->
        <div class="card mb-4">
            <div class="card-header bg-secondary text-white">
                <h5>1. <i class="fas fa-paint-brush me-2"></i>Apariencia del Sitio</h5>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <label for="site_logo" class="form-label">Logo del Sitio</label>
                        <input class="form-control" type="file" id="site_logo" name="site_logo" accept="image/*">
                        <div class="form-text">Sube un logo para el encabezado (JPG, PNG, SVG, GIF). Se redimensionará automáticamente.</div>
                        <?php if ($site_logo): ?>
                            <div class="mt-2">
                                <p><strong>Logo actual:</strong></p>
                                <img src="../<?= htmlspecialchars($site_logo) ?>" alt="Logo actual" style="max-height: 50px; background-color: #f0f0f0; padding: 5px; border-radius: 4px;">
                            </div>
                        <?php endif; ?>
                    </div>
                    <div class="col-md-6">
                        <label for="footer_text" class="form-label">Texto del Pie de Página</label>
                        <textarea class="form-control" id="footer_text" name="footer_text" rows="3"><?= htmlspecialchars($footer_text) ?></textarea>
                        <div class="form-text">Este texto aparecerá en el pie de página de todo el sitio.</div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Configuración de Registro -->
        <div class="card mb-4">
            <div class="card-header">
                <h5>2. Estado General del Registro</h5>
            </div>
            <div class="card-body">
                <p>Habilita o deshabilita completamente el acceso a la página de registro de estudiantes.</p>
                <div class="form-check form-switch">
                    <input class="form-check-input" type="radio" name="registration_status" id="status_enabled" value="1" <?= $current_status === '1' ? 'checked' : '' ?>>
                    <label class="form-check-label" for="status_enabled">
                        <strong>Habilitado:</strong> Los estudiantes pueden acceder a la página de registro.
                    </label>
                </div>
                <div class="form-check form-switch mt-2">
                    <input class="form-check-input" type="radio" name="registration_status" id="status_disabled" value="0" <?= $current_status === '0' ? 'checked' : '' ?>>
                    <label class="form-check-label" for="status_disabled">
                        <strong>Deshabilitado:</strong> La página de registro no será accesible.
                    </label>
                </div>
            </div>
        </div>

        <!-- Configuración de Registro Verificado -->
        <div class="card mb-4">
            <div class="card-header">
                <h5>3. Comportamiento para Estudiantes No Encontrados</h5>
            </div>
            <div class="card-body">
                <p>Esta opción define qué sucede si un estudiante intenta registrarse con una cédula que <strong>no</strong> está en la base de datos.</p>
                <div class="form-check">
                    <input class="form-check-input" type="radio" name="verified_only_status" id="verified_disabled" value="0" <?= $verified_only_status === '0' ? 'checked' : '' ?>>
                    <label class="form-check-label" for="verified_disabled">
                        <strong>Permitir Registro (Modo Flexible):</strong> Si el estudiante no es encontrado, se le permitirá llenar el formulario para crear una cuenta nueva.
                    </label>
                </div>
                <div class="form-check mt-2">
                    <input class="form-check-input" type="radio" name="verified_only_status" id="verified_enabled" value="1" <?= $verified_only_status === '1' ? 'checked' : '' ?>>
                    <label class="form-check-label" for="verified_enabled">
                        <strong>Bloquear Registro (Modo Estricto):</strong> Si el estudiante no es encontrado, se le mostrará un error y no podrá crear una cuenta.
                    </label>
                </div>
            </div>
        </div>

        <!-- Recuperación de Contraseña -->
        <div class="card mb-4">
            <div class="card-header">
                <h5>4. Recuperación de Contraseña</h5>
            </div>
            <div class="card-body">
                <p>Habilita o deshabilita la funcionalidad "Olvidé mi contraseña" en la página de login.</p>
                <div class="form-check form-switch">
                    <input class="form-check-input" type="radio" name="password_reset_status" id="reset_enabled" value="1" <?= $password_reset_status === '1' ? 'checked' : '' ?>>
                    <label class="form-check-label" for="reset_enabled">
                        <strong>Habilitado:</strong> Los usuarios podrán solicitar restablecer su contraseña por correo electrónico.
                    </label>
                </div>
                <div class="form-check form-switch mt-2">
                    <input class="form-check-input" type="radio" name="password_reset_status" id="reset_disabled" value="0" <?= $password_reset_status === '0' ? 'checked' : '' ?>>
                    <label class="form-check-label" for="reset_disabled">
                        <strong>Deshabilitado:</strong> La opción "Olvidé mi contraseña" no estará disponible.
                    </label>
                </div>
                <div class="alert alert-info mt-3">
                    <small>
                        <strong>Nota:</strong> Cuando está habilitado, los usuarios recibirán un enlace por correo para restablecer su contraseña. 
                        El enlace expirará después de 1 hora por seguridad.
                    </small>
                </div>
            </div>
        </div>

        <!-- Duración de Sesión - MEJORADA -->
        <div class="card mb-4">
            <div class="card-header bg-primary text-white">
                <h5>5. Configuración de Duración de Sesión</h5>
            </div>
            <div class="card-body">
                <p>Controla el comportamiento del sistema de sesiones mejorado.</p>
                
                <div class="alert alert-info mb-3">
                    <h6><i class="fas fa-info-circle me-2"></i>Sistema de Sesiones Mejorado</h6>
                    <p class="mb-1">El sistema ahora incluye:</p>
                    <ul class="mb-0">
                        <li><strong>Ventana emergente de advertencia</strong> 5 minutos antes de expirar</li>
                        <li><strong>Cuenta regresiva visual</strong> en tiempo real</li>
                        <li><strong>Detección inteligente</strong> de actividad real (no movimiento de mouse)</li>
                        <li><strong>Estado visual</strong> en el header que muestra tiempo restante</li>
                    </ul>
                </div>
                
                <div class="mb-3">
                    <label for="session_duration" class="form-label">Duración de la sesión (en segundos):</label>
                    <input type="number" class="form-control" id="session_duration" name="session_duration" 
                           value="<?= htmlspecialchars($session_duration) ?>" min="300" step="300" required>
                    <div class="form-text">
                        <strong>Duración actual:</strong> <span id="current_duration_display"><?= $current_duration_formatted ?></span><br>
                        <strong>Ejemplos recomendados:</strong><br>
                        • 900 = 15 minutos (alta seguridad)<br>
                        • 1800 = 30 minutos (seguridad media)<br>
                        • 3600 = 1 hora (seguridad estándar)<br>
                        • 7200 = 2 horas (baja seguridad)<br>
                        • 28800 = 8 horas (muy baja seguridad)
                    </div>
                </div>

                <div class="form-check form-switch mb-3">
                    <input class="form-check-input" type="radio" name="session_never_expire" id="session_never_expire_enabled" value="1" <?= $session_never_expire === '1' ? 'checked' : '' ?>>
                    <label class="form-check-label" for="session_never_expire_enabled">
                        <strong>Sesión que nunca expira:</strong> La sesión del usuario no expirará por inactividad.
                    </label>
                </div>
                <div class="form-check form-switch">
                    <input class="form-check-input" type="radio" name="session_never_expire" id="session_never_expire_disabled" value="0" <?= $session_never_expire === '0' ? 'checked' : '' ?>>
                    <label class="form-check-label" for="session_never_expire_disabled">
                        <strong>Usar duración configurada:</strong> La sesión expirará después del tiempo especificado arriba.
                    </label>
                </div>

                <div class="alert alert-warning mt-3">
                    <h6><i class="fas fa-exclamation-triangle me-2"></i>Comportamiento del Sistema</h6>
                    <ul class="mb-0">
                        <li><strong>Advertencia:</strong> Aparece 5 minutos antes de la expiración</li>
                        <li><strong>Actividad detectada:</strong> Clics, teclas, formularios, cambios en inputs</li>
                        <li><strong>Actividad ignorada:</strong> Movimiento de mouse, scroll, eventos automáticos</li>
                        <li><strong>Renovación:</strong> El usuario decide si extender o cerrar sesión</li>
                    </ul>
                </div>
            </div>
        </div>

        <button type="submit" class="btn btn-primary mt-3">
            <i class="fas fa-save me-2"></i>Guardar Cambios
        </button>
        <a href="index.php" class="btn btn-secondary mt-3">
            <i class="fas fa-times me-2"></i>Cancelar
        </a>
    </form>
</div>

<script>
// Actualizar la visualización de la duración en tiempo real
document.getElementById('session_duration').addEventListener('input', function() {
    const seconds = parseInt(this.value);
    if (!isNaN(seconds)) {
        const hours = Math.floor(seconds / 3600);
        const minutes = Math.floor((seconds % 3600) / 60);
        const secs = seconds % 60;
        
        let formatted = '';
        if (hours > 0) {
            formatted = `${hours} h ${minutes.toString().padStart(2, '0')} min ${secs.toString().padStart(2, '0')} seg`;
        } else if (minutes > 0) {
            formatted = `${minutes} min ${secs.toString().padStart(2, '0')} seg`;
        } else {
            formatted = `${secs} seg`;
        }
        
        // Actualizar el texto
        const durationDisplay = document.getElementById('current_duration_display');
        if (durationDisplay) {
            durationDisplay.textContent = formatted;
        }
    }
});

// Mostrar/ocultar campo de duración basado en la selección de "nunca expira"
document.querySelectorAll('input[name="session_never_expire"]').forEach(radio => {
    radio.addEventListener('change', function() {
        const durationField = document.getElementById('session_duration');
        const durationContainer = durationField.parentElement;
        if (this.value === '1') {
            durationField.disabled = true;
            durationContainer.style.opacity = '0.6';
            durationContainer.style.pointerEvents = 'none';
        } else {
            durationField.disabled = false;
            durationContainer.style.opacity = '1';
            durationContainer.style.pointerEvents = 'auto';
        }
    });
});

// Ejecutar al cargar la página para establecer estado inicial
document.addEventListener('DOMContentLoaded', function() {
    const neverExpireEnabled = document.getElementById('session_never_expire_enabled').checked;
    const durationField = document.getElementById('session_duration');
    const durationContainer = durationField.parentElement;
    if (neverExpireEnabled) {
        durationField.disabled = true;
        durationContainer.style.opacity = '0.6';
        durationContainer.style.pointerEvents = 'none';
    }
});
</script>

<?php
include __DIR__ . '/../includes/footer.php';
?>