<?php
// Configuración detallada de errores
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

session_start();

if (!isset($_SESSION['username']) || $_SESSION['rol'] != 'administrador') {
    header("Location: ../login.php");
    exit();
}

include '../includes/db.php';
if (!$conn) {
    die("Error de conexión: " . $conn->connect_error);
}
$conn->set_charset("utf8mb4");

// Obtener listados para filtros
$sql_postgrados = "SELECT * FROM postgrados ORDER BY nombre";
$result_postgrados = $conn->query($sql_postgrados);

$sql_promociones = "SELECT * FROM promociones ORDER BY nombre";
$result_promociones = $conn->query($sql_promociones);

$sql_escolaridades = "SELECT * FROM escolaridades ORDER BY nombre";
$result_escolaridades = $conn->query($sql_escolaridades);

$sql_anios_lectivos = "SELECT * FROM anios_lectivos ORDER BY nombre";
$result_anios_lectivos = $conn->query($sql_anios_lectivos);

// Procesar el formulario para graduar estudiante
if (isset($_POST['graduar_estudiante'])) {
    $estudiante_id = intval($_POST['estudiante_id']);
    $postgrado_id = intval($_POST['postgrado_id']);
    $promocion_id = intval($_POST['promocion_id']);
    $fecha_graduacion = $_POST['fecha_graduacion'];
    
    // Validar datos
    if (empty($fecha_graduacion)) {
        $_SESSION['error'] = "La fecha de graduación es requerida";
    } elseif (!DateTime::createFromFormat('Y-m-d', $fecha_graduacion)) {
        $_SESSION['error'] = "La fecha de graduación no es válida";
    } else {
        // Actualizar la relación estudiante-postgrado
        $sql = "UPDATE estudiante_postgrado 
                SET estado = 'graduado', 
                    fecha_graduacion = ?,
                    promocion_id = ?
                WHERE estudiante_id = ? 
                AND postgrado_id = ?";
        $stmt = $conn->prepare($sql);
        if ($stmt) {
            $stmt->bind_param("siii", $fecha_graduacion, $promocion_id, $estudiante_id, $postgrado_id);
            
            if ($stmt->execute()) {
                $_SESSION['success'] = "Estudiante graduado correctamente";
            } else {
                $_SESSION['error'] = "Error al graduar estudiante: " . $stmt->error;
            }
            $stmt->close();
        } else {
            $_SESSION['error'] = "Error en la consulta: " . $conn->error;
        }
    }
    header("Location: gestion_graduacion.php");
    exit();
}

// Obtener parámetros de filtro
$postgrado_id = isset($_GET['postgrado_id']) ? intval($_GET['postgrado_id']) : 0;
$escolaridad_id = isset($_GET['escolaridad_id']) ? intval($_GET['escolaridad_id']) : 0;
$anio_lectivo_id = isset($_GET['anio_lectivo_id']) ? intval($_GET['anio_lectivo_id']) : 0;
$fecha_graduacion_filtro = isset($_GET['fecha_graduacion']) ? $_GET['fecha_graduacion'] : '';
$cedula_filtro = isset($_GET['cedula']) ? trim($_GET['cedula']) : '';

// Obtener estudiantes que han completado todas las denominaciones (sin graduar) - MODIFICADO
$estudiantes_completados = [];
$sql_estudiantes = "SELECT e.id, e.nombres, e.apellidos, e.cedula, p.nombre AS postgrado_nombre, p.id AS postgrado_id
                   FROM estudiantes e
                   JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                   JOIN postgrados p ON ep.postgrado_id = p.id
                   WHERE ep.estado = 'activo'
                   AND NOT EXISTS (
                       SELECT d.id
                       FROM denominaciones d
                       WHERE d.postgrado_id = p.id
                       AND d.id NOT IN (
                           SELECT i.denominacion_id
                           FROM inscripciones i
                           WHERE i.estudiante_id = e.id
                           AND i.nota_definitiva IS NOT NULL
                       )
                   )";

$conditions_completados = [];
$params_completados = [];
$types_completados = "";

if ($postgrado_id > 0) {
    $conditions_completados[] = "p.id = ?";
    $params_completados[] = $postgrado_id;
    $types_completados .= "i";
}

if (!empty($cedula_filtro)) {
    $conditions_completados[] = "e.cedula LIKE ?";
    $params_completados[] = "%$cedula_filtro%";
    $types_completados .= "s";
}

if (!empty($conditions_completados)) {
    $sql_estudiantes .= " AND " . implode(" AND ", $conditions_completados);
}

$sql_estudiantes .= " ORDER BY p.nombre, e.apellidos, e.nombres";

$stmt = $conn->prepare($sql_estudiantes);
if ($stmt) {
    if (!empty($params_completados)) {
        $stmt->bind_param($types_completados, ...$params_completados);
    }
    if ($stmt->execute()) {
        $estudiantes_completados = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    } else {
        $_SESSION['error'] = "Error al obtener estudiantes completados: " . $stmt->error;
    }
    $stmt->close();
} else {
    $_SESSION['error'] = "Error en la consulta de estudiantes completados: " . $conn->error;
}

// Obtener estudiantes ya graduados (con fecha de grado)
$sql_graduados = "SELECT e.id, e.nombres, e.apellidos, e.cedula, 
                 ep.fecha_graduacion, p.nombre AS postgrado_nombre, 
                 pr.nombre AS promocion_nombre,
                 ep.postgrado_id, ep.promocion_id
                 FROM estudiantes e
                 JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                 JOIN postgrados p ON ep.postgrado_id = p.id
                 LEFT JOIN promociones pr ON ep.promocion_id = pr.id
                 WHERE ep.estado = 'graduado'";
                 
$conditions = [];
$params = [];
$types = "";

if ($postgrado_id > 0) {
    $conditions[] = "ep.postgrado_id = ?";
    $params[] = $postgrado_id;
    $types .= "i";
}

if ($escolaridad_id > 0) {
    $sql_graduados .= " JOIN inscripciones i ON e.id = i.estudiante_id";
    $conditions[] = "i.escolaridad_id = ?";
    $params[] = $escolaridad_id;
    $types .= "i";
}

if ($anio_lectivo_id > 0) {
    if (strpos($sql_graduados, 'JOIN inscripciones i') === false) {
        $sql_graduados .= " JOIN inscripciones i ON e.id = i.estudiante_id";
    }
    $conditions[] = "i.año_lectivo_id = ?";
    $params[] = $anio_lectivo_id;
    $types .= "i";
}

if (!empty($fecha_graduacion_filtro)) {
    $conditions[] = "ep.fecha_graduacion = ?";
    $params[] = $fecha_graduacion_filtro;
    $types .= "s";
}

if (!empty($cedula_filtro)) {
    $conditions[] = "e.cedula LIKE ?";
    $params[] = "%$cedula_filtro%";
    $types .= "s";
}

if (!empty($conditions)) {
    $sql_graduados .= " AND " . implode(" AND ", $conditions);
}

$sql_graduados .= " GROUP BY e.id ORDER BY ep.fecha_graduacion DESC";

$stmt_graduados = $conn->prepare($sql_graduados);
if ($stmt_graduados) {
    if (!empty($params)) {
        $stmt_graduados->bind_param($types, ...$params);
    }
    if ($stmt_graduados->execute()) {
        $graduados = $stmt_graduados->get_result()->fetch_all(MYSQLI_ASSOC);
    } else {
        $_SESSION['error'] = "Error al obtener graduados: " . $stmt_graduados->error;
    }
    $stmt_graduados->close();
} else {
    $_SESSION['error'] = "Error en la consulta de graduados: " . $conn->error;
}

$page_title = "Gestión de Graduación";
include '../includes/header.php';
?>

<div class="container">
    <!-- Botón para volver al panel -->
    <div class="mb-4">
        <a href="index.php" class="btn btn-secondary">
            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-arrow-left" viewBox="0 0 16 16">
                <path fill-rule="evenodd" d="M15 8a.5.5 0 0 0-.5-.5H2.707l3.147-3.146a.5.5 0 1 0-.708-.708l-4 4a.5.5 0 0 0 0 .708l4 4a.5.5 0 0 0 .708-.708L2.707 8.5H14.5A.5.5 0 0 0 15 8z"/>
            </svg>
            Volver al Panel
        </a>
    </div>

    <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger"><?= $_SESSION['error'] ?></div>
        <?php unset($_SESSION['error']); ?>
    <?php endif; ?>
    
    <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success"><?= $_SESSION['success'] ?></div>
        <?php unset($_SESSION['success']); ?>
    <?php endif; ?>

    <!-- Filtros para ambos listados -->
    <div class="card mb-4">
        <div class="card-header bg-primary text-white">
            <h3 class="mb-0">Filtros</h3>
        </div>
        <div class="card-body">
            <form method="get">
                <div class="row g-3">
                    <div class="col-md-3">
                        <label for="postgrado_id" class="form-label">Postgrado:</label>
                        <select name="postgrado_id" id="postgrado_id" class="form-select">
                            <option value="0">Todos</option>
                            <?php 
                            $result_postgrados->data_seek(0);
                            while($p = $result_postgrados->fetch_assoc()): ?>
                                <option value="<?= $p['id'] ?>" <?= $postgrado_id == $p['id'] ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($p['nombre']) ?>
                                </option>
                            <?php endwhile; ?>
                        </select>
                    </div>
                    
                    <div class="col-md-3">
                        <label for="escolaridad_id" class="form-label">Escolaridad:</label>
                        <select name="escolaridad_id" id="escolaridad_id" class="form-select">
                            <option value="0">Todas</option>
                            <?php 
                            $result_escolaridades->data_seek(0);
                            while($e = $result_escolaridades->fetch_assoc()): ?>
                                <option value="<?= $e['id'] ?>" <?= $escolaridad_id == $e['id'] ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($e['nombre']) ?>
                                </option>
                            <?php endwhile; ?>
                        </select>
                    </div>
                    
                    <div class="col-md-3">
                        <label for="anio_lectivo_id" class="form-label">Año Lectivo:</label>
                        <select name="anio_lectivo_id" id="anio_lectivo_id" class="form-select">
                            <option value="0">Todos</option>
                            <?php 
                            $result_anios_lectivos->data_seek(0);
                            while($a = $result_anios_lectivos->fetch_assoc()): ?>
                                <option value="<?= $a['id'] ?>" <?= $anio_lectivo_id == $a['id'] ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($a['nombre']) ?>
                                </option>
                            <?php endwhile; ?>
                        </select>
                    </div>
                    
                    <div class="col-md-3">
                        <label for="fecha_graduacion" class="form-label">Fecha de Graduación:</label>
                        <input type="date" name="fecha_graduacion" id="fecha_graduacion" class="form-control" value="<?= htmlspecialchars($fecha_graduacion_filtro) ?>">
                    </div>
                    
                    <div class="col-md-4">
                        <label for="cedula" class="form-label">Cédula:</label>
                        <input type="text" name="cedula" id="cedula" class="form-control" placeholder="Buscar por cédula" value="<?= htmlspecialchars($cedula_filtro) ?>">
                    </div>
                    
                    <div class="col-md-2 d-flex align-items-end">
                        <button type="submit" class="btn btn-primary w-100">Aplicar Filtros</button>
                    </div>
                    
                    <div class="col-md-2 d-flex align-items-end">
                        <a href="gestion_graduacion.php" class="btn btn-outline-secondary w-100">Limpiar Filtros</a>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Estudiantes que han completado todas las denominaciones (sin graduar) -->
    <div class="card mb-4">
        <div class="card-header bg-info text-white">
            <h3 class="mb-0">Estudiantes que han completado todas las denominaciones</h3>
        </div>
        <div class="card-body">
            <?php if (!empty($estudiantes_completados)): ?>
                <div class="table-responsive">
                    <table class="table table-striped table-hover">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Nombre</th>
                                <th>Apellidos</th>
                                <th>Cédula</th>
                                <th>Postgrado</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($estudiantes_completados as $estudiante): ?>
                                <tr>
                                    <td><?= $estudiante['id'] ?></td>
                                    <td><?= htmlspecialchars($estudiante['nombres']) ?></td>
                                    <td><?= htmlspecialchars($estudiante['apellidos']) ?></td>
                                    <td><?= htmlspecialchars($estudiante['cedula']) ?></td>
                                    <td><?= htmlspecialchars($estudiante['postgrado_nombre']) ?></td>
                                    <td>
                                        <form method="post" class="row g-3 align-items-center">
                                            <input type="hidden" name="estudiante_id" value="<?= $estudiante['id'] ?>">
                                            <input type="hidden" name="postgrado_id" value="<?= $estudiante['postgrado_id'] ?>">
                                            
                                            <div class="col-md-4">
                                                <label class="form-label">Promoción:</label>
                                                <select name="promocion_id" class="form-select" required>
                                                    <option value="">-- Seleccione --</option>
                                                    <?php 
                                                    $result_promociones->data_seek(0);
                                                    while($prom = $result_promociones->fetch_assoc()): ?>
                                                        <option value="<?= $prom['id'] ?>"><?= htmlspecialchars($prom['nombre']) ?></option>
                                                    <?php endwhile; ?>
                                                </select>
                                            </div>
                                            
                                            <div class="col-md-4">
                                                <label class="form-label">Fecha de Graduación:</label>
                                                <input type="date" name="fecha_graduacion" class="form-control" required value="<?= date('Y-m-d') ?>">
                                            </div>
                                            
                                            <div class="col-md-4">
                                                <button type="submit" name="graduar_estudiante" class="btn btn-success mt-4">Graduar Estudiante</button>
                                            </div>
                                        </form>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php else: ?>
                <div class="alert alert-info">
                    <?php if ($postgrado_id > 0 || !empty($cedula_filtro)): ?>
                        No hay estudiantes que hayan completado todas las denominaciones con los filtros aplicados.
                    <?php else: ?>
                        No hay estudiantes que hayan completado todas las denominaciones en ningún postgrado.
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Estudiantes ya graduados -->
    <div class="card">
        <div class="card-header bg-success text-white">
            <h3 class="mb-0">Estudiantes Graduados</h3>
        </div>
        <div class="card-body">
            <?php if (!empty($graduados)): ?>
                <div class="table-responsive">
                    <table class="table table-striped table-hover">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Nombre</th>
                                <th>Apellidos</th>
                                <th>Cédula</th>
                                <th>Postgrado</th>
                                <th>Promoción</th>
                                <th>Fecha de Graduación</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($graduados as $graduado): ?>
                                <tr>
                                    <td><?= $graduado['id'] ?></td>
                                    <td><?= htmlspecialchars($graduado['nombres']) ?></td>
                                    <td><?= htmlspecialchars($graduado['apellidos']) ?></td>
                                    <td><?= htmlspecialchars($graduado['cedula']) ?></td>
                                    <td><?= htmlspecialchars($graduado['postgrado_nombre']) ?></td>
                                    <td><?= htmlspecialchars($graduado['promocion_nombre'] ?? 'Sin promoción') ?></td>
                                    <td><?= htmlspecialchars($graduado['fecha_graduacion']) ?></td>
                                    <td>
                                        <form method="post" class="row g-3 align-items-center">
                                            <input type="hidden" name="estudiante_id" value="<?= $graduado['id'] ?>">
                                            <input type="hidden" name="postgrado_id" value="<?= $graduado['postgrado_id'] ?>">
                                            
                                            <div class="col-md-4">
                                                <label class="form-label">Promoción:</label>
                                                <select name="promocion_id" class="form-select" required>
                                                    <option value="">-- Seleccione --</option>
                                                    <?php 
                                                    $result_promociones->data_seek(0);
                                                    while($prom = $result_promociones->fetch_assoc()): ?>
                                                        <option value="<?= $prom['id'] ?>" <?= ($graduado['promocion_id'] ?? '') == $prom['id'] ? 'selected' : '' ?>>
                                                            <?= htmlspecialchars($prom['nombre']) ?>
                                                        </option>
                                                    <?php endwhile; ?>
                                                </select>
                                            </div>
                                            
                                            <div class="col-md-4">
                                                <label class="form-label">Fecha:</label>
                                                <input type="date" name="fecha_graduacion" class="form-control" value="<?= htmlspecialchars($graduado['fecha_graduacion']) ?>" required>
                                            </div>
                                            
                                            <div class="col-md-4">
                                                <button type="submit" name="graduar_estudiante" class="btn btn-primary mt-4">Actualizar Graduación</button>
                                            </div>
                                        </form>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php else: ?>
                <div class="alert alert-info">
                    No se encontraron estudiantes graduados con los filtros aplicados.
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<?php
include '../includes/footer.php';
$conn->close();
?>