<?php
session_start();

// Configurar encoding UTF-8
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');

if (!isset($_SESSION['username']) || $_SESSION['rol'] != 'administrador') {
    header("Location: ../login.php");
    exit();
}

include '../includes/db.php';

// Configurar conexión para usar UTF-8
$conn->set_charset("utf8mb4");
$conn->query("SET NAMES 'utf8mb4'");
$conn->query("SET CHARACTER SET utf8mb4");
$conn->query("SET COLLATION_CONNECTION = 'utf8mb4_unicode_ci'");

// --- LÓGICA PARA AGREGAR MÚLTIPLES DENOMINACIONES ---
if (isset($_POST['agregar_denominacion'])) {
    $postgrado_id = $_POST['postgrado_id'];
    $codigos = $_POST['codigo'];
    $nombres = $_POST['nombre'];
    $creditos = $_POST['credito'];
    $periodos = $_POST['periodo'];
    $tipos = $_POST['tipo'];
    $tipos_denominacion = $_POST['tipo_denominacion'];
    $plan_id = !empty($_POST['plan_id_hidden']) ? $_POST['plan_id_hidden'] : null;

    // VALIDACIÓN DE SEGURIDAD DEL LADO DEL SERVIDOR (ESTADO DEL POSTGRADO)
    $sql_check_estado = "SELECT estado FROM postgrados WHERE id = ?";
    $stmt_check = $conn->prepare($sql_check_estado);
    $stmt_check->bind_param("i", $postgrado_id);
    $stmt_check->execute();
    $estado_postgrado = $stmt_check->get_result()->fetch_assoc()['estado'];
    $stmt_check->close();

    if ($estado_postgrado === 'cerrado') {
        $_SESSION['error'] = "Acción no permitida. El postgrado está cerrado y no admite nuevas denominaciones.";
        header("Location: gestion_denominaciones.php");
        exit();
    }

    // VALIDACIÓN PARA EVITAR CÓDIGOS DUPLICADOS
    $stmt_check_duplicado = $conn->prepare("SELECT COUNT(*) as total FROM denominaciones WHERE codigo = ? AND postgrado_id = ?");
    $errores = [];
    $processed_codes = [];

    foreach ($codigos as $key => $codigo) {
        if (!empty($codigo) && !empty($nombres[$key])) {
            if (in_array($codigo, $processed_codes)) {
                $errores[] = "El código '".htmlspecialchars($codigo)."' está duplicado en el formulario.";
                continue;
            }

            $stmt_check_duplicado->bind_param("si", $codigo, $postgrado_id);
            $stmt_check_duplicado->execute();
            $result_duplicado = $stmt_check_duplicado->get_result()->fetch_assoc();
            
            if ($result_duplicado['total'] > 0) {
                $errores[] = "El código de denominación '".htmlspecialchars($codigo)."' ya existe para este postgrado.";
            }
            $processed_codes[] = $codigo;
        }
    }
    $stmt_check_duplicado->close();

    // Si hay errores, guardar los datos en sesión y mostrar errores
    if (!empty($errores)) {
        $_SESSION['form_data'] = $_POST;
        $_SESSION['error'] = implode("<br>", $errores);
        header("Location: gestion_denominaciones.php");
        exit();
    }

    $conn->begin_transaction();
    try {
        $sql = "INSERT INTO denominaciones (postgrado_id, codigo, nombre, credito, periodo, plan_id, tipo, tipo_denominacion) VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
        $stmt = $conn->prepare($sql);

        $added_count = 0;
        foreach ($codigos as $key => $codigo) {
            if (!empty($codigo) && !empty($nombres[$key]) && !empty($periodos[$key])) {
                $nombre = $nombres[$key];
                $credito = $creditos[$key];
                $periodo = $periodos[$key];
                $tipo = $tipos[$key];
                $tipo_denominacion = $tipos_denominacion[$key];

                $stmt->bind_param("issiiiss", $postgrado_id, $codigo, $nombre, $credito, $periodo, $plan_id, $tipo, $tipo_denominacion);
                $stmt->execute();
                $added_count++;
            }
        }

        $conn->commit();
        if ($added_count > 0) {
            $_SESSION['success'] = "$added_count denominacion(es) agregada(s) correctamente.";
            // Limpiar datos del formulario después de éxito
            unset($_SESSION['form_data']);
        } else {
            $_SESSION['error'] = "No se agregaron denominaciones. Por favor, complete todos los campos requeridos.";
            $_SESSION['form_data'] = $_POST;
        }

    } catch (Exception $e) {
        $conn->rollback();
        $_SESSION['error'] = "Error al agregar las denominaciones: " . $e->getMessage();
        $_SESSION['form_data'] = $_POST;
    }

    header("Location: gestion_denominaciones.php");
    exit();
}

// --- LÓGICA PARA ELIMINAR DENOMINACIÓN ---
if (isset($_GET['eliminar'])) {
    $id = $_GET['eliminar'];
    $sql_check = "SELECT COUNT(*) as total FROM inscripciones WHERE denominacion_id = ?";
    $stmt_check = $conn->prepare($sql_check);
    $stmt_check->bind_param("i", $id);
    $stmt_check->execute();
    $row_check = $stmt_check->get_result()->fetch_assoc();
    if ($row_check['total'] > 0) {
        $_SESSION['error'] = "No se puede eliminar la denominación porque tiene inscripciones asociadas.";
    } else {
        $sql = "DELETE FROM denominaciones WHERE id = ?";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("i", $id);
        if ($stmt->execute()) {
            $_SESSION['success'] = "Denominación eliminada correctamente.";
        } else {
            $_SESSION['error'] = "Error al eliminar la denominación: " . $stmt->error;
        }
    }
    header("Location: gestion_denominaciones.php");
    exit();
}

$page_title = "Gestión de Denominaciones";
include '../includes/header.php';

// Obtener datos del formulario guardados en sesión (si existen)
$form_data = isset($_SESSION['form_data']) ? $_SESSION['form_data'] : [];

// Obtener lista de postgrados
$sql_postgrados = "SELECT p.id, p.nombre, p.estado, pl.id as plan_id, pl.nombre as plan_nombre 
                  FROM postgrados p
                  LEFT JOIN planes pl ON p.plan_id = pl.id
                  ORDER BY p.nombre";
$result_postgrados = $conn->query($sql_postgrados);

// Filtrar por postgrado
$postgrado_filtro = isset($_GET['postgrado_filtro']) ? intval($_GET['postgrado_filtro']) : 0;
?>

<div class="container mt-4">
    <div class="mb-4">
        <a href="index.php" class="btn btn-secondary">
            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-arrow-left" viewBox="0 0 16 16">
                <path fill-rule="evenodd" d="M15 8a.5.5 0 0 0-.5-.5H2.707l3.147-3.146a.5.5 0 1 0-.708-.708l-4 4a.5.5 0 0 0 0 .708l4 4a.5.5 0 0 0 .708-.708L2.707 8.5H14.5A.5.5 0 0 0 15 8z"/>
            </svg>
            Volver al Panel
        </a>
    </div>

    <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert"><?= htmlspecialchars($_SESSION['success']) ?><button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button></div>
        <?php unset($_SESSION['success']); ?>
    <?php endif; ?>
    
    <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert"><?= $_SESSION['error'] ?><button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button></div>
        <?php unset($_SESSION['error']); ?>
    <?php endif; ?>

    <div class="card">
        <div class="card-header bg-primary text-white">
            <h3 class="mb-0">Agregar Nuevas Denominaciones</h3>
        </div>
        <div class="card-body">
            <form action="gestion_denominaciones.php" method="post" id="denominaciones-form">
                <div class="row g-3 mb-3">
                    <div class="col-md-6">
                        <label for="postgrado_id" class="form-label fw-bold">1. Postgrado (Afectará a todas las filas):</label>
                        <select class="form-select" id="postgrado_id" name="postgrado_id" required>
                            <option value="">Seleccione un postgrado</option>
                            <?php 
                            $result_postgrados->data_seek(0);
                            while ($postgrado = $result_postgrados->fetch_assoc()): 
                                $selected = (isset($form_data['postgrado_id']) && $form_data['postgrado_id'] == $postgrado['id']) ? 'selected' : '';
                            ?>
                                <option value="<?= $postgrado['id'] ?>" 
                                        data-plan-id="<?= htmlspecialchars($postgrado['plan_id'] ?? '') ?>" 
                                        data-plan-nombre="<?= htmlspecialchars($postgrado['plan_nombre'] ?? '') ?>"
                                        data-estado="<?= htmlspecialchars($postgrado['estado']) ?>"
                                        <?= $selected ?>>
                                    <?= htmlspecialchars($postgrado['nombre']) ?>
                                </option>
                            <?php endwhile; ?>
                        </select>
                        <input type="hidden" id="plan_id_hidden" name="plan_id_hidden" value="<?= htmlspecialchars($form_data['plan_id_hidden'] ?? '') ?>"> 
                    </div>
                    <div class="col-md-6">
                         <label for="plan_nombre_display" class="form-label fw-bold">Plan Asociado:</label>
                         <input type="text" class="form-control" id="plan_nombre_display" readonly 
                                value="<?= htmlspecialchars($form_data['plan_nombre_display'] ?? '') ?>" placeholder="Se asignará automáticamente">
                    </div>
                </div>

                <hr>
                
                <div id="mensaje-bloqueo" class="alert alert-warning" style="display: none;">
                    <strong>Atención:</strong> Este postgrado está cerrado y no admite nuevas denominaciones. Para habilitarlo, vaya a la <a href="gestionar_estado_postgrados.php">página de gestión de estados</a>.
                </div>

                <fieldset id="fieldset-denominaciones">
                    <h5 class="mt-3">2. Lista de Denominaciones a Agregar</h5>
                    <div id="denominaciones-container">
                        <?php
                        // Determinar cuántas filas mostrar
                        $num_filas = 1;
                        if (isset($form_data['codigo']) && is_array($form_data['codigo'])) {
                            $num_filas = count($form_data['codigo']);
                        }
                        
                        for ($i = 0; $i < $num_filas; $i++):
                        ?>
                        <div class="row g-3 align-items-center denominacion-row mb-2">
                            <div class="col-md-2">
                                <input type="text" class="form-control codigo-input" name="codigo[]" required 
                                       value="<?= htmlspecialchars($form_data['codigo'][$i] ?? '') ?>" placeholder="Código">
                            </div>
                            <div class="col-md-4">
                                <input type="text" class="form-control" name="nombre[]" required 
                                       value="<?= htmlspecialchars($form_data['nombre'][$i] ?? '') ?>" placeholder="Nombre de la Asignatura">
                            </div>
                            <div class="col-md-1">
                                <input type="number" class="form-control" name="credito[]" min="0" required 
                                       value="<?= htmlspecialchars($form_data['credito'][$i] ?? '') ?>" placeholder="Créd.">
                            </div>
                            <div class="col-md-2">
                                <select class="form-select" name="periodo[]" required>
                                    <option value="">Período</option>
                                    <?php for ($p = 1; $p <= 15; $p++): 
                                        $selected = (isset($form_data['periodo'][$i]) && $form_data['periodo'][$i] == $p) ? 'selected' : '';
                                    ?>
                                        <option value="<?= $p ?>" <?= $selected ?>>Período <?= $p ?></option>
                                    <?php endfor; ?>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <select class="form-select" name="tipo_denominacion[]" required>
                                    <option value="obligatoria" <?= (isset($form_data['tipo_denominacion'][$i]) && $form_data['tipo_denominacion'][$i] == 'obligatoria') ? 'selected' : '' ?>>Obligatoria</option>
                                    <option value="electiva" <?= (isset($form_data['tipo_denominacion'][$i]) && $form_data['tipo_denominacion'][$i] == 'electiva') ? 'selected' : '' ?>>Electiva</option>
                                </select>
                            </div>
                            <div class="col-md-1">
                                <select class="form-select" name="tipo[]" required>
                                    <option value="puntaje" <?= (isset($form_data['tipo'][$i]) && $form_data['tipo'][$i] == 'puntaje') ? 'selected' : '' ?>>Puntaje</option>
                                    <option value="Aprobado_Reprobado" <?= (isset($form_data['tipo'][$i]) && $form_data['tipo'][$i] == 'Aprobado_Reprobado') ? 'selected' : '' ?>>A/R</option>
                                </select>
                            </div>
                        </div>
                        <?php endfor; ?>
                    </div>

                    <div class="col-12 mt-3">
                        <button type="button" id="add-row-btn" class="btn btn-success me-2">Añadir Otra Denominación</button>
                        <button type="button" id="remove-last-row-btn" class="btn btn-warning">Borrar Última Denominación</button>
                    </div>
                    
                    <div class="col-12 mt-4 border-top pt-3">
                        <button type="submit" name="agregar_denominacion" class="btn btn-primary btn-lg">Guardar Todas las Denominaciones</button>
                        <button type="button" id="limpiar-formulario" class="btn btn-secondary btn-lg">Limpiar Formulario</button>
                    </div>
                </fieldset>
            </form>
        </div>
    </div>
    
    <!-- El resto del código se mantiene igual -->
    <div class="card mt-4">
        <div class="card-header bg-info text-white">
            <h3 class="mb-0">Filtrar Denominaciones</h3>
        </div>
        <div class="card-body">
            <form method="get" class="row g-3">
                <div class="col-md-6">
                    <label for="postgrado_filtro" class="form-label">Postgrado:</label>
                    <select name="postgrado_filtro" id="postgrado_filtro" class="form-select" onchange="this.form.submit()">
                        <option value="">Todos los postgrados</option>
                        <?php
                        $result_postgrados->data_seek(0);
                        while ($postgrado = $result_postgrados->fetch_assoc()):
                            $selected = ($postgrado_filtro == $postgrado['id']) ? 'selected' : '';
                        ?>
                            <option value="<?= $postgrado['id'] ?>" <?= $selected ?>>
                                <?= htmlspecialchars($postgrado['nombre']) ?>
                            </option>
                        <?php endwhile; ?>
                    </select>
                </div>
            </form>
        </div>
    </div>

    <div class="card mt-4">
        <div class="card-header bg-info text-white d-flex justify-content-between align-items-center">
            <h3 class="mb-0">Denominaciones Registradas</h3>
            <div>
                <button id="expandAllBtn" class="btn btn-sm btn-outline-light me-2">
                    <i class="fas fa-expand"></i> Abrir todos
                </button>
                <button id="collapseAllBtn" class="btn btn-sm btn-outline-light">
                    <i class="fas fa-compress"></i> Cerrar todos
                </button>
            </div>
        </div>
        <div class="card-body p-0">
            <?php
            // CONSULTA CORREGIDA - SIN FUNCIONES DE VENTANA PARA COMPATIBILIDAD
            $sql = "SELECT 
                        p.id as postgrado_id, 
                        p.nombre as postgrado_nombre, 
                        d.id as denominacion_id, 
                        d.codigo, 
                        d.nombre as denominacion_nombre, 
                        d.credito, 
                        d.periodo, 
                        d.tipo, 
                        d.tipo_denominacion, 
                        pl.nombre as plan_nombre,
                        (SELECT COUNT(*) FROM denominaciones WHERE postgrado_id = p.id) as total_denominaciones,
                        (SELECT COALESCE(SUM(credito), 0) FROM denominaciones WHERE postgrado_id = p.id) as total_creditos
                    FROM postgrados p 
                    LEFT JOIN denominaciones d ON p.id = d.postgrado_id 
                    LEFT JOIN planes pl ON d.plan_id = pl.id";
            
            if ($postgrado_filtro) { 
                $sql .= " WHERE p.id = $postgrado_filtro"; 
            }
            
            $sql .= " ORDER BY p.nombre, d.periodo, d.nombre";
            
            $result = $conn->query($sql);
            if ($result->num_rows > 0):
                $postgrados = [];
                while ($row = $result->fetch_assoc()) {
                    $postgrado_id = $row['postgrado_id'];
                    if (!isset($postgrados[$postgrado_id])) {
                        $postgrados[$postgrado_id] = [ 
                            'nombre' => $row['postgrado_nombre'], 
                            'total_denominaciones' => $row['total_denominaciones'], 
                            'total_creditos' => $row['total_creditos'], 
                            'denominaciones' => [] 
                        ];
                    }
                    if ($row['denominacion_id']) { 
                        $postgrados[$postgrado_id]['denominaciones'][] = $row; 
                    }
                }
            ?>
            
            <ul class="nav nav-tabs" id="postgradosTab" role="tablist">
                <?php foreach ($postgrados as $postgrado_id => $postgrado): ?>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link <?= ($postgrado_id == array_key_first($postgrados)) ? 'active' : '' ?>" id="tab-<?= $postgrado_id ?>" data-bs-toggle="tab" data-bs-target="#content-<?= $postgrado_id ?>" type="button" role="tab">
                            <?= htmlspecialchars($postgrado['nombre']) ?>
                            <span class="badge bg-primary ms-2"><?= $postgrado['total_denominaciones'] ?></span>
                        </button>
                    </li>
                <?php endforeach; ?>
            </ul>
            
            <div class="tab-content p-3 border border-top-0 rounded-bottom" id="postgradosTabContent">
                <?php foreach ($postgrados as $postgrado_id => $postgrado): ?>
                    <div class="tab-pane fade <?= ($postgrado_id == array_key_first($postgrados)) ? 'show active' : '' ?>" id="content-<?= $postgrado_id ?>" role="tabpanel">
                        <div class="mb-3">
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="card bg-light">
                                        <div class="card-body">
                                            <h5 class="card-title">Resumen del Postgrado</h5>
                                            <p class="card-text">
                                                <strong>Total Denominaciones:</strong> <?= $postgrado['total_denominaciones'] ?><br>
                                                <strong>Total Créditos:</strong> <?= $postgrado['total_creditos'] ?>
                                            </p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="accordion" id="accordionPeriodos-<?= $postgrado_id ?>">
                            <?php
                            $periodos = [];
                            foreach ($postgrado['denominaciones'] as $denominacion) {
                                $periodo = $denominacion['periodo'];
                                if (!isset($periodos[$periodo])) { $periodos[$periodo] = []; }
                                $periodos[$periodo][] = $denominacion;
                            }
                            foreach ($periodos as $periodo => $denominaciones): 
                                $periodo_id = "periodo-{$postgrado_id}-{$periodo}";
                                $total_creditos_periodo = array_sum(array_column($denominaciones, 'credito'));
                            ?>
                                <div class="accordion-item">
                                    <h2 class="accordion-header" id="heading-<?= $periodo_id ?>">
                                        <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapse-<?= $periodo_id ?>" aria-expanded="false" aria-controls="collapse-<?= $periodo_id ?>">
                                            Periodo <?= $periodo ?> 
                                            <span class="badge bg-secondary ms-2"><?= count($denominaciones) ?> denominaciones</span>
                                            <span class="badge bg-success ms-2"><?= $total_creditos_periodo ?> créditos</span>
                                        </button>
                                    </h2>
                                    <div id="collapse-<?= $periodo_id ?>" class="accordion-collapse collapse" aria-labelledby="heading-<?= $periodo_id ?>" data-bs-parent="#accordionPeriodos-<?= $postgrado_id ?>">
                                        <div class="accordion-body p-0">
                                            <div class="table-responsive">
                                                <table class="table table-striped table-hover mb-0">
                                                    <thead>
                                                        <tr><th>Código</th><th>Nombre</th><th>Créditos</th><th>Tipo</th><th>Tipo Denominación</th><th>Plan</th><th>Acciones</th></tr>
                                                    </thead>
                                                    <tbody>
                                                        <?php foreach ($denominaciones as $denominacion): ?>
                                                            <tr>
                                                                <td><?= htmlspecialchars($denominacion['codigo']) ?></td>
                                                                <td><?= htmlspecialchars($denominacion['denominacion_nombre']) ?></td>
                                                                <td><?= $denominacion['credito'] ?></td>
                                                                <td><?= $denominacion['tipo'] == 'puntaje' ? 'Puntaje' : 'Aprobado/Reprobado' ?></td>
                                                                <td>
                                                                    <span class="badge bg-<?= $denominacion['tipo_denominacion'] == 'obligatoria' ? 'primary' : 'warning' ?>">
                                                                        <?= ucfirst($denominacion['tipo_denominacion']) ?>
                                                                    </span>
                                                                </td>
                                                                <td><?= htmlspecialchars($denominacion['plan_nombre'] ?? '') ?></td>
                                                                <td>
                                                                    <a href='editar_denominacion.php?id=<?= $denominacion['denominacion_id'] ?>' class="btn btn-sm btn-primary">Editar</a>
                                                                    <a href='gestion_denominaciones.php?eliminar=<?= $denominacion['denominacion_id'] ?>' class="btn btn-sm btn-danger" onclick='return confirm("¿Estás seguro de eliminar esta denominación?")'>Eliminar</a>
                                                                </td>
                                                            </tr>
                                                        <?php endforeach; ?>
                                                    </tbody>
                                                </table>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
            <?php else: ?>
                <div class="alert alert-info m-3">No hay denominaciones registradas</div>
            <?php endif; ?>
        </div>
    </div>
</div>

<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script>
$(document).ready(function() {
    let currentPlanName = ''; 

    function setCodePrefix(rowElement) {
        const codigoInput = rowElement.find('.codigo-input');
        codigoInput.val(currentPlanName);
    }

    $("#add-row-btn").click(function() {
        if (!$('#postgrado_id').val()) {
            alert('Por favor, seleccione primero un Postgrado.');
            return;
        }
        const newRow = $(".denominacion-row:first").clone();
        newRow.find("input[type='text'], input[type='number']").val('');
        newRow.find("select").prop('selectedIndex', 0);
        setCodePrefix(newRow);
        $("#denominaciones-container").append(newRow);
        newRow.find("input[name='nombre[]']").focus();
    });

    $("#remove-last-row-btn").click(function() {
        if ($(".denominacion-row").length > 1) {
            $(".denominacion-row:last").remove();
        } else {
            alert("No se puede eliminar la última fila.");
        }
    });

    $('#postgrado_id').change(function() {
        const selectedOption = $(this).find('option:selected');
        const planId = selectedOption.data('plan-id');
        const planNombre = selectedOption.data('plan-nombre');
        const estado = selectedOption.data('estado');
        
        $('#plan_id_hidden').val(planId);
        $('#plan_nombre_display').val(planNombre);
        
        // Solo limpiar y resetear si no hay datos guardados
        if (!<?= !empty($form_data) ? 'true' : 'false' ?>) {
            $("#denominaciones-container .denominacion-row:not(:first)").remove();
            const firstRow = $(".denominacion-row:first");
            firstRow.find("input[type='text'], input[type='number']").val('');
            firstRow.find("select").prop('selectedIndex', 0);
        }
        
        currentPlanName = planNombre ? $.trim(planNombre) : ''; 
        
        // Aplicar prefijo solo a las filas nuevas o sin datos
        $(".denominacion-row").each(function() {
            const codigoInput = $(this).find('.codigo-input');
            if (!codigoInput.val()) {
                codigoInput.val(currentPlanName);
            }
        });

        if (estado === 'cerrado') {
            $('#fieldset-denominaciones').prop('disabled', true);
            $('#mensaje-bloqueo').show();
        } else {
            $('#fieldset-denominaciones').prop('disabled', false);
            $('#mensaje-bloqueo').hide();
        }
    });

    // Funcionalidad para expandir/colapsar todos los acordeones
    $('#expandAllBtn').click(function() {
        $('.accordion-collapse').each(function() {
            var collapse = new bootstrap.Collapse(this, {
                toggle: true
            });
            collapse.show();
        });
    });

    $('#collapseAllBtn').click(function() {
        $('.accordion-collapse').each(function() {
            var collapse = new bootstrap.Collapse(this, {
                toggle: true
            });
            collapse.hide();
        });
    });

    // Limpiar formulario
    $('#limpiar-formulario').click(function() {
        if (confirm('¿Está seguro de que desea limpiar todo el formulario? Se perderán todos los datos no guardados.')) {
            // Limpiar datos de sesión
            $.post('limpiar_formulario.php', function() {
                location.reload();
            });
        }
    });

    // Si hay datos guardados, activar el cambio del postgrado para cargar la información
    <?php if (!empty($form_data) && isset($form_data['postgrado_id'])): ?>
        $('#postgrado_id').trigger('change');
    <?php endif; ?>
});
</script>

<?php
include '../includes/footer.php';
$conn->close();

// Limpiar datos del formulario después de mostrarlos
unset($_SESSION['form_data']);
?>