<?php
// 97 EDITAR POSTGRADO - P09GINA PROTEGIDA

// 1. SIEMPRE PRIMERO - configuracin de sesin y encoding
require_once __DIR__ . '/../includes/session_config.php';

// Forzar encoding UTF-8 desde el inicio
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');
mb_http_output('UTF-8');
ob_start('mb_output_handler');

// 2. SIEMPRE SEGUNDO - verificacin de auth
$allowed_roles = ['administrador']; // Solo administradores pueden editar postgrados
require_once __DIR__ . '/../includes/auth_check.php';

// 3. DB Connection
require_once __DIR__ . '/../includes/db.php';

// Configurar conexin para usar UTF-8
$conn->set_charset("utf8mb4");
$conn->query("SET NAMES 'utf8mb4'");
$conn->query("SET CHARACTER SET utf8mb4");
$conn->query("SET COLLATION_CONNECTION = 'utf8mb4_unicode_ci'");

// 4. Procesar formulario de edicin
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['editar_postgrado'])) {
    $id = intval($_POST['id']);
    $nombre = trim($_POST['nombre']);
    $plan_id = intval($_POST['plan_id']);
    $tipo_postgrado = $_POST['tipo_postgrado'];
    $creditos_totales = intval($_POST['creditos_totales']);
    $creditos_obligatorios = intval($_POST['creditos_obligatorios']);
    $creditos_electivos = intval($_POST['creditos_electivos']);
    $estado = $_POST['estado'];
    
    // Validar que los crditos coincidan segn el tipo de postgrado
    if ($tipo_postgrado == 'tradicional') {
        $creditos_obligatorios = $creditos_totales;
        $creditos_electivos = 0;
    } elseif ($tipo_postgrado == 'electivo') {
        $creditos_obligatorios = 0;
        $creditos_electivos = $creditos_totales;
    } else { // mixto
        if ($creditos_obligatorios + $creditos_electivos != $creditos_totales) {
            $_SESSION['error'] = "La suma de creditos obligatorios y electivos debe ser igual a los creditos totales para postgrados mixtos.";
            header("Location: editar_postgrado.php?id=" . $id);
            exit();
        }
    }
    
    // Verificar si el nombre ya existe en otro postgrado
    $sql_check = "SELECT id FROM postgrados WHERE nombre = ? AND plan_id = ? AND id != ?";
    $stmt_check = $conn->prepare($sql_check);
    $stmt_check->bind_param("sii", $nombre, $plan_id, $id);
    $stmt_check->execute();
    $result_check = $stmt_check->get_result();
    
    if ($result_check->num_rows > 0) {
        $_SESSION['error'] = "Ya existe otro postgrado con el mismo nombre y plan.";
        header("Location: editar_postgrado.php?id=" . $id);
        exit();
    }
    
    // Actualizar el postgrado
    $sql = "UPDATE postgrados SET nombre = ?, plan_id = ?, tipo_postgrado = ?, 
            creditos_totales = ?, creditos_obligatorios = ?, creditos_electivos = ?, estado = ? 
            WHERE id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("sisiiisi", $nombre, $plan_id, $tipo_postgrado, 
                     $creditos_totales, $creditos_obligatorios, $creditos_electivos, $estado, $id);
    
    if ($stmt->execute()) {
        $_SESSION['success'] = "Postgrado actualizado correctamente";
        header("Location: gestion_postgrados.php");
        exit();
    } else {
        $_SESSION['error'] = "Error al actualizar el postgrado: " . $conn->error;
        header("Location: editar_postgrado.php?id=" . $id);
        exit();
    }
}

// 5. Obtener datos del postgrado a editar
if (!isset($_GET['id']) || empty($_GET['id'])) {
    $_SESSION['error'] = "ID de postgrado no especificado";
    header("Location: gestion_postgrados.php");
    exit();
}

$id = intval($_GET['id']);
$sql = "SELECT p.*, pl.nombre as plan_nombre 
        FROM postgrados p 
        JOIN planes pl ON p.plan_id = pl.id 
        WHERE p.id = ?";
$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $id);
$stmt->execute();
$result = $stmt->get_result();
$postgrado = $result->fetch_assoc();

if (!$postgrado) {
    $_SESSION['error'] = "Postgrado no encontrado";
    header("Location: gestion_postgrados.php");
    exit();
}

// 6. Configurar ttulo de pgina
$page_title = "Editar Postgrado";
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($page_title); ?> - Sistema de Postgrados</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.8.1/font/bootstrap-icons.css">
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-color: #f8f9fa;
        }
        .card {
            box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
            border: 1px solid rgba(0, 0, 0, 0.125);
            margin-bottom: 1rem;
        }
        .card-header {
            font-weight: 600;
        }
        .form-control:read-only {
            background-color: #e9ecef;
            opacity: 1;
        }
        .bi {
            margin-right: 5px;
        }
        .alert {
            border: none;
            border-left: 4px solid;
        }
        .alert-success {
            border-left-color: #198754;
            background-color: #d1e7dd;
        }
        .alert-danger {
            border-left-color: #dc3545;
            background-color: #f8d7da;
        }
        .badge {
            font-size: 0.875em;
        }
        .form-label {
            font-weight: 500;
            margin-bottom: 0.5rem;
        }
    </style>
</head>
<body>
    <div class="container mt-4">
        <!-- Botn para volver -->
        <div class="mb-4">
            <a href="gestion_postgrados.php" class="btn btn-secondary">
                <i class="bi bi-arrow-left"></i> Volver a Gestion de Postgrados
            </a>
        </div>

        <!-- Mostrar mensajes -->
        <?php if (isset($_SESSION['success'])): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <i class="bi bi-check-circle-fill"></i> <?php echo htmlspecialchars($_SESSION['success']); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
            <?php unset($_SESSION['success']); ?>
        <?php endif; ?>
        
        <?php if (isset($_SESSION['error'])): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <i class="bi bi-exclamation-triangle-fill"></i> <?php echo htmlspecialchars($_SESSION['error']); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
            <?php unset($_SESSION['error']); ?>
        <?php endif; ?>

        <!-- Formulario de edicion -->
        <div class="card">
            <div class="card-header bg-primary text-white">
                <h3 class="mb-0"><i class="bi bi-pencil-square"></i> Editar Postgrado</h3>
            </div>
            <div class="card-body">
                <form method="post" class="row g-3">
                    <input type="hidden" name="id" value="<?php echo $postgrado['id']; ?>">
                    
                    <div class="col-md-6">
                        <label for="nombre" class="form-label">Nombre del Postgrado:</label>
                        <input type="text" class="form-control" id="nombre" name="nombre" 
                               value="<?php echo htmlspecialchars($postgrado['nombre']); ?>" required>
                    </div>
                    
                    <div class="col-md-6">
                        <label for="plan_id" class="form-label">Plan de Estudio:</label>
                        <select class="form-select" id="plan_id" name="plan_id" required>
                            <option value="">Seleccione un plan</option>
                            <?php
                            $sql_planes = "SELECT * FROM planes ORDER BY nombre";
                            $result_planes = $conn->query($sql_planes);
                            while ($plan = $result_planes->fetch_assoc()) {
                                $selected = ($plan['id'] == $postgrado['plan_id']) ? 'selected' : '';
                                echo "<option value='" . $plan['id'] . "' $selected>" . htmlspecialchars($plan['nombre']) . "</option>";
                            }
                            ?>
                        </select>
                    </div>
                    
                    <div class="col-md-4">
                        <label for="tipo_postgrado" class="form-label">Tipo de Postgrado:</label>
                        <select class="form-select" id="tipo_postgrado" name="tipo_postgrado" required>
                            <option value="tradicional" <?php echo $postgrado['tipo_postgrado'] == 'tradicional' ? 'selected' : ''; ?>>Tradicional (Todas obligatorias)</option>
                            <option value="mixto" <?php echo $postgrado['tipo_postgrado'] == 'mixto' ? 'selected' : ''; ?>>Mixto (Obligatorias y Electivas)</option>
                            <option value="electivo" <?php echo $postgrado['tipo_postgrado'] == 'electivo' ? 'selected' : ''; ?>>Electivo (Todas electivas)</option>
                        </select>
                    </div>
                    
                    <div class="col-md-4">
                        <label for="estado" class="form-label">Estado:</label>
                        <select class="form-select" id="estado" name="estado" required>
                            <option value="abierto" <?php echo $postgrado['estado'] == 'abierto' ? 'selected' : ''; ?>>Abierto</option>
                            <option value="cerrado" <?php echo $postgrado['estado'] == 'cerrado' ? 'selected' : ''; ?>>Cerrado</option>
                        </select>
                    </div>
                    
                    <div class="col-md-4">
                        <label for="creditos_totales" class="form-label">Creditos Totales:</label>
                        <input type="number" class="form-control" id="creditos_totales" name="creditos_totales" 
                               value="<?php echo $postgrado['creditos_totales']; ?>" min="0" required>
                    </div>
                    
                    <div class="col-md-4">
                        <label for="creditos_obligatorios" class="form-label">Creditos Obligatorios:</label>
                        <input type="number" class="form-control" id="creditos_obligatorios" name="creditos_obligatorios" 
                               value="<?php echo $postgrado['creditos_obligatorios']; ?>" min="0" required>
                    </div>
                    
                    <div class="col-md-4">
                        <label for="creditos_electivos" class="form-label">Creditos Electivos:</label>
                        <input type="number" class="form-control" id="creditos_electivos" name="creditos_electivos" 
                               value="<?php echo $postgrado['creditos_electivos']; ?>" min="0" required>
                    </div>
                    
                    <div class="col-12">
                        <div class="card bg-light">
                            <div class="card-body">
                                <h6 class="card-title"><i class="bi bi-graph-up"></i> Resumen de Creditos</h6>
                                <div class="row">
                                    <div class="col-md-4">
                                        <strong>Creditos Totales:</strong> 
                                        <span id="resumen_totales" class="badge bg-primary"><?php echo $postgrado['creditos_totales']; ?></span>
                                    </div>
                                    <div class="col-md-4">
                                        <strong>Creditos Obligatorios:</strong> 
                                        <span id="resumen_obligatorios" class="badge bg-warning text-dark"><?php echo $postgrado['creditos_obligatorios']; ?></span>
                                    </div>
                                    <div class="col-md-4">
                                        <strong>Creditos Electivos:</strong> 
                                        <span id="resumen_electivos" class="badge bg-info"><?php echo $postgrado['creditos_electivos']; ?></span>
                                    </div>
                                </div>
                                <div class="mt-2" id="validacion_creditos"></div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-12 mt-4">
                        <button type="submit" name="editar_postgrado" class="btn btn-primary" id="btn-guardar">
                            <i class="bi bi-check-lg"></i> Guardar Cambios
                        </button>
                        <a href="gestion_postgrados.php" class="btn btn-secondary">
                            <i class="bi bi-x-lg"></i> Cancelar
                        </a>
                    </div>
                </form>
            </div>
        </div>

        <!-- Informacion del postgrado -->
        <div class="card mt-4">
            <div class="card-header bg-info text-white">
                <h5 class="mb-0"><i class="bi bi-info-circle"></i> Informacion del Postgrado</h5>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <p><strong>ID:</strong> <?php echo $postgrado['id']; ?></p>
                        <p><strong>Plan Actual:</strong> <?php echo htmlspecialchars($postgrado['plan_nombre']); ?></p>
                        <p><strong>Tipo Actual:</strong> 
                            <span class="badge bg-info"><?php echo ucfirst($postgrado['tipo_postgrado']); ?></span>
                        </p>
                    </div>
                    <div class="col-md-6">
                        <p><strong>Estado Actual:</strong> 
                            <span class="badge bg-<?php echo $postgrado['estado'] == 'abierto' ? 'success' : 'secondary'; ?>">
                                <?php echo ucfirst($postgrado['estado']); ?>
                            </span>
                        </p>
                        <p><strong>Creditos Actuales:</strong> 
                            <?php echo $postgrado['creditos_totales']; ?> totales 
                            (<?php echo $postgrado['creditos_obligatorios']; ?> obligatorios, 
                             <?php echo $postgrado['creditos_electivos']; ?> electivos)
                        </p>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        const tipoPostgrado = document.getElementById('tipo_postgrado');
        const creditosTotales = document.getElementById('creditos_totales');
        const creditosObligatorios = document.getElementById('creditos_obligatorios');
        const creditosElectivos = document.getElementById('creditos_electivos');
        const btnGuardar = document.getElementById('btn-guardar');
        
        // Elementos del resumen
        const resumenTotales = document.getElementById('resumen_totales');
        const resumenObligatorios = document.getElementById('resumen_obligatorios');
        const resumenElectivos = document.getElementById('resumen_electivos');
        const validacionCreditos = document.getElementById('validacion_creditos');

        function actualizarResumen() {
            const totales = parseInt(creditosTotales.value) || 0;
            const obligatorios = parseInt(creditosObligatorios.value) || 0;
            const electivos = parseInt(creditosElectivos.value) || 0;
            
            resumenTotales.textContent = totales;
            resumenObligatorios.textContent = obligatorios;
            resumenElectivos.textContent = electivos;
            
            // Validar segun el tipo de postgrado
            const tipo = tipoPostgrado.value;
            let mensaje = '';
            let esValido = true;
            
            if (tipo === 'tradicional') {
                if (obligatorios !== totales) {
                    mensaje = '<span class="text-danger"><i class="bi bi-exclamation-triangle"></i> Para postgrados tradicionales, los creditos obligatorios deben ser iguales a los creditos totales.</span>';
                    esValido = false;
                } else {
                    mensaje = '<span class="text-success"><i class="bi bi-check-circle"></i> Configuracion valida para postgrado tradicional.</span>';
                }
            } else if (tipo === 'electivo') {
                if (electivos !== totales) {
                    mensaje = '<span class="text-danger"><i class="bi bi-exclamation-triangle"></i> Para postgrados electivos, los creditos electivos deben ser iguales a los creditos totales.</span>';
                    esValido = false;
                } else {
                    mensaje = '<span class="text-success"><i class="bi bi-check-circle"></i> Configuracion valida para postgrado electivo.</span>';
                }
            } else { // mixto
                if (obligatorios + electivos !== totales) {
                    mensaje = '<span class="text-danger"><i class="bi bi-exclamation-triangle"></i> Para postgrados mixtos, la suma de creditos obligatorios y electivos debe ser igual a los creditos totales.</span>';
                    esValido = false;
                } else {
                    mensaje = '<span class="text-success"><i class="bi bi-check-circle"></i> Configuracion valida para postgrado mixto.</span>';
                }
            }
            
            validacionCreditos.innerHTML = mensaje;
            btnGuardar.disabled = !esValido;
        }

        function manejarTipoPostgrado() {
            const tipo = tipoPostgrado.value;
            
            if (tipo === 'tradicional') {
                creditosObligatorios.value = creditosTotales.value;
                creditosElectivos.value = 0;
                creditosObligatorios.readOnly = true;
                creditosElectivos.readOnly = true;
            } else if (tipo === 'electivo') {
                creditosObligatorios.value = 0;
                creditosElectivos.value = creditosTotales.value;
                creditosObligatorios.readOnly = true;
                creditosElectivos.readOnly = true;
            } else { // mixto
                creditosObligatorios.readOnly = false;
                creditosElectivos.readOnly = false;
            }
            
            actualizarResumen();
        }

        // Event listeners
        tipoPostgrado.addEventListener('change', manejarTipoPostgrado);
        creditosTotales.addEventListener('input', function() {
            if (tipoPostgrado.value === 'tradicional') {
                creditosObligatorios.value = this.value;
            } else if (tipoPostgrado.value === 'electivo') {
                creditosElectivos.value = this.value;
            }
            actualizarResumen();
        });
        
        creditosObligatorios.addEventListener('input', actualizarResumen);
        creditosElectivos.addEventListener('input', actualizarResumen);

        // Inicializar
        manejarTipoPostgrado();
        
        // Validacion del formulario antes de enviar
        document.querySelector('form').addEventListener('submit', function(e) {
            const tipo = tipoPostgrado.value;
            const totales = parseInt(creditosTotales.value);
            const obligatorios = parseInt(creditosObligatorios.value);
            const electivos = parseInt(creditosElectivos.value);
            
            let esValido = true;
            let mensaje = '';
            
            if (tipo === 'tradicional' && obligatorios !== totales) {
                esValido = false;
                mensaje = 'Para postgrados tradicionales, los creditos obligatorios deben ser iguales a los creditos totales.';
            } else if (tipo === 'electivo' && electivos !== totales) {
                esValido = false;
                mensaje = 'Para postgrados electivos, los creditos electivos deben ser iguales a los creditos totales.';
            } else if (tipo === 'mixto' && (obligatorios + electivos) !== totales) {
                esValido = false;
                mensaje = 'Para postgrados mixtos, la suma de creditos obligatorios y electivos debe ser igual a los creditos totales.';
            }
            
            if (!esValido) {
                e.preventDefault();
                alert('Error en la configuracion de creditos: ' + mensaje);
            }
        });
    });
    </script>
</body>
</html>
<?php
// 8. Cerrar conexion y finalizar
$conn->close();
?>