<?php
// Configuración detallada de errores
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Ruta específica para logs
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/constancia_errors.log');

// Aumentar límites para generación de PDFs
set_time_limit(300);
ini_set('memory_limit', '256M');
ini_set('max_execution_time', 300);

// Verificar sesión
session_start();
if (empty($_SESSION['username']) || $_SESSION['rol'] !== 'administrador') {
    header("Location: ../login.php");
    exit();
}

// =================================================================
// CONFIGURACIÓN ESPECÍFICA PARA TU INSTALACIÓN
// =================================================================

// Cargar primero el autoloader de Composer para resolver dependencias
require_once __DIR__ . '/../vendor/autoload.php';

// Cargar conexión a base de datos
require_once __DIR__ . '/../includes/db.php';

if (!$conn) {
    die("ERROR: No se pudo conectar a la base de datos.");
}

// Fuerza UTF-8 en la conexión
$conn->set_charset("utf8mb4");

// =================================================================
// VALIDACIÓN DE PARÁMETROS
// =================================================================

$required_params = ['estudiante_id', 'postgrado_id', 'fecha_constancia'];
foreach ($required_params as $param) {
    if (!isset($_GET[$param])) {
        error_log("Error: Parámetro faltante $param");
        die("ERROR: Faltan datos requeridos para generar la constancia.");
    }
}

$estudiante_id = filter_var($_GET['estudiante_id'], FILTER_VALIDATE_INT);
$postgrado_id = filter_var($_GET['postgrado_id'], FILTER_VALIDATE_INT);
$fecha_constancia = $_GET['fecha_constancia'];

if (!$estudiante_id || !$postgrado_id || !DateTime::createFromFormat('Y-m-d', $fecha_constancia)) {
    error_log("Error: Parámetros inválidos");
    die("ERROR: Datos proporcionados no válidos.");
}

// =================================================================
// CONSULTAS A LA BASE DE DATOS
// =================================================================

try {
    // Consulta estudiante - ACTUALIZADA CON NUEVOS CAMPOS DE DOCUMENTOS
    $sql_estudiante = "SELECT e.*, p.nombre AS postgrado_nombre, 
                      ep.fecha_graduacion, pr.nombre AS promocion,
                      e.tipo_documento, e.cedula_venezolana, e.cedula_extranjera, e.pasaporte
                      FROM estudiantes e
                      JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                      JOIN postgrados p ON ep.postgrado_id = p.id
                      LEFT JOIN promociones pr ON ep.promocion_id = pr.id
                      WHERE e.id = ? AND p.id = ? 
                      AND (ep.estado = 'graduado' OR ep.estado = 'activo')
                      LIMIT 1";

    $stmt_estudiante = $conn->prepare($sql_estudiante);
    if (!$stmt_estudiante) {
        throw new Exception("Error preparando consulta: ".$conn->error);
    }

    $stmt_estudiante->bind_param("ii", $estudiante_id, $postgrado_id);
    if (!$stmt_estudiante->execute()) {
        throw new Exception("Error ejecutando consulta: ".$stmt_estudiante->error);
    }

    $result_estudiante = $stmt_estudiante->get_result();
    $detalles_estudiante = $result_estudiante->fetch_assoc();
    $stmt_estudiante->close();

    if (empty($detalles_estudiante)) {
        throw new Exception("Estudiante no encontrado o no está activo/graduado");
    }

    // Función para obtener el número de documento según el tipo
    function obtenerNumeroDocumento($estudiante) {
        switch ($estudiante['tipo_documento']) {
            case 'V':
                return $estudiante['cedula_venezolana'] ?? $estudiante['cedula'];
            case 'E':
                return $estudiante['cedula_extranjera'] ?? $estudiante['cedula'];
            case 'P':
                return $estudiante['pasaporte'] ?? $estudiante['cedula'];
            default:
                return $estudiante['cedula'];
        }
    }

    // Función para obtener la etiqueta del tipo de documento
    function obtenerTipoDocumentoLabel($tipo) {
        switch ($tipo) {
            case 'V': return 'DE LA CÉDULA DE IDENTIDAD Nº V-';
            case 'E': return 'DE LA CÉDULA DE IDENTIDAD Nº E-';
            case 'P': return 'DEL PASAPORTE Nº ';
            default: return 'Documento';
        }
    }

    // Obtener número de documento correcto
    $numero_documento = obtenerNumeroDocumento($detalles_estudiante);
    $tipo_documento_label = obtenerTipoDocumentoLabel($detalles_estudiante['tipo_documento']);

    // Procesamiento de datos
    $nombre_postgrado = preg_replace('/\s*\-.*$/', '', $detalles_estudiante['postgrado_nombre']);

    // Consulta historial académico - CORREGIDA: usar 'credito' en lugar de 'creditos'
    $sql_historial = "SELECT d.codigo AS asignatura, d.nombre AS denominacion, 
                     d.tipo, i.nota_definitiva, `al`.`nombre` AS `año_lectivo`,
                     d.credito AS uc
                     FROM inscripciones i
                     INNER JOIN denominaciones d ON i.denominacion_id = d.id
                     LEFT JOIN `años_lectivos` `al` ON i.`año_lectivo_id` = `al`.`id`
                     WHERE i.estudiante_id = ? AND d.postgrado_id = ?
                     AND i.nota_definitiva IS NOT NULL
                     ORDER BY d.periodo, d.codigo";

    $stmt_historial = $conn->prepare($sql_historial);
    if (!$stmt_historial) {
        throw new Exception("Error preparando consulta: ".$conn->error);
    }

    $stmt_historial->bind_param("ii", $estudiante_id, $postgrado_id);
    if (!$stmt_historial->execute()) {
        throw new Exception("Error ejecutando consulta: ".$stmt_historial->error);
    }

    $result_historial = $stmt_historial->get_result();
    $historial_academico = $result_historial->fetch_all(MYSQLI_ASSOC);
    $stmt_historial->close();

    // Obtener firma
    $sql_firma = "SELECT nombre, cargo, rev_por FROM firmas_constancia_secretaria LIMIT 1";
    $result_firma = $conn->query($sql_firma);
    $firma = $result_firma && $result_firma->num_rows > 0 
             ? $result_firma->fetch_assoc() 
             : [
                 'nombre' => 'SECRETARIO(A)',
                 'cargo' => 'SECRETARIA DE LA UNIVERSIDAD CENTRAL DE VENEZUELA',
                 'rev_por' => 'ER REV POR:'
               ];

    // =================================================================
    // CONFIGURACIÓN Y GENERACIÓN DEL PDF
    // =================================================================

    // Crear directorio temporal si no existe
    $tempDir = __DIR__ . '/tmp';
    if (!file_exists($tempDir)) {
        if (!mkdir($tempDir, 0755, true)) {
            error_log("Error: No se pudo crear directorio temporal");
        }
    }

    // Configuración mPDF - TAMAÑO CARTA Y MÁRGENES AJUSTADOS
    $mpdf = new \Mpdf\Mpdf([
        'mode' => 'utf-8',
        'format' => 'Letter', // Tamaño Carta
        'margin_left' => 20,
        'margin_right' => 20,
        'margin_top' => 45, // 4.5 cm
        'margin_bottom' => 20, // Margen inferior reducido
        'margin_header' => 10,
        'margin_footer' => 10,
        'default_font_size' => 13, // Tamaño de letra aumentado
        'default_font' => 'arial',
        'tempDir' => $tempDir,
        'setAutoTopMargin' => 'stretch',
        'setAutoBottomMargin' => 'stretch'
    ]);

    // HTML de la constancia - ACTUALIZADO SEGÚN PDF MODELO
    $html = '<!DOCTYPE html>
    <html lang="es">
    <head>
        <meta charset="UTF-8">
        <title>Constancia Secretaría</title>
        <style>
            body { font-family: Arial, sans-serif; margin: 0; padding: 0; font-size: 13px; line-height: 1.3; }
            .texto-certificacion { margin: 0 0 15px 0; text-align: justify; line-height: 1.4; font-size: 13px; }
            table.historial { width: 100%; border-collapse: collapse; margin: 10px 0; }
            .historial th { padding: 6px; text-align: left; font-size: 12px; font-weight: bold; }
            .historial td { padding: 6px; font-size: 12px; vertical-align: top; }
            .centrado { text-align: center !important; }
            .nota-linea { white-space: nowrap; }
            .text-uppercase { text-transform: uppercase; }
            .firma { margin-top: 20px; } /* Reducido el margen */
            .firma-nombre { text-align: center; font-size: 12px; line-height: 1.0; margin: 0; padding: 0; }
            .firma-cargo { text-align: center; font-size: 12px; line-height: 1.5; margin: 0; padding: 0; }
            .page-break { page-break-after: always; }
            .firma-container { margin-left: 55%; width: 45%; }
            .contenido-final { margin-top: 12px; } /* Reducido el margen */
        </style>
    </head>
    <body>
        <div style="padding-top: 10px;">
            <div class="texto-certificacion">
                <p class="text-uppercase">LA SECRETARIA DE LA UNIVERSIDAD CENTRAL DE VENEZUELA, QUIEN SUSCRIBE, CERTIFICA: QUE EN EL EXPEDIENTE DE GRADO DE ESPECIALISTA EN '.htmlspecialchars($nombre_postgrado).', CORRESPONDIENTE AL CIUDADANO(A) '.htmlspecialchars($detalles_estudiante['apellidos'].' '.$detalles_estudiante['nombres']).' TITULAR '.htmlspecialchars($tipo_documento_label).htmlspecialchars($numero_documento).', SE ENCUENTRA UNA FICHA DOCENTE EN LA CUAL CONSTA QUE CURSO EN ÉSTA INSTITUCIÓN LAS ASIGNATURAS QUE A CONTINUACIÓN SE ESPECIFICAN CON LAS SIGUIENTES CALIFICACIONES:</p>
            </div>

            <table class="historial">
                <thead>
                    <tr>
                        <th style="width: 12%; text-align: left;">CÓDIGO</th>
                        <th style="width: 40%; text-align: left;">ASIGNATURA</th>
                        <th style="width: 8%; text-align: center;">UC</th>
                        <th style="width: 15%; text-align: left;">PERIODO</th>
                        <th style="width: 15%; text-align: left;">NOTA</th>
                        <th style="width: 5%; text-align: center;"></th>
                        <th style="width: 5%; text-align: center;">TIPO</th>
                    </tr>
                </thead>
                <tbody>';

    $total_materias = count($historial_academico);
    $materias_por_pagina = 23; // 23 materias por página en todas las páginas
    
    foreach ($historial_academico as $index => $historial) {
        // Procesar nota según tipo
        $nota_numero = '';
        $nota_texto = '';
        $tipo_columna = 'F'; // Siempre F según la corrección
        
        if ($historial['tipo'] == 'puntaje' && is_numeric($historial['nota_definitiva'])) {
            $nota_numero = intval($historial['nota_definitiva']);
            $nota_texto = numeroATexto($nota_numero);
        } elseif ($historial['tipo'] == 'Aprobado_Reprobado') {
            $nota_texto = (strtoupper($historial['nota_definitiva']) == 'APROBADO') ? 'APROBADO' : 'REPROBADO';
            $nota_numero = (strtoupper($historial['nota_definitiva']) == 'APROBADO') ? 'A' : 'R'; // A para Aprobado, R para Reprobado
        }
        
        $html .= '<tr>
                    <td style="text-align: left;">'.htmlspecialchars($historial['asignatura']).'</td>
                    <td style="text-align: left;">'.htmlspecialchars($historial['denominacion']).'</td>
                    <td style="text-align: center;">'.htmlspecialchars($historial['uc'] ?? '0').'</td>
                    <td style="text-align: left;">'.htmlspecialchars($historial['año_lectivo']).'</td>
                    <td style="text-align: left;" class="nota-linea">'.$nota_texto.'</td>
                    <td style="text-align: center;">'.$nota_numero.'</td>
                    <td style="text-align: center;">'.$tipo_columna.'</td>
                </tr>';
        
        // Insertar salto de página después de 23 materias, pero no en la última
        if (($index + 1) % $materias_por_pagina == 0 && ($index + 1) < $total_materias) {
            $html .= '</tbody>
            </table>
            <div class="page-break"></div>
            <table class="historial">
                <thead>
                    <tr>
                        <th style="width: 12%; text-align: left;">CÓDIGO</th>
                        <th style="width: 40%; text-align: left;">ASIGNATURA</th>
                        <th style="width: 8%; text-align: center;">UC</th>
                        <th style="width: 15%; text-align: left;">PERIODO</th>
                        <th style="width: 15%; text-align: left;">NOTA</th>
                        <th style="width: 5%; text-align: center;"></th>
                        <th style="width: 5%; text-align: center;">TIPO</th>
                    </tr>
                </thead>
                <tbody>';
        }
    }

    $html .= '</tbody>
            </table>

            <div class="contenido-final">
                <div class="texto-certificacion">
                    <p class="text-uppercase">IGUALMENTE CERTIFICA: QUE LA ESCALA DE CALIFICACIONES DE ÉSTA INSTITUCIÓN, ES COMO SIGUE: DE CERO (0) A VEINTE (20) PUNTOS. SE REQUIERE UNA NOTA MINIMA APROBATORIA DE DIEZ (10) PUNTOS. SE EXPIDE LA PRESENTE CERTIFICACION A SOLICITUD DE PARTE INTERESADA EN CARACAS, A LOS '.fechaATexto($fecha_constancia)['texto'].'. AÑOS '.fechaATexto($fecha_constancia)['anos'].'.</p>
                    <p>'.htmlspecialchars($firma['rev_por']).'</p>
                </div>
                <div class="firma">
                    <div class="firma-container">
                        <p class="firma-nombre">'.htmlspecialchars($firma['nombre']).'</p>
                        <p class="firma-cargo">'.htmlspecialchars($firma['cargo']).'</p>
                    </div>
                </div>
            </div>
        </div>
    </body>
    </html>';

    $mpdf->WriteHTML($html);
    
    $filename = 'Constancia_Secretaria_'.$numero_documento.'_'.date('Ymd').'.pdf';
    
    // Configurar headers para evitar caché
    header('Content-Type: application/pdf');
    header('Content-Disposition: inline; filename="'.$filename.'"');
    header('Cache-Control: private, max-age=0, must-revalidate');
    header('Pragma: public');
    header('Expires: 0');
    
    $mpdf->Output($filename, \Mpdf\Output\Destination::INLINE);

} catch (Exception $e) {
    error_log("Error al generar constancia: ".$e->getMessage());
    die("ERROR: Ocurrió un problema al generar la constancia. Por favor intente nuevamente o contacte al administrador. Error: ".$e->getMessage());
} finally {
    if (isset($conn) && $conn) {
        $conn->close();
    }
}

// =================================================================
// FUNCIONES AUXILIARES
// =================================================================

function numeroATexto($numero) {
    $unidades = ['', 'UNO', 'DOS', 'TRES', 'CUATRO', 'CINCO', 'SEIS', 'SIETE', 'OCHO', 'NUEVE'];
    $decenas = ['', 'DIEZ', 'VEINTE', 'TREINTA', 'CUARENTA', 'CINCUENTA', 'SESENTA', 'SETENTA', 'OCHENTA', 'NOVENTA'];
    $especiales = ['ONCE', 'DOCE', 'TRECE', 'CATORCE', 'QUINCE', 'DIECISEIS', 'DIECISIETE', 'DIECIOCHO', 'DIECINUEVE'];
    
    if ($numero == 0) return 'CERO';
    if ($numero == 10) return 'DIEZ';
    if ($numero == 20) return 'VEINTE';
    if ($numero > 10 && $numero < 20) return $especiales[$numero - 11];
    
    $texto = '';
    $decena = floor($numero / 10);
    $unidad = $numero % 10;
    
    if ($decena > 0) $texto .= $decenas[$decena];
    if ($unidad > 0 && $decena > 0) $texto .= ' Y ';
    if ($unidad > 0) $texto .= $unidades[$unidad];
    
    return $texto;
}

function numeroATextoAno($numero) {
    if ($numero == 2000) return 'DOS MIL';
    if ($numero == 2025) return 'DOS MIL VEINTICINCO';
    if ($numero == 2026) return 'DOS MIL VEINTISEIS';
    
    $millar = floor($numero / 1000);
    $centena = floor(($numero % 1000) / 100);
    $decena = floor(($numero % 100) / 10);
    $unidad = $numero % 10;
    
    $texto = '';
    
    if ($millar > 0) $texto .= numeroATexto($millar) . ' MIL';
    
    if ($texto && ($centena > 0 || $decena > 0 || $unidad > 0)) {
        $texto .= ' ';
    }
    
    if ($centena > 0) {
        if ($centena == 1 && $decena == 0 && $unidad == 0) {
            $texto .= 'CIEN';
        } else {
            $texto .= numeroATexto($centena) . 'CIENTOS';
        }
    }
    
    if ($texto && ($decena > 0 || $unidad > 0)) {
        $texto .= ' ';
    }
    
    if ($decena > 1) {
        $texto .= numeroATexto($decena * 10);
        if ($unidad > 0) {
            $texto .= ' Y ' . numeroATexto($unidad);
        }
    } elseif ($decena == 1) {
        if ($unidad > 0) {
            $texto .= numeroATexto(10 + $unidad);
        } else {
            $texto .= numeroATexto(10);
        }
    } elseif ($unidad > 0) {
        $texto .= numeroATexto($unidad);
    }
    
    return $texto;
}

function fechaATexto($fecha) {
    $dias = ['', 'UNO', 'DOS', 'TRES', 'CUATRO', 'CINCO', 'SEIS', 'SIETE', 'OCHO', 'NUEVE', 'DIEZ',
             'ONCE', 'DOCE', 'TRECE', 'CATORCE', 'QUINCE', 'DIECISEIS', 'DIECISIETE', 'DIECIOCHO', 'DIECINUEVE', 'VEINTE',
             'VEINTIUNO', 'VEINTIDOS', 'VEINTITRES', 'VEINTICUATRO', 'VEINTICINCO', 'VEINTISEIS', 'VEINTISIETE', 'VEINTIOCHO', 'VEINTINUEVE', 'TREINTA', 'TREINTA Y UNO'];
    
    $meses = ['', 'ENERO', 'FEBRERO', 'MARZO', 'ABRIL', 'MAYO', 'JUNIO', 'JULIO', 'AGOSTO', 'SEPTIEMBRE', 'OCTUBRE', 'NOVIEMBRE', 'DICIEMBRE'];
    
    $fechaObj = DateTime::createFromFormat('Y-m-d', $fecha);
    if (!$fechaObj) {
        $fechaObj = new DateTime();
    }
    
    $dia = $fechaObj->format('j');
    $mes = $fechaObj->format('n');
    $ano = $fechaObj->format('Y');
    $anoTexto = numeroATextoAno($ano);
    
    if ($ano == 2025) {
        $anosUcv = '215';
        $anosFacultad = '166';
    } else {
        $anosUcv = (215 + ($ano - 2025)) . '';
        $anosFacultad = (166 + ($ano - 2025)) . '';
    }
    
    return [
        'texto' => $dias[$dia] . ' DÍAS DEL MES DE ' . $meses[$mes] . ' DEL AÑO ' . $anoTexto,
        'anos' => $anosUcv . 'º Y ' . $anosFacultad . 'º'
    ];
}
?>