<?php
session_start();

// Configurar encoding UTF-8
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');

if (!isset($_SESSION['username']) || $_SESSION['rol'] != 'administrador') {
    header("Location: ../login.php");
    exit();
}

include '../includes/db.php';

// Configurar conexión para usar UTF-8
$conn->set_charset("utf8mb4");
$conn->query("SET NAMES 'utf8mb4'");
$conn->query("SET CHARACTER SET utf8mb4");
$conn->query("SET COLLATION_CONNECTION = 'utf8mb4_unicode_ci'");

// Verificar sesión y rol de administrador
if (!isset($_SESSION['username']) || $_SESSION['rol'] != 'administrador') {
    header("Location: ../login.php");
    exit();
}

// Procesar el formulario de cambio de postgrado
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['cambiar_postgrado'])) {
        $estudiante_id = intval($_POST['estudiante_id']);
        $postgrado_actual_id = intval($_POST['postgrado_actual_id']);
        $nuevo_postgrado_id = intval($_POST['nuevo_postgrado_id']);
        $motivo = trim($_POST['motivo']);

        // Verificar que el estudiante no esté ya en el nuevo postgrado
        if ($postgrado_actual_id == $nuevo_postgrado_id) {
            $_SESSION['error'] = "El estudiante ya está inscrito en este postgrado";
            header("Location: cambio_postgrado.php");
            exit();
        }

        // Iniciar transacción
        $conn->begin_transaction();

        try {
            // 1. Actualizar el estado del postgrado actual a 'cambio'
            $sql_actualizar = "UPDATE estudiante_postgrado 
                              SET estado = 'cambio', fecha_cambio = NOW()
                              WHERE estudiante_id = ? AND postgrado_id = ?";
            $stmt_actualizar = $conn->prepare($sql_actualizar);
            $stmt_actualizar->bind_param("ii", $estudiante_id, $postgrado_actual_id);
            $stmt_actualizar->execute();

            // 2. Crear nueva relación con el nuevo postgrado
            $sql_nuevo = "INSERT INTO estudiante_postgrado 
                         (estudiante_id, postgrado_id, fecha_inscripcion, estado)
                         VALUES (?, ?, NOW(), 'activo')";
            $stmt_nuevo = $conn->prepare($sql_nuevo);
            $stmt_nuevo->bind_param("ii", $estudiante_id, $nuevo_postgrado_id);
            $stmt_nuevo->execute();

            // 3. Registrar el cambio en la tabla de cambios_postgrado
            $sql_registro = "INSERT INTO cambios_postgrado 
                            (estudiante_id, postgrado_anterior, postgrado_nuevo, motivo, fecha, administrador_id)
                            VALUES (?, ?, ?, ?, NOW(), ?)";
            $stmt_registro = $conn->prepare($sql_registro);
            $stmt_registro->bind_param("iiisi", $estudiante_id, $postgrado_actual_id, $nuevo_postgrado_id, $motivo, $_SESSION['user_id']);
            $stmt_registro->execute();

            // Confirmar transacción
            $conn->commit();

            $_SESSION['success'] = "Postgrado cambiado exitosamente";
            header("Location: cambio_postgrado.php");
            exit();
        } catch (Exception $e) {
            // Revertir transacción en caso de error
            $conn->rollback();
            $_SESSION['error'] = "Error al cambiar el postgrado: " . $e->getMessage();
            header("Location: cambio_postgrado.php");
            exit();
        }
    }
    elseif (isset($_POST['buscar_cedula'])) {
        $cedula = trim($_POST['cedula_buscar']);
        $_SESSION['cedula_buscar'] = $cedula;
        header("Location: cambio_postgrado.php");
        exit();
    }
}

// Obtener parámetro de búsqueda por cédula
$cedula_buscar = isset($_SESSION['cedula_buscar']) ? $_SESSION['cedula_buscar'] : '';
unset($_SESSION['cedula_buscar']);

// CONSULTA ACTUALIZADA - Buscar en todos los campos de documentos
$sql_estudiantes = "SELECT e.id, e.cedula, e.nombres, e.apellidos, 
                   e.tipo_documento, e.cedula_venezolana, e.cedula_extranjera, e.pasaporte,
                   p.id as postgrado_id, p.nombre as postgrado_nombre
                   FROM estudiantes e
                   JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                   JOIN postgrados p ON ep.postgrado_id = p.id
                   WHERE ep.estado = 'activo'";

if (!empty($cedula_buscar)) {
    $sql_estudiantes .= " AND (e.cedula LIKE '%" . $conn->real_escape_string($cedula_buscar) . "%'
                          OR e.cedula_venezolana LIKE '%" . $conn->real_escape_string($cedula_buscar) . "%'
                          OR e.cedula_extranjera LIKE '%" . $conn->real_escape_string($cedula_buscar) . "%'
                          OR e.pasaporte LIKE '%" . $conn->real_escape_string($cedula_buscar) . "%')";
}

$sql_estudiantes .= " ORDER BY e.apellidos, e.nombres";
$result_estudiantes = $conn->query($sql_estudiantes);

// Obtener lista de todos los postgrados disponibles
$sql_postgrados = "SELECT id, nombre FROM postgrados ORDER BY nombre";
$result_postgrados = $conn->query($sql_postgrados);

$page_title = "Cambio de Postgrado";
include '../includes/header.php';
?>

<div class="container">
    <!-- Botón para volver al panel -->
    <div class="mb-4">
        <a href="index.php" class="btn btn-secondary">
            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-arrow-left" viewBox="0 0 16 16">
                <path fill-rule="evenodd" d="M15 8a.5.5 0 0 0-.5-.5H2.707l3.147-3.146a.5.5 0 1 0-.708-.708l-4 4a.5.5 0 0 0 0 .708l4 4a.5.5 0 0 0 .708-.708L2.707 8.5H14.5A.5.5 0 0 0 15 8z"/>
            </svg>
            Volver al Panel
        </a>
    </div>

    <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success"><?= $_SESSION['success'] ?></div>
        <?php unset($_SESSION['success']); ?>
    <?php endif; ?>
    
    <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger"><?= $_SESSION['error'] ?></div>
        <?php unset($_SESSION['error']); ?>
    <?php endif; ?>

    <!-- Formulario de búsqueda por documento -->
    <div class="card mb-4">
        <div class="card-header bg-info text-white">
            <h3 class="mb-0">Buscar Estudiante por Documento</h3>
        </div>
        <div class="card-body">
            <form method="post" class="row g-3">
                <div class="col-md-8">
                    <input type="text" class="form-control" name="cedula_buscar" placeholder="Ingrese cédula venezolana, extranjera o pasaporte" value="<?= htmlspecialchars($cedula_buscar) ?>">
                    <div class="form-text">Puede buscar por cualquier tipo de documento</div>
                </div>
                <div class="col-md-4">
                    <button type="submit" name="buscar_cedula" class="btn btn-primary w-100">Buscar</button>
                </div>
                <?php if (!empty($cedula_buscar)): ?>
                    <div class="col-12">
                        <a href="cambio_postgrado.php" class="btn btn-outline-secondary">Limpiar búsqueda</a>
                    </div>
                <?php endif; ?>
            </form>
        </div>
    </div>

    <?php if (!empty($cedula_buscar)): ?>
        <div class="alert alert-warning mb-4">
            Mostrando resultados para documento: <strong><?= htmlspecialchars($cedula_buscar) ?></strong>
        </div>
    <?php endif; ?>

    <div class="card mb-4">
        <div class="card-header bg-primary text-white">
            <h3 class="mb-0">Cambiar Postgrado a Estudiante</h3>
        </div>
        <div class="card-body">
            <form method="post">
                <div class="row g-3">
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="estudiante_id" class="form-label">Seleccionar Estudiante:</label>
                            <select class="form-select" id="estudiante_id" name="estudiante_id" required>
                                <option value="">-- Seleccione un estudiante --</option>
                                <?php if ($result_estudiantes->num_rows > 0): ?>
                                    <?php while ($estudiante = $result_estudiantes->fetch_assoc()): ?>
                                        <?php
                                        // Determinar qué documento mostrar
                                        $documento_mostrar = '';
                                        switch($estudiante['tipo_documento']) {
                                            case 'V':
                                                $documento_mostrar = 'V-' . $estudiante['cedula_venezolana'];
                                                break;
                                            case 'E':
                                                $documento_mostrar = 'E-' . $estudiante['cedula_extranjera'];
                                                break;
                                            case 'P':
                                                $documento_mostrar = 'P-' . $estudiante['pasaporte'];
                                                break;
                                            default:
                                                $documento_mostrar = $estudiante['cedula'];
                                        }
                                        ?>
                                        <option value="<?= $estudiante['id'] ?>" data-postgrado="<?= $estudiante['postgrado_id'] ?>">
                                            <?= htmlspecialchars($estudiante['apellidos'] . ', ' . $estudiante['nombres'] . ' (' . $documento_mostrar . ') - ' . $estudiante['postgrado_nombre']) ?>
                                        </option>
                                    <?php endwhile; ?>
                                <?php else: ?>
                                    <option value="">No se encontraron estudiantes</option>
                                <?php endif; ?>
                            </select>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="nuevo_postgrado_id" class="form-label">Nuevo Postgrado:</label>
                            <select class="form-select" id="nuevo_postgrado_id" name="nuevo_postgrado_id" required>
                                <option value="">-- Seleccione un postgrado --</option>
                                <?php while ($postgrado = $result_postgrados->fetch_assoc()): ?>
                                    <option value="<?= $postgrado['id'] ?>"><?= htmlspecialchars($postgrado['nombre']) ?></option>
                                <?php endwhile; ?>
                            </select>
                        </div>
                    </div>
                </div>

                <div class="mb-3">
                    <label for="motivo" class="form-label">Motivo del cambio:</label>
                    <textarea class="form-control" id="motivo" name="motivo" rows="3" required></textarea>
                </div>

                <input type="hidden" id="postgrado_actual_id" name="postgrado_actual_id" value="">

                <div class="d-grid gap-2 d-md-flex justify-content-md-end">
                    <button type="submit" name="cambiar_postgrado" class="btn btn-primary">Realizar Cambio</button>
                </div>
            </form>
        </div>
    </div>

    <div class="card">
        <div class="card-header bg-info text-white">
            <h3 class="mb-0">Historial de Cambios</h3>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-striped">
                    <thead>
                        <tr>
                            <th>Fecha</th>
                            <th>Estudiante</th>
                            <th>Documento</th>
                            <th>Postgrado Anterior</th>
                            <th>Postgrado Nuevo</th>
                            <th>Motivo</th>
                            <th>Administrador</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php
                        // CONSULTA ACTUALIZADA - Obtener historial de cambios con nuevos campos de documentos
                        $sql_historial = "SELECT c.fecha, e.nombres, e.apellidos, 
                                         e.tipo_documento, e.cedula_venezolana, e.cedula_extranjera, e.pasaporte,
                                         p1.nombre as postgrado_anterior, p2.nombre as postgrado_nuevo,
                                         c.motivo, u.username as administrador
                                         FROM cambios_postgrado c
                                         JOIN estudiantes e ON c.estudiante_id = e.id
                                         JOIN postgrados p1 ON c.postgrado_anterior = p1.id
                                         JOIN postgrados p2 ON c.postgrado_nuevo = p2.id
                                         JOIN usuarios u ON c.administrador_id = u.id
                                         ORDER BY c.fecha DESC
                                         LIMIT 10";
                        $result_historial = $conn->query($sql_historial);
                        
                        if ($result_historial->num_rows > 0):
                            while ($cambio = $result_historial->fetch_assoc()):
                                // Determinar qué documento mostrar
                                $documento_mostrar = '';
                                switch($cambio['tipo_documento']) {
                                    case 'V':
                                        $documento_mostrar = 'V-' . $cambio['cedula_venezolana'];
                                        break;
                                    case 'E':
                                        $documento_mostrar = 'E-' . $cambio['cedula_extranjera'];
                                        break;
                                    case 'P':
                                        $documento_mostrar = 'P-' . $cambio['pasaporte'];
                                        break;
                                    default:
                                        $documento_mostrar = 'N/A';
                                }
                        ?>
                            <tr>
                                <td><?= date('d/m/Y H:i', strtotime($cambio['fecha'])) ?></td>
                                <td><?= htmlspecialchars($cambio['apellidos'] . ', ' . $cambio['nombres']) ?></td>
                                <td><?= htmlspecialchars($documento_mostrar) ?></td>
                                <td><?= htmlspecialchars($cambio['postgrado_anterior']) ?></td>
                                <td><?= htmlspecialchars($cambio['postgrado_nuevo']) ?></td>
                                <td><?= htmlspecialchars($cambio['motivo']) ?></td>
                                <td><?= htmlspecialchars($cambio['administrador']) ?></td>
                            </tr>
                        <?php
                            endwhile;
                        else:
                        ?>
                            <tr>
                                <td colspan="7" class="text-center">No hay registros de cambios</td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
    // Actualizar el postgrado actual cuando se selecciona un estudiante
    document.getElementById('estudiante_id').addEventListener('change', function() {
        const selectedOption = this.options[this.selectedIndex];
        const postgradoActualId = selectedOption.getAttribute('data-postgrado');
        document.getElementById('postgrado_actual_id').value = postgradoActualId;
    });
</script>

<?php
include '../includes/footer.php';
$conn->close();
?>