<?php
session_start();

if (!isset($_SESSION['username'])) {
    header("Location: ../login.php");
    exit();
}

include '../includes/db.php';

// Configurar conexión para usar UTF-8
if ($conn) {
    $conn->set_charset("utf8mb4");
    $conn->query("SET NAMES 'utf8mb4'");
    $conn->query("SET CHARACTER SET utf8mb4");
    $conn->query("SET COLLATION_CONNECTION = 'utf8mb4_unicode_ci'");
}

// Obtener parámetros de búsqueda
$search = isset($_GET['search']) ? trim($_GET['search']) : '';
$promocion_id = isset($_GET['promocion_id']) ? intval($_GET['promocion_id']) : 0;
$año_lectivo_id = isset($_GET['año_lectivo_id']) ? intval($_GET['año_lectivo_id']) : 0;

// Obtener listas para filtros
$sql_promociones = "SELECT id, nombre FROM promociones ORDER BY nombre";
$promociones = $conn->query($sql_promociones);

// CORREGIDO: Usar comillas invertidas para el nombre de tabla con caracteres especiales
$sql_años_lectivos = "SELECT id, nombre FROM `años_lectivos` ORDER BY nombre";
$años_lectivos = $conn->query($sql_años_lectivos);

// Función para determinar el tipo de documento y obtener el valor
function obtenerDocumentoIdentidad($estudiante) {
    $tipo_documento = $estudiante['tipo_documento'] ?? 'V';
    $valor_documento = '';
    $etiqueta_documento = '';
    
    switch ($tipo_documento) {
        case 'V':
            $valor_documento = $estudiante['cedula_venezolana'] ?? $estudiante['cedula'];
            $etiqueta_documento = 'CÉDULA';
            break;
        case 'E':
            $valor_documento = $estudiante['cedula_extranjera'] ?? $estudiante['cedula'];
            $etiqueta_documento = 'CÉDULA';
            break;
        case 'P':
            $valor_documento = $estudiante['pasaporte'] ?? $estudiante['cedula'];
            $etiqueta_documento = 'PASAPORTE';
            break;
        default:
            $valor_documento = $estudiante['cedula'];
            $etiqueta_documento = 'CÉDULA';
    }
    
    return [
        'etiqueta' => $etiqueta_documento,
        'valor' => $valor_documento,
        'tipo' => $tipo_documento
    ];
}

// Consulta para obtener estudiantes con los nuevos campos de documentos
$sql_estudiantes = "SELECT 
                    e.id, e.nombres, e.apellidos, e.cedula,
                    e.tipo_documento, e.cedula_venezolana, e.cedula_extranjera, e.pasaporte
                   FROM estudiantes e
                   JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                   WHERE (e.cedula LIKE ? OR e.cedula_venezolana LIKE ? OR e.cedula_extranjera LIKE ? OR e.pasaporte LIKE ? OR e.nombres LIKE ? OR e.apellidos LIKE ? OR e.correo LIKE ?)";

$params = ["%$search%", "%$search%", "%$search%", "%$search%", "%$search%", "%$search%", "%$search%"];
$types = "sssssss";

if ($promocion_id > 0) {
    $sql_estudiantes .= " AND ep.promocion_id = ?";
    $params[] = $promocion_id;
    $types .= "i";
}

if ($año_lectivo_id > 0) {
    $sql_estudiantes .= " AND EXISTS (
        SELECT 1 FROM inscripciones i 
        WHERE i.estudiante_id = e.id 
        AND i.año_lectivo_id = ?
    )";
    $params[] = $año_lectivo_id;
    $types .= "i";
}

$sql_estudiantes .= " GROUP BY e.id ORDER BY e.apellidos";

$stmt_estudiantes = $conn->prepare($sql_estudiantes);
if (!$stmt_estudiantes) {
    die("Error en la preparación de la consulta: " . $conn->error);
}

$stmt_estudiantes->bind_param($types, ...$params);
$stmt_estudiantes->execute();
$estudiantes = $stmt_estudiantes->get_result()->fetch_all(MYSQLI_ASSOC);

// Para cada estudiante, obtener sus postgrados
foreach ($estudiantes as &$estudiante) {
    $sql_postgrados = "SELECT 
                        p.id AS postgrado_id, 
                        p.nombre AS postgrado_nombre,
                        pr.nombre AS promocion,
                        ep.estado AS estado_postgrado,
                        ep.fecha_graduacion
                       FROM estudiante_postgrado ep
                       JOIN postgrados p ON ep.postgrado_id = p.id
                       LEFT JOIN promociones pr ON ep.promocion_id = pr.id
                       WHERE ep.estudiante_id = ?
                       AND (ep.estado = 'activo' OR ep.estado = 'graduado')
                       ORDER BY ep.estado DESC, ep.fecha_inscripcion DESC";
    
    $stmt_postgrados = $conn->prepare($sql_postgrados);
    $stmt_postgrados->bind_param("i", $estudiante['id']);
    $stmt_postgrados->execute();
    $estudiante['postgrados'] = $stmt_postgrados->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt_postgrados->close();
    
    // Obtener información del documento de identidad
    $estudiante['documento_info'] = obtenerDocumentoIdentidad($estudiante);
}
unset($estudiante);

$page_title = "Buscar Estudiante - Constancia de Estudio";
include '../includes/header.php';
?>

<div class="container">
    <!-- Botón para volver al panel -->
    <div class="mb-4">
        <a href="index.php" class="btn btn-secondary">
            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-arrow-left" viewBox="0 0 16 16">
                <path fill-rule="evenodd" d="M15 8a.5.5 0 0 0-.5-.5H2.707l3.147-3.146a.5.5 0 1 0-.708-.708l-4 4a.5.5 0 0 0 0 .708l4 4a.5.5 0 0 0 .708-.708L2.707 8.5H14.5A.5.5 0 0 0 15 8z"/>
            </svg>
            Volver al Panel
        </a>
    </div>

    <div class="card">
        <div class="card-header bg-primary text-white">
            <h3 class="mb-0">Filtros de Búsqueda - Constancia de Estudio</h3>
        </div>
        <div class="card-body">
            <form method="get" class="row g-3 search-form">
                <div class="col-md-4">
                    <label for="search" class="form-label">Buscar Estudiante:</label>
                    <input type="text" class="form-control" name="search" id="search" 
                           placeholder="Cédula, Pasaporte, Nombre, Apellido o Correo" 
                           value="<?= htmlspecialchars($search) ?>">
                </div>
                
                <div class="col-md-3">
                    <label for="promocion_id" class="form-label">Promoción:</label>
                    <select class="form-select" id="promocion_id" name="promocion_id">
                        <option value="0">Todas las promociones</option>
                        <?php while ($promocion = $promociones->fetch_assoc()): ?>
                            <option value="<?= $promocion['id'] ?>" 
                                <?= $promocion_id == $promocion['id'] ? 'selected' : '' ?>>
                                <?= htmlspecialchars($promocion['nombre']) ?>
                            </option>
                        <?php endwhile; ?>
                    </select>
                </div>
                
                <div class="col-md-3">
                    <label for="año_lectivo_id" class="form-label">Año Lectivo:</label>
                    <select class="form-select" id="año_lectivo_id" name="año_lectivo_id">
                        <option value="0">Todos los años</option>
                        <?php while ($año = $años_lectivos->fetch_assoc()): ?>
                            <option value="<?= $año['id'] ?>" 
                                <?= $año_lectivo_id == $año['id'] ? 'selected' : '' ?>>
                                <?= htmlspecialchars($año['nombre']) ?>
                            </option>
                        <?php endwhile; ?>
                        </select>
                </div>
                
                <div class="col-md-2 d-flex align-items-end">
                    <button type="submit" class="btn btn-primary w-100">Buscar</button>
                </div>
            </form>
        </div>
    </div>

    <!-- Lista de estudiantes -->
    <?php if (!empty($estudiantes)): ?>
        <div class="card mt-4">
            <div class="card-header bg-info text-white">
                <h3 class="mb-0">Resultados de la Búsqueda</h3>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-striped table-hover">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Nombre</th>
                                <th>Apellidos</th>
                                <th>Documento</th>
                                <th>Tipo</th>
                                <th>Postgrados</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($estudiantes as $estudiante): ?>
                                <tr>
                                    <td><?= $estudiante['id'] ?></td>
                                    <td><?= htmlspecialchars($estudiante['nombres']) ?></td>
                                    <td><?= htmlspecialchars($estudiante['apellidos']) ?></td>
                                    <td><?= htmlspecialchars($estudiante['documento_info']['valor']) ?></td>
                                    <td>
                                        <?php 
                                        $tipo_doc = $estudiante['documento_info']['tipo'];
                                        $badge_class = '';
                                        switch($tipo_doc) {
                                            case 'V': 
                                                $tipo_text = 'Venezolana';
                                                $badge_class = 'bg-success';
                                                break;
                                            case 'E': 
                                                $tipo_text = 'Extranjera';
                                                $badge_class = 'bg-warning text-dark';
                                                break;
                                            case 'P': 
                                                $tipo_text = 'Pasaporte';
                                                $badge_class = 'bg-info';
                                                break;
                                            default: 
                                                $tipo_text = 'Desconocido';
                                                $badge_class = 'bg-secondary';
                                        }
                                        ?>
                                        <span class="badge <?= $badge_class ?>"><?= $tipo_text ?></span>
                                    </td>
                                    <td>
                                        <?php if (!empty($estudiante['postgrados'])): ?>
                                            <?php foreach ($estudiante['postgrados'] as $postgrado): ?>
                                                <div class="postgrado-item">
                                                    <strong><?= htmlspecialchars($postgrado['postgrado_nombre']) ?></strong><br>
                                                    Promoción: <?= htmlspecialchars($postgrado['promocion'] ?? 'N/A') ?><br>
                                                    Estado: <?= htmlspecialchars($postgrado['estado_postgrado']) ?>
                                                    <?php if ($postgrado['estado_postgrado'] == 'graduado' && !empty($postgrado['fecha_graduacion'])): ?>
                                                        <br>Graduación: <?= htmlspecialchars($postgrado['fecha_graduacion']) ?>
                                                    <?php endif; ?>
                                                </div>
                                            <?php endforeach; ?>
                                        <?php else: ?>
                                            <span class="no-postgrado">No asignado</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php if (!empty($estudiante['postgrados'])): ?>
                                            <div class="d-flex flex-column">
                                                <?php foreach ($estudiante['postgrados'] as $postgrado): ?>
                                                    <form action="constancia_estudiante_notas.php" method="get" class="mb-2" target="_blank">
                                                        <input type="hidden" name="estudiante_id" value="<?= $estudiante['id'] ?>">
                                                        <input type="hidden" name="postgrado_id" value="<?= $postgrado['postgrado_id'] ?>">
                                                        <div class="input-group">
                                                            <input type="date" name="fecha_constancia" class="form-control form-control-sm" required>
                                                            <button type="submit" class="btn btn-sm btn-primary" 
                                                                    title="Generar constancia para <?= htmlspecialchars($postgrado['postgrado_nombre']) ?>">
                                                                Constancia
                                                            </button>
                                                        </div>
                                                    </form>
                                                <?php endforeach; ?>
                                            </div>
                                        <?php else: ?>
                                            <span class="no-postgrado">No disponible</span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    <?php elseif (!empty($search) || $promocion_id > 0 || $año_lectivo_id > 0): ?>
        <div class="alert alert-info mt-4">
            No se encontraron estudiantes con los criterios de búsqueda especificados.
        </div>
    <?php endif; ?>
</div>

<script>
// Agregar funcionalidad para establecer fecha por defecto en los campos de fecha
document.addEventListener('DOMContentLoaded', function() {
    const today = new Date().toISOString().split('T')[0];
    const dateInputs = document.querySelectorAll('input[type="date"][name="fecha_constancia"]');
    
    dateInputs.forEach(input => {
        if (!input.value) {
            input.value = today;
        }
    });
});
</script>

<?php
include '../includes/footer.php';
$conn->close();
?>