<?php
session_start();
define('BASE_PATH', dirname(__FILE__));
include BASE_PATH . '/includes/db.php';
include BASE_PATH . '/includes/header.php';

$error_message = '';
$success_message = '';
$valid_token = false;
$user_id = null;

// Verificar token
if (isset($_GET['token'])) {
    $token = $_GET['token'];
    
    // Buscar token válido
    $sql = "SELECT user_id, expiration, used FROM password_reset_tokens 
            WHERE token = ? AND used = 0 AND expiration > NOW()";
    $stmt = $conn->prepare($sql);
    
    if ($stmt) {
        $stmt->bind_param('s', $token);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            $token_data = $result->fetch_assoc();
            $user_id = $token_data['user_id'];
            $valid_token = true;
        } else {
            $error_message = "El enlace de recuperación no es válido o ha expirado.";
        }
        $stmt->close();
    }
} else {
    $error_message = "No se proporcionó un token válido.";
}

// Procesar cambio de contraseña
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $valid_token) {
    $password = $_POST['password'];
    $password_confirm = $_POST['password_confirm'];
    
    if (empty($password) || empty($password_confirm)) {
        $error_message = "Por favor, completa ambos campos de contraseña.";
    } elseif ($password !== $password_confirm) {
        $error_message = "Las contraseñas no coinciden.";
    } elseif (strlen($password) < 8) {
        $error_message = "La contraseña debe tener al menos 8 caracteres.";
    } else {
        $conn->begin_transaction();
        try {
            // Actualizar contraseña
            $hashed_password = password_hash($password, PASSWORD_DEFAULT);
            $sql_update = "UPDATE usuarios SET password = ? WHERE id = ?";
            $stmt_update = $conn->prepare($sql_update);
            
            if ($stmt_update === false) {
                throw new Exception("Error al preparar consulta: " . $conn->error);
            }
            
            $stmt_update->bind_param('si', $hashed_password, $user_id);
            
            if (!$stmt_update->execute()) {
                throw new Exception("Error al actualizar contraseña: " . $stmt_update->error);
            }
            $stmt_update->close();
            
            // Marcar token como usado
            $sql_token = "UPDATE password_reset_tokens SET used = 1 WHERE token = ?";
            $stmt_token = $conn->prepare($sql_token);
            
            if ($stmt_token === false) {
                throw new Exception("Error al marcar token: " . $conn->error);
            }
            
            $stmt_token->bind_param('s', $token);
            $stmt_token->execute();
            $stmt_token->close();
            
            $conn->commit();
            $success_message = "¡Contraseña actualizada exitosamente! Ahora puedes iniciar sesión con tu nueva contraseña.";
            $valid_token = false; // Ya no es válido después del uso
            
        } catch (Exception $e) {
            $conn->rollback();
            $error_message = "Error al actualizar la contraseña: " . $e->getMessage();
        }
    }
}
?>

<div class="container mt-5">
    <div class="row justify-content-center">
        <div class="col-md-6">
            <div class="card">
                <div class="card-header">
                    <h4 class="mb-0">Restablecer Contraseña</h4>
                </div>
                <div class="card-body">
                    <?php if ($error_message): ?>
                        <div class="alert alert-danger"><?php echo $error_message; ?></div>
                    <?php endif; ?>
                    
                    <?php if ($success_message): ?>
                        <div class="alert alert-success"><?php echo $success_message; ?></div>
                        <div class="text-center">
                            <a href="login.php" class="btn btn-primary">Ir al Login</a>
                        </div>
                    <?php elseif ($valid_token): ?>
                        <p>Ingresa tu nueva contraseña.</p>
                        <form method="POST" action="">
                            <div class="mb-3">
                                <label for="password" class="form-label">Nueva Contraseña *</label>
                                <input type="password" class="form-control" id="password" name="password" required minlength="8">
                                <div class="form-text">Mínimo 8 caracteres.</div>
                            </div>
                            <div class="mb-3">
                                <label for="password_confirm" class="form-label">Confirmar Nueva Contraseña *</label>
                                <input type="password" class="form-control" id="password_confirm" name="password_confirm" required minlength="8">
                            </div>
                            <button type="submit" class="btn btn-primary w-100">Actualizar Contraseña</button>
                        </form>
                    <?php else: ?>
                        <div class="text-center">
                            <a href="login.php" class="btn btn-primary">Volver al Login</a>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<?php include BASE_PATH . '/includes/footer.php'; ?>