<?php
session_start();

// Verificación de sesión y roles
if (empty($_SESSION['username']) || ($_SESSION['rol'] !== 'administrador' && $_SESSION['rol'] !== 'coordinador1' && $_SESSION['rol'] !== 'coordinador2')) {
    header("Location: ../login.php");
    exit();
}

// Configuración para UTF-8
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');

include '../includes/db.php';

// Configurar conexión para usar UTF-8
$conn->set_charset("utf8mb4");
$conn->query("SET NAMES 'utf8mb4'");
$conn->query("SET CHARACTER SET utf8mb4");
$conn->query("SET COLLATION_CONNECTION = 'utf8mb4_unicode_ci'");

// Obtener postgrados asignados al coordinador
$postgrados_coordinador = [];
if ($_SESSION['rol'] === 'coordinador1' || $_SESSION['rol'] === 'coordinador2') {
    $sql_postgrados = "SELECT p.id, p.nombre 
                       FROM coordinador_postgrado cp 
                       JOIN postgrados p ON cp.postgrado_id = p.id 
                       WHERE cp.coordinador_id = ?";
    $stmt = $conn->prepare($sql_postgrados);
    $stmt->bind_param("i", $_SESSION['user_id']);
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) {
        $postgrados_coordinador[] = $row;
    }
    $stmt->close();
} else {
    // Administrador ve todos los postgrados
    $sql_postgrados = "SELECT id, nombre FROM postgrados ORDER BY nombre";
    $result = $conn->query($sql_postgrados);
    while ($row = $result->fetch_assoc()) {
        $postgrados_coordinador[] = $row;
    }
}

// Obtener escolaridades disponibles
$sql_escolaridades = "SELECT id, nombre FROM escolaridades ORDER BY nombre";
$result_escolaridades = $conn->query($sql_escolaridades);
$escolaridades = [];
while ($row = $result_escolaridades->fetch_assoc()) {
    $escolaridades[] = $row;
}

// Configuración de paginación
$registros_por_pagina = 25;
$pagina_actual = isset($_GET['pagina']) ? (int)$_GET['pagina'] : 1;
if ($pagina_actual < 1) $pagina_actual = 1;
$offset = ($pagina_actual - 1) * $registros_por_pagina;

// Procesar filtros
$filtros = [
    'postgrado_id' => $_GET['postgrado_id'] ?? '',
    'escolaridad_id' => $_GET['escolaridad_id'] ?? '',
    'cuatrimestre' => $_GET['cuatrimestre'] ?? '',
    'estudiante' => $_GET['estudiante'] ?? '',
    'fecha_desde' => $_GET['fecha_desde'] ?? '',
    'fecha_hasta' => $_GET['fecha_hasta'] ?? '',
    'estado_nota' => $_GET['estado_nota'] ?? 'todas',
    'usuario_nota' => $_GET['usuario_nota'] ?? 'todos' // Nuevo filtro: todos, mis_notas, otros_usuarios
];

// Construir consulta base
$where_conditions = [];
$params = [];
$types = '';

if (!empty($filtros['postgrado_id'])) {
    $where_conditions[] = "p.id = ?";
    $params[] = $filtros['postgrado_id'];
    $types .= 'i';
}

if (!empty($filtros['escolaridad_id'])) {
    $where_conditions[] = "i.escolaridad_id = ?";
    $params[] = $filtros['escolaridad_id'];
    $types .= 'i';
}

if (!empty($filtros['cuatrimestre'])) {
    $where_conditions[] = "al.nombre LIKE ?";
    $params[] = '%-' . $filtros['cuatrimestre'];
    $types .= 's';
}

if (!empty($filtros['estudiante'])) {
    $where_conditions[] = "(e.nombres LIKE ? OR e.apellidos LIKE ? OR e.cedula LIKE ?)";
    $params[] = '%' . $filtros['estudiante'] . '%';
    $params[] = '%' . $filtros['estudiante'] . '%';
    $params[] = '%' . $filtros['estudiante'] . '%';
    $types .= 'sss';
}

if (!empty($filtros['fecha_desde']) && !empty($filtros['fecha_hasta'])) {
    $where_conditions[] = "DATE(i.fecha_modificacion) BETWEEN ? AND ?";
    $params[] = $filtros['fecha_desde'];
    $params[] = $filtros['fecha_hasta'];
    $types .= 'ss';
}

// Filtro por estado de nota
if ($filtros['estado_nota'] === 'pendientes') {
    $where_conditions[] = "i.nota_definitiva IS NULL";
} elseif ($filtros['estado_nota'] === 'cargadas') {
    $where_conditions[] = "i.nota_definitiva IS NOT NULL";
}

// NUEVO FILTRO: Por usuario que cargó la nota
if ($filtros['usuario_nota'] === 'mis_notas') {
    $where_conditions[] = "i.usuario_modificacion = ?";
    $params[] = $_SESSION['user_id'];
    $types .= 'i';
} elseif ($filtros['usuario_nota'] === 'otros_usuarios') {
    $where_conditions[] = "i.usuario_modificacion IS NOT NULL AND i.usuario_modificacion != ?";
    $params[] = $_SESSION['user_id'];
    $types .= 'i';
}

// Aplicar restricción de postgrados para coordinadores
if (($_SESSION['rol'] === 'coordinador1' || $_SESSION['rol'] === 'coordinador2') && !empty($postgrados_coordinador)) {
    $postgrado_ids = array_column($postgrados_coordinador, 'id');
    $placeholders = implode(',', array_fill(0, count($postgrado_ids), '?'));
    $where_conditions[] = "p.id IN ($placeholders)";
    $params = array_merge($params, $postgrado_ids);
    $types .= str_repeat('i', count($postgrado_ids));
}

$where_clause = '';
if (!empty($where_conditions)) {
    $where_clause = "WHERE " . implode(' AND ', $where_conditions);
}

// Consulta para obtener el TOTAL de registros (para paginación)
$sql_count = "SELECT COUNT(*) as total
        FROM inscripciones i
        JOIN estudiantes e ON i.estudiante_id = e.id
        JOIN denominaciones d ON i.denominacion_id = d.id
        JOIN postgrados p ON d.postgrado_id = p.id
        JOIN años_lectivos al ON i.año_lectivo_id = al.id
        JOIN escolaridades esc ON i.escolaridad_id = esc.id
        LEFT JOIN usuarios u ON i.usuario_modificacion = u.id
        $where_clause";

$total_registros = 0;
if (!empty($params) || $where_clause === '') {
    $stmt_count = $conn->prepare($sql_count);
    if ($stmt_count) {
        if (!empty($params)) {
            $stmt_count->bind_param($types, ...$params);
        }
        $stmt_count->execute();
        $result_count = $stmt_count->get_result();
        $row_count = $result_count->fetch_assoc();
        $total_registros = $row_count['total'];
        $stmt_count->close();
    }
}

// Calcular total de páginas
$total_paginas = ceil($total_registros / $registros_por_pagina);
if ($pagina_actual > $total_paginas && $total_paginas > 0) {
    $pagina_actual = $total_paginas;
}

// Consulta para obtener datos CON PAGINACIÓN
$sql = "SELECT 
            e.id as estudiante_id,
            e.nombres,
            e.apellidos,
            e.cedula,
            p.nombre as postgrado_nombre,
            d.codigo,
            d.nombre as asignatura_nombre,
            d.credito,
            i.nota_definitiva,
            i.fecha_modificacion,
            al.nombre as año_lectivo,
            u.username as modificado_por,
            u.id as usuario_id,
            esc.nombre as escolaridad
        FROM inscripciones i
        JOIN estudiantes e ON i.estudiante_id = e.id
        JOIN denominaciones d ON i.denominacion_id = d.id
        JOIN postgrados p ON d.postgrado_id = p.id
        JOIN años_lectivos al ON i.año_lectivo_id = al.id
        JOIN escolaridades esc ON i.escolaridad_id = esc.id
        LEFT JOIN usuarios u ON i.usuario_modificacion = u.id
        $where_clause
        ORDER BY e.apellidos, e.nombres, al.nombre, d.codigo
        LIMIT ? OFFSET ?";

// Agregar parámetros de paginación
$params_paginacion = $params;
$types_paginacion = $types . 'ii';
$params_paginacion[] = $registros_por_pagina;
$params_paginacion[] = $offset;

$resultados = [];
if (!empty($params_paginacion) || $where_clause === '') {
    $stmt = $conn->prepare($sql);
    if ($stmt) {
        if (!empty($params_paginacion)) {
            $stmt->bind_param($types_paginacion, ...$params_paginacion);
        }
        $stmt->execute();
        $result = $stmt->get_result();
        $resultados = $result->fetch_all(MYSQLI_ASSOC);
        $stmt->close();
    }
}

// Para exportación (sin paginación)
if (isset($_GET['export'])) {
    $export_type = $_GET['export'];
    
    // Consulta sin límites para exportación
    $sql_export = "SELECT 
            e.id as estudiante_id,
            e.nombres,
            e.apellidos,
            e.cedula,
            p.nombre as postgrado_nombre,
            d.codigo,
            d.nombre as asignatura_nombre,
            d.credito,
            i.nota_definitiva,
            i.fecha_modificacion,
            al.nombre as año_lectivo,
            u.username as modificado_por,
            u.id as usuario_id,
            esc.nombre as escolaridad
        FROM inscripciones i
        JOIN estudiantes e ON i.estudiante_id = e.id
        JOIN denominaciones d ON i.denominacion_id = d.id
        JOIN postgrados p ON d.postgrado_id = p.id
        JOIN años_lectivos al ON i.año_lectivo_id = al.id
        JOIN escolaridades esc ON i.escolaridad_id = esc.id
        LEFT JOIN usuarios u ON i.usuario_modificacion = u.id
        $where_clause
        ORDER BY e.apellidos, e.nombres, al.nombre, d.codigo";
    
    $resultados_export = [];
    if (!empty($params) || $where_clause === '') {
        $stmt_export = $conn->prepare($sql_export);
        if ($stmt_export) {
            if (!empty($params)) {
                $stmt_export->bind_param($types, ...$params);
            }
            $stmt_export->execute();
            $result_export = $stmt_export->get_result();
            $resultados_export = $result_export->fetch_all(MYSQLI_ASSOC);
            $stmt_export->close();
        }
    }
    
    if ($export_type === 'pdf') {
        require_once '../vendor/autoload.php';
        
        try {
            $mpdf = new \Mpdf\Mpdf([
                'mode' => 'utf-8',
                'format' => 'Letter',
                'margin_left' => 10,
                'margin_right' => 10,
                'margin_top' => 15,
                'margin_bottom' => 15,
                'default_font_size' => 10,
                'default_font' => 'arial'
            ]);

            $html = generarHTMLReportePDF($resultados_export, $filtros, $postgrados_coordinador, $escolaridades);
            $mpdf->WriteHTML($html);
            
            $filename = 'Reporte_Notas_Cuatrimestral_' . date('Y-m-d') . '.pdf';
            $mpdf->Output($filename, \Mpdf\Output\Destination::DOWNLOAD);
            exit();
            
        } catch (\Mpdf\MpdfException $e) {
            die("Error al generar PDF: " . $e->getMessage());
        }
        
    } elseif ($export_type === 'excel') {
        header('Content-Type: application/vnd.ms-excel');
        header('Content-Disposition: attachment; filename="Reporte_Notas_Cuatrimestral_' . date('Y-m-d') . '.xls"');
        
        echo generarHTMLReporteExcel($resultados_export, $filtros, $postgrados_coordinador, $escolaridades);
        exit();
    }
}

// Función para generar HTML del reporte PDF
function generarHTMLReportePDF($resultados, $filtros, $postgrados_coordinador, $escolaridades) {
    $html = '
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset="UTF-8">
        <title>Reporte de Notas Cuatrimestral</title>
        <style>
            body { font-family: Arial, sans-serif; margin: 0; padding: 20px; }
            .header { text-align: center; margin-bottom: 20px; }
            .header h1 { margin: 0; font-size: 16px; }
            .header h2 { margin: 5px 0; font-size: 14px; color: #666; }
            .filtros { margin-bottom: 15px; font-size: 12px; }
            .filtros strong { display: inline-block; width: 120px; }
            table { width: 100%; border-collapse: collapse; font-size: 10px; }
            th, td { border: 1px solid #ddd; padding: 6px; text-align: left; }
            th { background-color: #f4f4f4; font-weight: bold; }
            .total { font-weight: bold; background-color: #e9ecef; }
        </style>
    </head>
    <body>
        <div class="header">
            <h1>UNIVERSIDAD CENTRAL DE VENEZUELA</h1>
            <h2>REPORTE DE NOTAS CARGADAS - SISTEMA DE POSTGRADO</h2>
            <h3>Constancia de Carga de Notas por Cuatrimestre</h3>
        </div>
        
        <div class="filtros">
            <p><strong>Fecha de reporte:</strong> ' . date('d/m/Y') . '</p>';
            
            if (!empty($filtros['postgrado_id'])) {
                $postgrado_nombre = '';
                foreach ($postgrados_coordinador as $postgrado) {
                    if ($postgrado['id'] == $filtros['postgrado_id']) {
                        $postgrado_nombre = $postgrado['nombre'];
                        break;
                    }
                }
                $html .= '<p><strong>Postgrado:</strong> ' . htmlspecialchars($postgrado_nombre) . '</p>';
            }
            
            if (!empty($filtros['escolaridad_id'])) {
                $escolaridad_nombre = '';
                foreach ($escolaridades as $escolaridad) {
                    if ($escolaridad['id'] == $filtros['escolaridad_id']) {
                        $escolaridad_nombre = $escolaridad['nombre'];
                        break;
                    }
                }
                $html .= '<p><strong>Escolaridad:</strong> ' . htmlspecialchars($escolaridad_nombre) . '</p>';
            }
            
            if (!empty($filtros['cuatrimestre'])) {
                $html .= '<p><strong>Cuatrimestre:</strong> ' . htmlspecialchars($filtros['cuatrimestre']) . '</p>';
            }
            
            if (!empty($filtros['estado_nota'])) {
                $estado_texto = [
                    'todas' => 'Todas las notas',
                    'pendientes' => 'Solo notas pendientes',
                    'cargadas' => 'Solo notas cargadas'
                ];
                $html .= '<p><strong>Estado de nota:</strong> ' . $estado_texto[$filtros['estado_nota']] . '</p>';
            }
            
            // NUEVO: Mostrar filtro de usuario aplicado
            if (!empty($filtros['usuario_nota'])) {
                $usuario_texto = [
                    'todos' => 'Todos los usuarios',
                    'mis_notas' => 'Solo mis notas',
                    'otros_usuarios' => 'Notas de otros usuarios'
                ];
                $html .= '<p><strong>Notas cargadas por:</strong> ' . $usuario_texto[$filtros['usuario_nota']] . '</p>';
            }
            
            $html .= '<p><strong>Total de registros:</strong> ' . count($resultados) . '</p>
        </div>
        
        <table>
            <thead>
                <tr>
                    <th>Estudiante</th>
                    <th>Cédula</th>
                    <th>Postgrado</th>
                    <th>Asignatura</th>
                    <th>Crédito</th>
                    <th>Nota</th>
                    <th>Año Lectivo</th>
                    <th>Escolaridad</th>
                    <th>Fecha Modificación</th>
                    <th>Modificado Por</th>
                </tr>
            </thead>
            <tbody>';
    
    foreach ($resultados as $row) {
        $html .= '
                <tr>
                    <td>' . htmlspecialchars($row['apellidos'] . ', ' . $row['nombres']) . '</td>
                    <td>' . htmlspecialchars($row['cedula']) . '</td>
                    <td>' . htmlspecialchars($row['postgrado_nombre']) . '</td>
                    <td>' . htmlspecialchars($row['codigo'] . ' - ' . $row['asignatura_nombre']) . '</td>
                    <td>' . htmlspecialchars($row['credito']) . '</td>
                    <td>' . htmlspecialchars($row['nota_definitiva'] ?? 'PENDIENTE') . '</td>
                    <td>' . htmlspecialchars($row['año_lectivo']) . '</td>
                    <td>' . htmlspecialchars($row['escolaridad']) . '</td>
                    <td>' . ($row['fecha_modificacion'] ? date('d/m/Y H:i', strtotime($row['fecha_modificacion'])) : 'N/A') . '</td>
                    <td>' . htmlspecialchars($row['modificado_por'] ?? 'N/A') . '</td>
                </tr>';
    }
    
    $html .= '
            </tbody>
        </table>
    </body>
    </html>';
    
    return $html;
}

// Función para generar HTML del reporte Excel
function generarHTMLReporteExcel($resultados, $filtros, $postgrados_coordinador, $escolaridades) {
    $html = '<html>
    <head>
        <meta charset="UTF-8">
        <title>Reporte de Notas Cuatrimestral</title>
    </head>
    <body>
        <table border="1">
            <tr>
                <th colspan="10" style="text-align: center; background-color: #d9edf7; font-size: 14px;">
                    UNIVERSIDAD CENTRAL DE VENEZUELA - REPORTE DE NOTAS CARGADAS
                </th>
            </tr>
            <tr>
                <th colspan="10" style="text-align: center; background-color: #f5f5f5; font-size: 12px;">
                    Constancia de Carga de Notas por Cuatrimestre - ' . date('d/m/Y') . '
                </th>
            </tr>';
            
            // Filtros aplicados
            $filtros_texto = [];
            if (!empty($filtros['postgrado_id'])) {
                $postgrado_nombre = '';
                foreach ($postgrados_coordinador as $postgrado) {
                    if ($postgrado['id'] == $filtros['postgrado_id']) {
                        $postgrado_nombre = $postgrado['nombre'];
                        break;
                    }
                }
                $filtros_texto[] = 'Postgrado: ' . $postgrado_nombre;
            }
            
            if (!empty($filtros['escolaridad_id'])) {
                $escolaridad_nombre = '';
                foreach ($escolaridades as $escolaridad) {
                    if ($escolaridad['id'] == $filtros['escolaridad_id']) {
                        $escolaridad_nombre = $escolaridad['nombre'];
                        break;
                    }
                }
                $filtros_texto[] = 'Escolaridad: ' . $escolaridad_nombre;
            }
            
            if (!empty($filtros['cuatrimestre'])) {
                $filtros_texto[] = 'Cuatrimestre: ' . $filtros['cuatrimestre'];
            }
            
            if (!empty($filtros['estado_nota'])) {
                $estado_texto = [
                    'todas' => 'Todas las notas',
                    'pendientes' => 'Solo notas pendientes',
                    'cargadas' => 'Solo notas cargadas'
                ];
                $filtros_texto[] = 'Estado: ' . $estado_texto[$filtros['estado_nota']];
            }
            
            // NUEVO: Mostrar filtro de usuario aplicado
            if (!empty($filtros['usuario_nota'])) {
                $usuario_texto = [
                    'todos' => 'Todos los usuarios',
                    'mis_notas' => 'Solo mis notas',
                    'otros_usuarios' => 'Notas de otros usuarios'
                ];
                $filtros_texto[] = 'Notas cargadas por: ' . $usuario_texto[$filtros['usuario_nota']];
            }
            
            if (!empty($filtros_texto)) {
                $html .= '<tr><td colspan="10" style="background-color: #f9f9f9;"><strong>Filtros aplicados:</strong> ' . implode(' | ', $filtros_texto) . '</td></tr>';
            }
            
            $html .= '
            <tr>
                <th style="background-color: #337ab7; color: white;">Estudiante</th>
                <th style="background-color: #337ab7; color: white;">Cédula</th>
                <th style="background-color: #337ab7; color: white;">Postgrado</th>
                <th style="background-color: #337ab7; color: white;">Asignatura</th>
                <th style="background-color: #337ab7; color: white;">Crédito</th>
                <th style="background-color: #337ab7; color: white;">Nota</th>
                <th style="background-color: #337ab7; color: white;">Año Lectivo</th>
                <th style="background-color: #337ab7; color: white;">Escolaridad</th>
                <th style="background-color: #337ab7; color: white;">Fecha Modificación</th>
                <th style="background-color: #337ab7; color: white;">Modificado Por</th>
            </tr>';
    
    foreach ($resultados as $row) {
        $html .= '
            <tr>
                <td>' . htmlspecialchars($row['apellidos'] . ', ' . $row['nombres']) . '</td>
                <td>' . htmlspecialchars($row['cedula']) . '</td>
                <td>' . htmlspecialchars($row['postgrado_nombre']) . '</td>
                <td>' . htmlspecialchars($row['codigo'] . ' - ' . $row['asignatura_nombre']) . '</td>
                <td>' . htmlspecialchars($row['credito']) . '</td>
                <td>' . htmlspecialchars($row['nota_definitiva'] ?? 'PENDIENTE') . '</td>
                <td>' . htmlspecialchars($row['año_lectivo']) . '</td>
                <td>' . htmlspecialchars($row['escolaridad']) . '</td>
                <td>' . ($row['fecha_modificacion'] ? date('d/m/Y H:i', strtotime($row['fecha_modificacion'])) : 'N/A') . '</td>
                <td>' . htmlspecialchars($row['modificado_por'] ?? 'N/A') . '</td>
            </tr>';
    }
    
    $html .= '
            <tr>
                <td colspan="10" style="text-align: center; background-color: #f0f0f0; font-weight: bold;">
                    Total de registros: ' . count($resultados) . '
                </td>
            </tr>
        </table>
    </body>
    </html>';
    
    return $html;
}

$page_title = "Reportes Cuatrimestrales - Constancia de Carga de Notas";
include '../includes/header.php';
?>

<div class="container-fluid mt-4">
    <div class="row">
        <div class="col-md-12">
            <div class="card">
                <div class="card-header bg-primary text-white">
                    <h4 class="mb-0">
                        <i class="bi bi-graph-up"></i> Reporte de Notas Cargadas - Constancia Cuatrimestral
                    </h4>
                </div>
                <div class="card-body">
                    <!-- Formulario de Filtros -->
                    <form method="GET" class="mb-4">
                        <input type="hidden" name="pagina" value="1">
                        <div class="row g-3">
                            <div class="col-md-3">
                                <label class="form-label">Postgrado:</label>
                                <select name="postgrado_id" class="form-select">
                                    <option value="">Todos los postgrados</option>
                                    <?php foreach ($postgrados_coordinador as $postgrado): ?>
                                        <option value="<?= $postgrado['id'] ?>" 
                                            <?= ($filtros['postgrado_id'] == $postgrado['id']) ? 'selected' : '' ?>>
                                            <?= htmlspecialchars($postgrado['nombre']) ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <div class="col-md-2">
                                <label class="form-label">Escolaridad:</label>
                                <select name="escolaridad_id" class="form-select">
                                    <option value="">Todas</option>
                                    <?php foreach ($escolaridades as $escolaridad): ?>
                                        <option value="<?= $escolaridad['id'] ?>" 
                                            <?= ($filtros['escolaridad_id'] == $escolaridad['id']) ? 'selected' : '' ?>>
                                            <?= htmlspecialchars($escolaridad['nombre']) ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <div class="col-md-2">
                                <label class="form-label">Cuatrimestre:</label>
                                <select name="cuatrimestre" class="form-select">
                                    <option value="">Todos</option>
                                    <option value="01C" <?= ($filtros['cuatrimestre'] == '01C') ? 'selected' : '' ?>>1º Cuatrimestre</option>
                                    <option value="02C" <?= ($filtros['cuatrimestre'] == '02C') ? 'selected' : '' ?>>2º Cuatrimestre</option>
                                    <option value="03C" <?= ($filtros['cuatrimestre'] == '03C') ? 'selected' : '' ?>>3º Cuatrimestre</option>
                                </select>
                            </div>
                            
                            <div class="col-md-2">
                                <label class="form-label">Estado Nota:</label>
                                <select name="estado_nota" class="form-select">
                                    <option value="todas" <?= ($filtros['estado_nota'] == 'todas') ? 'selected' : '' ?>>Todas</option>
                                    <option value="pendientes" <?= ($filtros['estado_nota'] == 'pendientes') ? 'selected' : '' ?>>Solo Pendientes</option>
                                    <option value="cargadas" <?= ($filtros['estado_nota'] == 'cargadas') ? 'selected' : '' ?>>Solo Cargadas</option>
                                </select>
                            </div>

                            <!-- NUEVO FILTRO: Usuario que cargó la nota -->
                            <div class="col-md-2">
                                <label class="form-label">Notas Cargadas Por:</label>
                                <select name="usuario_nota" class="form-select">
                                    <option value="todos" <?= ($filtros['usuario_nota'] == 'todos') ? 'selected' : '' ?>>Todos los usuarios</option>
                                    <option value="mis_notas" <?= ($filtros['usuario_nota'] == 'mis_notas') ? 'selected' : '' ?>>Solo mis notas</option>
                                    <option value="otros_usuarios" <?= ($filtros['usuario_nota'] == 'otros_usuarios') ? 'selected' : '' ?>>Notas de otros usuarios</option>
                                </select>
                            </div>
                            
                            <div class="col-md-3">
                                <label class="form-label">Buscar Estudiante:</label>
                                <input type="text" name="estudiante" class="form-control" 
                                       value="<?= htmlspecialchars($filtros['estudiante']) ?>" 
                                       placeholder="Nombres, apellidos o cédula">
                            </div>
                        </div>
                        
                        <div class="row g-3 mt-2">
                            <div class="col-md-3">
                                <label class="form-label">Fecha Desde:</label>
                                <input type="date" name="fecha_desde" class="form-control" 
                                       value="<?= htmlspecialchars($filtros['fecha_desde']) ?>">
                            </div>
                            
                            <div class="col-md-3">
                                <label class="form-label">Fecha Hasta:</label>
                                <input type="date" name="fecha_hasta" class="form-control" 
                                       value="<?= htmlspecialchars($filtros['fecha_hasta']) ?>">
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">&nbsp;</label>
                                <div class="d-flex gap-2">
                                    <button type="submit" class="btn btn-primary flex-fill">
                                        <i class="bi bi-search"></i> Buscar
                                    </button>
                                    <a href="?" class="btn btn-secondary flex-fill">
                                        <i class="bi bi-arrow-clockwise"></i> Limpiar
                                    </a>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Botones de Exportación y Previsualización -->
                        <?php if (!empty($resultados) || $total_registros > 0): ?>
                        <div class="row g-3 mt-3">
                            <div class="col-md-12">
                                <div class="d-flex gap-2">
                                    <button type="submit" name="export" value="pdf" class="btn btn-danger flex-fill">
                                        <i class="bi bi-file-pdf"></i> Exportar PDF
                                    </button>
                                    <button type="submit" name="export" value="excel" class="btn btn-success flex-fill">
                                        <i class="bi bi-file-excel"></i> Exportar Excel
                                    </button>
                                    <button type="button" id="btn-previsualizar" class="btn btn-info flex-fill">
                                        <i class="bi bi-eye"></i> Previsualizar Excel
                                    </button>
                                </div>
                            </div>
                        </div>
                        <?php endif; ?>
                    </form>

                    <!-- Resumen de Filtros y Paginación -->
                    <?php if (!empty($filtros['postgrado_id']) || !empty($filtros['escolaridad_id']) || !empty($filtros['cuatrimestre']) || !empty($filtros['estado_nota']) || !empty($filtros['usuario_nota'])): ?>
                        <div class="alert alert-info">
                            <div class="row align-items-center">
                                <div class="col-md-8">
                                    <h6 class="mb-2">Filtros aplicados:</h6>
                                    <div class="row">
                                        <?php if (!empty($filtros['postgrado_id'])): 
                                            $postgrado_nombre = '';
                                            foreach ($postgrados_coordinador as $postgrado) {
                                                if ($postgrado['id'] == $filtros['postgrado_id']) {
                                                    $postgrado_nombre = $postgrado['nombre'];
                                                    break;
                                                }
                                            }
                                        ?>
                                            <div class="col-md-3">
                                                <strong>Postgrado:</strong> <?= htmlspecialchars($postgrado_nombre) ?>
                                            </div>
                                        <?php endif; ?>
                                        
                                        <?php if (!empty($filtros['escolaridad_id'])): 
                                            $escolaridad_nombre = '';
                                            foreach ($escolaridades as $escolaridad) {
                                                if ($escolaridad['id'] == $filtros['escolaridad_id']) {
                                                    $escolaridad_nombre = $escolaridad['nombre'];
                                                    break;
                                                }
                                            }
                                        ?>
                                            <div class="col-md-3">
                                                <strong>Escolaridad:</strong> <?= htmlspecialchars($escolaridad_nombre) ?>
                                            </div>
                                        <?php endif; ?>
                                        
                                        <?php if (!empty($filtros['cuatrimestre'])): ?>
                                            <div class="col-md-2">
                                                <strong>Cuatrimestre:</strong> 
                                                <?= htmlspecialchars($filtros['cuatrimestre']) ?>
                                            </div>
                                        <?php endif; ?>
                                        
                                        <?php if (!empty($filtros['estado_nota'])): ?>
                                            <div class="col-md-2">
                                                <strong>Estado Nota:</strong> 
                                                <?= $filtros['estado_nota'] === 'todas' ? 'Todas' : ($filtros['estado_nota'] === 'pendientes' ? 'Solo Pendientes' : 'Solo Cargadas') ?>
                                            </div>
                                        <?php endif; ?>

                                        <!-- NUEVO: Mostrar filtro de usuario aplicado -->
                                        <?php if (!empty($filtros['usuario_nota'])): ?>
                                            <div class="col-md-2">
                                                <strong>Notas Cargadas Por:</strong> 
                                                <?= $filtros['usuario_nota'] === 'todos' ? 'Todos' : ($filtros['usuario_nota'] === 'mis_notas' ? 'Solo Mis Notas' : 'Otros Usuarios') ?>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <div class="col-md-4 text-end">
                                    <div class="pagination-info">
                                        <strong>Total:</strong> <?= number_format($total_registros) ?> registros
                                        <?php if ($total_paginas > 1): ?>
                                            | <strong>Página:</strong> <?= $pagina_actual ?> de <?= $total_paginas ?>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    <?php endif; ?>

                    <!-- Controles de Paginación -->
                    <?php if ($total_paginas > 1): ?>
                    <div class="d-flex justify-content-between align-items-center mb-3">
                        <div>
                            <small class="text-muted">
                                Mostrando <?= count($resultados) ?> de <?= number_format($total_registros) ?> registros
                            </small>
                        </div>
                        <nav>
                            <ul class="pagination pagination-sm mb-0">
                                <?php if ($pagina_actual > 1): ?>
                                    <li class="page-item">
                                        <a class="page-link" href="?<?= http_build_query(array_merge($filtros, ['pagina' => 1])) ?>" aria-label="Primera">
                                            <span aria-hidden="true">&laquo;&laquo;</span>
                                        </a>
                                    </li>
                                    <li class="page-item">
                                        <a class="page-link" href="?<?= http_build_query(array_merge($filtros, ['pagina' => $pagina_actual - 1])) ?>" aria-label="Anterior">
                                            <span aria-hidden="true">&laquo;</span>
                                        </a>
                                    </li>
                                <?php else: ?>
                                    <li class="page-item disabled">
                                        <span class="page-link">&laquo;&laquo;</span>
                                    </li>
                                    <li class="page-item disabled">
                                        <span class="page-link">&laquo;</span>
                                    </li>
                                <?php endif; ?>

                                <?php
                                // Mostrar páginas cercanas a la actual
                                $inicio = max(1, $pagina_actual - 2);
                                $fin = min($total_paginas, $pagina_actual + 2);
                                
                                for ($i = $inicio; $i <= $fin; $i++):
                                ?>
                                    <li class="page-item <?= ($i == $pagina_actual) ? 'active' : '' ?>">
                                        <a class="page-link" href="?<?= http_build_query(array_merge($filtros, ['pagina' => $i])) ?>">
                                            <?= $i ?>
                                        </a>
                                    </li>
                                <?php endfor; ?>

                                <?php if ($pagina_actual < $total_paginas): ?>
                                    <li class="page-item">
                                        <a class="page-link" href="?<?= http_build_query(array_merge($filtros, ['pagina' => $pagina_actual + 1])) ?>" aria-label="Siguiente">
                                            <span aria-hidden="true">&raquo;</span>
                                        </a>
                                    </li>
                                    <li class="page-item">
                                        <a class="page-link" href="?<?= http_build_query(array_merge($filtros, ['pagina' => $total_paginas])) ?>" aria-label="Última">
                                            <span aria-hidden="true">&raquo;&raquo;</span>
                                        </a>
                                    </li>
                                <?php else: ?>
                                    <li class="page-item disabled">
                                        <span class="page-link">&raquo;</span>
                                    </li>
                                    <li class="page-item disabled">
                                        <span class="page-link">&raquo;&raquo;</span>
                                    </li>
                                <?php endif; ?>
                            </ul>
                        </nav>
                    </div>
                    <?php endif; ?>

                    <!-- Tabla de Resultados -->
                    <?php if (!empty($resultados)): ?>
                        <div class="table-responsive" id="tabla-resultados">
                            <table class="table table-striped table-bordered table-hover">
                                <thead class="table-dark">
                                    <tr>
                                        <th>Estudiante</th>
                                        <th>Cédula</th>
                                        <th>Postgrado</th>
                                        <th>Asignatura</th>
                                        <th>Crédito</th>
                                        <th>Nota</th>
                                        <th>Año Lectivo</th>
                                        <th>Escolaridad</th>
                                        <th>Fecha Modificación</th>
                                        <th>Modificado Por</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($resultados as $row): ?>
                                        <tr>
                                            <td><?= htmlspecialchars($row['apellidos'] . ', ' . $row['nombres']) ?></td>
                                            <td><?= htmlspecialchars($row['cedula']) ?></td>
                                            <td><?= htmlspecialchars($row['postgrado_nombre']) ?></td>
                                            <td>
                                                <small class="text-muted"><?= htmlspecialchars($row['codigo']) ?></small><br>
                                                <?= htmlspecialchars($row['asignatura_nombre']) ?>
                                            </td>
                                            <td class="text-center"><?= htmlspecialchars($row['credito']) ?></td>
                                            <td class="text-center">
                                                <?php if ($row['nota_definitiva']): ?>
                                                    <span class="badge bg-success"><?= htmlspecialchars($row['nota_definitiva']) ?></span>
                                                <?php else: ?>
                                                    <span class="badge bg-warning">PENDIENTE</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <span class="badge bg-info"><?= htmlspecialchars($row['año_lectivo']) ?></span>
                                            </td>
                                            <td><?= htmlspecialchars($row['escolaridad']) ?></td>
                                            <td>
                                                <small class="text-muted">
                                                    <?= $row['fecha_modificacion'] ? date('d/m/Y H:i', strtotime($row['fecha_modificacion'])) : 'N/A' ?>
                                                </small>
                                            </td>
                                            <td>
                                                <?php if ($row['usuario_id'] == $_SESSION['user_id']): ?>
                                                    <span class="badge bg-primary"><?= htmlspecialchars($row['modificado_por'] ?? 'N/A') ?> (Yo)</span>
                                                <?php else: ?>
                                                    <?= htmlspecialchars($row['modificado_por'] ?? 'N/A') ?>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php else: ?>
                        <div class="text-center py-5">
                            <i class="bi bi-inbox display-1 text-muted"></i>
                            <h4 class="text-muted mt-3">No se encontraron resultados</h4>
                            <p class="text-muted">Utilice los filtros para buscar notas cargadas</p>
                        </div>
                    <?php endif; ?>

                    <!-- Controles de Paginación (abajo) -->
                    <?php if ($total_paginas > 1): ?>
                    <div class="d-flex justify-content-between align-items-center mt-3">
                        <div>
                            <small class="text-muted">
                                Mostrando <?= count($resultados) ?> de <?= number_format($total_registros) ?> registros
                            </small>
                        </div>
                        <nav>
                            <ul class="pagination pagination-sm mb-0">
                                <?php if ($pagina_actual > 1): ?>
                                    <li class="page-item">
                                        <a class="page-link" href="?<?= http_build_query(array_merge($filtros, ['pagina' => 1])) ?>" aria-label="Primera">
                                            <span aria-hidden="true">&laquo;&laquo;</span>
                                        </a>
                                    </li>
                                    <li class="page-item">
                                        <a class="page-link" href="?<?= http_build_query(array_merge($filtros, ['pagina' => $pagina_actual - 1])) ?>" aria-label="Anterior">
                                            <span aria-hidden="true">&laquo;</span>
                                        </a>
                                    </li>
                                <?php else: ?>
                                    <li class="page-item disabled">
                                        <span class="page-link">&laquo;&laquo;</span>
                                    </li>
                                    <li class="page-item disabled">
                                        <span class="page-link">&laquo;</span>
                                    </li>
                                <?php endif; ?>

                                <?php
                                // Mostrar páginas cercanas a la actual
                                $inicio = max(1, $pagina_actual - 2);
                                $fin = min($total_paginas, $pagina_actual + 2);
                                
                                for ($i = $inicio; $i <= $fin; $i++):
                                ?>
                                    <li class="page-item <?= ($i == $pagina_actual) ? 'active' : '' ?>">
                                        <a class="page-link" href="?<?= http_build_query(array_merge($filtros, ['pagina' => $i])) ?>">
                                            <?= $i ?>
                                        </a>
                                    </li>
                                <?php endfor; ?>

                                <?php if ($pagina_actual < $total_paginas): ?>
                                    <li class="page-item">
                                        <a class="page-link" href="?<?= http_build_query(array_merge($filtros, ['pagina' => $pagina_actual + 1])) ?>" aria-label="Siguiente">
                                            <span aria-hidden="true">&raquo;</span>
                                        </a>
                                    </li>
                                    <li class="page-item">
                                        <a class="page-link" href="?<?= http_build_query(array_merge($filtros, ['pagina' => $total_paginas])) ?>" aria-label="Última">
                                            <span aria-hidden="true">&raquo;&raquo;</span>
                                        </a>
                                    </li>
                                <?php else: ?>
                                    <li class="page-item disabled">
                                        <span class="page-link">&raquo;</span>
                                    </li>
                                    <li class="page-item disabled">
                                        <span class="page-link">&raquo;&raquo;</span>
                                    </li>
                                <?php endif; ?>
                            </ul>
                        </nav>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Modal para Previsualización Excel -->
<div class="modal fade" id="modalPrevisualizacion" tabindex="-1" aria-labelledby="modalPrevisualizacionLabel" aria-hidden="true">
    <div class="modal-dialog modal-xl">
        <div class="modal-content">
            <div class="modal-header bg-info text-white">
                <h5 class="modal-title" id="modalPrevisualizacionLabel">
                    <i class="bi bi-eye"></i> Previsualización - Formato Excel
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body p-0">
                <div class="table-responsive">
                    <table class="table table-bordered mb-0" id="tabla-previsualizacion">
                        <!-- Contenido se llena con JavaScript -->
                    </table>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cerrar</button>
                <button type="submit" form="form-export" name="export" value="excel" class="btn btn-success">
                    <i class="bi bi-download"></i> Descargar Excel
                </button>
            </div>
        </div>
    </div>
</div>

<style>
    .table th {
        background-color: #343a40;
        color: white;
        font-weight: 600;
    }
    .badge {
        font-size: 0.75em;
    }
    #tabla-previsualizacion {
        width: 100%;
        font-size: 0.9em;
    }
    #tabla-previsualizacion th {
        background-color: #337ab7;
        color: white;
        text-align: center;
        font-weight: bold;
        border: 1px solid #2e6da4;
        padding: 10px;
    }
    #tabla-previsualizacion td {
        border: 1px solid #dee2e6;
        padding: 8px;
        vertical-align: middle;
    }
    #tabla-previsualizacion tr:nth-child(even) {
        background-color: #f8f9fa;
    }
    #tabla-previsualizacion tr:hover {
        background-color: #e9ecef;
    }
    .pagination-info {
        font-size: 0.9rem;
    }
    
    /* Asegurar que los badges se vean bien */
    .badge.bg-success, .badge.bg-warning, .badge.bg-info {
        color: white !important;
        padding: 4px 8px;
        border-radius: 4px;
    }
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Previsualización Excel
    document.getElementById('btn-previsualizar').addEventListener('click', function() {
        const tablaResultados = document.querySelector('#tabla-resultados table');
        const tablaPrevisualizacion = document.getElementById('tabla-previsualizacion');
        
        if (!tablaResultados) {
            alert('No hay datos para previsualizar');
            return;
        }
        
        // Limpiar tabla de previsualización
        tablaPrevisualizacion.innerHTML = '';
        
        // Crear encabezado de previsualización
        const thead = document.createElement('thead');
        const headerRow = document.createElement('tr');
        
        // Agregar encabezados
        const headers = ['Estudiante', 'Cédula', 'Postgrado', 'Asignatura', 'Crédito', 'Nota', 'Año Lectivo', 'Escolaridad', 'Fecha Modificación', 'Modificado Por'];
        headers.forEach(headerText => {
            const th = document.createElement('th');
            th.textContent = headerText;
            th.style.backgroundColor = '#337ab7';
            th.style.color = 'white';
            th.style.padding = '10px';
            th.style.textAlign = 'center';
            th.style.fontWeight = 'bold';
            th.style.border = '1px solid #2e6da4';
            headerRow.appendChild(th);
        });
        thead.appendChild(headerRow);
        tablaPrevisualizacion.appendChild(thead);
        
        // Crear cuerpo de la tabla
        const tbody = document.createElement('tbody');
        
        // Copiar filas de la tabla principal
        const filasOriginales = tablaResultados.querySelectorAll('tbody tr');
        
        filasOriginales.forEach((fila, filaIndex) => {
            const nuevaFila = document.createElement('tr');
            const celdas = fila.querySelectorAll('td');
            
            // Alternar colores de fondo para mejor legibilidad
            if (filaIndex % 2 === 0) {
                nuevaFila.style.backgroundColor = '#ffffff';
            } else {
                nuevaFila.style.backgroundColor = '#f8f9fa';
            }
            
            celdas.forEach((celda, index) => {
                const nuevaCelda = document.createElement('td');
                nuevaCelda.style.border = '1px solid #dee2e6';
                nuevaCelda.style.padding = '8px';
                nuevaCelda.style.verticalAlign = 'middle';
                
                switch(index) {
                    case 0: // Estudiante
                    case 1: // Cédula
                    case 2: // Postgrado
                        nuevaCelda.textContent = celda.textContent.trim();
                        break;
                        
                    case 3: // Asignatura (código + nombre)
                        nuevaCelda.innerHTML = celda.innerHTML;
                        break;
                        
                    case 4: // Crédito - SIN ESTILOS ESPECIALES
                        nuevaCelda.textContent = celda.textContent.trim();
                        nuevaCelda.style.textAlign = 'center';
                        break;
                        
                    case 5: // Nota
                        nuevaCelda.style.textAlign = 'center';
                        const badgeNota = celda.querySelector('.badge');
                        if (badgeNota) {
                            // Clonar el badge para mantener estilos
                            const badgeClon = badgeNota.cloneNode(true);
                            nuevaCelda.appendChild(badgeClon);
                        } else {
                            nuevaCelda.textContent = celda.textContent.trim();
                        }
                        break;
                        
                    case 6: // Año Lectivo
                        const badgeAnio = celda.querySelector('.badge');
                        if (badgeAnio) {
                            const badgeClon = badgeAnio.cloneNode(true);
                            nuevaCelda.appendChild(badgeClon);
                        } else {
                            nuevaCelda.textContent = celda.textContent.trim();
                        }
                        nuevaCelda.style.textAlign = 'center';
                        break;
                        
                    case 7: // Escolaridad
                    case 9: // Modificado Por
                        nuevaCelda.textContent = celda.textContent.trim();
                        break;
                        
                    case 8: // Fecha Modificación
                        const smallElement = celda.querySelector('small');
                        if (smallElement) {
                            nuevaCelda.textContent = smallElement.textContent.trim();
                        } else {
                            nuevaCelda.textContent = celda.textContent.trim();
                        }
                        nuevaCelda.style.textAlign = 'center';
                        nuevaCelda.style.fontSize = '0.85em';
                        break;
                        
                    default:
                        nuevaCelda.textContent = celda.textContent.trim();
                }
                
                nuevaFila.appendChild(nuevaCelda);
            });
            
            tbody.appendChild(nuevaFila);
        });
        
        tablaPrevisualizacion.appendChild(tbody);
        
        // Mostrar modal
        const modal = new bootstrap.Modal(document.getElementById('modalPrevisualizacion'));
        modal.show();
    });
});
</script>

<?php include '../includes/footer.php'; ?>