<?php
// 1. Configuración de sesión
require_once __DIR__ . '/../includes/session_config.php';

// 2. Verificación de auth
$allowed_roles = ['administrador', 'coordinador1', 'coordinador2'];
require_once __DIR__ . '/../includes/auth_check.php';

// 3. Configurar título de página
$page_title = "Reporte de Notas Cargadas";

// 4. Configurar encoding UTF-8
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');

// 5. Lógica específica de la página
include '../includes/db.php';

// Configurar conexión para usar UTF-8
$conn->set_charset("utf8mb4");

// -----------------------------------------------------------------------------
// OBTENER PERMISOS DEL USUARIO
// -----------------------------------------------------------------------------
$usuario_id = $_SESSION['user_id'];
$rol_actual = $_SESSION['rol'];

if ($rol_actual == 'administrador') {
    // Administrador ve todos los postgrados
    $sql_postgrados = "SELECT id, nombre FROM postgrados ORDER BY nombre";
    $stmt_postgrados = $conn->prepare($sql_postgrados);
    $stmt_postgrados->execute();
    $postgrados = $stmt_postgrados->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt_postgrados->close();
} else {
    // Coordinadores ven solo sus postgrados asignados
    $sql_postgrados = "SELECT DISTINCT p.id, p.nombre 
                       FROM postgrados p 
                       JOIN coordinador_postgrado cp ON p.id = cp.postgrado_id 
                       WHERE cp.coordinador_id = ?";
    $stmt_postgrados = $conn->prepare($sql_postgrados);
    $stmt_postgrados->bind_param("i", $usuario_id);
    $stmt_postgrados->execute();
    $postgrados = $stmt_postgrados->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt_postgrados->close();
}

if (empty($postgrados)) {
    include '../includes/header.php';
    echo '<div class="container mt-5"><div class="alert alert-danger">Error: No tiene ningún postgrado asignado. Por favor, contacte a un administrador.</div></div>';
    include '../includes/footer.php';
    exit();
}

// Obtener años lectivos
$sql_anios = "SELECT id, nombre FROM años_lectivos ORDER BY nombre DESC";
$anios_lectivos = $conn->query($sql_anios)->fetch_all(MYSQLI_ASSOC);

// Obtener años disponibles para filtro (últimos 10 años)
$anios_disponibles = [];
for ($i = 0; $i < 10; $i++) {
    $anios_disponibles[] = date('Y') - $i;
}

// Procesar filtros
$postgrado_seleccionado = isset($_GET['postgrado_id']) ? intval($_GET['postgrado_id']) : '';
$anio_lectivo = isset($_GET['anio_lectivo']) ? $conn->real_escape_string($_GET['anio_lectivo']) : '';
$cedula_busqueda = isset($_GET['cedula']) ? $conn->real_escape_string($_GET['cedula']) : '';
$trimestre = isset($_GET['trimestre']) ? $_GET['trimestre'] : '';
$anio_trimestre = isset($_GET['anio_trimestre']) ? intval($_GET['anio_trimestre']) : date('Y');
$formato = isset($_GET['formato']) ? $_GET['formato'] : 'preview';

// Verificar que el postgrado seleccionado esté entre los permitidos
if (!empty($postgrado_seleccionado)) {
    $postgrado_permitido = false;
    foreach ($postgrados as $p) {
        if ($p['id'] == $postgrado_seleccionado) {
            $postgrado_permitido = true;
            break;
        }
    }
    if (!$postgrado_permitido) {
        $_SESSION['error'] = "No tiene permisos para acceder a este postgrado.";
        $postgrado_seleccionado = '';
    }
}

// Obtener datos del reporte
$datos_reporte = [];
$estadisticas = [
    'total_estudiantes' => 0,
    'total_asignaturas' => 0,
    'con_notas' => 0,
    'sin_notas' => 0,
    'aprobados' => 0,
    'reprobados' => 0
];

if (!empty($postgrado_seleccionado)) {
    // Construir consulta base
    $sql_reporte = "SELECT 
                    e.id as estudiante_id,
                    e.nombres,
                    e.apellidos,
                    e.cedula_venezolana,
                    e.cedula_extranjera,
                    e.pasaporte,
                    e.tipo_documento,
                    d.codigo,
                    d.nombre as denominacion,
                    d.credito,
                    d.tipo,
                    i.nota_definitiva,
                    i.nota_final,
                    i.fecha_modificacion,
                    al.nombre as año_lectivo,
                    p.nombre as postgrado_nombre,
                    esc.nombre as escolaridad,
                    ep.estado as estado_estudiante,
                    u_mod.nombres as modificado_por_nombres,
                    u_mod.apellidos as modificado_por_apellidos,
                    u_mod.username as modificado_por_username
                FROM inscripciones i
                INNER JOIN estudiantes e ON i.estudiante_id = e.id
                INNER JOIN denominaciones d ON i.denominacion_id = d.id
                INNER JOIN años_lectivos al ON i.año_lectivo_id = al.id
                INNER JOIN escolaridades esc ON i.escolaridad_id = esc.id
                INNER JOIN postgrados p ON d.postgrado_id = p.id
                INNER JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id AND p.id = ep.postgrado_id
                LEFT JOIN usuarios u_mod ON i.usuario_modificacion = u_mod.id
                WHERE d.postgrado_id = ?";
    
    $params = [$postgrado_seleccionado];
    $types = "i";

    // Aplicar filtro de año lectivo
    if (!empty($anio_lectivo)) {
        $sql_reporte .= " AND al.nombre = ?";
        $params[] = $anio_lectivo;
        $types .= "s";
    }

    // Aplicar filtro de cédula/pasaporte
    if (!empty($cedula_busqueda)) {
        $sql_reporte .= " AND (e.cedula_venezolana LIKE ? OR e.cedula_extranjera LIKE ? OR e.pasaporte LIKE ?)";
        $params[] = "%$cedula_busqueda%";
        $params[] = "%$cedula_busqueda%";
        $params[] = "%$cedula_busqueda%";
        $types .= "sss";
    }

    // Aplicar filtro de trimestre
    if (!empty($trimestre) && !empty($anio_trimestre)) {
        switch ($trimestre) {
            case '1':
                // 1º cuatrimestre: enero a abril
                $sql_reporte .= " AND (MONTH(i.fecha_modificacion) BETWEEN 1 AND 4) AND YEAR(i.fecha_modificacion) = ?";
                break;
            case '2':
                // 2º cuatrimestre: mayo a agosto
                $sql_reporte .= " AND (MONTH(i.fecha_modificacion) BETWEEN 5 AND 8) AND YEAR(i.fecha_modificacion) = ?";
                break;
            case '3':
                // 3º cuatrimestre: septiembre a diciembre
                $sql_reporte .= " AND (MONTH(i.fecha_modificacion) BETWEEN 9 AND 12) AND YEAR(i.fecha_modificacion) = ?";
                break;
        }
        $params[] = $anio_trimestre;
        $types .= "i";
    }

    $sql_reporte .= " ORDER BY e.apellidos, e.nombres, d.codigo";

    $stmt_reporte = $conn->prepare($sql_reporte);
    if (!empty($params)) {
        $stmt_reporte->bind_param($types, ...$params);
    }
    $stmt_reporte->execute();
    $result_reporte = $stmt_reporte->get_result();
    $datos_reporte = $result_reporte->fetch_all(MYSQLI_ASSOC);
    $stmt_reporte->close();

    // Calcular estadísticas
    $estudiantes_unicos = [];
    foreach ($datos_reporte as $registro) {
        $estudiante_id = $registro['estudiante_id'];
        if (!in_array($estudiante_id, $estudiantes_unicos)) {
            $estudiantes_unicos[] = $estudiante_id;
            $estadisticas['total_estudiantes']++;
        }
        
        $estadisticas['total_asignaturas']++;
        
        if ($registro['nota_definitiva'] !== null) {
            $estadisticas['con_notas']++;
            
            if ($registro['tipo'] == 'puntaje') {
                if (floatval($registro['nota_definitiva']) >= 10) {
                    $estadisticas['aprobados']++;
                } else {
                    $estadisticas['reprobados']++;
                }
            } elseif ($registro['tipo'] == 'Aprobado_Reprobado') {
                if (strtoupper($registro['nota_definitiva']) == 'APROBADO') {
                    $estadisticas['aprobados']++;
                } else {
                    $estadisticas['reprobados']++;
                }
            }
        } else {
            $estadisticas['sin_notas']++;
        }
    }
}

// FUNCIÓN PARA OBTENER EL NÚMERO DE DOCUMENTO SEGÚN EL TIPO
function obtener_numero_documento($tipo_documento, $cedula_venezolana, $cedula_extranjera, $pasaporte) {
    switch ($tipo_documento) {
        case 'V': return $cedula_venezolana;
        case 'E': return $cedula_extranjera;
        case 'P': return $pasaporte;
        default: return $cedula_venezolana;
    }
}

// FUNCIÓN PARA OBTENER EL TIPO DE DOCUMENTO EN TEXTO
function obtener_tipo_documento_texto($tipo_documento) {
    switch ($tipo_documento) {
        case 'V': return 'Cédula Venezolana';
        case 'E': return 'Cédula Extranjera';
        case 'P': return 'Pasaporte';
        default: return 'Cédula Venezolana';
    }
}

// FUNCIÓN PARA FORMATEAR FECHA
function formatearFecha($fecha) {
    if (empty($fecha) || $fecha == '0000-00-00 00:00:00') {
        return 'N/A';
    }
    return date('d/m/Y H:i', strtotime($fecha));
}

// Generar reporte en PDF
if ($formato == 'pdf' && !empty($datos_reporte)) {
    require_once __DIR__ . '/../vendor/autoload.php';
    
    try {
        $mpdf = new \Mpdf\Mpdf([
            'mode' => 'utf-8',
            'format' => 'Letter',
            'margin_left' => 10,
            'margin_right' => 10,
            'margin_top' => 15,
            'margin_bottom' => 15,
            'default_font_size' => 9,
            'default_font' => 'arial'
        ]);
        
        $postgrado_nombre = '';
        foreach ($postgrados as $p) {
            if ($p['id'] == $postgrado_seleccionado) {
                $postgrado_nombre = $p['nombre'];
                break;
            }
        }

        $html = '
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <style>
                body { font-family: Arial, sans-serif; font-size: 12px; }
                .header { text-align: center; margin-bottom: 20px; }
                .filtros { margin-bottom: 15px; padding: 10px; background: #f5f5f5; border: 1px solid #ddd; }
                table { width: 100%; border-collapse: collapse; margin-top: 10px; }
                th, td { border: 1px solid #ddd; padding: 6px; text-align: left; }
                th { background: #2c3e50; color: white; }
                .estadisticas { margin: 15px 0; padding: 10px; background: #e8f4f8; border: 1px solid #bce8f1; }
                .numero { text-align: right; }
                .sin-nota { background-color: #fff3cd; }
            </style>
        </head>
        <body>
            <div class="header">
                <h2>UNIVERSIDAD CENTRAL DE VENEZUELA</h2>
                <h3>FACULTAD DE MEDICINA - POSTGRADO DE MEDICINA</h3>
                <h3>REPORTE DE NOTAS CARGADAS</h3>
            </div>
            
            <div class="filtros">
                <strong>Postgrado:</strong> ' . $postgrado_nombre . ' | 
                <strong>Fecha Generación:</strong> ' . date('d/m/Y H:i:s') . '<br>';

        if (!empty($anio_lectivo)) {
            $html .= '<strong>Año Lectivo:</strong> ' . $anio_lectivo . ' | ';
        }
        if (!empty($cedula_busqueda)) {
            $html .= '<strong>Documento:</strong> ' . $cedula_busqueda . ' | ';
        }
        if (!empty($trimestre)) {
            $trimestre_texto = '';
            switch ($trimestre) {
                case '1': $trimestre_texto = '1º Cuatrimestre (Ene-Abr)'; break;
                case '2': $trimestre_texto = '2º Cuatrimestre (May-Ago)'; break;
                case '3': $trimestre_texto = '3º Cuatrimestre (Sep-Dic)'; break;
            }
            $html .= '<strong>Periodo:</strong> ' . $trimestre_texto . ' ' . $anio_trimestre . ' | ';
        }

        $html .= '</div>
            
            <div class="estadisticas">
                <strong>ESTADÍSTICAS:</strong> 
                Total Estudiantes: ' . $estadisticas['total_estudiantes'] . ' | 
                Total Asignaturas: ' . $estadisticas['total_asignaturas'] . ' | 
                Con Notas: ' . $estadisticas['con_notas'] . ' | 
                Sin Notas: ' . $estadisticas['sin_notas'] . ' | 
                Aprobados: ' . $estadisticas['aprobados'] . ' | 
                Reprobados: ' . $estadisticas['reprobados'] . '
            </div>
            
            <table>
                <thead>
                    <tr>
                        <th>#</th>
                        <th>Estudiante</th>
                        <th>Documento</th>
                        <th>Asignatura</th>
                        <th>Denominación</th>
                        <th>Créditos</th>
                        <th>Nota Definitiva</th>
                        <th>Nota Final</th>
                        <th>Escolaridad</th>
                        <th>Fecha Modificación</th>
                        <th>Modificado Por</th>
                    </tr>
                </thead>
                <tbody>';
        
        $contador = 1;
        foreach ($datos_reporte as $fila) {
            $numero_documento = obtener_numero_documento(
                $fila['tipo_documento'], 
                $fila['cedula_venezolana'], 
                $fila['cedula_extranjera'], 
                $fila['pasaporte']
            );
            
            $clase_fila = $fila['nota_definitiva'] === null ? 'sin-nota' : '';
            
            $html .= '<tr class="' . $clase_fila . '">
                        <td>' . $contador . '</td>
                        <td>' . htmlspecialchars($fila['apellidos'] . ', ' . $fila['nombres']) . '</td>
                        <td>' . htmlspecialchars($numero_documento) . '</td>
                        <td>' . htmlspecialchars($fila['codigo']) . '</td>
                        <td>' . htmlspecialchars($fila['denominacion']) . '</td>
                        <td class="numero">' . htmlspecialchars($fila['credito']) . '</td>
                        <td class="numero">' . htmlspecialchars($fila['nota_definitiva'] ?? 'PENDIENTE') . '</td>
                        <td class="numero">' . htmlspecialchars($fila['nota_final'] ?? 'PENDIENTE') . '</td>
                        <td>' . htmlspecialchars($fila['escolaridad']) . '</td>
                        <td>' . formatearFecha($fila['fecha_modificacion']) . '</td>
                        <td>' . htmlspecialchars($fila['modificado_por_nombres'] . ' ' . $fila['modificado_por_apellidos'] ?? 'Sistema') . '</td>
                      </tr>';
            $contador++;
        }
        
        $html .= '</tbody>
            </table>
        </body>
        </html>';
        
        $mpdf->WriteHTML($html);
        $filename = 'Reporte_Notas_' . date('Y-m-d') . '.pdf';
        $mpdf->Output($filename, \Mpdf\Output\Destination::INLINE);
        exit;
        
    } catch (\Mpdf\MpdfException $e) {
        die("Error al generar PDF: " . $e->getMessage());
    }
}

// Generar reporte en Excel
if ($formato == 'excel' && !empty($datos_reporte)) {
    header('Content-Type: application/vnd.ms-excel; charset=utf-8');
    header('Content-Disposition: attachment; filename="Reporte_Notas_' . date('Y-m-d') . '.xls"');
    
    $postgrado_nombre = '';
    foreach ($postgrados as $p) {
        if ($p['id'] == $postgrado_seleccionado) {
            $postgrado_nombre = $p['nombre'];
            break;
        }
    }

    echo "<html>";
    echo "<meta charset='UTF-8'>";
    echo "<table border='1'>";
    echo "<tr><th colspan='11'>REPORTE DE NOTAS CARGADAS - " . $postgrado_nombre . "</th></tr>";
    echo "<tr><th colspan='11'>Generado: " . date('d/m/Y H:i:s');
    
    if (!empty($anio_lectivo)) {
        echo " | Año Lectivo: " . $anio_lectivo;
    }
    if (!empty($cedula_busqueda)) {
        echo " | Documento: " . $cedula_busqueda;
    }
    if (!empty($trimestre)) {
        $trimestre_texto = '';
        switch ($trimestre) {
            case '1': $trimestre_texto = '1º Cuatrimestre (Ene-Abr)'; break;
            case '2': $trimestre_texto = '2º Cuatrimestre (May-Ago)'; break;
            case '3': $trimestre_texto = '3º Cuatrimestre (Sep-Dic)'; break;
        }
        echo " | Periodo: " . $trimestre_texto . ' ' . $anio_trimestre;
    }
    
    echo "</th></tr>";
    echo "<tr><th colspan='11'></th></tr>";
    echo "<tr>
            <th>#</th>
            <th>ESTUDIANTE</th>
            <th>DOCUMENTO</th>
            <th>ASIGNATURA</th>
            <th>DENOMINACIÓN</th>
            <th>CRÉDITOS</th>
            <th>NOTA DEFINITIVA</th>
            <th>NOTA FINAL</th>
            <th>ESCOLARIDAD</th>
            <th>FECHA MODIFICACIÓN</th>
            <th>MODIFICADO POR</th>
          </tr>";
    
    $contador = 1;
    foreach ($datos_reporte as $fila) {
        $numero_documento = obtener_numero_documento(
            $fila['tipo_documento'], 
            $fila['cedula_venezolana'], 
            $fila['cedula_extranjera'], 
            $fila['pasaporte']
        );
        
        echo "<tr>
                <td>" . $contador . "</td>
                <td>" . htmlspecialchars($fila['apellidos'] . ', ' . $fila['nombres']) . "</td>
                <td>" . htmlspecialchars($numero_documento) . "</td>
                <td>" . htmlspecialchars($fila['codigo']) . "</td>
                <td>" . htmlspecialchars($fila['denominacion']) . "</td>
                <td>" . htmlspecialchars($fila['credito']) . "</td>
                <td>" . htmlspecialchars($fila['nota_definitiva'] ?? 'PENDIENTE') . "</td>
                <td>" . htmlspecialchars($fila['nota_final'] ?? 'PENDIENTE') . "</td>
                <td>" . htmlspecialchars($fila['escolaridad']) . "</td>
                <td>" . formatearFecha($fila['fecha_modificacion']) . "</td>
                <td>" . htmlspecialchars($fila['modificado_por_nombres'] . ' ' . $fila['modificado_por_apellidos'] ?? 'Sistema') . "</td>
              </tr>";
        $contador++;
    }
    
    echo "</table>";
    echo "</html>";
    exit;
}

// 6. Incluir header
include '../includes/header.php';
?>

<div class="container-fluid px-3 px-md-4">
    <!-- Botón para volver -->
    <div class="mb-3 mb-md-4">
        <a href="index.php" class="btn btn-secondary btn-sm">
            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-arrow-left" viewBox="0 0 16 16">
                <path fill-rule="evenodd" d="M15 8a.5.5 0 0 0-.5-.5H2.707l3.147-3.146a.5.5 0 1 0-.708-.708l-4 4a.5.5 0 0 0 0 .708l4 4a.5.5 0 0 0 .708-.708L2.707 8.5H14.5A.5.5 0 0 0 15 8z"/>
            </svg>
            Volver al Panel
        </a>
    </div>

    <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <?= htmlspecialchars($_SESSION['success']) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
        <?php unset($_SESSION['success']); ?>
    <?php endif; ?>
    
    <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <?= htmlspecialchars($_SESSION['error']) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
        <?php unset($_SESSION['error']); ?>
    <?php endif; ?>

    <!-- Información del usuario -->
    <div class="card mb-3 mb-md-4">
        <div class="card-header bg-info text-white py-2 py-md-3">
            <h4 class="mb-0">
                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" fill="currentColor" class="bi bi-graph-up" viewBox="0 0 16 16">
                    <path fill-rule="evenodd" d="M0 0h1v15h15v1H0V0Zm10 3.5a.5.5 0 0 1 .5-.5h4a.5.5 0 0 1 .5.5v4a.5.5 0 0 1-1 0V4.9l-3.613 4.417a.5.5 0 0 1-.74.037L7.06 6.767l-3.656 5.027a.5.5 0 0 1-.808-.588l4-5.5a.5.5 0 0 1 .758-.06l2.609 2.61L13.445 4H10.5a.5.5 0 0 1-.5-.5Z"/>
                </svg>
                Reporte de Notas Cargadas
            </h4>
        </div>
        <div class="card-body p-2 p-md-3">
            <div class="row">
                <div class="col-md-6">
                    <p class="mb-2"><strong>Usuario:</strong> <?php echo htmlspecialchars($_SESSION['username']); ?></p>
                    <p class="mb-2"><strong>Rol:</strong> <?php echo htmlspecialchars(ucfirst($rol_actual)); ?></p>
                </div>
                <div class="col-md-6">
                    <p class="mb-2"><strong>Postgrados disponibles:</strong> <?php echo count($postgrados); ?></p>
                    <p class="mb-0"><strong>Fecha:</strong> <?php echo date('d/m/Y H:i:s'); ?></p>
                </div>
            </div>
        </div>
    </div>

    <!-- Tarjeta de Filtros -->
    <div class="card mb-3 mb-md-4">
        <div class="card-header bg-primary text-white py-2 py-md-3">
            <h4 class="mb-0">Filtros del Reporte</h4>
        </div>
        <div class="card-body p-2 p-md-3">
            <form method="get" class="mb-0">
                <div class="row g-2 g-md-3">
                    <!-- Filtro de postgrado -->
                    <div class="col-12 col-md-6">
                        <label for="postgrado_id" class="form-label">Postgrado *</label>
                        <select name="postgrado_id" id="postgrado_id" class="form-select form-select-sm" required>
                            <option value="">Seleccionar Postgrado</option>
                            <?php foreach ($postgrados as $p): ?>
                                <option value="<?php echo htmlspecialchars($p['id']); ?>" 
                                    <?php echo ($p['id'] == $postgrado_seleccionado) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($p['nombre']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <!-- Filtro de año lectivo -->
                    <div class="col-12 col-md-6">
                        <label for="anio_lectivo" class="form-label">Año Lectivo</label>
                        <select id="anio_lectivo" name="anio_lectivo" class="form-select form-select-sm">
                            <option value="">Todos los años lectivos</option>
                            <?php foreach ($anios_lectivos as $al): ?>
                                <option value="<?php echo htmlspecialchars($al['nombre']); ?>" 
                                    <?php echo ($al['nombre'] == $anio_lectivo) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($al['nombre']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <!-- Filtro de cédula/pasaporte -->
                    <div class="col-12 col-md-6">
                        <label for="cedula" class="form-label">Cédula o Pasaporte</label>
                        <input type="text" id="cedula" name="cedula" class="form-control form-select-sm" 
                               value="<?php echo htmlspecialchars($cedula_busqueda); ?>" 
                               placeholder="Buscar por documento">
                    </div>

                    <!-- Filtro de trimestre -->
                    <div class="col-12 col-md-6">
                        <label for="trimestre" class="form-label">Periodo de Carga</label>
                        <div class="row g-1">
                            <div class="col-6">
                                <select id="trimestre" name="trimestre" class="form-select form-select-sm">
                                    <option value="">Todos los periodos</option>
                                    <option value="1" <?php echo ($trimestre == '1') ? 'selected' : ''; ?>>1º Cuatrimestre (Ene-Abr)</option>
                                    <option value="2" <?php echo ($trimestre == '2') ? 'selected' : ''; ?>>2º Cuatrimestre (May-Ago)</option>
                                    <option value="3" <?php echo ($trimestre == '3') ? 'selected' : ''; ?>>3º Cuatrimestre (Sep-Dic)</option>
                                </select>
                            </div>
                            <div class="col-6">
                                <select name="anio_trimestre" class="form-select form-select-sm">
                                    <?php foreach ($anios_disponibles as $anio): ?>
                                        <option value="<?php echo $anio; ?>" <?php echo ($anio == $anio_trimestre) ? 'selected' : ''; ?>>
                                            <?php echo $anio; ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                    </div>

                    <!-- Botones de acción -->
                    <div class="col-12">
                        <div class="d-flex flex-wrap gap-2 mt-2">
                            <button type="submit" name="formato" value="preview" class="btn btn-primary btn-sm flex-fill">
                                <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" fill="currentColor" class="bi bi-eye me-1" viewBox="0 0 16 16">
                                    <path d="M16 8s-3-5.5-8-5.5S0 8 0 8s3 5.5 8 5.5S16 8 16 8zM1.173 8a13.133 13.133 0 0 1 1.66-2.043C4.12 4.668 5.88 3.5 8 3.5c2.12 0 3.879 1.168 5.168 2.457A13.133 13.133 0 0 1 14.828 8c-.058.087-.122.183-.195.288-.335.48-.83 1.12-1.465 1.755C11.879 11.332 10.119 12.5 8 12.5c-2.12 0-3.879-1.168-5.168-2.457A13.134 13.134 0 0 1 1.172 8z"/>
                                    <path d="M8 5.5a2.5 2.5 0 1 0 0 5 2.5 2.5 0 0 0 0-5zM4.5 8a3.5 3.5 0 1 1 7 0 3.5 3.5 0 0 1-7 0z"/>
                                </svg>
                                Vista Previa
                            </button>
                            <button type="submit" name="formato" value="pdf" class="btn btn-danger btn-sm flex-fill" <?php echo empty($datos_reporte) ? 'disabled' : ''; ?>>
                                <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" fill="currentColor" class="bi bi-file-pdf me-1" viewBox="0 0 16 16">
                                    <path d="M5.523 10.424c.14-.082.293-.162.459-.238a7.878 7.878 0 0 1-.45.606c-.28.337-.498.516-.635.572a.266.266 0 0 1-.035.012.282.282 0 0 1-.026-.044c-.056-.11-.054-.216.04-.36.106-.165.319-.354.647-.548zm2.455-1.647c-.119.025-.237.05-.356.078a21.148 21.148 0 0 0 .5-1.05 12.045 12.045 0 0 0 .51.858c-.217.032-.436.07-.654.114zm2.525.939a3.881 3.881 0 0 1-.435-.41c.228.005.434.022.612.054.317.057.466.147.518.209a.095.095 0 0 1 .026.064.436.436 0 0 1-.06.2.307.307 0 0 1-.094.124.107.107 0 0 1-.069.015c-.09-.003-.258-.066-.498-.256zM8.278 4.97c-.04.244-.108.524-.2.829a4.86 4.86 0 0 1-.089-.346c-.076-.353-.087-.63-.046-.822.038-.177.11-.248.196-.283a.517.517 0 0 1 .145-.04c.013.03.028.092.032.198.005.122-.007.277-.038.465z"/>
                                    <path fill-rule="evenodd" d="M4 0h8a2 2 0 0 1 2 2v12a2 2 0 0 1-2 2H4a2 2 0 0 1-2-2V2a2 2 0 0 1 2-2zm0 1a1 1 0 0 0-1 1v12a1 1 0 0 0 1 1h8a1 1 0 0 0 1-1V2a1 1 0 0 0-1-1H4z"/>
                                </svg>
                                Descargar PDF
                            </button>
                            <button type="submit" name="formato" value="excel" class="btn btn-success btn-sm flex-fill" <?php echo empty($datos_reporte) ? 'disabled' : ''; ?>>
                                <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" fill="currentColor" class="bi bi-file-excel me-1" viewBox="0 0 16 16">
                                    <path d="M5.18 4.616a.5.5 0 0 1 .704.064L8 7.203l2.116-2.523a.5.5 0 1 1 .768.641L8.651 8l2.233 2.68a.5.5 0 0 1-.768.64L8 8.796l-2.116 2.523a.5.5 0 0 1-.768-.641L7.349 8 5.116 5.32a.5.5 0 0 1 .064-.704z"/>
                                    <path d="M4 0a2 2 0 0 0-2 2v12a2 2 0 0 0 2 2h8a2 2 0 0 0 2-2V2a2 2 0 0 0-2-2H4zm0 1h8a1 1 0 0 1 1 1v12a1 1 0 0 1-1 1H4a1 1 0 0 1-1-1V2a1 1 0 0 1 1-1z"/>
                                </svg>
                                Descargar Excel
                            </button>
                        </div>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Tarjeta de Resultados -->
    <?php if (!empty($datos_reporte)): ?>
    <div class="card">
        <div class="card-header bg-success text-white py-2 py-md-3 d-flex justify-content-between align-items-center">
            <h4 class="mb-0">Resultados del Reporte</h4>
            <span class="badge bg-light text-dark"><?= $estadisticas['total_estudiantes'] ?> estudiante(s) - <?= $estadisticas['total_asignaturas'] ?> asignatura(s)</span>
        </div>
        <div class="card-body p-2 p-md-3">
            <!-- Estadísticas -->
            <div class="alert alert-info mb-2 mb-md-3 p-2">
                <strong class="small">Estadísticas:</strong>
                <div class="row mt-1">
                    <div class="col-6 col-md-4 col-lg-2">
                        <span class="badge bg-primary">Total: <?= $estadisticas['total_asignaturas'] ?></span>
                    </div>
                    <div class="col-6 col-md-4 col-lg-2">
                        <span class="badge bg-success">Con Notas: <?= $estadisticas['con_notas'] ?></span>
                    </div>
                    <div class="col-6 col-md-4 col-lg-2">
                        <span class="badge bg-warning">Sin Notas: <?= $estadisticas['sin_notas'] ?></span>
                    </div>
                    <div class="col-6 col-md-4 col-lg-2">
                        <span class="badge bg-info">Aprobados: <?= $estadisticas['aprobados'] ?></span>
                    </div>
                    <div class="col-6 col-md-4 col-lg-2">
                        <span class="badge bg-danger">Reprobados: <?= $estadisticas['reprobados'] ?></span>
                    </div>
                    <div class="col-6 col-md-4 col-lg-2">
                        <span class="badge bg-secondary">Estudiantes: <?= $estadisticas['total_estudiantes'] ?></span>
                    </div>
                </div>
            </div>

            <!-- Información de filtros aplicados -->
            <div class="row mb-3">
                <div class="col-md-6">
                    <div class="card">
                        <div class="card-body py-2">
                            <small class="text-muted">
                                <strong>Postgrado:</strong> <?= $postgrados[array_search($postgrado_seleccionado, array_column($postgrados, 'id'))]['nombre'] ?><br>
                                <?php if (!empty($anio_lectivo)): ?>
                                    <strong>Año Lectivo:</strong> <?= $anio_lectivo ?><br>
                                <?php endif; ?>
                                <?php if (!empty($cedula_busqueda)): ?>
                                    <strong>Documento:</strong> <?= htmlspecialchars($cedula_busqueda) ?><br>
                                <?php endif; ?>
                            </small>
                        </div>
                    </div>
                </div>
                <div class="col-md-6">
                    <div class="card">
                        <div class="card-body py-2">
                            <small class="text-muted">
                                <?php if (!empty($trimestre)): 
                                    $trimestre_texto = '';
                                    switch ($trimestre) {
                                        case '1': $trimestre_texto = '1º Cuatrimestre (Ene-Abr)'; break;
                                        case '2': $trimestre_texto = '2º Cuatrimestre (May-Ago)'; break;
                                        case '3': $trimestre_texto = '3º Cuatrimestre (Sep-Dic)'; break;
                                    }
                                ?>
                                    <strong>Periodo:</strong> <?= $trimestre_texto ?> <?= $anio_trimestre ?><br>
                                <?php endif; ?>
                                <strong>Fecha Generación:</strong> <?= date('d/m/Y H:i:s') ?><br>
                                <strong>Generado por:</strong> <?= htmlspecialchars($_SESSION['username']) ?>
                            </small>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Tabla de resultados -->
            <div class="table-responsive-container">
                <div class="table-responsive">
                    <table class="table table-striped table-hover table-sm mb-2">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Estudiante</th>
                                <th>Documento</th>
                                <th>Asignatura</th>
                                <th>Denominación</th>
                                <th>Créditos</th>
                                <th>Nota Definitiva</th>
                                <th>Nota Final</th>
                                <th>Escolaridad</th>
                                <th>Fecha Modificación</th>
                                <th>Modificado Por</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php $contador = 1; ?>
                            <?php foreach ($datos_reporte as $fila): 
                                $numero_documento = obtener_numero_documento(
                                    $fila['tipo_documento'], 
                                    $fila['cedula_venezolana'], 
                                    $fila['cedula_extranjera'], 
                                    $fila['pasaporte']
                                );
                                $sin_nota = $fila['nota_definitiva'] === null;
                            ?>
                                <tr class="<?= $sin_nota ? 'table-warning' : '' ?>">
                                    <td><?= $contador ?></td>
                                    <td>
                                        <span class="fw-bold"><?= htmlspecialchars($fila['apellidos'] . ', ' . $fila['nombres']) ?></span>
                                    </td>
                                    <td>
                                        <small><?= htmlspecialchars($numero_documento) ?></small>
                                    </td>
                                    <td><?= htmlspecialchars($fila['codigo']) ?></td>
                                    <td><?= htmlspecialchars($fila['denominacion']) ?></td>
                                    <td class="text-center"><?= htmlspecialchars($fila['credito']) ?></td>
                                    <td class="text-center fw-bold <?= $sin_nota ? 'text-warning' : '' ?>">
                                        <?= htmlspecialchars($fila['nota_definitiva'] ?? 'PENDIENTE') ?>
                                    </td>
                                    <td class="text-center">
                                        <?= htmlspecialchars($fila['nota_final'] ?? 'PENDIENTE') ?>
                                    </td>
                                    <td>
                                        <span class="badge bg-secondary"><?= htmlspecialchars($fila['escolaridad']) ?></span>
                                    </td>
                                    <td>
                                        <small class="text-muted"><?= formatearFecha($fila['fecha_modificacion']) ?></small>
                                    </td>
                                    <td>
                                        <small class="text-muted">
                                            <?= htmlspecialchars($fila['modificado_por_nombres'] . ' ' . $fila['modificado_por_apellidos'] ?? 'Sistema') ?>
                                        </small>
                                    </td>
                                </tr>
                                <?php $contador++; ?>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
    <?php elseif (!empty($postgrado_seleccionado)): ?>
        <div class="card">
            <div class="card-body text-center py-4">
                <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" fill="currentColor" class="bi bi-search text-muted mb-2" viewBox="0 0 16 16">
                    <path d="M11.742 10.344a6.5 6.5 0 1 0-1.397 1.398h-.001c.03.04.062.078.098.115l3.85 3.85a1 1 0 0 0 1.415-1.414l-3.85-3.85a1.007 1.007 0 0 0-.115-.1zM12 6.5a5.5 5.5 0 1 1-11 0 5.5 5.5 0 0 1 11 0z"/>
                </svg>
                <h5 class="text-muted">No se encontraron datos</h5>
                <p class="text-muted small">
                    No hay registros de notas que coincidan con los filtros aplicados.
                </p>
            </div>
        </div>
    <?php endif; ?>
</div>

<style>
    .container-fluid {
        max-width: 100%;
        overflow-x: hidden;
    }
    
    .card {
        border: 1px solid #dee2e6;
        border-radius: 8px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        margin-bottom: 15px;
    }
    
    .card-header {
        border-radius: 8px 8px 0 0 !important;
        font-weight: 600;
    }
    
    .table-responsive-container {
        position: relative;
        width: 100%;
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
        border: 1px solid #dee2e6;
        border-radius: 6px;
    }
    
    .table-responsive {
        min-width: 1200px;
        margin-bottom: 0;
    }
    
    .table th {
        background-color: #2c3e50;
        color: white;
        font-weight: 600;
        white-space: nowrap;
        font-size: 0.875rem;
        padding: 8px 12px;
    }
    
    .table td {
        vertical-align: middle;
        padding: 8px 12px;
        font-size: 0.875rem;
    }
    
    .btn-primary {
        background-color: #007bff;
        border-color: #007bff;
        font-size: 0.8rem;
    }
    
    .btn-primary:hover {
        background-color: #0056b3;
        border-color: #0056b3;
    }
    
    .badge {
        font-size: 0.7em;
        font-weight: 600;
    }
    
    .form-label {
        font-weight: 500;
        color: #2c3e50;
        font-size: 0.9rem;
    }
    
    .alert {
        border: none;
        border-radius: 6px;
        font-size: 0.875rem;
    }
    
    .table-warning {
        background-color: #fff3cd !important;
    }
    
    /* Scroll horizontal suave */
    .table-responsive-container::-webkit-scrollbar {
        height: 8px;
    }
    
    .table-responsive-container::-webkit-scrollbar-track {
        background: #f1f1f1;
        border-radius: 4px;
    }
    
    .table-responsive-container::-webkit-scrollbar-thumb {
        background: #c1c1c1;
        border-radius: 4px;
    }
    
    .table-responsive-container::-webkit-scrollbar-thumb:hover {
        background: #a8a8a8;
    }
</style>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Mejorar la experiencia táctil en móviles
    const tableContainer = document.querySelector('.table-responsive-container');
    if (tableContainer) {
        let startX;
        let scrollLeft;
        
        tableContainer.addEventListener('touchstart', (e) => {
            startX = e.touches[0].pageX - tableContainer.offsetLeft;
            scrollLeft = tableContainer.scrollLeft;
        });
        
        tableContainer.addEventListener('touchmove', (e) => {
            if (!startX) return;
            const x = e.touches[0].pageX - tableContainer.offsetLeft;
            const walk = (x - startX) * 2;
            tableContainer.scrollLeft = scrollLeft - walk;
        });
    }

    // Habilitar/deshabilitar botones de exportación
    const form = document.querySelector('form');
    const pdfBtn = form.querySelector('button[value="pdf"]');
    const excelBtn = form.querySelector('button[value="excel"]');
    
    form.addEventListener('submit', function() {
        // Los botones ya están controlados por PHP, pero podemos agregar validación adicional
        const postgrado = document.getElementById('postgrado_id').value;
        if (!postgrado) {
            alert('Por favor seleccione un postgrado.');
            return false;
        }
    });
});
</script>

<?php
// 7. Incluir footer
include '../includes/footer.php';
$conn->close();
?>