<?php
session_start();

if (!isset($_SESSION['username']) || $_SESSION['rol'] != 'administrador') {
    header("Location: ../login.php");
    exit();
}

include '../includes/db.php';
$conn->set_charset("utf8mb4");

$usuario_id = isset($_GET['usuario_id']) ? intval($_GET['usuario_id']) : 0;
$anio_lectivo = isset($_GET['anio_lectivo']) ? $_GET['anio_lectivo'] : '';
$cuatrimestre = isset($_GET['cuatrimestre']) ? $_GET['cuatrimestre'] : '';

if ($usuario_id <= 0) {
    die("Usuario no válido");
}

// Obtener información del usuario
$sql_usuario = "SELECT nombres, apellidos, username, rol FROM usuarios WHERE id = ?";
$stmt_usuario = $conn->prepare($sql_usuario);
$stmt_usuario->bind_param("i", $usuario_id);
$stmt_usuario->execute();
$usuario_info = $stmt_usuario->get_result()->fetch_assoc();

if (!$usuario_info) {
    die("Usuario no encontrado");
}

// Obtener cargas del usuario con filtros
$sql_cargas = "SELECT 
                i.fecha_modificacion,
                i.nota_definitiva,
                i.veces_modificada,
                e.nombres as estudiante_nombre,
                e.apellidos as estudiante_apellido,
                d.nombre as asignatura_nombre,
                d.codigo as asignatura_codigo,
                p.nombre as postgrado_nombre,
                esc.nombre as escolaridad_nombre,
                al.nombre as año_lectivo
                FROM inscripciones i
                JOIN estudiantes e ON i.estudiante_id = e.id
                JOIN denominaciones d ON i.denominacion_id = d.id
                JOIN postgrados p ON d.postgrado_id = p.id
                JOIN escolaridades esc ON i.escolaridad_id = esc.id
                LEFT JOIN años_lectivos al ON i.año_lectivo_id = al.id
                WHERE i.usuario_modificacion = ?";

$params = [$usuario_id];
$types = "i";

if (!empty($anio_lectivo)) {
    $sql_cargas .= " AND al.nombre LIKE ?";
    $params[] = $anio_lectivo . '%';
    $types .= "s";
}

if (!empty($cuatrimestre) && $cuatrimestre != 'TODOS') {
    $sql_cargas .= " AND al.nombre LIKE ?";
    $params[] = '%' . $cuatrimestre;
    $types .= "s";
}

$sql_cargas .= " ORDER BY i.fecha_modificacion DESC";

$stmt_cargas = $conn->prepare($sql_cargas);
$stmt_cargas->bind_param($types, ...$params);
$stmt_cargas->execute();
$cargas = $stmt_cargas->get_result()->fetch_all(MYSQLI_ASSOC);

// Estadísticas
$total_cargas = count($cargas);
$notas_cargadas = 0;
$notas_pendientes = 0;

foreach ($cargas as $carga) {
    if (!empty($carga['nota_definitiva'])) {
        $notas_cargadas++;
    } else {
        $notas_pendientes++;
    }
}

$page_title = "Detalle de Cargas - " . $usuario_info['nombres'];
include '../includes/header.php';
?>

<div class="container-fluid">
    <div class="mb-3">
        <a href="monitoreo_cargas.php" class="btn btn-secondary btn-sm">
            <i class="bi bi-arrow-left"></i> Volver al Monitoreo
        </a>
    </div>

    <div class="card mb-4">
        <div class="card-header bg-primary text-white">
            <h4 class="mb-0">
                <i class="bi bi-person"></i> 
                Detalle de Cargas: <?= htmlspecialchars($usuario_info['nombres'] . ' ' . $usuario_info['apellidos']) ?>
            </h4>
        </div>
        <div class="card-body">
            <div class="row">
                <div class="col-md-3">
                    <div class="card bg-light">
                        <div class="card-body text-center">
                            <h5 class="card-title">Total Cargas</h5>
                            <h2 class="text-primary"><?= $total_cargas ?></h2>
                        </div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="card bg-success text-white">
                        <div class="card-body text-center">
                            <h5 class="card-title">Notas Cargadas</h5>
                            <h2><?= $notas_cargadas ?></h2>
                        </div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="card bg-warning">
                        <div class="card-body text-center">
                            <h5 class="card-title">Notas Pendientes</h5>
                            <h2><?= $notas_pendientes ?></h2>
                        </div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="card bg-info text-white">
                        <div class="card-body text-center">
                            <h5 class="card-title">Porcentaje</h5>
                            <h2><?= $total_cargas > 0 ? round(($notas_cargadas / $total_cargas) * 100, 2) : 0 ?>%</h2>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Filtros aplicados -->
            <?php if (!empty($anio_lectivo) || !empty($cuatrimestre)): ?>
                <div class="mt-3 p-2 bg-light rounded">
                    <strong>Filtros aplicados:</strong>
                    <?php 
                    $filtros = [];
                    if (!empty($anio_lectivo)) $filtros[] = "Año: " . htmlspecialchars($anio_lectivo);
                    if (!empty($cuatrimestre) && $cuatrimestre != 'TODOS') {
                        $cuatrimestres_nombres = [
                            '01C' => '1º Cuatrimestre',
                            '02C' => '2º Cuatrimestre', 
                            '03C' => '3º Cuatrimestre'
                        ];
                        $filtros[] = "Cuatrimestre: " . $cuatrimestres_nombres[$cuatrimestre];
                    }
                    echo implode(' | ', $filtros);
                    ?>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <div class="card">
        <div class="card-header bg-info text-white">
            <h5 class="mb-0"><i class="bi bi-list-check"></i> Historial de Cargas</h5>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-striped table-hover">
                    <thead>
                        <tr>
                            <th>Fecha/Hora</th>
                            <th>Estudiante</th>
                            <th>Asignatura</th>
                            <th>Postgrado</th>
                            <th>Escolaridad</th>
                            <th>Año/Cuatrimestre</th>
                            <th>Nota</th>
                            <th>Modificaciones</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($cargas as $carga): ?>
                            <tr>
                                <td><?= date('d/m/Y H:i', strtotime($carga['fecha_modificacion'])) ?></td>
                                <td><?= htmlspecialchars($carga['estudiante_nombre'] . ' ' . $carga['estudiante_apellido']) ?></td>
                                <td>
                                    <small class="text-muted"><?= htmlspecialchars($carga['asignatura_codigo']) ?></small><br>
                                    <?= htmlspecialchars($carga['asignatura_nombre']) ?>
                                </td>
                                <td><?= htmlspecialchars($carga['postgrado_nombre']) ?></td>
                                <td>
                                    <span class="badge bg-info"><?= htmlspecialchars($carga['escolaridad_nombre']) ?></span>
                                </td>
                                <td><?= htmlspecialchars($carga['año_lectivo']) ?></td>
                                <td>
                                    <span class="badge bg-<?= !empty($carga['nota_definitiva']) ? 'success' : 'warning' ?>">
                                        <?= !empty($carga['nota_definitiva']) ? htmlspecialchars($carga['nota_definitiva']) : 'PENDIENTE' ?>
                                    </span>
                                </td>
                                <td>
                                    <span class="badge bg-<?= $carga['veces_modificada'] > 1 ? 'warning' : 'secondary' ?>">
                                        <?= $carga['veces_modificada'] ?> vez(es)
                                    </span>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<?php
include '../includes/footer.php';
$conn->close();
?>