<?php
session_start();

// Definir la ruta base
define('BASE_PATH', dirname(__FILE__));

// Incluir archivos con rutas absolutas
include BASE_PATH . '/includes/db.php';

// Verificar conexión a la base de datos
if ($conn->connect_error) {
    die("Error de conexión a la base de datos: " . $conn->connect_error);
}

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

// Cargar PHPMailer
require BASE_PATH . '/vendor/autoload.php';
include BASE_PATH . '/includes/header.php';

// 1. OBTENER CONFIGURACIÓN DE LA BASE DE DATOS
$config = [];
$sql_config = "SELECT setting_key, setting_value FROM app_settings WHERE setting_key IN ('student_registration_enabled', 'verified_students_only')";
$result_config = $conn->query($sql_config);
if ($result_config) {
    while ($row = $result_config->fetch_assoc()) {
        $config[$row['setting_key']] = $row['setting_value'];
    }
}

$registro_habilitado = isset($config['student_registration_enabled']) && $config['student_registration_enabled'] == '1';
$registro_verificado_solo = isset($config['verified_students_only']) && $config['verified_students_only'] == '1';

// 2. VERIFICAR SI EL REGISTRO ESTÁ HABILITADO
if (!$registro_habilitado) {
    echo '<div class="container mt-5"><div class="alert alert-warning">El registro de nuevos estudiantes se encuentra deshabilitado por el administrador.</div></div>';
    include BASE_PATH . '/includes/footer.php';
    exit();
}

$step = 'verify'; // Pasos: verify, register
$error_message = '';
$success_message = '';
$student_data = [];

// 3. LÓGICA DE PROCESAMIENTO DEL FORMULARIO
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // --- PASO 1: VERIFICACIÓN DE CÉDULA ---
    if (isset($_POST['verify_student'])) {
        $cedula = trim($_POST['cedula']);
        
        if (empty($cedula)) {
            $error_message = "Por favor, ingresa tu número de cédula o pasaporte.";
        } else {
            // Verificar si el estudiante ya tiene un usuario
            $sql_user = "SELECT u.id FROM usuarios u JOIN estudiantes e ON u.username = e.cedula WHERE e.cedula = ?";
            $stmt_user = $conn->prepare($sql_user);
            
            if ($stmt_user === false) {
                $error_message = "Error en la consulta de verificación: " . $conn->error;
            } else {
                $stmt_user->bind_param('s', $cedula);
                $stmt_user->execute();
                $result_user = $stmt_user->get_result();

                if ($result_user->num_rows > 0) {
                    $error_message = "Ya existe un usuario registrado con esta cédula. Si olvidaste tu contraseña, contacta al administrador.";
                } else {
                    // Buscar al estudiante en la tabla de estudiantes
                    $sql_student = "SELECT * FROM estudiantes WHERE cedula = ?";
                    $stmt_student = $conn->prepare($sql_student);
                    
                    if ($stmt_student === false) {
                        $error_message = "Error en la consulta de estudiante: " . $conn->error;
                    } else {
                        $stmt_student->bind_param('s', $cedula);
                        $stmt_student->execute();
                        $result_student = $stmt_student->get_result();

                        if ($result_student->num_rows > 0) {
                            $student_data = $result_student->fetch_assoc();
                            $step = 'register'; // Estudiante encontrado, pasar al registro
                        } else {
                            $error_message = "Tu cédula no se encuentra en nuestra base de datos de estudiantes verificados. Para registrarte, debes estar previamente inscrito en el sistema.";
                        }
                        $stmt_student->close();
                    }
                }
                $stmt_user->close();
            }
        }
    }
    // --- PASO 2: REGISTRO DE USUARIO ---
    elseif (isset($_POST['register_user'])) {
        $step = 'register';
        $cedula = trim($_POST['cedula']);
        $password = $_POST['password'];
        $password_confirm = $_POST['password_confirm'];
        $student_id = intval($_POST['student_id']);

        // Validaciones básicas
        if (empty($password) || empty($password_confirm)) {
            $error_message = "Por favor, completa ambos campos de contraseña.";
        } elseif ($password !== $password_confirm) {
            $error_message = "Las contraseñas no coinciden.";
        } elseif (strlen($password) < 8) {
            $error_message = "La contraseña debe tener al menos 8 caracteres.";
        } else {
            $conn->begin_transaction();
            try {
                // VERIFICAR QUE EL ESTUDIANTE EXISTA EN LA TABLA ESTUDIANTES
                $sql_verify = "SELECT * FROM estudiantes WHERE id = ? AND cedula = ?";
                $stmt_verify = $conn->prepare($sql_verify);
                
                if ($stmt_verify === false) {
                    throw new Exception("Error en la consulta de verificación: " . $conn->error);
                }
                
                $stmt_verify->bind_param('is', $student_id, $cedula);
                $stmt_verify->execute();
                $result_verify = $stmt_verify->get_result();

                if ($result_verify->num_rows === 0) {
                    throw new Exception("Error de verificación: Los datos del estudiante no coinciden.");
                }

                $verified_student = $result_verify->fetch_assoc();
                $stmt_verify->close();

                // USAR LOS DATOS VERIFICADOS DE LA TABLA ESTUDIANTES
                $nombres = $verified_student['nombres'];
                $apellidos = $verified_student['apellidos'];
                $correo = $verified_student['correo'];
                $telefono = $verified_student['telefono'] ?? '';
                $direccion = $verified_student['direccion'] ?? '';

                // Verificar que el correo esté presente
                if (empty($correo)) {
                    throw new Exception("No se puede completar el registro. El correo electrónico no está registrado en el sistema. Contacta al administrador.");
                }

                // Crear el usuario con los datos VERIFICADOS
                $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                $token = bin2hex(random_bytes(50));
                $expiracion = date('Y-m-d H:i:s', strtotime('+1 day'));
                $rol = 'estudiante';
                $estado_cuenta = 'pendiente';

                // Primero verificar si la tabla usuarios tiene la columna 'correo' o 'email'
                $sql_check_columns = "SHOW COLUMNS FROM usuarios LIKE 'correo'";
                $result_check = $conn->query($sql_check_columns);
                $has_correo_column = ($result_check->num_rows > 0);
                
                if ($has_correo_column) {
                    // Si existe columna 'correo'
                    $sql_create_user = "INSERT INTO usuarios (username, password, rol, correo, telefono, direccion, estado_cuenta, token_activacion, token_expiracion, estudiante_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
                    $stmt_create_user = $conn->prepare($sql_create_user);
                    if ($stmt_create_user === false) {
                        throw new Exception("Error al preparar consulta de usuario: " . $conn->error);
                    }
                    $stmt_create_user->bind_param('sssssssssi', $cedula, $hashed_password, $rol, $correo, $telefono, $direccion, $estado_cuenta, $token, $expiracion, $student_id);
                } else {
                    // Si no existe columna 'correo', usar 'email'
                    $sql_create_user = "INSERT INTO usuarios (username, password, rol, email, telefono, direccion, estado_cuenta, token_activacion, token_expiracion, estudiante_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
                    $stmt_create_user = $conn->prepare($sql_create_user);
                    if ($stmt_create_user === false) {
                        throw new Exception("Error al preparar consulta de usuario: " . $conn->error);
                    }
                    $stmt_create_user->bind_param('sssssssssi', $cedula, $hashed_password, $rol, $correo, $telefono, $direccion, $estado_cuenta, $token, $expiracion, $student_id);
                }
                
                if (!$stmt_create_user->execute()) {
                    throw new Exception("Error al crear el usuario: " . $stmt_create_user->error);
                }
                $stmt_create_user->close();

                $conn->commit();

                // ==================== CONFIGURACIÓN Y ENVÍO DE CORREO ====================
                include BASE_PATH . '/config/mail_config.php';
                
                // Verificar que la configuración esté completa
                if (!verificar_configuracion_correo()) {
                    throw new Exception("Configuración de correo incompleta. Contacta al administrador.");
                }

                // Enviar correo de activación
                $mail_sent = false;
                $mail_error = '';
                
                try {
                    $mail = new PHPMailer(true);
                    
                    // ==================== CONFIGURACIÓN MEJORADA DEL SERVIDOR ====================
                    $mail->isSMTP();
                    
                    // Configuración principal
                    $mail->Host       = MAIL_HOST;
                    $mail->SMTPAuth   = MAIL_SMTP_AUTH;
                    $mail->Username   = MAIL_USERNAME;
                    $mail->Password   = MAIL_PASSWORD;
                    $mail->SMTPSecure = MAIL_ENCRYPTION;
                    $mail->Port       = MAIL_PORT;
                    $mail->CharSet    = 'UTF-8';
                    
                    // ==================== CONFIGURACIONES CRÍTICAS PARA EVITAR TIMEOUT ====================
                    $mail->SMTPDebug   = MAIL_DEBUG;
                    $mail->Timeout     = MAIL_TIMEOUT;
                    $mail->SMTPKeepAlive = MAIL_SMTP_KEEPALIVE;
                    
                    // Configuraciones adicionales de seguridad
                    $mail->SMTPOptions = array(
                        'ssl' => array(
                            'verify_peer' => false,
                            'verify_peer_name' => false,
                            'allow_self_signed' => true
                        )
                    );
                    
                    // ==================== SISTEMA DE REINTENTOS CON CONFIGURACIÓN ALTERNATIVA ====================
                    $configuraciones = [
                        ['port' => MAIL_PORT, 'encryption' => MAIL_ENCRYPTION, 'name' => 'SSL/Puerto 465'],
                        ['port' => MAIL_ALT_PORT, 'encryption' => MAIL_ALT_ENCRYPTION, 'name' => 'TLS/Puerto 587']
                    ];
                    
                    $conexion_exitosa = false;
                    $ultimo_error = '';
                    
                    foreach ($configuraciones as $config) {
                        try {
                            // Configurar puerto y encriptación
                            $mail->Port = $config['port'];
                            $mail->SMTPSecure = $config['encryption'];
                            
                            // Limpiar destinatarios anteriores
                            $mail->clearAddresses();
                            $mail->clearReplyTos();
                            
                            // ==================== CONFIGURACIÓN DEL REMITENTE Y DESTINATARIO ====================
                            $mail->setFrom(MAIL_FROM_EMAIL, MAIL_FROM_NAME);
                            $mail->addAddress($correo, $nombres . ' ' . $apellidos);
                            $mail->addReplyTo(SUPPORT_EMAIL, 'Soporte del Sistema');

                            // ==================== CONTENIDO DEL CORREO ====================
                            $mail->isHTML(true);
                            $mail->Subject = 'Activa tu cuenta - Sistema de Postgrado Medicina UCV';
                            
                            // Enlace de activación
                            $activation_link = SITE_URL . "/activar_cuenta.php?token=" . $token;
                            
                            // Contenido HTML del correo
                            $mail->Body = "
                            <!DOCTYPE html>
                            <html lang='es'>
                            <head>
                                <meta charset='UTF-8'>
                                <meta name='viewport' content='width=device-width, initial-scale=1.0'>
                                <title>Activación de Cuenta</title>
                                <style>
                                    body { 
                                        font-family: 'Arial', sans-serif; 
                                        line-height: 1.6; 
                                        color: #333; 
                                        margin: 0; 
                                        padding: 0; 
                                        background-color: #f4f4f4;
                                    }
                                    .container { 
                                        max-width: 600px; 
                                        margin: 0 auto; 
                                        background: #ffffff; 
                                        padding: 20px; 
                                        border-radius: 10px; 
                                        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
                                    }
                                    .header { 
                                        background: linear-gradient(135deg, #007bff, #0056b3); 
                                        color: white; 
                                        padding: 20px; 
                                        text-align: center; 
                                        border-radius: 10px 10px 0 0;
                                    }
                                    .content { 
                                        padding: 30px 20px; 
                                    }
                                    .button { 
                                        display: inline-block; 
                                        background: #007bff; 
                                        color: white; 
                                        padding: 12px 30px; 
                                        text-decoration: none; 
                                        border-radius: 5px; 
                                        font-weight: bold; 
                                        margin: 20px 0;
                                    }
                                    .footer { 
                                        text-align: center; 
                                        padding: 20px; 
                                        color: #666; 
                                        font-size: 12px; 
                                        border-top: 1px solid #eee;
                                    }
                                    .alert { 
                                        background: #fff3cd; 
                                        border: 1px solid #ffeaa7; 
                                        padding: 15px; 
                                        border-radius: 5px; 
                                        margin: 15px 0;
                                    }
                                    .code { 
                                        background: #f8f9fa; 
                                        padding: 15px; 
                                        border-radius: 5px; 
                                        border: 1px solid #e9ecef; 
                                        word-break: break-all; 
                                        font-family: monospace;
                                    }
                                </style>
                            </head>
                            <body>
                                <div class='container'>
                                    <div class='header'>
                                        <h1>¡Bienvenido al Sistema de Postgrado!</h1>
                                    </div>
                                    
                                    <div class='content'>
                                        <p>Hola <strong>{$nombres} {$apellidos}</strong>,</p>
                                        
                                        <p>Gracias por registrarte en nuestro Sistema de Postgrado. Estás a un paso de activar tu cuenta y acceder a todos nuestros servicios.</p>
                                        
                                        <p style='text-align: center;'>
                                            <a href='{$activation_link}' class='button'>Activar Mi Cuenta</a>
                                        </p>
                                        
                                        <p>Si el botón no funciona, copia y pega el siguiente enlace en tu navegador:</p>
                                        
                                        <div class='code'>{$activation_link}</div>
                                        
                                        <div class='alert'>
                                            <strong>⚠️ Importante:</strong> 
                                            <ul>
                                                <li>Este enlace expirará en <strong>24 horas</strong></li>
                                                <li>Si no activas tu cuenta en este tiempo, deberás solicitar un nuevo enlace</li>
                                                <li>Guarda este correo para futuras referencias</li>
                                            </ul>
                                        </div>
                                        
                                        <p>Si tienes problemas para activar tu cuenta o no solicitaste este registro, por favor contacta a nuestro equipo de soporte inmediatamente.</p>
                                        
                                        <p>¡Estamos emocionados de tenerte con nosotros!</p>
                                    </div>
                                    
                                    <div class='footer'>
                                        <p><strong>Sistema de Postgrado - Medicina UCV</strong></p>
                                        <p>Este es un mensaje automático, por favor no respondas a este correo.</p>
                                        <p>Si necesitas ayuda, contacta a: <a href='mailto:" . SUPPORT_EMAIL . "'>" . SUPPORT_EMAIL . "</a></p>
                                    </div>
                                </div>
                            </body>
                            </html>";

                            // Versión alternativa en texto plano
                            $mail->AltBody = "ACTIVACIÓN DE CUENTA - SISTEMA DE POSTGRADO MEDICINA UCV\n\n" .
                                            "Hola {$nombres} {$apellidos},\n\n" .
                                            "Gracias por registrarte en el Sistema de Postgrado.\n\n" .
                                            "Para activar tu cuenta, por favor visita el siguiente enlace:\n" .
                                            "{$activation_link}\n\n" .
                                            "Este enlace expirará en 24 horas.\n\n" .
                                            "Si no solicitaste este registro, ignora este mensaje.\n\n" .
                                            "Saludos cordiales,\n" .
                                            "Sistema de Postgrado - Medicina UCV\n" .
                                            "Soporte: " . SUPPORT_EMAIL;

                            // ==================== INTENTAR ENVÍO ====================
                            error_log("Intentando enviar correo con configuración: " . $config['name']);
                            
                            if ($mail->send()) {
                                $mail_sent = true;
                                error_log("✅ Correo enviado EXITOSAMENTE a: " . $correo . " usando " . $config['name']);
                                break;
                            }
                            
                        } catch (Exception $e) {
                            $ultimo_error = "Configuración " . $config['name'] . " falló: " . $e->getMessage();
                            error_log("❌ " . $ultimo_error);
                        }
                    }
                    
                    // Si ninguna configuración funcionó
                    if (!$mail_sent) {
                        $mail_error = $ultimo_error ?: "Todas las configuraciones fallaron";
                        error_log("❌ TODAS las configuraciones de correo fallaron. Último error: " . $mail_error);
                    }

                } catch (Exception $e) {
                    $mail_error = $e->getMessage();
                    error_log("❌ Excepción general en PHPMailer: " . $mail_error);
                }

                // ==================== MENSAJES DE RESPUESTA ====================
                if ($mail_sent) {
                    $success_message = '¡Registro exitoso! Se ha enviado un correo de activación a <strong>' . htmlspecialchars($correo) . '</strong>. ' .
                                      'Por favor, revisa tu bandeja de entrada (y la carpeta de spam) para activar tu cuenta. ' .
                                      'El enlace de activación expirará en 24 horas.';
                } else {
                    $success_message = "¡Registro exitoso! Sin embargo, no se pudo enviar el correo de activación automáticamente. " .
                                      "Error: " . htmlspecialchars($mail_error) . ". " .
                                      "Por favor, contacta al administrador en " . SUPPORT_EMAIL . " para activar tu cuenta manualmente.";
                }

                echo '<div class="container mt-5">
                        <div class="alert alert-success">' . $success_message . ' 
                            <a href="login.php" class="btn btn-primary mt-3">Ir a Iniciar Sesión</a>
                        </div>
                      </div>';
                include BASE_PATH . '/includes/footer.php';
                exit();

            } catch (Exception $e) {
                $conn->rollback();
                // Error de duplicado
                if ($conn->errno == 1062) {
                    $error_message = "Error: Ya existe un usuario registrado con esta cédula.";
                } else {
                    $error_message = "Error al registrar: " . $e->getMessage();
                }
                // Recargar datos del estudiante para mostrar en el formulario
                $sql_reload = "SELECT * FROM estudiantes WHERE id = ?";
                $stmt_reload = $conn->prepare($sql_reload);
                if ($stmt_reload) {
                    $stmt_reload->bind_param('i', $student_id);
                    $stmt_reload->execute();
                    $result_reload = $stmt_reload->get_result();
                    if ($result_reload->num_rows > 0) {
                        $student_data = $result_reload->fetch_assoc();
                    }
                    $stmt_reload->close();
                }
            }
        }
    }
}

?>

<div class="container mt-5">
    <h2>Registro de Nuevo Estudiante</h2>

    <?php if ($error_message): ?>
        <div class="alert alert-danger"><?php echo $error_message; ?></div>
    <?php endif; ?>

    <?php if ($step === 'verify'): ?>
    <!-- PASO 1: FORMULARIO DE VERIFICACIÓN -->
    <div class="card">
        <div class="card-header">Paso 1: Verificar Identidad</div>
        <div class="card-body">
            <p>Para registrarte en el sistema, debes estar previamente inscrito como estudiante.</p>
            <div class="alert alert-info">
                <strong>Importante:</strong> Introduce tu número de cédula o pasaporte exactamente como aparece en tus documentos oficiales.
            </div>
            <form method="POST" action="">
                <div class="mb-3">
                    <label for="cedula" class="form-label">Cédula o Pasaporte *</label>
                    <input type="text" class="form-control" id="cedula" name="cedula" required autofocus value="<?php echo isset($_POST['cedula']) ? htmlspecialchars($_POST['cedula']) : ''; ?>">
                    <div class="form-text">Debe coincidir exactamente con tu cédula registrada en el sistema.</div>
                </div>
                <button type="submit" name="verify_student" class="btn btn-primary">Verificar Identidad</button>
            </form>
        </div>
    </div>
    <?php endif; ?>

    <?php if ($step === 'register' && !empty($student_data)): ?>
    <!-- PASO 2: FORMULARIO DE REGISTRO -->
    <div class="card">
        <div class="card-header">Paso 2: Crear Cuenta de Usuario</div>
        <div class="card-body">
            <div class="alert alert-success">
                <strong>¡Verificación exitosa!</strong> Hemos encontrado tus datos en el sistema. 
                Ahora puedes crear tu cuenta de acceso.
            </div>

            <!-- MOSTRAR DATOS VERIFICADOS (SOLO LECTURA) -->
            <div class="card mb-4">
                <div class="card-header bg-light">
                    <h6 class="mb-0">Tus Datos Verificados</h6>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-4">
                            <strong>Cédula/Pasaporte:</strong><br>
                            <?php echo htmlspecialchars($student_data['cedula']); ?>
                        </div>
                        <div class="col-md-4">
                            <strong>Nombres:</strong><br>
                            <?php echo htmlspecialchars($student_data['nombres']); ?>
                        </div>
                        <div class="col-md-4">
                            <strong>Apellidos:</strong><br>
                            <?php echo htmlspecialchars($student_data['apellidos']); ?>
                        </div>
                    </div>
                    <div class="row mt-2">
                        <div class="col-md-6">
                            <strong>Correo Electrónico:</strong><br>
                            <?php echo htmlspecialchars($student_data['correo']); ?>
                        </div>
                        <div class="col-md-6">
                            <strong>Teléfono:</strong><br>
                            <?php echo htmlspecialchars($student_data['telefono'] ?? 'No registrado'); ?>
                        </div>
                    </div>
                    <?php if (!empty($student_data['direccion'])): ?>
                    <div class="row mt-2">
                        <div class="col-12">
                            <strong>Dirección:</strong><br>
                            <?php echo htmlspecialchars($student_data['direccion']); ?>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
            </div>

            <div class="alert alert-warning">
                <strong>Nota:</strong> Tus datos personales están verificados y no pueden ser modificados. 
                Si encuentras algún error, contacta al administrador del sistema.
            </div>

            <form method="POST" action="">
                <input type="hidden" name="student_id" value="<?php echo htmlspecialchars($student_data['id']); ?>">
                <input type="hidden" name="cedula" value="<?php echo htmlspecialchars($student_data['cedula']); ?>">
                
                <h5 class="mb-3">Crear Contraseña de Acceso</h5>

                <div class="row">
                    <div class="col-md-6 mb-3">
                        <label for="password" class="form-label">Contraseña *</label>
                        <input type="password" class="form-control" id="password" name="password" required minlength="8">
                        <div class="form-text">Mínimo 8 caracteres.</div>
                    </div>
                    <div class="col-md-6 mb-3">
                        <label for="password_confirm" class="form-label">Confirmar Contraseña *</label>
                        <input type="password" class="form-control" id="password_confirm" name="password_confirm" required minlength="8">
                    </div>
                </div>

                <div class="alert alert-info">
                    <small>
                        <strong>Información de seguridad:</strong> Tu cuenta será creada con los datos verificados del sistema. 
                        El correo de activación se enviará a <strong><?php echo htmlspecialchars($student_data['correo']); ?></strong>.
                        El enlace de activación expirará en 24 horas.
                    </small>
                </div>

                <button type="submit" name="register_user" class="btn btn-success">Crear Mi Cuenta</button>
                <a href="registro_estudiante.php" class="btn btn-secondary">Cancelar</a>
            </form>
        </div>
    </div>
    <?php endif; ?>

</div>

<?php include BASE_PATH . '/includes/footer.php'; ?>