<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Verificador de Entorno para PDFs</title>
    <style>
        body { font-family: Arial, sans-serif; line-height: 1.6; margin: 20px; }
        .success { color: green; font-weight: bold; }
        .error { color: red; font-weight: bold; }
        .warning { color: orange; font-weight: bold; }
        pre { background: #f4f4f4; padding: 10px; border-radius: 5px; }
        .section { margin-bottom: 30px; border-bottom: 1px solid #ccc; padding-bottom: 20px; }
    </style>
</head>
<body>
    <h1>Verificador de Entorno para Generación de PDFs</h1>
    
    <div class="section">
        <h2>1. Información del Servidor</h2>
        <?php
        echo '<p><strong>Servidor:</strong> ' . $_SERVER['SERVER_SOFTWARE'] . '</p>';
        echo '<p><strong>PHP Version:</strong> ' . phpversion() . '</p>';
        echo '<p><strong>Sistema Operativo:</strong> ' . php_uname() . '</p>';
        ?>
    </div>

    <div class="section">
        <h2>2. Extensiones PHP Requeridas</h2>
        <?php
        $required_extensions = [
            'gd' => 'Para manipulación de imágenes',
            'mbstring' => 'Para manejo de caracteres especiales',
            'dom' => 'Para manipulación de documentos XML/HTML',
            'openssl' => 'Para conexiones seguras',
            'zip' => 'Para compresión de archivos (algunos PDF)',
            'fileinfo' => 'Para identificación de tipos de archivo'
        ];

        foreach ($required_extensions as $ext => $description) {
            if (extension_loaded($ext)) {
                echo "<p class='success'>✓ $ext: Instalado ($description)</p>";
            } else {
                echo "<p class='error'>✗ $ext: FALTA ($description)</p>";
            }
        }
        ?>
    </div>

    <div class="section">
        <h2>3. Configuración PHP Crítica</h2>
        <?php
        $configs = [
            'memory_limit' => ['>=128M', '128M o más recomendado para PDFs'],
            'max_execution_time' => ['>=30', '30 segundos o más recomendado'],
            'upload_max_filesize' => ['>=2M', 'Para subir plantillas'],
            'post_max_size' => ['>=8M', 'Para enviar datos complejos']
        ];

        foreach ($configs as $key => $value) {
            $current = ini_get($key);
            $required = $value[0];
            $description = $value[1];
            
            // Comparación numérica
            $current_num = (int)$current;
            $required_num = (int)filter_var($required, FILTER_SANITIZE_NUMBER_INT);
            
            $operator = substr($required, 0, 2);
            
            $ok = false;
            if ($operator == '>=') {
                $ok = ($current_num >= $required_num);
            }
            
            if ($ok) {
                echo "<p class='success'>✓ $key: $current ($description)</p>";
            } else {
                echo "<p class='error'>✗ $key: $current (Se recomienda $required - $description)</p>";
            }
        }
        ?>
    </div>

    <div class="section">
        <h2>4. Prueba de Composer y Autoload</h2>
        <?php
        $composer_working = false;
        $vendor_dir = __DIR__.'/vendor/autoload.php';
        
        if (file_exists($vendor_dir)) {
            echo "<p class='success'>✓ Se encontró vendor/autoload.php</p>";
            
            try {
                require $vendor_dir;
                echo "<p class='success'>✓ Autoload de Composer cargado correctamente</p>";
                $composer_working = true;
                
                // Prueba para librerías comunes de PDF
                $pdf_libs = [
                    'Dompdf\Dompdf' => 'Dompdf',
                    'TCPDF' => 'TCPDF',
                    'mPDF' => 'mPDF'
                ];
                
                foreach ($pdf_libs as $class => $name) {
                    if (class_exists($class)) {
                        echo "<p class='success'>✓ $name detectado en vendor</p>";
                    }
                }
                
            } catch (Exception $e) {
                echo "<p class='error'>✗ Error al cargar autoload: " . htmlspecialchars($e->getMessage()) . "</p>";
            }
        } else {
            echo "<p class='warning'>No se encontró vendor/autoload.php (Composer no instalado o en otra ubicación)</p>";
        }
        ?>
    </div>

    <div class="section">
        <h2>5. Prueba de Escritura de Archivos</h2>
        <?php
        $test_dir = __DIR__.'/temp_pdf_test';
        $test_file = $test_dir.'/test_'.time().'.txt';
        
        if (!file_exists($test_dir)) {
            if (mkdir($test_dir, 0755)) {
                echo "<p class='success'>✓ Directorio temporal creado</p>";
            } else {
                echo "<p class='error'>✗ No se pudo crear directorio temporal (ver permisos)</p>";
            }
        }
        
        if (file_put_contents($test_file, 'Prueba de escritura')) {
            echo "<p class='success'>✓ Escritura de archivos funcionando</p>";
            unlink($test_file);
        } else {
            echo "<p class='error'>✗ No se pudo escribir archivo (ver permisos)</p>";
        }
        ?>
    </div>

    <div class="section">
        <h2>6. Información Completa de PHP</h2>
        <p><a href="phpinfo.php">Ver phpinfo() completo</a> (crea un archivo phpinfo.php con solo <?php phpinfo(); ?>)</p>
    </div>

    <div class="section">
        <h2>Resultado General</h2>
        <?php
        $all_ok = $composer_working && extension_loaded('gd') && extension_loaded('dom');
        
        if ($all_ok) {
            echo '<p class="success">✓ Tu entorno parece estar listo para generación de PDFs</p>';
        } else {
            echo '<p class="error">✗ Hay problemas en tu configuración que podrían afectar la generación de PDFs</p>';
            echo '<p>Recomendaciones:';
            echo '<ul>';
            
            if (!extension_loaded('gd')) {
                echo '<li>Solicitar al hosting que active la extensión GD</li>';
            }
            
            if (!extension_loaded('dom')) {
                echo '<li>Solicitar al hosting que active la extensión DOM</li>';
            }
            
            if (!$composer_working) {
                echo '<li>Verificar la instalación de Composer o copiar correctamente la carpeta vendor</li>';
            }
            
            echo '</ul></p>';
        }
        ?>
    </div>
</body>
</html>