<?php
/**
 * index.php - Página principal del sistema de postgrado
 * Redirige a los usuarios según su rol después de verificar la sesión
 */

// 1. Cargar la configuración central (sesión, rutas, bd, codificación)
require_once __DIR__ . '/includes/session_config.php';

// 2. Centralized authentication and session check
require_once ROOT_PATH . '/includes/auth_check.php';

// Verificar que el rol del usuario esté definido
if (!isset($_SESSION['rol'])) {
    error_log("Error: Rol de usuario no definido para " . $_SESSION['username']);
    session_destroy();
    header("Location: " . BASE_URL . "/login.php?error=invalid_role");
    exit();
}

// Mapeo de redirecciones según el rol
$redirect_map = [
    'administrador' => 'admin/index.php',
    'coordinador1' => 'coordinador1/index.php',
    'coordinador2' => 'coordinador2/index.php', 
    'secretaria' => 'secretaria/index.php',
    'estudiante' => 'estudiante/index.php'
];

// Obtener el rol del usuario
$user_role = $_SESSION['rol'];

// Verificar si el rol tiene una redirección definida
if (isset($redirect_map[$user_role])) {
    $redirect_path = $redirect_map[$user_role];
    
    // Verificar que el archivo de destino exista
    $target_file = ROOT_PATH . '/' . $redirect_path;
    if (!file_exists($target_file)) {
        error_log("Error: Archivo de destino no encontrado: " . $target_file);
        
        // Mostrar página de error amigable
        include_once ROOT_PATH . '/includes/header.php';
        ?>
        <div class="container mt-5">
            <div class="row justify-content-center">
                <div class="col-md-8">
                    <div class="card shadow-lg">
                        <div class="card-header bg-danger text-white">
                            <h3 class="card-title mb-0">
                                <i class="fas fa-exclamation-triangle me-2"></i>Error del Sistema
                            </h3>
                        </div>
                        <div class="card-body text-center py-5">
                            <i class="fas fa-cogs fa-5x text-muted mb-4"></i>
                            <h4 class="text-danger mb-3">Módulo no disponible</h4>
                            <p class="text-muted mb-4">
                                El módulo correspondiente a su rol (<strong><?php echo htmlspecialchars($user_role); ?></strong>) 
                                no se encuentra disponible en este momento.
                            </p>
                            <div class="alert alert-warning">
                                <small>
                                    <i class="fas fa-info-circle me-1"></i>
                                    Esto puede deberse a mantenimiento del sistema o configuración incompleta.
                                </small>
                            </div>
                            <div class="mt-4">
                                <a href="logout.php" class="btn btn-primary me-2">
                                    <i class="fas fa-sign-out-alt me-1"></i>Cerrar Sesión
                                </a>
                                <button onclick="window.location.reload()" class="btn btn-outline-secondary">
                                    <i class="fas fa-redo me-1"></i>Reintentar
                                </button>
                            </div>
                        </div>
                        <div class="card-footer text-muted text-center">
                            <small>Si el problema persiste, contacte al administrador del sistema.</small>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
        include_once ROOT_PATH . '/includes/footer.php';
        exit();
    }
    
    // Redirigir al destino correspondiente
    header("Location: " . BASE_URL . '/' . $redirect_path);
    exit();
    
} else {
    // Rol no válido
    error_log("Error: Rol de usuario no válido: " . $user_role . " para usuario: " . $_SESSION['username']);
    
    // Mostrar página de error para rol no válido
    include_once ROOT_PATH . '/includes/header.php';
    ?>
    <div class="container mt-5">
        <div class="row justify-content-center">
            <div class="col-md-8">
                <div class="card shadow-lg border-0">
                    <div class="card-header bg-warning text-dark">
                        <h3 class="card-title mb-0">
                            <i class="fas fa-user-slash me-2"></i>Rol No Reconocido
                        </h3>
                    </div>
                    <div class="card-body text-center py-5">
                        <i class="fas fa-user-times fa-5x text-warning mb-4"></i>
                        <h4 class="text-warning mb-3">Configuración de Usuario Incompleta</h4>
                        <p class="text-muted mb-4">
                            Su cuenta tiene un rol (<strong><?php echo htmlspecialchars($user_role); ?></strong>) 
                            que no está configurado en el sistema.
                        </p>
                        
                        <div class="row mt-4">
                            <div class="col-md-6 mb-3">
                                <div class="card h-100">
                                    <div class="card-body">
                                        <i class="fas fa-user-cog fa-2x text-primary mb-3"></i>
                                        <h5>Contactar Administrador</h5>
                                        <p class="small text-muted">
                                            Solicite al administrador que configure los permisos para su rol.
                                        </p>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6 mb-3">
                                <div class="card h-100">
                                    <div class="card-body">
                                        <i class="fas fa-sign-out-alt fa-2x text-danger mb-3"></i>
                                        <h5>Cerrar Sesión</h5>
                                        <p class="small text-muted">
                                            Finalice la sesión actual e intente ingresar nuevamente.
                                        </p>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="mt-4">
                            <a href="logout.php" class="btn btn-danger me-2">
                                <i class="fas fa-sign-out-alt me-1"></i>Cerrar Sesión
                            </a>
                            <a href="mailto:administrador@medicina.ucv.ve" class="btn btn-primary">
                                <i class="fas fa-envelope me-1"></i>Contactar Soporte
                            </a>
                        </div>
                    </div>
                    <div class="card-footer text-center">
                        <small class="text-muted">
                            Sistema de Postgrado - Facultad de Medicina UCV | 
                            Usuario: <?php echo htmlspecialchars($_SESSION['username']); ?> | 
                            IP: <?php echo $_SERVER['REMOTE_ADDR']; ?>
                        </small>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <?php
    include_once ROOT_PATH . '/includes/footer.php';
    exit();
}

// Código de respaldo (no debería ejecutarse nunca)
error_log("Error crítico: Flujo inesperado en index.php para usuario: " . $_SESSION['username']);
session_destroy();
header("Location: " . BASE_URL . "/login.php?error=system_error");
exit();
?>
