<?php
// Incluir el archivo de configuración central.
// Este se encarga de la sesión, codificación, rutas y conexión a la BD.
require_once __DIR__ . '/session_config.php';

// Variables para el header
$site_logo = '';
$user_menu = [];
$app_settings = ['session_duration' => 3600, 'session_never_expire' => 0, 'menu_style' => 'classic']; // Defaults

// Inicializar variables para evitar errores
$initial_time_remaining = 0;
$session_time_remaining = '--:--';
$session_settings = $app_settings; // Inicializar con valores por defecto

// Obtener información del usuario actual para mostrar nombre - SOLUCIÓN DEL PROBLEMA
$nombre_completo_usuario = $_SESSION['username'] ?? 'Usuario';
if (isset($_SESSION['user_id']) && isset($conn)) {
    $user_id = $_SESSION['user_id'];
    $sql_user = "SELECT nombres, apellidos FROM usuarios WHERE id = ?";
    $stmt_user = $conn->prepare($sql_user);
    $stmt_user->bind_param("i", $user_id);
    $stmt_user->execute();
    $result_user = $stmt_user->get_result();
    
    if ($row_user = $result_user->fetch_assoc()) {
        $nombre_completo_usuario = trim($row_user['nombres'] . ' ' . $row_user['apellidos']);
        // Si está vacío, usar el username
        if (empty($nombre_completo_usuario)) {
            $nombre_completo_usuario = $_SESSION['username'] ?? 'Usuario';
        }
    }
}

if (isset($conn)) {
    // Obtener configuraciones de la app
    $sql_settings = "SELECT setting_key, setting_value FROM app_settings 
                     WHERE setting_key IN ('site_logo', 'session_duration', 'session_never_expire', 'menu_style')";
    $result_settings = $conn->query($sql_settings);
    if ($result_settings) {
        while($row = $result_settings->fetch_assoc()) {
            if ($row['setting_key'] == 'site_logo') {
                $site_logo = $row['setting_value'];
            } else {
                $app_settings[$row['setting_key']] = $row['setting_value'];
            }
        }
    }

    // Obtener menú para el rol del usuario
    if (isset($_SESSION['rol'])) {
        $rol = $_SESSION['rol'];
        $items_by_id = []; // Inicializar la variable aquí para evitar errores

        // Cargar menú dinámico solo si está seleccionado
        if ($app_settings['menu_style'] === 'dynamic') {
            $sql_menu = "SELECT mi.* FROM menu_items mi JOIN menu_roles mr ON mi.id = mr.menu_item_id WHERE mr.rol = ? ORDER BY mi.sort_order, mi.label";
            $stmt_menu = $conn->prepare($sql_menu);
            $stmt_menu->bind_param("s", $rol);
            $stmt_menu->execute();
            $result_menu = $stmt_menu->get_result();
            
            $items = [];
            while ($row = $result_menu->fetch_assoc()) {
                $items[] = $row;
            }

            // Construir árbol de menú
            foreach ($items as $item) {
                $items_by_id[$item['id']] = $item;
            }

            foreach ($items as $key => $item) {
                if ($item['parent_id'] && isset($items_by_id[$item['parent_id']])) {
                    $items_by_id[$item['parent_id']]['children'][] = &$items_by_id[$item['id']];
                }
            } 
        }
        // Filtrar solo los padres
        $user_menu = array_filter($items_by_id, function($item){ return empty($item['parent_id']); });
    }
}

// Configuración de sesión para JavaScript
$session_settings = $app_settings; // Para compatibilidad con el modal de sesión
$page_title = $page_title ?? 'Sistema de Postgrado';

// Calcular tiempo restante de sesión para mostrar inicialmente
if (isset($_SESSION['LAST_ACTIVITY']) && $session_settings['session_never_expire'] !== '1') {
    $session_duration = intval($session_settings['session_duration']);
    $time_elapsed = time() - $_SESSION['LAST_ACTIVITY'];
    $initial_time_remaining = max(0, $session_duration - $time_elapsed);
    
    if ($initial_time_remaining > 0) {
        $minutes = floor($initial_time_remaining / 60);
        $seconds = $initial_time_remaining % 60;
        $session_time_remaining = sprintf("%02d:%02d", $minutes, $seconds);
    }
}

// Obtener hora del servidor en formato 12h y 24h
$server_time_24h = date('H:i:s');
$server_time_12h = date('h:i:s A');
$server_date = date('d/m/Y');

// Verificar si el rol puede ver el panel de tiempo (solo admin y coordinador2)
$show_time_panel = false;
if (isset($_SESSION['rol']) && in_array($_SESSION['rol'], ['administrador', 'coordinador2'])) {
    $show_time_panel = true;
}

// Función para renderizar el menú dinámico
function render_menu($menu_items) {
    foreach ($menu_items as $item) {
        if (!empty($item['children'])) { // Es un dropdown
            echo '<li class="nav-item dropdown">';
            echo '<a class="nav-link dropdown-toggle" href="#" id="navbarDropdown' . $item['id'] . '" role="button" data-bs-toggle="dropdown" aria-expanded="false">';
            if ($item['icon']) echo '<i class="' . htmlspecialchars($item['icon']) . ' me-1"></i>';
            echo htmlspecialchars($item['label']);
            echo '</a>';
            echo '<ul class="dropdown-menu" aria-labelledby="navbarDropdown' . $item['id'] . '">';
            render_menu($item['children']); // Recursión para los hijos
            echo '</ul>';
            echo '</li>';
        } else { // Es un enlace simple
            echo '<li class="nav-item">';
            // CORREGIDO: Usar ruta absoluta para el menú dinámico
            $url = htmlspecialchars($item['url']);
            // Si la URL no comienza con /cepmed/, hacerla absoluta
            if (!preg_match('#^/cepmed/#', $url) && !preg_match('#^https?://#', $url)) {
                $url = '/cepmed' . (strpos($url, '/') === 0 ? $url : '/' . $url);
            }
            echo '<a class="nav-link" href="' . $url . '">';
            if ($item['icon']) echo '<i class="' . htmlspecialchars($item['icon']) . ' me-1"></i>';
            echo htmlspecialchars($item['label']);
            echo '</a>';
            echo '</li>';
        }
    }
}

// Función para renderizar el menú clásico (estático)
function render_classic_menu($role) {
    echo '<div class="list-group list-group-flush">';
    
    // Enlace al Panel Principal (Dashboard) - CORREGIDO: rutas absolutas
    $dashboard_link = '/cepmed/index.php'; // Enlace por defecto
    if ($role === 'administrador') $dashboard_link = '/cepmed/admin/index.php';
    if ($role === 'coordinador1') $dashboard_link = '/cepmed/coordinador1/index.php';
    if ($role === 'coordinador2') $dashboard_link = '/cepmed/coordinador2/index.php';
    if ($role === 'secretaria') $dashboard_link = '/cepmed/secretaria/index.php';
    if ($role === 'estudiante') $dashboard_link = '/cepmed/estudiante/index.php';
    
    echo '<a href="' . $dashboard_link . '" class="list-group-item list-group-item-action bg-primary text-white"><i class="fas fa-tachometer-alt fa-fw me-2 text-white"></i>Panel Principal</a>';
    
    if ($role === 'administrador') {
        echo '<div class="accordion" id="adminMenuAccordion">';
        
        // Gestión de Usuarios
        echo '<div class="accordion-item">';
        echo '<h2 class="accordion-header">';
        echo '<button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapseUsers" aria-expanded="false" aria-controls="collapseUsers">';
        echo '<i class="fas fa-users-cog fa-fw me-2"></i>Gestión de Usuarios';
        echo '</button>';
        echo '</h2>';
        echo '<div id="collapseUsers" class="accordion-collapse collapse" data-bs-parent="#adminMenuAccordion">';
        echo '<div class="accordion-body p-0">';
        echo '<a href="/cepmed/admin/gestion_usuarios.php" class="list-group-item list-group-item-action border-0"><i class="fas fa-users-cog fa-fw me-2"></i>Administrar Usuarios</a>';
        echo '<a href="/cepmed/admin/asignar_postgrados_1.php" class="list-group-item list-group-item-action border-0"><i class="fas fa-user-plus fa-fw me-2"></i>Asignar a Coord. 1</a>';
        echo '<a href="/cepmed/admin/asignar_postgrados.php" class="list-group-item list-group-item-action border-0"><i class="fas fa-user-plus fa-fw me-2"></i>Asignar a Coord. 2</a>';
        echo '<a href="/cepmed/admin/habilitar_escolaridad.php" class="list-group-item list-group-item-action border-0"><i class="fas fa-calendar-check fa-fw me-2"></i>Habilitar Escolaridad</a>';
        echo '</div>';
        echo '</div>';
        echo '</div>';

        // Gestión Académica
        echo '<div class="accordion-item">';
        echo '<h2 class="accordion-header">';
        echo '<button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapseAcademic" aria-expanded="false" aria-controls="collapseAcademic">';
        echo '<i class="fas fa-graduation-cap fa-fw me-2"></i>Gestión Académica';
        echo '</button>';
        echo '</h2>';
        echo '<div id="collapseAcademic" class="accordion-collapse collapse" data-bs-parent="#adminMenuAccordion">';
        echo '<div class="accordion-body p-0">';
        echo '<a href="/cepmed/admin/gestion_postgrados.php" class="list-group-item list-group-item-action border-0"><i class="fas fa-graduation-cap fa-fw me-2"></i>Gestionar Postgrados</a>';
        echo '<a href="/cepmed/admin/gestion_denominaciones.php" class="list-group-item list-group-item-action border-0"><i class="fas fa-book fa-fw me-2"></i>Gestionar Denominaciones</a>';
        echo '<a href="/cepmed/admin/gestionar_estado_postgrados.php" class="list-group-item list-group-item-action border-0"><i class="fas fa-toggle-on fa-fw me-2"></i>Gestionar Estados Postgrados</a>';
        echo '<a href="/cepmed/admin/cambio_postgrado.php" class="list-group-item list-group-item-action border-0"><i class="fas fa-exchange-alt fa-fw me-2"></i>Cambio de Postgrado</a>';
        echo '</div>';
        echo '</div>';
        echo '</div>';

        // Gestión de Estudiantes
        echo '<div class="accordion-item">';
        echo '<h2 class="accordion-header">';
        echo '<button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapseStudents" aria-expanded="false" aria-controls="collapseStudents">';
        echo '<i class="fas fa-user-graduate fa-fw me-2"></i>Gestión de Estudiantes';
        echo '</button>';
        echo '</h2>';
        echo '<div id="collapseStudents" class="accordion-collapse collapse" data-bs-parent="#adminMenuAccordion">';
        echo '<div class="accordion-body p-0">';
        echo '<a href="/cepmed/admin/gestion_estudiantes.php" class="list-group-item list-group-item-action border-0"><i class="fas fa-user-graduate fa-fw me-2"></i>Estudiantes</a>';
        echo '<a href="/cepmed/admin/gestion_inscripciones.php" class="list-group-item list-group-item-action border-0"><i class="fas fa-clipboard-list fa-fw me-2"></i>Inscripciones</a>';
        echo '<a href="/cepmed/admin/cargar_notas.php" class="list-group-item list-group-item-action border-0"><i class="fas fa-file-upload fa-fw me-2"></i>Cargar Notas</a>';
        echo '<a href="/cepmed/admin/gestion_graduacion.php" class="list-group-item list-group-item-action border-0"><i class="fas fa-user-graduate fa-fw me-2"></i>Graduación</a>';        
        echo '<a href="/cepmed/retiros/retiro_estudiantes.php" class="list-group-item list-group-item-action border-0"><i class="fas fa-user-minus fa-fw me-2"></i>Retirar Estudiantes</a>';
        echo '<a href="/cepmed/admin/buscar_estudiante.php" class="list-group-item list-group-item-action border-0"><i class="fas fa-id-card fa-fw me-2"></i>Ficha de Grado</a>';
        echo '<a href="/cepmed/admin/buscar_estudiante_secretaria.php" class="list-group-item list-group-item-action border-0"><i class="fas fa-file-alt fa-fw me-2"></i>Constancia de Secretaría</a>';
        echo '<a href="/cepmed/admin/buscar_estudiante_constancia.php" class="list-group-item list-group-item-action border-0"><i class="fas fa-file-invoice fa-fw me-2"></i>Constancia de Estudio</a>';
        echo '<a href="/cepmed/admin/kardex_estudiantes.php" class="list-group-item list-group-item-action border-0"><i class="fas fa-archive fa-fw me-2"></i>Kardex Estudiantil</a>';
        echo '</div>';
        echo '</div>';
        echo '</div>';

        // Reportes
        echo '<div class="accordion-item">';
        echo '<h2 class="accordion-header">';
        echo '<button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapseReports" aria-expanded="false" aria-controls="collapseReports">';
        echo '<i class="fas fa-chart-bar fa-fw me-2"></i>Reportes';
        echo '</button>';
        echo '</h2>';
        echo '<div id="collapseReports" class="accordion-collapse collapse" data-bs-parent="#adminMenuAccordion">';
        echo '<div class="accordion-body p-0">';
        echo '<a href="/cepmed/reportes/reporte1.php" class="list-group-item list-group-item-action border-0"><i class="fas fa-chart-bar fa-fw me-2"></i>Reporte de Notas</a>';
        echo '<a href="/cepmed/reportes/seleccion_reporte.php" class="list-group-item list-group-item-action border-0"><i class="fas fa-user-check fa-fw me-2"></i>Reporte de Graduando</a>';
        echo '<a href="/cepmed/reportes/reporte_cambios_postgrado.php" class="list-group-item list-group-item-action border-0"><i class="fas fa-history fa-fw me-2"></i>Reporte de Cambios</a>';
        echo '</div>';
        echo '</div>';
        echo '</div>';

        // Configuraciones
        echo '<div class="accordion-item">';
        echo '<h2 class="accordion-header">';
        echo '<button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapseSettings" aria-expanded="false" aria-controls="collapseSettings">';
        echo '<i class="fas fa-cogs fa-fw me-2"></i>Configuraciones';
        echo '</button>';
        echo '</h2>';
        echo '<div id="collapseSettings" class="accordion-collapse collapse" data-bs-parent="#adminMenuAccordion">';
        echo '<div class="accordion-body p-0">';
        echo '<a href="/cepmed/admin/gestion_datos_fijos.php" class="list-group-item list-group-item-action border-0"><i class="fas fa-database fa-fw me-2"></i>Datos Fijos</a>';
        echo '<a href="/cepmed/admin/agregar_suspension.php" class="list-group-item list-group-item-action border-0"><i class="fas fa-user-slash fa-fw me-2"></i>Suspensiones</a>';
        echo '<a href="/cepmed/admin/configurar_firmas.php" class="list-group-item list-group-item-action border-0"><i class="fas fa-signature fa-fw me-2"></i>Firmas Ficha de Grado</a>';
        echo '<a href="/cepmed/admin/configurar_firmas_secretaria.php" class="list-group-item list-group-item-action border-0"><i class="fas fa-signature fa-fw me-2"></i>Firmas Secretaría</a>';
        echo '<a href="/cepmed/admin/configurar_firmas_constancia_estudio.php" class="list-group-item list-group-item-action border-0"><i class="fas fa-signature fa-fw me-2"></i>Firmas Constancia Estudio</a>';
        echo '<a href="/cepmed/admin/gestion_registro.php" class="list-group-item list-group-item-action border-0"><i class="fas fa-cogs fa-fw me-2"></i>Configuración General</a>';
        echo '</div>';
        echo '</div>';
        echo '</div>';

        echo '</div>'; // Cierre del accordion
    } elseif ($role === 'coordinador1') {
        echo '<div class="list-group-item"><h6 class="text-muted">Gestión de Notas</h6></div>';
        echo '<a href="/cepmed/coordinador1/cargar_notas.php" class="list-group-item list-group-item-action"><i class="fas fa-edit fa-fw me-2"></i>Cargar Notas</a>';
    } elseif ($role === 'coordinador2') {
        echo '<a href="/cepmed/coordinador2/inscripcion.php" class="list-group-item list-group-item-action"><i class="fas fa-user-plus fa-fw me-2"></i>Inscripción</a>';
        echo '<a href="/cepmed/coordinador2/cargar_notas.php" class="list-group-item list-group-item-action"><i class="fas fa-edit fa-fw me-2"></i>Cargar Notas</a>';
        echo '<a href="/cepmed/coordinador2/estudiantes.php" class="list-group-item list-group-item-action"><i class="fas fa-users fa-fw me-2"></i>Gestión de Estudiantes</a>';
        echo '<a href="/cepmed/coordinador2/crear_estudiante.php" class="list-group-item list-group-item-action"><i class="fas fa-user-plus fa-fw me-2"></i>Crear Nuevo Estudiante</a>';
        echo '<a href="/cepmed/retiros/retiro_estudiantes.php" class="list-group-item list-group-item-action text-danger"><i class="fas fa-user-minus fa-fw me-2"></i>Retiro de Estudiantes</a>';
        echo '<a href="/cepmed/coordinador2/buscar_estudiante.php" class="list-group-item list-group-item-action"><i class="fas fa-id-card fa-fw me-2"></i>Ficha de Grado</a>';
    } elseif ($role === 'secretaria') {
        echo '<a href="/cepmed/secretaria/buscar_estudiante_secretaria.php" class="list-group-item list-group-item-action"><i class="fas fa-search fa-fw me-2"></i>Generar Constancias</a>';
    } elseif ($role === 'estudiante') {
        echo '<a href="/cepmed/estudiante/index.php" class="list-group-item list-group-item-action"><i class="fas fa-book-reader fa-fw me-2"></i>Mis Calificaciones</a>';
    } else {
        echo '<div class="alert alert-info">El menú clásico no está definido para este rol.</div>';
    }
    
    echo '</div>';
}

?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($page_title); ?></title>
    <!-- CORREGIDO: Usar rutas absolutas para los recursos -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet" integrity="sha384-QWTKZyjpPEjISv5WaRU9OFeRpok6YctnYmDr5pNlyT2bRjXh0JMhjY6hW+ALEwIH" crossorigin="anonymous">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="/cepmed/css/styles.css">
    <style>
        body { 
            display: flex; 
            flex-direction: column; 
            min-height: 100vh; 
            background-color: #f8f9fa; 
            padding-top: 120px; /* Espacio para el header fijo */
        }
        .body-wrapper { display: flex; flex: 1; }
        .full-width-header { 
            position: fixed; /* Fijo en la parte superior */
            top: 0;
            left: 0;
            width: 100%; 
            background-color: #2c3e50; 
            padding: 1rem 0; 
            box-shadow: 0 2px 5px rgba(0,0,0,0.1); 
            z-index: 1030;
        }
        .main-container { 
            flex: 1; 
            width: 100%; 
            max-width: 1200px; 
            margin: 0 auto; 
            padding: 0 15px; 
        }
        .header-container { 
            max-width: 1200px; 
            margin: 0 auto; 
            padding: 0 15px; 
        }
        .header-logo { 
            max-height: 50px; 
            margin-right: 15px; 
        }
        .navbar-custom { 
            position: sticky;
            top: 120px; /* Debajo del header */
            background-color: #f8f9fa; 
            border-bottom: 1px solid #dee2e6; 
            z-index: 1029; 
        }
        .offcanvas-start { width: 280px; } /* Menú principal más compacto */
        .offcanvas-end { width: 240px; } /* Menú de acciones más compacto */
        .offcanvas-header { 
            background-color: #2c3e50; 
            color: white; 
            padding: 1rem;
        }
        .offcanvas-body .list-group-item { 
            border: none; 
            padding: 0.6rem 1rem; /* Menos padding para más compacto */
        }
        .offcanvas-body .list-group-item i { color: #6c757d; }
        /* Estilos mejorados para el menú de acciones */
        .action-menu .list-group-item { padding: 0.5rem 0.8rem; }
        /* Icono blanco para Panel Principal */
        .bg-primary .text-white { color: white !important; }
        .bg-primary .fa-fw { color: white !important; }
        /* Estilos para el acordeón del menú */
        .accordion-button:not(.collapsed) { background-color: #e9ecef; color: #2c3e50; }
        .accordion-button:focus { box-shadow: none; border-color: rgba(0,0,0,.125); }
        .accordion-body { padding: 0 !important; }
        .accordion-button { 
            padding: 0.7rem 1rem; /* Más compacto */
            font-size: 0.9rem;
        }
        /* Ajustes para el contenido principal */
        .main-content {
            padding: 1rem 0;
        }
        /* Panel de información de tiempo */
        .time-info-panel {
            background: rgba(255,255,255,0.15);
            border-radius: 10px;
            padding: 8px 12px;
            margin-left: 15px;
            border: 1px solid rgba(255,255,255,0.3);
            min-width: 240px;
            position: relative;
        }
        .time-display {
            font-size: 0.8rem;
            line-height: 1.2;
        }
        .time-label {
            font-size: 0.7rem;
            color: #e0e0e0 !important; /* Color claro para mejor contraste */
            margin-bottom: 2px;
            font-weight: 500;
        }
        .time-label i {
            color: #bbdefb !important; /* Azul claro para iconos */
        }
        .time-value {
            font-weight: 600;
            font-size: 0.85rem;
        }
        .session-time {
            color: #4fc3f7 !important; /* Azul claro */
        }
        .local-time {
            color: #81c784 !important; /* Verde claro */
        }
        .server-time {
            color: #ffb74d !important; /* Naranja */
        }
        /* Botón de formato reposicionado */
        .time-format-section {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 4px;
        }
        .time-format-toggle {
            background: rgba(255,255,255,0.2);
            border: none;
            border-radius: 4px;
            color: #e0e0e0;
            font-size: 0.65rem;
            padding: 2px 6px;
            cursor: pointer;
            margin-left: 8px;
        }
        .time-format-toggle:hover {
            background: rgba(255,255,255,0.3);
            color: white;
        }
        .session-time-warning {
            background: #ffc107 !important;
            animation: pulse 2s infinite;
        }
        .session-time-warning .time-label,
        .session-time-warning .time-value {
            color: #000 !important;
        }
        .session-time-critical {
            background: #dc3545 !important;
            animation: pulse 1s infinite;
        }
        .session-time-critical .time-label,
        .session-time-critical .time-value {
            color: #fff !important;
        }
        @keyframes pulse {
            0% { opacity: 1; }
            50% { opacity: 0.7; }
            100% { opacity: 1; }
        }
        /* Contador en movimiento */
        .countdown-active {
            font-weight: bold;
        }
        /* Para dispositivos móviles */
        @media (max-width: 768px) {
            .time-info-panel {
                min-width: 220px;
                padding: 6px 10px;
            }
            .time-display {
                font-size: 0.75rem;
            }
            .time-value {
                font-size: 0.8rem;
            }
        }
    </style>
</head>
<body>
    <header class="full-width-header">
        <div class="header-container">
            <div class="d-flex justify-content-between align-items-center">
                <div class="d-flex align-items-center">
                    <?php if (isset($_SESSION['username']) && $app_settings['menu_style'] === 'classic'): ?>
                        <button class="btn btn-outline-light me-3" type="button" data-bs-toggle="offcanvas" data-bs-target="#classicMenuOffcanvas" aria-controls="classicMenuOffcanvas" title="Abrir menú">
                            <i class="fas fa-bars"></i>
                        </button>
                    <?php endif; ?>
                    <?php if ($site_logo && file_exists(__DIR__ . '/../' . $site_logo)): ?>
                        <!-- CORREGIDO: Ruta absoluta para el logo -->
                        <img src="/cepmed/<?= htmlspecialchars($site_logo) ?>" alt="Logo" class="header-logo">
                    <?php endif; ?>
                    <div>
                        <h1 class="text-white mb-1" style="font-size: 1.8rem;"><?php echo htmlspecialchars($page_title); ?></h1>
                        <?php if (isset($_SESSION['username'])): ?>
                            <div class="user-info text-light" style="font-size: 1rem;">
                                <i class="fas fa-user"></i> 
                                <!-- SOLUCIÓN: Usar la variable corregida que obtiene nombres y apellidos de la BD -->
                                <?php echo htmlspecialchars($nombre_completo_usuario); ?>
                                <span class="badge bg-light text-dark ms-2"><?php echo htmlspecialchars($_SESSION['rol'] ?? 'Usuario'); ?></span>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
                <?php if (isset($_SESSION['username'])): ?>
                    <div class="d-flex align-items-center">
                        <!-- Panel de información de tiempo - SOLO para admin y coordinador2 -->
                        <?php if ($show_time_panel): ?>
                            <?php if ($session_settings['session_never_expire'] !== '1'): ?>
                            <div class="time-info-panel" id="timePanel">
                                <div class="time-display">
                                    <!-- Sección de sesión con botón de formato -->
                                    <div class="time-format-section">
                                        <div class="d-flex justify-content-between align-items-center flex-grow-1">
                                            <span class="time-label"><i class="fas fa-clock me-1"></i>Sesión:</span>
                                            <span id="sessionTimeDisplay" class="time-value session-time countdown-active"><?= $session_time_remaining ?></span>
                                        </div>
                                        <button class="time-format-toggle" id="timeFormatToggle" title="Cambiar formato de hora">12h</button>
                                    </div>
                                    <!-- Hora local -->
                                    <div class="d-flex justify-content-between align-items-center mb-1">
                                        <span class="time-label"><i class="fas fa-desktop me-1"></i>Local:</span>
                                        <span id="localTimeDisplay" class="time-value local-time">--:--:--</span>
                                    </div>
                                    <!-- Hora servidor -->
                                    <div class="d-flex justify-content-between align-items-center">
                                        <span class="time-label"><i class="fas fa-server me-1"></i>Servidor:</span>
                                        <span id="serverTimeDisplay" class="time-value server-time"><?= $server_time_12h ?></span>
                                    </div>
                                </div>
                            </div>
                            <?php else: ?>
                            <!-- Solo mostrar horas cuando la sesión nunca expira -->
                            <div class="time-info-panel" id="timePanel">
                                <div class="time-display">
                                    <!-- Sección de formato sin sesión -->
                                    <div class="time-format-section">
                                        <span class="time-label"><i class="fas fa-clock me-1"></i>Formato:</span>
                                        <button class="time-format-toggle" id="timeFormatToggle" title="Cambiar formato de hora">12h</button>
                                    </div>
                                    <!-- Hora local -->
                                    <div class="d-flex justify-content-between align-items-center mb-1">
                                        <span class="time-label"><i class="fas fa-desktop me-1"></i>Local:</span>
                                        <span id="localTimeDisplay" class="time-value local-time">--:--:--</span>
                                    </div>
                                    <!-- Hora servidor -->
                                    <div class="d-flex justify-content-between align-items-center">
                                        <span class="time-label"><i class="fas fa-server me-1"></i>Servidor:</span>
                                        <span id="serverTimeDisplay" class="time-value server-time"><?= $server_time_12h ?></span>
                                    </div>
                                </div>
                            </div>
                            <?php endif; ?>
                        <?php endif; ?>
                        
                        <!-- Botón de hamburguesa para menú de acciones rápidas -->
                        <button class="btn btn-outline-light ms-3" type="button" data-bs-toggle="offcanvas" data-bs-target="#appMenuOffcanvas" aria-controls="appMenuOffcanvas" title="Mi Perfil">
                            <i class="fas fa-user"></i>
                        </button>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </header>

    <?php if (isset($_SESSION['username']) && $app_settings['menu_style'] === 'dynamic' && !empty($user_menu)): ?>
    <nav class="navbar navbar-expand-lg navbar-custom">
        <div class="container-fluid" style="max-width: 1200px; margin: 0 auto;">
            <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#mainNavbar" aria-controls="mainNavbar" aria-expanded="false" aria-label="Toggle navigation">
                <span class="navbar-toggler-icon"></span>
            </button>
            <div class="collapse navbar-collapse" id="mainNavbar">
                <ul class="navbar-nav me-auto mb-2 mb-lg-0">
                    <?php render_menu($user_menu); ?>
                </ul>
            </div>
        </div>
    </nav>
    <?php endif; ?>

    <!-- Offcanvas Menu -->
    <div class="offcanvas offcanvas-end" tabindex="-1" id="appMenuOffcanvas" aria-labelledby="appMenuOffcanvasLabel">
        <div class="offcanvas-header">
            <h5 id="appMenuOffcanvasLabel">Menú de Acciones</h5>
            <button type="button" class="btn-close text-reset btn-close-white" data-bs-dismiss="offcanvas" aria-label="Close"></button>
        </div>
        <div class="offcanvas-body p-0 action-menu">
            <div class="list-group list-group-flush">
                <!-- CORREGIDO: Usar ruta absoluta para perfil -->
                <a href="/cepmed/perfil.php" class="list-group-item list-group-item-action"><i class="fas fa-user-cog fa-fw me-2"></i>Mi Perfil</a>
                <!-- CORREGIDO: Usar ruta absoluta para logout -->
                <a href="/cepmed/logout.php" class="list-group-item list-group-item-action text-danger"><i class="fas fa-sign-out-alt fa-fw me-2"></i>Cerrar Sesión</a>
            </div>
        </div>
    </div>

    <!-- Menú Clásico Offcanvas (Barra Lateral Izquierda) -->
    <?php if (isset($_SESSION['username']) && $app_settings['menu_style'] === 'classic'): ?>
    <div class="offcanvas offcanvas-start" tabindex="-1" id="classicMenuOffcanvas" aria-labelledby="classicMenuLabel">
        <div class="offcanvas-header">
            <h5 id="classicMenuLabel">Menú Principal</h5>
            <button type="button" class="btn-close text-reset btn-close-white" data-bs-dismiss="offcanvas" aria-label="Close"></button>
        </div>
        <div class="offcanvas-body">
            <div class="list-group list-group-flush">
                <?php render_classic_menu($_SESSION['rol'] ?? 'default'); ?>
            </div>
        </div>
    </div>
    <?php endif; ?>
    
    <main class="main-container">
        <div class="main-content">
            <?php if (isset($_SESSION['success'])): ?>
                <div class="alert alert-success alert-dismissible fade show">
                    <?= htmlspecialchars($_SESSION['success']) ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
                <?php unset($_SESSION['success']); ?>
            <?php endif; ?>
            
            <?php if (isset($_SESSION['error'])): ?>
                <div class="alert alert-danger alert-dismissible fade show">
                    <?= htmlspecialchars($_SESSION['error']) ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
                <?php unset($_SESSION['error']); ?>
            <?php endif; ?>

<!-- El footer se incluirá después de este punto -->

<!-- Modal de Advertencia de Sesión -->
<div class="modal fade" id="sessionWarningModal" tabindex="-1" aria-labelledby="sessionWarningModalLabel" aria-hidden="true" data-bs-backdrop="static" data-bs-keyboard="false">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header bg-warning text-dark">
                <h5 class="modal-title" id="sessionWarningModalLabel"><i class="fas fa-exclamation-triangle"></i> Su sesión está a punto de expirar</h5>
            </div>
            <div class="modal-body text-center">
                <p>Su sesión se cerrará automáticamente por inactividad en:</p>
                <p class="h1" id="sessionCountdown">05:00</p>
                <p>¿Desea permanecer conectado?</p>
                <div class="mt-3">
                    <small class="text-muted">
                        <i class="fas fa-info-circle me-1"></i>
                        La sesión se renovará automáticamente si continúa navegando
                    </small>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" id="logoutSessionBtn">
                    <i class="fas fa-sign-out-alt me-1"></i>Cerrar Sesión
                </button>
                <button type="button" class="btn btn-primary" id="extendSessionBtn">
                    <i class="fas fa-sync-alt me-1"></i>Permanecer Conectado
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Script para la gestión de sesión COMPLETAMENTE CORREGIDO -->
<script>
// Configuración de sesión inyectada desde PHP
const sessionConfig = {
    duration: <?= intval($session_settings['session_duration'] ?? 3600) ?>,
    neverExpire: <?= json_encode($session_settings['session_never_expire'] ?? '0') ?>,
    warningTime: 300, // 5 minutos en segundos
    baseUrl: '<?= BASE_URL ?>',
    // Información de tiempo inicial desde PHP
    initialTimeRemaining: <?= intval($initial_time_remaining) ?>,
    // Horas del servidor desde PHP
    serverTime12h: '<?= $server_time_12h ?>',
    serverTime24h: '<?= $server_time_24h ?>',
    serverDate: '<?= $server_date ?>',
    // Control de visibilidad por rol
    showTimePanel: <?= $show_time_panel ? 'true' : 'false' ?>
};

// Variables globales - UNA SOLA FUENTE DE VERDAD
let currentTimeRemaining = sessionConfig.initialTimeRemaining;
let countdownInterval = null;
let timeUpdateInterval = null;
let timeFormat = '12h';
let modalShown = false;
let modalInstance = null;

// Función para formatear el tiempo
function formatTime(date, format) {
    if (format === '24h') {
        const hours = date.getHours().toString().padStart(2, '0');
        const minutes = date.getMinutes().toString().padStart(2, '0');
        const seconds = date.getSeconds().toString().padStart(2, '0');
        return `${hours}:${minutes}:${seconds}`;
    } else {
        let hours = date.getHours();
        const minutes = date.getMinutes().toString().padStart(2, '0');
        const seconds = date.getSeconds().toString().padStart(2, '0');
        const ampm = hours >= 12 ? 'PM' : 'AM';
        hours = hours % 12;
        hours = hours ? hours : 12;
        return `${hours.toString().padStart(2, '0')}:${minutes}:${seconds} ${ampm}`;
    }
}

// Función para actualizar las horas
function updateTimeDisplays() {
    // Hora local
    const localTime = new Date();
    const localTimeDisplay = document.getElementById('localTimeDisplay');
    if (localTimeDisplay) {
        localTimeDisplay.textContent = formatTime(localTime, timeFormat);
    }
    
    // Hora del servidor
    const serverTimeDisplay = document.getElementById('serverTimeDisplay');
    if (serverTimeDisplay) {
        const [serverHours, serverMinutes, serverSeconds] = sessionConfig.serverTime24h.split(':').map(Number);
        const serverTime = new Date();
        serverTime.setHours(serverHours, serverMinutes, serverSeconds);
        const timeSinceLoad = Math.floor((Date.now() - window.pageLoadTime) / 1000);
        serverTime.setSeconds(serverTime.getSeconds() + timeSinceLoad);
        serverTimeDisplay.textContent = formatTime(serverTime, timeFormat);
    }
}

// Función para actualizar el indicador de tiempo de sesión - SIMPLIFICADA
function updateSessionTimeDisplay(seconds) {
    const minutes = Math.floor(seconds / 60);
    const secs = seconds % 60;
    const timeString = `${minutes.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
    
    // Actualizar el indicador de sesión en el header
    const indicator = document.getElementById('sessionTimeDisplay');
    if (indicator) {
        indicator.textContent = timeString;
    }
    
    // Actualizar también el modal si está visible
    const modalCountdown = document.getElementById('sessionCountdown');
    if (modalCountdown) {
        modalCountdown.textContent = timeString;
    }
    
    // Aplicar estilos según el tiempo restante
    const timePanel = document.getElementById('timePanel');
    if (timePanel) {
        if (seconds <= 60) {
            timePanel.classList.add('session-time-critical');
            timePanel.classList.remove('session-time-warning');
        } else if (seconds <= 300) {
            timePanel.classList.add('session-time-warning');
            timePanel.classList.remove('session-time-critical');
        } else {
            timePanel.classList.remove('session-time-warning', 'session-time-critical');
        }
    }
}

// Función para alternar el formato de hora
function toggleTimeFormat() {
    timeFormat = timeFormat === '12h' ? '24h' : '12h';
    const toggleBtn = document.getElementById('timeFormatToggle');
    if (toggleBtn) {
        toggleBtn.textContent = timeFormat;
        toggleBtn.title = timeFormat === '12h' ? 'Cambiar a formato 24h (militar)' : 'Cambiar a formato 12h (AM/PM)';
    }
    updateTimeDisplays();
}

// Función para iniciar la cuenta regresiva de sesión - SIMPLIFICADA Y ESTABLE
function startSessionCountdown() {
    // Limpiar cualquier intervalo existente
    if (countdownInterval) {
        clearInterval(countdownInterval);
    }
    
    // Iniciar el contador
    countdownInterval = setInterval(() => {
        if (currentTimeRemaining > 0) {
            currentTimeRemaining--;
            updateSessionTimeDisplay(currentTimeRemaining);
            
            // Mostrar advertencia cuando queden 5 minutos (300 segundos)
            if (currentTimeRemaining === sessionConfig.warningTime && !modalShown) {
                modalShown = true;
                showWarningModal();
            }
            
            // Cerrar sesión cuando llegue a 0
            if (currentTimeRemaining <= 0) {
                clearInterval(countdownInterval);
                forceLogout();
            }
        }
    }, 1000);
}

// Función para iniciar la actualización de horas
function startTimeUpdates() {
    if (timeUpdateInterval) {
        clearInterval(timeUpdateInterval);
    }
    updateTimeDisplays();
    timeUpdateInterval = setInterval(updateTimeDisplays, 1000);
}

// Función para mostrar el modal de advertencia
function showWarningModal() {
    const modal = document.getElementById('sessionWarningModal');
    if (!modal) return;
    
    // Asegurarse de que el contador del modal muestre el tiempo correcto
    updateSessionTimeDisplay(currentTimeRemaining);
    
    modalInstance = new bootstrap.Modal(modal);
    
    // Configurar event listeners
    const extendBtn = document.getElementById('extendSessionBtn');
    const logoutBtn = document.getElementById('logoutSessionBtn');
    
    if (extendBtn) {
        extendBtn.onclick = function() {
            extendSession();
            if (modalInstance) {
                modalInstance.hide();
            }
            modalShown = false;
        };
    }
    
    if (logoutBtn) {
        logoutBtn.onclick = function() {
            if (modalInstance) {
                modalInstance.hide();
            }
            forceLogout();
        };
    }
    
    // Evento cuando se cierra el modal
    modal.addEventListener('hidden.bs.modal', function() {
        modalShown = false;
        modalInstance = null;
    });
    
    modalInstance.show();
}

// Función para extender la sesión
async function extendSession() {
    try {
        const response = await fetch(sessionConfig.baseUrl + '/keep_alive.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            }
        });

        if (response.ok) {
            const data = await response.json();
            if (data.status === 'success') {
                // Reiniciar el contador con la duración completa
                currentTimeRemaining = sessionConfig.duration;
                modalShown = false;
                
                // Reiniciar el contador
                if (countdownInterval) {
                    clearInterval(countdownInterval);
                }
                startSessionCountdown();
                
                // Actualizar la visualización inmediatamente
                updateSessionTimeDisplay(currentTimeRemaining);
                
                return;
            }
        }
    } catch (error) {
        console.error('Error de red:', error);
    }
    
    // En caso de error, hacer un reinicio completo como fallback
    currentTimeRemaining = sessionConfig.duration;
    modalShown = false;
    
    if (countdownInterval) {
        clearInterval(countdownInterval);
    }
    startSessionCountdown();
    updateSessionTimeDisplay(currentTimeRemaining);
}

// Función para forzar logout
function forceLogout() {
    window.location.href = sessionConfig.baseUrl + '/logout.php?reason=session_expired';
}

// Inicialización cuando se carga la página
document.addEventListener('DOMContentLoaded', function() {
    // Guardar el tiempo de carga de la página
    window.pageLoadTime = Date.now();
    
    // Asegurar que currentTimeRemaining sea un número entero
    currentTimeRemaining = parseInt(currentTimeRemaining) || 0;
    
    // Solo inicializar el sistema de tiempo si el panel está visible
    if (sessionConfig.showTimePanel) {
        // Configurar el botón de alternar formato
        const formatToggle = document.getElementById('timeFormatToggle');
        if (formatToggle) {
            formatToggle.addEventListener('click', toggleTimeFormat);
        }
        
        // Iniciar actualización de horas
        startTimeUpdates();
        
        // Solo inicializar el sistema de sesión si no está en modo "nunca expira"
        if (sessionConfig.neverExpire !== '1') {
            startSessionCountdown();
        }
    }
});

// Cleanup cuando se cierra la página
window.addEventListener('beforeunload', function() {
    if (countdownInterval) {
        clearInterval(countdownInterval);
    }
    if (timeUpdateInterval) {
        clearInterval(timeUpdateInterval);
    }
});
</script>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js" integrity="sha384-YvpcrYf0tY3lHB60NNkmXc5s9fDVZLESaAA55NDzOxhy9GkcIdslK1eN7N6jIeHz" crossorigin="anonymous"></script>