<?php
session_start();
if (!isset($_SESSION['username']) || $_SESSION['rol'] != 'coordinador2') {
    header("Location: ../login.php");
    exit();
}

include '../includes/db.php';

// Verificar que se haya proporcionado un ID de estudiante
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    header("Location: estudiantes.php");
    exit();
}

$estudiante_id = $_GET['id'];
$coordinador_id = $_SESSION['user_id'];

// Obtener información básica del estudiante
$sql = "SELECT e.*, ep.*, p.nombre AS postgrado_nombre, al.nombre AS anio_lectivo
        FROM estudiantes e
        JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
        JOIN postgrados p ON ep.postgrado_id = p.id
        JOIN coordinador_postgrado cp ON ep.postgrado_id = cp.postgrado_id
        JOIN inscripciones i ON e.id = i.estudiante_id
        JOIN años_lectivos al ON i.año_lectivo_id = al.id
        WHERE e.id = ? AND cp.coordinador_id = ?
        LIMIT 1";

$stmt = $conn->prepare($sql);
$stmt->bind_param("ii", $estudiante_id, $coordinador_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    // El estudiante no existe o no pertenece a los postgrados del coordinador
    header("Location: estudiantes.php");
    exit();
}

$estudiante = $result->fetch_assoc();

// Obtener las inscripciones del estudiante
$sql_inscripciones = "SELECT i.*, d.nombre AS denominacion_nombre, d.codigo AS denominacion_codigo, 
                      d.periodo, esc.nombre AS escolaridad, al.nombre AS anio_lectivo, 
                      s.nombre AS seccion, p.nombre AS plan_nombre
                      FROM inscripciones i
                      JOIN denominaciones d ON i.denominacion_id = d.id
                      JOIN escolaridades esc ON i.escolaridad_id = esc.id
                      JOIN años_lectivos al ON i.año_lectivo_id = al.id
                      JOIN secciones s ON i.seccion_id = s.id
                      JOIN planes p ON d.plan_id = p.id
                      WHERE i.estudiante_id = ?
                      ORDER BY d.periodo, d.nombre";

$stmt_insc = $conn->prepare($sql_inscripciones);
$stmt_insc->bind_param("i", $estudiante_id);
$stmt_insc->execute();
$inscripciones = $stmt_insc->get_result()->fetch_all(MYSQLI_ASSOC);

// Obtener suspensiones del estudiante
$sql_suspensiones = "SELECT * FROM suspensiones 
                     WHERE estudiante_id = ? 
                     ORDER BY fecha_inicio DESC";
$stmt_susp = $conn->prepare($sql_suspensiones);
$stmt_susp->bind_param("i", $estudiante_id);
$stmt_susp->execute();
$suspensiones = $stmt_susp->get_result()->fetch_all(MYSQLI_ASSOC);

// Obtener cambios de postgrado (si existen)
$sql_cambios = "SELECT cp.*, p1.nombre AS postgrado_anterior_nombre, p2.nombre AS postgrado_nuevo_nombre
                FROM cambios_postgrado cp
                JOIN postgrados p1 ON cp.postgrado_anterior = p1.id
                JOIN postgrados p2 ON cp.postgrado_nuevo = p2.id
                WHERE cp.estudiante_id = ?
                ORDER BY cp.fecha DESC";
$stmt_cambios = $conn->prepare($sql_cambios);
$stmt_cambios->bind_param("i", $estudiante_id);
$stmt_cambios->execute();
$cambios = $stmt_cambios->get_result()->fetch_all(MYSQLI_ASSOC);
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Detalles del Estudiante</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            margin: 20px;
            background-color: #f8f9fa;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: white;
            padding: 25px;
            border-radius: 10px;
            box-shadow: 0 2px 15px rgba(0,0,0,0.1);
        }
        
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 2px solid #e9ecef;
        }
        
        .info-section {
            margin-bottom: 30px;
        }
        
        .info-section h2 {
            color: #2c3e50;
            border-bottom: 1px solid #e9ecef;
            padding-bottom: 10px;
            margin-bottom: 15px;
        }
        
        .info-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 15px;
            margin-bottom: 20px;
        }
        
        .info-item {
            margin-bottom: 10px;
        }
        
        .info-label {
            font-weight: bold;
            color: #495057;
        }
        
        .info-value {
            color: #212529;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
        }
        
        th, td {
            padding: 12px 15px;
            border: 1px solid #dee2e6;
            text-align: left;
        }
        
        th {
            background-color: #2c3e50;
            color: white;
        }
        
        .btn {
            padding: 8px 15px;
            border-radius: 5px;
            text-decoration: none;
            font-size: 14px;
            transition: all 0.3s;
            border: none;
            cursor: pointer;
            margin-right: 5px;
        }
        
        .btn-primary { background-color: #3498db; color: white; }
        .btn-info { background-color: #17a2b8; color: white; }
        .btn-success { background-color: #28a745; color: white; }
        .btn-danger { background-color: #dc3545; color: white; }
        .btn-secondary { background-color: #6c757d; color: white; }
        
        .estado-activo { color: #28a745; }
        .estado-graduado { color: #17a2b8; }
        .estado-retirado { color: #dc3545; }
        .estado-cambio { color: #ffc107; }
        
        .nota-aprobada { color: #28a745; font-weight: bold; }
        .nota-reprobada { color: #dc3545; font-weight: bold; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>Detalles del Estudiante</h1>
            <div>
                <a href="editar_estudiante.php?id=<?= $estudiante['id'] ?>" class="btn btn-primary">Editar</a>
                <a href="cargar_notas.php?estudiante_id=<?= $estudiante['id'] ?>" class="btn btn-success">Notas</a>
                <?php if ($estudiante['estado'] == 'activo'): ?>
                    <a href="suspender_estudiante.php?id=<?= $estudiante['id'] ?>" class="btn btn-danger">Suspender</a>
                <?php endif; ?>
                <a href="estudiantes.php" class="btn btn-secondary">Volver</a>
            </div>
        </div>
        
        <div class="info-section">
            <h2>Información Personal</h2>
            <div class="info-grid">
                <div class="info-item">
                    <span class="info-label">Cédula:</span>
                    <span class="info-value"><?= htmlspecialchars($estudiante['cedula']) ?></span>
                </div>
                <div class="info-item">
                    <span class="info-label">Nombres:</span>
                    <span class="info-value"><?= htmlspecialchars($estudiante['nombres']) ?></span>
                </div>
                <div class="info-item">
                    <span class="info-label">Apellidos:</span>
                    <span class="info-value"><?= htmlspecialchars($estudiante['apellidos']) ?></span>
                </div>
                <div class="info-item">
                    <span class="info-label">Correo:</span>
                    <span class="info-value"><?= htmlspecialchars($estudiante['correo']) ?></span>
                </div>
                <div class="info-item">
                    <span class="info-label">Teléfono:</span>
                    <span class="info-value"><?= htmlspecialchars($estudiante['telefono'] ?? 'N/A') ?></span>
                </div>
                <div class="info-item">
                    <span class="info-label">Celular:</span>
                    <span class="info-value"><?= htmlspecialchars($estudiante['celular'] ?? 'N/A') ?></span>
                </div>
                <div class="info-item">
                    <span class="info-label">Dirección:</span>
                    <span class="info-value"><?= htmlspecialchars($estudiante['direccion'] ?? 'N/A') ?></span>
                </div>
            </div>
        </div>
        
        <div class="info-section">
            <h2>Información Académica</h2>
            <div class="info-grid">
                <div class="info-item">
                    <span class="info-label">Postgrado:</span>
                    <span class="info-value"><?= htmlspecialchars($estudiante['postgrado_nombre']) ?></span>
                </div>
                <div class="info-item">
                    <span class="info-label">Año Lectivo:</span>
                    <span class="info-value"><?= htmlspecialchars($estudiante['anio_lectivo']) ?></span>
                </div>
                <div class="info-item">
                    <span class="info-label">Fecha de Inscripción:</span>
                    <span class="info-value"><?= date('d/m/Y', strtotime($estudiante['fecha_inscripcion'])) ?></span>
                </div>
                <div class="info-item">
                    <span class="info-label">Estado:</span>
                    <span class="info-value estado-<?= $estudiante['estado'] ?>">
                        <?= ucfirst($estudiante['estado']) ?>
                    </span>
                </div>
                <?php if ($estudiante['estado'] == 'graduado' && !empty($estudiante['fecha_graduacion'])): ?>
                <div class="info-item">
                    <span class="info-label">Fecha de Graduación:</span>
                    <span class="info-value"><?= date('d/m/Y', strtotime($estudiante['fecha_graduacion'])) ?></span>
                </div>
                <?php endif; ?>
            </div>
        </div>
        
        <?php if (!empty($suspensiones)): ?>
        <div class="info-section">
            <h2>Historial de Suspensiones</h2>
            <table>
                <thead>
                    <tr>
                        <th>Fecha Inicio</th>
                        <th>Fecha Fin</th>
                        <th>Motivo</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($suspensiones as $suspension): ?>
                    <tr>
                        <td><?= date('d/m/Y', strtotime($suspension['fecha_inicio'])) ?></td>
                        <td><?= date('d/m/Y', strtotime($suspension['fecha_fin'])) ?></td>
                        <td><?= htmlspecialchars($suspension['motivo'] ?? 'No especificado') ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php endif; ?>
        
        <?php if (!empty($cambios)): ?>
        <div class="info-section">
            <h2>Historial de Cambios de Postgrado</h2>
            <table>
                <thead>
                    <tr>
                        <th>Fecha</th>
                        <th>Postgrado Anterior</th>
                        <th>Postgrado Nuevo</th>
                        <th>Motivo</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($cambios as $cambio): ?>
                    <tr>
                        <td><?= date('d/m/Y H:i', strtotime($cambio['fecha'])) ?></td>
                        <td><?= htmlspecialchars($cambio['postgrado_anterior_nombre']) ?></td>
                        <td><?= htmlspecialchars($cambio['postgrado_nuevo_nombre']) ?></td>
                        <td><?= htmlspecialchars($cambio['motivo']) ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php endif; ?>
        
        <div class="info-section">
            <h2>Inscripciones y Notas</h2>
            <table>
                <thead>
                    <tr>
                        <th>Código</th>
                        <th>Denominación</th>
                        <th>Plan</th>
                        <th>Periodo</th>
                        <th>Escolaridad</th>
                        <th>Sección</th>
                        <th>Nota Final</th>
                        <th>Estado</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (!empty($inscripciones)): ?>
                        <?php foreach ($inscripciones as $insc): ?>
                        <tr>
                            <td><?= htmlspecialchars($insc['denominacion_codigo']) ?></td>
                            <td><?= htmlspecialchars($insc['denominacion_nombre']) ?></td>
                            <td><?= htmlspecialchars($insc['plan_nombre']) ?></td>
                            <td><?= htmlspecialchars($insc['periodo'] ?? 'N/A') ?></td>
                            <td><?= htmlspecialchars($insc['escolaridad']) ?></td>
                            <td><?= htmlspecialchars($insc['seccion']) ?></td>
                            <td class="<?= 
                                ($insc['nota_final'] !== null && $insc['nota_final'] !== 'NULL') ? 
                                (is_numeric($insc['nota_final']) && $insc['nota_final'] >= 10 ? 'nota-aprobada' : 'nota-reprobada') : '' ?>">
                                <?= ($insc['nota_final'] !== null && $insc['nota_final'] !== 'NULL') ? htmlspecialchars($insc['nota_final']) : 'Pendiente' ?>
                            </td>
                            <td><?= ucfirst($insc['estado']) ?></td>
                        </tr>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="8" class="text-center">No se encontraron inscripciones</td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</body>
</html>
<?php $conn->close(); ?>