<?php
// Configurar para mostrar errores (solo en desarrollo)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Centralized authentication and session check
$allowed_roles = ['coordinador2', 'administrador'];
require_once __DIR__ . '/../includes/auth_check.php';

// Verificar conexión a la base de datos
if ($conn->connect_error) {
    die("Error de conexión: " . $conn->connect_error);
}

// Obtener ID del usuario y rol
$coordinador_id = $_SESSION['user_id'];
$es_administrador = ($_SESSION['rol'] == 'administrador');

// Configuración de paginación
$registros_por_pagina = 10;
$pagina_actual = isset($_GET['pagina']) ? (int)$_GET['pagina'] : 1;
if ($pagina_actual < 1) $pagina_actual = 1;
$offset = ($pagina_actual - 1) * $registros_por_pagina;

// Obtener parámetros de búsqueda
$busqueda = isset($_GET['busqueda']) ? trim($_GET['busqueda']) : '';
$tipo_busqueda = isset($_GET['tipo_busqueda']) ? $_GET['tipo_busqueda'] : 'cedula';

// CONSULTA PRINCIPAL DE ESTUDIANTES
if ($es_administrador) {
    $sql_base = "SELECT e.id, e.cedula, e.nombres, e.apellidos, 
                p.nombre as postgrado_nombre, ep.postgrado_id, ep.estado as estado_postgrado,
                IF(ep.id IS NULL, 'sin_postgrado', IF(ep.estado = 'activo', 'activo', IF(ep.estado = 'graduado', 'graduado', 'inactivo'))) as estado_estudiante
                FROM estudiantes e
                LEFT JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                LEFT JOIN postgrados p ON ep.postgrado_id = p.id";
    
    $sql_contador = "SELECT COUNT(DISTINCT e.id) as total FROM estudiantes e
                    LEFT JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id";
} else {
    $sql_base = "SELECT e.id, e.cedula, e.nombres, e.apellidos, 
                p.nombre as postgrado_nombre, ep.postgrado_id, ep.estado as estado_postgrado,
                IF(ep.id IS NULL, 'sin_postgrado', IF(ep.estado = 'activo', 'activo', IF(ep.estado = 'graduado', 'graduado', 'inactivo'))) as estado_estudiante
                FROM estudiantes e
                LEFT JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                LEFT JOIN postgrados p ON ep.postgrado_id = p.id
                LEFT JOIN coordinador_postgrado cp ON ep.postgrado_id = cp.postgrado_id
                WHERE (cp.coordinador_id = ? OR ep.id IS NULL)";
    
    $sql_contador = "SELECT COUNT(DISTINCT e.id) as total FROM estudiantes e
                    LEFT JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                    LEFT JOIN coordinador_postgrado cp ON ep.postgrado_id = cp.postgrado_id
                    WHERE (cp.coordinador_id = ? OR ep.id IS NULL)";
}

// Aplicar filtros de búsqueda
if (!empty($busqueda)) {
    $busqueda_like = "%$busqueda%";
    
    switch ($tipo_busqueda) {
        case 'cedula':
            $sql_base .= " AND e.cedula LIKE ?";
            $sql_contador .= " AND e.cedula LIKE ?";
            break;
        case 'nombres':
            $sql_base .= " AND (e.nombres LIKE ? OR e.apellidos LIKE ?)";
            $sql_contador .= " AND (e.nombres LIKE ? OR e.apellidos LIKE ?)";
            break;
        case 'apellidos':
            $sql_base .= " AND e.apellidos LIKE ?";
            $sql_contador .= " AND e.apellidos LIKE ?";
            break;
    }
}

$sql_base .= " GROUP BY e.id ORDER BY e.apellidos, e.nombres LIMIT ? OFFSET ?";

// Preparar y ejecutar consulta principal
$stmt_estudiantes = $conn->prepare($sql_base);
if (!$stmt_estudiantes) {
    die("Error en la preparación de la consulta: " . $conn->error);
}

// Preparar y ejecutar consulta de conteo
$stmt_contador = $conn->prepare($sql_contador);
if (!$stmt_contador) {
    die("Error en la preparación de la consulta de conteo: " . $conn->error);
}

// Bind parameters según el tipo de usuario y búsqueda
if ($es_administrador) {
    if (!empty($busqueda)) {
        if ($tipo_busqueda == 'nombres') {
            $stmt_estudiantes->bind_param("ssii", $busqueda_like, $busqueda_like, $registros_por_pagina, $offset);
        } else {
            $stmt_estudiantes->bind_param("sii", $busqueda_like, $registros_por_pagina, $offset);
        }
    } else {
        $stmt_estudiantes->bind_param("ii", $registros_por_pagina, $offset);
    }
} else {
    if (!empty($busqueda)) {
        if ($tipo_busqueda == 'nombres') {
            $stmt_estudiantes->bind_param("issii", $coordinador_id, $busqueda_like, $busqueda_like, $registros_por_pagina, $offset);
        } else {
            $stmt_estudiantes->bind_param("isii", $coordinador_id, $busqueda_like, $registros_por_pagina, $offset);
        }
    } else {
        $stmt_estudiantes->bind_param("iii", $coordinador_id, $registros_por_pagina, $offset);
    }
}

// Bind parameters para la consulta de conteo
if ($es_administrador) {
    if (!empty($busqueda)) {
        if ($tipo_busqueda == 'nombres') {
            $stmt_contador->bind_param("ss", $busqueda_like, $busqueda_like);
        } else {
            $stmt_contador->bind_param("s", $busqueda_like);
        }
    }
    // Para administrador sin búsqueda, no hay parámetros que vincular
} else {
    if (!empty($busqueda)) {
        if ($tipo_busqueda == 'nombres') {
            $stmt_contador->bind_param("iss", $coordinador_id, $busqueda_like, $busqueda_like);
        } else {
            $stmt_contador->bind_param("is", $coordinador_id, $busqueda_like);
        }
    } else {
        // Solo el coordinador_id cuando no hay búsqueda
        $stmt_contador->bind_param("i", $coordinador_id);
    }
}

if (!$stmt_estudiantes->execute()) {
    die("Error al ejecutar la consulta: " . $stmt_estudiantes->error);
}

$result_estudiantes = $stmt_estudiantes->get_result();

if (!$stmt_contador->execute()) {
    die("Error al ejecutar la consulta de conteo: " . $stmt_contador->error);
}

$result_contador = $stmt_contador->get_result();
$total_registros = $result_contador->fetch_assoc()['total'];
$total_paginas = ceil($total_registros / $registros_por_pagina);

// Procesar formulario de asignación de postgrado
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['asignar_postgrado'])) {
    $estudiante_id = intval($_POST['estudiante_id']);
    $postgrado_id = intval($_POST['postgrado_id']);
    
    // Verificar permisos para asignar postgrado
    if (!$es_administrador) {
        $sql_verificar_postgrado = "SELECT 1 FROM coordinador_postgrado WHERE postgrado_id = ? AND coordinador_id = ?";
        $stmt_verificar = $conn->prepare($sql_verificar_postgrado);
        $stmt_verificar->bind_param("ii", $postgrado_id, $coordinador_id);
        $stmt_verificar->execute();
        $result_verificar = $stmt_verificar->get_result();
        
        if ($result_verificar->num_rows == 0) {
            $_SESSION['error'] = "No tienes permisos para asignar este postgrado";
            header("Location: inscripcion.php");
            exit();
        }
    }
    
    // Verificar que el estudiante no tenga ya este postgrado (activo o graduado)
    $sql_verificar_estudiante = "SELECT 1 FROM estudiante_postgrado 
                                WHERE estudiante_id = ? AND postgrado_id = ? AND (estado = 'activo' OR estado = 'graduado')";
    $stmt_verificar_est = $conn->prepare($sql_verificar_estudiante);
    $stmt_verificar_est->bind_param("ii", $estudiante_id, $postgrado_id);
    $stmt_verificar_est->execute();
    $result_verificar_est = $stmt_verificar_est->get_result();
    
    if ($result_verificar_est->num_rows > 0) {
        $_SESSION['error'] = "El estudiante ya ha cursado este postgrado";
        header("Location: inscripcion.php?estudiante_id=$estudiante_id");
        exit();
    }
    
    // Asignar postgrado al estudiante
    $sql_asignar = "INSERT INTO estudiante_postgrado 
                   (estudiante_id, postgrado_id, fecha_inscripcion, estado)
                   VALUES (?, ?, NOW(), 'activo')";
    $stmt_asignar = $conn->prepare($sql_asignar);
    $stmt_asignar->bind_param("ii", $estudiante_id, $postgrado_id);
    
    if ($stmt_asignar->execute()) {
        $_SESSION['success'] = "Postgrado asignado correctamente";
        header("Location: inscripcion.php?estudiante_id=$estudiante_id");
        exit();
    } else {
        $_SESSION['error'] = "Error al asignar postgrado: " . $conn->error;
        header("Location: inscripcion.php");
        exit();
    }
}

// Procesar formulario de inscripción de denominaciones
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['estudiante_id']) && isset($_POST['denominaciones'])) {
    $estudiante_id = intval($_POST['estudiante_id']);
    $denominaciones = $_POST['denominaciones'];
    $escolaridad_id = intval($_POST['escolaridad_id']);
    $año_lectivo_id = intval($_POST['año_lectivo_id']);
    $seccion_id = intval($_POST['seccion_id']);
    
    // Verificar que el estudiante pertenece a este coordinador
    $sql_verificar = "SELECT 1 FROM estudiante_postgrado ep
                     JOIN coordinador_postgrado cp ON ep.postgrado_id = cp.postgrado_id
                     WHERE ep.estudiante_id = ? AND cp.coordinador_id = ? AND ep.estado = 'activo'";
    $stmt_verificar = $conn->prepare($sql_verificar);
    if (!$stmt_verificar) {
        die("Error en la preparación de verificación: " . $conn->error);
    }
    
    $stmt_verificar->bind_param("ii", $estudiante_id, $coordinador_id);
    if (!$stmt_verificar->execute()) {
        die("Error al verificar estudiante: " . $stmt_verificar->error);
    }
    
    $result_verificar = $stmt_verificar->get_result();
    
    if ($result_verificar->num_rows == 0 && !$es_administrador) {
        $_SESSION['error'] = "No tienes permisos para inscribir a este estudiante";
        header("Location: inscripcion.php");
        exit();
    }
    
    // Consulta para insertar inscripciones
    $sql_insert = "INSERT INTO inscripciones 
                  (estudiante_id, denominacion_id, escolaridad_id, año_lectivo_id, seccion_id, fecha_inscripcion)
                  VALUES (?, ?, ?, ?, ?, NOW())";
    
    $stmt_insert = $conn->prepare($sql_insert);
    if (!$stmt_insert) {
        die("Error en la preparación de inserción: " . $conn->error);
    }
    
    $inserciones_exitosas = 0;
    foreach ($denominaciones as $denominacion_id) {
        $denominacion_id = intval($denominacion_id);
        
        // Verificar que la denominación pertenece al postgrado del estudiante
        $sql_verificar_den = "SELECT 1 FROM denominaciones d
                             JOIN estudiante_postgrado ep ON d.postgrado_id = ep.postgrado_id
                             WHERE d.id = ? AND ep.estudiante_id = ? AND ep.estado = 'activo'";
        $stmt_verificar_den = $conn->prepare($sql_verificar_den);
        if (!$stmt_verificar_den) {
            die("Error en la preparación de verificación de denominación: " . $conn->error);
        }
        
        $stmt_verificar_den->bind_param("ii", $denominacion_id, $estudiante_id);
        if (!$stmt_verificar_den->execute()) {
            die("Error al verificar denominación: " . $stmt_verificar_den->error);
        }
        
        $result_verificar_den = $stmt_verificar_den->get_result();
        
        if ($result_verificar_den->num_rows == 1) {
            $stmt_insert->bind_param("iiiii", $estudiante_id, $denominacion_id, $escolaridad_id, $año_lectivo_id, $seccion_id);
            if ($stmt_insert->execute()) {
                $inserciones_exitosas++;
            }
        }
    }
    
    if ($inserciones_exitosas > 0) {
        $_SESSION['success'] = "Se han inscrito $inserciones_exitosas denominaciones correctamente";
    } else {
        $_SESSION['error'] = "No se pudo realizar ninguna inscripción";
    }
    
    header("Location: inscripcion.php?estudiante_id=$estudiante_id");
    exit();
}

// Obtener denominaciones disponibles para estudiante seleccionado
$denominaciones_disponibles = [];
$estudiante_seleccionado = null;
$postgrados_disponibles = [];
$escolaridades = [];
$años_lectivos = [];
$secciones = [];

// Obtener listas para formularios
$sql_escolaridades = "SELECT id, nombre FROM escolaridades ORDER BY nombre";
$result_escolaridades = $conn->query($sql_escolaridades);
while ($row = $result_escolaridades->fetch_assoc()) {
    $escolaridades[] = $row;
}

$sql_años_lectivos = "SELECT id, nombre FROM años_lectivos ORDER BY nombre";
$result_años_lectivos = $conn->query($sql_años_lectivos);
while ($row = $result_años_lectivos->fetch_assoc()) {
    $años_lectivos[] = $row;
}

$sql_secciones = "SELECT id, nombre FROM secciones ORDER BY nombre";
$result_secciones = $conn->query($sql_secciones);
while ($row = $result_secciones->fetch_assoc()) {
    $secciones[] = $row;
}

if (isset($_GET['estudiante_id']) && !empty($_GET['estudiante_id'])) {
    $estudiante_id = intval($_GET['estudiante_id']);
    
    // Consulta para obtener información del estudiante
    $sql_estudiante = "SELECT e.id, e.cedula, e.nombres, e.apellidos, 
                      p.nombre as postgrado_nombre, ep.postgrado_id, ep.estado as estado_postgrado,
                      IF(ep.id IS NULL, 'sin_postgrado', IF(ep.estado = 'activo', 'activo', IF(ep.estado = 'graduado', 'graduado', 'inactivo'))) as estado_estudiante
                      FROM estudiantes e
                      LEFT JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                      LEFT JOIN postgrados p ON ep.postgrado_id = p.id
                      WHERE e.id = ?";
    
    $stmt_estudiante = $conn->prepare($sql_estudiante);
    $stmt_estudiante->bind_param("i", $estudiante_id);
    $stmt_estudiante->execute();
    $estudiante_seleccionado = $stmt_estudiante->get_result()->fetch_assoc();
    
    if ($estudiante_seleccionado) {
        // Si el estudiante no tiene postgrado, obtener postgrados disponibles para asignar
        if ($estudiante_seleccionado['estado_estudiante'] == 'sin_postgrado') {
            if ($es_administrador) {
                $sql_postgrados = "SELECT id, nombre FROM postgrados ORDER BY nombre";
            } else {
                $sql_postgrados = "SELECT p.id, p.nombre 
                                  FROM postgrados p
                                  JOIN coordinador_postgrado cp ON p.id = cp.postgrado_id
                                  WHERE cp.coordinador_id = ?
                                  ORDER BY p.nombre";
            }
            
            $stmt_postgrados = $conn->prepare($sql_postgrados);
            if (!$es_administrador) {
                $stmt_postgrados->bind_param("i", $coordinador_id);
            }
            $stmt_postgrados->execute();
            $result_postgrados = $stmt_postgrados->get_result();
            
            while ($row = $result_postgrados->fetch_assoc()) {
                $postgrados_disponibles[] = $row;
            }
        } 
        // Si el estudiante tiene postgrado activo, obtener denominaciones disponibles
        elseif ($estudiante_seleccionado['estado_estudiante'] == 'activo') {
            $sql_denominaciones = "SELECT d.id, d.codigo, d.nombre, d.credito, d.periodo, d.tipo
                                 FROM denominaciones d
                                 WHERE d.postgrado_id = ?
                                 AND NOT EXISTS (
                                     SELECT 1 FROM inscripciones i
                                     WHERE i.denominacion_id = d.id
                                     AND i.estudiante_id = ?
                                     AND (i.estado = 'activa' OR 
                                         (d.tipo = 'puntaje' AND i.nota_definitiva >= 10) OR
                                         (d.tipo = 'Aprobado_Reprobado' AND i.nota_definitiva = 'Aprobado'))
                                 )
                                 ORDER BY d.periodo, d.codigo";
            
            $stmt_denominaciones = $conn->prepare($sql_denominaciones);
            $stmt_denominaciones->bind_param("ii", $estudiante_seleccionado['postgrado_id'], $estudiante_id);
            $stmt_denominaciones->execute();
            $result_denominaciones = $stmt_denominaciones->get_result();
            
            while ($row = $result_denominaciones->fetch_assoc()) {
                $denominaciones_disponibles[] = $row;
            }
        }
    }
}

$page_title = "Inscripción de Estudiantes";
include '../includes/header.php';
?>

<div class="container">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h2>Inscripción de Estudiantes</h2>
        <div>
            <a href="registrar_estudiante.php" class="btn btn-success me-2">
                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-person-plus" viewBox="0 0 16 16">
                    <path d="M6 8a3 3 0 1 0 0-6 3 3 0 0 0 0 6zm2-3a2 2 0 1 1-4 0 2 2 0 0 1 4 0zm4 8c0 1-1 1-1 1H1s-1 0-1-1 1-4 6-4 6 3 6 4zm-1-.004c-.001-.246-.154-.986-.832-1.664C9.516 10.68 8.289 10 6 10c-2.29 0-3.516.68-4.168 1.332-.678.678-.83 1.418-.832 1.664h10z"/>
                    <path fill-rule="evenodd" d="M13.5 5a.5.5 0 0 1 .5.5V7h1.5a.5.5 0 0 1 0 1H14v1.5a.5.5 0 0 1-1 0V8h-1.5a.5.5 0 0 1 0-1H13V5.5a.5.5 0 0 1 .5-.5z"/>
                </svg>
                Registrar Nuevo
            </a>
            <a href="../index.php" class="btn btn-secondary">
                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-arrow-left" viewBox="0 0 16 16">
                    <path fill-rule="evenodd" d="M15 8a.5.5 0 0 0-.5-.5H2.707l3.147-3.146a.5.5 0 1 0-.708-.708l-4 4a.5.5 0 0 0 0 .708l4 4a.5.5 0 0 0 .708-.708L2.707 8.5H14.5A.5.5 0 0 0 15 8z"/>
                </svg>
                Volver
            </a>
        </div>
    </div>
    
    <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger"><?= $_SESSION['error'] ?></div>
        <?php unset($_SESSION['error']); ?>
    <?php endif; ?>
    
    <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success"><?= $_SESSION['success'] ?></div>
        <?php unset($_SESSION['success']); ?>
    <?php endif; ?>
    
    <div class="card mb-4">
        <div class="card-header">
            <h3 class="mb-0">Búsqueda Avanzada</h3>
        </div>
        <div class="card-body">
            <form method="get" action="inscripcion.php">
                <div class="row">
                    <div class="col-md-8">
                        <div class="form-group">
                            <label for="busqueda" class="form-label">Término de búsqueda:</label>
                            <input type="text" class="form-control" id="busqueda" name="busqueda" 
                                   value="<?= htmlspecialchars($busqueda) ?>" placeholder="Ingrese cédula o nombre del estudiante">
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="form-group">
                            <label for="tipo_busqueda" class="form-label">Buscar por:</label>
                            <select class="form-select" id="tipo_busqueda" name="tipo_busqueda">
                                <option value="cedula" <?= $tipo_busqueda == 'cedula' ? 'selected' : '' ?>>Cédula</option>
                                <option value="nombres" <?= $tipo_busqueda == 'nombres' ? 'selected' : '' ?>>Nombres</option>
                                <option value="apellidos" <?= $tipo_busqueda == 'apellidos' ? 'selected' : '' ?>>Apellidos</option>
                            </select>
                        </div>
                    </div>
                </div>
                <div class="row mt-3">
                    <div class="col-md-12 d-flex justify-content-between">
                        <div>
                            <button type="submit" class="btn btn-primary me-2">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-search" viewBox="0 0 16 16">
                                    <path d="M11.742 10.344a6.5 6.5 0 1 0-1.397 1.398h-.001c.03.04.062.078.098.115l3.85 3.85a1 1 0 0 0 1.415-1.414l-3.85-3.85a1.007 1.007 0 0 0-.115-.1zM12 6.5a5.5 5.5 0 1 1-11 0 5.5 5.5 0 0 1 11 0z"/>
                                </svg>
                                Buscar
                            </button>
                            <a href="inscripcion.php" class="btn btn-secondary">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-arrow-counterclockwise" viewBox="0 0 16 16">
                                    <path fill-rule="evenodd" d="M8 3a5 5 0 1 1-4.546 2.914.5.5 0 0 0-.908-.417A6 6 0 1 0 8 2v1z"/>
                                    <path d="M8 4.466V.534a.25.25 0 0 0-.41-.192L5.23 2.308a.25.25 0 0 0 0 .384l2.36 1.966A.25.25 0 0 0 8 4.466z"/>
                                </svg>
                                Limpiar
                            </a>
                        </div>
                        <?php if (!empty($busqueda)): ?>
                            <div class="text-muted">
                                <?= $total_registros ?> resultados encontrados
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </form>
        </div>
    </div>
    
    <div class="card mb-4">
        <div class="card-header d-flex justify-content-between align-items-center">
            <h3 class="mb-0">Lista de Estudiantes</h3>
            <div class="text-muted">
                Mostrando <?= min($registros_por_pagina, $result_estudiantes->num_rows) ?> de <?= $total_registros ?> registros
            </div>
        </div>
        <div class="card-body">
            <?php if ($result_estudiantes->num_rows > 0): ?>
                <div class="table-responsive">
                    <table class="table table-striped table-hover">
                        <thead class="table-light">
                            <tr>
                                <th>Cédula</th>
                                <th>Apellidos</th>
                                <th>Nombres</th>
                                <th>Postgrado</th>
                                <th>Estado</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php while ($estudiante = $result_estudiantes->fetch_assoc()): ?>
                                <tr>
                                    <td><?= htmlspecialchars($estudiante['cedula']) ?></td>
                                    <td><?= htmlspecialchars($estudiante['apellidos']) ?></td>
                                    <td><?= htmlspecialchars($estudiante['nombres']) ?></td>
                                    <td><?= $estudiante['postgrado_nombre'] ? htmlspecialchars($estudiante['postgrado_nombre']) : 'Sin postgrado' ?></td>
                                    <td>
                                        <?php 
                                        $estado_clase = '';
                                        switch ($estudiante['estado_estudiante']) {
                                            case 'activo': $estado_clase = 'success'; break;
                                            case 'graduado': $estado_clase = 'info'; break;
                                            case 'inactivo': $estado_clase = 'warning'; break;
                                            default: $estado_clase = 'secondary'; break;
                                        }
                                        ?>
                                        <span class="badge bg-<?= $estado_clase ?>">
                                            <?= ucfirst($estudiante['estado_estudiante']) ?>
                                        </span>
                                    </td>
                                    <td>
                                        <a href="inscripcion.php?estudiante_id=<?= $estudiante['id'] ?>" class="btn btn-sm btn-primary">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-pencil-square" viewBox="0 0 16 16">
                                                <path d="M15.502 1.94a.5.5 0 0 1 0 .706L14.459 3.69l-2-2L13.502.646a.5.5 0 0 1 .707 0l1.293 1.293zm-1.75 2.456-2-2L4.939 9.21a.5.5 0 0 0-.121.196l-.805 2.414a.25.25 0 0 0 .316.316l2.414-.805a.5.5 0 0 0 .196-.12l6.813-6.814z"/>
                                                <path fill-rule="evenodd" d="M1 13.5A1.5 1.5 0 0 0 2.5 15h11a1.5 1.5 0 0 0 1.5-1.5v-6a.5.5 0 0 0-1 0v6a.5.5 0 0 1-.5.5h-11a.5.5 0 0 1-.5-.5v-11a.5.5 0 0 1 .5-.5H9a.5.5 0 0 0 0-1H2.5A1.5 1.5 0 0 0 1 2.5v11z"/>
                                            </svg>
                                            Gestionar
                                        </a>
                                    </td>
                                </tr>
                            <?php endwhile; ?>
                        </tbody>
                    </table>
                </div>
                
                <!-- Paginación -->
                <?php if ($total_paginas > 1): ?>
                    <nav aria-label="Paginación">
                        <ul class="pagination justify-content-center mt-4">
                            <?php if ($pagina_actual > 1): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['pagina' => 1])) ?>" aria-label="Primera">
                                        <span aria-hidden="true">&laquo;&laquo;</span>
                                    </a>
                                </li>
                                <li class="page-item">
                                    <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['pagina' => $pagina_actual - 1])) ?>" aria-label="Anterior">
                                        <span aria-hidden="true">&laquo;</span>
                                    </a>
                                </li>
                            <?php endif; ?>
                            
                            <?php 
                            $inicio = max(1, $pagina_actual - 2);
                            $fin = min($total_paginas, $pagina_actual + 2);
                            
                            if ($inicio > 1) {
                                echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                            }
                            
                            for ($i = $inicio; $i <= $fin; $i++): ?>
                                <li class="page-item <?= $i == $pagina_actual ? 'active' : '' ?>">
                                    <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['pagina' => $i])) ?>"><?= $i ?></a>
                                </li>
                            <?php endfor;
                            
                            if ($fin < $total_paginas) {
                                echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                            }
                            ?>
                            
                            <?php if ($pagina_actual < $total_paginas): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['pagina' => $pagina_actual + 1])) ?>" aria-label="Siguiente">
                                        <span aria-hidden="true">&raquo;</span>
                                    </a>
                                </li>
                                <li class="page-item">
                                    <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['pagina' => $total_paginas])) ?>" aria-label="Última">
                                        <span aria-hidden="true">&raquo;&raquo;</span>
                                    </a>
                                </li>
                            <?php endif; ?>
                        </ul>
                    </nav>
                <?php endif; ?>
            <?php else: ?>
                <div class="alert alert-info text-center">
                    No se encontraron estudiantes. 
                    <?php if (!empty($busqueda)): ?>
                        ¿Desea <a href="registrar_estudiante.php?cedula=<?= urlencode($busqueda) ?>" class="alert-link">registrar un nuevo estudiante</a>?
                    <?php else: ?>
                        No hay estudiantes registrados en el sistema.
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        </div>
    </div>
    
    <?php if (isset($estudiante_seleccionado)): ?>
        <div class="card">
            <div class="card-header">
                <h3 class="mb-0">Gestionar Estudiante: <?= htmlspecialchars($estudiante_seleccionado['nombres'] . ' ' . $estudiante_seleccionado['apellidos']) ?></h3>
            </div>
            <div class="card-body">
                <div class="mb-3">
                    <strong>Cédula:</strong> <?= htmlspecialchars($estudiante_seleccionado['cedula']) ?><br>
                    <strong>Postgrado:</strong> <?= $estudiante_seleccionado['postgrado_nombre'] ? htmlspecialchars($estudiante_seleccionado['postgrado_nombre']) : 'Sin postgrado' ?><br>
                    <strong>Estado:</strong> 
                    <span class="badge bg-<?= 
                        $estudiante_seleccionado['estado_estudiante'] == 'activo' ? 'success' : 
                        ($estudiante_seleccionado['estado_estudiante'] == 'graduado' ? 'info' : 
                        ($estudiante_seleccionado['estado_estudiante'] == 'inactivo' ? 'warning' : 'secondary')) ?>">
                        <?= ucfirst($estudiante_seleccionado['estado_estudiante']) ?>
                    </span>
                </div>
                
                <?php if ($estudiante_seleccionado['estado_estudiante'] == 'sin_postgrado'): ?>
                    <?php if (!empty($postgrados_disponibles)): ?>
                        <form method="post" action="inscripcion.php">
                            <input type="hidden" name="asignar_postgrado" value="1">
                            <input type="hidden" name="estudiante_id" value="<?= $estudiante_seleccionado['id'] ?>">
                            
                            <div class="form-group mb-3">
                                <label for="postgrado_id" class="form-label">Seleccione un postgrado:</label>
                                <select class="form-select" id="postgrado_id" name="postgrado_id" required>
                                    <option value="">-- Seleccione un postgrado --</option>
                                    <?php foreach ($postgrados_disponibles as $postgrado): ?>
                                        <?php 
                                        // Verificar si el estudiante ya completó este postgrado
                                        $sql_verificar_completado = "SELECT 1 FROM estudiante_postgrado 
                                                                    WHERE estudiante_id = ? AND postgrado_id = ? AND estado = 'graduado'";
                                        $stmt_verificar = $conn->prepare($sql_verificar_completado);
                                        $stmt_verificar->bind_param("ii", $estudiante_seleccionado['id'], $postgrado['id']);
                                        $stmt_verificar->execute();
                                        $result_verificar = $stmt_verificar->get_result();
                                        
                                        // Solo mostrar si no está completado
                                        if ($result_verificar->num_rows == 0): ?>
                                            <option value="<?= $postgrado['id'] ?>"><?= htmlspecialchars($postgrado['nombre']) ?></option>
                                        <?php endif; ?>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <button type="submit" class="btn btn-success">Asignar Postgrado</button>
                        </form>
                    <?php else: ?>
                        <div class="alert alert-warning">
                            No hay postgrados disponibles para asignar a este estudiante.
                        </div>
                    <?php endif; ?>
                <?php elseif ($estudiante_seleccionado['estado_estudiante'] == 'activo'): ?>
                    <?php if (!empty($denominaciones_disponibles)): ?>
                        <form method="post" action="inscripcion.php">
                            <input type="hidden" name="estudiante_id" value="<?= $estudiante_seleccionado['id'] ?>">
                            
                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <label for="escolaridad_id" class="form-label">Escolaridad:</label>
                                    <select class="form-select" id="escolaridad_id" name="escolaridad_id" required>
                                        <option value="">-- Seleccione escolaridad --</option>
                                        <?php foreach ($escolaridades as $escolaridad): ?>
                                            <option value="<?= $escolaridad['id'] ?>"><?= htmlspecialchars($escolaridad['nombre']) ?></option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="col-md-6">
                                    <label for="año_lectivo_id" class="form-label">Año Lectivo:</label>
                                    <select class="form-select" id="año_lectivo_id" name="año_lectivo_id" required>
                                        <option value="">-- Seleccione año lectivo --</option>
                                        <?php foreach ($años_lectivos as $año): ?>
                                            <option value="<?= $año['id'] ?>"><?= htmlspecialchars($año['nombre']) ?></option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                            </div>
                            
                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <label for="seccion_id" class="form-label">Sección:</label>
                                    <select class="form-select" id="seccion_id" name="seccion_id" required>
                                        <option value="">-- Seleccione sección --</option>
                                        <?php foreach ($secciones as $seccion): ?>
                                            <option value="<?= $seccion['id'] ?>"><?= htmlspecialchars($seccion['nombre']) ?></option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                            </div>
                            
                            <?php 
                            // Agrupar denominaciones por período
                            $denominaciones_por_periodo = [];
                            foreach ($denominaciones_disponibles as $denominacion) {
                                $periodo = $denominacion['periodo'];
                                if (!isset($denominaciones_por_periodo[$periodo])) {
                                    $denominaciones_por_periodo[$periodo] = [];
                                }
                                $denominaciones_por_periodo[$periodo][] = $denominacion;
                            }
                            ?>
                            
                            <?php foreach ($denominaciones_por_periodo as $periodo => $denominaciones): ?>
                                <div class="periodo-title">Período <?= $periodo ?></div>
                                <?php foreach ($denominaciones as $denominacion): ?>
                                    <div class="denominacion-item form-check">
                                        <input class="form-check-input" type="checkbox" 
                                               name="denominaciones[]" 
                                               value="<?= $denominacion['id'] ?>" 
                                               id="den-<?= $denominacion['id'] ?>">
                                        <label class="form-check-label" for="den-<?= $denominacion['id'] ?>">
                                            <strong><?= htmlspecialchars($denominacion['codigo']) ?></strong> - 
                                            <?= htmlspecialchars($denominacion['nombre']) ?> 
                                            (<?= $denominacion['credito'] ?> créditos, <?= $denominacion['tipo'] ?>)
                                        </label>
                                    </div>
                                <?php endforeach; ?>
                            <?php endforeach; ?>
                            
                            <button type="submit" class="btn btn-primary mt-3">Inscribir Denominaciones Seleccionadas</button>
                        </form>
                    <?php else: ?>
                        <div class="alert alert-info">
                            No hay denominaciones disponibles para inscribir a este estudiante.
                        </div>
                    <?php endif; ?>
                <?php elseif ($estudiante_seleccionado['estado_estudiante'] == 'graduado'): ?>
                    <div class="alert alert-info">
                        Este estudiante ya ha completado su postgrado. No puede asignarse el mismo postgrado nuevamente.
                    </div>
                    <?php 
                    // Mostrar opción para asignar un nuevo postgrado diferente
                    if ($es_administrador) {
                        $sql_postgrados = "SELECT id, nombre FROM postgrados ORDER BY nombre";
                    } else {
                        $sql_postgrados = "SELECT p.id, p.nombre 
                                          FROM postgrados p
                                          JOIN coordinador_postgrado cp ON p.id = cp.postgrado_id
                                          WHERE cp.coordinador_id = ?
                                          ORDER BY p.nombre";
                    }
                    
                    $stmt_postgrados = $conn->prepare($sql_postgrados);
                    if (!$es_administrador) {
                        $stmt_postgrados->bind_param("i", $coordinador_id);
                    }
                    $stmt_postgrados->execute();
                    $result_postgrados = $stmt_postgrados->get_result();
                    
                    if ($result_postgrados->num_rows > 0): ?>
                        <form method="post" action="inscripcion.php">
                            <input type="hidden" name="asignar_postgrado" value="1">
                            <input type="hidden" name="estudiante_id" value="<?= $estudiante_seleccionado['id'] ?>">
                            
                            <div class="form-group mb-3">
                                <label for="postgrado_id" class="form-label">Asignar un nuevo postgrado:</label>
                                <select class="form-select" id="postgrado_id" name="postgrado_id" required>
                                    <option value="">-- Seleccione un postgrado --</option>
                                    <?php while ($postgrado = $result_postgrados->fetch_assoc()): ?>
                                        <?php 
                                        // Verificar que no sea el mismo postgrado ya completado
                                        if ($postgrado['id'] != $estudiante_seleccionado['postgrado_id']): ?>
                                            <option value="<?= $postgrado['id'] ?>"><?= htmlspecialchars($postgrado['nombre']) ?></option>
                                        <?php endif; ?>
                                    <?php endwhile; ?>
                                </select>
                            </div>
                            
                            <button type="submit" class="btn btn-success">Asignar Nuevo Postgrado</button>
                        </form>
                    <?php else: ?>
                        <div class="alert alert-warning">
                            No hay otros postgrados disponibles para asignar a este estudiante.
                        </div>
                    <?php endif; ?>
                <?php else: ?>
                    <div class="alert alert-warning">
                        Este estudiante tiene un postgrado inactivo. Contacte al administrador para reactivarlo.
                    </div>
                <?php endif; ?>
            </div>
        </div>
    <?php endif; ?>
</div>

<?php
include '../includes/footer.php';
$conn->close();
?>