<?php
session_start();
include '../includes/db.php';

// Verificar sesión y rol
if (!isset($_SESSION['username']) || $_SESSION['rol'] != 'coordinador2') {
    header("Location: ../login.php");
    exit();
}

// Verificar conexión a la base de datos
if ($conn->connect_error) {
    die("Error de conexión: " . $conn->connect_error);
}

$coordinador_id = $_SESSION['user_id'];

// Configuración de paginación
$registros_por_pagina = 10;
$pagina_actual = isset($_GET['pagina']) ? (int)$_GET['pagina'] : 1;
if ($pagina_actual < 1) $pagina_actual = 1;
$offset = ($pagina_actual - 1) * $registros_por_pagina;

// Parámetros de búsqueda
$busqueda = isset($_GET['busqueda']) ? trim($_GET['busqueda']) : '';
$tipo_busqueda = isset($_GET['tipo_busqueda']) ? $_GET['tipo_busqueda'] : 'cedula';
$filtro_estado = isset($_GET['estado']) ? $_GET['estado'] : 'todos';

// Consulta base para estudiantes
$sql_base = "SELECT e.id, e.cedula, e.nombres, e.apellidos, e.correo, e.telefono, e.celular, e.direccion,
            p.nombre as postgrado_nombre, ep.estado as estado_postgrado, ep.fecha_inscripcion, ep.fecha_graduacion,
            IF(ep.id IS NULL, 'sin_postgrado', 
               IF(ep.estado = 'activo', 'activo', 
                  IF(ep.estado = 'graduado', 'graduado', 'inactivo'))) as estado_estudiante
            FROM estudiantes e
            LEFT JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
            LEFT JOIN postgrados p ON ep.postgrado_id = p.id
            LEFT JOIN coordinador_postgrado cp ON ep.postgrado_id = cp.postgrado_id
            WHERE (cp.coordinador_id = ? OR ep.id IS NULL)";

$sql_contador = "SELECT COUNT(DISTINCT e.id) as total 
                FROM estudiantes e
                LEFT JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                LEFT JOIN coordinador_postgrado cp ON ep.postgrado_id = cp.postgrado_id
                WHERE (cp.coordinador_id = ? OR ep.id IS NULL)";

// Aplicar filtros
if ($filtro_estado != 'todos') {
    if ($filtro_estado == 'sin_postgrado') {
        $sql_base .= " AND ep.id IS NULL";
        $sql_contador .= " AND ep.id IS NULL";
    } else {
        $sql_base .= " AND ep.estado = ?";
        $sql_contador .= " AND ep.estado = ?";
    }
}

if (!empty($busqueda)) {
    $busqueda_like = "%$busqueda%";
    switch ($tipo_busqueda) {
        case 'cedula': $sql_base .= " AND e.cedula LIKE ?"; $sql_contador .= " AND e.cedula LIKE ?"; break;
        case 'nombres': $sql_base .= " AND e.nombres LIKE ?"; $sql_contador .= " AND e.nombres LIKE ?"; break;
        case 'apellidos': $sql_base .= " AND e.apellidos LIKE ?"; $sql_contador .= " AND e.apellidos LIKE ?"; break;
        case 'correo': $sql_base .= " AND e.correo LIKE ?"; $sql_contador .= " AND e.correo LIKE ?"; break;
        case 'postgrado': $sql_base .= " AND p.nombre LIKE ?"; $sql_contador .= " AND p.nombre LIKE ?"; break;
    }
}

$sql_base .= " GROUP BY e.id ORDER BY e.apellidos, e.nombres LIMIT ? OFFSET ?";

// Preparar consultas
$stmt_estudiantes = $conn->prepare($sql_base);
$stmt_contador = $conn->prepare($sql_contador);

// Bind parameters
$param_types = "i";
$param_values = [$coordinador_id];

if ($filtro_estado != 'todos' && $filtro_estado != 'sin_postgrado') {
    $param_types .= "s";
    $param_values[] = $filtro_estado;
}

if (!empty($busqueda)) {
    $param_types .= "s";
    $param_values[] = $busqueda_like;
}

$param_types_main = $param_types . "ii";
$param_values_main = array_merge($param_values, [$registros_por_pagina, $offset]);

$stmt_estudiantes->bind_param($param_types_main, ...$param_values_main);
$stmt_contador->bind_param($param_types, ...$param_values);

// Ejecutar consultas
$stmt_estudiantes->execute();
$result_estudiantes = $stmt_estudiantes->get_result();

$stmt_contador->execute();
$result_contador = $stmt_contador->get_result();
$total_registros = $result_contador->fetch_assoc()['total'];
$total_paginas = ceil($total_registros / $registros_por_pagina);

// Obtener detalles del estudiante si se seleccionó uno
$estudiante_detalle = null;
$postgrados_estudiante = [];
$denominaciones_estudiante = [];

if (isset($_GET['detalle_id'])) {
    $detalle_id = (int)$_GET['detalle_id'];
    
    // Información básica del estudiante
    $sql_estudiante = "SELECT * FROM estudiantes WHERE id = ?";
    $stmt_estudiante = $conn->prepare($sql_estudiante);
    $stmt_estudiante->bind_param("i", $detalle_id);
    $stmt_estudiante->execute();
    $estudiante_detalle = $stmt_estudiante->get_result()->fetch_assoc();
    
    // Postgrados del estudiante
    $sql_postgrados = "SELECT ep.*, p.nombre as postgrado_nombre, pr.nombre as promocion_nombre
                      FROM estudiante_postgrado ep
                      JOIN postgrados p ON ep.postgrado_id = p.id
                      LEFT JOIN promociones pr ON ep.promocion_id = pr.id
                      WHERE ep.estudiante_id = ?
                      ORDER BY ep.estado DESC, ep.fecha_inscripcion DESC";
    $stmt_postgrados = $conn->prepare($sql_postgrados);
    $stmt_postgrados->bind_param("i", $detalle_id);
    $stmt_postgrados->execute();
    $postgrados_estudiante = $stmt_postgrados->get_result()->fetch_all(MYSQLI_ASSOC);
    
    // Denominaciones cursadas con notas
    if (!empty($postgrados_estudiante)) {
        $sql_denominaciones = "SELECT i.*, d.codigo, d.nombre as denominacion_nombre, d.credito, d.periodo, d.tipo,
                             a.nombre as año_lectivo, s.nombre as seccion, esc.nombre as escolaridad
                             FROM inscripciones i
                             JOIN denominaciones d ON i.denominacion_id = d.id
                             LEFT JOIN años_lectivos a ON i.año_lectivo_id = a.id
                             LEFT JOIN secciones s ON i.seccion_id = s.id
                             LEFT JOIN escolaridades esc ON i.escolaridad_id = esc.id
                             WHERE i.estudiante_id = ?
                             ORDER BY d.periodo, d.codigo";
        $stmt_denominaciones = $conn->prepare($sql_denominaciones);
        $stmt_denominaciones->bind_param("i", $detalle_id);
        $stmt_denominaciones->execute();
        $denominaciones_estudiante = $stmt_denominaciones->get_result()->fetch_all(MYSQLI_ASSOC);
    }
}

$page_title = "Gestión de Estudiantes";
include '../includes/header.php';
?>

<div class="container">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h2>Gestión de Estudiantes</h2>
        <div>
            <a href="../index.php" class="btn btn-secondary">
                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-arrow-left" viewBox="0 0 16 16">
                    <path fill-rule="evenodd" d="M15 8a.5.5 0 0 0-.5-.5H2.707l3.147-3.146a.5.5 0 1 0-.708-.708l-4 4a.5.5 0 0 0 0 .708l4 4a.5.5 0 0 0 .708-.708L2.707 8.5H14.5A.5.5 0 0 0 15 8z"/>
                </svg>
                Volver al Panel
            </a>
        </div>
    </div>

    <!-- Card de Búsqueda -->
    <div class="card mb-4">
        <div class="card-header">
            <h3 class="mb-0">Búsqueda Avanzada</h3>
        </div>
        <div class="card-body">
            <form method="get" action="gestion_estudiantes.php">
                <input type="hidden" name="detalle_id" value="<?= isset($_GET['detalle_id']) ? $_GET['detalle_id'] : '' ?>">
                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group">
                            <label for="busqueda" class="form-label">Término de búsqueda:</label>
                            <input type="text" class="form-control" id="busqueda" name="busqueda" 
                                   value="<?= htmlspecialchars($busqueda) ?>" placeholder="Ingrese término de búsqueda">
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="tipo_busqueda" class="form-label">Buscar por:</label>
                            <select class="form-select" id="tipo_busqueda" name="tipo_busqueda">
                                <option value="cedula" <?= $tipo_busqueda == 'cedula' ? 'selected' : '' ?>>Cédula</option>
                                <option value="nombres" <?= $tipo_busqueda == 'nombres' ? 'selected' : '' ?>>Nombres</option>
                                <option value="apellidos" <?= $tipo_busqueda == 'apellidos' ? 'selected' : '' ?>>Apellidos</option>
                                <option value="correo" <?= $tipo_busqueda == 'correo' ? 'selected' : '' ?>>Correo</option>
                                <option value="postgrado" <?= $tipo_busqueda == 'postgrado' ? 'selected' : '' ?>>Postgrado</option>
                            </select>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="estado" class="form-label">Filtrar por estado:</label>
                            <select class="form-select" id="estado" name="estado">
                                <option value="todos" <?= $filtro_estado == 'todos' ? 'selected' : '' ?>>Todos</option>
                                <option value="activo" <?= $filtro_estado == 'activo' ? 'selected' : '' ?>>Activos</option>
                                <option value="graduado" <?= $filtro_estado == 'graduado' ? 'selected' : '' ?>>Graduados</option>
                                <option value="sin_postgrado" <?= $filtro_estado == 'sin_postgrado' ? 'selected' : '' ?>>Sin postgrado</option>
                            </select>
                        </div>
                    </div>
                </div>
                <div class="row mt-3">
                    <div class="col-md-12 d-flex justify-content-between">
                        <div>
                            <button type="submit" class="btn btn-primary me-2">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-search" viewBox="0 0 16 16">
                                    <path d="M11.742 10.344a6.5 6.5 0 1 0-1.397 1.398h-.001c.03.04.062.078.098.115l3.85 3.85a1 1 0 0 0 1.415-1.414l-3.85-3.85a1.007 1.007 0 0 0-.115-.1zM12 6.5a5.5 5.5 0 1 1-11 0 5.5 5.5 0 0 1 11 0z"/>
                                </svg>
                                Buscar
                            </button>
                            <a href="gestion_estudiantes.php" class="btn btn-secondary">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-arrow-counterclockwise" viewBox="0 0 16 16">
                                    <path fill-rule="evenodd" d="M8 3a5 5 0 1 1-4.546 2.914.5.5 0 0 0-.908-.417A6 6 0 1 0 8 2v1z"/>
                                    <path d="M8 4.466V.534a.25.25 0 0 0-.41-.192L5.23 2.308a.25.25 0 0 0 0 .384l2.36 1.966A.25.25 0 0 0 8 4.466z"/>
                                </svg>
                                Limpiar
                            </a>
                        </div>
                        <?php if (!empty($busqueda)): ?>
                            <div class="text-muted">
                                <?= $total_registros ?> resultados encontrados
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Lista de Estudiantes -->
    <div class="card mb-4">
        <div class="card-header d-flex justify-content-between align-items-center">
            <h3 class="mb-0">Lista de Estudiantes</h3>
            <div class="text-muted">
                Mostrando <?= min($registros_por_pagina, $result_estudiantes->num_rows) ?> de <?= $total_registros ?> registros
            </div>
        </div>
        <div class="card-body">
            <?php if ($result_estudiantes->num_rows > 0): ?>
                <div class="table-responsive">
                    <table class="table table-striped table-hover">
                        <thead class="table-light">
                            <tr>
                                <th>Cédula</th>
                                <th>Apellidos</th>
                                <th>Nombres</th>
                                <th>Correo</th>
                                <th>Postgrado</th>
                                <th>Estado</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php while ($estudiante = $result_estudiantes->fetch_assoc()): ?>
                                <tr>
                                    <td><?= htmlspecialchars($estudiante['cedula']) ?></td>
                                    <td><?= htmlspecialchars($estudiante['apellidos']) ?></td>
                                    <td><?= htmlspecialchars($estudiante['nombres']) ?></td>
                                    <td><?= htmlspecialchars($estudiante['correo']) ?></td>
                                    <td><?= $estudiante['postgrado_nombre'] ? htmlspecialchars($estudiante['postgrado_nombre']) : 'Sin postgrado' ?></td>
                                    <td>
                                        <?php 
                                        $estado_clase = '';
                                        switch ($estudiante['estado_estudiante']) {
                                            case 'activo': $estado_clase = 'success'; break;
                                            case 'graduado': $estado_clase = 'info'; break;
                                            case 'inactivo': $estado_clase = 'warning'; break;
                                            default: $estado_clase = 'secondary'; break;
                                        }
                                        ?>
                                        <span class="badge bg-<?= $estado_clase ?>">
                                            <?= ucfirst($estudiante['estado_estudiante']) ?>
                                        </span>
                                    </td>
                                    <td>
                                        <a href="?<?= http_build_query(array_merge($_GET, ['detalle_id' => $estudiante['id'], 'pagina' => 1])) ?>#ficha" 
                                           class="btn btn-sm btn-primary" title="Ver ficha">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-file-earmark-text" viewBox="0 0 16 16">
                                                <path d="M5.5 7a.5.5 0 0 0 0 1h5a.5.5 0 0 0 0-1h-5zM5 9.5a.5.5 0 0 1 .5-.5h5a.5.5 0 0 1 0 1h-5a.5.5 0 0 1-.5-.5zm0 2a.5.5 0 0 1 .5-.5h2a.5.5 0 0 1 0 1h-2a.5.5 0 0 1-.5-.5z"/>
                                                <path d="M9.5 0H4a2 2 0 0 0-2 2v12a2 2 0 0 0 2 2h8a2 2 0 0 0 2-2V4.5L9.5 0zm0 1v2A1.5 1.5 0 0 0 11 4.5h2V14a1 1 0 0 1-1 1H4a1 1 0 0 1-1-1V2a1 1 0 0 1 1-1h5.5z"/>
                                            </svg>
                                        </a>
                                        <a href="inscripcion.php?estudiante_id=<?= $estudiante['id'] ?>" 
                                           class="btn btn-sm btn-success" title="Gestionar">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-pencil-square" viewBox="0 0 16 16">
                                                <path d="M15.502 1.94a.5.5 0 0 1 0 .706L14.459 3.69l-2-2L13.502.646a.5.5 0 0 1 .707 0l1.293 1.293zm-1.75 2.456-2-2L4.939 9.21a.5.5 0 0 0-.121.196l-.805 2.414a.25.25 0 0 0 .316.316l2.414-.805a.5.5 0 0 0 .196-.12l6.813-6.814z"/>
                                                <path fill-rule="evenodd" d="M1 13.5A1.5 1.5 0 0 0 2.5 15h11a1.5 1.5 0 0 0 1.5-1.5v-6a.5.5 0 0 0-1 0v6a.5.5 0 0 1-.5.5h-11a.5.5 0 0 1-.5-.5v-11a.5.5 0 0 1 .5-.5H9a.5.5 0 0 0 0-1H2.5A1.5 1.5 0 0 0 1 2.5v11z"/>
                                            </svg>
                                        </a>
                                    </td>
                                </tr>
                            <?php endwhile; ?>
                        </tbody>
                    </table>
                </div>
                
                <!-- Paginación -->
                <?php if ($total_paginas > 1): ?>
                    <nav aria-label="Paginación">
                        <ul class="pagination justify-content-center mt-4">
                            <?php if ($pagina_actual > 1): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['pagina' => 1])) ?>#ficha" aria-label="Primera">
                                        <span aria-hidden="true">&laquo;&laquo;</span>
                                    </a>
                                </li>
                                <li class="page-item">
                                    <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['pagina' => $pagina_actual - 1])) ?>#ficha" aria-label="Anterior">
                                        <span aria-hidden="true">&laquo;</span>
                                    </a>
                                </li>
                            <?php endif; ?>
                            
                            <?php 
                            $inicio = max(1, $pagina_actual - 2);
                            $fin = min($total_paginas, $pagina_actual + 2);
                            
                            if ($inicio > 1) {
                                echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                            }
                            
                            for ($i = $inicio; $i <= $fin; $i++): ?>
                                <li class="page-item <?= $i == $pagina_actual ? 'active' : '' ?>">
                                    <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['pagina' => $i])) ?>#ficha"><?= $i ?></a>
                                </li>
                            <?php endfor;
                            
                            if ($fin < $total_paginas) {
                                echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                            }
                            ?>
                            
                            <?php if ($pagina_actual < $total_paginas): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['pagina' => $pagina_actual + 1])) ?>#ficha" aria-label="Siguiente">
                                        <span aria-hidden="true">&raquo;</span>
                                    </a>
                                </li>
                                <li class="page-item">
                                    <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['pagina' => $total_paginas])) ?>#ficha" aria-label="Última">
                                        <span aria-hidden="true">&raquo;&raquo;</span>
                                    </a>
                                </li>
                            <?php endif; ?>
                        </ul>
                    </nav>
                <?php endif; ?>
            <?php else: ?>
                <div class="alert alert-info text-center">
                    No se encontraron estudiantes con los criterios de búsqueda seleccionados.
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Ficha del Estudiante -->
    <?php if ($estudiante_detalle): ?>
        <div class="card mb-4" id="ficha">
            <div class="card-header">
                <h3 class="mb-0">Ficha del Estudiante: <?= htmlspecialchars($estudiante_detalle['nombres'] . ' ' . $estudiante_detalle['apellidos']) ?></h3>
            </div>
            <div class="card-body">
                <!-- Información Personal -->
                <div class="mb-4">
                    <h4>Información Personal</h4>
                    <div class="row">
                        <div class="col-md-3">
                            <p><strong>Cédula:</strong> <?= htmlspecialchars($estudiante_detalle['cedula']) ?></p>
                        </div>
                        <div class="col-md-3">
                            <p><strong>Correo:</strong> <?= htmlspecialchars($estudiante_detalle['correo']) ?></p>
                        </div>
                        <div class="col-md-3">
                            <p><strong>Teléfono:</strong> <?= htmlspecialchars($estudiante_detalle['telefono']) ?: 'No registrado' ?></p>
                        </div>
                        <div class="col-md-3">
                            <p><strong>Celular:</strong> <?= htmlspecialchars($estudiante_detalle['celular']) ?: 'No registrado' ?></p>
                        </div>
                    </div>
                    <div class="row">
                        <div class="col-md-12">
                            <p><strong>Dirección:</strong> <?= htmlspecialchars($estudiante_detalle['direccion']) ?: 'No registrada' ?></p>
                        </div>
                    </div>
                </div>

                <!-- Historial de Postgrados -->
                <div class="mb-4">
                    <h4>Historial de Postgrados</h4>
                    <?php if (!empty($postgrados_estudiante)): ?>
                        <div class="table-responsive">
                            <table class="table table-striped">
                                <thead>
                                    <tr>
                                        <th>Postgrado</th>
                                        <th>Promoción</th>
                                        <th>Fecha Inscripción</th>
                                        <th>Fecha Graduación</th>
                                        <th>Estado</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($postgrados_estudiante as $postgrado): ?>
                                        <tr>
                                            <td><?= htmlspecialchars($postgrado['postgrado_nombre']) ?></td>
                                            <td><?= htmlspecialchars($postgrado['promocion_nombre'] ?? 'No asignada') ?></td>
                                            <td><?= date('d/m/Y', strtotime($postgrado['fecha_inscripcion'])) ?></td>
                                            <td><?= $postgrado['fecha_graduacion'] ? date('d/m/Y', strtotime($postgrado['fecha_graduacion'])) : 'No graduado' ?></td>
                                            <td>
                                                <?php 
                                                $estado_clase = $postgrado['estado'] == 'activo' ? 'success' : 
                                                              ($postgrado['estado'] == 'graduado' ? 'info' : 'warning');
                                                ?>
                                                <span class="badge bg-<?= $estado_clase ?>">
                                                    <?= ucfirst($postgrado['estado']) ?>
                                                </span>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php else: ?>
                        <div class="alert alert-info">
                            El estudiante no está inscrito en ningún postgrado.
                        </div>
                    <?php endif; ?>
                </div>

                <!-- Denominaciones Cursadas -->
                <?php if (!empty($denominaciones_estudiante)): ?>
                    <div class="mb-4">
                        <h4>Denominaciones Cursadas</h4>
                        <?php 
                        // Agrupar denominaciones por postgrado y período
                        $denominaciones_agrupadas = [];
                        foreach ($denominaciones_estudiante as $denominacion) {
                            $periodo = $denominacion['periodo'];
                            if (!isset($denominaciones_agrupadas[$periodo])) {
                                $denominaciones_agrupadas[$periodo] = [];
                            }
                            $denominaciones_agrupadas[$periodo][] = $denominacion;
                        }
                        ksort($denominaciones_agrupadas);
                        ?>

                        <?php foreach ($denominaciones_agrupadas as $periodo => $denominaciones): ?>
                            <h5 class="mt-3">Período <?= $periodo ?></h5>
                            <div class="table-responsive">
                                <table class="table table-striped">
                                    <thead>
                                        <tr>
                                            <th>Código</th>
                                            <th>Denominación</th>
                                            <th>Créditos</th>
                                            <th>Tipo</th>
                                            <th>Año Lectivo</th>
                                            <th>Escolaridad</th>
                                            <th>Nota</th>
                                            <th>Estado</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($denominaciones as $denominacion): ?>
                                            <tr>
                                                <td><?= htmlspecialchars($denominacion['codigo']) ?></td>
                                                <td><?= htmlspecialchars($denominacion['denominacion_nombre']) ?></td>
                                                <td><?= htmlspecialchars($denominacion['credito']) ?></td>
                                                <td><?= htmlspecialchars($denominacion['tipo']) ?></td>
                                                <td><?= htmlspecialchars($denominacion['año_lectivo']) ?></td>
                                                <td><?= htmlspecialchars($denominacion['escolaridad']) ?></td>
                                                <td>
                                                    <?php if ($denominacion['tipo'] == 'puntaje'): ?>
                                                        <?= htmlspecialchars($denominacion['nota_definitiva']) ?: 'En curso' ?>
                                                    <?php else: ?>
                                                        <?= htmlspecialchars($denominacion['nota_definitiva']) ?: 'En curso' ?>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <?php 
                                                    $estado_den = $denominacion['estado'] == 'finalizada' ? 'Completada' : 'En curso';
                                                    $clase_den = $denominacion['estado'] == 'finalizada' ? 'success' : 'warning';
                                                    ?>
                                                    <span class="badge bg-<?= $clase_den ?>"><?= $estado_den ?></span>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php elseif (!empty($postgrados_estudiante)): ?>
                    <div class="alert alert-info">
                        El estudiante no tiene denominaciones registradas.
                    </div>
                <?php endif; ?>
            </div>
        </div>
    <?php endif; ?>
</div>

<?php
include '../includes/footer.php';
$conn->close();
?>