<?php
session_start();
include '../includes/db.php';

// Verificar sesión y rol
if (!isset($_SESSION['username']) || $_SESSION['rol'] != 'coordinador2') {
    header("Location: ../login.php");
    exit();
}

// Obtener postgrados asignados al coordinador
$coordinador_id = $_SESSION['user_id'];
$postgrados = [];

$sql_postgrados = "SELECT p.id, p.nombre 
                  FROM coordinador_postgrado cp
                  JOIN postgrados p ON cp.postgrado_id = p.id
                  WHERE cp.coordinador_id = ?";
$stmt_postgrados = $conn->prepare($sql_postgrados);
$stmt_postgrados->bind_param("i", $coordinador_id);
$stmt_postgrados->execute();
$result_postgrados = $stmt_postgrados->get_result();

while ($row = $result_postgrados->fetch_assoc()) {
    $postgrados[] = $row;
}

// Procesar el formulario
$error = '';
$success = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    // Validar datos
    $cedula = trim($_POST['cedula']);
    $nombres = trim($_POST['nombres']);
    $apellidos = trim($_POST['apellidos']);
    $correo = trim($_POST['correo']);
    $telefono = trim($_POST['telefono'] ?? '');
    $celular = trim($_POST['celular'] ?? '');
    $direccion = trim($_POST['direccion'] ?? '');
    $postgrado_id = isset($_POST['postgrado_id']) ? intval($_POST['postgrado_id']) : null;

    // Validaciones básicas
    if (empty($cedula) || empty($nombres) || empty($apellidos) || empty($correo)) {
        $_SESSION['error'] = "Los campos marcados con * son obligatorios";
    } else {
        // Verificar si la cédula ya existe
        $sql_verificar = "SELECT id FROM estudiantes WHERE cedula = ?";
        $stmt_verificar = $conn->prepare($sql_verificar);
        $stmt_verificar->bind_param("s", $cedula);
        $stmt_verificar->execute();
        $result_verificar = $stmt_verificar->get_result();

        if ($result_verificar->num_rows > 0) {
            $_SESSION['error'] = "Ya existe un estudiante con esta cédula";
        } else {
            // Iniciar transacción
            $conn->begin_transaction();

            try {
                // Insertar estudiante
                $sql_estudiante = "INSERT INTO estudiantes 
                                 (cedula, nombres, apellidos, correo, telefono, celular, direccion)
                                 VALUES (?, ?, ?, ?, ?, ?, ?)";
                $stmt_estudiante = $conn->prepare($sql_estudiante);
                $stmt_estudiante->bind_param("sssssss", $cedula, $nombres, $apellidos, $correo, $telefono, $celular, $direccion);
                $stmt_estudiante->execute();
                $estudiante_id = $stmt_estudiante->insert_id;

                // Asignar al postgrado si se seleccionó uno
                if ($postgrado_id) {
                    $sql_asignar = "INSERT INTO estudiante_postgrado 
                                   (estudiante_id, postgrado_id, fecha_inscripcion, estado)
                                   VALUES (?, ?, NOW(), 'activo')";
                    $stmt_asignar = $conn->prepare($sql_asignar);
                    $stmt_asignar->bind_param("ii", $estudiante_id, $postgrado_id);
                    $stmt_asignar->execute();
                }

                $conn->commit();
                $_SESSION['success'] = $postgrado_id 
                    ? "Estudiante registrado y asignado al postgrado exitosamente" 
                    : "Estudiante registrado exitosamente (sin postgrado asignado)";
                
                // Limpiar campos del formulario
                $_POST = [];
            } catch (Exception $e) {
                $conn->rollback();
                $_SESSION['error'] = "Error al registrar el estudiante: " . $e->getMessage();
            }
        }
    }
    // Redirigir para evitar reenvío de formulario
    header("Location: crear_estudiante.php");
    exit();
}

$page_title = "Registrar Nuevo Estudiante";
include '../includes/header.php';
?>
<style>
    .form-label {
        font-weight: 500;
        margin-bottom: 0.3rem;
    }
    .required-field::after {
        content: " *";
        color: #dc3545;
    }
    .form-section {
        margin-bottom: 1.5rem;
    }
    .form-section h5 {
        font-size: 1rem;
        color: #495057;
        margin-bottom: 1rem;
        padding-bottom: 0.5rem;
        border-bottom: 1px solid #dee2e6;
    }
</style>

<div class="card">
    <div class="card-body">
        <h2 class="card-title mb-4">Registrar Nuevo Estudiante</h2>
        
        <form method="post">
            <div class="form-section">
                <h5>Información Personal</h5>
                <div class="row g-3">
                    <div class="col-md-6">
                        <label for="cedula" class="form-label required-field">Cédula</label>
                        <input type="text" class="form-control" id="cedula" name="cedula" 
                               value="<?= htmlspecialchars($_POST['cedula'] ?? '') ?>" required>
                    </div>
                    <div class="col-md-6">
                        <label for="nombres" class="form-label required-field">Nombres</label>
                        <input type="text" class="form-control" id="nombres" name="nombres" 
                               value="<?= htmlspecialchars($_POST['nombres'] ?? '') ?>" required>
                    </div>
                    <div class="col-md-6">
                        <label for="apellidos" class="form-label required-field">Apellidos</label>
                        <input type="text" class="form-control" id="apellidos" name="apellidos" 
                               value="<?= htmlspecialchars($_POST['apellidos'] ?? '') ?>" required>
                    </div>
                    <div class="col-md-6">
                        <label for="correo" class="form-label required-field">Correo Electrónico</label>
                        <input type="email" class="form-control" id="correo" name="correo" 
                               value="<?= htmlspecialchars($_POST['correo'] ?? '') ?>" required>
                    </div>
                </div>
            </div>
            
            <div class="form-section">
                <h5>Información de Contacto</h5>
                <div class="row g-3">
                    <div class="col-md-6">
                        <label for="telefono" class="form-label">Teléfono</label>
                        <input type="tel" class="form-control" id="telefono" name="telefono" 
                               value="<?= htmlspecialchars($_POST['telefono'] ?? '') ?>">
                    </div>
                    <div class="col-md-6">
                        <label for="celular" class="form-label">Celular</label>
                        <input type="tel" class="form-control" id="celular" name="celular" 
                               value="<?= htmlspecialchars($_POST['celular'] ?? '') ?>">
                    </div>
                    <div class="col-12">
                        <label for="direccion" class="form-label">Dirección</label>
                        <textarea class="form-control" id="direccion" name="direccion" rows="2"><?= htmlspecialchars($_POST['direccion'] ?? '') ?></textarea>
                    </div>
                </div>
            </div>
            
            <?php if (!empty($postgrados)): ?>
            <div class="form-section">
                <h5>Asignación a Postgrado</h5>
                <div class="mb-3">
                    <label for="postgrado_id" class="form-label">Seleccionar Postgrado</label>
                    <select class="form-select" id="postgrado_id" name="postgrado_id">
                        <option value="">-- No asignar a postgrado --</option>
                        <?php foreach ($postgrados as $postgrado): ?>
                            <option value="<?= $postgrado['id'] ?>" <?= isset($_POST['postgrado_id']) && $_POST['postgrado_id'] == $postgrado['id'] ? 'selected' : '' ?>>
                                <?= htmlspecialchars($postgrado['nombre']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <small class="text-muted">Opcional - Puede asignar al estudiante a un postgrado después</small>
                </div>
            </div>
            <?php endif; ?>
            
            <div class="d-flex justify-content-between mt-4">
                <a href="index.php" class="btn btn-outline-secondary">Cancelar</a>
                <button type="submit" class="btn btn-primary">Registrar Estudiante</button>
            </div>
        </form>
    </div>
</div>

<?php
include '../includes/footer.php';
$conn->close();
?>
