<?php
session_start();

// Verificar sesión y rol de administrador
if (!isset($_SESSION['username']) || $_SESSION['rol'] != 'coordinador2') {
    header("Location: ../login.php");
    exit();
}

// Verificar e incluir mPDF
$mpdfPath = __DIR__ . '/../vendor/autoload.php';
if (!file_exists($mpdfPath)) {
    die("Error: La biblioteca mPDF no está instalada. Ejecuta 'composer require mpdf/mpdf' para instalarla.");
}
require_once $mpdfPath;

include '../includes/db.php';

// Verificar si se proporcionó el estudiante_id y postgrado_id en la URL
if (!isset($_GET['estudiante_id']) || !isset($_GET['postgrado_id'])) {
    die("Error: No se proporcionó el ID del estudiante o del postgrado.");
}

// Inicializar variables
$estudiante_id = intval($_GET['estudiante_id']);
$postgrado_id = intval($_GET['postgrado_id']);
$detalles_estudiante = [];
$historial_academico = [];
$firmas = [];

// Función para convertir números a letras
function numeroALetras($numero) {
    $unidades = ['', 'UNO', 'DOS', 'TRES', 'CUATRO', 'CINCO', 'SEIS', 'SIETE', 'OCHO', 'NUEVE'];
    $decenas = ['', 'DIEZ', 'VEINTE', 'TREINTA', 'CUARENTA', 'CINCUENTA', 'SESENTA', 'SETENTA', 'OCHENTA', 'NOVENTA'];
    $especiales = ['', 'ONCE', 'DOCE', 'TRECE', 'CATORCE', 'QUINCE', 'DIECISEIS', 'DIECISIETE', 'DIECIOCHO', 'DIECINUEVE'];
    
    if ($numero == 0) return 'CERO';
    if ($numero == 10) return 'DIEZ';
    if ($numero == 20) return 'VEINTE';
    
    if ($numero < 10) return $unidades[$numero];
    if ($numero > 10 && $numero < 20) return $especiales[$numero - 10];
    
    $unidad = $numero % 10;
    $decena = floor($numero / 10);
    
    if ($unidad == 0) {
        return $decenas[$decena];
    } else {
        return $decenas[$decena] . ($decena == 2 ? 'I' : '') . $unidades[$unidad];
    }
}

// Función para formatear fecha en formato día/mes/año
function formatearFecha($fecha) {
    if (empty($fecha)) {
        return 'N/A';
    }
    
    try {
        $fechaObj = new DateTime($fecha);
        return $fechaObj->format('d/m/Y');
    } catch (Exception $e) {
        return 'N/A';
    }
}

if ($estudiante_id > 0 && $postgrado_id > 0) {
    // Obtener detalles del estudiante
    $sql_detalles = "SELECT e.*, p.nombre AS postgrado_nombre, p.id AS postgrado_id,
                    ep.fecha_graduacion, pr.nombre AS promocion
                    FROM estudiantes e
                    JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                    JOIN postgrados p ON ep.postgrado_id = p.id
                    LEFT JOIN promociones pr ON ep.promocion_id = pr.id
                    WHERE e.id = ? AND p.id = ? AND (ep.estado = 'graduado' OR ep.estado = 'activo')
                    LIMIT 1";
    
    $stmt_detalles = $conn->prepare($sql_detalles);
    if ($stmt_detalles) {
        $stmt_detalles->bind_param("ii", $estudiante_id, $postgrado_id);
        $stmt_detalles->execute();
        $result_detalles = $stmt_detalles->get_result();
        $detalles_estudiante = $result_detalles->fetch_assoc() ?: [];
        $stmt_detalles->close();
    }

    if (empty($detalles_estudiante)) {
        die("Error: No se encontró al estudiante o no está activo/graduado en el postgrado seleccionado.");
    }

    // Obtener total de asignaturas inscritas
    $sql_todas_inscripciones = "SELECT COUNT(DISTINCT i.denominacion_id) AS total_inscritas
                               FROM inscripciones i
                               JOIN denominaciones d ON i.denominacion_id = d.id
                               WHERE i.estudiante_id = ?
                               AND d.postgrado_id = ?";
    
    $stmt_todas = $conn->prepare($sql_todas_inscripciones);
    if ($stmt_todas) {
        $stmt_todas->bind_param("ii", $estudiante_id, $postgrado_id);
        $stmt_todas->execute();
        $result_todas = $stmt_todas->get_result();
        $total_inscritas_row = $result_todas->fetch_assoc();
        $total_asignaturas = $total_inscritas_row['total_inscritas'] ?? 0;
        $stmt_todas->close();
    }

    // Obtener historial académico
    $sql_historial = "SELECT d.codigo AS asignatura, 
                             d.nombre AS denominacion, 
                             d.credito,
                             d.tipo,
                             i.nota_definitiva, 
                             i.nota_final,
                             al.nombre AS año_lectivo
                      FROM inscripciones i
                      INNER JOIN denominaciones d ON i.denominacion_id = d.id
                      LEFT JOIN años_lectivos al ON i.año_lectivo_id = al.id
                      WHERE i.estudiante_id = ? 
                      AND d.postgrado_id = ?
                      AND i.nota_definitiva IS NOT NULL
                      ORDER BY d.periodo, d.codigo";

    $stmt_historial = $conn->prepare($sql_historial);
    if ($stmt_historial) {
        $stmt_historial->bind_param("ii", $estudiante_id, $postgrado_id);
        $stmt_historial->execute();
        $result_historial = $stmt_historial->get_result();
        $historial_academico = $result_historial->fetch_all(MYSQLI_ASSOC) ?: [];
        $stmt_historial->close();
    }

    // Obtener asignaturas aprobadas
    $sql_aprobadas = "SELECT COUNT(DISTINCT i.denominacion_id) AS total_aprobadas
                     FROM inscripciones i
                     INNER JOIN denominaciones d ON i.denominacion_id = d.id
                     WHERE i.estudiante_id = ? 
                     AND d.postgrado_id = ?
                     AND i.nota_definitiva IS NOT NULL
                     AND (
                         (d.tipo = 'puntaje' AND CAST(i.nota_definitiva AS DECIMAL(10,2)) >= 10) OR
                         (d.tipo = 'Aprobado_Reprobado' AND UPPER(i.nota_definitiva) = 'APROBADO')
                     )";
    
    $stmt_aprobadas = $conn->prepare($sql_aprobadas);
    $asignaturas_aprobadas = 0;
    if ($stmt_aprobadas) {
        $stmt_aprobadas->bind_param("ii", $estudiante_id, $postgrado_id);
        $stmt_aprobadas->execute();
        $result_aprobadas = $stmt_aprobadas->get_result();
        $row = $result_aprobadas->fetch_assoc();
        $asignaturas_aprobadas = $row['total_aprobadas'] ?? 0;
        $stmt_aprobadas->close();
    }

    // Calcular promedios
    $total_creditos = 0;
    $nota_simple = 0;
    $nota_ponderada = 0;
    $asignaturas_puntaje_aprobadas = 0;

    foreach ($historial_academico as $h) {
        if ($h['tipo'] == 'puntaje' && is_numeric($h['nota_definitiva']) && floatval($h['nota_definitiva']) >= 10) {
            $valor = floatval($h['nota_definitiva']);
            $credito = intval($h['credito']);
            $nota_simple += $valor;
            $nota_ponderada += ($valor * $credito);
            $total_creditos += $credito;
            $asignaturas_puntaje_aprobadas++;
        }
    }

    $nota_simple_promedio = $asignaturas_puntaje_aprobadas > 0 ? $nota_simple / $asignaturas_puntaje_aprobadas : 0;
    $nota_ponderada_promedio = $total_creditos > 0 ? $nota_ponderada / $total_creditos : 0;
}

// Obtener firmas
$sql_firmas = "SELECT nombre, cargo FROM firmas_constancia ORDER BY id";
$result_firmas = $conn->query($sql_firmas);

if ($result_firmas && $result_firmas->num_rows > 0) {
    $firmas = $result_firmas->fetch_all(MYSQLI_ASSOC);
} else {
    $firmas = [
        ['nombre' => 'Prof. MARIO PATIÑO TORRES', 'cargo' => 'DECANO'],
        ['nombre' => 'Prof. JOSÈ RAMÒN GARCÌA RODRÌGUEZ', 'cargo' => 'COORDINADOR DE ESTUDIOS DE POSTGRADO MED'],
        ['nombre' => 'Ing. ZULAY DÌAZ', 'cargo' => 'ÀREA DE POSTGRADO CENTRAL'],
        ['nombre' => 'JESSE LEE HERNÀNDEZ', 'cargo' => 'CONTROL DE ESTUDIOS DE POSTGRADO MED']
    ];
}

try {
    // Configurar PDF
    $mpdf = new \Mpdf\Mpdf([
        'mode' => 'utf-8',
        'format' => 'A4',
        'margin_left' => 10,
        'margin_right' => 10,
        'margin_top' => 15,
        'margin_bottom' => 15,
        'default_font_size' => 10,
        'default_font' => 'arial',
        'tempDir' => __DIR__ . '/tmp'
    ]);

    $mpdf->SetTitle('Constancia de Notas - ' . $detalles_estudiante['nombres'] . ' ' . $detalles_estudiante['apellidos']);
    $mpdf->SetAuthor('Universidad Central de Venezuela');
    $mpdf->SetCreator('Sistema de Postgrado');

    $html = '<!DOCTYPE html>
    <html lang="es">
    <head>
        <meta charset="UTF-8">
        <title>Constancia de Notas</title>
        <style>
            body { font-family: Arial, sans-serif; margin: 0; padding: 0; }
            .constancia { width: 100%; padding: 20px; box-sizing: border-box; }
            .header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px; }
            .header h1 { font-size: 14px; margin: 0; text-align: left; line-height: 1.5; }
            .header h2 { font-size: 12px; margin: 0; text-align: right; line-height: 1.5; }
            .datos-estudiante { margin-bottom: 20px; text-align: left; }
            .datos-estudiante p { margin: 5px 0; font-size: 12px; line-height: 1.5; }
            .linea-negra { border-top: 2px solid #000; margin: 20px 0; }
            table { width: 100%; border-collapse: collapse; margin-top: 10px; page-break-inside: avoid; }
            th, td { padding: 6px; text-align: left; font-size: 10px; border: 1px solid #ddd; }
            th { background-color: #f4f4f4; }
            .resumen { margin: 50px auto 0; width: 90%; }
            .resumen h3 { text-align: center; margin-bottom: 15px; }
            .resumen-line { display: flex; justify-content: space-between; margin: 8px 0; }
            .resumen-label { flex-grow: 1; font-size: 12px; }
            .resumen-dots { flex-grow: 0; white-space: nowrap; font-size: 12px; padding: 0 5px; }
            .resumen-value { width: 80px; text-align: right; font-size: 12px; font-weight: bold; }
            .firmas { margin-top: 100px; text-align: center; }
            .firmas table { width: 100%; border-collapse: collapse; border: none; }
            .firmas td { width: 50%; text-align: center; vertical-align: top; padding: 30px; border: none; }
            .firmas p { margin: 0; font-size: 12px; line-height: 1.5; }
            .page-break { page-break-after: always; }
            .avoid-break { page-break-inside: avoid; }
        </style>
    </head>
    <body>';

    if ($estudiante_id > 0 && $postgrado_id > 0 && !empty($detalles_estudiante)) {
        $denominaciones_por_pagina = 28;
        $total_paginas_denominaciones = ceil(count($historial_academico) / $denominaciones_por_pagina);
        $total_paginas = $total_paginas_denominaciones + 1;

        // Primero generamos todas las páginas de asignaturas
        for ($pagina = 1; $pagina <= $total_paginas_denominaciones; $pagina++) {
            $html .= '
            <div class="constancia' . ($pagina < $total_paginas_denominaciones ? ' page-break' : '') . '">
                <div class="header">
                    <h1>UNIVERSIDAD CENTRAL DE VENEZUELA<br>OFICINA CENTRAL DE CONTROL DE ESTUDIOS<br>ÁREA DE POSTGRADO</h1>
                    <h2>FECHA DE GRADO: ' . formatearFecha($detalles_estudiante['fecha_graduacion']) . '<br>Página ' . $pagina . ' de ' . $total_paginas . '</h2>
                </div>

                <div class="datos-estudiante">
                    <h3 style="text-align: center;">*** FICHA DE GRADO ***</h3>
                    <p><strong>FACULTAD DE MEDICINA</strong></p>
                    <p><strong>POSTGRADO DE MEDICINA</strong></p>
                    <p>&nbsp;</p>
                    <p><strong>ESPECIALISTA EN:</strong> ' . htmlspecialchars($detalles_estudiante['postgrado_nombre'] ?? 'ANESTESIOLOGÍA – HDL') . '</p>
                    <p><strong>ESTUDIANTE:</strong> ' . htmlspecialchars($detalles_estudiante['nombres'] . ' ' . $detalles_estudiante['apellidos']) . '</p>
                    <p><strong>PASAPORTE:</strong> ' . htmlspecialchars($detalles_estudiante['cedula']) . '</p>
                </div>

                <div class="linea-negra"></div>

                <table>
                    <thead>
                        <tr>
                            <th>ASIGNATURA</th>
                            <th>DENOMINACIÓN</th>
                            <th>CRÉDITO</th>
                            <th>NOTA DEFINITIVA</th>
                            <th>AÑO LECTIVO</th>
                        </tr>
                    </thead>
                    <tbody>';

            $inicio = ($pagina - 1) * $denominaciones_por_pagina;
            $denominaciones_pagina = array_slice($historial_academico, $inicio, $denominaciones_por_pagina);

            foreach ($denominaciones_pagina as $historial) {
                $nota_display = 'N/A';
                
                if ($historial['tipo'] == 'Aprobado_Reprobado') {
                    $estado = strtoupper($historial['nota_definitiva']);
                    $letra = ($estado == 'APROBADO') ? 'A' : 'R';
                    $nota_display = $letra . ' (' . $estado . ')';
                } elseif (is_numeric($historial['nota_definitiva'])) {
                    $nota_num = (int)$historial['nota_definitiva'];
                    $nota_display = $nota_num . ' (' . numeroALetras($nota_num) . ')';
                }
                
                $html .= '
                        <tr>
                            <td>' . htmlspecialchars($historial['asignatura']) . '</td>
                            <td>' . htmlspecialchars($historial['denominacion']) . '</td>
                            <td>' . htmlspecialchars($historial['credito']) . '</td>
                            <td>' . $nota_display . '</td>
                            <td>' . htmlspecialchars($historial['año_lectivo']) . '</td>
                        </tr>';
            }

            $html .= '
                    </tbody>
                </table>
            </div>';
        }

        // Ahora generamos la página final con el resumen y firmas
        $html .= '
        <div class="constancia avoid-break">
            <div class="header">
                <h1>UNIVERSIDAD CENTRAL DE VENEZUELA<br>OFICINA CENTRAL DE CONTROL DE ESTUDIOS<br>ÁREA DE POSTGRADO</h1>
                <h2>FECHA DE GRADO: ' . formatearFecha($detalles_estudiante['fecha_graduacion']) . '<br>Página ' . $total_paginas . ' de ' . $total_paginas . '</h2>
            </div>

            <div class="datos-estudiante">
                <h3 style="text-align: center;">*** FICHA DE GRADO ***</h3>
                <p><strong>FACULTAD DE MEDICINA</strong></p>
                <p><strong>POSTGRADO DE MEDICINA</strong></p>
                <p>&nbsp;</p>
                <p><strong>ESPECIALISTA EN:</strong> ' . htmlspecialchars($detalles_estudiante['postgrado_nombre'] ?? 'ANESTESIOLOGÍA – HDL') . '</p>
                <p><strong>ESTUDIANTE:</strong> ' . htmlspecialchars($detalles_estudiante['nombres'] . ' ' . $detalles_estudiante['apellidos']) . '</p>
                <p><strong>PASAPORTE:</strong> ' . htmlspecialchars($detalles_estudiante['cedula']) . '</p>
            </div>

            <div class="resumen avoid-break">
                <h3>RESUMEN ACADÉMICO</h3>
                <div class="resumen-line">
                    <span class="resumen-label">NRO. DE UNIDADES INSCRITAS</span>
                    <span class="resumen-dots">.....................................</span>
                    <span class="resumen-value">' . $total_asignaturas . '</span>
                </div>
                <div class="resumen-line">
                    <span class="resumen-label">NRO. DE UNIDADES APROBADAS</span>
                    <span class="resumen-dots">.....................................</span>
                    <span class="resumen-value">' . $asignaturas_aprobadas . '</span>
                </div>
                <div class="resumen-line">
                    <span class="resumen-label">NRO. DE UNIDADES APROBADAS POR RECONOC. CRED</span>
                    <span class="resumen-dots">...</span>
                    <span class="resumen-value">0</span>
                </div>
                <div class="resumen-line">
                    <span class="resumen-label">NRO. DE UNIDADES APROBADAS POR ASIG.CRED</span>
                    <span class="resumen-dots">............</span>
                    <span class="resumen-value">0</span>
                </div>
                <div class="resumen-line">
                    <span class="resumen-label">PROMEDIO GENERAL DE NOTAS APROBADAS</span>
                    <span class="resumen-dots">...................</span>
                    <span class="resumen-value">' . number_format($nota_simple_promedio, 2) . '</span>
                </div>
                <div class="resumen-line">
                    <span class="resumen-label">PROMEDIO PONDERADO DE NOTAS APROBADAS</span>
                    <span class="resumen-dots">............</span>
                    <span class="resumen-value">' . number_format($nota_ponderada_promedio, 2) . '</span>
                </div>
            </div>

            <div class="firmas avoid-break">
                <table>';

        foreach (array_chunk($firmas, 2) as $fila) {
            $html .= '<tr>';
            foreach ($fila as $firma) {
                $html .= '
                    <td>
                        <p><strong>' . htmlspecialchars($firma['nombre']) . '</strong></p>
                        <p>' . htmlspecialchars($firma['cargo']) . '</p>
                    </td>';
            }
            $html .= '</tr>';
        }

        $html .= '
                </table>
            </div>
        </div>';
    }

    $html .= '
    </body>
    </html>';

    $mpdf->WriteHTML($html);
    $filename = 'Constancia_Notas_' . $detalles_estudiante['cedula'] . '.pdf';
    $mpdf->Output($filename, \Mpdf\Output\Destination::INLINE);

} catch (\Mpdf\MpdfException $e) {
    die("Error al generar el PDF: " . $e->getMessage());
}
?>