<?php
// Centralized authentication and session check
$allowed_roles = ['coordinador2'];
require_once __DIR__ . '/../includes/auth_check.php';

if ($conn->connect_error) {
    die("Error de conexión: " . $conn->connect_error);
}

mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

// CSRF Token
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

$coordinador_id = $_SESSION['user_id'];
$estudiante_id = isset($_GET['estudiante_id']) ? intval($_GET['estudiante_id']) : 0;
$postgrado_seleccionado = isset($_GET['postgrado_id']) ? intval($_GET['postgrado_id']) : 0;

// Verificar acceso al postgrado
$sql_verificar_postgrado = "SELECT COUNT(*) as count FROM coordinador_postgrado 
                           WHERE coordinador_id = ? AND postgrado_id = ?";
$stmt_verificar = $conn->prepare($sql_verificar_postgrado);
$stmt_verificar->bind_param("ii", $coordinador_id, $postgrado_seleccionado);
$stmt_verificar->execute();
$result_verificar = $stmt_verificar->get_result()->fetch_assoc();

if ($result_verificar['count'] == 0) {
    $_SESSION['error'] = "No tienes permisos para acceder a este postgrado.";
    header("Location: cargar_notas.php");
    exit();
}

// Obtener información del estudiante
$sql_estudiante_info = "SELECT e.*, ep.estado as estado_postgrado, p.nombre as postgrado_nombre
                       FROM estudiantes e
                       JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                       JOIN postgrados p ON ep.postgrado_id = p.id
                       WHERE e.id = ? AND ep.postgrado_id = ?";
$stmt_estudiante_info = $conn->prepare($sql_estudiante_info);
$stmt_estudiante_info->bind_param("ii", $estudiante_id, $postgrado_seleccionado);
$stmt_estudiante_info->execute();
$estudiante_info = $stmt_estudiante_info->get_result()->fetch_assoc();

if (!$estudiante_info) {
    $_SESSION['error'] = "No se encontró al estudiante o no pertenece al postgrado seleccionado.";
    header("Location: cargar_notas.php?postgrado_id=".$postgrado_seleccionado);
    exit();
}

// Obtener años lectivos
$sql_anios_lectivos = "SELECT DISTINCT al.id, al.nombre 
                      FROM años_lectivos al
                      JOIN inscripciones i ON al.id = i.año_lectivo_id
                      JOIN estudiante_postgrado ep ON i.estudiante_id = ep.estudiante_id
                      WHERE ep.postgrado_id = ?
                      ORDER BY al.nombre DESC";
$stmt_anios = $conn->prepare($sql_anios_lectivos);
$stmt_anios->bind_param("i", $postgrado_seleccionado);
$stmt_anios->execute();
$anios_lectivos_permitidos = $stmt_anios->get_result()->fetch_all(MYSQLI_ASSOC);

$anios_ids = array_column($anios_lectivos_permitidos, 'id');

// Variables para estadísticas
$denominaciones = [];
$periodos = [];
$totales = [
    'total_materias' => 0, 'total_materias_puntaje' => 0, 'total_materias_cualitativas' => 0,
    'pendientes_puntaje' => 0, 'pendientes_cualitativas' => 0, 'aprobadas_puntaje' => 0,
    'aprobadas_cualitativas' => 0, 'promedio_simple' => 0, 'promedio_ponderado' => 0,
    'suma_notas_simples' => 0, 'suma_notas_ponderadas' => 0, 'suma_creditos_aprobados' => 0
];
$promedios_por_periodo = [];

if (!empty($anios_ids)) {
    $sql_denominaciones = "SELECT d.id, d.nombre, d.codigo, d.tipo, d.periodo, d.credito,
                          i.nota_final, i.nota_definitiva, i.tipo_evaluacion, i.estado,
                          i.año_lectivo_id, al.nombre as anio_lectivo_nombre,
                          i.fecha_modificacion, u.username as modificado_por
                          FROM denominaciones d
                          JOIN inscripciones i ON d.id = i.denominacion_id
                          JOIN años_lectivos al ON i.año_lectivo_id = al.id
                          LEFT JOIN usuarios u ON i.usuario_modificacion = u.id
                          WHERE i.estudiante_id = ? AND d.postgrado_id = ?
                          AND i.año_lectivo_id IN (" . implode(",", $anios_ids) . ")
                          ORDER BY d.periodo, d.codigo";
    $stmt_denominaciones = $conn->prepare($sql_denominaciones);
    $stmt_denominaciones->bind_param("ii", $estudiante_id, $postgrado_seleccionado);
    $stmt_denominaciones->execute();
    $denominaciones = $stmt_denominaciones->get_result()->fetch_all(MYSQLI_ASSOC);
    
    foreach ($denominaciones as $denominacion) {
        $periodo = $denominacion['periodo'];
        $periodos[$periodo][] = $denominacion;
        
        if (!isset($promedios_por_periodo[$periodo])) {
            $promedios_por_periodo[$periodo] = [
                'suma_notas_simples' => 0,
                'suma_notas_ponderadas' => 0,
                'suma_creditos_aprobados' => 0,
                'aprobadas_puntaje' => 0,
                'promedio_simple' => 0,
                'promedio_ponderado' => 0
            ];
        }
        
        $totales['total_materias']++;
        if ($denominacion['tipo'] == 'puntaje') {
            $totales['total_materias_puntaje']++;
            $credito = intval($denominacion['credito']);
            if (empty($denominacion['nota_definitiva'])) {
                $totales['pendientes_puntaje']++;
            } else {
                $nota = floatval($denominacion['nota_definitiva']);
                if ($nota >= 10) {
                    $totales['suma_notas_simples'] += $nota;
                    $totales['suma_notas_ponderadas'] += ($nota * $credito);
                    $totales['suma_creditos_aprobados'] += $credito;
                    $totales['aprobadas_puntaje']++;
                    
                    $promedios_por_periodo[$periodo]['suma_notas_simples'] += $nota;
                    $promedios_por_periodo[$periodo]['suma_notas_ponderadas'] += ($nota * $credito);
                    $promedios_por_periodo[$periodo]['suma_creditos_aprobados'] += $credito;
                    $promedios_por_periodo[$periodo]['aprobadas_puntaje']++;
                }
            }
        } else {
            $totales['total_materias_cualitativas']++;
            if ($denominacion['estado'] != 'finalizada' || empty($denominacion['nota_definitiva'])) {
                $totales['pendientes_cualitativas']++;
            } elseif (strtoupper($denominacion['nota_definitiva']) == 'APROBADO') {
                $totales['aprobadas_cualitativas']++;
            }
        }
    }
    
    if ($totales['aprobadas_puntaje'] > 0) {
        $totales['promedio_simple'] = $totales['suma_notas_simples'] / $totales['aprobadas_puntaje'];
    }
    if ($totales['suma_creditos_aprobados'] > 0) {
        $totales['promedio_ponderado'] = $totales['suma_notas_ponderadas'] / $totales['suma_creditos_aprobados'];
    }
    
    foreach ($promedios_por_periodo as $periodo => $datos) {
        if ($datos['aprobadas_puntaje'] > 0) {
            $promedios_por_periodo[$periodo]['promedio_simple'] = $datos['suma_notas_simples'] / $datos['aprobadas_puntaje'];
            $promedios_por_periodo[$periodo]['promedio_ponderado'] = $datos['suma_notas_ponderadas'] / $datos['suma_creditos_aprobados'];
        }
    }
}

// Procesar actualización de notas
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['guardar_notas'])) {
    if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        $_SESSION['error'] = "Token de seguridad inválido.";
        header("Location: cargar_notas_estudiante.php?estudiante_id=$estudiante_id&postgrado_id=$postgrado_seleccionado");
        exit();
    }

    $conn->begin_transaction();
    try {
        $periodo_activo = $_POST['periodo_activo'] ?? null;
        
        foreach ($_POST['denominaciones'] as $denominacion_id => $datos) {
            $denominacion_id = intval($denominacion_id);
            $tipo_evaluacion = 'Final';
            
            $sql_verificar = "SELECT d.periodo, d.tipo FROM denominaciones d
                             JOIN inscripciones i ON d.id = i.denominacion_id
                             WHERE i.estudiante_id = ? AND i.denominacion_id = ? AND d.postgrado_id = ?";
            $stmt_verificar = $conn->prepare($sql_verificar);
            $stmt_verificar->bind_param("iii", $estudiante_id, $denominacion_id, $postgrado_seleccionado);
            $stmt_verificar->execute();
            $result_verificar = $stmt_verificar->get_result();
            
            if ($result_verificar->num_rows === 0) continue;
            
            $verificacion = $result_verificar->fetch_assoc();
            if ($periodo_activo != $verificacion['periodo']) continue;
            
            $fecha_actual = date('Y-m-d H:i:s');
            
            if ($verificacion['tipo'] == 'puntaje') {
                $nota = !empty($datos['nota']) ? floatval($datos['nota']) : null;
                
                if ($nota !== null && ($nota < 0 || $nota > 20)) {
                    throw new Exception("Las notas deben estar entre 0 y 20.");
                }
                
                $sql_update = "UPDATE inscripciones SET 
                              nota_final = ?, nota_definitiva = ?, tipo_evaluacion = ?,
                              estado = IF(? IS NOT NULL, 'finalizada', estado),
                              fecha_modificacion = ?, usuario_modificacion = ?
                              WHERE estudiante_id = ? AND denominacion_id = ?";
                $stmt = $conn->prepare($sql_update);
                $stmt->bind_param("ddsssiii", $nota, $nota, $tipo_evaluacion, $nota, $fecha_actual, $coordinador_id, $estudiante_id, $denominacion_id);
                $stmt->execute();
            } else {
                $nota_definitiva = $conn->real_escape_string($datos['nota']);
                
                if (in_array($nota_definitiva, ['Aprobado', 'Reprobado'])) {
                    $estado_inscripcion = ($nota_definitiva === 'Aprobado') ? 'finalizada' : 'activa';
                    $sql_update = "UPDATE inscripciones SET 
                                  nota_final = ?, nota_definitiva = ?, tipo_evaluacion = ?, estado = ?,
                                  fecha_modificacion = ?, usuario_modificacion = ?
                                  WHERE estudiante_id = ? AND denominacion_id = ?";
                    $stmt = $conn->prepare($sql_update);
                    $stmt->bind_param("ssssssii", $nota_definitiva, $nota_definitiva, $tipo_evaluacion, $estado_inscripcion, $fecha_actual, $coordinador_id, $estudiante_id, $denominacion_id);
                    $stmt->execute();
                }
            }
        }
        $conn->commit();
        $_SESSION['success'] = "Notas actualizadas correctamente.";
    } catch (Exception $e) {
        $conn->rollback();
        $_SESSION['error'] = "Error al actualizar las notas: " . $e->getMessage();
    }
    header("Location: cargar_notas_estudiante.php?estudiante_id=$estudiante_id&postgrado_id=$postgrado_seleccionado");
    exit();
}

$page_title = "Cargar Notas - " . htmlspecialchars($estudiante_info['nombres'] . ' ' . $estudiante_info['apellidos']);
include '../includes/header.php';
?>

<div class="container">
    <div class="mb-4">
        <a href="cargar_notas.php?postgrado_id=<?= htmlspecialchars($postgrado_seleccionado) ?>" class="btn btn-secondary">
            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-arrow-left" viewBox="0 0 16 16">
                <path fill-rule="evenodd" d="M15 8a.5.5 0 0 0-.5-.5H2.707l3.147-3.146a.5.5 0 1 0-.708-.708l-4 4a.5.5 0 0 0 0 .708l4 4a.5.5 0 0 0 .708-.708L2.707 8.5H14.5A.5.5 0 0 0 15 8z"/>
            </svg>
            Volver al Listado
        </a>
    </div>

    <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success"><?= htmlspecialchars($_SESSION['success']) ?></div>
        <?php unset($_SESSION['success']); ?>
    <?php endif; ?>
    
    <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger"><?= htmlspecialchars($_SESSION['error']) ?></div>
        <?php unset($_SESSION['error']); ?>
    <?php endif; ?>

    <div class="card border-0 shadow-sm">
        <div class="card-body p-0">
            <div class="p-3 bg-light border-bottom">
                <h3 class="mb-0">Notas para: <?= htmlspecialchars($estudiante_info['nombres'] . ' ' . $estudiante_info['apellidos']) ?> (Cédula: <?= htmlspecialchars($estudiante_info['cedula']) ?>)</h3>
                <p class="mb-0">Postgrado: <strong><?= htmlspecialchars($estudiante_info['postgrado_nombre']) ?></strong></p>
                <p class="mb-0">Estado del Postgrado: 
                    <span class="estado-<?= strtolower(htmlspecialchars($estudiante_info['estado_postgrado'])) ?>">
                        <?= htmlspecialchars(ucfirst($estudiante_info['estado_postgrado'])) ?>
                    </span>
                </p>
            </div>

            <?php if (!empty($denominaciones)): ?>
                <div class="p-3 border-bottom">
                    <h4 class="mb-3">Resumen Académico</h4>
                    <div class="row">
                        <div class="col-md-4">
                            <div class="card bg-light mb-3">
                                <div class="card-body">
                                    <h5 class="card-title">Total Asignaturas</h5>
                                    <p class="card-text display-6"><?= $totales['total_materias'] ?></p>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="card bg-light mb-3">
                                <div class="card-body">
                                    <h5 class="card-title">Promedio Simple</h5>
                                    <p class="card-text display-6 <?= ($totales['promedio_simple'] < 15) ? 'promedio-bajo' : 'promedio-alto' ?>">
                                        <?= number_format($totales['promedio_simple'], 2) ?>
                                    </p>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="card bg-light mb-3">
                                <div class="card-body">
                                    <h5 class="card-title">Promedio Ponderado</h5>
                                    <p class="card-text display-6 <?= ($totales['promedio_ponderado'] < 15) ? 'promedio-bajo' : 'promedio-alto' ?>">
                                        <?= number_format($totales['promedio_ponderado'], 2) ?>
                                    </p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="p-3">
                    <form method="post" id="form-notas">
                        <input type="hidden" name="estudiante_id" value="<?= $estudiante_id ?>">
                        <input type="hidden" name="postgrado_id" value="<?= $postgrado_seleccionado ?>">
                        <input type="hidden" name="guardar_notas" value="1">
                        <input type="hidden" name="csrf_token" value="<?= $_SESSION['csrf_token'] ?>">
                        <input type="hidden" name="periodo_activo" id="periodo-activo" value="<?= isset(array_keys($periodos)[0]) ? htmlspecialchars(array_keys($periodos)[0]) : '' ?>">

                        <div class="d-flex flex-wrap tabs-container mb-3" id="periodos-tabs">
                            <?php 
                            $periodos_keys = array_keys($periodos);
                            $tabs_per_row = 8;
                            $rows = ceil(count($periodos_keys) / $tabs_per_row);
                            
                            for ($row = 0; $row < $rows; $row++): ?>
                                <div class="d-flex w-100 mb-2">
                                    <?php 
                                    $start = $row * $tabs_per_row;
                                    $end = min(($row + 1) * $tabs_per_row, count($periodos_keys));
                                    
                                    for ($i = $start; $i < $end; $i++): 
                                        $periodo_nombre = $periodos_keys[$i];
                                    ?>
                                        <button type="button" class="btn btn-outline-primary tab-button <?= ($i === 0) ? 'active' : '' ?>" 
                                                data-target="tab-<?= $i ?>" 
                                                data-periodo="<?= htmlspecialchars($periodo_nombre) ?>">
                                            Periodo <?= htmlspecialchars($periodo_nombre) ?>
                                        </button>
                                    <?php endfor; ?>
                                </div>
                            <?php endfor; ?>
                        </div>
                        
                        <?php foreach ($periodos as $periodo_nombre => $denom_list): ?>
                            <?php 
                            $periodo_id = array_search($periodo_nombre, array_keys($periodos));
                            $promedio_simple_periodo = isset($promedios_por_periodo[$periodo_nombre]['promedio_simple']) ? number_format($promedios_por_periodo[$periodo_nombre]['promedio_simple'], 2) : 'N/A';
                            $promedio_ponderado_periodo = isset($promedios_por_periodo[$periodo_nombre]['promedio_ponderado']) ? number_format($promedios_por_periodo[$periodo_nombre]['promedio_ponderado'], 2) : 'N/A';
                            ?>
                            <div id="tab-<?= $periodo_id ?>" class="tab-content <?= ($periodo_id === 0) ? 'active' : '' ?>">
                                <div class="d-flex justify-content-between align-items-center mb-3">
                                    <h5>Periodo: <?= htmlspecialchars($periodo_nombre) ?></h5>
                                    <div>
                                        <span class="badge bg-primary me-2">Prom. Simple: <?= $promedio_simple_periodo ?></span>
                                        <span class="badge bg-primary">Prom. Ponderado: <?= $promedio_ponderado_periodo ?></span>
                                    </div>
                                </div>
                                
                                <div class="table-responsive">
                                    <table class="table table-striped">
                                        <thead class="table-light">
                                            <tr>
                                                <th>Código</th>
                                                <th>Denominación</th>
                                                <th>Crédito</th>
                                                <th>Nota</th>
                                                <th>Estado</th>
                                                <th>Última Modificación</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($denom_list as $denominacion): ?>
                                                <?php 
                                                $nota_bloqueada = false; // Coordinador2 puede modificar todas las notas
                                                $fecha_modificacion = !empty($denominacion['fecha_modificacion']) ? date('d/m/Y H:i', strtotime($denominacion['fecha_modificacion'])) : 'No modificada';
                                                $modificado_por = !empty($denominacion['modificado_por']) ? $denominacion['modificado_por'] : 'N/A';
                                                ?>
                                                <tr>
                                                    <td><?= htmlspecialchars($denominacion['codigo']) ?></td>
                                                    <td><?= htmlspecialchars($denominacion['nombre']) ?></td>
                                                    <td><?= htmlspecialchars($denominacion['credito']) ?></td>
                                                    <td>
                                                        <?php if ($denominacion['tipo'] == 'puntaje'): ?>
                                                            <input type="number" step="0.01" min="0" max="20" 
                                                                   name="denominaciones[<?= $denominacion['id'] ?>][nota]" 
                                                                   class="form-control nota-input" 
                                                                   value="<?= htmlspecialchars($denominacion['nota_definitiva'] ?? '') ?>">
                                                        <?php else: ?>
                                                            <select name="denominaciones[<?= $denominacion['id'] ?>][nota]" class="form-control">
                                                                <option value="">- Seleccionar -</option>
                                                                <option value="Aprobado" <?= (strtoupper($denominacion['nota_definitiva'] ?? '') == 'APROBADO') ? 'selected' : '' ?>>Aprobado</option>
                                                                <option value="Reprobado" <?= (strtoupper($denominacion['nota_definitiva'] ?? '') == 'REPROBADO') ? 'selected' : '' ?>>Reprobado</option>
                                                            </select>
                                                        <?php endif; ?>
                                                    </td>
                                                    <td>
                                                        <span class="estado-<?= strtolower(htmlspecialchars($denominacion['estado'])) ?>">
                                                            <?= htmlspecialchars(ucfirst($denominacion['estado'])) ?>
                                                        </span>
                                                    </td>
                                                    <td>
                                                        <small><?= $fecha_modificacion ?><br>
                                                        <strong><?= $modificado_por ?></strong></small>
                                                    </td>
                                                </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        <?php endforeach; ?>
                        <button type="submit" class="btn btn-primary mt-3" id="btn-guardar">Guardar Notas</button>
                    </form>
                </div>
            <?php else: ?>
                <div class="p-3">
                    <div class="alert alert-info">El estudiante seleccionado no tiene asignaturas inscritas en este postgrado.</div>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<style>
    .estado-activo { color: #28a745; font-weight: bold; }
    .estado-suspendido, .estado-reprobado, .promedio-bajo { color: #dc3545; font-weight: bold; }
    .estado-graduado, .estado-finalizada { color: #17a2b8; font-weight: bold; }
    .estado-retirado { color: #6c757d; font-weight: bold; }
    .estado-aprobado, .promedio-alto { color: #28a745; font-weight: bold; }
    
    .tab-content { display: none; }
    .tab-content.active { display: block; }
    .tab-button {
        margin-right: 5px;
        margin-bottom: 5px;
    }
    .tab-button.active {
        background-color: #0d6efd;
        color: white;
    }
    .nota-input {
        width: 100px;
        text-align: center;
        font-weight: bold;
    }
    .tabs-container { gap: 5px; }
    .badge {
        font-size: 0.9rem;
        padding: 0.5em 0.75em;
    }

</style>

<script>
    // Manejo de pestañas
    document.querySelectorAll('.tab-button').forEach(button => {
        button.addEventListener('click', function() {
            document.querySelectorAll('.tab-button').forEach(btn => {
                btn.classList.remove('active');
            });
            this.classList.add('active');
            
            document.querySelectorAll('.tab-content').forEach(content => {
                content.classList.remove('active');
            });
            
            const target = this.getAttribute('data-target');
            document.getElementById(target).classList.add('active');
            
            document.getElementById('periodo-activo').value = this.getAttribute('data-periodo');
        });
    });

    // Validación del formulario
    const form = document.getElementById('form-notas');
    if (form) {
        form.addEventListener('submit', function(e) {
            let isValid = true;
            const numberInputs = form.querySelectorAll('input[type="number"]');
            
            numberInputs.forEach(function(input) {
                const value = parseFloat(input.value);
                if (input.value !== '' && (value < 0 || value > 20)) {
                    isValid = false;
                    input.style.border = '2px solid red';
                    input.style.backgroundColor = '#f8d7da';
                } else {
                    input.style.border = '1px solid #ced4da';
                    input.style.backgroundColor = 'white';
                }
            });
            
            if (!isValid) {
                e.preventDefault();
                alert('Error: Las notas numéricas deben estar entre 0 y 20. Por favor, corrija los campos marcados en rojo.');
            }
        });
    }
</script>

<?php
include '../includes/footer.php';
$conn->close();
?>