<?php
session_start();

// Headers de seguridad adicionales
header("X-Frame-Options: DENY");
header("X-Content-Type-Options: nosniff");
header("X-XSS-Protection: 1; mode=block");
header("Referrer-Policy: strict-origin-when-cross-origin");

// Generar token CSRF
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Autenticación y rol
if (!isset($_SESSION['username']) || $_SESSION['rol'] != 'coordinador2') {
    header("Location: ../login.php");
    exit();
}

include '../includes/db.php';

if ($conn->connect_error) {
    die("Error de conexión: " . $conn->connect_error);
}

// Habilitar reportes de error para depuración
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

// Obtener parámetros de búsqueda con sanitización
$search = isset($_GET['search']) ? $conn->real_escape_string(trim($_GET['search'])) : '';

// Consulta para obtener estudiantes con sus postgrados asignados al coordinador2
$sql_estudiantes = "SELECT 
                    e.id, e.nombres, e.apellidos, e.cedula, e.correo, e.telefono 
                   FROM estudiantes e
                   JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                   JOIN coordinador_postgrado cp ON ep.postgrado_id = cp.postgrado_id
                   WHERE cp.coordinador_id = ?
                   AND (e.cedula LIKE ? OR e.nombres LIKE ? OR e.apellidos LIKE ? OR e.correo LIKE ?)
                   GROUP BY e.id
                   ORDER BY e.apellidos";

$stmt_estudiantes = $conn->prepare($sql_estudiantes);

if (!$stmt_estudiantes) {
    die("Error en la preparación de la consulta: " . $conn->error);
}

$search_term = "%$search%";
$stmt_estudiantes->bind_param("issss", $_SESSION['user_id'], $search_term, $search_term, $search_term, $search_term);
$stmt_estudiantes->execute();
$estudiantes = $stmt_estudiantes->get_result()->fetch_all(MYSQLI_ASSOC);

// Para cada estudiante, obtener sus postgrados asignados al coordinador2
foreach ($estudiantes as &$estudiante) {
    $sql_postgrados = "SELECT 
                        p.id AS postgrado_id, 
                        p.nombre AS postgrado_nombre,
                        pr.nombre AS promocion,
                        ep.estado AS estado_postgrado,
                        ep.fecha_inscripcion,
                        ep.fecha_graduacion
                       FROM estudiante_postgrado ep
                       JOIN postgrados p ON ep.postgrado_id = p.id
                       JOIN coordinador_postgrado cp ON ep.postgrado_id = cp.postgrado_id
                       LEFT JOIN promociones pr ON ep.promocion_id = pr.id
                       WHERE ep.estudiante_id = ?
                       AND cp.coordinador_id = ?
                       AND (ep.estado = 'activo' OR ep.estado = 'graduado')
                       ORDER BY ep.estado DESC, ep.fecha_inscripcion DESC";
    
    $stmt_postgrados = $conn->prepare($sql_postgrados);
    $stmt_postgrados->bind_param("ii", $estudiante['id'], $_SESSION['user_id']);
    $stmt_postgrados->execute();
    $estudiante['postgrados'] = $stmt_postgrados->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt_postgrados->close();
}
unset($estudiante); // Romper la referencia

// Generar PDF si se solicita
if (isset($_GET['generar_pdf']) && $_GET['generar_pdf'] == '1') {
    require_once '../lib/tcpdf/tcpdf.php';
    
    // Crear nuevo documento PDF
    $pdf = new TCPDF(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);
    
    // Configurar documento
    $pdf->SetCreator(PDF_CREATOR);
    $pdf->SetAuthor('Sistema de Postgrados');
    $pdf->SetTitle('Reporte de Estudiantes - Coordinador2');
    $pdf->SetSubject('Reporte de Estudiantes');
    $pdf->SetKeywords('Estudiantes, Postgrados, Reporte');
    
    // Eliminar márgenes
    $pdf->SetMargins(10, 10, 10);
    $pdf->SetHeaderMargin(0);
    $pdf->SetFooterMargin(0);
    
    // Añadir página
    $pdf->AddPage();
    
    // Contenido del PDF
    $html = '
    <style>
        h1 { color: #2c3e50; font-size: 18px; text-align: center; }
        table { width: 100%; border-collapse: collapse; margin-top: 10px; }
        th { background-color: #f2f2f2; font-weight: bold; text-align: left; padding: 5px; border: 1px solid #ddd; }
        td { padding: 5px; border: 1px solid #ddd; }
        .postgrado-item { margin-bottom: 5px; }
    </style>
    <h1>Reporte de Estudiantes - Coordinador2</h1>
    <p style="text-align: right;">Fecha: '.date('d/m/Y').'</p>';
    
    if (!empty($estudiantes)) {
        $html .= '<table>
            <tr>
                <th width="8%">ID</th>
                <th width="15%">Nombre</th>
                <th width="15%">Apellidos</th>
                <th width="12%">Cédula</th>
                <th width="25%">Postgrados</th>
                <th width="15%">Contacto</th>
            </tr>';
        
        foreach ($estudiantes as $estudiante) {
            $postgrados_html = '';
            if (!empty($estudiante['postgrados'])) {
                foreach ($estudiante['postgrados'] as $postgrado) {
                    $postgrados_html .= '<div class="postgrado-item">
                        <strong>'.$postgrado['postgrado_nombre'].'</strong><br>
                        Promoción: '.($postgrado['promocion'] ?? 'N/A').'<br>
                        Estado: '.ucfirst($postgrado['estado_postgrado']).'<br>
                        Inscripción: '.date('d/m/Y', strtotime($postgrado['fecha_inscripcion']));
                    
                    if ($postgrado['estado_postgrado'] == 'graduado' && !empty($postgrado['fecha_graduacion'])) {
                        $postgrados_html .= '<br>Graduación: '.date('d/m/Y', strtotime($postgrado['fecha_graduacion']));
                    }
                    
                    $postgrados_html .= '</div>';
                }
            } else {
                $postgrados_html = 'No asignado';
            }
            
            $html .= '<tr>
                <td>'.$estudiante['id'].'</td>
                <td>'.$estudiante['nombres'].'</td>
                <td>'.$estudiante['apellidos'].'</td>
                <td>'.$estudiante['cedula'].'</td>
                <td>'.$postgrados_html.'</td>
                <td>
                    Correo: '.$estudiante['correo'].'<br>
                    Teléfono: '.($estudiante['telefono'] ?? 'N/A').'
                </td>
            </tr>';
        }
        
        $html .= '</table>';
    } else {
        $html .= '<p>No se encontraron estudiantes.</p>';
    }
    
    // Escribir contenido HTML
    $pdf->writeHTML($html, true, false, true, false, '');
    
    // Generar y descargar PDF
    $pdf->Output('reporte_estudiantes_coordinador2_'.date('Ymd').'.pdf', 'D');
    exit();
}

$page_title = "Buscar Estudiante - Coordinador2";
include '../includes/header.php';
?>

<div class="container">
    <div class="mb-4">
        <a href="index.php" class="btn btn-secondary">
            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-arrow-left" viewBox="0 0 16 16">
                <path fill-rule="evenodd" d="M15 8a.5.5 0 0 0-.5-.5H2.707l3.147-3.146a.5.5 0 1 0-.708-.708l-4 4a.5.5 0 0 0 0 .708l4 4a.5.5 0 0 0 .708-.708L2.707 8.5H14.5A.5.5 0 0 0 15 8z"/>
            </svg>
            Volver al Panel
        </a>
    </div>

    <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success"><?= htmlspecialchars($_SESSION['success']) ?></div>
        <?php unset($_SESSION['success']); ?>
    <?php endif; ?>
    
    <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger"><?= htmlspecialchars($_SESSION['error']) ?></div>
        <?php unset($_SESSION['error']); ?>
    <?php endif; ?>

    <div class="card border-0 shadow-sm">
        <div class="card-body p-0">
            <h2 class="card-title p-3 mb-0 bg-light border-bottom">Buscar Estudiante - Coordinador2</h2>

            <div class="p-3">
                <!-- Buscador de estudiantes -->
                <form method="get" class="mb-4">
                    <div class="row g-3">
                        <div class="col-md-8">
                            <label for="search" class="form-label">Buscar Estudiante:</label>
                            <input type="text" id="search" name="search" class="form-control" 
                                   value="<?= htmlspecialchars($search) ?>" 
                                   placeholder="Cédula, Nombre, Apellido o Correo">
                        </div>
                        <div class="col-md-4 d-flex align-items-end">
                            <button type="submit" class="btn btn-primary me-2">Buscar</button>
                            <a href="buscar_estudiante.php" class="btn btn-secondary">Limpiar</a>
                        </div>
                    </div>
                </form>
                
                <?php if (!empty($estudiantes)): ?>
                <div class="mb-3">
                    <a href="buscar_estudiante.php?search=<?= urlencode($search) ?>&generar_pdf=1" class="btn btn-success">
                        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-file-earmark-pdf" viewBox="0 0 16 16">
                            <path d="M14 14V4.5L9.5 0H4a2 2 0 0 0-2 2v12a2 2 0 0 0 2 2h8a2 2 0 0 0 2-2zM9.5 3A1.5 1.5 0 0 0 11 4.5h2V14a1 1 0 0 1-1 1H4a1 1 0 0 1-1-1V2a1 1 0 0 1 1-1h5.5v2z"/>
                            <path d="M4.603 14.087a.81.81 0 0 1-.438-.42c-.195-.388-.13-.776.08-1.102.198-.307.526-.568.897-.787a7.68 7.68 0 0 1 1.482-.645 19.697 19.697 0 0 0 1.062-2.227 7.269 7.269 0 0 1-.43-1.295c-.086-.4-.119-.796-.046-1.136.075-.354.274-.672.65-.823.192-.077.4-.12.602-.077a.7.7 0 0 1 .477.365c.088.164.12.356.127.538.007.188-.012.396-.047.614-.084.51-.27 1.134-.52 1.794a10.954 10.954 0 0 0 .98 1.686 5.753 5.753 0 0 1 1.334.05c.364.066.734.195.96.465.12.144.193.32.2.518.007.192-.047.382-.138.563a1.04 1.04 0 0 1-.354.416.856.856 0 0 1-.51.138c-.331-.014-.654-.196-.933-.417a5.712 5.712 0 0 1-.911-.95 11.651 11.651 0 0 0-1.997.406 11.307 11.307 0 0 1-1.02 1.51c-.292.35-.609.656-.927.787a.793.793 0 0 1-.58.029z"/>
                        </svg>
                        Generar PDF
                    </a>
                </div>
                <?php endif; ?>
            </div>

            <div class="p-3 border-top">
                <?php if (!empty($estudiantes)): ?>
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead class="table-light">
                                <tr>
                                    <th>ID</th>
                                    <th>Nombre</th>
                                    <th>Apellidos</th>
                                    <th>Cédula</th>
                                    <th>Contacto</th>
                                    <th>Postgrados</th>
                                    <th>Acciones</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($estudiantes as $estudiante): ?>
                                    <tr>
                                        <td><?= $estudiante['id'] ?></td>
                                        <td><?= htmlspecialchars($estudiante['nombres']) ?></td>
                                        <td><?= htmlspecialchars($estudiante['apellidos']) ?></td>
                                        <td><?= htmlspecialchars($estudiante['cedula']) ?></td>
                                        <td>
                                            <small>
                                                <?= htmlspecialchars($estudiante['correo']) ?><br>
                                                Tel: <?= htmlspecialchars($estudiante['telefono'] ?? 'N/A') ?>
                                            </small>
                                        </td>
                                        <td>
                                            <?php if (!empty($estudiante['postgrados'])): ?>
                                                <?php foreach ($estudiante['postgrados'] as $postgrado): ?>
                                                    <div class="mb-2 p-2 border rounded">
                                                        <strong><?= htmlspecialchars($postgrado['postgrado_nombre']) ?></strong><br>
                                                        <small class="text-muted">
                                                            Promoción: <?= htmlspecialchars($postgrado['promocion'] ?? 'N/A') ?><br>
                                                            Estado: <span class="estado-<?= strtolower(htmlspecialchars($postgrado['estado_postgrado'])) ?>">
                                                                <?= htmlspecialchars(ucfirst($postgrado['estado_postgrado'])) ?>
                                                            </span><br>
                                                            Inscripción: <?= date('d/m/Y', strtotime($postgrado['fecha_inscripcion'])) ?>
                                                            <?php if ($postgrado['estado_postgrado'] == 'graduado' && !empty($postgrado['fecha_graduacion'])): ?>
                                                                <br>Graduación: <?= date('d/m/Y', strtotime($postgrado['fecha_graduacion'])) ?>
                                                            <?php endif; ?>
                                                        </small>
                                                    </div>
                                                <?php endforeach; ?>
                                            <?php else: ?>
                                                <span class="text-muted">No asignado</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if (!empty($estudiante['postgrados'])): ?>
                                                <?php foreach ($estudiante['postgrados'] as $postgrado): ?>
                                                    <div class="mb-2">
                                                        <a href="constancia_notas.php?estudiante_id=<?= $estudiante['id'] ?>&postgrado_id=<?= $postgrado['postgrado_id'] ?>" 
                                                           class="btn btn-sm btn-primary" 
                                                           title="Generar constancia para <?= htmlspecialchars($postgrado['postgrado_nombre']) ?>">
                                                            Constancia <?= htmlspecialchars($postgrado['postgrado_nombre']) ?>
                                                        </a>
                                                    </div>
                                                <?php endforeach; ?>
                                            <?php else: ?>
                                                <span class="text-muted">No disponible</span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <div class="alert alert-info">No se encontraron estudiantes para los criterios de búsqueda.</div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<style>
    .estado-activo { color: #28a745; font-weight: bold; }
    .estado-graduado { color: #17a2b8; font-weight: bold; }
</style>

<?php
include '../includes/footer.php';
$conn->close();
?>