<?php
// Centralized authentication and session check
$allowed_roles = ['coordinador1'];
require_once __DIR__ . '/../includes/auth_check.php';

// Headers de seguridad adicionales (pueden permanecer)
header("X-Content-Type-Options: nosniff");

require_once '../includes/db.php'; // Conexión a la base de datos

// Verificar si hay conexión a la base de datos
if (!$conn) {
    die("Error de conexión a la base de datos");
}

// Obtener ID del coordinador actual
$coordinador_id = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : 0;

// Obtener postgrados asignados a este coordinador
$postgrados_query = "SELECT p.id, p.nombre 
                    FROM coordinador_postgrado cp 
                    JOIN postgrados p ON cp.postgrado_id = p.id 
                    WHERE cp.coordinador_id = ?";
$stmt_postgrados = $conn->prepare($postgrados_query);
$stmt_postgrados->bind_param("i", $coordinador_id);
$stmt_postgrados->execute();
$postgrados_result = $stmt_postgrados->get_result();

// Verificar si la consulta de postgrados fue exitosa
if (!$postgrados_result) {
    die("Error en la consulta de postgrados: " . $conn->error);
}

// Obtener escolaridades permitidas para este coordinador
$escolaridades_query = "SELECT DISTINCT e.id, e.nombre 
                       FROM coordinador_postgrado_escolaridad cpe
                       JOIN escolaridades e ON cpe.escolaridad_id = e.id
                       WHERE cpe.coordinador_id = ?";
$stmt_escolaridades = $conn->prepare($escolaridades_query);
$stmt_escolaridades->bind_param("i", $coordinador_id);
$stmt_escolaridades->execute();
$escolaridades_result = $stmt_escolaridades->get_result();

// Verificar si la consulta de escolaridades fue exitosa
if (!$escolaridades_result) {
    die("Error en la consulta de escolaridades: " . $conn->error);
}

$page_title = "Panel del Coordinador 1";
include '../includes/header.php';
?>

<div class="container">
    <div class="text-center mb-4">
        <h1 class="display-4">Bienvenido, Coordinador 1</h1>
        <p class="lead">Usuario: <?php echo htmlspecialchars($_SESSION['username']); ?></p>
        <p class="lead">Rol: <?php echo htmlspecialchars($_SESSION['rol']); ?></p>
    </div>

    <div class="card mb-4">
        <div class="card-header bg-primary text-white">
            <h5>Estadísticas por Postgrado</h5>
        </div>
        <div class="card-body">
            <div class="row">
                <?php 
                if ($postgrados_result->num_rows > 0) {
                    while($postgrado = $postgrados_result->fetch_assoc()): 
                        // Contar estudiantes activos en este postgrado
                        $estudiantes_query = "SELECT COUNT(*) as total FROM estudiante_postgrado 
                                            WHERE postgrado_id = ? AND estado = 'activo'";
                        $stmt_estudiantes = $conn->prepare($estudiantes_query);
                        $stmt_estudiantes->bind_param("i", $postgrado['id']);
                        $stmt_estudiantes->execute();
                        $estudiantes_result = $stmt_estudiantes->get_result();
                        
                        $total_estudiantes = 0;
                        if ($estudiantes_result) {
                            $row = $estudiantes_result->fetch_assoc();
                            $total_estudiantes = $row['total'];
                        }
                        
                        // Contar estudiantes con promedio bajo (simple O ponderado < 15)
                        $bajos_promedios_query = "SELECT COUNT(*) as total_bajos FROM (
                                                    SELECT e.id,
                                                        (SELECT AVG(CAST(i.nota_definitiva AS DECIMAL(10,2))) 
                                                        FROM inscripciones i JOIN denominaciones d ON i.denominacion_id = d.id
                                                        WHERE i.estudiante_id = e.id AND d.postgrado_id = ?
                                                        AND i.nota_definitiva IS NOT NULL AND i.nota_definitiva != 'Aprobado'
                                                        AND i.estado = 'finalizada') as promedio_simple,
                                                        (SELECT SUM(CAST(i.nota_definitiva AS DECIMAL(10,2)) * d.credito) / NULLIF(SUM(d.credito), 0)
                                                        FROM inscripciones i JOIN denominaciones d ON i.denominacion_id = d.id
                                                        WHERE i.estudiante_id = e.id AND d.postgrado_id = ?
                                                        AND i.nota_definitiva IS NOT NULL AND i.nota_definitiva != 'Aprobado'
                                                        AND i.estado = 'finalizada') as promedio_ponderado
                                                    FROM estudiantes e
                                                    JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                                                    WHERE ep.postgrado_id = ? AND ep.estado = 'activo'
                                                    HAVING promedio_simple < 15 OR promedio_ponderado < 15
                                                ) AS subquery";
                        $stmt_bajos = $conn->prepare($bajos_promedios_query);
                        $stmt_bajos->bind_param("iii", $postgrado['id'], $postgrado['id'], $postgrado['id']);
                        $stmt_bajos->execute();
                        $bajos_promedios_result = $stmt_bajos->get_result();
                        
                        $total_bajos_promedios = 0;
                        if ($bajos_promedios_result) {
                            $row = $bajos_promedios_result->fetch_assoc();
                            $total_bajos_promedios = $row['total_bajos'];
                        }
                ?>
                <div class="col-md-4 mb-4">
                    <div class="card h-100">
                        <div class="card-header bg-info text-white">
                            <h6 class="mb-0"><?php echo htmlspecialchars($postgrado['nombre']); ?></h6>
                        </div>
                        <div class="card-body">
                            <p class="card-text">
                                <strong>Estudiantes activos:</strong> <?php echo $total_estudiantes; ?><br>
                                <strong class="text-danger">Promedio < 15:</strong> <?php echo $total_bajos_promedios; ?>
                            </p>
                        </div>
                    </div>
                </div>
                <?php 
                    endwhile;
                } else {
                    echo '<div class="col-12"><div class="alert alert-info">No tienes postgrados asignados</div></div>';
                }
                ?>
            </div>
        </div>
    </div>

    <div class="card mb-4">
        <div class="card-header bg-warning text-dark">
            <h5>Estudiantes con Promedio Bajo (<15 puntos)</h5>
        </div>
        <div class="card-body">
            <?php 
            // Re-ejecutar la consulta de postgrados para el listado
            $stmt_postgrados->execute();
            $postgrados_result = $stmt_postgrados->get_result();
            
            if ($postgrados_result->num_rows > 0) {
                while($postgrado = $postgrados_result->fetch_assoc()): 
                    // Obtener estudiantes con promedio simple bajo con más detalles
                    $bajos_promedios_query = "SELECT e.id, e.cedula, e.nombres, e.apellidos,
                                            (SELECT AVG(CAST(i.nota_definitiva AS DECIMAL(10,2))) 
                                            FROM inscripciones i 
                                            JOIN denominaciones d ON i.denominacion_id = d.id
                                            WHERE i.estudiante_id = e.id AND d.postgrado_id = ?
                                            AND i.nota_definitiva IS NOT NULL AND i.nota_definitiva != 'Aprobado'
                                            AND i.estado = 'finalizada') as promedio_simple,
                                            (SELECT SUM(CAST(i.nota_definitiva AS DECIMAL(10,2)) * d.credito) / NULLIF(SUM(d.credito), 0)
                                            FROM inscripciones i 
                                            JOIN denominaciones d ON i.denominacion_id = d.id
                                            WHERE i.estudiante_id = e.id AND d.postgrado_id = ?
                                            AND i.nota_definitiva IS NOT NULL AND i.nota_definitiva != 'Aprobado'
                                            AND i.estado = 'finalizada') as promedio_ponderado
                                            FROM estudiantes e
                                            JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                                            WHERE ep.postgrado_id = ? AND ep.estado = 'activo'
                                            HAVING promedio_simple < 15 OR promedio_ponderado < 15
                                            ORDER BY promedio_simple ASC";
                    $stmt_detalle = $conn->prepare($bajos_promedios_query);
                    $stmt_detalle->bind_param("iii", $postgrado['id'], $postgrado['id'], $postgrado['id']);
                    $stmt_detalle->execute();
                    $bajos_promedios_result = $stmt_detalle->get_result();
                    
                    if ($bajos_promedios_result && $bajos_promedios_result->num_rows > 0):
            ?>
            <h6 class="mb-3"><?php echo htmlspecialchars($postgrado['nombre']); ?></h6>
            <div class="table-responsive">
                <table class="table table-sm table-bordered table-hover">
                    <thead class="thead-light">
                        <tr>
                            <th>Cédula</th>
                            <th>Estudiante</th>
                            <th class="text-danger">Prom. Simple</th>
                            <th class="text-danger">Prom. Ponderado</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php while($estudiante = $bajos_promedios_result->fetch_assoc()): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($estudiante['cedula']); ?></td>
                            <td><?php echo htmlspecialchars($estudiante['nombres'].' '.$estudiante['apellidos']); ?></td>
                            <td class="text-danger font-weight-bold"><?php echo isset($estudiante['promedio_simple']) ? number_format($estudiante['promedio_simple'], 2) : 'N/A'; ?></td>
                            <td class="text-danger font-weight-bold"><?php echo isset($estudiante['promedio_ponderado']) ? number_format($estudiante['promedio_ponderado'], 2) : 'N/A'; ?></td>
                        </tr>
                        <?php endwhile; ?>
                    </tbody>
                </table>
            </div>
            <?php else: ?>
            <div class="alert alert-info">
                No hay estudiantes con promedio bajo en <?php echo htmlspecialchars($postgrado['nombre']); ?>
            </div>
            <?php endif; ?>
            <?php 
                endwhile;
            } else {
                echo '<div class="alert alert-info">No tienes postgrados asignados</div>';
            }
            ?>
        </div>
    </div>

    <div class="card mb-4">
        <div class="card-header bg-success text-white">
            <h5>Escolaridades Permitidas para Cargar Notas</h5>
        </div>
        <div class="card-body">
            <?php 
            $stmt_escolaridades->execute();
            $escolaridades_result = $stmt_escolaridades->get_result();

            if ($escolaridades_result->num_rows > 0): 
                echo '<div class="d-flex flex-wrap">';
                while($escolaridad = $escolaridades_result->fetch_assoc()): 
            ?>
                <span class="badge bg-success me-2 mb-2 fs-6">
                    <?php echo htmlspecialchars($escolaridad['nombre']); ?>
                </span>
            <?php 
                endwhile; 
                echo '</div>';
            else: 
            ?>
                <div class="alert alert-warning">No tienes escolaridades asignadas</div>
            <?php endif; ?>
        </div>
    </div>

</div>

<style>
    .text-danger {
        color: #dc3545 !important;
    }
    .font-weight-bold {
        font-weight: bold !important;
    }
</style>

<?php
include '../includes/footer.php'; // Asegurarse de que el footer se incluye
?>