<?php
// 1. Configuración de sesión
require_once __DIR__ . '/../includes/session_config.php';

// 2. Verificación de auth
$allowed_roles = ['coordinador1'];
require_once __DIR__ . '/../includes/auth_check.php';

// 3. Configurar título de página
$page_title = "Cargar Notas - Estudiante";

// 4. Configurar encoding UTF-8
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');

// 5. Lógica específica de la página
include '../includes/db.php';

$conn->set_charset("utf8mb4");
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

$estudiante_id = isset($_GET['estudiante_id']) ? intval($_GET['estudiante_id']) : 0;
$postgrado_seleccionado = isset($_GET['postgrado_id']) ? intval($_GET['postgrado_id']) : 0;
$coordinador_id = $_SESSION['user_id'];

// VERIFICACIÓN COMPLETA DE PERMISOS
$tiene_permiso = false;
$error_permisos = "";

// Primero, verificar si el coordinador tiene permiso para alguna escolaridad del postgrado
$sql_permiso_escolaridad = "SELECT COUNT(*) as count 
                           FROM coordinador_postgrado_escolaridad 
                           WHERE coordinador_id = ? AND postgrado_id = ?";
$stmt_permiso_escolaridad = $conn->prepare($sql_permiso_escolaridad);
$stmt_permiso_escolaridad->bind_param("ii", $coordinador_id, $postgrado_seleccionado);
$stmt_permiso_escolaridad->execute();
$result_permiso_escolaridad = $stmt_permiso_escolaridad->get_result()->fetch_assoc();

if ($result_permiso_escolaridad['count'] > 0) {
    // Tiene permisos específicos por escolaridad
    $tiene_permiso = true;
} else {
    // Verificar permisos generales en coordinador_postgrado
    $sql_permiso_general = "SELECT COUNT(*) as count 
                           FROM coordinador_postgrado 
                           WHERE coordinador_id = ? AND postgrado_id = ?";
    $stmt_permiso_general = $conn->prepare($sql_permiso_general);
    $stmt_permiso_general->bind_param("ii", $coordinador_id, $postgrado_seleccionado);
    $stmt_permiso_general->execute();
    $result_permiso_general = $stmt_permiso_general->get_result()->fetch_assoc();
    
    if ($result_permiso_general['count'] > 0) {
        // Tiene permisos generales al postgrado
        $tiene_permiso = true;
    } else {
        $tiene_permiso = false;
        $error_permisos = "No tiene permisos para gestionar este postgrado.";
    }
}

if (!$tiene_permiso) {
    $_SESSION['error'] = $error_permisos;
    header("Location: cargar_notas.php");
    exit();
}

// Obtener información del estudiante
$sql_estudiante_info = "SELECT e.*, ep.estado as estado_postgrado, p.nombre as postgrado_nombre
                       FROM estudiantes e
                       JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                       JOIN postgrados p ON ep.postgrado_id = p.id
                       WHERE e.id = ? AND ep.postgrado_id = ?";
$stmt_estudiante_info = $conn->prepare($sql_estudiante_info);
$stmt_estudiante_info->bind_param("ii", $estudiante_id, $postgrado_seleccionado);
$stmt_estudiante_info->execute();
$estudiante_info = $stmt_estudiante_info->get_result()->fetch_assoc();

if (!$estudiante_info) {
    $_SESSION['error'] = "No se encontró al estudiante o no pertenece al postgrado seleccionado.";
    header("Location: cargar_notas.php?postgrado_id=".$postgrado_seleccionado);
    exit();
}

// FUNCIÓN PARA OBTENER EL NÚMERO DE DOCUMENTO SEGÚN EL TIPO
function obtener_numero_documento($tipo_documento, $cedula_venezolana, $cedula_extranjera, $pasaporte) {
    switch ($tipo_documento) {
        case 'V': return $cedula_venezolana;
        case 'E': return $cedula_extranjera;
        case 'P': return $pasaporte;
        default: return $cedula_venezolana;
    }
}

// FUNCIÓN PARA OBTENER EL TIPO DE DOCUMENTO EN TEXTO
function obtener_tipo_documento_texto($tipo_documento) {
    switch ($tipo_documento) {
        case 'V': return 'Cédula Venezolana';
        case 'E': return 'Cédula Extranjera';
        case 'P': return 'Pasaporte';
        default: return 'Cédula Venezolana';
    }
}

// Obtener el número de documento del estudiante
$numero_documento = obtener_numero_documento(
    $estudiante_info['tipo_documento'], 
    $estudiante_info['cedula_venezolana'], 
    $estudiante_info['cedula_extranjera'], 
    $estudiante_info['pasaporte']
);
$tipo_documento_texto = obtener_tipo_documento_texto($estudiante_info['tipo_documento']);

// Verificar qué columnas existen realmente para manejo de modificaciones
$sql_check_columns = "SHOW COLUMNS FROM inscripciones";
$result = $conn->query($sql_check_columns);
$has_veces_modificada = false;
$has_veces_modificado = false;
$has_modificado = false;

while ($row = $result->fetch_assoc()) {
    if ($row['Field'] == 'veces_modificada') {
        $has_veces_modificada = true;
    }
    if ($row['Field'] == 'veces_modificado') {
        $has_veces_modificado = true;
    }
    if ($row['Field'] == 'modificado') {
        $has_modificado = true;
    }
}

// Usar 'veces_modificado' como columna principal
$main_modifications_column = 'veces_modificado';
$use_secondary_column = $has_veces_modificada;

// Obtener denominaciones filtrando por escolaridades permitidas al coordinador
$sql_denominaciones = "SELECT d.id, d.nombre, d.codigo, d.tipo, d.periodo, d.credito,
                      i.id as id_inscripcion, i.nota_definitiva, i.estado,
                      i.fecha_creacion, i.fecha_modificacion, i.modificado,
                      i.escolaridad_id,
                      al.nombre as anio_lectivo_nombre,
                      esc.nombre as escolaridad_nombre,
                      u_mod.nombres as modificado_por_nombres,
                      u_mod.apellidos as modificado_por_apellidos,
                      u_mod.username as modificado_por_username";
                      
// Obtener las columnas de modificaciones
if ($has_veces_modificado) {
    $sql_denominaciones .= ", i.veces_modificado";
} else {
    $sql_denominaciones .= ", 0 as veces_modificado";
}

if ($has_veces_modificada) {
    $sql_denominaciones .= ", i.veces_modificada";
} else {
    $sql_denominaciones .= ", 0 as veces_modificada";
}

$sql_denominaciones .= " FROM denominaciones d
                      JOIN inscripciones i ON d.id = i.denominacion_id
                      JOIN anios_lectivos al ON i.año_lectivo_id = al.id
                      JOIN escolaridades esc ON i.escolaridad_id = esc.id
                      LEFT JOIN usuarios u_mod ON i.usuario_modificacion = u_mod.id
                      WHERE i.estudiante_id = ? AND d.postgrado_id = ?";

// Filtrar por escolaridades permitidas al coordinador
$sql_permisos_escolaridad_lista = "SELECT escolaridad_id 
                                  FROM coordinador_postgrado_escolaridad 
                                  WHERE coordinador_id = ? AND postgrado_id = ?";
$stmt_permisos = $conn->prepare($sql_permisos_escolaridad_lista);
$stmt_permisos->bind_param("ii", $coordinador_id, $postgrado_seleccionado);
$stmt_permisos->execute();
$result_permisos = $stmt_permisos->get_result();

$escolaridades_permitidas = [];
while ($row = $result_permisos->fetch_assoc()) {
    $escolaridades_permitidas[] = $row['escolaridad_id'];
}

// Si el coordinador tiene permisos específicos por escolaridad
if (!empty($escolaridades_permitidas)) {
    $placeholders = implode(',', array_fill(0, count($escolaridades_permitidas), '?'));
    $sql_denominaciones .= " AND i.escolaridad_id IN ($placeholders)";
    $tipos = str_repeat('i', count($escolaridades_permitidas));
    $sql_denominaciones .= " ORDER BY d.periodo, d.codigo";
    
    $stmt_denominaciones = $conn->prepare($sql_denominaciones);
    
    // Construir array de parámetros: estudiante_id, postgrado_id, ...escolaridades
    $params = array_merge([$estudiante_id, $postgrado_seleccionado], $escolaridades_permitidas);
    
    // Vincular parámetros dinámicamente
    $stmt_denominaciones->bind_param("ii" . $tipos, ...$params);
} else {
    // Si no tiene permisos específicos por escolaridad, usar permisos generales
    $sql_denominaciones .= " ORDER BY d.periodo, d.codigo";
    $stmt_denominaciones = $conn->prepare($sql_denominaciones);
    $stmt_denominaciones->bind_param("ii", $estudiante_id, $postgrado_seleccionado);
}

$stmt_denominaciones->execute();
$denominaciones = $stmt_denominaciones->get_result()->fetch_all(MYSQLI_ASSOC);

// Verificar si hay escolaridades no permitidas para este coordinador
$sql_escolaridades_estudiante = "SELECT DISTINCT esc.id, esc.nombre
                                FROM inscripciones i
                                JOIN denominaciones d ON i.denominacion_id = d.id
                                JOIN escolaridades esc ON i.escolaridad_id = esc.id
                                WHERE i.estudiante_id = ? AND d.postgrado_id = ?";
$stmt_escolaridades = $conn->prepare($sql_escolaridades_estudiante);
$stmt_escolaridades->bind_param("ii", $estudiante_id, $postgrado_seleccionado);
$stmt_escolaridades->execute();
$escolaridades_estudiante = $stmt_escolaridades->get_result()->fetch_all(MYSQLI_ASSOC);

$escolaridades_sin_acceso = [];
foreach ($escolaridades_estudiante as $esc) {
    if (!empty($escolaridades_permitidas) && !in_array($esc['id'], $escolaridades_permitidas)) {
        $escolaridades_sin_acceso[] = $esc['nombre'];
    }
}

$periodos = [];
$totales = [
    'total_materias' => 0, 
    'total_materias_puntaje' => 0, 
    'total_materias_cualitativas' => 0,
    'aprobadas_puntaje' => 0,
    'aprobadas_cualitativas' => 0, 
    'sin_nota_puntaje' => 0, 
    'sin_nota_cualitativas' => 0,
    'reprobadas_puntaje' => 0, 
    'reprobadas_cualitativas' => 0,
    'promedio_simple' => 0,
    'promedio_ponderado' => 0,
    'suma_notas_simples' => 0,
    'suma_notas_ponderadas' => 0,
    'suma_creditos_aprobados' => 0
];

// Array para almacenar promedios por periodo
$promedios_por_periodo = array();

foreach ($denominaciones as $denominacion) {
    $periodo = $denominacion['periodo'];
    if (!isset($periodos[$periodo])) {
        $periodos[$periodo] = array();
    }
    $periodos[$periodo][] = $denominacion;
    
    // Inicializar promedios por periodo si no existen
    if (!isset($promedios_por_periodo[$periodo])) {
        $promedios_por_periodo[$periodo] = array(
            'suma_notas_simples' => 0,
            'suma_notas_ponderadas' => 0,
            'suma_creditos_aprobados' => 0,
            'aprobadas_puntaje' => 0,
            'sin_nota_puntaje' => 0,
            'reprobadas_puntaje' => 0,
            'promedio_simple' => 0,
            'promedio_ponderado' => 0
        );
    }
    
    $totales['total_materias']++;
    
    if ($denominacion['tipo'] == 'puntaje') {
        $totales['total_materias_puntaje']++;
        $credito = intval($denominacion['credito']);
        
        if (empty($denominacion['nota_definitiva'])) {
            $totales['sin_nota_puntaje']++;
            $promedios_por_periodo[$periodo]['sin_nota_puntaje']++;
        } else {
            $nota = floatval($denominacion['nota_definitiva']);
            if ($nota >= 10) {
                $totales['suma_notas_simples'] += $nota;
                $totales['suma_notas_ponderadas'] += ($nota * $credito);
                $totales['suma_creditos_aprobados'] += $credito;
                $totales['aprobadas_puntaje']++;
                
                // Acumular para promedios por periodo
                $promedios_por_periodo[$periodo]['suma_notas_simples'] += $nota;
                $promedios_por_periodo[$periodo]['suma_notas_ponderadas'] += ($nota * $credito);
                $promedios_por_periodo[$periodo]['suma_creditos_aprobados'] += $credito;
                $promedios_por_periodo[$periodo]['aprobadas_puntaje']++;
            } else {
                $totales['reprobadas_puntaje']++;
                $promedios_por_periodo[$periodo]['reprobadas_puntaje']++;
            }
        }
    } else {
        $totales['total_materias_cualitativas']++;
        
        if (empty($denominacion['nota_definitiva'])) {
            $totales['sin_nota_cualitativas']++;
        } else {
            $nota_cualitativa = trim($denominacion['nota_definitiva']);
            if (strtoupper($nota_cualitativa) == 'APROBADO') {
                $totales['aprobadas_cualitativas']++;
            } else {
                $totales['reprobadas_cualitativas']++;
            }
        }
    }
}

// Calcular promedios generales (solo para materias de puntaje)
if ($totales['aprobadas_puntaje'] > 0) {
    $totales['promedio_simple'] = $totales['suma_notas_simples'] / $totales['aprobadas_puntaje'];
}
if ($totales['suma_creditos_aprobados'] > 0) {
    $totales['promedio_ponderado'] = $totales['suma_notas_ponderadas'] / $totales['suma_creditos_aprobados'];
}

// Calcular promedios por periodo (solo para materias de puntaje)
foreach ($promedios_por_periodo as $periodo => $datos) {
    if ($datos['aprobadas_puntaje'] > 0) {
        $promedios_por_periodo[$periodo]['promedio_simple'] = $datos['suma_notas_simples'] / $datos['aprobadas_puntaje'];
        if ($datos['suma_creditos_aprobados'] > 0) {
            $promedios_por_periodo[$periodo]['promedio_ponderado'] = $datos['suma_notas_ponderadas'] / $datos['suma_creditos_aprobados'];
        }
    }
}

// LÓGICA DE GUARDADO DE NOTAS - CORREGIDA PARA USAR FECHA_CREACION COMO FECHA DE PRIMERA NOTA
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['guardar_notas'])) {
    // Validar token CSRF
    if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        $_SESSION['error'] = "Token de seguridad inválido.";
        header("Location: " . $_SERVER['PHP_SELF'] . "?" . http_build_query($_GET));
        exit();
    }

    // Verificar que el coordinador tenga acceso a las escolaridades antes de guardar
    $conn->begin_transaction();
    try {
        $periodo_activo = $_POST['periodo_activo'];
        $notas_guardadas = 0;
        $notas_bloqueadas = 0;
        $notas_sin_acceso = 0;
        $notas_sin_cambio = 0;
        $notas_primer_ingreso = 0;

        foreach ($_POST['denominaciones'] as $inscripcion_id => $datos) {
            $inscripcion_id = intval($inscripcion_id);
            
            // Obtener información completa de la inscripción
            $sql_nota_actual = "SELECT i.nota_definitiva, i.fecha_creacion, i.escolaridad_id, d.periodo, d.tipo, 
                               i.veces_modificado, i.veces_modificada, i.modificado,
                               i.fecha_modificacion
                               FROM inscripciones i
                               JOIN denominaciones d ON i.denominacion_id = d.id
                               WHERE i.id = ? AND i.estudiante_id = ?";
            
            $stmt_nota_actual = $conn->prepare($sql_nota_actual);
            $stmt_nota_actual->bind_param("ii", $inscripcion_id, $estudiante_id);
            $stmt_nota_actual->execute();
            $nota_db = $stmt_nota_actual->get_result()->fetch_assoc();

            if (!$nota_db || $nota_db['periodo'] != $periodo_activo) {
                continue;
            }

            // Verificar acceso a la escolaridad
            if (!empty($escolaridades_permitidas) && !in_array($nota_db['escolaridad_id'], $escolaridades_permitidas)) {
                $notas_sin_acceso++;
                continue; // No tiene acceso a esta escolaridad
            }

            $nota_nueva_str = trim($datos['nota']);
            $nota_actual_str = $nota_db['nota_definitiva'] ? trim($nota_db['nota_definitiva']) : '';

            // Detectar si hay un cambio real
            $hay_cambio = false;
            
            if ($nota_db['tipo'] == 'puntaje') {
                // Para notas de puntaje, verificar si es número entero
                if ($nota_nueva_str !== '') {
                    if (!preg_match('/^\d+$/', $nota_nueva_str)) {
                        throw new Exception("Las notas de puntaje deben ser números enteros sin decimales.");
                    }
                    $nota_nueva_int = intval($nota_nueva_str);
                    if ($nota_nueva_int < 0 || $nota_nueva_int > 20) {
                        throw new Exception("Las notas deben estar entre 0 y 20.");
                    }
                }
                
                $nota_nueva_int = $nota_nueva_str === '' ? null : intval($nota_nueva_str);
                $nota_actual_int = $nota_actual_str === '' ? null : intval($nota_actual_str);
                
                if ($nota_nueva_int === null && $nota_actual_int !== null) {
                    $hay_cambio = true;
                } elseif ($nota_nueva_int !== null && $nota_actual_int === null) {
                    $hay_cambio = true;
                } elseif ($nota_nueva_int !== null && $nota_actual_int !== null) {
                    if ($nota_nueva_int !== $nota_actual_int) {
                        $hay_cambio = true;
                    }
                }
            } else {
                // Para notas cualitativas
                if ($nota_nueva_str === '') {
                    // Si la nota nueva está vacía, verificar si había una nota anterior
                    if (!empty($nota_actual_str)) {
                        $hay_cambio = true; // Cambio de tener nota a no tener nota
                    } else {
                        $notas_sin_cambio++;
                        continue;
                    }
                } else {
                    // Solo procesar si es "Aprobado" o "Reprobado"
                    $nota_nueva_str = ucfirst(strtolower(trim($nota_nueva_str)));
                    if (!in_array($nota_nueva_str, ['Aprobado', 'Reprobado'])) {
                        $notas_sin_cambio++;
                        continue;
                    }
                    
                    // Verificar si hay cambio
                    if ($nota_nueva_str !== $nota_actual_str) {
                        $hay_cambio = true;
                    } else {
                        $notas_sin_cambio++;
                        continue;
                    }
                }
            }

            if (!$hay_cambio) {
                $notas_sin_cambio++;
                continue;
            }

            // LÓGICA DE RESTRICCIÓN CORREGIDA - IMPORTANTE CAMBIO AQUÍ
            $es_primer_ingreso = empty($nota_actual_str);
            
            // SOLO aplicar restricciones si NO es primer ingreso
            if (!$es_primer_ingreso) {
                // Obtener el número de modificaciones - usar veces_modificado como principal
                $veces_modificada_valor = isset($nota_db['veces_modificado']) ? intval($nota_db['veces_modificado']) : 0;

                // CORRECCIÓN: Si nunca se ha modificado (veces_modificado = 0), entonces la fecha de creación
                // es cuando se creó la inscripción, NO cuando se puso la nota por primera vez
                // Por lo tanto, si veces_modificado = 0, NO aplicar restricción de tiempo
                
                if ($veces_modificada_valor >= 3) {
                    $notas_bloqueadas++;
                    continue; // Ya se modificó 3 veces, no se puede cambiar más
                }
                
                // Si ya se ha modificado al menos una vez, verificar tiempo desde la última modificación
                if ($veces_modificada_valor > 0 && !empty($nota_db['fecha_modificacion'])) {
                    $fecha_ultima_mod = new DateTime($nota_db['fecha_modificacion']);
                    $ahora = new DateTime();
                    $diferencia_dias = $ahora->diff($fecha_ultima_mod)->days;
                    
                    if ($diferencia_dias > 30) {
                        $notas_bloqueadas++;
                        continue; // Pasó más de 30 días desde la última modificación
                    }
                }
            } else {
                $notas_primer_ingreso++;
            }

            // Si pasa la validación, proceder a guardar
            $fecha_actual = date('Y-m-d H:i:s');
            $usuario_id = $_SESSION['user_id'];

            if ($nota_db['tipo'] == 'puntaje') {
                $nota_nueva = $nota_nueva_str === '' ? null : intval($nota_nueva_str);
                $estado_inscripcion = ($nota_nueva !== null && $nota_nueva >= 10) ? 'finalizada' : 'activa';
                
                // Determinar qué SQL usar basado en si es primer ingreso
                if ($es_primer_ingreso) {
                    // PRIMER INGRESO: Establecer fecha_creacion como fecha actual si es la primera vez que se pone nota
                    $sql_update = "UPDATE inscripciones SET nota_definitiva = ?, estado = ?, 
                                  fecha_modificacion = ?, usuario_modificacion = ?";
                    
                    // IMPORTANTE: Si es primer ingreso y no tiene fecha_creacion, establecerla ahora
                    if (empty($nota_db['fecha_creacion'])) {
                        $sql_update .= ", fecha_creacion = ?";
                    }
                    
                    // Establecer contadores en 0 para primer ingreso
                    if ($has_veces_modificado) {
                        $sql_update .= ", veces_modificado = 0";
                    }
                    if ($has_veces_modificada) {
                        $sql_update .= ", veces_modificada = 0";
                    }
                    if ($has_modificado) {
                        $sql_update .= ", modificado = 1";
                    }
                    
                    $sql_update .= " WHERE id = ?";
                    
                    $stmt_update = $conn->prepare($sql_update);
                    if (empty($nota_db['fecha_creacion'])) {
                        $stmt_update->bind_param("issssii", $nota_nueva, $estado_inscripcion, $fecha_actual, $usuario_id, $fecha_actual, $inscripcion_id);
                    } else {
                        $stmt_update->bind_param("issii", $nota_nueva, $estado_inscripcion, $fecha_actual, $usuario_id, $inscripcion_id);
                    }
                } else {
                    // MODIFICACIÓN: Incrementar contadores
                    $sql_update = "UPDATE inscripciones SET nota_definitiva = ?, estado = ?, 
                                  fecha_modificacion = ?, usuario_modificacion = ?";
                    
                    // Incrementar contadores de modificaciones
                    if ($has_veces_modificado) {
                        $sql_update .= ", veces_modificado = veces_modificado + 1";
                    }
                    if ($has_veces_modificada) {
                        $sql_update .= ", veces_modificada = veces_modificada + 1";
                    }
                    if ($has_modificado) {
                        $sql_update .= ", modificado = 1";
                    }
                    
                    $sql_update .= " WHERE id = ?";
                    
                    $stmt_update = $conn->prepare($sql_update);
                    $stmt_update->bind_param("issii", $nota_nueva, $estado_inscripcion, $fecha_actual, $usuario_id, $inscripcion_id);
                }
            } else { // Cualitativa
                $nota_nueva = $nota_nueva_str === '' ? null : $nota_nueva_str;
                $estado_inscripcion = ($nota_nueva === 'Aprobado') ? 'finalizada' : 'activa';
                
                // Determinar qué SQL usar basado en si es primer ingreso
                if ($es_primer_ingreso) {
                    // PRIMER INGRESO: Establecer fecha_creacion como fecha actual si es la primera vez que se pone nota
                    $sql_update = "UPDATE inscripciones SET nota_definitiva = ?, estado = ?, 
                                  fecha_modificacion = ?, usuario_modificacion = ?";
                    
                    // IMPORTANTE: Si es primer ingreso y no tiene fecha_creacion, establecerla ahora
                    if (empty($nota_db['fecha_creacion'])) {
                        $sql_update .= ", fecha_creacion = ?";
                    }
                    
                    // Establecer contadores en 0 para primer ingreso
                    if ($has_veces_modificado) {
                        $sql_update .= ", veces_modificado = 0";
                    }
                    if ($has_veces_modificada) {
                        $sql_update .= ", veces_modificada = 0";
                    }
                    if ($has_modificado) {
                        $sql_update .= ", modificado = 1";
                    }
                    
                    $sql_update .= " WHERE id = ?";
                    
                    $stmt_update = $conn->prepare($sql_update);
                    if (empty($nota_db['fecha_creacion'])) {
                        if ($nota_nueva === null) {
                            $stmt_update->bind_param("sssssii", $nota_nueva, $estado_inscripcion, $fecha_actual, $usuario_id, $fecha_actual, $inscripcion_id);
                        } else {
                            $stmt_update->bind_param("sssssii", $nota_nueva, $estado_inscripcion, $fecha_actual, $usuario_id, $fecha_actual, $inscripcion_id);
                        }
                    } else {
                        if ($nota_nueva === null) {
                            $stmt_update->bind_param("sssii", $nota_nueva, $estado_inscripcion, $fecha_actual, $usuario_id, $inscripcion_id);
                        } else {
                            $stmt_update->bind_param("sssii", $nota_nueva, $estado_inscripcion, $fecha_actual, $usuario_id, $inscripcion_id);
                        }
                    }
                } else {
                    // MODIFICACIÓN: Incrementar contadores
                    $sql_update = "UPDATE inscripciones SET nota_definitiva = ?, estado = ?, 
                                  fecha_modificacion = ?, usuario_modificacion = ?";
                    
                    // Incrementar contadores de modificaciones
                    if ($has_veces_modificado) {
                        $sql_update .= ", veces_modificado = veces_modificado + 1";
                    }
                    if ($has_veces_modificada) {
                        $sql_update .= ", veces_modificada = veces_modificada + 1";
                    }
                    if ($has_modificado) {
                        $sql_update .= ", modificado = 1";
                    }
                    
                    $sql_update .= " WHERE id = ?";
                    
                    $stmt_update = $conn->prepare($sql_update);
                    if ($nota_nueva === null) {
                        $stmt_update->bind_param("sssii", $nota_nueva, $estado_inscripcion, $fecha_actual, $usuario_id, $inscripcion_id);
                    } else {
                        $stmt_update->bind_param("sssii", $nota_nueva, $estado_inscripcion, $fecha_actual, $usuario_id, $inscripcion_id);
                    }
                }
            }
            
            $stmt_update->execute();
            $notas_guardadas++;
        }

        $conn->commit();

        $mensaje = "Proceso finalizado. ";
        if ($notas_guardadas > 0) $mensaje .= "$notas_guardadas nota(s) guardada(s) correctamente. ";
        if ($notas_primer_ingreso > 0) $mensaje .= "($notas_primer_ingreso primer ingreso) ";
        if ($notas_bloqueadas > 0) $mensaje .= "$notas_bloqueadas nota(s) no se pudieron modificar por estar bloqueadas. ";
        if ($notas_sin_acceso > 0) $mensaje .= "$notas_sin_acceso nota(s) no se pudieron modificar por falta de permisos. ";
        if ($notas_sin_cambio > 0 && $notas_guardadas == 0 && $notas_bloqueadas == 0 && $notas_sin_acceso == 0) {
            $mensaje = "No se detectaron cambios para guardar.";
        }
        
        $_SESSION['success'] = $mensaje;

    } catch (Exception $e) {
        $conn->rollback();
        $_SESSION['error'] = "Error al actualizar: " . $e->getMessage();
    }
    header("Location: " . $_SERVER['PHP_SELF'] . "?" . http_build_query($_GET));
    exit();
}

// Generar token CSRF
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// 6. Incluir header
include '../includes/header.php';
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        .estado-activo { color: #28a745; font-weight: bold; }
        .estado-suspendido, .estado-reprobado, .promedio-bajo { color: #dc3545; font-weight: bold; }
        .estado-graduado, .estado-finalizada { color: #17a2b8; font-weight: bold; }
        .estado-retirado { color: #6c757d; font-weight: bold; }
        .estado-aprobado, .promedio-alto { color: #28a745; font-weight: bold; }
        .estado-pendiente { color: #ffc107; font-weight: bold; }
        
        .tab-content {
            display: none;
        }
        .tab-content.show {
            display: block;
        }
        .tab-button {
            margin-right: 5px;
            margin-bottom: 5px;
        }
        .tab-button.active {
            background-color: #0d6efd;
            color: white;
        }
        .nota-input {
            width: 90px;
            text-align: center;
            font-weight: bold;
        }
        .tabs-container {
            gap: 5px;
        }
        .badge {
            font-size: 0.75rem;
        }
        .card {
            box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
            border: 1px solid rgba(0, 0, 0, 0.125);
        }
        .table th {
            background-color: #f8f9fa;
            border-bottom: 2px solid #dee2e6;
        }
        .resumen-card {
            height: 100%;
            transition: transform 0.2s;
        }
        .resumen-card:hover {
            transform: translateY(-2px);
        }
        .sin-nota {
            background-color: #fff3cd !important;
        }
        .sin-acceso {
            background-color: #f8d7da !important;
            opacity: 0.6;
        }
        .primer-ingreso {
            background-color: #d1e7dd !important;
        }
        .anio-lectivo-badge {
            font-size: 0.7rem;
        }
        .resumen-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            margin-bottom: 1rem;
        }
        .resumen-item {
            background: #f8f9fa;
            border-radius: 0.375rem;
            padding: 1rem;
            text-align: center;
            border-left: 4px solid #0d6efd;
        }
        .resumen-valor {
            font-size: 1.5rem;
            font-weight: bold;
            margin-bottom: 0.5rem;
        }
        .resumen-label {
            font-size: 0.875rem;
            color: #6c757d;
        }
        .table-responsive {
            margin-bottom: 1rem;
        }
        .select-cualitativo {
            width: 120px;
        }
        .form-control:focus {
            border-color: #86b7fe;
            box-shadow: 0 0 0 0.25rem rgba(13, 110, 253, 0.25);
        }
        .is-invalid {
            border-color: #dc3545;
            box-shadow: 0 0 0 0.25rem rgba(220, 53, 69, 0.25);
        }
        .changed {
            background-color: #e7f3ff !important;
            border: 2px solid #0d6efd !important;
        }
        .nota-bloqueada {
            background-color: #f8d7da !important;
            color: #721c24 !important;
        }
        .nota-modificable {
            background-color: #d1ecf1 !important;
            color: #0c5460 !important;
        }
        .primer-ingreso-input {
            background-color: #d1e7dd !important;
            border: 2px solid #198754 !important;
        }
        .modificaciones-info {
            font-size: 0.75rem;
            margin-top: 2px;
        }
        .info-primer-ingreso {
            font-size: 0.75rem;
            color: #198754;
            font-weight: bold;
        }
        .decimal-warning {
            font-size: 0.7rem;
            color: #dc3545;
            margin-top: 2px;
        }
        .cualitativa-vacia {
            background-color: #fff3cd !important;
            border: 2px solid #ffc107 !important;
        }
        .fecha-info {
            font-size: 0.7rem;
            color: #6c757d;
            margin-top: 2px;
        }
    </style>
</head>
<body>
    <div class="container-fluid px-3 px-md-4 mt-4">
        <div class="mb-3 mb-md-4">
            <a href="cargar_notas.php?postgrado_id=<?php echo $postgrado_seleccionado; ?>" class="btn btn-secondary btn-sm">
                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-arrow-left" viewBox="0 0 16 16">
                    <path fill-rule="evenodd" d="M15 8a.5.5 0 0 0-.5-.5H2.707l3.147-3.146a.5.5 0 1 0-.708-.708l-4 4a.5.5 0 0 0 0 .708l4 4a.5.5 0 0 0 .708-.708L2.707 8.5H14.5A.5.5 0 0 0 15 8z"/>
                </svg>
                Volver al Listado
            </a>
        </div>

        <?php if (isset($_SESSION['success'])): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <?= $_SESSION['success'] ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
            <?php unset($_SESSION['success']); ?>
        <?php endif; ?>
        
        <?php if (isset($_SESSION['error'])): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <?= $_SESSION['error'] ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
            <?php unset($_SESSION['error']); ?>
        <?php endif; ?>

        <div class="card border-0 shadow-sm">
            <div class="card-body p-0">
                <div class="p-3 bg-light border-bottom">
                    <h3 class="mb-0">Notas para: <?php echo htmlspecialchars($estudiante_info['nombres'] . ' ' . $estudiante_info['apellidos']); ?></h3>
                    <p class="mb-0">
                        <strong><?php echo htmlspecialchars($tipo_documento_texto); ?>:</strong> <?php echo htmlspecialchars($numero_documento); ?>
                    </p>
                    <p class="mb-0">Postgrado: <strong><?php echo htmlspecialchars($estudiante_info['postgrado_nombre']); ?></strong></p>
                    <p class="mb-0">Estado del Postgrado: 
                        <span class="estado-<?php echo strtolower(htmlspecialchars($estudiante_info['estado_postgrado'])); ?>">
                            <?php echo htmlspecialchars(ucfirst($estudiante_info['estado_postgrado'])); ?>
                        </span>
                    </p>
                    
                    <?php if (!empty($escolaridades_sin_acceso)): ?>
                        <div class="alert alert-warning mt-2 mb-0 p-2">
                            <small>
                                <strong>Advertencia:</strong> No tiene acceso a las siguientes escolaridades: 
                                <?php echo htmlspecialchars(implode(', ', $escolaridades_sin_acceso)); ?>
                            </small>
                        </div>
                    <?php endif; ?>
                    
                    <?php 
                    // Información sobre las columnas detectadas
                    echo '<div class="alert alert-info mt-2 mb-0 p-2">';
                    echo '<small><strong>Configuración detectada:</strong><br>';
                    echo '- Columna "veces_modificado": ' . ($has_veces_modificado ? 'SÍ' : 'NO') . '<br>';
                    echo '- Columna "veces_modificada": ' . ($has_veces_modificada ? 'SÍ' : 'NO') . '<br>';
                    echo '- Columna "modificado": ' . ($has_modificado ? 'SÍ' : 'NO');
                    echo '</small>';
                    echo '</div>';
                    ?>
                </div>

                <?php if (!empty($denominaciones)): ?>
                    <div class="p-3 border-bottom">
                        <h4 class="mb-3">Resumen Académico</h4>
                        
                        <div class="resumen-grid">
                            <div class="resumen-item">
                                <div class="resumen-valor"><?php echo $totales['total_materias']; ?></div>
                                <div class="resumen-label">Total Asignaturas</div>
                                <small class="text-muted">(Con acceso)</small>
                            </div>
                            <div class="resumen-item">
                                <div class="resumen-valor text-warning"><?php echo $totales['sin_nota_puntaje'] + $totales['sin_nota_cualitativas']; ?></div>
                                <div class="resumen-label">Sin Nota</div>
                                <small class="text-muted">
                                    P: <?php echo $totales['sin_nota_puntaje']; ?> | 
                                    C: <?php echo $totales['sin_nota_cualitativas']; ?>
                                </small>
                            </div>
                            <div class="resumen-item">
                                <div class="resumen-valor text-success"><?php echo $totales['aprobadas_puntaje'] + $totales['aprobadas_cualitativas']; ?></div>
                                <div class="resumen-label">Aprobadas</div>
                                <small class="text-muted">
                                    P: <?php echo $totales['aprobadas_puntaje']; ?> | 
                                    C: <?php echo $totales['aprobadas_cualitativas']; ?>
                                </small>
                            </div>
                            <div class="resumen-item">
                                <div class="resumen-valor text-danger"><?php echo $totales['reprobadas_puntaje'] + $totales['reprobadas_cualitativas']; ?></div>
                                <div class="resumen-label">Reprobadas</div>
                                <small class="text-muted">
                                    P: <?php echo $totales['reprobadas_puntaje']; ?> | 
                                    C: <?php echo $totales['reprobadas_cualitativas']; ?>
                                </small>
                            </div>
                            <div class="resumen-item">
                                <div class="resumen-valor <?php echo ($totales['promedio_simple'] < 15 && $totales['promedio_simple'] > 0) ? 'text-danger' : 'text-primary'; ?>">
                                    <?php echo number_format($totales['promedio_simple'], 2); ?>
                                </div>
                                <div class="resumen-label">Promedio Simple</div>
                            </div>
                            <div class="resumen-item">
                                <div class="resumen-valor <?php echo ($totales['promedio_ponderado'] < 15 && $totales['promedio_ponderado'] > 0) ? 'text-danger' : 'text-primary'; ?>">
                                    <?php echo number_format($totales['promedio_ponderado'], 2); ?>
                                </div>
                                <div class="resumen-label">Promedio Ponderado</div>
                            </div>
                            <div class="resumen-item">
                                <div class="resumen-valor text-info"><?php echo $totales['suma_creditos_aprobados']; ?></div>
                                <div class="resumen-label">Créditos Aprobados</div>
                            </div>
                        </div>

                        <?php if (!empty($escolaridades_sin_acceso)): ?>
                            <div class="alert alert-warning mt-3">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-exclamation-triangle me-2" viewBox="0 0 16 16">
                                    <path d="M7.938 2.016A.13.13 0 0 1 8.002 2a.13.13 0 0 1 .063.016.146.146 0 0 1 .054.057l6.857 11.667c.036.06.035.124.002.183a.163.163 0 0 1-.054.06.116.116 0 0 1-.066.017H1.146a.115.115 0 0 1-.066-.017.163.163 0 0 1-.054-.06.176.176 0 0 1 .002-.183L7.884 2.073a.147.147 0 0 1 .054-.057zm1.044-.45a1.13 1.13 0 0 0-1.96 0L.165 13.233c-.457.778.091 1.767.98 1.767h13.713c.889 0 1.438-.99.98-1.767L8.982 1.566z"/>
                                    <path d="M7.002 12a1 1 0 1 1 2 0 1 1 0 0 1-2 0zM7.1 5.995a.905.905 0 1 1 1.8 0l-.35 3.507a.552.552 0 0 1-1.1 0L7.1 5.995z"/>
                                </svg>
                                <strong>Atención:</strong> No tiene permisos para gestionar todas las escolaridades de este estudiante.
                                Solo puede modificar asignaturas de escolaridades a las que tiene acceso.
                            </div>
                        <?php endif; ?>

                        <div class="alert alert-info mt-3">
                            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-info-circle me-2" viewBox="0 0 16 16">
                                <path d="M8 15A7 7 0 1 1 8 1a7 7 0 0 1 0 14zm0 1A8 8 0 1 0 8 0a8 8 0 0 0 0 16z"/>
                                <path d="m8.93 6.588-2.29.287-.082.38.45.083c.294.07.352.176.288.469l-.738 3.468c-.194.897.105 1.319.808 1.319.545 0 1.178-.252 1.465-.598l.088-.416c-.2.176-.492.246-.686.246-.275 0-.375-.193-.304-.533L8.93 6.588zM9 4.5a1 1 0 1 1-2 0 1 1 0 0 1 2 0z"/>
                            </svg>
                            <strong>Información:</strong> 
                            <ul class="mb-0 mt-2">
                                <li>Las notas de <strong>puntaje</strong> deben ser <strong>números enteros del 0 al 20</strong> (sin decimales).</li>
                                <li>Las notas <strong>cualitativas</strong> deben ser "Aprobado" o "Reprobado".</li>
                                <li>Las notas <strong>sin valor (primer ingreso)</strong> pueden colocarse en cualquier momento.</li>
                                <li>Las notas <strong>ya existentes</strong> pueden modificarse hasta 3 veces como máximo dentro de 30 días desde la última modificación.</li>
                                <li><strong>IMPORTANTE:</strong> La fecha de creación mostrada es cuando se creó la inscripción. La fecha de primera nota puede ser diferente.</li>
                            </ul>
                        </div>
                    </div>

                    <div class="p-3">
                        <form method="post" id="form-notas">
                            <input type="hidden" name="estudiante_id" value="<?php echo $estudiante_id; ?>">
                            <input type="hidden" name="postgrado_id" value="<?php echo $postgrado_seleccionado; ?>">
                            <input type="hidden" name="guardar_notas" value="1">
                            <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                            <input type="hidden" name="periodo_activo" id="periodo-activo" value="<?php echo isset(array_keys($periodos)[0]) ? array_keys($periodos)[0] : ''; ?>">

                            <div class="d-flex flex-wrap tabs-container mb-3" id="periodos-tabs">
                                <?php 
                                $periodos_keys = array_keys($periodos);
                                $tabs_per_row = 6;
                                $rows = ceil(count($periodos_keys) / $tabs_per_row);
                                
                                for ($row = 0; $row < $rows; $row++): ?>
                                    <div class="d-flex w-100 mb-2">
                                        <?php 
                                        $start = $row * $tabs_per_row;
                                        $end = min(($row + 1) * $tabs_per_row, count($periodos_keys));
                                        
                                        for ($i = $start; $i < $end; $i++): 
                                            $periodo_nombre = $periodos_keys[$i];
                                            $sin_nota_periodo = isset($promedios_por_periodo[$periodo_nombre]['sin_nota_puntaje']) ? $promedios_por_periodo[$periodo_nombre]['sin_nota_puntaje'] : 0;
                                            $sin_nota_cualitativas_periodo = 0;
                                            foreach ($periodos[$periodo_nombre] as $denom) {
                                                if ($denom['tipo'] == 'cualitativa' && empty($denom['nota_definitiva'])) {
                                                    $sin_nota_cualitativas_periodo++;
                                                }
                                            }
                                            $total_sin_nota = $sin_nota_periodo + $sin_nota_cualitativas_periodo;
                                        ?>
                                            <button type="button" class="btn btn-outline-primary tab-button position-relative <?php echo ($i === 0) ? 'active' : ''; ?>" 
                                                    data-target="tab-<?php echo $i; ?>" 
                                                    data-periodo="<?php echo htmlspecialchars($periodo_nombre); ?>">
                                                Periodo <?php echo htmlspecialchars($periodo_nombre); ?>
                                                <?php if ($total_sin_nota > 0): ?>
                                                    <span class="position-absolute top-0 start-100 translate-middle badge rounded-pill bg-warning">
                                                        <?php echo $total_sin_nota; ?>
                                                    </span>
                                                <?php endif; ?>
                                            </button>
                                        <?php endfor; ?>
                                    </div>
                                <?php endfor; ?>
                            </div>

                            <?php 
                            $periodo_index = 0;
                            foreach ($periodos as $periodo_nombre => $denom_list): 
                                $promedio_simple_periodo = isset($promedios_por_periodo[$periodo_nombre]['promedio_simple']) ? number_format($promedios_por_periodo[$periodo_nombre]['promedio_simple'], 2) : 'N/A';
                                $promedio_ponderado_periodo = isset($promedios_por_periodo[$periodo_nombre]['promedio_ponderado']) ? number_format($promedios_por_periodo[$periodo_nombre]['promedio_ponderado'], 2) : 'N/A';
                                $sin_nota_periodo = isset($promedios_por_periodo[$periodo_nombre]['sin_nota_puntaje']) ? $promedios_por_periodo[$periodo_nombre]['sin_nota_puntaje'] : 0;
                                $sin_nota_cualitativas_periodo = 0;
                                foreach ($denom_list as $denom) {
                                    if ($denom['tipo'] == 'cualitativa' && empty($denom['nota_definitiva'])) {
                                        $sin_nota_cualitativas_periodo++;
                                    }
                                }
                                $total_sin_nota_periodo = $sin_nota_periodo + $sin_nota_cualitativas_periodo;
                            ?>
                                <div id="tab-<?php echo $periodo_index; ?>" class="tab-content <?php echo ($periodo_index === 0) ? 'show active' : ''; ?>">
                                    <div class="d-flex justify-content-between align-items-center mb-3">
                                        <h5>Periodo: <?php echo htmlspecialchars($periodo_nombre); ?></h5>
                                        <div>
                                            <?php if ($total_sin_nota_periodo > 0): ?>
                                                <span class="badge bg-warning me-2">Sin nota: <?php echo $total_sin_nota_periodo; ?> (P:<?php echo $sin_nota_periodo; ?>/C:<?php echo $sin_nota_cualitativas_periodo; ?>)</span>
                                            <?php endif; ?>
                                            <span class="badge bg-primary me-2">Prom. Simple: <?php echo $promedio_simple_periodo; ?></span>
                                            <span class="badge bg-primary">Prom. Ponderado: <?php echo $promedio_ponderado_periodo; ?></span>
                                        </div>
                                    </div>
                                    
                                    <div class="table-responsive">
                                        <table class="table table-striped table-hover">
                                            <thead class="table-light">
                                                <tr>
                                                    <th>Código</th>
                                                    <th>Denominación</th>
                                                    <th>Crédito</th>
                                                    <th>Tipo</th>
                                                    <th>Escolaridad</th>
                                                    <th>Año Lectivo</th>
                                                    <th>Nota</th>
                                                    <th>Estado</th>
                                                    <th>Última Modificación</th>
                                                    <th>Modificado Por</th>
                                                    <th>Fecha Creación Inscripción</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach ($denom_list as $denominacion): ?>
                                                    <?php 
                                                    // Verificar acceso a la escolaridad
                                                    $tiene_acceso_escolaridad = empty($escolaridades_permitidas) || in_array($denominacion['escolaridad_id'], $escolaridades_permitidas);
                                                    
                                                    // Determinar si es primer ingreso (nota vacía)
                                                    $es_primer_ingreso = empty($denominacion['nota_definitiva']);
                                                    
                                                    // Lógica para determinar si un campo está bloqueado
                                                    $is_locked = false;
                                                    $lock_reason = '';
                                                    $modificaciones_restantes = 3;
                                                    $estado_modificacion = 'editabled';
                                                    $dias_desde_modificacion = 0;
                                                    $fecha_creacion_formateada = '';
                                                    $fecha_modificacion_formateada = '';
                                                    
                                                    if (!empty($denominacion['fecha_creacion'])) {
                                                        $fecha_creacion_formateada = date('d/m/Y H:i', strtotime($denominacion['fecha_creacion']));
                                                    }
                                                    
                                                    if (!empty($denominacion['fecha_modificacion'])) {
                                                        $fecha_modificacion_formateada = date('d/m/Y H:i', strtotime($denominacion['fecha_modificacion']));
                                                    }
                                                    
                                                    if (!$tiene_acceso_escolaridad) {
                                                        $is_locked = true;
                                                        $lock_reason = 'No tiene acceso a esta escolaridad.';
                                                        $estado_modificacion = 'sin_acceso';
                                                    } elseif (!$es_primer_ingreso) {
                                                        // Solo aplicar restricciones si NO es primer ingreso
                                                        // Obtener el número de modificaciones
                                                        $veces_modificada_valor = isset($denominacion['veces_modificado']) ? intval($denominacion['veces_modificado']) : 0;
                                                        
                                                        $modificaciones_restantes = 3 - $veces_modificada_valor;
                                                        
                                                        if ($veces_modificada_valor >= 3) {
                                                            $is_locked = true;
                                                            $lock_reason = 'Esta nota ya fue modificada 3 veces y no puede volver a cambiarse.';
                                                            $estado_modificacion = 'bloqueada';
                                                        } elseif ($veces_modificada_valor > 0 && !empty($denominacion['fecha_modificacion'])) {
                                                            // Si ya se ha modificado al menos una vez, verificar tiempo desde la última modificación
                                                            $fecha_ultima_mod = new DateTime($denominacion['fecha_modificacion']);
                                                            $ahora = new DateTime();
                                                            $dias_desde_modificacion = $ahora->diff($fecha_ultima_mod)->days;
                                                            
                                                            if ($dias_desde_modificacion > 30) {
                                                                $is_locked = true;
                                                                $lock_reason = 'Han pasado ' . $dias_desde_modificacion . ' días desde la última modificación (> 30 días).';
                                                                $estado_modificacion = 'bloqueada';
                                                            } else {
                                                                $estado_modificacion = 'modificable';
                                                            }
                                                        } else {
                                                            // Si nunca se ha modificado (veces_modificado = 0), permitir modificación
                                                            $estado_modificacion = 'modificable';
                                                        }
                                                    }
                                                    
                                                    $fecha_modificacion = !empty($denominacion['fecha_modificacion']) ? date('d/m/Y H:i', strtotime($denominacion['fecha_modificacion'])) : 'No modificada';
                                                    $modificado_por = !empty($denominacion['modificado_por_nombres']) ? 
                                                        htmlspecialchars($denominacion['modificado_por_nombres'] . ' ' . $denominacion['modificado_por_apellidos']) : 
                                                        (empty($denominacion['nota_definitiva']) ? 'N/A' : 'Sistema');
                                                    $sin_nota = empty($denominacion['nota_definitiva']);
                                                    $tipo_denominacion = ($denominacion['tipo'] == 'puntaje') ? 'Numérica' : 'Cualitativa';
                                                    ?>
                                                    <tr class="<?php echo $sin_nota ? 'sin-nota primer-ingreso' : ''; ?> <?php echo !$tiene_acceso_escolaridad ? 'sin-acceso' : ($is_locked ? 'nota-bloqueada' : ($estado_modificacion == 'modificable' ? 'nota-modificable' : '')); ?>">
                                                        <td><?php echo htmlspecialchars($denominacion['codigo']); ?></td>
                                                        <td><?php echo htmlspecialchars($denominacion['nombre']); ?></td>
                                                        <td><?php echo htmlspecialchars($denominacion['credito']); ?></td>
                                                        <td>
                                                            <span class="badge <?php echo $denominacion['tipo'] == 'puntaje' ? 'bg-primary' : 'bg-info'; ?>">
                                                                <?php echo $tipo_denominacion; ?>
                                                            </span>
                                                        </td>
                                                        <td>
                                                            <span class="badge <?php echo $tiene_acceso_escolaridad ? 'bg-secondary' : 'bg-danger'; ?>">
                                                                <?php echo htmlspecialchars($denominacion['escolaridad_nombre']); ?>
                                                                <?php if (!$tiene_acceso_escolaridad): ?>
                                                                    <i class="bi bi-lock ms-1"></i>
                                                                <?php endif; ?>
                                                            </span>
                                                        </td>
                                                        <td>
                                                            <span class="badge bg-dark anio-lectivo-badge">
                                                                <?php echo htmlspecialchars($denominacion['anio_lectivo_nombre']); ?>
                                                            </span>
                                                        </td>
                                                        <td>
                                                            <?php if ($denominacion['tipo'] == 'puntaje'): ?>
                                                                <input type="number" step="1" min="0" max="20" 
                                                                       name="denominaciones[<?php echo $denominacion['id_inscripcion']; ?>][nota]" 
                                                                       class="form-control nota-input <?php echo $sin_nota ? 'border-warning primer-ingreso-input' : ''; ?> <?php echo $is_locked ? 'nota-bloqueada' : ''; ?>" 
                                                                       value="<?php echo htmlspecialchars($denominacion['nota_definitiva'] ?? ''); ?>"
                                                                       placeholder="0-20"
                                                                       <?php if ($is_locked) echo 'readonly'; ?>>
                                                            <?php else: ?>
                                                                <select name="denominaciones[<?php echo $denominacion['id_inscripcion']; ?>][nota]" class="form-control select-cualitativo <?php echo $sin_nota ? 'border-warning primer-ingreso-input' : ''; ?> <?php echo $is_locked ? 'nota-bloqueada' : ''; ?>" <?php if ($is_locked) echo 'disabled'; ?>>
                                                                    <option value="">- Seleccionar -</option>
                                                                    <option value="Aprobado" <?php echo ($denominacion['nota_definitiva'] == 'Aprobado') ? 'selected' : ''; ?>>Aprobado</option>
                                                                    <option value="Reprobado" <?php echo ($denominacion['nota_definitiva'] == 'Reprobado') ? 'selected' : ''; ?>>Reprobado</option>
                                                                </select>
                                                            <?php endif; ?>
                                                            <div class="modificaciones-info <?php echo $is_locked ? 'text-danger' : ($es_primer_ingreso ? 'text-success' : 'text-info'); ?>">
                                                                <?php if (!$tiene_acceso_escolaridad): ?>
                                                                    <svg xmlns="http://www.w3.org/2000/svg" width="10" height="10" fill="currentColor" class="bi bi-ban me-1" viewBox="0 0 16 16">
                                                                        <path d="M15 8a6.973 6.973 0 0 0-1.71-4.584l-9.874 9.875A7 7 0 0 0 15 8M2.71 12.584l9.874-9.875a7 7 0 0 0-9.874 9.874ZM16 8A8 8 0 1 1 0 8a8 8 0 0 1 16 0"/>
                                                                    </svg>
                                                                    Sin acceso
                                                                <?php elseif ($is_locked): ?>
                                                                    <svg xmlns="http://www.w3.org/2000/svg" width="10" height="10" fill="currentColor" class="bi bi-lock-fill me-1" viewBox="0 0 16 16">
                                                                        <path d="M8 1a2 2 0 0 1 2 2v4H6V3a2 2 0 0 1 2-2zm3 6V3a3 3 0 0 0-6 0v4a2 2 0 0 0-2 2v5a2 2 0 0 0 2 2h6a2 2 0 0 0 2-2V9a2 2 0 0 0-2-2z"/>
                                                                    </svg>
                                                                    Bloqueada: <?php echo $lock_reason; ?>
                                                                <?php elseif ($es_primer_ingreso): ?>
                                                                    <svg xmlns="http://www.w3.org/2000/svg" width="10" height="10" fill="currentColor" class="bi bi-plus-circle me-1" viewBox="0 0 16 16">
                                                                        <path d="M8 15A7 7 0 1 1 8 1a7 7 0 0 1 0 14zm0 1A8 8 0 1 0 8 0a8 8 0 0 0 0 16z"/>
                                                                        <path d="M8 4a.5.5 0 0 1 .5.5v3h3a.5.5 0 0 1 0 1h-3v3a.5.5 0 0 1-1 0v-3h-3a.5.5 0 0 1 0-1h3v-3A.5.5 0 0 1 8 4z"/>
                                                                    </svg>
                                                                    <span class="info-primer-ingreso">Primer ingreso</span>
                                                                <?php else: ?>
                                                                    <svg xmlns="http://www.w3.org/2000/svg" width="10" height="10" fill="currentColor" class="bi bi-pencil-square me-1" viewBox="0 0 16 16">
                                                                        <path d="M15.502 1.94a.5.5 0 0 1 0 .706L14.459 3.69l-2-2L13.502.646a.5.5 0 0 1 .707 0l1.293 1.293zm-1.75 2.456-2-2L4.939 9.21a.5.5 0 0 0-.121.196l-.805 2.414a.25.25 0 0 0 .316.316l2.414-.805a.5.5 0 0 0 .196-.12l6.813-6.814z"/>
                                                                        <path fill-rule="evenodd" d="M1 13.5A1.5 1.5 0 0 0 2.5 15h11a1.5 1.5 0 0 0 1.5-1.5v-6a.5.5 0 0 0-1 0v6a.5.5 0 0 1-.5.5h-11a.5.5 0 0 1-.5-.5v-11a.5.5 0 0 1 .5-.5H9a.5.5 0 0 0 0-1H2.5A1.5 1.5 0 0 0 1 2.5v11z"/>
                                                                    </svg>
                                                                    Modificaciones restantes: <?php echo $modificaciones_restantes; ?> de 3
                                                                    <?php if ($dias_desde_modificacion > 0): ?>
                                                                        <br><small>Última modificación hace <?php echo $dias_desde_modificacion; ?> días</small>
                                                                    <?php endif; ?>
                                                                <?php endif; ?>
                                                            </div>
                                                            <?php if ($denominacion['tipo'] == 'puntaje'): ?>
                                                                <div class="decimal-warning">
                                                                    <small>Solo números enteros (0-20)</small>
                                                                </div>
                                                            <?php else: ?>
                                                                <div class="decimal-warning">
                                                                    <small>Solo "Aprobado" o "Reprobado"</small>
                                                                </div>
                                                            <?php endif; ?>
                                                        </td>
                                                        <td>
                                                            <span class="estado-<?php echo strtolower($denominacion['estado']); ?>">
                                                                <?php echo htmlspecialchars(ucfirst($denominacion['estado'])); ?>
                                                            </span>
                                                        </td>
                                                        <td>
                                                            <small class="text-muted">
                                                                <?php echo $fecha_modificacion; ?>
                                                            </small>
                                                            <?php if (!empty($fecha_modificacion_formateada)): ?>
                                                                <div class="fecha-info">
                                                                    <small><?php echo $fecha_modificacion_formateada; ?></small>
                                                                </div>
                                                            <?php endif; ?>
                                                        </td>
                                                        <td>
                                                            <small class="text-muted">
                                                                <?php echo $modificado_por; ?>
                                                            </small>
                                                        </td>
                                                        <td>
                                                            <?php if (!empty($fecha_creacion_formateada)): ?>
                                                                <small class="text-muted">
                                                                    <?php echo $fecha_creacion_formateada; ?>
                                                                </small>
                                                                <div class="fecha-info">
                                                                    <small>Creación de inscripción</small>
                                                                </div>
                                                            <?php else: ?>
                                                                <small class="text-muted">N/A</small>
                                                            <?php endif; ?>
                                                        </td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                                <?php $periodo_index++; ?>
                            <?php endforeach; ?>

                            <div class="mt-4">
                                <button type="submit" class="btn btn-success btn-lg">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-check-circle me-1" viewBox="0 0 16 16">
                                        <path d="M8 15A7 7 0 1 1 8 1a7 7 0 0 1 0 14zm0 1A8 8 0 1 0 8 0a8 8 0 0 0 0 16z"/>
                                        <path d="M10.97 4.97a.235.235 0 0 0-.02.022L7.477 9.417 5.384 7.323a.75.75 0 0 0-1.06 1.06L6.97 11.03a.75.75 0 0 0 1.079-.02l3.992-4.99a.75.75 0 0 0-1.071-1.05z"/>
                                    </svg>
                                    Guardar Cambios
                                </button>
                            </div>
                        </form>
                    </div>
                <?php else: ?>
                    <div class="p-4 text-center">
                        <div class="alert alert-info">
                            <h4>No hay asignaturas inscritas</h4>
                            <p>El estudiante no tiene asignaturas inscritas en este postgrado o no tiene acceso a sus escolaridades.</p>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Manejo de pestañas
            const tabButtons = document.querySelectorAll('.tab-button');
            const tabContents = document.querySelectorAll('.tab-content');
            const periodoActivoInput = document.getElementById('periodo-activo');

            tabButtons.forEach(button => {
                button.addEventListener('click', function() {
                    const target = this.getAttribute('data-target');
                    const periodo = this.getAttribute('data-periodo');
                    
                    // Actualizar pestaña activa
                    tabButtons.forEach(btn => btn.classList.remove('active'));
                    this.classList.add('active');
                    
                    // Mostrar contenido correspondiente
                    tabContents.forEach(content => {
                        content.classList.remove('show', 'active');
                    });
                    document.getElementById(target).classList.add('show', 'active');
                    
                    // Actualizar periodo activo en el formulario
                    periodoActivoInput.value = periodo;
                });
            });

            // Validación de notas (0-20) solo números enteros
            document.querySelectorAll('.nota-input').forEach(input => {
                input.addEventListener('input', function() {
                    if (this.readOnly) return;
                    
                    // Remover cualquier caracter no numérico
                    this.value = this.value.replace(/[^0-9]/g, '');
                    
                    if (this.value === '') return;
                    
                    let value = parseInt(this.value);
                    if (isNaN(value)) {
                        this.value = '';
                    } else if (value < 0) {
                        this.value = 0;
                    } else if (value > 20) {
                        this.value = 20;
                    }
                });
                
                input.addEventListener('blur', function() {
                    if (this.readOnly) return;
                    
                    if (this.value === '') return;
                    let value = parseInt(this.value);
                    if (isNaN(value) || value < 0 || value > 20) {
                        this.classList.add('is-invalid');
                    } else {
                        this.classList.remove('is-invalid');
                    }
                });
            });

            // Validación del formulario al enviar
            document.getElementById('form-notas').addEventListener('submit', function(e) {
                let hasErrors = false;
                const currentTab = document.querySelector('.tab-content.show.active');
                const inputs = currentTab.querySelectorAll('.nota-input');
                const selects = currentTab.querySelectorAll('.select-cualitativo');
                
                // Validar inputs numéricos
                inputs.forEach(input => {
                    if (!input.readOnly && input.value) {
                        const value = parseInt(input.value);
                        if (isNaN(value) || value < 0 || value > 20) {
                            input.classList.add('is-invalid');
                            hasErrors = true;
                        } else {
                            input.classList.remove('is-invalid');
                        }
                    }
                });
                
                // Validar selects cualitativos
                selects.forEach(select => {
                    if (!select.disabled && select.value) {
                        const value = select.value.trim();
                        if (!['Aprobado', 'Reprobado'].includes(value)) {
                            select.classList.add('is-invalid');
                            hasErrors = true;
                        } else {
                            select.classList.remove('is-invalid');
                        }
                    }
                });
                
                if (hasErrors) {
                    e.preventDefault();
                    alert('Por favor, corrija las notas:\n- Notas numéricas: enteros del 0 al 20\n- Notas cualitativas: "Aprobado" o "Reprobado"');
                }
            });
        });
    </script>
</body>
</html>

<?php
// 7. Incluir footer
include '../includes/footer.php';
$conn->close();
?>