<?php
// 1. Configuración de sesión
require_once __DIR__ . '/../includes/session_config.php';

// 2. Verificación de auth
$allowed_roles = ['coordinador1'];
require_once __DIR__ . '/../includes/auth_check.php';

// 3. Configurar título de página
$page_title = "Cargar Notas - Coordinador";

// 4. Configurar encoding UTF-8
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');

// 5. Lógica específica de la página
include '../includes/db.php';

// Configurar conexión para usar UTF-8
$conn->set_charset("utf8mb4");
$conn->query("SET NAMES 'utf8mb4'");
$conn->query("SET CHARACTER SET utf8mb4");
$conn->query("SET COLLATION_CONNECTION = 'utf8mb4_unicode_ci'");

if ($conn->connect_error) {
    die("Error de conexión: " . $conn->connect_error);
}

// Habilitar reportes de error para depuración
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

// -----------------------------------------------------------------------------
// OBTENER PERMISOS DEL COORDINADOR
// -----------------------------------------------------------------------------
$coordinador_id = $_SESSION['user_id'];

// Obtener información completa del coordinador
$sql_coordinador_info = "SELECT u.nombres, u.apellidos, u.username, u.email 
                        FROM usuarios u 
                        WHERE u.id = ?";
$stmt_coordinador = $conn->prepare($sql_coordinador_info);
$stmt_coordinador->bind_param("i", $coordinador_id);
$stmt_coordinador->execute();
$coordinador_info = $stmt_coordinador->get_result()->fetch_assoc();

$sql_permisos = "SELECT cp.postgrado_id, p.nombre as postgrado_nombre 
                FROM coordinador_postgrado cp
                JOIN postgrados p ON cp.postgrado_id = p.id
                WHERE cp.coordinador_id = ?";
$stmt_permisos = $conn->prepare($sql_permisos);
$stmt_permisos->bind_param("i", $coordinador_id);
$stmt_permisos->execute();
$result_permisos = $stmt_permisos->get_result();
$postgrados_permitidos_ids = [];
$postgrados_permitidos_info = [];
while ($row = $result_permisos->fetch_assoc()) {
    $postgrados_permitidos_ids[] = $row['postgrado_id'];
    $postgrados_permitidos_info[$row['postgrado_id']] = $row['postgrado_nombre'];
}

if (empty($postgrados_permitidos_ids)) {
    include '../includes/header.php';
    echo '<div class="container mt-5"><div class="alert alert-danger">Error: No tiene ningún postgrado asignado. Por favor, contacte a un administrador.</div></div>';
    include '../includes/footer.php';
    exit();
}

// Obtener solo los postgrados permitidos para el coordinador
$placeholders_postgrados = implode(',', array_fill(0, count($postgrados_permitidos_ids), '?'));
$sql_postgrados = "SELECT p.id, p.nombre 
                  FROM postgrados p
                  WHERE p.id IN ($placeholders_postgrados)
                  ORDER BY p.nombre";
$stmt_postgrados = $conn->prepare($sql_postgrados);
$types_postgrados = str_repeat('i', count($postgrados_permitidos_ids));
$stmt_postgrados->bind_param($types_postgrados, ...$postgrados_permitidos_ids);
$stmt_postgrados->execute();
$result_postgrados = $stmt_postgrados->get_result();
$postgrados = $result_postgrados->fetch_all(MYSQLI_ASSOC);

// Variables para filtros (con sanitización)
$cedula = isset($_GET['cedula']) ? $conn->real_escape_string($_GET['cedula']) : '';
$postgrado_seleccionado = isset($_GET['postgrado_id']) ? intval($_GET['postgrado_id']) : '';
$anio_lectivo = isset($_GET['anio_lectivo']) ? $conn->real_escape_string($_GET['anio_lectivo']) : '';
$escolaridad = isset($_GET['escolaridad']) ? $conn->real_escape_string($_GET['escolaridad']) : '';

// Verificar que el postgrado seleccionado esté entre los permitidos
if (!empty($postgrado_seleccionado) && !in_array($postgrado_seleccionado, $postgrados_permitidos_ids)) {
    $_SESSION['error'] = "No tiene permisos para acceder a este postgrado.";
    $postgrado_seleccionado = '';
}

// Configuración de paginación
$estudiantes_por_pagina = 25; // 25 estudiantes por página
$pagina_actual = isset($_GET['pagina']) ? (int)$_GET['pagina'] : 1;
if ($pagina_actual < 1) $pagina_actual = 1;

// Determinar si hay búsqueda activa
$busqueda_activa = !empty($cedula) || !empty($postgrado_seleccionado) || !empty($anio_lectivo) || !empty($escolaridad);

// Obtener años lectivos y escolaridades disponibles (ya filtrado por los postgrados permitidos)
$sql_filtros = "SELECT DISTINCT al.nombre as anio_lectivo, esc.nombre as escolaridad 
               FROM inscripciones i
               JOIN estudiante_postgrado ep ON i.estudiante_id = ep.estudiante_id
               JOIN anios_lectivos al ON i.año_lectivo_id = al.id
               JOIN escolaridades esc ON i.escolaridad_id = esc.id
               WHERE ep.postgrado_id IN ($placeholders_postgrados)";

$params_filtros = $postgrados_permitidos_ids;
$types_filtros = str_repeat('i', count($postgrados_permitidos_ids));

if (!empty($cedula)) {
    $sql_filtros .= " AND EXISTS (SELECT 1 FROM estudiantes e WHERE e.id = i.estudiante_id AND 
                    (e.cedula_venezolana LIKE ? OR e.cedula_extranjera LIKE ? OR e.pasaporte LIKE ?))";
    $params_filtros[] = "%$cedula%";
    $params_filtros[] = "%$cedula%";
    $params_filtros[] = "%$cedula%";
    $types_filtros .= "sss";
}

if (!empty($postgrado_seleccionado)) {
    $sql_filtros .= " AND ep.postgrado_id = ?";
    $params_filtros[] = $postgrado_seleccionado;
    $types_filtros .= "i";
}

$sql_filtros .= " ORDER BY al.nombre DESC, esc.nombre";

$stmt_filtros = $conn->prepare($sql_filtros);
$stmt_filtros->bind_param($types_filtros, ...$params_filtros);
$stmt_filtros->execute();
$result_filtros = $stmt_filtros->get_result();

$anios_lectivos = [];
$escolaridades = [];
while ($filtro = $result_filtros->fetch_assoc()) {
    if (!in_array($filtro['anio_lectivo'], $anios_lectivos)) $anios_lectivos[] = $filtro['anio_lectivo'];
    if (!in_array($filtro['escolaridad'], $escolaridades)) $escolaridades[] = $filtro['escolaridad'];
}

// FUNCIÓN PARA OBTENER EL NÚMERO DE DOCUMENTO SEGÚN EL TIPO
function obtener_numero_documento($tipo_documento, $cedula_venezolana, $cedula_extranjera, $pasaporte) {
    switch ($tipo_documento) {
        case 'V': return $cedula_venezolana;
        case 'E': return $cedula_extranjera;
        case 'P': return $pasaporte;
        default: return $cedula_venezolana;
    }
}

// FUNCIÓN PARA OBTENER EL TIPO DE DOCUMENTO EN TEXTO
function obtener_tipo_documento_texto($tipo_documento) {
    switch ($tipo_documento) {
        case 'V': return 'Cédula Venezolana';
        case 'E': return 'Cédula Extranjera';
        case 'P': return 'Pasaporte';
        default: return 'Cédula Venezolana';
    }
}

// CONSULTA PARA CONTAR TOTAL DE ESTUDIANTES (con filtros de coordinador)
$sql_count = "SELECT COUNT(DISTINCT e.id) as total
              FROM estudiantes e
              JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
              JOIN postgrados p ON ep.postgrado_id = p.id
              JOIN inscripciones i ON e.id = i.estudiante_id
              JOIN anios_lectivos al ON i.año_lectivo_id = al.id
              JOIN escolaridades esc ON i.escolaridad_id = esc.id
              WHERE ep.postgrado_id IN ($placeholders_postgrados) AND ep.estado = 'activo'";

$params_count = $postgrados_permitidos_ids;
$types_count = str_repeat('i', count($postgrados_permitidos_ids));

if (!empty($cedula)) {
    $sql_count .= " AND (e.cedula_venezolana LIKE ? OR e.cedula_extranjera LIKE ? OR e.pasaporte LIKE ?)";
    $params_count[] = "%$cedula%";
    $params_count[] = "%$cedula%";
    $params_count[] = "%$cedula%";
    $types_count .= "sss";
}

if (!empty($postgrado_seleccionado)) {
    $sql_count .= " AND ep.postgrado_id = ?";
    $params_count[] = $postgrado_seleccionado;
    $types_count .= "i";
}

if (!empty($anio_lectivo)) {
    $sql_count .= " AND al.nombre = ?";
    $params_count[] = $anio_lectivo;
    $types_count .= "s";
}

if (!empty($escolaridad)) {
    $sql_count .= " AND esc.nombre = ?";
    $params_count[] = $escolaridad;
    $types_count .= "s";
}

$stmt_count = $conn->prepare($sql_count);
if (!empty($params_count)) {
    $stmt_count->bind_param($types_count, ...$params_count);
}
$stmt_count->execute();
$result_count = $stmt_count->get_result();
$total_estudiantes = $result_count->fetch_assoc()['total'];
$total_paginas = ceil($total_estudiantes / $estudiantes_por_pagina);

// Ajustar página actual si es mayor al total
if ($pagina_actual > $total_paginas) $pagina_actual = $total_paginas;

// Calcular offset para la paginación
$offset = ($pagina_actual - 1) * $estudiantes_por_pagina;

// CONSULTA PARA OBTENER ESTUDIANTES (con filtros de coordinador)
$sql_estudiantes = "SELECT DISTINCT e.id, e.cedula, e.nombres, e.apellidos, 
                   e.tipo_documento, e.cedula_venezolana, e.cedula_extranjera, e.pasaporte,
                   p.nombre as postgrado_nombre, p.id as postgrado_id,
                   GROUP_CONCAT(DISTINCT esc.nombre SEPARATOR ', ') as escolaridad,
                   ep.estado as estado_postgrado
                   FROM estudiantes e
                   JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                   JOIN postgrados p ON ep.postgrado_id = p.id
                   JOIN inscripciones i ON e.id = i.estudiante_id
                   JOIN anios_lectivos al ON i.año_lectivo_id = al.id
                   JOIN escolaridades esc ON i.escolaridad_id = esc.id
                   WHERE ep.postgrado_id IN ($placeholders_postgrados) AND ep.estado = 'activo'";

$params = $postgrados_permitidos_ids;
$types = str_repeat('i', count($postgrados_permitidos_ids));

// Aplicar filtros
if (!empty($cedula)) {
    $sql_estudiantes .= " AND (e.cedula_venezolana LIKE ? OR e.cedula_extranjera LIKE ? OR e.pasaporte LIKE ?)";
    $params[] = "%$cedula%";
    $params[] = "%$cedula%";
    $params[] = "%$cedula%";
    $types .= "sss";
}

if (!empty($postgrado_seleccionado)) {
    $sql_estudiantes .= " AND ep.postgrado_id = ?";
    $params[] = $postgrado_seleccionado;
    $types .= "i";
}

if (!empty($anio_lectivo)) {
    $sql_estudiantes .= " AND al.nombre = ?";
    $params[] = $anio_lectivo;
    $types .= "s";
}

if (!empty($escolaridad)) {
    $sql_estudiantes .= " AND esc.nombre = ?";
    $params[] = $escolaridad;
    $types .= "s";
}

$sql_estudiantes .= " GROUP BY e.id, p.id ORDER BY e.apellidos, e.nombres LIMIT ? OFFSET ?";

// Agregar parámetros de paginación
$params[] = $estudiantes_por_pagina;
$params[] = $offset;
$types .= "ii";

$stmt_estudiantes = $conn->prepare($sql_estudiantes);
if (!empty($params)) {
    $stmt_estudiantes->bind_param($types, ...$params);
}
$stmt_estudiantes->execute();
$result_estudiantes = $stmt_estudiantes->get_result();
$estudiantes = $result_estudiantes->fetch_all(MYSQLI_ASSOC);

// 6. Incluir header
include '../includes/header.php';
?>

<div class="container-fluid px-3 px-md-4">
    <!-- Botón para volver -->
    <div class="mb-3 mb-md-4">
        <a href="index.php" class="btn btn-secondary btn-sm">
            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-arrow-left" viewBox="0 0 16 16">
                <path fill-rule="evenodd" d="M15 8a.5.5 0 0 0-.5-.5H2.707l3.147-3.146a.5.5 0 1 0-.708-.708l-4 4a.5.5 0 0 0 0 .708l4 4a.5.5 0 0 0 .708-.708L2.707 8.5H14.5A.5.5 0 0 0 15 8z"/>
            </svg>
            Volver al Panel
        </a>
    </div>

    <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <?= htmlspecialchars($_SESSION['success']) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
        <?php unset($_SESSION['success']); ?>
    <?php endif; ?>
    
    <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <?= htmlspecialchars($_SESSION['error']) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
        <?php unset($_SESSION['error']); ?>
    <?php endif; ?>

    <!-- Información del coordinador -->
    <div class="card mb-3 mb-md-4">
        <div class="card-header bg-info text-white py-2 py-md-3">
            <h4 class="mb-0">
                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" fill="currentColor" class="bi bi-person-badge me-2" viewBox="0 0 16 16">
                    <path d="M6.5 2a.5.5 0 0 0 0 1h3a.5.5 0 0 0 0-1h-3zM11 8a3 3 0 1 1-6 0 3 3 0 0 1 6 0z"/>
                    <path d="M4.5 0A2.5 2.5 0 0 0 2 2.5V14a2 2 0 0 0 2 2h8a2 2 0 0 0 2-2V2.5A2.5 2.5 0 0 0 11.5 0h-7zM3 2.5A1.5 1.5 0 0 1 4.5 1h7A1.5 1.5 0 0 1 13 2.5v10.795a4.2 4.2 0 0 0-.776-.492C11.392 12.387 10.063 12 8 12s-3.392.387-4.224.803a4.2 4.2 0 0 0-.776.492V2.5z"/>
                </svg>
                Información del Coordinador
            </h4>
        </div>
        <div class="card-body p-2 p-md-3">
            <div class="row">
                <div class="col-md-6">
                    <p class="mb-2"><strong>Coordinador:</strong> <?php echo htmlspecialchars($coordinador_info['nombres'] . ' ' . $coordinador_info['apellidos']); ?></p>
                    <p class="mb-2"><strong>Usuario:</strong> <?php echo htmlspecialchars($coordinador_info['username']); ?></p>
                </div>
                <div class="col-md-6">
                    <p class="mb-2"><strong>Postgrados asignados:</strong> <?php echo count($postgrados_permitidos_ids); ?></p>
                    <p class="mb-0"><strong>Total estudiantes activos:</strong> <?php echo $total_estudiantes; ?></p>
                </div>
            </div>
        </div>
    </div>

    <!-- Tarjeta de Búsqueda -->
    <div class="card mb-3 mb-md-4">
        <div class="card-header bg-primary text-white py-2 py-md-3">
            <h4 class="mb-0">Buscar Estudiantes para Cargar Notas</h4>
        </div>
        <div class="card-body p-2 p-md-3">
            <form method="get" class="mb-0">
                <input type="hidden" name="pagina" value="1">
                <div class="row g-2 g-md-3">
                    <!-- Búsqueda por número de documento -->
                    <div class="col-12 col-md-4">
                        <label for="cedula" class="form-label">Número de Documento</label>
                        <input type="text" id="cedula" name="cedula" class="form-control form-control-sm" 
                               value="<?php echo htmlspecialchars($cedula); ?>" 
                               placeholder="Ej: V12345678, E87654321, PAB123456">
                        <div class="form-text small">Buscar por cédula venezolana, extranjera o pasaporte</div>
                    </div>
                    
                    <!-- Filtro de postgrado -->
                    <div class="col-12 col-md-4">
                        <label for="postgrado_id" class="form-label">Postgrado</label>
                        <select name="postgrado_id" id="postgrado_id" class="form-select form-select-sm">
                            <option value="">Todos mis postgrados</option>
                            <?php foreach ($postgrados as $p): ?>
                                <option value="<?php echo htmlspecialchars($p['id']); ?>" 
                                    <?php echo ($p['id'] == $postgrado_seleccionado) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($p['nombre']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <!-- Filtro de año lectivo -->
                    <div class="col-6 col-md-2">
                        <label for="anio_lectivo" class="form-label">Año Lectivo</label>
                        <select id="anio_lectivo" name="anio_lectivo" class="form-select form-select-sm">
                            <option value="">Todos</option>
                            <?php foreach ($anios_lectivos as $al): ?>
                                <option value="<?php echo htmlspecialchars($al); ?>" 
                                    <?php echo ($al == $anio_lectivo) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($al); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <!-- Filtro de escolaridad -->
                    <div class="col-6 col-md-2">
                        <label for="escolaridad" class="form-label">Escolaridad</label>
                        <select id="escolaridad" name="escolaridad" class="form-select form-select-sm">
                            <option value="">Todas</option>
                            <?php foreach ($escolaridades as $esc): ?>
                                <option value="<?php echo htmlspecialchars($esc); ?>" 
                                    <?php echo ($esc == $escolaridad) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($esc); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <!-- Botones de acción -->
                    <div class="col-12">
                        <div class="d-flex flex-wrap gap-2 mt-2">
                            <button type="submit" class="btn btn-primary btn-sm flex-fill">
                                <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" fill="currentColor" class="bi bi-search me-1" viewBox="0 0 16 16">
                                    <path d="M11.742 10.344a6.5 6.5 0 1 0-1.397 1.398h-.001c.03.04.062.078.098.115l3.85 3.85a1 1 0 0 0 1.415-1.414l-3.85-3.85a1.007 1.007 0 0 0-.115-.1zM12 6.5a5.5 5.5 0 1 1-11 0 5.5 5.5 0 0 1 11 0z"/>
                                </svg>
                                Buscar Estudiantes
                            </button>
                            <a href="cargar_notas.php" class="btn btn-secondary btn-sm flex-fill">
                                <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" fill="currentColor" class="bi bi-arrow-clockwise me-1" viewBox="0 0 16 16">
                                    <path fill-rule="evenodd" d="M8 3a5 5 0 1 0 4.546 2.914.5.5 0 0 1 .908-.417A6 6 0 1 1 8 2v1z"/>
                                    <path d="M8 4.466V.534a.25.25 0 0 1 .41-.192l2.36 1.966c.12.1.12.284 0 .384L8.41 4.658A.25.25 0 0 1 8 4.466z"/>
                                </svg>
                                Limpiar Filtros
                            </a>
                        </div>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Tarjeta de Resultados -->
    <div class="card">
        <div class="card-header bg-info text-white py-2 py-md-3 d-flex justify-content-between align-items-center">
            <h4 class="mb-0">Resultados de la Búsqueda</h4>
            <span class="badge bg-light text-dark"><?= $total_estudiantes ?> estudiante(s)</span>
        </div>
        <div class="card-body p-2 p-md-3">
            <?php if ($busqueda_activa): ?>
                <div class="alert alert-info mb-2 mb-md-3 p-2">
                    <strong class="small">Filtros aplicados:</strong>
                    <ul class="mb-0 mt-1 small">
                        <?php if (!empty($cedula)): ?>
                            <li>Documento: "<?php echo htmlspecialchars($cedula); ?>"</li>
                        <?php endif; ?>
                        <?php if (!empty($postgrado_seleccionado)): ?>
                            <li>Postgrado: <?php 
                                $postgrado_nombre = '';
                                foreach ($postgrados as $p) {
                                    if ($p['id'] == $postgrado_seleccionado) {
                                        $postgrado_nombre = $p['nombre'];
                                        break;
                                    }
                                }
                                echo htmlspecialchars($postgrado_nombre);
                            ?></li>
                        <?php endif; ?>
                        <?php if (!empty($anio_lectivo)): ?>
                            <li>Año lectivo: <?php echo htmlspecialchars($anio_lectivo); ?></li>
                        <?php endif; ?>
                        <?php if (!empty($escolaridad)): ?>
                            <li>Escolaridad: <?php echo htmlspecialchars($escolaridad); ?></li>
                        <?php endif; ?>
                    </ul>
                </div>
            <?php else: ?>
                <div class="alert alert-warning mb-2 mb-md-3 p-2">
                    <strong class="small">Nota:</strong> No se han aplicado filtros. Se muestran todos los estudiantes disponibles de sus postgrados asignados.
                    Use los filtros arriba para refinar la búsqueda.
                </div>
            <?php endif; ?>

            <?php if (!empty($estudiantes)): ?>
                <div class="table-responsive-container">
                    <div class="table-responsive">
                        <table class="table table-striped table-hover table-sm mb-2">
                            <thead>
                                <tr>
                                    <th class="d-none d-md-table-cell">Tipo Documento</th>
                                    <th>Documento</th>
                                    <th>Estudiante</th>
                                    <th class="d-none d-md-table-cell">Postgrado</th>
                                    <th class="d-none d-lg-table-cell">Escolaridades</th>
                                    <th class="d-none d-sm-table-cell">Estado</th>
                                    <th>Acciones</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($estudiantes as $est): 
                                    $numero_documento = obtener_numero_documento(
                                        $est['tipo_documento'], 
                                        $est['cedula_venezolana'], 
                                        $est['cedula_extranjera'], 
                                        $est['pasaporte']
                                    );
                                    $tipo_documento_texto = obtener_tipo_documento_texto($est['tipo_documento']);
                                ?>
                                    <tr>
                                        <td class="d-none d-md-table-cell">
                                            <span class="badge bg-secondary small"><?php echo htmlspecialchars($tipo_documento_texto); ?></span>
                                        </td>
                                        <td>
                                            <span class="d-md-none small text-muted">Doc:</span>
                                            <span class="small"><?php echo htmlspecialchars($numero_documento); ?></span>
                                        </td>
                                        <td>
                                            <div class="d-flex flex-column">
                                                <span class="fw-bold"><?php echo htmlspecialchars($est['nombres'] . ' ' . $est['apellidos']); ?></span>
                                                <small class="text-muted d-md-none">
                                                    <?php echo htmlspecialchars($est['postgrado_nombre']); ?>
                                                </small>
                                                <small class="text-muted d-md-none">
                                                    <span class="badge bg-<?= 
                                                        $est['estado_postgrado'] == 'activo' ? 'success' : 
                                                        ($est['estado_postgrado'] == 'graduado' ? 'info' : 
                                                        ($est['estado_postgrado'] == 'suspendido' ? 'warning' : 
                                                        ($est['estado_postgrado'] == 'retirado' ? 'danger' : 'secondary'))) ?>">
                                                        <?php echo htmlspecialchars(ucfirst($est['estado_postgrado'])); ?>
                                                    </span>
                                                </small>
                                            </div>
                                        </td>
                                        <td class="d-none d-md-table-cell"><?php echo htmlspecialchars($est['postgrado_nombre']); ?></td>
                                        <td class="d-none d-lg-table-cell">
                                            <small><?php echo htmlspecialchars($est['escolaridad']); ?></small>
                                        </td>
                                        <td class="d-none d-sm-table-cell">
                                            <span class="badge bg-<?= 
                                                $est['estado_postgrado'] == 'activo' ? 'success' : 
                                                ($est['estado_postgrado'] == 'graduado' ? 'info' : 
                                                ($est['estado_postgrado'] == 'suspendido' ? 'warning' : 
                                                ($est['estado_postgrado'] == 'retirado' ? 'danger' : 'secondary'))) ?>">
                                                <?php echo htmlspecialchars(ucfirst($est['estado_postgrado'])); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <div class="btn-group-vertical btn-group-sm" role="group">
                                                <a href="cargar_notas_estudiante.php?estudiante_id=<?php echo htmlspecialchars($est['id']); ?>&postgrado_id=<?php echo htmlspecialchars($est['postgrado_id']); ?>&cedula=<?php echo urlencode($cedula); ?>&anio_lectivo=<?php echo urlencode($anio_lectivo); ?>&escolaridad=<?php echo urlencode($escolaridad); ?>&pagina=<?php echo $pagina_actual; ?>" 
                                                   class="btn btn-primary btn-sm">
                                                    <svg xmlns="http://www.w3.org/2000/svg" width="12" height="12" fill="currentColor" class="bi bi-pencil-square me-1" viewBox="0 0 16 16">
                                                        <path d="M15.502 1.94a.5.5 0 0 1 0 .706L14.459 3.69l-2-2L13.502.646a.5.5 0 0 1 .707 0l1.293 1.293zm-1.75 2.456-2-2L4.939 9.21a.5.5 0 0 0-.121.196l-.805 2.414a.25.25 0 0 0 .316.316l2.414-.805a.5.5 0 0 0 .196-.12l6.813-6.814z"/>
                                                        <path fill-rule="evenodd" d="M1 13.5A1.5 1.5 0 0 0 2.5 15h11a1.5 1.5 0 0 0 1.5-1.5v-6a.5.5 0 0 0-1 0v6a.5.5 0 0 1-.5.5h-11a.5.5 0 0 1-.5-.5v-11a.5.5 0 0 1 .5-.5H9a.5.5 0 0 0 0-1H2.5A1.5 1.5 0 0 0 1 2.5v11z"/>
                                                    </svg>
                                                    Cargar Notas
                                                </a>
                                                <button type="button" class="btn btn-outline-info btn-sm d-md-none" 
                                                        data-bs-toggle="popover" 
                                                        data-bs-placement="left"
                                                        data-bs-title="Información del Estudiante"
                                                        data-bs-content="
                                                            <strong>Documento:</strong> <?php echo htmlspecialchars($numero_documento); ?><br>
                                                            <strong>Tipo:</strong> <?php echo htmlspecialchars($tipo_documento_texto); ?><br>
                                                            <strong>Postgrado:</strong> <?php echo htmlspecialchars($est['postgrado_nombre']); ?><br>
                                                            <strong>Escolaridades:</strong> <?php echo htmlspecialchars($est['escolaridad']); ?>
                                                        ">
                                                    <svg xmlns="http://www.w3.org/2000/svg" width="12" height="12" fill="currentColor" class="bi bi-info-circle" viewBox="0 0 16 16">
                                                        <path d="M8 15A7 7 0 1 1 8 1a7 7 0 0 1 0 14zm0 1A8 8 0 1 0 8 0a8 8 0 0 0 0 16z"/>
                                                        <path d="m8.93 6.588-2.29.287-.082.38.45.083c.294.07.352.176.288.469l-.738 3.468c-.194.897.105 1.319.808 1.319.545 0 1.178-.252 1.465-.598l.088-.416c-.2.176-.492.246-.686.246-.275 0-.375-.193-.304-.533L8.93 6.588zM9 4.5a1 1 0 1 1-2 0 1 1 0 0 1 2 0z"/>
                                                    </svg>
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
                
                <!-- PAGINACIÓN -->
                <?php if ($total_paginas > 1): ?>
                <nav aria-label="Paginación de estudiantes">
                    <ul class="pagination justify-content-center pagination-sm mt-2 mt-md-4">
                        <!-- Botón Anterior -->
                        <li class="page-item <?= $pagina_actual == 1 ? 'disabled' : '' ?>">
                            <a class="page-link" href="?pagina=<?= $pagina_actual - 1 ?>&cedula=<?= urlencode($cedula) ?>&postgrado_id=<?= $postgrado_seleccionado ?>&anio_lectivo=<?= urlencode($anio_lectivo) ?>&escolaridad=<?= urlencode($escolaridad) ?>" aria-label="Anterior">
                                <span aria-hidden="true">&laquo;</span>
                            </a>
                        </li>
                        
                        <!-- Páginas -->
                        <?php
                        $inicio = max(1, $pagina_actual - 2);
                        $fin = min($total_paginas, $pagina_actual + 2);
                        
                        if ($inicio > 1) {
                            echo '<li class="page-item"><a class="page-link" href="?pagina=1&cedula='.urlencode($cedula).'&postgrado_id='.$postgrado_seleccionado.'&anio_lectivo='.urlencode($anio_lectivo).'&escolaridad='.urlencode($escolaridad).'">1</a></li>';
                            if ($inicio > 2) echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                        }
                        
                        for ($i = $inicio; $i <= $fin; $i++): ?>
                            <li class="page-item <?= $i == $pagina_actual ? 'active' : '' ?>">
                                <a class="page-link" href="?pagina=<?= $i ?>&cedula=<?= urlencode($cedula) ?>&postgrado_id=<?= $postgrado_seleccionado ?>&anio_lectivo=<?= urlencode($anio_lectivo) ?>&escolaridad=<?= urlencode($escolaridad) ?>"><?= $i ?></a>
                            </li>
                        <?php endfor;
                        
                        if ($fin < $total_paginas) {
                            if ($fin < $total_paginas - 1) echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                            echo '<li class="page-item"><a class="page-link" href="?pagina='.$total_paginas.'&cedula='.urlencode($cedula).'&postgrado_id='.$postgrado_seleccionado.'&anio_lectivo='.urlencode($anio_lectivo).'&escolaridad='.urlencode($escolaridad).'">'.$total_paginas.'</a></li>';
                        }
                        ?>
                        
                        <!-- Botón Siguiente -->
                        <li class="page-item <?= $pagina_actual == $total_paginas ? 'disabled' : '' ?>">
                            <a class="page-link" href="?pagina=<?= $pagina_actual + 1 ?>&cedula=<?= urlencode($cedula) ?>&postgrado_id=<?= $postgrado_seleccionado ?>&anio_lectivo=<?= urlencode($anio_lectivo) ?>&escolaridad=<?= urlencode($escolaridad) ?>" aria-label="Siguiente">
                                <span aria-hidden="true">&raquo;</span>
                            </a>
                        </li>
                    </ul>
                </nav>
                <?php endif; ?>
                
                <div class="mt-2 text-muted text-center">
                    <small>Mostrando <?= count($estudiantes) ?> de <?= $total_estudiantes ?> estudiante(s) - Página <?= $pagina_actual ?> de <?= $total_paginas ?></small>
                </div>
                
            <?php else: ?>
                <div class="text-center py-4">
                    <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" fill="currentColor" class="bi bi-search text-muted mb-2" viewBox="0 0 16 16">
                        <path d="M11.742 10.344a6.5 6.5 0 1 0-1.397 1.398h-.001c.03.04.062.078.098.115l3.85 3.85a1 1 0 0 0 1.415-1.414l-3.85-3.85a1.007 1.007 0 0 0-.115-.1zM12 6.5a5.5 5.5 0 1 1-11 0 5.5 5.5 0 0 1 11 0z"/>
                    </svg>
                    <h5 class="text-muted">No se encontraron estudiantes</h5>
                    <p class="text-muted small">
                        <?php if ($busqueda_activa): ?>
                            No hay estudiantes que coincidan con los criterios de búsqueda.<br>
                            Intente ajustar los filtros o buscar con criterios diferentes.
                        <?php else: ?>
                            No hay estudiantes registrados en sus postgrados asignados.
                        <?php endif; ?>
                    </p>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<style>
    .container-fluid {
        max-width: 100%;
        overflow-x: hidden;
    }
    
    .card {
        border: 1px solid #dee2e6;
        border-radius: 8px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        margin-bottom: 15px;
    }
    
    .card-header {
        border-radius: 8px 8px 0 0 !important;
        font-weight: 600;
    }
    
    .table-responsive-container {
        position: relative;
        width: 100%;
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
        border: 1px solid #dee2e6;
        border-radius: 6px;
    }
    
    .table-responsive {
        min-width: 800px;
        margin-bottom: 0;
    }
    
    .table th {
        background-color: #2c3e50;
        color: white;
        font-weight: 600;
        white-space: nowrap;
        font-size: 0.875rem;
        padding: 8px 12px;
    }
    
    .table td {
        vertical-align: middle;
        padding: 8px 12px;
        font-size: 0.875rem;
    }
    
    .btn-primary {
        background-color: #007bff;
        border-color: #007bff;
        font-size: 0.8rem;
    }
    
    .btn-primary:hover {
        background-color: #0056b3;
        border-color: #0056b3;
    }
    
    .badge {
        font-size: 0.7em;
        font-weight: 600;
    }
    
    .form-label {
        font-weight: 500;
        color: #2c3e50;
        font-size: 0.9rem;
    }
    
    .alert {
        border: none;
        border-radius: 6px;
        font-size: 0.875rem;
    }
    
    .pagination .page-item.active .page-link {
        background-color: #2c3e50;
        border-color: #2c3e50;
        color: white !important;
    }
    
    .pagination .page-link {
        color: #2c3e50;
        font-size: 0.875rem;
        padding: 0.375rem 0.75rem;
    }
    
    .pagination .page-link:hover {
        color: #1a252f;
        background-color: #e9ecef;
    }
    
    .pagination .page-item.disabled .page-link {
        color: #6c757d;
    }
    
    /* Mejoras específicas para móviles */
    @media (max-width: 768px) {
        .container-fluid {
            padding-left: 10px;
            padding-right: 10px;
        }
        
        .card-header h4 {
            font-size: 1.1rem;
        }
        
        .table-responsive {
            min-width: 600px;
        }
        
        .btn-group-vertical .btn {
            margin-bottom: 2px;
        }
        
        .form-control-sm, .form-select-sm {
            font-size: 0.875rem;
        }
    }
    
    @media (max-width: 576px) {
        .container-fluid {
            padding-left: 8px;
            padding-right: 8px;
        }
        
        .card-body {
            padding: 12px;
        }
        
        .table-responsive {
            min-width: 500px;
        }
        
        .btn-sm {
            padding: 0.25rem 0.5rem;
            font-size: 0.75rem;
        }
        
        .badge {
            font-size: 0.65em;
        }
        
        .pagination .page-link {
            padding: 0.25rem 0.5rem;
            font-size: 0.8rem;
        }
    }
    
    /* Scroll horizontal suave */
    .table-responsive-container::-webkit-scrollbar {
        height: 8px;
    }
    
    .table-responsive-container::-webkit-scrollbar-track {
        background: #f1f1f1;
        border-radius: 4px;
    }
    
    .table-responsive-container::-webkit-scrollbar-thumb {
        background: #c1c1c1;
        border-radius: 4px;
    }
    
    .table-responsive-container::-webkit-scrollbar-thumb:hover {
        background: #a8a8a8;
    }
</style>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    const cedulaInput = document.getElementById('cedula');
    
    // Focus automático en el campo de documento
    if (cedulaInput && cedulaInput.value === '') {
        cedulaInput.focus();
    }
    
    // Inicializar popovers de Bootstrap
    const popoverTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="popover"]'))
    const popoverList = popoverTriggerList.map(function (popoverTriggerEl) {
        return new bootstrap.Popover(popoverTriggerEl, {
            html: true,
            sanitize: false
        })
    });
    
    // Mejorar la experiencia táctil en móviles
    const tableContainer = document.querySelector('.table-responsive-container');
    if (tableContainer) {
        let startX;
        let scrollLeft;
        
        tableContainer.addEventListener('touchstart', (e) => {
            startX = e.touches[0].pageX - tableContainer.offsetLeft;
            scrollLeft = tableContainer.scrollLeft;
        });
        
        tableContainer.addEventListener('touchmove', (e) => {
            if (!startX) return;
            const x = e.touches[0].pageX - tableContainer.offsetLeft;
            const walk = (x - startX) * 2;
            tableContainer.scrollLeft = scrollLeft - walk;
        });
    }
});
</script>

<?php
// 7. Incluir footer
include '../includes/footer.php';
$conn->close();
?>