<?php
// Centralized authentication and session check
$allowed_roles = ['administrador', 'coordinador1', 'coordinador2', 'secretaria'];
require_once __DIR__ . '/../includes/auth_check.php';

// DB Connection
require_once __DIR__ . '/../includes/db.php';

// Configurar conexión para usar UTF-8
$conn->set_charset("utf8mb4");
$conn->query("SET NAMES 'utf8mb4'");
$conn->query("SET CHARACTER SET utf8mb4");
$conn->query("SET COLLATION_CONNECTION = 'utf8mb4_unicode_ci'");

// Variables
$mensaje = '';
$tipo_mensaje = '';

// Obtener ID de inscripción desde GET
$inscripcion_id = isset($_GET['inscripcion']) ? intval($_GET['inscripcion']) : 0;

if ($inscripcion_id <= 0) {
    $_SESSION['error'] = "ID de inscripción inválido. Complete el proceso de inscripción desde el principio.";
    header("Location: gestion_inscripciones.php");
    exit();
}

// Obtener información de la inscripción
$sql_inscripcion = "SELECT ep.id, ep.estudiante_id, ep.postgrado_id, e.nombres, e.apellidos, p.nombre as postgrado_nombre, 
                   p.tipo_postgrado, p.creditos_totales, p.creditos_obligatorios, p.creditos_electivos, ep.anio_inicio
                   FROM estudiante_postgrado ep
                   JOIN estudiantes e ON ep.estudiante_id = e.id
                   JOIN postgrados p ON ep.postgrado_id = p.id
                   WHERE ep.id = ?";
$stmt_inscripcion = $conn->prepare($sql_inscripcion);
$stmt_inscripcion->bind_param("i", $inscripcion_id);
$stmt_inscripcion->execute();
$result_inscripcion = $stmt_inscripcion->get_result();
$inscripcion = $result_inscripcion->fetch_assoc();

if (!$inscripcion) {
    $_SESSION['error'] = "Inscripción no encontrada";
    header("Location: gestion_inscripciones.php");
    exit();
}

// Verificar que esta inscripción necesita selección de electivas
if ($inscripcion['tipo_postgrado'] == 'tradicional') {
    $_SESSION['error'] = "Este postgrado no requiere selección de electivas";
    header("Location: gestion_inscripciones.php");
    exit();
}

// Procesar selección de electivas
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['seleccionar_electivas'])) {
    $electivas_seleccionadas = isset($_POST['electivas']) ? $_POST['electivas'] : [];
    $fecha_inscripcion = date('Y-m-d');
    $anio_inicio = $inscripcion['anio_inicio'];
    
    // Validar créditos
    $total_creditos_seleccionados = 0;
    foreach ($electivas_seleccionadas as $denominacion_id) {
        $sql_credito = "SELECT credito, periodo FROM denominaciones WHERE id = ?";
        $stmt_credito = $conn->prepare($sql_credito);
        $stmt_credito->bind_param("i", $denominacion_id);
        $stmt_credito->execute();
        $result_credito = $stmt_credito->get_result();
        $denominacion = $result_credito->fetch_assoc();
        $total_creditos_seleccionados += $denominacion['credito'];
    }
    
    // Validar límites de créditos
    if ($total_creditos_seleccionados < $inscripcion['creditos_electivos']) {
        $_SESSION['error'] = "Debe seleccionar al menos " . $inscripcion['creditos_electivos'] . " créditos electivos. Actualmente ha seleccionado $total_creditos_seleccionados créditos.";
        header("Location: seleccion_electivas.php?inscripcion=" . $inscripcion_id);
        exit();
    }
    
    // Iniciar transacción
    $conn->begin_transaction();
    
    try {
        // Inscribir cada electiva seleccionada
        foreach ($electivas_seleccionadas as $denominacion_id) {
            // Obtener información de la denominación
            $sql_denominacion = "SELECT credito, periodo FROM denominaciones WHERE id = ?";
            $stmt_denominacion = $conn->prepare($sql_denominacion);
            $stmt_denominacion->bind_param("i", $denominacion_id);
            $stmt_denominacion->execute();
            $denominacion = $stmt_denominacion->get_result()->fetch_assoc();
            
            // Calcular año lectivo basado en el período de la denominación
            $periodo = intval($denominacion['periodo']);
            $anios_adicionales = floor(($periodo - 1) / 3);
            $trimestre = (($periodo - 1) % 3) + 1;
            $anio_actual = $anio_inicio + $anios_adicionales;
            $escolaridad = $anio_actual;
            $anio_lectivo = $anio_actual . '-' . str_pad($trimestre, 2, '0', STR_PAD_LEFT) . 'C';
            
            // Obtener o crear escolaridad
            $escolaridad_id = obtener_o_crear($conn, 'escolaridades', 'nombre', $escolaridad);
            
            // Obtener o crear año lectivo
            $anio_lectivo_id = obtener_o_crear($conn, 'anios_lectivos', 'nombre', $anio_lectivo);
            
            // Obtener sección por defecto
            $seccion_id = $conn->query("SELECT id FROM secciones LIMIT 1")->fetch_assoc()['id'];
            
            // Crear inscripción a la denominación
            $sql_insc_denom = "INSERT INTO inscripciones 
                              (estudiante_id, denominacion_id, escolaridad_id, año_lectivo_id, seccion_id, fecha_inscripcion, estado) 
                              VALUES (?, ?, ?, ?, ?, ?, 'activa')";
            $stmt_insc_denom = $conn->prepare($sql_insc_denom);
            $stmt_insc_denom->bind_param("iiiiis", $inscripcion['estudiante_id'], $denominacion_id, $escolaridad_id, $anio_lectivo_id, $seccion_id, $fecha_inscripcion);
            $stmt_insc_denom->execute();
        }
        
        $conn->commit();
        $_SESSION['success'] = "Electivas seleccionadas correctamente. Total de créditos inscritos: $total_creditos_seleccionados";
        header("Location: gestion_inscripciones.php");
        exit();
        
    } catch (Exception $e) {
        $conn->rollback();
        $_SESSION['error'] = "Error al inscribir las electivas: " . $e->getMessage();
        header("Location: seleccion_electivas.php?inscripcion=" . $inscripcion_id);
        exit();
    }
}

// Obtener electivas disponibles para este postgrado
$sql_electivas = "SELECT d.id, d.codigo, d.nombre, d.credito, d.periodo
                  FROM denominaciones d
                  WHERE d.postgrado_id = ? AND d.tipo_denominacion = 'electiva' AND d.activa = TRUE
                  ORDER BY d.periodo, d.nombre";
$stmt_electivas = $conn->prepare($sql_electivas);
$stmt_electivas->bind_param("i", $inscripcion['postgrado_id']);
$stmt_electivas->execute();
$result_electivas = $stmt_electivas->get_result();
$electivas = $result_electivas->fetch_all(MYSQLI_ASSOC);

// Obtener electivas ya inscritas por el estudiante
$sql_inscritas = "SELECT i.denominacion_id 
                  FROM inscripciones i
                  JOIN denominaciones d ON i.denominacion_id = d.id
                  WHERE i.estudiante_id = ? AND d.postgrado_id = ? AND d.tipo_denominacion = 'electiva'";
$stmt_inscritas = $conn->prepare($sql_inscritas);
$stmt_inscritas->bind_param("ii", $inscripcion['estudiante_id'], $inscripcion['postgrado_id']);
$stmt_inscritas->execute();
$result_inscritas = $stmt_inscritas->get_result();
$electivas_inscritas = [];
while ($row = $result_inscritas->fetch_assoc()) {
    $electivas_inscritas[] = $row['denominacion_id'];
}

// Función auxiliar para obtener o crear registros
function obtener_o_crear($conn, $tabla, $campo, $valor) {
    $sql = "SELECT id FROM $tabla WHERE $campo = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("s", $valor);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        return $result->fetch_assoc()['id'];
    } else {
        $sql_insert = "INSERT INTO $tabla ($campo) VALUES (?)";
        $stmt_insert = $conn->prepare($sql_insert);
        $stmt_insert->bind_param("s", $valor);
        $stmt_insert->execute();
        return $stmt_insert->insert_id;
    }
}

$page_title = "Selección de Electivas";
include '../includes/header.php';
?>

<div class="container">
    <!-- Botón para volver -->
    <div class="mb-4">
        <a href="gestion_inscripciones.php" class="btn btn-secondary">
            <i class="bi bi-arrow-left"></i> Volver a Inscripciones
        </a>
    </div>

    <h2 class="my-4">Selección de Asignaturas Electivas</h2>
    
    <!-- Mostrar mensajes -->
    <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger"><?= $_SESSION['error'] ?></div>
        <?php unset($_SESSION['error']); ?>
    <?php endif; ?>
    
    <!-- Información del estudiante y postgrado -->
    <div class="card mb-4">
        <div class="card-header bg-info text-white">
            <h3 class="mb-0">Información de la Inscripción</h3>
        </div>
        <div class="card-body">
            <div class="row">
                <div class="col-md-6">
                    <p><strong>Estudiante:</strong> <?= htmlspecialchars($inscripcion['nombres'] . ' ' . $inscripcion['apellidos']) ?></p>
                    <p><strong>Postgrado:</strong> <?= htmlspecialchars($inscripcion['postgrado_nombre']) ?></p>
                </div>
                <div class="col-md-6">
                    <p><strong>Tipo de Postgrado:</strong> <?= ucfirst($inscripcion['tipo_postgrado']) ?></p>
                    <p><strong>Créditos Electivos Requeridos:</strong> <?= $inscripcion['creditos_electivos'] ?></p>
                    <p><strong>Año de Inicio:</strong> <?= $inscripcion['anio_inicio'] ?></p>
                </div>
            </div>
        </div>
    </div>

    <!-- Formulario de selección de electivas -->
    <div class="card">
        <div class="card-header bg-primary text-white">
            <h3 class="mb-0">Asignaturas Electivas Disponibles</h3>
        </div>
        <div class="card-body">
            <?php if (empty($electivas)): ?>
                <div class="alert alert-warning">
                    No hay asignaturas electivas disponibles para este postgrado.
                </div>
            <?php else: ?>
                <form method="post" id="form-electivas">
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead class="table-dark">
                                <tr>
                                    <th>Seleccionar</th>
                                    <th>Código</th>
                                    <th>Nombre</th>
                                    <th>Créditos</th>
                                    <th>Período</th>
                                    <th>Año Lectivo Calculado</th>
                                    <th>Estado</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php 
                                $total_creditos_seleccionados = 0;
                                foreach ($electivas as $electiva): 
                                    $ya_inscrita = in_array($electiva['id'], $electivas_inscritas);
                                    
                                    // Calcular año lectivo automáticamente
                                    $periodo = intval($electiva['periodo']);
                                    $anios_adicionales = floor(($periodo - 1) / 3);
                                    $trimestre = (($periodo - 1) % 3) + 1;
                                    $anio_actual = $inscripcion['anio_inicio'] + $anios_adicionales;
                                    $anio_lectivo = $anio_actual . '-' . str_pad($trimestre, 2, '0', STR_PAD_LEFT) . 'C';
                                ?>
                                    <tr>
                                        <td>
                                            <input type="checkbox" name="electivas[]" value="<?= $electiva['id'] ?>" 
                                                   class="electiva-checkbox" 
                                                   <?= $ya_inscrita ? 'checked disabled' : '' ?>
                                                   data-creditos="<?= $electiva['credito'] ?>">
                                        </td>
                                        <td><?= htmlspecialchars($electiva['codigo']) ?></td>
                                        <td><?= htmlspecialchars($electiva['nombre']) ?></td>
                                        <td><?= $electiva['credito'] ?></td>
                                        <td><?= $electiva['periodo'] ?></td>
                                        <td><?= $anio_lectivo ?></td>
                                        <td>
                                            <?php if ($ya_inscrita): ?>
                                                <span class="badge bg-success">Inscrita</span>
                                            <?php else: ?>
                                                <span class="badge bg-secondary">Disponible</span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                    <?php if ($ya_inscrita) $total_creditos_seleccionados += $electiva['credito']; ?>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <div class="mt-4 p-3 border rounded bg-light">
                        <div class="row">
                            <div class="col-md-6">
                                <h5>Resumen de Selección</h5>
                                <p><strong>Créditos requeridos:</strong> <?= $inscripcion['creditos_electivos'] ?></p>
                                <p><strong>Créditos seleccionados:</strong> <span id="creditos-seleccionados"><?= $total_creditos_seleccionados ?></span></p>
                                <p><strong>Estado:</strong> <span id="estado-seleccion" class="badge bg-danger">No cumple</span></p>
                            </div>
                            <div class="col-md-6">
                                <div class="d-grid">
                                    <button type="submit" name="seleccionar_electivas" class="btn btn-primary btn-lg" id="btn-guardar" disabled>
                                        <i class="bi bi-save"></i> Guardar Selección
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </form>
            <?php endif; ?>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const checkboxes = document.querySelectorAll('.electiva-checkbox');
    const creditosSeleccionados = document.getElementById('creditos-seleccionados');
    const estadoSeleccion = document.getElementById('estado-seleccion');
    const btnGuardar = document.getElementById('btn-guardar');
    const creditosRequeridos = <?= $inscripcion['creditos_electivos'] ?>;
    
    function actualizarResumen() {
        let total = <?= $total_creditos_seleccionados ?>;
        
        checkboxes.forEach(checkbox => {
            if (checkbox.checked && !checkbox.disabled) {
                total += parseInt(checkbox.dataset.creditos);
            }
        });
        
        creditosSeleccionados.textContent = total;
        
        if (total >= creditosRequeridos) {
            estadoSeleccion.textContent = 'Cumple';
            estadoSeleccion.className = 'badge bg-success';
            btnGuardar.disabled = false;
        } else {
            estadoSeleccion.textContent = 'No cumple';
            estadoSeleccion.className = 'badge bg-danger';
            btnGuardar.disabled = true;
        }
    }
    
    checkboxes.forEach(checkbox => {
        checkbox.addEventListener('change', actualizarResumen);
    });
    
    // Inicializar el resumen
    actualizarResumen();
});
</script>

<?php
include '../includes/footer.php';
?>