<?php
session_start();

// Configurar encoding UTF-8
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');

if (!isset($_SESSION['username']) || $_SESSION['rol'] != 'administrador') {
    header("Location: ../login.php");
    exit();
}

include '../includes/db.php';

// Configurar conexión para usar UTF-8
$conn->set_charset("utf8mb4");
$conn->query("SET NAMES 'utf8mb4'");
$conn->query("SET CHARACTER SET utf8mb4");
$conn->query("SET COLLATION_CONNECTION = 'utf8mb4_unicode_ci'");

// Procesar la asignación de escolaridades
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['asignar_escolaridades'])) {
        $coordinador_id = intval($_POST['coordinador_id']);
        $postgrado_id = intval($_POST['postgrado_id']);
        $escolaridades = isset($_POST['escolaridades']) ? $_POST['escolaridades'] : [];

        if ($coordinador_id > 0 && $postgrado_id > 0) {
            $conn->begin_transaction();
            try {
                // Eliminar asignaciones existentes para este coordinador y postgrado
                $delete_sql = "DELETE FROM coordinador_postgrado_escolaridad WHERE coordinador_id = ? AND postgrado_id = ?";
                $stmt = $conn->prepare($delete_sql);
                $stmt->bind_param("ii", $coordinador_id, $postgrado_id);
                $stmt->execute();

                // Insertar nuevas asignaciones
                if (!empty($escolaridades)) {
                    $insert_sql = "INSERT INTO coordinador_postgrado_escolaridad (coordinador_id, postgrado_id, escolaridad_id) VALUES (?, ?, ?)";
                    $stmt = $conn->prepare($insert_sql);
                    foreach ($escolaridades as $escolaridad_id) {
                        $escolaridad_id = intval($escolaridad_id);
                        $stmt->bind_param("iii", $coordinador_id, $postgrado_id, $escolaridad_id);
                        $stmt->execute();
                    }
                }
                $conn->commit();
                $_SESSION['success'] = "Escolaridades asignadas correctamente.";
            } catch (Exception $e) {
                $conn->rollback();
                $_SESSION['error'] = "Error al asignar escolaridades: " . $e->getMessage();
            }
        } else {
            $_SESSION['error'] = "Debe seleccionar un coordinador y un postgrado válidos.";
        }
        
        header("Location: habilitar_escolaridad.php");
        exit();
    }
}

// Obtener todos los coordinadores con rol 'coordinador1'
$sql_coordinadores = "SELECT id, username FROM usuarios WHERE rol = 'coordinador1' ORDER BY username";
$coordinadores = $conn->query($sql_coordinadores);

// Obtener todas las asignaciones existentes para mostrar en la tabla
$sql_existing_assignments = "SELECT 
                                cpe.coordinador_id, u.username AS coordinador_nombre,
                                cpe.postgrado_id, p.nombre AS postgrado_nombre,
                                cpe.escolaridad_id, e.nombre AS escolaridad_nombre,
                                GROUP_CONCAT(DISTINCT al.nombre ORDER BY al.nombre SEPARATOR ', ') AS años_lectivos
                            FROM coordinador_postgrado_escolaridad cpe
                            JOIN usuarios u ON cpe.coordinador_id = u.id
                            JOIN postgrados p ON cpe.postgrado_id = p.id
                            JOIN escolaridades e ON cpe.escolaridad_id = e.id
                            LEFT JOIN años_lectivos al ON al.nombre LIKE CONCAT(e.nombre, '-%')
                            GROUP BY cpe.coordinador_id, cpe.postgrado_id, cpe.escolaridad_id
                            ORDER BY u.username, p.nombre, e.nombre";
$existing_assignments_result = $conn->query($sql_existing_assignments);

$page_title = "Asignar Escolaridad a Coordinador 1";
include '../includes/header.php';
?>

<div class="container mt-4">
    <div class="mb-4">
        <a href="index.php" class="btn btn-secondary">
            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-arrow-left" viewBox="0 0 16 16">
                <path fill-rule="evenodd" d="M15 8a.5.5 0 0 0-.5-.5H2.707l3.147-3.146a.5.5 0 1 0-.708-.708l-4 4a.5.5 0 0 0 0 .708l4 4a.5.5 0 0 0 .708-.708L2.707 8.5H14.5A.5.5 0 0 0 15 8z"/>
            </svg>
            Volver al Panel
        </a>
    </div>

    <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <?= htmlspecialchars($_SESSION['success']) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
        <?php unset($_SESSION['success']); ?>
    <?php endif; ?>
    
    <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <?= htmlspecialchars($_SESSION['error']) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
        <?php unset($_SESSION['error']); ?>
    <?php endif; ?>

    <div class="card">
        <div class="card-header bg-primary text-white">
            <h3 class="mb-0">Asignar Escolaridad a Coordinador</h3>
        </div>
        <div class="card-body">
            <form method="post" id="asignacionEscolaridadForm">
                <div class="row g-3">
                    <div class="col-md-6">
                        <label for="coordinador_id" class="form-label fw-bold">1. Seleccionar Coordinador:</label>
                        <select name="coordinador_id" id="coordinador_id" class="form-select" required>
                            <option value="">-- Seleccione un coordinador --</option>
                            <?php while ($coordinador = $coordinadores->fetch_assoc()): ?>
                                <option value="<?= $coordinador['id'] ?>"><?= htmlspecialchars($coordinador['username']) ?></option>
                            <?php endwhile; ?>
                        </select>
                    </div>
                    <div class="col-md-6">
                        <label for="postgrado_id" class="form-label fw-bold">2. Seleccionar Postgrado Asignado:</label>
                        <select name="postgrado_id" id="postgrado_id" class="form-select" required disabled>
                            <option value="">-- Primero seleccione un coordinador --</option>
                        </select>
                    </div>
                </div>

                <div class="mt-4">
                    <label class="form-label fw-bold">3. Escolaridades Disponibles para el Postgrado:</label>
                    <div id="escolaridades_container" class="border p-3 rounded" style="min-height: 100px; max-height: 250px; overflow-y: auto;">
                        <p class="text-muted">Seleccione un coordinador y un postgrado para ver las escolaridades.</p>
                    </div>
                </div>

                <div class="mt-4">
                    <button type="submit" name="asignar_escolaridades" class="btn btn-primary">
                        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-save" viewBox="0 0 16 16">
                            <path d="M2 1a1 1 0 0 0-1 1v12a1 1 0 0 0 1 1h12a1 1 0 0 0 1-1V2a1 1 0 0 0-1-1H9.5a1 1 0 0 0-1 1v7.293l2.646-2.647a.5.5 0 0 1 .708.708l-3.5 3.5a.5.5 0 0 1-.708 0l-3.5-3.5a.5.5 0 1 1 .708-.708L7.5 9.293V2a2 2 0 0 1 2-2H14a2 2 0 0 1 2 2v12a2 2 0 0 1-2 2H2a2 2 0 0 1-2-2V2a2 2 0 0 1 2-2h2.5a.5.5 0 0 1 0 1H2z"/>
                        </svg>
                        Guardar Asignaciones
                    </button>
                </div>
            </form>
        </div>
    </div>

    <div class="card mt-4">
        <div class="card-header bg-info text-white">
            <h3 class="mb-0">Asignaciones de Escolaridad Existentes</h3>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-striped table-hover">
                    <thead>
                        <tr>
                            <th>Coordinador</th>
                            <th>Postgrado</th>
                            <th>Escolaridad</th>
                            <th>Años Lectivos</th>
                            <th class="text-center">Acciones</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if ($existing_assignments_result->num_rows > 0): ?>
                            <?php while ($assignment = $existing_assignments_result->fetch_assoc()): ?>
                                <tr>
                                    <td><?= htmlspecialchars($assignment['coordinador_nombre']) ?></td>
                                    <td><?= htmlspecialchars($assignment['postgrado_nombre']) ?></td>
                                    <td><?= htmlspecialchars($assignment['escolaridad_nombre']) ?></td>
                                    <td><?= $assignment['años_lectivos'] ? htmlspecialchars($assignment['años_lectivos']) : 'N/A' ?></td>
                                    <td class="text-center">
                                        <a href='#' class="btn btn-sm btn-danger delete-assignment-btn" 
                                           data-coordinador-id="<?= $assignment['coordinador_id'] ?>"
                                           data-postgrado-id="<?= $assignment['postgrado_id'] ?>"
                                           data-escolaridad-id="<?= $assignment['escolaridad_id'] ?>">
                                            Eliminar
                                        </a>
                                    </td>
                                </tr>
                            <?php endwhile; ?>
                        <?php else: ?>
                            <tr>
                                <td colspan="5" class="text-center">No hay asignaciones de escolaridad registradas.</td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script>
$(document).ready(function() {
    // --- Cargar postgrados al cambiar de coordinador ---
    $('#coordinador_id').change(function() {
        var coordinadorId = $(this).val();
        var postgradoSelect = $('#postgrado_id');
        var escolaridadesContainer = $('#escolaridades_container');

        postgradoSelect.html('<option value="">-- Primero seleccione un coordinador --</option>').prop('disabled', true);
        escolaridadesContainer.html('<p class="text-muted">Seleccione un coordinador y un postgrado para ver las escolaridades.</p>');

        if (coordinadorId) {
            $.ajax({
                url: 'api/get_postgrados_by_coordinador.php',
                type: 'GET',
                data: { coordinador_id: coordinadorId, rol: 'coordinador1' },
                dataType: 'json',
                success: function(data) {
                    postgradoSelect.html('<option value="">-- Seleccione un postgrado --</option>');
                    if (data.length > 0) {
                        $.each(data, function(i, postgrado) {
                            postgradoSelect.append($('<option>', {
                                value: postgrado.id,
                                text: postgrado.nombre
                            }));
                        });
                        postgradoSelect.prop('disabled', false);
                    } else {
                        postgradoSelect.html('<option value="">-- No hay postgrados asignados --</option>');
                    }
                },
                error: function() {
                    alert("Error al cargar los postgrados del coordinador.");
                }
            });
        }
    });

    // --- Cargar escolaridades al cambiar de postgrado ---
    $('#postgrado_id').change(function() {
        var coordinadorId = $('#coordinador_id').val();
        var postgradoId = $(this).val();
        var escolaridadesContainer = $('#escolaridades_container');

        escolaridadesContainer.html('<p class="text-muted">Cargando...</p>');

        if (coordinadorId && postgradoId) {
            $.ajax({
                url: 'api/get_escolaridades_for_assignment.php',
                type: 'GET',
                data: { 
                    coordinador_id: coordinadorId, 
                    postgrado_id: postgradoId,
                    incluir_años_lectivos: true
                },
                dataType: 'json',
                success: function(data) {
                    escolaridadesContainer.empty();
                    if (data.all_escolaridades.length > 0) {
                        $.each(data.all_escolaridades, function(i, escolaridad) {
                            var isChecked = data.assigned_escolaridad_ids.includes(escolaridad.id);
                            var checkbox = '<div class="form-check mb-2">' +
                                '<input class="form-check-input" type="checkbox" name="escolaridades[]" value="' + escolaridad.id + '" id="escolaridad_' + escolaridad.id + '"' + (isChecked ? ' checked' : '') + '>' +
                                '<label class="form-check-label" for="escolaridad_' + escolaridad.id + '">' + 
                                escolaridad.nombre + 
                                (escolaridad.años_lectivos ? ' <small class="text-muted">(' + escolaridad.años_lectivos.join(', ') + ')</small>' : '') +
                                '</label>' +
                                '</div>';
                            escolaridadesContainer.append(checkbox);
                        });
                    } else {
                        escolaridadesContainer.html('<p class="text-muted">No hay escolaridades disponibles para este postgrado.</p>');
                    }
                },
                error: function() {
                    alert("Error al cargar las escolaridades.");
                }
            });
        } else {
            escolaridadesContainer.html('<p class="text-muted">Seleccione un coordinador y un postgrado para ver las escolaridades.</p>');
        }
    });

    // --- Manejar el clic en el botón de eliminar ---
    $('.delete-assignment-btn').click(function(e) {
        e.preventDefault();

        if (confirm("¿Está seguro de que desea eliminar esta asignación?")) {
            var $thisButton = $(this);
            var coordinadorId = $thisButton.data('coordinador-id');
            var postgradoId = $thisButton.data('postgrado-id');
            var escolaridadId = $thisButton.data('escolaridad-id');
            var $row = $thisButton.closest('tr');

            $.ajax({
                url: 'api/delete_coordinador_escolaridad.php',
                type: 'POST',
                data: { 
                    coordinador_id: coordinadorId, 
                    postgrado_id: postgradoId, 
                    escolaridad_id: escolaridadId 
                },
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        $row.fadeOut(400, function() {
                            $(this).remove();
                        });
                        
                        if ($('#coordinador_id').val() == coordinadorId && $('#postgrado_id').val() == postgradoId) {
                            $('#postgrado_id').trigger('change');
                        }
                    } else {
                        alert('Error: ' + response.message);
                    }
                },
                error: function() {
                    alert("Error de conexión al intentar eliminar la asignación.");
                }
            });
        }
    });
});
</script>

<?php
include '../includes/footer.php';
$conn->close();
?>