<?php
// Centralized authentication and session check
$allowed_roles = ['administrador'];
require_once __DIR__ . '/../includes/auth_check.php';

// DB Connection
require_once __DIR__ . '/../includes/db.php';

// Configurar conexión para usar UTF-8
$conn->set_charset("utf8mb4");
$conn->query("SET NAMES 'utf8mb4'");
$conn->query("SET CHARACTER SET utf8mb4");
$conn->query("SET COLLATION_CONNECTION = 'utf8mb4_unicode_ci'");

// Variables para mensajes
$mensaje = '';
$tipo_mensaje = '';
$busqueda = isset($_GET['busqueda']) ? trim($_GET['busqueda']) : '';

// Procesar mensajes de sesión
if (isset($_SESSION['error'])) {
    $mensaje = $_SESSION['error'];
    $tipo_mensaje = 'danger';
    unset($_SESSION['error']);
}

if (isset($_SESSION['success'])) {
    $mensaje = $_SESSION['success'];
    $tipo_mensaje = 'success';
    unset($_SESSION['success']);
}

// Obtener lista de estudiantes con búsqueda
$estudiantes = [];
$sql_estudiantes = "SELECT e.id, e.cedula, e.nombres, e.apellidos, 
                   (SELECT COUNT(*) FROM suspensiones s 
                    WHERE s.estudiante_id = e.id 
                    AND s.fecha_fin >= CURDATE()) as suspensiones_activas,
                   (SELECT COUNT(*) FROM estudiante_postgrado ep 
                    WHERE ep.estudiante_id = e.id 
                    AND ep.estado IN ('activo', 'cambio')) as postgrados_activos
                   FROM estudiantes e
                   WHERE e.cedula LIKE ? OR e.nombres LIKE ? OR e.apellidos LIKE ? 
                   ORDER BY e.apellidos, e.nombres";
$stmt_estudiantes = $conn->prepare($sql_estudiantes);
$param_busqueda = "%$busqueda%";
$stmt_estudiantes->bind_param("sss", $param_busqueda, $param_busqueda, $param_busqueda);
$stmt_estudiantes->execute();
$result_estudiantes = $stmt_estudiantes->get_result();
$estudiantes = $result_estudiantes->fetch_all(MYSQLI_ASSOC);

// Obtener lista de postgrados
$postgrados = [];
$sql_postgrados = "SELECT id, nombre, tipo_postgrado, creditos_electivos FROM postgrados ORDER BY nombre";
$result_postgrados = $conn->query($sql_postgrados);
$postgrados = $result_postgrados->fetch_all(MYSQLI_ASSOC);

// Obtener inscripciones existentes con búsqueda y paginación
$limit = 10; // Número de registros por página
$page = isset($_GET['page']) ? intval($_GET['page']) : 1;
$offset = ($page - 1) * $limit;

// Consulta para contar total de registros
$sql_count = "SELECT COUNT(*) as total 
              FROM estudiante_postgrado ep
              JOIN estudiantes e ON ep.estudiante_id = e.id
              JOIN postgrados p ON ep.postgrado_id = p.id
              WHERE e.cedula LIKE ? OR e.nombres LIKE ? OR e.apellidos LIKE ? OR p.nombre LIKE ?";
$stmt_count = $conn->prepare($sql_count);
$stmt_count->bind_param("ssss", $param_busqueda, $param_busqueda, $param_busqueda, $param_busqueda);
$stmt_count->execute();
$result_count = $stmt_count->get_result();
$total_rows = $result_count->fetch_assoc()['total'];
$total_pages = ceil($total_rows / $limit);

// Consulta para obtener los datos con paginación - MODIFICADA PARA INCLUIR USUARIO QUE INSCRIBIÓ
$sql_inscripciones = "SELECT ep.id, e.cedula, e.nombres, e.apellidos, p.nombre as postgrado, 
                     ep.fecha_inscripcion, ep.estado, ep.postgrado_id as postgrado_actual_id, p.tipo_postgrado,
                     ep.anio_inicio, u.nombres as usuario_nombres, u.apellidos as usuario_apellidos, u.username,
                     COALESCE(CONCAT(u.nombres, ' ', u.apellidos), 'Sistema') as inscrito_por
                     FROM estudiante_postgrado ep
                     JOIN estudiantes e ON ep.estudiante_id = e.id
                     JOIN postgrados p ON ep.postgrado_id = p.id
                     LEFT JOIN usuarios u ON ep.usuario_inscripcion_id = u.id
                     WHERE e.cedula LIKE ? OR e.nombres LIKE ? OR e.apellidos LIKE ? OR p.nombre LIKE ?
                     ORDER BY ep.fecha_inscripcion DESC
                     LIMIT ? OFFSET ?";
$stmt_inscripciones = $conn->prepare($sql_inscripciones);
$stmt_inscripciones->bind_param("ssssii", $param_busqueda, $param_busqueda, $param_busqueda, $param_busqueda, $limit, $offset);
$stmt_inscripciones->execute();
$result_inscripciones = $stmt_inscripciones->get_result();
$inscripciones = $result_inscripciones->fetch_all(MYSQLI_ASSOC);

// Procesar formulario de inscripción
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['inscribir'])) {
        $estudiante_id = intval($_POST['estudiante_id']);
        $postgrado_id = intval($_POST['postgrado_id']);
        $anio_inicio = intval($_POST['anio_inicio']);
        $fecha_inscripcion = !empty($_POST['fecha_inscripcion']) ? $_POST['fecha_inscripcion'] : date('Y-m-d');
        $denominaciones_seleccionadas = isset($_POST['denominaciones']) ? $_POST['denominaciones'] : [];
        $usuario_actual_id = $_SESSION['user_id'];

        // DEBUG: Registrar información del usuario
        error_log("=== NUEVA INSCRIPCIÓN ===");
        error_log("Usuario ID: " . $usuario_actual_id);
        error_log("Estudiante ID: " . $estudiante_id);
        error_log("Postgrado ID: " . $postgrado_id);
        error_log("Año inicio: " . $anio_inicio);

        // Validaciones básicas
        if ($estudiante_id <= 0 || $postgrado_id <= 0 || $anio_inicio < 1900) {
            $_SESSION['error'] = "Datos de inscripción inválidos. El año debe ser un valor mayor o igual a 1900.";
            header("Location: gestion_inscripciones.php");
            exit();
        }

        // Verificar si el estudiante está suspendido
        $sql_suspension = "SELECT COUNT(*) as suspensiones_activas 
                          FROM suspensiones 
                          WHERE estudiante_id = ? AND fecha_fin >= CURDATE()";
        $stmt_suspension = $conn->prepare($sql_suspension);
        $stmt_suspension->bind_param("i", $estudiante_id);
        $stmt_suspension->execute();
        $result_suspension = $stmt_suspension->get_result();
        $suspension = $result_suspension->fetch_assoc();
        
        if ($suspension['suspensiones_activas'] > 0) {
            $_SESSION['error'] = "El estudiante está actualmente suspendido y no puede ser inscrito";
            header("Location: gestion_inscripciones.php");
            exit();
        }

        // VERIFICAR SI EL ESTUDIANTE YA TIENE UN POSTGRADO ACTIVO
        $sql_postgrado_activo = "SELECT COUNT(*) as total_activos 
                                FROM estudiante_postgrado 
                                WHERE estudiante_id = ? AND estado IN ('activo', 'cambio')";
        $stmt_postgrado_activo = $conn->prepare($sql_postgrado_activo);
        $stmt_postgrado_activo->bind_param("i", $estudiante_id);
        $stmt_postgrado_activo->execute();
        $result_postgrado_activo = $stmt_postgrado_activo->get_result();
        $postgrado_activo = $result_postgrado_activo->fetch_assoc();

        if ($postgrado_activo['total_activos'] > 0) {
            $_SESSION['error'] = "El estudiante ya tiene un postgrado activo. Debe terminar (graduarse o retirarse) del postgrado actual antes de inscribirse en uno nuevo.";
            header("Location: gestion_inscripciones.php");
            exit();
        }

        // Verificar el estado actual del estudiante en el postgrado específico
        $sql_check = "SELECT estado FROM estudiante_postgrado 
                     WHERE estudiante_id = ? AND postgrado_id = ?";
        $stmt_check = $conn->prepare($sql_check);
        $stmt_check->bind_param("ii", $estudiante_id, $postgrado_id);
        $stmt_check->execute();
        $result_check = $stmt_check->get_result();
        
        if ($result_check->num_rows > 0) {
            $estado_actual = $result_check->fetch_assoc()['estado'];
            
            if ($estado_actual == 'activo') {
                $_SESSION['error'] = "El estudiante ya está activo en este postgrado";
                header("Location: gestion_inscripciones.php");
                exit();
            }
            
            if ($estado_actual == 'graduado') {
                $_SESSION['error'] = "El estudiante ya está graduado de este postgrado";
                header("Location: gestion_inscripciones.php");
                exit();
            }
            
            // Si está retirado, actualizamos el estado a activo en lugar de crear nueva inscripción
            if ($estado_actual == 'retirado') {
                $sql_reactivar = "UPDATE estudiante_postgrado SET estado = 'activo', fecha_inscripcion = ?, anio_inicio = ?, usuario_inscripcion_id = ?
                                 WHERE estudiante_id = ? AND postgrado_id = ?";
                $stmt_reactivar = $conn->prepare($sql_reactivar);
                $stmt_reactivar->bind_param("siiii", $fecha_inscripcion, $anio_inicio, $usuario_actual_id, $estudiante_id, $postgrado_id);
                
                if ($stmt_reactivar->execute()) {
                    $_SESSION['success'] = "Estudiante reactivado correctamente en el postgrado";
                } else {
                    $_SESSION['error'] = "Error al reactivar al estudiante: " . $conn->error;
                }
                
                header("Location: gestion_inscripciones.php?busqueda=" . urlencode($busqueda));
                exit();
            }
        }

        // Obtener información del postgrado
        $sql_postgrado_info = "SELECT tipo_postgrado, creditos_electivos FROM postgrados WHERE id = ?";
        $stmt_postgrado_info = $conn->prepare($sql_postgrado_info);
        $stmt_postgrado_info->bind_param("i", $postgrado_id);
        $stmt_postgrado_info->execute();
        $postgrado_info = $stmt_postgrado_info->get_result()->fetch_assoc();
        $tipo_postgrado = $postgrado_info['tipo_postgrado'];
        $creditos_electivos_requeridos = $postgrado_info['creditos_electivos'];

        // Iniciar transacción
        $conn->begin_transaction();

        try {
            // INSERCIÓN CORREGIDA: Incluir usuario_inscripcion_id correctamente
            $sql_insc = "INSERT INTO estudiante_postgrado 
                        (estudiante_id, postgrado_id, fecha_inscripcion, estado, anio_inicio, usuario_inscripcion_id) 
                        VALUES (?, ?, ?, 'activo', ?, ?)";
            $stmt_insc = $conn->prepare($sql_insc);
            $stmt_insc->bind_param("iisii", $estudiante_id, $postgrado_id, $fecha_inscripcion, $anio_inicio, $usuario_actual_id);
            
            if (!$stmt_insc->execute()) {
                throw new Exception("Error al insertar en estudiante_postgrado: " . $conn->error);
            }
            
            $inscripcion_id = $stmt_insc->insert_id;
            error_log("Inscripción creada con ID: " . $inscripcion_id . " por usuario: " . $usuario_actual_id);

            // Para postgrados tradicionales, inscribir todas las denominaciones obligatorias
            if ($tipo_postgrado == 'tradicional') {
                // Obtener denominaciones obligatorias del postgrado
                $sql_denominaciones = "SELECT id, periodo FROM denominaciones 
                                      WHERE postgrado_id = ? AND tipo_denominacion = 'obligatoria' ORDER BY periodo, codigo";
                $stmt_denominaciones = $conn->prepare($sql_denominaciones);
                $stmt_denominaciones->bind_param("i", $postgrado_id);
                $stmt_denominaciones->execute();
                $result_denominaciones = $stmt_denominaciones->get_result();
                
                if ($result_denominaciones->num_rows == 0) {
                    throw new Exception("El postgrado no tiene asignaturas obligatorias definidas");
                }

                // Procesar cada denominación obligatoria
                while ($denominacion = $result_denominaciones->fetch_assoc()) {
                    $periodo = intval($denominacion['periodo']);
                    $anios_adicionales = floor(($periodo - 1) / 3);
                    $trimestre = (($periodo - 1) % 3) + 1;
                    $anio_actual = $anio_inicio + $anios_adicionales;
                    $escolaridad = $anio_actual;
                    $anio_lectivo = $anio_actual . '-' . str_pad($trimestre, 2, '0', STR_PAD_LEFT) . 'C';
                    
                    // Obtener o crear escolaridad
                    $escolaridad_id = obtener_o_crear($conn, 'escolaridades', 'nombre', $escolaridad);
                    
                    // Obtener o crear año lectivo
                    $anio_lectivo_id = obtener_o_crear($conn, 'anios_lectivos', 'nombre', $anio_lectivo);
                    
                    // Obtener sección por defecto
                    $seccion_id = $conn->query("SELECT id FROM secciones LIMIT 1")->fetch_assoc()['id'];
                    
                    // Crear inscripción a la denominación
                    $sql_insc_denom = "INSERT INTO inscripciones 
                                      (estudiante_id, denominacion_id, escolaridad_id, año_lectivo_id, seccion_id, fecha_inscripcion, estado) 
                                      VALUES (?, ?, ?, ?, ?, ?, 'activa')";
                    $stmt_insc_denom = $conn->prepare($sql_insc_denom);
                    $stmt_insc_denom->bind_param("iiiiis", $estudiante_id, $denominacion['id'], $escolaridad_id, $anio_lectivo_id, $seccion_id, $fecha_inscripcion);
                    
                    if (!$stmt_insc_denom->execute()) {
                        throw new Exception("Error al inscribir denominación: " . $conn->error);
                    }
                }
                
                $_SESSION['success'] = "Estudiante inscrito correctamente en todas las asignaturas obligatorias del postgrado";
            } 
            // Para postgrados mixtos o electivos
            else {
                // Inscribir denominaciones obligatorias (si las hay)
                $sql_denominaciones_obligatorias = "SELECT id, periodo FROM denominaciones 
                                                   WHERE postgrado_id = ? AND tipo_denominacion = 'obligatoria' ORDER BY periodo, codigo";
                $stmt_denominaciones_obligatorias = $conn->prepare($sql_denominaciones_obligatorias);
                $stmt_denominaciones_obligatorias->bind_param("i", $postgrado_id);
                $stmt_denominaciones_obligatorias->execute();
                $result_denominaciones_obligatorias = $stmt_denominaciones_obligatorias->get_result();
                
                $denominaciones_obligatorias_inscritas = 0;
                while ($denominacion = $result_denominaciones_obligatorias->fetch_assoc()) {
                    $periodo = intval($denominacion['periodo']);
                    $anios_adicionales = floor(($periodo - 1) / 3);
                    $trimestre = (($periodo - 1) % 3) + 1;
                    $anio_actual = $anio_inicio + $anios_adicionales;
                    $escolaridad = $anio_actual;
                    $anio_lectivo = $anio_actual . '-' . str_pad($trimestre, 2, '0', STR_PAD_LEFT) . 'C';
                    
                    // Obtener o crear escolaridad
                    $escolaridad_id = obtener_o_crear($conn, 'escolaridades', 'nombre', $escolaridad);
                    
                    // Obtener o crear año lectivo
                    $anio_lectivo_id = obtener_o_crear($conn, 'anios_lectivos', 'nombre', $anio_lectivo);
                    
                    // Obtener sección por defecto
                    $seccion_id = $conn->query("SELECT id FROM secciones LIMIT 1")->fetch_assoc()['id'];
                    
                    // Crear inscripción a la denominación
                    $sql_insc_denom = "INSERT INTO inscripciones 
                                      (estudiante_id, denominacion_id, escolaridad_id, año_lectivo_id, seccion_id, fecha_inscripcion, estado) 
                                      VALUES (?, ?, ?, ?, ?, ?, 'activa')";
                    $stmt_insc_denom = $conn->prepare($sql_insc_denom);
                    $stmt_insc_denom->bind_param("iiiiis", $estudiante_id, $denominacion['id'], $escolaridad_id, $anio_lectivo_id, $seccion_id, $fecha_inscripcion);
                    
                    if (!$stmt_insc_denom->execute()) {
                        throw new Exception("Error al inscribir denominación obligatoria: " . $conn->error);
                    }
                    $denominaciones_obligatorias_inscritas++;
                }

                // Inscribir denominaciones electivas seleccionadas
                $denominaciones_electivas_inscritas = 0;
                $total_creditos_electivos = 0;
                
                foreach ($denominaciones_seleccionadas as $denominacion_id) {
                    // Obtener información de la denominación electiva
                    $sql_electiva = "SELECT credito, periodo FROM denominaciones WHERE id = ?";
                    $stmt_electiva = $conn->prepare($sql_electiva);
                    $stmt_electiva->bind_param("i", $denominacion_id);
                    $stmt_electiva->execute();
                    $electiva = $stmt_electiva->get_result()->fetch_assoc();
                    
                    $periodo = intval($electiva['periodo']);
                    $anios_adicionales = floor(($periodo - 1) / 3);
                    $trimestre = (($periodo - 1) % 3) + 1;
                    $anio_actual = $anio_inicio + $anios_adicionales;
                    $escolaridad = $anio_actual;
                    $anio_lectivo = $anio_actual . '-' . str_pad($trimestre, 2, '0', STR_PAD_LEFT) . 'C';
                    
                    // Obtener o crear escolaridad
                    $escolaridad_id = obtener_o_crear($conn, 'escolaridades', 'nombre', $escolaridad);
                    
                    // Obtener o crear año lectivo
                    $anio_lectivo_id = obtener_o_crear($conn, 'anios_lectivos', 'nombre', $anio_lectivo);
                    
                    // Obtener sección por defecto
                    $seccion_id = $conn->query("SELECT id FROM secciones LIMIT 1")->fetch_assoc()['id'];
                    
                    // Crear inscripción a la denominación electiva
                    $sql_insc_denom = "INSERT INTO inscripciones 
                                      (estudiante_id, denominacion_id, escolaridad_id, año_lectivo_id, seccion_id, fecha_inscripcion, estado) 
                                      VALUES (?, ?, ?, ?, ?, ?, 'activa')";
                    $stmt_insc_denom = $conn->prepare($sql_insc_denom);
                    $stmt_insc_denom->bind_param("iiiiis", $estudiante_id, $denominacion_id, $escolaridad_id, $anio_lectivo_id, $seccion_id, $fecha_inscripcion);
                    
                    if (!$stmt_insc_denom->execute()) {
                        throw new Exception("Error al inscribir denominación electiva: " . $conn->error);
                    }
                    
                    $denominaciones_electivas_inscritas++;
                    $total_creditos_electivos += $electiva['credito'];
                }

                // Validar créditos electivos
                if ($tipo_postgrado == 'electivo' && $total_creditos_electivos < $creditos_electivos_requeridos) {
                    throw new Exception("Debe seleccionar al menos $creditos_electivos_requeridos créditos electivos. Actualmente ha seleccionado $total_creditos_electivos créditos.");
                }

                if ($tipo_postgrado == 'mixto') {
                    $_SESSION['success'] = "Estudiante inscrito correctamente. Se han inscrito $denominaciones_obligatorias_inscritas asignatura(s) obligatoria(s) y $denominaciones_electivas_inscritas asignatura(s) electiva(s). Total créditos electivos: $total_creditos_electivos";
                } else {
                    $_SESSION['success'] = "Estudiante inscrito correctamente. Se han inscrito $denominaciones_electivas_inscritas asignatura(s) electiva(s). Total créditos: $total_creditos_electivos";
                }
            }
            
            $conn->commit();
            error_log("Inscripción completada exitosamente");
            
        } catch (Exception $e) {
            $conn->rollback();
            error_log("Error en inscripción: " . $e->getMessage());
            $_SESSION['error'] = "Error al inscribir al estudiante: " . $e->getMessage();
        }
        
        header("Location: gestion_inscripciones.php?busqueda=" . urlencode($busqueda));
        exit();
    }
    
    // Procesar modificación de inscripción
    if (isset($_POST['modificar'])) {
        $inscripcion_id = intval($_POST['inscripcion_id']);
        $nuevo_postgrado_id = intval($_POST['postgrado_id']);
        $nuevo_anio_inicio = intval($_POST['anio_inicio']);
        $fecha_modificacion = !empty($_POST['fecha_modificacion']) ? $_POST['fecha_modificacion'] : date('Y-m-d');
        $denominaciones_seleccionadas = isset($_POST['denominaciones']) ? $_POST['denominaciones'] : [];
        $usuario_actual_id = $_SESSION['user_id'];
        
        error_log("=== MODIFICACIÓN DE INSCRIPCIÓN ===");
        error_log("Inscripción ID: " . $inscripcion_id);
        error_log("Nuevo postgrado ID: " . $nuevo_postgrado_id);
        error_log("Nuevo año inicio: " . $nuevo_anio_inicio);
        error_log("Usuario modificando: " . $usuario_actual_id);
        
        // Validaciones básicas
        if ($inscripcion_id <= 0 || $nuevo_anio_inicio < 1900) {
            $_SESSION['error'] = "Datos de modificación inválidos. El año debe ser un valor mayor o igual a 1900.";
            header("Location: gestion_inscripciones.php?busqueda=" . urlencode($busqueda));
            exit();
        }
        
        // Obtener información actual de la inscripción
        $sql_actual = "SELECT estudiante_id, postgrado_id, anio_inicio FROM estudiante_postgrado WHERE id = ?";
        $stmt_actual = $conn->prepare($sql_actual);
        $stmt_actual->bind_param("i", $inscripcion_id);
        $stmt_actual->execute();
        $result_actual = $stmt_actual->get_result();
        $inscripcion_actual = $result_actual->fetch_assoc();
        $estudiante_id = $inscripcion_actual['estudiante_id'];
        $postgrado_actual_id = $inscripcion_actual['postgrado_id'];
        $anio_inicio_actual = $inscripcion_actual['anio_inicio'];
        
        error_log("Estudiante ID: " . $estudiante_id);
        error_log("Postgrado actual ID: " . $postgrado_actual_id);
        error_log("Año inicio actual: " . $anio_inicio_actual);
        
        // VERIFICAR SI EL ESTUDIANTE TIENE NOTAS CARGADAS - CORREGIDO
        $sql_tiene_notas = "SELECT COUNT(*) as total_notas 
                           FROM inscripciones i
                           JOIN denominaciones d ON i.denominacion_id = d.id
                           WHERE i.estudiante_id = ? 
                           AND d.postgrado_id = ?
                           AND (i.nota_final IS NOT NULL OR i.nota_definitiva IS NOT NULL)";
        $stmt_tiene_notas = $conn->prepare($sql_tiene_notas);
        $stmt_tiene_notas->bind_param("ii", $estudiante_id, $postgrado_actual_id);
        $stmt_tiene_notas->execute();
        $result_tiene_notas = $stmt_tiene_notas->get_result();
        $tiene_notas = $result_tiene_notas->fetch_assoc()['total_notas'] > 0;
        
        error_log("¿Tiene notas?: " . ($tiene_notas ? 'SÍ' : 'NO'));
        
        // Si tiene notas cargadas, NO permitir cambio de postgrado
        if ($tiene_notas && $nuevo_postgrado_id != $postgrado_actual_id) {
            $_SESSION['error'] = "No se puede cambiar de postgrado porque el estudiante ya tiene notas cargadas. Solo se permite modificar el año de inicio, escolaridad y año lectivo.";
            header("Location: gestion_inscripciones.php?busqueda=" . urlencode($busqueda));
            exit();
        }
        
        // Obtener información del nuevo postgrado (si no hay notas o es el mismo postgrado)
        $postgrado_a_usar = $tiene_notas ? $postgrado_actual_id : $nuevo_postgrado_id;
        
        $sql_postgrado_info = "SELECT tipo_postgrado, creditos_electivos FROM postgrados WHERE id = ?";
        $stmt_postgrado_info = $conn->prepare($sql_postgrado_info);
        $stmt_postgrado_info->bind_param("i", $postgrado_a_usar);
        $stmt_postgrado_info->execute();
        $postgrado_info = $stmt_postgrado_info->get_result()->fetch_assoc();
        $tipo_postgrado = $postgrado_info['tipo_postgrado'];
        $creditos_electivos_requeridos = $postgrado_info['creditos_electivos'];
        
        error_log("Postgrado a usar: " . $postgrado_a_usar);
        error_log("Tipo postgrado: " . $tipo_postgrado);
        
        // Iniciar transacción
        $conn->begin_transaction();
        
        try {
            // Actualizar la inscripción principal CON usuario_modificacion_id
            $sql_update = "UPDATE estudiante_postgrado 
                          SET postgrado_id = ?, fecha_inscripcion = ?, anio_inicio = ?, usuario_modificacion_id = ?, fecha_modificacion = NOW()
                          WHERE id = ?";
            $stmt_update = $conn->prepare($sql_update);
            $stmt_update->bind_param("isiii", $postgrado_a_usar, $fecha_modificacion, $nuevo_anio_inicio, $usuario_actual_id, $inscripcion_id);
            
            if (!$stmt_update->execute()) {
                throw new Exception("Error al actualizar estudiante_postgrado: " . $conn->error);
            }
            
            error_log("Estudiante_postgrado actualizado correctamente");
            
            // Si tiene notas, solo actualizar año lectivo y escolaridad de las inscripciones existentes SIN MODIFICAR NOTAS
            if ($tiene_notas) {
                error_log("Actualizando año de inicio, escolaridad y año lectivo (hay notas - NO se modifican las notas)");
                
                // Obtener todas las inscripciones activas del estudiante con sus períodos
                $sql_inscripciones_actuales = "SELECT i.id, i.denominacion_id, d.periodo, d.codigo, d.nombre,
                                              i.nota_final, i.nota_definitiva
                                              FROM inscripciones i
                                              JOIN denominaciones d ON i.denominacion_id = d.id
                                              WHERE i.estudiante_id = ? 
                                              AND d.postgrado_id = ?
                                              AND i.estado = 'activa'";
                $stmt_inscripciones_actuales = $conn->prepare($sql_inscripciones_actuales);
                $stmt_inscripciones_actuales->bind_param("ii", $estudiante_id, $postgrado_actual_id);
                $stmt_inscripciones_actuales->execute();
                $result_inscripciones_actuales = $stmt_inscripciones_actuales->get_result();
                
                // Contar inscripciones encontradas
                $total_inscripciones = $result_inscripciones_actuales->num_rows;
                error_log("Inscripciones encontradas: $total_inscripciones");
                
                // Preparar consulta para actualizar cada inscripción individualmente
                $sql_actualizar_inscripcion = "UPDATE inscripciones 
                                              SET escolaridad_id = ?, 
                                                  año_lectivo_id = ?, 
                                                  fecha_modificacion = NOW(), 
                                                  usuario_modificacion = ?
                                              WHERE id = ?";
                $stmt_actualizar_inscripcion = $conn->prepare($sql_actualizar_inscripcion);
                
                $inscripciones_actualizadas = 0;
                
                // Procesar cada inscripción existente
                while ($inscripcion = $result_inscripciones_actuales->fetch_assoc()) {
                    $periodo = intval($inscripcion['periodo']);
                    $anios_adicionales = floor(($periodo - 1) / 3);
                    $trimestre = (($periodo - 1) % 3) + 1;
                    $anio_actual = $nuevo_anio_inicio + $anios_adicionales;
                    $escolaridad = $anio_actual;
                    $anio_lectivo = $anio_actual . '-' . str_pad($trimestre, 2, '0', STR_PAD_LEFT) . 'C';
                    
                    // Obtener o crear escolaridad
                    $escolaridad_id = obtener_o_crear($conn, 'escolaridades', 'nombre', $escolaridad);
                    
                    // Obtener o crear año lectivo
                    $anio_lectivo_id = obtener_o_crear($conn, 'anios_lectivos', 'nombre', $anio_lectivo);
                    
                    error_log("Asignatura: {$inscripcion['codigo']} - Periodo: $periodo - Año nuevo: $anio_actual - Año lectivo: $anio_lectivo");
                    error_log("Nota final: {$inscripcion['nota_final']} - Nota definitiva: {$inscripcion['nota_definitiva']}");
                    
                    // Actualizar la inscripción específica (SOLO escolaridad y año lectivo, NO las notas)
                    $stmt_actualizar_inscripcion->bind_param("iiii", $escolaridad_id, $anio_lectivo_id, $usuario_actual_id, $inscripcion['id']);
                    if ($stmt_actualizar_inscripcion->execute()) {
                        $inscripciones_actualizadas++;
                        error_log("Inscripción {$inscripcion['id']} actualizada correctamente (solo escolaridad y año lectivo)");
                    } else {
                        error_log("ERROR al actualizar inscripción {$inscripcion['id']}: " . $conn->error);
                        throw new Exception("Error al actualizar inscripción ID {$inscripcion['id']}");
                    }
                }
                
                $_SESSION['success'] = "Año de inicio actualizado correctamente. Se han ajustado la escolaridad y año lectivo de $inscripciones_actualizadas asignaturas manteniendo todas las notas intactas.";
            } 
            // Si NO tiene notas, proceder con cambio completo de postgrado
            else {
                error_log("Realizando cambio completo de postgrado (no hay notas)");
                
                // Eliminar las inscripciones a denominaciones antiguas
                $sql_delete = "DELETE i FROM inscripciones i
                              JOIN denominaciones d ON i.denominacion_id = d.id
                              WHERE i.estudiante_id = ? AND d.postgrado_id = ?";
                $stmt_delete = $conn->prepare($sql_delete);
                $stmt_delete->bind_param("ii", $estudiante_id, $postgrado_actual_id);
                
                if (!$stmt_delete->execute()) {
                    throw new Exception("Error al eliminar inscripciones antiguas: " . $conn->error);
                }
                
                error_log("Inscripciones antiguas eliminadas");
                
                // Para postgrados tradicionales, inscribir todas las denominaciones obligatorias
                if ($tipo_postgrado == 'tradicional') {
                    // Obtener denominaciones obligatorias del postgrado
                    $sql_denominaciones = "SELECT id, periodo FROM denominaciones 
                                          WHERE postgrado_id = ? AND tipo_denominacion = 'obligatoria' ORDER BY periodo, codigo";
                    $stmt_denominaciones = $conn->prepare($sql_denominaciones);
                    $stmt_denominaciones->bind_param("i", $nuevo_postgrado_id);
                    $stmt_denominaciones->execute();
                    $result_denominaciones = $stmt_denominaciones->get_result();
                    
                    if ($result_denominaciones->num_rows == 0) {
                        throw new Exception("El postgrado no tiene asignaturas obligatorias definidas");
                    }

                    // Procesar cada denominación obligatoria
                    while ($denominacion = $result_denominaciones->fetch_assoc()) {
                        $periodo = intval($denominacion['periodo']);
                        $anios_adicionales = floor(($periodo - 1) / 3);
                        $trimestre = (($periodo - 1) % 3) + 1;
                        $anio_actual = $nuevo_anio_inicio + $anios_adicionales;
                        $escolaridad = $anio_actual;
                        $anio_lectivo = $anio_actual . '-' . str_pad($trimestre, 2, '0', STR_PAD_LEFT) . 'C';
                        
                        // Obtener o crear escolaridad
                        $escolaridad_id = obtener_o_crear($conn, 'escolaridades', 'nombre', $escolaridad);
                        
                        // Obtener o crear año lectivo
                        $anio_lectivo_id = obtener_o_crear($conn, 'anios_lectivos', 'nombre', $anio_lectivo);
                        
                        // Obtener sección por defecto
                        $seccion_id = $conn->query("SELECT id FROM secciones LIMIT 1")->fetch_assoc()['id'];
                        
                        // Crear inscripción a la denominación
                        $sql_insc_denom = "INSERT INTO inscripciones 
                                          (estudiante_id, denominacion_id, escolaridad_id, año_lectivo_id, seccion_id, fecha_inscripcion, estado) 
                                          VALUES (?, ?, ?, ?, ?, ?, 'activa')";
                        $stmt_insc_denom = $conn->prepare($sql_insc_denom);
                        $stmt_insc_denom->bind_param("iiiiis", $estudiante_id, $denominacion['id'], $escolaridad_id, $anio_lectivo_id, $seccion_id, $fecha_modificacion);
                        
                        if (!$stmt_insc_denom->execute()) {
                            throw new Exception("Error al inscribir denominación: " . $conn->error);
                        }
                    }
                    
                    $_SESSION['success'] = "Inscripción modificada correctamente. Se han inscrito todas las asignaturas obligatorias del postgrado tradicional.";
                } 
                // Para postgrados mixtos o electivos
                else {
                    // Inscribir denominaciones obligatorias (si las hay)
                    $sql_denominaciones_obligatorias = "SELECT id, periodo FROM denominaciones 
                                                       WHERE postgrado_id = ? AND tipo_denominacion = 'obligatoria' ORDER BY periodo, codigo";
                    $stmt_denominaciones_obligatorias = $conn->prepare($sql_denominaciones_obligatorias);
                    $stmt_denominaciones_obligatorias->bind_param("i", $nuevo_postgrado_id);
                    $stmt_denominaciones_obligatorias->execute();
                    $result_denominaciones_obligatorias = $stmt_denominaciones_obligatorias->get_result();
                    
                    $denominaciones_obligatorias_inscritas = 0;
                    while ($denominacion = $result_denominaciones_obligatorias->fetch_assoc()) {
                        $periodo = intval($denominacion['periodo']);
                        $anios_adicionales = floor(($periodo - 1) / 3);
                        $trimestre = (($periodo - 1) % 3) + 1;
                        $anio_actual = $nuevo_anio_inicio + $anios_adicionales;
                        $escolaridad = $anio_actual;
                        $anio_lectivo = $anio_actual . '-' . str_pad($trimestre, 2, '0', STR_PAD_LEFT) . 'C';
                        
                        // Obtener o crear escolaridad
                        $escolaridad_id = obtener_o_crear($conn, 'escolaridades', 'nombre', $escolaridad);
                        
                        // Obtener o crear año lectivo
                        $anio_lectivo_id = obtener_o_crear($conn, 'anios_lectivos', 'nombre', $anio_lectivo);
                        
                        // Obtener sección por defecto
                        $seccion_id = $conn->query("SELECT id FROM secciones LIMIT 1")->fetch_assoc()['id'];
                        
                        // Crear inscripción a la denominación
                        $sql_insc_denom = "INSERT INTO inscripciones 
                                          (estudiante_id, denominacion_id, escolaridad_id, año_lectivo_id, seccion_id, fecha_inscripcion, estado) 
                                          VALUES (?, ?, ?, ?, ?, ?, 'activa')";
                        $stmt_insc_denom = $conn->prepare($sql_insc_denom);
                        $stmt_insc_denom->bind_param("iiiiis", $estudiante_id, $denominacion['id'], $escolaridad_id, $anio_lectivo_id, $seccion_id, $fecha_modificacion);
                        
                        if (!$stmt_insc_denom->execute()) {
                            throw new Exception("Error al inscribir denominación obligatoria: " . $conn->error);
                        }
                        $denominaciones_obligatorias_inscritas++;
                    }

                    // Inscribir denominaciones electivas seleccionadas
                    $denominaciones_electivas_inscritas = 0;
                    $total_creditos_electivos = 0;
                    
                    foreach ($denominaciones_seleccionadas as $denominacion_id) {
                        // Obtener información de la denominación electiva
                        $sql_electiva = "SELECT credito, periodo FROM denominaciones WHERE id = ?";
                        $stmt_electiva = $conn->prepare($sql_electiva);
                        $stmt_electiva->bind_param("i", $denominacion_id);
                        $stmt_electiva->execute();
                        $electiva = $stmt_electiva->get_result()->fetch_assoc();
                        
                        $periodo = intval($electiva['periodo']);
                        $anios_adicionales = floor(($periodo - 1) / 3);
                        $trimestre = (($periodo - 1) % 3) + 1;
                        $anio_actual = $nuevo_anio_inicio + $anios_adicionales;
                        $escolaridad = $anio_actual;
                        $anio_lectivo = $anio_actual . '-' . str_pad($trimestre, 2, '0', STR_PAD_LEFT) . 'C';
                        
                        // Obtener o crear escolaridad
                        $escolaridad_id = obtener_o_crear($conn, 'escolaridades', 'nombre', $escolaridad);
                        
                        // Obtener o crear año lectivo
                        $anio_lectivo_id = obtener_o_crear($conn, 'anios_lectivos', 'nombre', $anio_lectivo);
                        
                        // Obtener sección por defecto
                        $seccion_id = $conn->query("SELECT id FROM secciones LIMIT 1")->fetch_assoc()['id'];
                        
                        // Crear inscripción a la denominación electiva
                        $sql_insc_denom = "INSERT INTO inscripciones 
                                          (estudiante_id, denominacion_id, escolaridad_id, año_lectivo_id, seccion_id, fecha_inscripcion, estado) 
                                          VALUES (?, ?, ?, ?, ?, ?, 'activa')";
                        $stmt_insc_denom = $conn->prepare($sql_insc_denom);
                        $stmt_insc_denom->bind_param("iiiiis", $estudiante_id, $denominacion_id, $escolaridad_id, $anio_lectivo_id, $seccion_id, $fecha_modificacion);
                        
                        if (!$stmt_insc_denom->execute()) {
                            throw new Exception("Error al inscribir denominación electiva: " . $conn->error);
                        }
                        
                        $denominaciones_electivas_inscritas++;
                        $total_creditos_electivos += $electiva['credito'];
                    }

                    // Validar créditos electivos
                    if ($tipo_postgrado == 'electivo' && $total_creditos_electivos < $creditos_electivos_requeridos) {
                        throw new Exception("Debe seleccionar al menos $creditos_electivos_requeridos créditos electivos. Actualmente ha seleccionado $total_creditos_electivos créditos.");
                    }

                    if ($tipo_postgrado == 'mixto') {
                        $_SESSION['success'] = "Inscripción modificada correctamente. Se han inscrito $denominaciones_obligatorias_inscritas asignatura(s) obligatoria(s) y $denominaciones_electivas_inscritas asignatura(s) electiva(s). Total créditos electivos: $total_creditos_electivos";
                    } else {
                        $_SESSION['success'] = "Inscripción modificada correctamente. Se han inscrito $denominaciones_electivas_inscritas asignatura(s) electiva(s). Total créditos: $total_creditos_electivos";
                    }
                }
            }
            
            $conn->commit();
            error_log("Modificación completada exitosamente");
            
        } catch (Exception $e) {
            $conn->rollback();
            error_log("Error en modificación: " . $e->getMessage());
            $_SESSION['error'] = "Error al modificar la inscripción: " . $e->getMessage();
        }
        
        header("Location: gestion_inscripciones.php?busqueda=" . urlencode($busqueda));
        exit();
    }
}

// Función auxiliar para obtener o crear registros
function obtener_o_crear($conn, $tabla, $campo, $valor) {
    $sql = "SELECT id FROM $tabla WHERE $campo = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("s", $valor);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        return $result->fetch_assoc()['id'];
    } else {
        $sql_insert = "INSERT INTO $tabla ($campo) VALUES (?)";
        $stmt_insert = $conn->prepare($sql_insert);
        $stmt_insert->bind_param("s", $valor);
        $stmt_insert->execute();
        return $stmt_insert->insert_id;
    }
}

// Función para obtener denominaciones por postgrado
function obtener_denominaciones_por_postgrado($conn, $postgrado_id, $tipo_postgrado) {
    if ($tipo_postgrado == 'tradicional') {
        $sql = "SELECT id, codigo, nombre, credito, periodo, tipo_denominacion 
                FROM denominaciones 
                WHERE postgrado_id = ? AND tipo_denominacion = 'obligatoria'
                ORDER BY periodo, codigo";
    } else {
        $sql = "SELECT id, codigo, nombre, credito, periodo, tipo_denominacion 
                FROM denominaciones 
                WHERE postgrado_id = ? 
                ORDER BY tipo_denominacion, periodo, codigo";
    }
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $postgrado_id);
    $stmt->execute();
    $result = $stmt->get_result();
    return $result->fetch_all(MYSQLI_ASSOC);
}

$page_title = "Gestión de Inscripciones";
include '../includes/header.php';
?>
    <style>
        .table-responsive {
            overflow-x: auto;
        }
        .table td, .table th {
            vertical-align: middle;
            white-space: nowrap;
        }
        .btn-action {
            min-width: 80px;
        }
        .badge {
            font-size: 0.9em;
        }
        .suspended {
            color: #dc3545 !important;
            font-weight: bold;
        }
        .postgrado-activo {
            color: #ffc107 !important;
            font-weight: bold;
        }
        select option:disabled {
            color: #dc3545 !important;
            font-weight: bold;
        }
        .pagination .page-item.active .page-link {
            background-color: #0d6efd;
            border-color: #0d6efd;
        }
        .search-card {
            margin-top: 20px;
            margin-bottom: 20px;
        }
        .date-field {
            max-width: 200px;
        }
        .denominaciones-container {
            max-height: 400px;
            overflow-y: auto;
            border: 1px solid #dee2e6;
            border-radius: 0.375rem;
            padding: 10px;
        }
        .electiva-requerida {
            background-color: #fff3cd !important;
        }
        .modal-lg {
            max-width: 800px;
        }
        .usuario-info {
            font-size: 0.85em;
            color: #6c757d;
        }
        .alert-warning {
            background-color: #fff3cd;
            border-color: #ffeaa7;
        }
    </style>
<div class="container">
    <!-- Botón para volver -->
    <div class="mb-4">
        <a href="index.php" class="btn btn-secondary">
            <i class="bi bi-arrow-left"></i> Volver al Panel
        </a>
    </div>

    <h2 class="my-4">Gestión de Inscripciones</h2>
    
    <!-- Mostrar mensajes -->
    <?php if ($mensaje): ?>
        <div class="alert alert-<?= $tipo_mensaje ?>"><?= $mensaje ?></div>
    <?php endif; ?>
    
    <!-- Formulario de búsqueda principal -->
    <div class="card mb-4">
        <div class="card-body">
            <form method="get" class="row g-3">
                <div class="col-md-8">
                    <input type="text" class="form-control" name="busqueda" value="<?= htmlspecialchars($busqueda) ?>" 
                           placeholder="Buscar por cédula, nombres, apellidos o postgrado">
                </div>
                <div class="col-md-2">
                    <button type="submit" class="btn btn-primary w-100">
                        <i class="bi bi-search"></i> Buscar
                    </button>
                </div>
                <div class="col-md-2">
                    <a href="gestion_inscripciones.php" class="btn btn-secondary w-100">
                        <i class="bi bi-x-circle"></i> Limpiar
                    </a>
                </div>
            </form>
        </div>
    </div>
    
    <div class="row">
        <!-- Columna para nueva inscripción -->
        <div class="col-md-6">
            <div class="card">
                <div class="card-header bg-primary text-white">
                    <h3 class="mb-0"><i class="bi bi-person-plus"></i> Nueva Inscripción</h3>
                </div>
                <div class="card-body">
                    <form method="post" id="form-inscripcion">
                        <!-- Campo hidden para el usuario que hace la inscripción -->
                        <input type="hidden" name="usuario_inscripcion_id" value="<?= $_SESSION['user_id'] ?>">
                        
                        <div class="mb-3">
                            <label for="estudiante_id" class="form-label">Estudiante:</label>
                            <select class="form-select" id="estudiante_id" name="estudiante_id" required>
                                <option value="">Seleccionar estudiante</option>
                                <?php foreach ($estudiantes as $estudiante): ?>
                                    <option value="<?= $estudiante['id'] ?>" 
                                        <?= $estudiante['suspensiones_activas'] > 0 ? 'disabled class="suspended"' : 
                                           ($estudiante['postgrados_activos'] > 0 ? 'disabled class="postgrado-activo"' : '') ?>>
                                        <?= htmlspecialchars("{$estudiante['apellidos']}, {$estudiante['nombres']} ({$estudiante['cedula']})") ?>
                                        <?= $estudiante['suspensiones_activas'] > 0 ? ' (SUSPENDIDO)' : '' ?>
                                        <?= $estudiante['postgrados_activos'] > 0 ? ' (POSTGRADO ACTIVO)' : '' ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <div class="form-text">
                                <span class="text-danger">● SUSPENDIDO:</span> No puede ser seleccionado<br>
                                <span class="text-warning">● POSTGRADO ACTIVO:</span> Ya tiene un postgrado en curso
                            </div>
                        </div>
                        
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <label for="postgrado_id" class="form-label">Postgrado:</label>
                                <select class="form-select" id="postgrado_id" name="postgrado_id" required>
                                    <option value="">Seleccionar postgrado</option>
                                    <?php foreach ($postgrados as $postgrado): ?>
                                        <option value="<?= $postgrado['id'] ?>" data-tipo="<?= $postgrado['tipo_postgrado'] ?>" data-creditos="<?= $postgrado['creditos_electivos'] ?>">
                                            <?= htmlspecialchars($postgrado['nombre']) ?> 
                                            (<?= ucfirst($postgrado['tipo_postgrado']) ?>)
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-6">
                                <label for="anio_inicio" class="form-label">Año de inicio:</label>
                                <input type="number" class="form-control" id="anio_inicio" name="anio_inicio" 
                                       min="1900" max="2100" value="<?= date('Y') ?>" required>
                                <small class="text-muted">Ingrese el año de inicio (ej: 1990, 2000, 2023, etc.)</small>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="fecha_inscripcion" class="form-label">Fecha de inscripción (opcional):</label>
                            <input type="date" class="form-control date-field" id="fecha_inscripcion" name="fecha_inscripcion" 
                                   value="<?= date('Y-m-d') ?>">
                            <small class="text-muted">Si no se especifica, se usará la fecha actual</small>
                        </div>
                        
                        <div class="alert alert-info" id="info-tipo-postgrado" style="display: none;">
                            <strong>Información:</strong> 
                            <span id="texto-tipo-postgrado"></span>
                        </div>
                        
                        <!-- Sección para selección de denominaciones -->
                        <div id="seccion-denominaciones" style="display: none;">
                            <h5 class="mt-4">Selección de Asignaturas</h5>
                            <div class="denominaciones-container" id="lista-denominaciones">
                                <!-- Las denominaciones se cargarán aquí por JavaScript -->
                            </div>
                            <div class="mt-3" id="resumen-creditos" style="display: none;">
                                <strong>Créditos seleccionados: </strong>
                                <span id="total-creditos">0</span> / 
                                <span id="creditos-requeridos">0</span>
                                <span id="estado-creditos" class="badge bg-danger ms-2">No cumple</span>
                            </div>
                        </div>
                        
                        <button type="submit" name="inscribir" class="btn btn-primary mt-3" id="btn-inscribir">
                            <i class="bi bi-save"></i> Inscribir Estudiante
                        </button>
                    </form>
                </div>
            </div>
        </div>
        
        <!-- Columna para modificación de inscripción -->
        <div class="col-md-6">
            <div class="card">
                <div class="card-header bg-info text-white">
                    <h3 class="mb-0"><i class="bi bi-pencil-square"></i> Modificar Inscripción</h3>
                </div>
                <div class="card-body">
                    <div class="alert alert-warning">
                        <strong>Nota importante:</strong> 
                        <ul class="mb-0 mt-2">
                            <li>Si el estudiante <strong>NO tiene notas cargadas</strong>: Puede cambiar de postgrado y año de inicio</li>
                            <li>Si el estudiante <strong>YA tiene notas cargadas</strong>: Solo puede modificar el año de inicio (se ajustarán automáticamente la escolaridad y año lectivo manteniendo todas las notas intactas)</li>
                        </ul>
                    </div>
                    <p class="text-muted">Seleccione una inscripción de la lista de abajo y haga clic en "Modificar" para editarla.</p>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Listado de inscripciones existentes -->
    <div class="card mt-4">
        <div class="card-header bg-secondary text-white">
            <div class="d-flex justify-content-between align-items-center">
                <h3 class="mb-0"><i class="bi bi-list-ul"></i> Inscripciones Existentes</h3>
                <span class="badge bg-light text-dark">Total: <?= $total_rows ?></span>
            </div>
        </div>
        <div class="card-body">
            <?php if (empty($inscripciones)): ?>
                <div class="alert alert-info">No hay inscripciones registradas <?= !empty($busqueda) ? 'que coincidan con la búsqueda' : '' ?></div>
            <?php else: ?>
                <div class="table-responsive">
                    <table class="table table-striped table-hover">
                        <thead class="table-dark">
                            <tr>
                                <th>#</th>
                                <th>Estudiante</th>
                                <th>Postgrado</th>
                                <th>Tipo</th>
                                <th>Fecha Inscripción</th>
                                <th>Año Inicio</th>
                                <th>Inscrito por</th>
                                <th>Estado</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php 
                            $contador = ($page - 1) * $limit + 1;
                            foreach ($inscripciones as $insc): ?>
                                <tr>
                                    <td><?= $contador ?></td>
                                    <td><?= htmlspecialchars("{$insc['apellidos']}, {$insc['nombres']}") ?></td>
                                    <td><?= htmlspecialchars($insc['postgrado']) ?></td>
                                    <td>
                                        <span class="badge bg-<?= 
                                            $insc['tipo_postgrado'] == 'tradicional' ? 'primary' : 
                                            ($insc['tipo_postgrado'] == 'mixto' ? 'warning' : 'info') ?>">
                                            <?= ucfirst($insc['tipo_postgrado']) ?>
                                        </span>
                                    </td>
                                    <td><?= date('d/m/Y', strtotime($insc['fecha_inscripcion'])) ?></td>
                                    <td><?= $insc['anio_inicio'] ?></td>
                                    <td class="usuario-info">
                                        <?php if (!empty($insc['usuario_nombres'])): ?>
                                            <?= htmlspecialchars("{$insc['usuario_nombres']} {$insc['usuario_apellidos']}") ?>
                                            <br><small>(<?= htmlspecialchars($insc['username']) ?>)</small>
                                        <?php else: ?>
                                            <span class="text-muted">No registrado</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <span class="badge bg-<?= 
                                            $insc['estado'] == 'activo' ? 'success' : 
                                            ($insc['estado'] == 'graduado' ? 'info' : 
                                            ($insc['estado'] == 'suspendido' ? 'warning' : 
                                            ($insc['estado'] == 'retirado' ? 'secondary' : 'dark'))) ?>">
                                            <?= ucfirst($insc['estado']) ?>
                                        </span>
                                    </td>
                                    <td>
                                        <div class="d-flex gap-2">
                                            <button type="button" class="btn btn-warning btn-sm btn-modificar" 
                                                    data-inscripcion-id="<?= $insc['id'] ?>"
                                                    data-postgrado-id="<?= $insc['postgrado_actual_id'] ?>"
                                                    data-anio-inicio="<?= $insc['anio_inicio'] ?>"
                                                    data-tipo-postgrado="<?= $insc['tipo_postgrado'] ?>">
                                                <i class="bi bi-pencil"></i> Modificar
                                            </button>
                                            
                                            <a href="detalle_inscripcion.php?id=<?= $insc['id'] ?>" 
                                               class="btn btn-sm btn-primary btn-action" 
                                               title="Ver detalles">
                                               <i class="bi bi-eye"></i> Ver
                                            </a>
                                        </div>
                                    </td>
                                </tr>
                            <?php 
                            $contador++;
                            endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Paginación -->
                <?php if ($total_pages > 1): ?>
                    <nav aria-label="Page navigation">
                        <ul class="pagination justify-content-center mt-4">
                            <?php if ($page > 1): ?>
                                <li class="page-item">
                                    <a class="page-link" href="gestion_inscripciones.php?page=<?= $page - 1 ?><?= !empty($busqueda) ? '&busqueda='.urlencode($busqueda) : '' ?>" aria-label="Previous">
                                        <span aria-hidden="true">&laquo;</span>
                                    </a>
                                </li>
                            <?php endif; ?>
                            
                            <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                                <li class="page-item <?= $i == $page ? 'active' : '' ?>">
                                    <a class="page-link" href="gestion_inscripciones.php?page=<?= $i ?><?= !empty($busqueda) ? '&busqueda='.urlencode($busqueda) : '' ?>"><?= $i ?></a>
                                </li>
                            <?php endfor; ?>
                            
                            <?php if ($page < $total_pages): ?>
                                <li class="page-item">
                                    <a class="page-link" href="gestion_inscripciones.php?page=<?= $page + 1 ?><?= !empty($busqueda) ? '&busqueda='.urlencode($busqueda) : '' ?>" aria-label="Next">
                                        <span aria-hidden="true">&raquo;</span>
                                    </a>
                                </li>
                            <?php endif; ?>
                        </ul>
                    </nav>
                <?php endif; ?>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- Modal para Modificar Inscripción -->
<div class="modal fade" id="modalModificar" tabindex="-1" aria-labelledby="modalModificarLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="modalModificarLabel">Modificar Inscripción</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="post" id="form-modificar">
                <div class="modal-body">
                    <input type="hidden" id="modificar_inscripcion_id" name="inscripcion_id">
                    <!-- Campo hidden para el usuario que modifica -->
                    <input type="hidden" name="usuario_modificacion_id" value="<?= $_SESSION['user_id'] ?>">
                    
                    <div class="row mb-3">
                        <div class="col-md-6">
                            <label for="modificar_postgrado_id" class="form-label">Postgrado:</label>
                            <select class="form-select" id="modificar_postgrado_id" name="postgrado_id" required>
                                <option value="">Seleccionar postgrado</option>
                                <?php foreach ($postgrados as $postgrado): ?>
                                    <option value="<?= $postgrado['id'] ?>" data-tipo="<?= $postgrado['tipo_postgrado'] ?>" data-creditos="<?= $postgrado['creditos_electivos'] ?>">
                                        <?= htmlspecialchars($postgrado['nombre']) ?> 
                                        (<?= ucfirst($postgrado['tipo_postgrado']) ?>)
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-6">
                            <label for="modificar_anio_inicio" class="form-label">Año de inicio:</label>
                            <input type="number" class="form-control" id="modificar_anio_inicio" name="anio_inicio" 
                                   min="1900" max="2100" required>
                            <small class="text-muted">Ingrese el año de inicio</small>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="modificar_fecha_modificacion" class="form-label">Fecha de modificación (opcional):</label>
                        <input type="date" class="form-control date-field" id="modificar_fecha_modificacion" name="fecha_modificacion" 
                               value="<?= date('Y-m-d') ?>">
                        <small class="text-muted">Si no se especifica, se usará la fecha actual</small>
                    </div>
                    
                    <div class="alert alert-info" id="modificar-info-tipo-postgrado" style="display: none;">
                        <strong>Información:</strong> 
                        <span id="modificar-texto-tipo-postgrado"></span>
                    </div>
                    
                    <!-- Sección para selección de denominaciones en modificación -->
                    <div id="modificar-seccion-denominaciones" style="display: none;">
                        <h5 class="mt-4">Selección de Asignaturas</h5>
                        <div class="denominaciones-container" id="modificar-lista-denominaciones">
                            <!-- Las denominaciones se cargarán aquí por JavaScript -->
                        </div>
                        <div class="mt-3" id="modificar-resumen-creditos" style="display: none;">
                            <strong>Créditos seleccionados: </strong>
                            <span id="modificar-total-creditos">0</span> / 
                            <span id="modificar-creditos-requeridos">0</span>
                            <span id="modificar-estado-creditos" class="badge bg-danger ms-2">No cumple</span>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" name="modificar" class="btn btn-primary" id="btn-modificar-inscripcion">
                        <i class="bi bi-save"></i> Guardar Cambios
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const postgradoSelect = document.getElementById('postgrado_id');
    const estudianteSelect = document.getElementById('estudiante_id');
    const seccionDenominaciones = document.getElementById('seccion-denominaciones');
    const listaDenominaciones = document.getElementById('lista-denominaciones');
    const infoTipoPostgrado = document.getElementById('info-tipo-postgrado');
    const textoInfo = document.getElementById('texto-tipo-postgrado');
    const resumenCreditos = document.getElementById('resumen-creditos');
    const totalCreditosSpan = document.getElementById('total-creditos');
    const creditosRequeridosSpan = document.getElementById('creditos-requeridos');
    const estadoCreditos = document.getElementById('estado-creditos');
    const btnInscribir = document.getElementById('btn-inscribir');

    // Elementos para modificación
    const modificarPostgradoSelect = document.getElementById('modificar_postgrado_id');
    const modificarSeccionDenominaciones = document.getElementById('modificar-seccion-denominaciones');
    const modificarListaDenominaciones = document.getElementById('modificar-lista-denominaciones');
    const modificarInfoTipoPostgrado = document.getElementById('modificar-info-tipo-postgrado');
    const modificarTextoInfo = document.getElementById('modificar-texto-tipo-postgrado');
    const modificarResumenCreditos = document.getElementById('modificar-resumen-creditos');
    const modificarTotalCreditosSpan = document.getElementById('modificar-total-creditos');
    const modificarCreditosRequeridosSpan = document.getElementById('modificar-creditos-requeridos');
    const modificarEstadoCreditos = document.getElementById('modificar-estado-creditos');
    const modificarBtnGuardar = document.getElementById('btn-modificar-inscripcion');

    let denominacionesData = [];
    let creditosRequeridos = 0;
    let modificarDenominacionesData = [];
    let modificarCreditosRequeridos = 0;

    // Función para cargar denominaciones
    function cargarDenominaciones(postgradoId, tipoPostgrado, creditosRequeridos, contenedorLista, contenedorInfo, contenedorPrincipal, resumenContainer, totalSpan, requeridosSpan, estadoSpan, btnSubmit) {
        if (!postgradoId) {
            contenedorPrincipal.style.display = 'none';
            resumenContainer.style.display = 'none';
            return;
        }

        // Hacer una petición AJAX para obtener las denominaciones
        fetch(`obtener_denominaciones.php?postgrado_id=${postgradoId}&tipo_postgrado=${tipoPostgrado}`)
            .then(response => {
                if (!response.ok) {
                    throw new Error('Error en la respuesta del servidor');
                }
                return response.json();
            })
            .then(data => {
                if (data.denominaciones) {
                    // Mostrar las denominaciones
                    let html = '';
                    let tieneObligatorias = false;
                    let tieneElectivas = false;

                    data.denominaciones.forEach(denominacion => {
                        const esObligatoria = denominacion.tipo_denominacion === 'obligatoria';
                        const esElectiva = denominacion.tipo_denominacion === 'electiva';
                        
                        if (esObligatoria) tieneObligatorias = true;
                        if (esElectiva) tieneElectivas = true;

                        if (tipoPostgrado === 'mixto' && esObligatoria) {
                            // Para mixto, las obligatorias son automáticas (solo mostrar)
                            html += `
                                <div class="form-check mb-2">
                                    <input class="form-check-input" type="checkbox" checked disabled 
                                           data-credito="${denominacion.credito}" data-tipo="obligatoria">
                                    <label class="form-check-label">
                                        <strong>${denominacion.codigo} - ${denominacion.nombre}</strong>
                                        (${denominacion.credito} créditos, Período ${denominacion.periodo})
                                        <span class="badge bg-primary">Obligatoria</span>
                                    </label>
                                </div>
                            `;
                        } else if ((tipoPostgrado === 'mixto' && esElectiva) || tipoPostgrado === 'electivo') {
                            // Para electivas o postgrados electivos puros
                            html += `
                                <div class="form-check mb-2 ${tipoPostgrado === 'electivo' ? 'electiva-requerida' : ''}">
                                    <input class="form-check-input denominacion-checkbox" type="checkbox" 
                                           name="denominaciones[]" value="${denominacion.id}"
                                           data-credito="${denominacion.credito}" data-tipo="electiva">
                                    <label class="form-check-label">
                                        ${denominacion.codigo} - ${denominacion.nombre}
                                        (${denominacion.credito} créditos, Período ${denominacion.periodo})
                                        <span class="badge bg-warning">Electiva</span>
                                    </label>
                                </div>
                            `;
                        }
                    });

                    if (tipoPostgrado === 'mixto') {
                        if (tieneObligatorias) {
                            html = `<h6>Asignaturas Obligatorias (se inscriben automáticamente)</h6>` + html;
                        }
                        if (tieneElectivas) {
                            html += `<h6 class="mt-3">Asignaturas Electivas (seleccione las deseadas)</h6>`;
                        }
                    } else if (tipoPostgrado === 'electivo') {
                        html = `<h6>Seleccione las asignaturas electivas (mínimo ${creditosRequeridos} créditos)</h6>` + html;
                    }

                    contenedorLista.innerHTML = html;
                    contenedorPrincipal.style.display = 'block';
                    
                    // Actualizar resumen de créditos
                    if (tipoPostgrado === 'electivo') {
                        requeridosSpan.textContent = creditosRequeridos;
                        resumenContainer.style.display = 'block';
                        actualizarResumenCreditos(contenedorLista, totalSpan, requeridosSpan, estadoSpan, btnSubmit, creditosRequeridos);
                    } else if (tipoPostgrado === 'mixto' && tieneElectivas) {
                        requeridosSpan.textContent = creditosRequeridos;
                        resumenContainer.style.display = 'block';
                        actualizarResumenCreditos(contenedorLista, totalSpan, requeridosSpan, estadoSpan, btnSubmit, creditosRequeridos);
                    } else {
                        resumenContainer.style.display = 'none';
                        if (btnSubmit) btnSubmit.disabled = false;
                    }

                    // Agregar event listeners a los checkboxes
                    contenedorLista.querySelectorAll('.denominacion-checkbox').forEach(checkbox => {
                        checkbox.addEventListener('change', function() {
                            actualizarResumenCreditos(contenedorLista, totalSpan, requeridosSpan, estadoSpan, btnSubmit, creditosRequeridos);
                        });
                    });

                    // Actualizar inicialmente
                    actualizarResumenCreditos(contenedorLista, totalSpan, requeridosSpan, estadoSpan, btnSubmit, creditosRequeridos);
                } else {
                    console.error('Error al cargar denominaciones:', data.error);
                    contenedorLista.innerHTML = '<div class="alert alert-danger">Error al cargar las asignaturas</div>';
                }
            })
            .catch(error => {
                console.error('Error al cargar denominaciones:', error);
                contenedorLista.innerHTML = '<div class="alert alert-danger">Error al cargar las asignaturas</div>';
            });
    }

    function actualizarResumenCreditos(contenedorLista, totalSpan, requeridosSpan, estadoSpan, btnSubmit, creditosRequeridos) {
        let totalCreditos = 0;
        
        contenedorLista.querySelectorAll('.denominacion-checkbox:checked').forEach(checkbox => {
            totalCreditos += parseInt(checkbox.getAttribute('data-credito'));
        });
        
        totalSpan.textContent = totalCreditos;
        
        if (totalCreditos >= creditosRequeridos) {
            estadoSpan.textContent = 'Cumple';
            estadoSpan.className = 'badge bg-success ms-2';
            if (btnSubmit) btnSubmit.disabled = false;
        } else {
            estadoSpan.textContent = 'No cumple';
            estadoSpan.className = 'badge bg-danger ms-2';
            if (btnSubmit) btnSubmit.disabled = true;
        }
    }

    // Event listeners para el formulario de inscripción
    postgradoSelect.addEventListener('change', function() {
        const selectedOption = this.options[this.selectedIndex];
        const tipoPostgrado = selectedOption.getAttribute('data-tipo');
        const postgradoId = this.value;
        const creditosRequeridos = parseInt(selectedOption.getAttribute('data-creditos')) || 0;

        // Mostrar información del tipo de postgrado
        if (tipoPostgrado) {
            let mensaje = '';
            if (tipoPostgrado === 'tradicional') {
                mensaje = 'Este postgrado tiene un plan de estudios fijo. El estudiante será inscrito en todas las asignaturas obligatorias automáticamente.';
                seccionDenominaciones.style.display = 'none';
                resumenCreditos.style.display = 'none';
                btnInscribir.disabled = false;
            } else if (tipoPostgrado === 'mixto') {
                mensaje = `Este postgrado tiene asignaturas obligatorias y electivas. Las obligatorias se inscribirán automáticamente, debe seleccionar las electivas (mínimo ${creditosRequeridos} créditos).`;
                cargarDenominaciones(postgradoId, tipoPostgrado, creditosRequeridos, listaDenominaciones, textoInfo, seccionDenominaciones, resumenCreditos, totalCreditosSpan, creditosRequeridosSpan, estadoCreditos, btnInscribir);
            } else if (tipoPostgrado === 'electivo') {
                mensaje = `Este postgrado es completamente electivo. Debe seleccionar las asignaturas que desea cursar (mínimo ${creditosRequeridos} créditos).`;
                cargarDenominaciones(postgradoId, tipoPostgrado, creditosRequeridos, listaDenominaciones, textoInfo, seccionDenominaciones, resumenCreditos, totalCreditosSpan, creditosRequeridosSpan, estadoCreditos, btnInscribir);
            }
            
            textoInfo.textContent = mensaje;
            infoTipoPostgrado.style.display = 'block';
        } else {
            infoTipoPostgrado.style.display = 'none';
            seccionDenominaciones.style.display = 'none';
            resumenCreditos.style.display = 'none';
        }
    });

    // Event listeners para el formulario de modificación
    modificarPostgradoSelect.addEventListener('change', function() {
        const selectedOption = this.options[this.selectedIndex];
        const tipoPostgrado = selectedOption.getAttribute('data-tipo');
        const postgradoId = this.value;
        const creditosRequeridos = parseInt(selectedOption.getAttribute('data-creditos')) || 0;

        // Mostrar información del tipo de postgrado
        if (tipoPostgrado) {
            let mensaje = '';
            if (tipoPostgrado === 'tradicional') {
                mensaje = 'Este postgrado tiene un plan de estudios fijo. El estudiante será inscrito en todas las asignaturas obligatorias automáticamente.';
                modificarSeccionDenominaciones.style.display = 'none';
                modificarResumenCreditos.style.display = 'none';
                modificarBtnGuardar.disabled = false;
            } else if (tipoPostgrado === 'mixto') {
                mensaje = `Este postgrado tiene asignaturas obligatorias y electivas. Las obligatorias se inscribirán automáticamente, debe seleccionar las electivas (mínimo ${creditosRequeridos} créditos).`;
                cargarDenominaciones(postgradoId, tipoPostgrado, creditosRequeridos, modificarListaDenominaciones, modificarTextoInfo, modificarSeccionDenominaciones, modificarResumenCreditos, modificarTotalCreditosSpan, modificarCreditosRequeridosSpan, modificarEstadoCreditos, modificarBtnGuardar);
            } else if (tipoPostgrado === 'electivo') {
                mensaje = `Este postgrado es completamente electivo. Debe seleccionar las asignaturas que desea cursar (mínimo ${creditosRequeridos} créditos).`;
                cargarDenominaciones(postgradoId, tipoPostgrado, creditosRequeridos, modificarListaDenominaciones, modificarTextoInfo, modificarSeccionDenominaciones, modificarResumenCreditos, modificarTotalCreditosSpan, modificarCreditosRequeridosSpan, modificarEstadoCreditos, modificarBtnGuardar);
            }
            
            modificarTextoInfo.textContent = mensaje;
            modificarInfoTipoPostgrado.style.display = 'block';
        } else {
            modificarInfoTipoPostgrado.style.display = 'none';
            modificarSeccionDenominaciones.style.display = 'none';
            modificarResumenCreditos.style.display = 'none';
        }
    });

    // Configurar botones de modificación
    document.querySelectorAll('.btn-modificar').forEach(button => {
        button.addEventListener('click', function() {
            const inscripcionId = this.getAttribute('data-inscripcion-id');
            const postgradoId = this.getAttribute('data-postgrado-id');
            const anioInicio = this.getAttribute('data-anio-inicio');
            const tipoPostgrado = this.getAttribute('data-tipo-postgrado');
            
            // Llenar el formulario modal
            document.getElementById('modificar_inscripcion_id').value = inscripcionId;
            document.getElementById('modificar_postgrado_id').value = postgradoId;
            document.getElementById('modificar_anio_inicio').value = anioInicio;
            
            // Disparar el evento change para cargar las denominaciones
            const event = new Event('change');
            modificarPostgradoSelect.dispatchEvent(event);
            
            // Mostrar el modal
            const modal = new bootstrap.Modal(document.getElementById('modalModificar'));
            modal.show();
        });
    });

    // Validación del formulario de inscripción
    document.getElementById('form-inscripcion').addEventListener('submit', function(e) {
        const postgradoId = postgradoSelect.value;
        const estudianteId = estudianteSelect.value;
        const tipoPostgrado = postgradoSelect.options[postgradoSelect.selectedIndex]?.getAttribute('data-tipo');
        
        if (!postgradoId || !estudianteId) {
            e.preventDefault();
            alert('Por favor, complete todos los campos requeridos.');
            return;
        }

        if ((tipoPostgrado === 'mixto' || tipoPostgrado === 'electivo') && btnInscribir.disabled) {
            e.preventDefault();
            alert('Debe seleccionar el mínimo de créditos requeridos para este postgrado.');
            return;
        }
    });

    // Validación del formulario de modificación
    document.getElementById('form-modificar').addEventListener('submit', function(e) {
        const postgradoId = modificarPostgradoSelect.value;
        const tipoPostgrado = modificarPostgradoSelect.options[modificarPostgradoSelect.selectedIndex]?.getAttribute('data-tipo');
        
        if (!postgradoId) {
            e.preventDefault();
            alert('Por favor, complete todos los campos requeridos.');
            return;
        }

        if ((tipoPostgrado === 'mixto' || tipoPostgrado === 'electivo') && modificarBtnGuardar.disabled) {
            e.preventDefault();
            alert('Debe seleccionar el mínimo de créditos requeridos para este postgrado.');
            return;
        }
    });
});

    // Activar tooltips
    document.addEventListener('DOMContentLoaded', function() {
        const tooltipTriggerList = [].slice.call(document.querySelectorAll('[title]'));
        tooltipTriggerList.map(function (tooltipTriggerEl) {
            return new bootstrap.Tooltip(tooltipTriggerEl);
        });
    });
</script>

<?php
include '../includes/footer.php';
?>