<?php
// Configuración detallada de errores
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', '/home/postpemv/public_html/cepmed/admin/constancia_masiva_notas_errors.log');

// Aumentar límites para generación de PDFs
set_time_limit(300);
ini_set('memory_limit', '512M');
ini_set('max_execution_time', 300);

session_start();

if (!isset($_SESSION['username']) || $_SESSION['rol'] !== 'administrador') {
    header("HTTP/1.1 403 Forbidden");
    exit(json_encode(['error' => 'Acceso denegado']));
}

include __DIR__ . '/../includes/db.php';
if (!$conn) {
    header("HTTP/1.1 500 Internal Server Error");
    exit(json_encode(['error' => 'Error de conexión a la base de datos']));
}
$conn->set_charset("utf8mb4");

// Validar parámetros
if (!isset($_POST['promocion_id'])) {
    header("HTTP/1.1 400 Bad Request");
    exit(json_encode(['error' => 'Faltan parámetros requeridos']));
}

$promocion_id = (int)$_POST['promocion_id'];
$postgrado_id = isset($_POST['postgrado_id']) ? (int)$_POST['postgrado_id'] : 0;

// Crear buffer de salida para evitar corrupción del PDF
ob_start();

try {
    // Incluir mPDF
    $mpdfPath = __DIR__ . '/../vendor/autoload.php';
    if (!file_exists($mpdfPath)) {
        throw new Exception("Biblioteca mPDF no encontrada en: $mpdfPath");
    }
    require_once $mpdfPath;

    // Configuración de mPDF (igual que constancia_notas.php)
    $mpdf = new \Mpdf\Mpdf([
        'mode' => 'utf-8',
        'format' => 'Letter', 
        'margin_left' => 15,
        'margin_right' => 15,
        'margin_top' => 15,
        'margin_bottom' => 20,
        'default_font_size' => 10,
        'default_font' => 'arial',
        'tempDir' => __DIR__ . '/tmp'
    ]);

    // Obtener estudiantes con sus postgrados (SOLO GRADUADOS)
    $sql_estudiantes = "SELECT 
                        e.id, e.nombres, e.apellidos, e.cedula,
                        e.tipo_documento, e.cedula_venezolana, e.cedula_extranjera, e.pasaporte,
                        p.id AS postgrado_id, p.nombre AS postgrado_nombre,
                        ep.fecha_graduacion, pr.nombre AS promocion
                        FROM estudiantes e
                        JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                        JOIN postgrados p ON ep.postgrado_id = p.id
                        LEFT JOIN promociones pr ON ep.promocion_id = pr.id
                        WHERE ep.promocion_id = ?
                        AND ep.estado = 'graduado'";
    
    if ($postgrado_id > 0) {
        $sql_estudiantes .= " AND ep.postgrado_id = ?";
    }
    
    $sql_estudiantes .= " ORDER BY e.apellidos, e.nombres";

    $stmt_estudiantes = $conn->prepare($sql_estudiantes);
    if (!$stmt_estudiantes) {
        throw new Exception("Error preparando consulta de estudiantes: " . $conn->error);
    }

    if ($postgrado_id > 0) {
        $stmt_estudiantes->bind_param("ii", $promocion_id, $postgrado_id);
    } else {
        $stmt_estudiantes->bind_param("i", $promocion_id);
    }

    if (!$stmt_estudiantes->execute()) {
        throw new Exception("Error ejecutando consulta de estudiantes: " . $stmt_estudiantes->error);
    }

    $result_estudiantes = $stmt_estudiantes->get_result();
    $estudiantes = $result_estudiantes->fetch_all(MYSQLI_ASSOC);
    $stmt_estudiantes->close();

    if (empty($estudiantes)) {
        throw new Exception("No se encontraron estudiantes graduados para esta promoción");
    }

    // Obtener firmas (igual que en constancia_notas.php)
    $sql_firmas = "SELECT nombre, cargo FROM firmas_constancia ORDER BY id";
    $result_firmas = $conn->query($sql_firmas);

    if ($result_firmas && $result_firmas->num_rows > 0) {
        $firmas = $result_firmas->fetch_all(MYSQLI_ASSOC);
    } else {
        $firmas = [
            ['nombre' => 'Prof. MARIO PATIÑO TORRES', 'cargo' => 'DECANO'],
            ['nombre' => 'Prof. JOSÈ RAMÒN GARCÌA RODRÌGUEZ', 'cargo' => 'COORDINADOR DE ESTUDIOS DE POSTGRADO MED'],
            ['nombre' => 'Ing. ZULAY DÌAZ', 'cargo' => 'ÀREA DE POSTGRADO CENTRAL'],
            ['nombre' => 'JESSE LEE HERNÀNDEZ', 'cargo' => 'CONTROL DE ESTUDIOS DE POSTGRADO MED']
        ];
    }

    // Función para determinar el tipo de documento y obtener el valor
    function obtenerDocumentoIdentidad($estudiante) {
        $tipo_documento = $estudiante['tipo_documento'] ?? 'V';
        $valor_documento = '';
        $etiqueta_documento = '';
        
        switch ($tipo_documento) {
            case 'V':
                $valor_documento = $estudiante['cedula_venezolana'] ?? $estudiante['cedula'];
                $etiqueta_documento = 'CÉDULA';
                break;
            case 'E':
                $valor_documento = $estudiante['cedula_extranjera'] ?? $estudiante['cedula'];
                $etiqueta_documento = 'CÉDULA';
                break;
            case 'P':
                $valor_documento = $estudiante['pasaporte'] ?? $estudiante['cedula'];
                $etiqueta_documento = 'PASAPORTE';
                break;
            default:
                $valor_documento = $estudiante['cedula'];
                $etiqueta_documento = 'CÉDULA';
        }
        
        return [
            'etiqueta' => $etiqueta_documento,
            'valor' => $valor_documento
        ];
    }

    // Función para convertir números a letras
    function numeroALetras($numero) {
        $unidades = ['', 'UNO', 'DOS', 'TRES', 'CUATRO', 'CINCO', 'SEIS', 'SIETE', 'OCHO', 'NUEVE'];
        $decenas = ['', 'DIEZ', 'VEINTE', 'TREINTA', 'CUARENTA', 'CINCUENTA', 'SESENTA', 'SETENTA', 'OCHENTA', 'NOVENTA'];
        $especiales = ['', 'ONCE', 'DOCE', 'TRECE', 'CATORCE', 'QUINCE', 'DIECISEIS', 'DIECISIETE', 'DIECIOCHO', 'DIECINUEVE'];
        
        if ($numero == 0) return 'CERO';
        if ($numero == 10) return 'DIEZ';
        if ($numero == 20) return 'VEINTE';
        
        if ($numero < 10) return $unidades[$numero];
        if ($numero > 10 && $numero < 20) return $especiales[$numero - 10];
        
        $unidad = $numero % 10;
        $decena = floor($numero / 10);
        
        if ($unidad == 0) {
            return $decenas[$decena];
        } else {
            return $decenas[$decena] . ($decena == 2 ? 'I' : '') . $unidades[$unidad];
        }
    }

    // Función para formatear fecha en formato día/mes/año
    function formatearFecha($fecha) {
        if (empty($fecha)) {
            return 'N/A';
        }
        
        try {
            $fechaObj = new DateTime($fecha);
            return $fechaObj->format('d/m/Y');
        } catch (Exception $e) {
            return 'N/A';
        }
    }

    // Procesar cada estudiante
    foreach ($estudiantes as $index => $estudiante) {
        // Obtener información del documento de identidad
        $documento_info = obtenerDocumentoIdentidad($estudiante);

        // Obtener historial académico
        $sql_historial = "SELECT d.codigo AS asignatura, d.nombre AS denominacion, 
                         d.credito, d.tipo, i.nota_definitiva, al.nombre AS año_lectivo
                         FROM inscripciones i
                         INNER JOIN denominaciones d ON i.denominacion_id = d.id
                         LEFT JOIN años_lectivos al ON i.año_lectivo_id = al.id
                         WHERE i.estudiante_id = ? AND d.postgrado_id = ?
                         AND i.nota_definitiva IS NOT NULL
                         ORDER BY d.periodo, d.codigo";

        $stmt_historial = $conn->prepare($sql_historial);
        if (!$stmt_historial) {
            throw new Exception("Error preparando consulta de historial: " . $conn->error);
        }

        $stmt_historial->bind_param("ii", $estudiante['id'], $estudiante['postgrado_id']);
        if (!$stmt_historial->execute()) {
            throw new Exception("Error ejecutando consulta de historial: " . $stmt_historial->error);
        }

        $result_historial = $stmt_historial->get_result();
        $historial_academico = $result_historial->fetch_all(MYSQLI_ASSOC);
        $stmt_historial->close();

        // Obtener total de créditos inscritos por el estudiante
        $sql_creditos_inscritos = "SELECT SUM(d.credito) AS total_creditos_inscritos
                                  FROM inscripciones i
                                  JOIN denominaciones d ON i.denominacion_id = d.id
                                  WHERE i.estudiante_id = ?
                                  AND d.postgrado_id = ?";
        $stmt_creditos_inscritos = $conn->prepare($sql_creditos_inscritos);
        $total_creditos_inscritos = 0;
        if ($stmt_creditos_inscritos) {
            $stmt_creditos_inscritos->bind_param("ii", $estudiante['id'], $estudiante['postgrado_id']);
            $stmt_creditos_inscritos->execute();
            $result_creditos_inscritos = $stmt_creditos_inscritos->get_result();
            $row_creditos_inscritos = $result_creditos_inscritos->fetch_assoc();
            $total_creditos_inscritos = $row_creditos_inscritos['total_creditos_inscritos'] ?? 0;
            $stmt_creditos_inscritos->close();
        }

        // Obtener créditos aprobados
        $sql_creditos_aprobados = "SELECT SUM(d.credito) AS total_creditos_aprobados
                                  FROM inscripciones i
                                  INNER JOIN denominaciones d ON i.denominacion_id = d.id
                                  WHERE i.estudiante_id = ? 
                                  AND d.postgrado_id = ?
                                  AND i.nota_definitiva IS NOT NULL
                                  AND (
                                      (d.tipo = 'puntaje' AND CAST(i.nota_definitiva AS DECIMAL(10,2)) >= 10) OR
                                      (d.tipo = 'Aprobado_Reprobado' AND UPPER(TRIM(i.nota_definitiva)) = 'APROBADO')
                                  )";
        $stmt_creditos_aprobados = $conn->prepare($sql_creditos_aprobados);
        $total_creditos_aprobados = 0;
        if ($stmt_creditos_aprobados) {
            $stmt_creditos_aprobados->bind_param("ii", $estudiante['id'], $estudiante['postgrado_id']);
            $stmt_creditos_aprobados->execute();
            $result_creditos_aprobados = $stmt_creditos_aprobados->get_result();
            $row_creditos_aprobados = $result_creditos_aprobados->fetch_assoc();
            $total_creditos_aprobados = $row_creditos_aprobados['total_creditos_aprobados'] ?? 0;
            $stmt_creditos_aprobados->close();
        }

        // Calcular promedios
        $total_creditos = 0;
        $nota_simple = 0;
        $nota_ponderada = 0;
        $asignaturas_puntaje_aprobadas = 0;

        foreach ($historial_academico as $h) {
            if ($h['tipo'] == 'puntaje' && is_numeric($h['nota_definitiva']) && floatval($h['nota_definitiva']) >= 10) {
                $valor = floatval($h['nota_definitiva']);
                $credito = intval($h['credito']);
                $nota_simple += $valor;
                $nota_ponderada += ($valor * $credito);
                $total_creditos += $credito;
                $asignaturas_puntaje_aprobadas++;
            }
        }

        $nota_simple_promedio = $asignaturas_puntaje_aprobadas > 0 ? $nota_simple / $asignaturas_puntaje_aprobadas : 0;
        $nota_ponderada_promedio = $total_creditos > 0 ? $nota_ponderada / $total_creditos : 0;

        // Usar la fecha de graduación del estudiante (obligatoria para graduados)
        $fecha_graduacion_mostrar = !empty($estudiante['fecha_graduacion']) ? 
                                   formatearFecha($estudiante['fecha_graduacion']) : 
                                   formatearFecha(date('Y-m-d'));

        // Generar HTML para esta constancia (formato de constancia_notas.php)
        $denominaciones_por_pagina = 25;
        $total_paginas_denominaciones = ceil(count($historial_academico) / $denominaciones_por_pagina);
        $total_paginas = $total_paginas_denominaciones + 1; // +1 para la página de resumen

        $html = '<!DOCTYPE html>
        <html lang="es">
        <head>
            <meta charset="UTF-8">
            <title>Constancia de Notas</title>
            <style>
                body { font-family: Arial, sans-serif; margin: 0; padding: 0; }
                .constancia { width: 100%; padding: 20px; box-sizing: border-box; }
                .header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px; }
                .header h1 { font-size: 14px; margin: 0; text-align: left; line-height: 1.2; }
                .header h2 { font-size: 12px; margin: 0; text-align: right; line-height: 1.2; }
                .datos-estudiante { margin-bottom: 15px; text-align: left; }
                .datos-estudiante p { margin: 3px 0; font-size: 12px; line-height: 1.2; }
                .linea-negra { border-top: 2px solid #000; margin: 15px 0; }
                table { width: 100%; border-collapse: collapse; margin-top: 8px; page-break-inside: avoid; }
                th, td { padding: 4px; text-align: left; font-size: 12px; border: 1px solid #ddd; line-height: 1.2; }
                th { background-color: #f4f4f4; }
                .resumen-tabla { 
                    margin: 50px 0 0 40px;
                    width: 65%;
                    border: none;
                }
                .resumen-tabla td { 
                    border: none; 
                    padding: 4px 0px;
                    font-size: 12px; 
                    vertical-align: top; 
                    line-height: 1.1;
                }
                .resumen-label { 
                    text-align: left; 
                    width: 50%;
                    padding-right: 1px;
                }
                .resumen-value { 
                    text-align: right; 
                    width: 15%;
                    font-weight: bold; 
                    padding-left: 1px;
                }
                .firmas { margin-top: 120px; text-align: center; }
                .firmas table { width: 100%; border-collapse: collapse; border: none; }
                .firmas td { width: 50%; text-align: center; vertical-align: top; padding: 60px 40px; border: none; }
                .firmas p { margin: 0; font-size: 12px; line-height: 1.5; }
                .page-break { page-break-after: always; }
                .avoid-break { page-break-inside: avoid; }
            </style>
        </head>
        <body>';

        // Primero generamos todas las páginas de asignaturas
        for ($pagina = 1; $pagina <= $total_paginas_denominaciones; $pagina++) {
            $html .= '
            <div class="constancia' . ($pagina < $total_paginas_denominaciones ? ' page-break' : '') . '">
                <div class="header">
                    <h1>UNIVERSIDAD CENTRAL DE VENEZUELA<br>OFICINA CENTRAL DE CONTROL DE ESTUDIOS<br>ÁREA DE POSTGRADO</h1>
                    <h2>FECHA DE GRADUACIÓN: ' . $fecha_graduacion_mostrar . '<br>Página ' . $pagina . ' de ' . $total_paginas . '</h2>
                </div>

                <div class="datos-estudiante">
                    <h3 style="text-align: center;">*** FICHA DE GRADO ***</h3>
                    <p><strong>FACULTAD DE MEDICINA</strong></p>
                    <p><strong>POSTGRADO DE MEDICINA</strong></p>
                    <p>&nbsp;</p>
                    <p><strong>ESPECIALISTA EN:</strong> ' . htmlspecialchars($estudiante['postgrado_nombre']) . '</p>
                    <p><strong>ESTUDIANTE:</strong> ' . htmlspecialchars($estudiante['nombres'] . ' ' . $estudiante['apellidos']) . '</p>
                    <p><strong>' . $documento_info['etiqueta'] . ':</strong> ' . htmlspecialchars($documento_info['valor']) . '</p>
                </div>

                <div class="linea-negra"></div>

                <table>
                    <thead>
                        <tr>
                            <th>ASIGNATURA</th>
                            <th>DENOMINACIÓN</th>
                            <th>CRÉDITO</th>
                            <th>NOTA DEFINITIVA</th>
                            <th>AÑO LECTIVO</th>
                        </tr>
                    </thead>
                    <tbody>';

            $inicio = ($pagina - 1) * $denominaciones_por_pagina;
            $denominaciones_pagina = array_slice($historial_academico, $inicio, $denominaciones_por_pagina);

            foreach ($denominaciones_pagina as $historial) {
                $nota_display = 'N/A';
                
                if ($historial['tipo'] == 'Aprobado_Reprobado') {
                    $estado = strtoupper($historial['nota_definitiva']);
                    $letra = ($estado == 'APROBADO') ? 'A' : 'R';
                    $nota_display = $letra . ' (' . $estado . ')';
                } elseif (is_numeric($historial['nota_definitiva'])) {
                    $nota_num = (int)$historial['nota_definitiva'];
                    $nota_display = $nota_num . ' (' . numeroALetras($nota_num) . ')';
                }
                
                $html .= '
                        <tr>
                            <td>' . htmlspecialchars($historial['asignatura']) . '</td>
                            <td>' . htmlspecialchars($historial['denominacion']) . '</td>
                            <td>' . htmlspecialchars($historial['credito']) . '</td>
                            <td>' . $nota_display . '</td>
                            <td>' . htmlspecialchars($historial['año_lectivo']) . '</td>
                        </tr>';
            }

            $html .= '
                    </tbody>
                </table>
            </div>';
        }

        // Ahora generamos la página final con el resumen y firmas
        // Solo agregamos salto de página si hay páginas de asignaturas anteriores
        if ($total_paginas_denominaciones > 0) {
            $html .= '<div class="page-break"></div>';
        }

        $html .= '
        <div class="constancia avoid-break">
            <div class="header">
                <h1>UNIVERSIDAD CENTRAL DE VENEZUELA<br>OFICINA CENTRAL DE CONTROL DE ESTUDIOS<br>ÁREA DE POSTGRADO</h1>
                <h2>FECHA DE GRADUACIÓN: ' . $fecha_graduacion_mostrar . '<br>Página ' . $total_paginas . ' de ' . $total_paginas . '</h2>
            </div>

            <div class="datos-estudiante">
                <h3 style="text-align: center;">*** FICHA DE GRADO ***</h3>
                <p><strong>FACULTAD DE MEDICINA</strong></p>
                <p><strong>POSTGRADO DE MEDICINA</strong></p>
                <p>&nbsp;</p>
                <p><strong>ESPECIALISTA EN:</strong> ' . htmlspecialchars($estudiante['postgrado_nombre']) . '</p>
                <p><strong>ESTUDIANTE:</strong> ' . htmlspecialchars($estudiante['nombres'] . ' ' . $estudiante['apellidos']) . '</p>
                <p><strong>' . $documento_info['etiqueta'] . ':</strong> ' . htmlspecialchars($documento_info['valor']) . '</p>
            </div>

            <div class="avoid-break">
                <h3 style="text-align: center; margin-bottom: 20px;">RESUMEN ACADÉMICO</h3>
                <table class="resumen-tabla">
                    <tr>
                        <td class="resumen-label">NRO. DE UNIDADES INSCRITAS</td>
                        <td class="resumen-value">' . $total_creditos_inscritos . '</td>
                    </tr>
                    <tr>
                        <td class="resumen-label">NRO. DE UNIDADES APROBADAS</td>
                        <td class="resumen-value">' . $total_creditos_aprobados . '</td>
                    </tr>
                    <tr>
                        <td class="resumen-label">NRO. DE UNIDADES APROBADAS POR RECONOC. CRED</td>
                        <td class="resumen-value">0</td>
                    </tr>
                    <tr>
                        <td class="resumen-label">NRO. DE UNIDADES APROBADAS POR ASIG.CRED</td>
                        <td class="resumen-value">0</td>
                    </tr>
                    <tr>
                        <td class="resumen-label">PROMEDIO GENERAL DE NOTAS APROBADAS</td>
                        <td class="resumen-value">' . number_format($nota_simple_promedio, 2) . '</td>
                    </tr>
                    <tr>
                        <td class="resumen-label">PROMEDIO PONDERADO DE NOTAS APROBADAS</td>
                        <td class="resumen-value">' . number_format($nota_ponderada_promedio, 2) . '</td>
                    </tr>
                </table>
            </div>

            <div class="firmas avoid-break">
                <table>';

        foreach (array_chunk($firmas, 2) as $fila) {
            $html .= '<tr>';
            foreach ($fila as $firma) {
                $html .= '
                    <td>
                        <p><strong>' . htmlspecialchars($firma['nombre']) . '</strong></p>
                        <p>' . htmlspecialchars($firma['cargo']) . '</p>
                    </td>';
            }
            $html .= '</tr>';
        }

        $html .= '
                </table>
            </div>
        </div>';

        $html .= '
        </body>
        </html>';

        // Agregar página al PDF
        $mpdf->WriteHTML($html);
        
        // NO agregar ningún tipo de separador de página - dejar que mPDF maneje automáticamente
        // Esto evita páginas en blanco antes de cada constancia
    }

    // Generar nombre del archivo
    $filename = 'constancias_notas_promocion_' . $promocion_id;
    if ($postgrado_id > 0) {
        $filename .= '_postgrado_' . $postgrado_id;
    }
    $filename .= '_' . date('Ymd_His') . '.pdf';
    
    // Limpiar buffer de salida antes de enviar el PDF
    ob_end_clean();
    
    // Enviar headers y contenido
    header('Content-Type: application/pdf');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Cache-Control: private, max-age=0, must-revalidate');
    header('Pragma: public');
    
    // Enviar el PDF
    $mpdf->Output($filename, \Mpdf\Output\Destination::DOWNLOAD);

} catch (Exception $e) {
    ob_end_clean();
    header("HTTP/1.1 500 Internal Server Error");
    exit(json_encode(['error' => 'Error al generar constancias: ' . $e->getMessage()]));
} finally {
    if (isset($conn)) {
        $conn->close();
    }
}