<?php
session_start();
if (!isset($_SESSION['username']) || $_SESSION['rol'] != 'administrador') {
    header("Location: ../login.php");
    exit();
}

include '../includes/db.php';

$error_message = '';
$success_message = '';

// Obtener el ID del usuario a editar
$id = isset($_GET['id']) ? intval($_GET['id']) : 0;

if ($id <= 0) {
    header("Location: gestion_usuarios.php");
    exit();
}

// Obtener datos actuales del usuario - CONSULTA ACTUALIZADA CON NUEVOS CAMPOS
$sql = "SELECT u.id, u.username, u.rol, u.email, u.telefono, u.direccion, u.estado_cuenta, 
               u.estudiante_id, u.nombres, u.apellidos,
               e.cedula, e.nombres as nombres_est, e.apellidos as apellidos_est, 
               e.correo as correo_est, e.telefono as telefono_est, e.direccion as direccion_est,
               e.tipo_documento, e.cedula_venezolana, e.cedula_extranjera, e.pasaporte
        FROM usuarios u 
        LEFT JOIN estudiantes e ON u.estudiante_id = e.id 
        WHERE u.id = ?";
$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    header("Location: gestion_usuarios.php");
    exit();
}

$usuario = $result->fetch_assoc();
$stmt->close();

// Procesar el formulario de edición
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['editar_usuario'])) {
    $username = trim($_POST['username']);
    $rol = $_POST['rol'];
    $estado_cuenta = $_POST['estado_cuenta'];
    $password = $_POST['password'] ?: null;
    
    // Datos para la tabla correspondiente
    $nombres = isset($_POST['nombres']) ? trim($_POST['nombres']) : null;
    $apellidos = isset($_POST['apellidos']) ? trim($_POST['apellidos']) : null;
    $email = isset($_POST['email']) ? trim($_POST['email']) : null;
    $telefono = isset($_POST['telefono']) ? trim($_POST['telefono']) : null;
    $direccion = isset($_POST['direccion']) ? trim($_POST['direccion']) : null;

    // Validaciones básicas
    if (empty($username)) {
        $error_message = "El nombre de usuario es obligatorio.";
    } else {
        // Verificar que el username no esté duplicado (excluyendo el actual)
        $stmt = $conn->prepare("SELECT id FROM usuarios WHERE username = ? AND id != ?");
        $stmt->bind_param("si", $username, $id);
        $stmt->execute();
        $stmt->store_result();
        
        if ($stmt->num_rows > 0) {
            $error_message = "El nombre de usuario ya está en uso.";
        } else {
            $stmt->close();

            // Iniciar transacción para asegurar consistencia
            $conn->begin_transaction();

            try {
                // Si el usuario es estudiante, actualizar también la tabla estudiantes
                if ($rol === 'estudiante' && $usuario['estudiante_id']) {
                    $sql_update_estudiante = "UPDATE estudiantes SET cedula=?, nombres=?, apellidos=?, correo=?, telefono=?, direccion=? WHERE id=?";
                    $stmt_estudiante = $conn->prepare($sql_update_estudiante);
                    $stmt_estudiante->bind_param("ssssssi", $username, $nombres, $apellidos, $email, $telefono, $direccion, $usuario['estudiante_id']);
                    
                    if (!$stmt_estudiante->execute()) {
                        throw new Exception("Error al actualizar datos del estudiante: " . $stmt_estudiante->error);
                    }
                    $stmt_estudiante->close();
                }

                // Actualizar tabla usuarios
                if ($password) {
                    $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                    $sql_update_usuario = "UPDATE usuarios SET username=?, rol=?, email=?, telefono=?, direccion=?, nombres=?, apellidos=?, estado_cuenta=?, password=? WHERE id=?";
                    $stmt_usuario = $conn->prepare($sql_update_usuario);
                    $stmt_usuario->bind_param("sssssssssi", $username, $rol, $email, $telefono, $direccion, $nombres, $apellidos, $estado_cuenta, $hashed_password, $id);
                } else {
                    $sql_update_usuario = "UPDATE usuarios SET username=?, rol=?, email=?, telefono=?, direccion=?, nombres=?, apellidos=?, estado_cuenta=? WHERE id=?";
                    $stmt_usuario = $conn->prepare($sql_update_usuario);
                    // CORRECCIÓN: Cambiado de "sssssssi" a "ssssssssi" para 9 parámetros
                    $stmt_usuario->bind_param("ssssssssi", $username, $rol, $email, $telefono, $direccion, $nombres, $apellidos, $estado_cuenta, $id);
                }

                if (!$stmt_usuario->execute()) {
                    throw new Exception("Error al actualizar usuario: " . $stmt_usuario->error);
                }
                $stmt_usuario->close();

                $conn->commit();
                $success_message = "Usuario actualizado correctamente.";
                
                // Recargar los datos del usuario
                $stmt = $conn->prepare($sql);
                $stmt->bind_param("i", $id);
                $stmt->execute();
                $result = $stmt->get_result();
                $usuario = $result->fetch_assoc();
                $stmt->close();

            } catch (Exception $e) {
                $conn->rollback();
                $error_message = $e->getMessage();
            }
        }
    }
}

$page_title = "Editar Usuario";
include '../includes/header.php';
?>

<div class="container-fluid">
    <div class="mb-4">
        <a href="gestion_usuarios.php" class="btn btn-secondary">Volver a Gestión de Usuarios</a>
    </div>

    <?php if ($error_message): ?>
        <div class="alert alert-danger"><?php echo $error_message; ?></div>
    <?php endif; ?>

    <?php if ($success_message): ?>
        <div class="alert alert-success"><?php echo $success_message; ?></div>
    <?php endif; ?>

    <div class="card">
        <div class="card-header bg-primary text-white">
            <h3 class="mb-0">Editar Usuario</h3>
        </div>
        <div class="card-body">
            <form method="POST" action="">
                <div class="row">
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="username" class="form-label">Nombre de Usuario *</label>
                            <input type="text" class="form-control" id="username" name="username" value="<?php echo htmlspecialchars($usuario['username']); ?>" required>
                            <div class="form-text">Para estudiantes, debe ser la cédula.</div>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="rol" class="form-label">Rol *</label>
                            <select class="form-select" id="rol" name="rol" required>
                                <option value="administrador" <?php echo $usuario['rol'] == 'administrador' ? 'selected' : ''; ?>>Administrador</option>
                                <option value="secretaria" <?php echo $usuario['rol'] == 'secretaria' ? 'selected' : ''; ?>>Secretaria</option>
                                <option value="coordinador1" <?php echo $usuario['rol'] == 'coordinador1' ? 'selected' : ''; ?>>Coordinador 1</option>
                                <option value="coordinador2" <?php echo $usuario['rol'] == 'coordinador2' ? 'selected' : ''; ?>>Coordinador 2</option>
                                <option value="estudiante" <?php echo $usuario['rol'] == 'estudiante' ? 'selected' : ''; ?>>Estudiante</option>
                            </select>
                        </div>
                    </div>
                </div>

                <!-- Datos personales -->
                <h5 class="mt-4 mb-3">Datos Personales</h5>
                <div class="row">
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="nombres" class="form-label">Nombres</label>
                            <input type="text" class="form-control" id="nombres" name="nombres" 
                                   value="<?php echo $usuario['rol'] === 'estudiante' && !empty($usuario['nombres_est']) ? htmlspecialchars($usuario['nombres_est']) : htmlspecialchars($usuario['nombres']); ?>">
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="apellidos" class="form-label">Apellidos</label>
                            <input type="text" class="form-control" id="apellidos" name="apellidos" 
                                   value="<?php echo $usuario['rol'] === 'estudiante' && !empty($usuario['apellidos_est']) ? htmlspecialchars($usuario['apellidos_est']) : htmlspecialchars($usuario['apellidos']); ?>">
                        </div>
                    </div>
                </div>

                <div class="row">
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="email" class="form-label">Correo Electrónico</label>
                            <input type="email" class="form-control" id="email" name="email" 
                                   value="<?php echo $usuario['rol'] === 'estudiante' && !empty($usuario['correo_est']) ? htmlspecialchars($usuario['correo_est']) : htmlspecialchars($usuario['email']); ?>">
                            <div class="form-text">Importante para la verificación de cuenta</div>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="telefono" class="form-label">Teléfono</label>
                            <input type="text" class="form-control" id="telefono" name="telefono" 
                                   value="<?php echo $usuario['rol'] === 'estudiante' && !empty($usuario['telefono_est']) ? htmlspecialchars($usuario['telefono_est']) : htmlspecialchars($usuario['telefono']); ?>">
                        </div>
                    </div>
                </div>

                <div class="mb-3">
                    <label for="direccion" class="form-label">Dirección</label>
                    <input type="text" class="form-control" id="direccion" name="direccion" 
                           value="<?php echo $usuario['rol'] === 'estudiante' && !empty($usuario['direccion_est']) ? htmlspecialchars($usuario['direccion_est']) : htmlspecialchars($usuario['direccion']); ?>">
                </div>

                <div class="mb-3">
                    <label for="estado_cuenta" class="form-label">Estado de Cuenta</label>
                    <select class="form-select" id="estado_cuenta" name="estado_cuenta">
                        <option value="activa" <?php echo ($usuario['estado_cuenta'] == 'activa') ? 'selected' : ''; ?>>Activa</option>
                        <option value="pendiente" <?php echo ($usuario['estado_cuenta'] == 'pendiente') ? 'selected' : ''; ?>>Pendiente</option>
                        <option value="inactiva" <?php echo ($usuario['estado_cuenta'] == 'inactiva') ? 'selected' : ''; ?>>Inactiva</option>
                        <option value="bloqueada" <?php echo ($usuario['estado_cuenta'] == 'bloqueada') ? 'selected' : ''; ?>>Bloqueada</option>
                    </select>
                </div>

                <div class="mb-3">
                    <label for="password" class="form-label">Nueva Contraseña</label>
                    <input type="password" class="form-control" id="password" name="password">
                    <div class="form-text">Dejar en blanco para mantener la contraseña actual.</div>
                </div>

                <button type="submit" name="editar_usuario" class="btn btn-primary">Actualizar Usuario</button>
                <a href="gestion_usuarios.php" class="btn btn-secondary">Cancelar</a>
            </form>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const rolSelect = document.getElementById('rol');
    
    rolSelect.addEventListener('change', function() {
        // Si cambia a estudiante, mostrar advertencia sobre la cédula
        if (this.value === 'estudiante') {
            if (!confirm('Al cambiar a rol estudiante, asegúrese de que el nombre de usuario sea la cédula del estudiante.')) {
                this.value = '<?php echo $usuario["rol"]; ?>';
            }
        }
    });
});
</script>

<?php
include '../includes/footer.php';
$conn->close();
?>