<?php
session_start();

// Configurar encoding UTF-8
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');

if (!isset($_SESSION['username'])) {
    header("Location: ../login.php");
    exit();
}

include '../includes/db.php';

// Configurar conexión para usar UTF-8
$conn->set_charset("utf8mb4");
$conn->query("SET NAMES 'utf8mb4'");
$conn->query("SET CHARACTER SET utf8mb4");
$conn->query("SET COLLATION_CONNECTION = 'utf8mb4_unicode_ci'");

// Verificar si se proporcionó un ID de inscripción
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    header("Location: gestion_inscripciones.php");
    exit;
}

$inscripcion_id = intval($_GET['id']);

// Obtener información principal de la inscripción
$sql_inscripcion = "SELECT ep.id, e.cedula, e.nombres, e.apellidos, e.correo, e.telefono, e.celular, 
                   p.nombre as postgrado, ep.fecha_inscripcion, ep.fecha_graduacion, 
                   ep.estado, pr.nombre as promocion, ep.postgrado_id, ep.estudiante_id
                   FROM estudiante_postgrado ep
                   JOIN estudiantes e ON ep.estudiante_id = e.id
                   JOIN postgrados p ON ep.postgrado_id = p.id
                   LEFT JOIN promociones pr ON ep.promocion_id = pr.id
                   WHERE ep.id = ?";
$stmt_inscripcion = $conn->prepare($sql_inscripcion);
$stmt_inscripcion->bind_param("i", $inscripcion_id);
$stmt_inscripcion->execute();
$result_inscripcion = $stmt_inscripcion->get_result();

if ($result_inscripcion->num_rows == 0) {
    header("Location: gestion_inscripciones.php");
    exit;
}

$inscripcion = $result_inscripcion->fetch_assoc();

// Obtener todas las asignaturas inscritas para este postgrado
$sql_asignaturas = "SELECT i.id, d.codigo, d.nombre, d.credito, d.periodo, 
                   a.nombre as año_lectivo, e.nombre as escolaridad,
                   i.nota_final, i.nota_definitiva, i.estado, i.fecha_inscripcion
                   FROM inscripciones i
                   JOIN denominaciones d ON i.denominacion_id = d.id
                   JOIN años_lectivos a ON i.año_lectivo_id = a.id
                   JOIN escolaridades e ON i.escolaridad_id = e.id
                   WHERE i.estudiante_id = ? AND d.postgrado_id = ?
                   ORDER BY d.periodo, d.codigo";
$stmt_asignaturas = $conn->prepare($sql_asignaturas);
$stmt_asignaturas->bind_param("ii", $inscripcion['estudiante_id'], $inscripcion['postgrado_id']);
$stmt_asignaturas->execute();
$result_asignaturas = $stmt_asignaturas->get_result();
$asignaturas = $result_asignaturas->fetch_all(MYSQLI_ASSOC);

// Calcular créditos aprobados y totales
$creditos_aprobados = 0;
$creditos_totales = 0;
$asignaturas_aprobadas = 0;
$asignaturas_totales = count($asignaturas);

foreach ($asignaturas as $asig) {
    $creditos_totales += $asig['credito'];
    if ($asig['estado'] == 'finalizada' && ($asig['nota_definitiva'] >= 10 || $asig['nota_definitiva'] == 'Aprobado')) {
        $creditos_aprobados += $asig['credito'];
        $asignaturas_aprobadas++;
    }
}

// Obtener información de suspensiones si existen
$sql_suspensiones = "SELECT * FROM suspensiones WHERE estudiante_id = ? ORDER BY fecha_inicio";
$stmt_suspensiones = $conn->prepare($sql_suspensiones);
$stmt_suspensiones->bind_param("i", $inscripcion['estudiante_id']);
$stmt_suspensiones->execute();
$result_suspensiones = $stmt_suspensiones->get_result();
$suspensiones = $result_suspensiones->fetch_all(MYSQLI_ASSOC);

include '../includes/header.php';
?>

<div class="container">
    <!-- Botón para volver -->
    <div class="mb-4">
        <a href="gestion_inscripciones.php" class="btn btn-secondary">
            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-arrow-left" viewBox="0 0 16 16">
                <path fill-rule="evenodd" d="M15 8a.5.5 0 0 0-.5-.5H2.707l3.147-3.146a.5.5 0 1 0-.708-.708l-4 4a.5.5 0 0 0 0 .708l4 4a.5.5 0 0 0 .708-.708L2.707 8.5H14.5A.5.5 0 0 0 15 8z"/>
            </svg>
            Volver a Gestión de Inscripciones
        </a>
    </div>

    <h2 class="my-4">Detalle de Inscripción</h2>
    
    <!-- Información del estudiante -->
    <div class="card mb-4">
        <div class="card-header bg-primary text-white">
            <h3 class="mb-0">Información del Estudiante</h3>
        </div>
        <div class="card-body">
            <div class="row">
                <div class="col-md-6">
                    <p><strong>Cédula:</strong> <?= htmlspecialchars($inscripcion['cedula']) ?></p>
                    <p><strong>Nombres:</strong> <?= htmlspecialchars($inscripcion['nombres']) ?></p>
                    <p><strong>Apellidos:</strong> <?= htmlspecialchars($inscripcion['apellidos']) ?></p>
                </div>
                <div class="col-md-6">
                    <p><strong>Correo:</strong> <?= htmlspecialchars($inscripcion['correo']) ?></p>
                    <p><strong>Teléfono:</strong> <?= htmlspecialchars($inscripcion['telefono'] ?? '') ?></p>
                    <p><strong>Celular:</strong> <?= htmlspecialchars($inscripcion['celular'] ?? '') ?></p>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Información de la inscripción -->
    <div class="card mb-4">
        <div class="card-header bg-info text-white">
            <h3 class="mb-0">Información del Postgrado</h3>
        </div>
        <div class="card-body">
            <div class="row">
                <div class="col-md-4">
                    <p><strong>Postgrado:</strong> <?= htmlspecialchars($inscripcion['postgrado']) ?></p>
                    <p><strong>Promoción:</strong> <?= $inscripcion['promocion'] ? htmlspecialchars($inscripcion['promocion']) : 'No asignada' ?></p>
                </div>
                <div class="col-md-4">
                    <p><strong>Fecha de Inscripción:</strong> <?= $inscripcion['fecha_inscripcion'] ?></p>
                    <p><strong>Estado:</strong> 
                        <span class="badge bg-<?= 
                            $inscripcion['estado'] == 'activo' ? 'success' : 
                            ($inscripcion['estado'] == 'graduado' ? 'info' : 
                            ($inscripcion['estado'] == 'suspendido' ? 'warning' : 'secondary')) ?>">
                            <?= ucfirst($inscripcion['estado']) ?>
                        </span>
                    </p>
                </div>
                <div class="col-md-4">
                    <?php if ($inscripcion['fecha_graduacion']): ?>
                        <p><strong>Fecha de Graduación:</strong> <?= $inscripcion['fecha_graduacion'] ?></p>
                    <?php endif; ?>
                    <p><strong>Progreso:</strong> 
                        <?php if ($asignaturas_totales > 0): ?>
                            <?= $asignaturas_aprobadas ?> de <?= $asignaturas_totales ?> asignaturas 
                            (<?= round(($asignaturas_aprobadas/$asignaturas_totales)*100, 2) ?>%)
                        <?php else: ?>
                            No hay asignaturas registradas
                        <?php endif; ?>
                    </p>
                    <p><strong>Créditos:</strong> 
                        <?php if ($creditos_totales > 0): ?>
                            <?= $creditos_aprobados ?> de <?= $creditos_totales ?> créditos 
                            (<?= round(($creditos_aprobados/$creditos_totales)*100, 2) ?>%)
                        <?php else: ?>
                            No hay créditos registrados
                        <?php endif; ?>
                    </p>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Suspensiones (si existen) -->
    <?php if (!empty($suspensiones)): ?>
    <div class="card mb-4">
        <div class="card-header bg-warning text-white">
            <h3 class="mb-0">Historial de Suspensiones</h3>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-striped">
                    <thead>
                        <tr>
                            <th>Fecha Inicio</th>
                            <th>Fecha Fin</th>
                            <th>Motivo</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($suspensiones as $susp): ?>
                            <tr>
                                <td><?= $susp['fecha_inicio'] ?></td>
                                <td><?= $susp['fecha_fin'] ?></td>
                                <td><?= htmlspecialchars($susp['motivo']) ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    <?php endif; ?>
    
    <!-- Asignaturas inscritas -->
    <div class="card">
        <div class="card-header bg-secondary text-white">
            <h3 class="mb-0">Asignaturas Inscritas</h3>
        </div>
        <div class="card-body">
            <?php if (count($asignaturas) > 0): ?>
            <div class="table-responsive">
                <table class="table table-striped">
                    <thead>
                        <tr>
                            <th>Código</th>
                            <th>Asignatura</th>
                            <th>Créditos</th>
                            <th>Periodo</th>
                            <th>Año Lectivo</th>
                            <th>Escolaridad</th>
                            <th>Nota Final</th>
                            <th>Estado</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($asignaturas as $asig): ?>
                            <tr>
                                <td><?= htmlspecialchars($asig['codigo']) ?></td>
                                <td><?= htmlspecialchars($asig['nombre']) ?></td>
                                <td><?= $asig['credito'] ?></td>
                                <td><?= $asig['periodo'] ?></td>
                                <td><?= htmlspecialchars($asig['año_lectivo']) ?></td>
                                <td><?= htmlspecialchars($asig['escolaridad']) ?></td>
                                <td>
                                    <?php if ($asig['estado'] == 'finalizada'): ?>
                                        <span class="badge bg-<?= 
                                            ($asig['nota_definitiva'] >= 10 || $asig['nota_definitiva'] == 'Aprobado') ? 'success' : 'danger' ?>">
                                            <?= htmlspecialchars($asig['nota_definitiva']) ?>
                                        </span>
                                    <?php else: ?>
                                        <span class="badge bg-secondary">En curso</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <span class="badge bg-<?= $asig['estado'] == 'finalizada' ? 'info' : 'primary' ?>">
                                        <?= ucfirst($asig['estado']) ?>
                                    </span>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            <?php else: ?>
                <div class="alert alert-info">No se encontraron asignaturas registradas para esta inscripción.</div>
            <?php endif; ?>
        </div>
    </div>
</div>

<?php
include '../includes/footer.php';
$conn->close();
?>