<?php
session_start();

// Configurar encoding UTF-8
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');

if (!isset($_SESSION['username'])) {
    header("Location: ../login.php");
    exit();
}

include '../includes/db.php';

// Configurar conexión para usar UTF-8
$conn->set_charset("utf8mb4");
$conn->query("SET NAMES 'utf8mb4'");
$conn->query("SET CHARACTER SET utf8mb4");
$conn->query("SET COLLATION_CONNECTION = 'utf8mb4_unicode_ci'");

if ($conn->connect_error) {
    die("Error de conexión: " . $conn->connect_error);
}

// Habilitar reportes de error para depuración
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

$estudiante_id = isset($_GET['estudiante_id']) ? intval($_GET['estudiante_id']) : 0;
$postgrado_seleccionado = isset($_GET['postgrado_id']) ? intval($_GET['postgrado_id']) : 0;

// Verificar que el postgrado seleccionado exista
$sql_verificar_postgrado = "SELECT COUNT(*) as count FROM postgrados WHERE id = ?";
$stmt_verificar = $conn->prepare($sql_verificar_postgrado);
$stmt_verificar->bind_param("i", $postgrado_seleccionado);
$stmt_verificar->execute();
$result_verificar = $stmt_verificar->get_result()->fetch_assoc();

if ($result_verificar['count'] == 0) {
    $_SESSION['error'] = "El postgrado seleccionado no existe.";
    header("Location: cargar_notas.php");
    exit();
}

// Obtener información del estudiante
$sql_estudiante_info = "SELECT e.*, ep.estado as estado_postgrado, p.nombre as postgrado_nombre
                       FROM estudiantes e
                       JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                       JOIN postgrados p ON ep.postgrado_id = p.id
                       WHERE e.id = ? AND ep.postgrado_id = ?";
$stmt_estudiante_info = $conn->prepare($sql_estudiante_info);
$stmt_estudiante_info->bind_param("ii", $estudiante_id, $postgrado_seleccionado);
$stmt_estudiante_info->execute();
$estudiante_info = $stmt_estudiante_info->get_result()->fetch_assoc();

if (!$estudiante_info) {
    $_SESSION['error'] = "No se encontró al estudiante o no pertenece al postgrado seleccionado.";
    header("Location: cargar_notas.php?postgrado_id=".$postgrado_seleccionado);
    exit();
}

// FUNCIÓN PARA OBTENER EL NÚMERO DE DOCUMENTO SEGÚN EL TIPO
function obtener_numero_documento($tipo_documento, $cedula_venezolana, $cedula_extranjera, $pasaporte) {
    switch ($tipo_documento) {
        case 'V':
            return $cedula_venezolana;
        case 'E':
            return $cedula_extranjera;
        case 'P':
            return $pasaporte;
        default:
            return $cedula_venezolana;
    }
}

// FUNCIÓN PARA OBTENER EL TIPO DE DOCUMENTO EN TEXTO
function obtener_tipo_documento_texto($tipo_documento) {
    switch ($tipo_documento) {
        case 'V':
            return 'Cédula Venezolana';
        case 'E':
            return 'Cédula Extranjera';
        case 'P':
            return 'Pasaporte';
        default:
            return 'Cédula Venezolana';
    }
}

// Obtener el número de documento del estudiante
$numero_documento = obtener_numero_documento(
    $estudiante_info['tipo_documento'], 
    $estudiante_info['cedula_venezolana'], 
    $estudiante_info['cedula_extranjera'], 
    $estudiante_info['pasaporte']
);
$tipo_documento_texto = obtener_tipo_documento_texto($estudiante_info['tipo_documento']);

// Obtener todos los años lectivos para este postgrado
$sql_anios_lectivos = "SELECT DISTINCT al.id, al.nombre 
                      FROM anios_lectivos al
                      JOIN inscripciones i ON al.id = i.año_lectivo_id
                      JOIN estudiante_postgrado ep ON i.estudiante_id = ep.estudiante_id
                      WHERE ep.postgrado_id = ?
                      ORDER BY al.nombre DESC";
$stmt_anios = $conn->prepare($sql_anios_lectivos);
$stmt_anios->bind_param("i", $postgrado_seleccionado);
$stmt_anios->execute();
$anios_lectivos_permitidos = $stmt_anios->get_result()->fetch_all(MYSQLI_ASSOC);

// Convertir a formato para consulta SQL
$anios_ids = array_column($anios_lectivos_permitidos, 'id');

// Obtener denominaciones y notas para el cálculo de promedios
$denominaciones = array();
$periodos = array();
$totales = array(
    'total_materias' => 0, 
    'total_materias_puntaje' => 0, 
    'total_materias_cualitativas' => 0,
    'pendientes_puntaje' => 0, 
    'pendientes_cualitativas' => 0, 
    'aprobadas_puntaje' => 0,
    'aprobadas_cualitativas' => 0, 
    'sin_nota_puntaje' => 0, 
    'sin_nota_cualitativas' => 0,
    'reprobadas_puntaje' => 0, 
    'reprobadas_cualitativas' => 0,
    'promedio_simple' => 0, 
    'promedio_ponderado' => 0,
    'suma_notas_simples' => 0, 
    'suma_notas_ponderadas' => 0, 
    'suma_creditos_aprobados' => 0
);

// Array para almacenar promedios por periodo
$promedios_por_periodo = array();

if (!empty($anios_ids)) {
    // Consulta mejorada para incluir escolaridad y año lectivo
    $sql_denominaciones = "SELECT d.id, d.nombre, d.codigo, d.tipo, d.periodo, d.credito,
                          i.id as id_inscripcion, i.nota_final, i.nota_definitiva, i.tipo_evaluacion, i.estado,
                          i.año_lectivo_id, al.nombre as anio_lectivo_nombre,
                          i.escolaridad_id, esc.nombre as escolaridad_nombre,
                          i.fecha_modificacion, u.username as modificado_por
                          FROM denominaciones d
                          JOIN inscripciones i ON d.id = i.denominacion_id
                          JOIN anios_lectivos al ON i.año_lectivo_id = al.id
                          JOIN escolaridades esc ON i.escolaridad_id = esc.id
                          LEFT JOIN usuarios u ON i.usuario_modificacion = u.id
                          WHERE i.estudiante_id = ? AND d.postgrado_id = ?
                          AND i.año_lectivo_id IN (" . implode(",", $anios_ids) . ")
                          ORDER BY d.periodo, d.codigo";
    $stmt_denominaciones = $conn->prepare($sql_denominaciones);
    $stmt_denominaciones->bind_param("ii", $estudiante_id, $postgrado_seleccionado);
    $stmt_denominaciones->execute();
    $denominaciones = $stmt_denominaciones->get_result()->fetch_all(MYSQLI_ASSOC);
    
    foreach ($denominaciones as $denominacion) {
        $periodo = $denominacion['periodo'];
        if (!isset($periodos[$periodo])) {
            $periodos[$periodo] = array();
        }
        $periodos[$periodo][] = $denominacion;
        
        // Inicializar promedios por periodo si no existen
        if (!isset($promedios_por_periodo[$periodo])) {
            $promedios_por_periodo[$periodo] = array(
                'suma_notas_simples' => 0,
                'suma_notas_ponderadas' => 0,
                'suma_creditos_aprobados' => 0,
                'aprobadas_puntaje' => 0,
                'sin_nota_puntaje' => 0,
                'reprobadas_puntaje' => 0,
                'promedio_simple' => 0,
                'promedio_ponderado' => 0
            );
        }
        
        $totales['total_materias']++;
        
        // CORRECCIÓN: Verificar correctamente el tipo de denominación
        if ($denominacion['tipo'] == 'puntaje') {
            $totales['total_materias_puntaje']++;
            $credito = intval($denominacion['credito']);
            
            if (empty($denominacion['nota_definitiva'])) {
                $totales['pendientes_puntaje']++;
                $totales['sin_nota_puntaje']++;
                $promedios_por_periodo[$periodo]['sin_nota_puntaje']++;
            } else {
                $nota = floatval($denominacion['nota_definitiva']);
                if ($nota >= 10) {
                    $totales['suma_notas_simples'] += $nota;
                    $totales['suma_notas_ponderadas'] += ($nota * $credito);
                    $totales['suma_creditos_aprobados'] += $credito;
                    $totales['aprobadas_puntaje']++;
                    
                    // Acumular para promedios por periodo
                    $promedios_por_periodo[$periodo]['suma_notas_simples'] += $nota;
                    $promedios_por_periodo[$periodo]['suma_notas_ponderadas'] += ($nota * $credito);
                    $promedios_por_periodo[$periodo]['suma_creditos_aprobados'] += $credito;
                    $promedios_por_periodo[$periodo]['aprobadas_puntaje']++;
                } else {
                    $totales['reprobadas_puntaje']++;
                    $promedios_por_periodo[$periodo]['reprobadas_puntaje']++;
                }
            }
        } else { 
            // CORRECCIÓN: Para tipo 'Aprobado_Reprobado' (cualitativas)
            $totales['total_materias_cualitativas']++;
            
            if (empty($denominacion['nota_definitiva'])) {
                $totales['pendientes_cualitativas']++;
                $totales['sin_nota_cualitativas']++;
            } else {
                // CORRECCIÓN: Verificar correctamente si está aprobada o reprobada
                $nota_cualitativa = trim($denominacion['nota_definitiva']);
                if (strtoupper($nota_cualitativa) == 'APROBADO') {
                    $totales['aprobadas_cualitativas']++;
                } else {
                    $totales['reprobadas_cualitativas']++;
                }
            }
        }
    }
    
    // Calcular promedios generales (solo para materias de puntaje)
    if ($totales['aprobadas_puntaje'] > 0) {
        $totales['promedio_simple'] = $totales['suma_notas_simples'] / $totales['aprobadas_puntaje'];
    }
    if ($totales['suma_creditos_aprobados'] > 0) {
        $totales['promedio_ponderado'] = $totales['suma_notas_ponderadas'] / $totales['suma_creditos_aprobados'];
    }
    
    // Calcular promedios por periodo (solo para materias de puntaje)
    foreach ($promedios_por_periodo as $periodo => $datos) {
        if ($datos['aprobadas_puntaje'] > 0) {
            $promedios_por_periodo[$periodo]['promedio_simple'] = $datos['suma_notas_simples'] / $datos['aprobadas_puntaje'];
            if ($datos['suma_creditos_aprobados'] > 0) {
                $promedios_por_periodo[$periodo]['promedio_ponderado'] = $datos['suma_notas_ponderadas'] / $datos['suma_creditos_aprobados'];
            }
        }
    }
}

// Obtener escolaridades disponibles para el modal
$sql_escolaridades = "SELECT id, nombre FROM escolaridades ORDER BY nombre";
$result_escolaridades = $conn->query($sql_escolaridades);
$escolaridades = $result_escolaridades->fetch_all(MYSQLI_ASSOC);

// Obtener años lectivos disponibles para el modal
$sql_todos_anios = "SELECT id, nombre FROM anios_lectivos ORDER BY nombre DESC";
$result_todos_anios = $conn->query($sql_todos_anios);
$todos_anios = $result_todos_anios->fetch_all(MYSQLI_ASSOC);

// Procesar actualización de notas
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    // Validar token CSRF
    if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        $_SESSION['error'] = "Token de seguridad inválido.";
        header("Location: cargar_notas_estudiante.php?estudiante_id=$estudiante_id&postgrado_id=$postgrado_seleccionado");
        exit();
    }

    $conn->begin_transaction();
    try {
        // Procesar actualización de año lectivo y escolaridad (CORREGIDO)
        if (isset($_POST['actualizar_configuracion'])) {
            $inscripcion_id = intval($_POST['inscripcion_id']);
            $nuevo_anio_lectivo_id = intval($_POST['nuevo_anio_lectivo']);
            $nueva_escolaridad_id = intval($_POST['nueva_escolaridad']);
            $fecha_actual = date('Y-m-d H:i:s');
            
            // Verificar que la inscripción pertenezca al estudiante
            $sql_verificar_inscripcion = "SELECT estudiante_id FROM inscripciones WHERE id = ? AND estudiante_id = ?";
            $stmt_verificar = $conn->prepare($sql_verificar_inscripcion);
            $stmt_verificar->bind_param("ii", $inscripcion_id, $estudiante_id);
            $stmt_verificar->execute();
            
            if ($stmt_verificar->get_result()->num_rows === 1) {
                // CORRECCIÓN: Actualizar solo la inscripción específica usando WHERE id = ?
                $sql_update_config = "UPDATE inscripciones SET 
                                    año_lectivo_id = ?, escolaridad_id = ?,
                                    fecha_modificacion = ?, usuario_modificacion = ?
                                    WHERE id = ?";
                $stmt_update = $conn->prepare($sql_update_config);
                $stmt_update->bind_param("iisii", $nuevo_anio_lectivo_id, $nueva_escolaridad_id, $fecha_actual, $_SESSION['user_id'], $inscripcion_id);
                $stmt_update->execute();
                
                $_SESSION['success'] = "Configuración actualizada correctamente.";
            } else {
                throw new Exception("La inscripción no pertenece al estudiante.");
            }
        }
        // Procesar actualización de notas
        elseif (isset($_POST['guardar_notas'])) {
            $periodo_activo = isset($_POST['periodo_activo']) ? $_POST['periodo_activo'] : null;
            $modificaciones_realizadas = false;
            
            foreach ($_POST['denominaciones'] as $denominacion_id => $datos) {
                $denominacion_id = intval($denominacion_id);
                $tipo_evaluacion = 'Final';
                
                // Verificar que la denominación pertenezca al estudiante y postgrado
                $sql_verificar = "SELECT i.id as inscripcion_id, i.nota_definitiva, d.periodo, d.tipo
                                 FROM inscripciones i
                                 JOIN denominaciones d ON i.denominacion_id = d.id
                                 WHERE i.estudiante_id = ? AND i.denominacion_id = ?";
                $stmt_verificar = $conn->prepare($sql_verificar);
                $stmt_verificar->bind_param("ii", $estudiante_id, $denominacion_id);
                $stmt_verificar->execute();
                $result_verificar = $stmt_verificar->get_result();
                
                if ($result_verificar->num_rows === 0) {
                    continue;
                }
                
                $verificacion = $result_verificar->fetch_assoc();
                $periodo_actual = $verificacion['periodo'];
                $tipo = $verificacion['tipo'];
                $nota_actual = $verificacion['nota_definitiva'];
                $inscripcion_id = $verificacion['inscripcion_id'];
                
                // Solo procesar las notas del periodo activo
                if ($periodo_activo != $periodo_actual) {
                    continue;
                }
                
                $fecha_actual = date('Y-m-d H:i:s');
                
                if ($tipo == 'puntaje') {
                    // MODIFICACIÓN: Solo aceptar números enteros
                    $nota_nueva = isset($datos['nota']) && $datos['nota'] !== '' ? intval($datos['nota']) : null;
                    
                    // Determinar si hay cambio
                    $hay_cambio = false;
                    
                    if ($nota_nueva === null && $nota_actual !== null) {
                        // Cambio de tener valor a vacío
                        $hay_cambio = true;
                    } elseif ($nota_nueva !== null && $nota_actual === null) {
                        // Cambio de vacío a tener valor
                        $hay_cambio = true;
                    } elseif ($nota_nueva !== null && $nota_actual !== null) {
                        // Comparar como enteros
                        $nota_actual_int = intval($nota_actual);
                        if ($nota_nueva !== $nota_actual_int) {
                            $hay_cambio = true;
                        }
                    }
                    
                    if ($hay_cambio) {
                        if ($nota_nueva !== null && ($nota_nueva < 0 || $nota_nueva > 20)) {
                            throw new Exception("Las notas deben estar entre 0 y 20.");
                        }
                        
                        $estado_inscripcion = ($nota_nueva !== null && $nota_nueva >= 10) ? 'finalizada' : 'activa';
                        
                        $sql_update = "UPDATE inscripciones SET 
                                        nota_final = ?, nota_definitiva = ?, tipo_evaluacion = ?,
                                        estado = ?, fecha_modificacion = ?, usuario_modificacion = ?
                                       WHERE id = ?";
                        $stmt = $conn->prepare($sql_update);
                        $stmt->bind_param("ddsssii", $nota_nueva, $nota_nueva, $tipo_evaluacion, $estado_inscripcion, $fecha_actual, $_SESSION['user_id'], $inscripcion_id);
                        $stmt->execute();
                        $modificaciones_realizadas = true;
                    }
                } else { 
                    // Para notas cualitativas
                    $nota_nueva_cualitativa = isset($datos['nota']) ? trim($datos['nota']) : '';
                    
                    // Determinar si hay cambio
                    $hay_cambio = false;
                    $nota_actual_trim = $nota_actual ? trim($nota_actual) : '';
                    
                    if ($nota_nueva_cualitativa === '' && $nota_actual_trim !== '') {
                        // Cambio de tener valor a vacío
                        $hay_cambio = true;
                    } elseif ($nota_nueva_cualitativa !== '' && $nota_nueva_cualitativa !== $nota_actual_trim) {
                        // Cambio de valor
                        $hay_cambio = true;
                    }
                    
                    if ($hay_cambio) {
                        if (empty($nota_nueva_cualitativa)) {
                            // Si se deja vacío, limpiar la nota
                            $sql_update = "UPDATE inscripciones SET 
                                           nota_final = NULL, nota_definitiva = NULL, tipo_evaluacion = NULL,
                                           estado = 'activa', fecha_modificacion = ?, usuario_modificacion = ?
                                           WHERE id = ?";
                            $stmt = $conn->prepare($sql_update);
                            $stmt->bind_param("sii", $fecha_actual, $_SESSION['user_id'], $inscripcion_id);
                        } else {
                            // Validar que sea un valor permitido
                            if (!in_array($nota_nueva_cualitativa, ['Aprobado', 'Reprobado'])) {
                                throw new Exception("Valor no permitido para nota cualitativa: " . $nota_nueva_cualitativa);
                            }
                            
                            // Si se selecciona Aprobado o Reprobado
                            $estado_inscripcion = ($nota_nueva_cualitativa === 'Aprobado') ? 'finalizada' : 'activa';
                            $sql_update = "UPDATE inscripciones SET 
                                           nota_final = ?, nota_definitiva = ?, tipo_evaluacion = ?, estado = ?,
                                           fecha_modificacion = ?, usuario_modificacion = ?
                                           WHERE id = ?";
                            $stmt = $conn->prepare($sql_update);
                            $stmt->bind_param("sssssii", $nota_nueva_cualitativa, $nota_nueva_cualitativa, $tipo_evaluacion, $estado_inscripcion, $fecha_actual, $_SESSION['user_id'], $inscripcion_id);
                        }
                        $stmt->execute();
                        $modificaciones_realizadas = true;
                    }
                }
            }
            
            if ($modificaciones_realizadas) {
                $_SESSION['success'] = "Notas actualizadas correctamente.";
            } else {
                $_SESSION['info'] = "No se realizaron cambios. Las notas se mantienen igual.";
            }
        }
        
        $conn->commit();
        header("Location: cargar_notas_estudiante.php?estudiante_id=$estudiante_id&postgrado_id=$postgrado_seleccionado");
        exit();
    } catch (Exception $e) {
        $conn->rollback();
        $_SESSION['error'] = "Error al actualizar: " . $e->getMessage();
        header("Location: cargar_notas_estudiante.php?estudiante_id=$estudiante_id&postgrado_id=$postgrado_seleccionado");
        exit();
    }
}

// Generar token CSRF si no existe
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

$page_title = "Cargar Notas - " . $estudiante_info['nombres'] . " " . $estudiante_info['apellidos'];
include '../includes/header.php';
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        .estado-activo { color: #28a745; font-weight: bold; }
        .estado-suspendido, .estado-reprobado, .promedio-bajo { color: #dc3545; font-weight: bold; }
        .estado-graduado, .estado-finalizada { color: #17a2b8; font-weight: bold; }
        .estado-retirado { color: #6c757d; font-weight: bold; }
        .estado-aprobado, .promedio-alto { color: #28a745; font-weight: bold; }
        .estado-pendiente { color: #ffc107; font-weight: bold; }
        
        .tab-content {
            display: none;
        }
        .tab-content.show {
            display: block;
        }
        .tab-button {
            margin-right: 5px;
            margin-bottom: 5px;
        }
        .tab-button.active {
            background-color: #0d6efd;
            color: white;
        }
        .nota-input {
            width: 90px;
            text-align: center;
            font-weight: bold;
        }
        .tabs-container {
            gap: 5px;
        }
        .badge {
            font-size: 0.75rem;
        }
        .card {
            box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
            border: 1px solid rgba(0, 0, 0, 0.125);
        }
        .table th {
            background-color: #f8f9fa;
            border-bottom: 2px solid #dee2e6;
        }
        .resumen-card {
            height: 100%;
            transition: transform 0.2s;
        }
        .resumen-card:hover {
            transform: translateY(-2px);
        }
        .sin-nota {
            background-color: #fff3cd !important;
        }
        .anio-lectivo-badge {
            font-size: 0.7rem;
        }
        .resumen-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            margin-bottom: 1rem;
        }
        .resumen-item {
            background: #f8f9fa;
            border-radius: 0.375rem;
            padding: 1rem;
            text-align: center;
            border-left: 4px solid #0d6efd;
        }
        .resumen-valor {
            font-size: 1.5rem;
            font-weight: bold;
            margin-bottom: 0.5rem;
        }
        .resumen-label {
            font-size: 0.875rem;
            color: #6c757d;
        }
        .config-btn {
            padding: 0.25rem 0.5rem;
            font-size: 0.75rem;
        }
        .table-responsive {
            margin-bottom: 1rem;
        }
        .select-cualitativo {
            width: 120px;
        }
        .form-control:focus {
            border-color: #86b7fe;
            box-shadow: 0 0 0 0.25rem rgba(13, 110, 253, 0.25);
        }
        .is-invalid {
            border-color: #dc3545;
            box-shadow: 0 0 0 0.25rem rgba(220, 53, 69, 0.25);
        }
        .changed {
            background-color: #e7f3ff !important;
            border: 2px solid #0d6efd !important;
        }
    </style>
</head>
<body>
    <div class="container mt-4">
        <div class="mb-4">
            <a href="cargar_notas.php?postgrado_id=<?php echo $postgrado_seleccionado; ?>" class="btn btn-secondary">
                <i class="bi bi-arrow-left"></i> Volver al Listado
            </a>
        </div>

        <?php if (isset($_SESSION['success'])): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <?= $_SESSION['success'] ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
            <?php unset($_SESSION['success']); ?>
        <?php endif; ?>
        
        <?php if (isset($_SESSION['info'])): ?>
            <div class="alert alert-info alert-dismissible fade show" role="alert">
                <?= $_SESSION['info'] ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
            <?php unset($_SESSION['info']); ?>
        <?php endif; ?>
        
        <?php if (isset($_SESSION['error'])): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <?= $_SESSION['error'] ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
            <?php unset($_SESSION['error']); ?>
        <?php endif; ?>

        <div class="card border-0 shadow-sm">
            <div class="card-body p-0">
                <div class="p-3 bg-light border-bottom">
                    <h3 class="mb-0">Notas para: <?php echo htmlspecialchars($estudiante_info['nombres'] . ' ' . $estudiante_info['apellidos']); ?></h3>
                    <p class="mb-0">
                        <strong><?php echo htmlspecialchars($tipo_documento_texto); ?>:</strong> <?php echo htmlspecialchars($numero_documento); ?>
                    </p>
                    <p class="mb-0">Postgrado: <strong><?php echo htmlspecialchars($estudiante_info['postgrado_nombre']); ?></strong></p>
                    <p class="mb-0">Estado del Postgrado: 
                        <span class="estado-<?php echo strtolower(htmlspecialchars($estudiante_info['estado_postgrado'])); ?>">
                            <?php echo htmlspecialchars(ucfirst($estudiante_info['estado_postgrado'])); ?>
                        </span>
                    </p>
                </div>

                <?php if (!empty($denominaciones)): ?>
                    <div class="p-3 border-bottom">
                        <h4 class="mb-3">Resumen Académico</h4>
                        
                        <div class="resumen-grid">
                            <div class="resumen-item">
                                <div class="resumen-valor"><?php echo $totales['total_materias']; ?></div>
                                <div class="resumen-label">Total Asignaturas</div>
                            </div>
                            <div class="resumen-item">
                                <div class="resumen-valor text-warning"><?php echo $totales['sin_nota_puntaje'] + $totales['sin_nota_cualitativas']; ?></div>
                                <div class="resumen-label">Sin Nota</div>
                                <small class="text-muted">
                                    P: <?php echo $totales['sin_nota_puntaje']; ?> | 
                                    C: <?php echo $totales['sin_nota_cualitativas']; ?>
                                </small>
                            </div>
                            <div class="resumen-item">
                                <div class="resumen-valor text-success"><?php echo $totales['aprobadas_puntaje'] + $totales['aprobadas_cualitativas']; ?></div>
                                <div class="resumen-label">Aprobadas</div>
                                <small class="text-muted">
                                    P: <?php echo $totales['aprobadas_puntaje']; ?> | 
                                    C: <?php echo $totales['aprobadas_cualitativas']; ?>
                                </small>
                            </div>
                            <div class="resumen-item">
                                <div class="resumen-valor text-danger"><?php echo $totales['reprobadas_puntaje'] + $totales['reprobadas_cualitativas']; ?></div>
                                <div class="resumen-label">Reprobadas</div>
                                <small class="text-muted">
                                    P: <?php echo $totales['reprobadas_puntaje']; ?> | 
                                    C: <?php echo $totales['reprobadas_cualitativas']; ?>
                                </small>
                            </div>
                            <div class="resumen-item">
                                <div class="resumen-valor <?php echo ($totales['promedio_simple'] < 15 && $totales['promedio_simple'] > 0) ? 'text-danger' : 'text-primary'; ?>">
                                    <?php echo number_format($totales['promedio_simple'], 2); ?>
                                </div>
                                <div class="resumen-label">Promedio Simple</div>
                            </div>
                            <div class="resumen-item">
                                <div class="resumen-valor <?php echo ($totales['promedio_ponderado'] < 15 && $totales['promedio_ponderado'] > 0) ? 'text-danger' : 'text-primary'; ?>">
                                    <?php echo number_format($totales['promedio_ponderado'], 2); ?>
                                </div>
                                <div class="resumen-label">Promedio Ponderado</div>
                            </div>
                            <div class="resumen-item">
                                <div class="resumen-valor text-info"><?php echo $totales['suma_creditos_aprobados']; ?></div>
                                <div class="resumen-label">Créditos Aprobados</div>
                            </div>
                        </div>
                    </div>

                    <div class="p-3">
                        <form method="post" id="form-notas">
                            <input type="hidden" name="estudiante_id" value="<?php echo $estudiante_id; ?>">
                            <input type="hidden" name="postgrado_id" value="<?php echo $postgrado_seleccionado; ?>">
                            <input type="hidden" name="guardar_notas" value="1">
                            <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                            <input type="hidden" name="periodo_activo" id="periodo-activo" value="<?php echo isset(array_keys($periodos)[0]) ? array_keys($periodos)[0] : ''; ?>">

                            <div class="d-flex flex-wrap tabs-container mb-3" id="periodos-tabs">
                                <?php 
                                $periodos_keys = array_keys($periodos);
                                $tabs_per_row = 6;
                                $rows = ceil(count($periodos_keys) / $tabs_per_row);
                                
                                for ($row = 0; $row < $rows; $row++): ?>
                                    <div class="d-flex w-100 mb-2">
                                        <?php 
                                        $start = $row * $tabs_per_row;
                                        $end = min(($row + 1) * $tabs_per_row, count($periodos_keys));
                                        
                                        for ($i = $start; $i < $end; $i++): 
                                            $periodo_nombre = $periodos_keys[$i];
                                            $sin_nota_periodo = isset($promedios_por_periodo[$periodo_nombre]['sin_nota_puntaje']) ? $promedios_por_periodo[$periodo_nombre]['sin_nota_puntaje'] : 0;
                                        ?>
                                            <button type="button" class="btn btn-outline-primary tab-button position-relative <?php echo ($i === 0) ? 'active' : ''; ?>" 
                                                    data-target="tab-<?php echo $i; ?>" 
                                                    data-periodo="<?php echo htmlspecialchars($periodo_nombre); ?>">
                                                Periodo <?php echo htmlspecialchars($periodo_nombre); ?>
                                                <?php if ($sin_nota_periodo > 0): ?>
                                                    <span class="position-absolute top-0 start-100 translate-middle badge rounded-pill bg-warning">
                                                        <?php echo $sin_nota_periodo; ?>
                                                    </span>
                                                <?php endif; ?>
                                            </button>
                                        <?php endfor; ?>
                                    </div>
                                <?php endfor; ?>
                            </div>

                            <?php 
                            $periodo_index = 0;
                            foreach ($periodos as $periodo_nombre => $denom_list): 
                                $promedio_simple_periodo = isset($promedios_por_periodo[$periodo_nombre]['promedio_simple']) ? number_format($promedios_por_periodo[$periodo_nombre]['promedio_simple'], 2) : 'N/A';
                                $promedio_ponderado_periodo = isset($promedios_por_periodo[$periodo_nombre]['promedio_ponderado']) ? number_format($promedios_por_periodo[$periodo_nombre]['promedio_ponderado'], 2) : 'N/A';
                                $sin_nota_periodo = isset($promedios_por_periodo[$periodo_nombre]['sin_nota_puntaje']) ? $promedios_por_periodo[$periodo_nombre]['sin_nota_puntaje'] : 0;
                            ?>
                                <div id="tab-<?php echo $periodo_index; ?>" class="tab-content <?php echo ($periodo_index === 0) ? 'show active' : ''; ?>">
                                    <div class="d-flex justify-content-between align-items-center mb-3">
                                        <h5>Periodo: <?php echo htmlspecialchars($periodo_nombre); ?></h5>
                                        <div>
                                            <?php if ($sin_nota_periodo > 0): ?>
                                                <span class="badge bg-warning me-2">Sin nota: <?php echo $sin_nota_periodo; ?></span>
                                            <?php endif; ?>
                                            <span class="badge bg-primary me-2">Prom. Simple: <?php echo $promedio_simple_periodo; ?></span>
                                            <span class="badge bg-primary">Prom. Ponderado: <?php echo $promedio_ponderado_periodo; ?></span>
                                        </div>
                                    </div>
                                    
                                    <div class="table-responsive">
                                        <table class="table table-striped table-hover">
                                            <thead class="table-light">
                                                <tr>
                                                    <th>Código</th>
                                                    <th>Denominación</th>
                                                    <th>Crédito</th>
                                                    <th>Tipo</th>
                                                    <th>Escolaridad</th>
                                                    <th>Año Lectivo</th>
                                                    <th>Nota</th>
                                                    <th>Estado</th>
                                                    <th>Última Modificación</th>
                                                    <th>Configuración</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach ($denom_list as $denominacion): ?>
                                                    <?php 
                                                    $fecha_modificacion = !empty($denominacion['fecha_modificacion']) ? date('d/m/Y H:i', strtotime($denominacion['fecha_modificacion'])) : 'No modificada';
                                                    $modificado_por = !empty($denominacion['modificado_por']) ? $denominacion['modificado_por'] : 'N/A';
                                                    $sin_nota = empty($denominacion['nota_definitiva']);
                                                    $tipo_denominacion = ($denominacion['tipo'] == 'puntaje') ? 'Numérica' : 'Cualitativa';
                                                    ?>
                                                    <tr class="<?php echo $sin_nota ? 'sin-nota' : ''; ?>">
                                                        <td><?php echo htmlspecialchars($denominacion['codigo']); ?></td>
                                                        <td><?php echo htmlspecialchars($denominacion['nombre']); ?></td>
                                                        <td><?php echo htmlspecialchars($denominacion['credito']); ?></td>
                                                        <td>
                                                            <span class="badge <?php echo $denominacion['tipo'] == 'puntaje' ? 'bg-primary' : 'bg-info'; ?>">
                                                                <?php echo $tipo_denominacion; ?>
                                                            </span>
                                                        </td>
                                                        <td>
                                                            <span class="badge bg-secondary"><?php echo htmlspecialchars($denominacion['escolaridad_nombre']); ?></span>
                                                        </td>
                                                        <td>
                                                            <span class="badge bg-dark anio-lectivo-badge">
                                                                <?php echo htmlspecialchars($denominacion['anio_lectivo_nombre']); ?>
                                                            </span>
                                                        </td>
                                                        <td>
                                                            <?php if ($denominacion['tipo'] == 'puntaje'): ?>
                                                                <!-- MODIFICACIÓN: Solo números enteros, sin decimales -->
                                                                <input type="number" step="1" min="0" max="20" 
                                                                       name="denominaciones[<?php echo $denominacion['id']; ?>][nota]" 
                                                                       class="form-control nota-input <?php echo $sin_nota ? 'border-warning' : ''; ?>" 
                                                                       value="<?php echo htmlspecialchars($denominacion['nota_definitiva'] ?? ''); ?>"
                                                                       placeholder="0-20">
                                                            <?php else: ?>
                                                                <select name="denominaciones[<?php echo $denominacion['id']; ?>][nota]" class="form-control select-cualitativo <?php echo $sin_nota ? 'border-warning' : ''; ?>">
                                                                    <option value="">- Seleccionar -</option>
                                                                    <option value="Aprobado" <?php echo ($denominacion['nota_definitiva'] == 'Aprobado') ? 'selected' : ''; ?>>Aprobado</option>
                                                                    <option value="Reprobado" <?php echo ($denominacion['nota_definitiva'] == 'Reprobado') ? 'selected' : ''; ?>>Reprobado</option>
                                                                </select>
                                                            <?php endif; ?>
                                                        </td>
                                                        <td>
                                                            <span class="estado-<?php echo strtolower($denominacion['estado']); ?>">
                                                                <?php echo htmlspecialchars(ucfirst($denominacion['estado'])); ?>
                                                            </span>
                                                        </td>
                                                        <td>
                                                            <small class="text-muted">
                                                                <?php echo $fecha_modificacion; ?><br>
                                                                Por: <?php echo htmlspecialchars($modificado_por); ?>
                                                            </small>
                                                        </td>
                                                        <td>
                                                            <button type="button" class="btn btn-sm btn-outline-secondary config-btn" 
                                                                    data-bs-toggle="modal" data-bs-target="#configModal"
                                                                    data-inscripcion="<?php echo $denominacion['id_inscripcion']; ?>"
                                                                    data-anio-actual="<?php echo $denominacion['año_lectivo_id']; ?>"
                                                                    data-escolaridad-actual="<?php echo $denominacion['escolaridad_id']; ?>"
                                                                    data-denominacion="<?php echo htmlspecialchars($denominacion['nombre']); ?>">
                                                                <i class="bi bi-gear"></i>
                                                            </button>
                                                        </td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                                <?php $periodo_index++; ?>
                            <?php endforeach; ?>

                            <div class="mt-4">
                                <button type="submit" class="btn btn-success btn-lg">
                                    <i class="bi bi-check-circle"></i> Guardar Cambios
                                </button>
                            </div>
                        </form>
                    </div>
                <?php else: ?>
                    <div class="p-4 text-center">
                        <div class="alert alert-info">
                            <h4>No hay asignaturas inscritas</h4>
                            <p>El estudiante no tiene asignaturas inscritas en este postgrado.</p>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Modal para Configuración -->
    <div class="modal fade" id="configModal" tabindex="-1" aria-labelledby="configModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="configModalLabel">Configurar Denominación</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form method="post" id="form-config">
                    <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                    <input type="hidden" name="actualizar_configuracion" value="1">
                    <input type="hidden" name="inscripcion_id" id="inscripcion_id">
                    
                    <div class="modal-body">
                        <div class="mb-3">
                            <label class="form-label"><strong>Denominación:</strong></label>
                            <p id="denominacion-nombre" class="form-control-plaintext"></p>
                        </div>
                        
                        <div class="mb-3">
                            <label for="nuevo_anio_lectivo" class="form-label">Año Lectivo:</label>
                            <select class="form-select" id="nuevo_anio_lectivo" name="nuevo_anio_lectivo" required>
                                <option value="">Seleccionar año lectivo</option>
                                <?php foreach ($todos_anios as $anio): ?>
                                    <option value="<?php echo $anio['id']; ?>"><?php echo htmlspecialchars($anio['nombre']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="mb-3">
                            <label for="nueva_escolaridad" class="form-label">Escolaridad:</label>
                            <select class="form-select" id="nueva_escolaridad" name="nueva_escolaridad" required>
                                <option value="">Seleccionar escolaridad</option>
                                <?php foreach ($escolaridades as $escolaridad): ?>
                                    <option value="<?php echo $escolaridad['id']; ?>"><?php echo htmlspecialchars($escolaridad['nombre']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" class="btn btn-primary">Guardar Cambios</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Manejo de pestañas
            const tabButtons = document.querySelectorAll('.tab-button');
            const tabContents = document.querySelectorAll('.tab-content');
            const periodoActivoInput = document.getElementById('periodo-activo');

            tabButtons.forEach(button => {
                button.addEventListener('click', function() {
                    const target = this.getAttribute('data-target');
                    const periodo = this.getAttribute('data-periodo');
                    
                    // Actualizar pestaña activa
                    tabButtons.forEach(btn => btn.classList.remove('active'));
                    this.classList.add('active');
                    
                    // Mostrar contenido correspondiente
                    tabContents.forEach(content => {
                        content.classList.remove('show', 'active');
                    });
                    document.getElementById(target).classList.add('show', 'active');
                    
                    // Actualizar periodo activo en el formulario
                    periodoActivoInput.value = periodo;
                });
            });

            // Configuración del modal
            const configModal = document.getElementById('configModal');
            configModal.addEventListener('show.bs.modal', function(event) {
                const button = event.relatedTarget;
                const inscripcionId = button.getAttribute('data-inscripcion');
                const anioActual = button.getAttribute('data-anio-actual');
                const escolaridadActual = button.getAttribute('data-escolaridad-actual');
                const denominacionNombre = button.getAttribute('data-denominacion');
                
                document.getElementById('inscripcion_id').value = inscripcionId;
                document.getElementById('denominacion-nombre').textContent = denominacionNombre;
                document.getElementById('nuevo_anio_lectivo').value = anioActual;
                document.getElementById('nueva_escolaridad').value = escolaridadActual;
            });

            // Validación de notas (0-20) solo enteros
            document.querySelectorAll('.nota-input').forEach(input => {
                input.addEventListener('input', function() {
                    let value = parseInt(this.value);
                    if (isNaN(value)) {
                        this.value = '';
                    } else if (value < 0) {
                        this.value = 0;
                    } else if (value > 20) {
                        this.value = 20;
                    } else {
                        this.value = value; // Asegurar que sea entero
                    }
                });
                
                input.addEventListener('blur', function() {
                    if (this.value === '') return;
                    let value = parseInt(this.value);
                    if (isNaN(value) || value < 0 || value > 20) {
                        this.classList.add('is-invalid');
                    } else {
                        this.classList.remove('is-invalid');
                    }
                });
            });

            // Navegación con teclado para inputs de notas
            document.querySelectorAll('.nota-input, .select-cualitativo').forEach((input, index, inputs) => {
                input.addEventListener('keydown', function(e) {
                    const currentIndex = Array.from(inputs).indexOf(this);
                    
                    // Enter o Flecha Abajo: ir al siguiente campo
                    if (e.key === 'Enter' || e.key === 'ArrowDown') {
                        e.preventDefault();
                        if (currentIndex < inputs.length - 1) {
                            inputs[currentIndex + 1].focus();
                            if (inputs[currentIndex + 1].tagName === 'SELECT') {
                                inputs[currentIndex + 1].click(); // Abrir dropdown si es select
                            }
                        }
                    }
                    
                    // Flecha Arriba: ir al campo anterior
                    if (e.key === 'ArrowUp') {
                        e.preventDefault();
                        if (currentIndex > 0) {
                            inputs[currentIndex - 1].focus();
                            if (inputs[currentIndex - 1].tagName === 'SELECT') {
                                inputs[currentIndex - 1].click(); // Abrir dropdown si es select
                            }
                        }
                    }
                });
            });

            // Validación del formulario al enviar
            document.getElementById('form-notas').addEventListener('submit', function(e) {
                let hasErrors = false;
                const currentTab = document.querySelector('.tab-content.show.active');
                const inputs = currentTab.querySelectorAll('.nota-input');
                
                inputs.forEach(input => {
                    if (input.value) {
                        const value = parseInt(input.value);
                        if (isNaN(value) || value < 0 || value > 20) {
                            input.classList.add('is-invalid');
                            hasErrors = true;
                        } else {
                            input.classList.remove('is-invalid');
                        }
                    }
                });
                
                if (hasErrors) {
                    e.preventDefault();
                    alert('Por favor, corrija las notas que están fuera del rango permitido (0-20).');
                }
            });
        });
    </script>
</body>
</html>

<?php
$conn->close();
?>