<?php
session_start();

// Configurar encoding UTF-8
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');

if (!isset($_SESSION['username']) || $_SESSION['rol'] != 'administrador') {
    header("Location: ../login.php");
    exit();
}

include '../includes/db.php';

// Configurar conexión para usar UTF-8
$conn->set_charset("utf8mb4");
$conn->query("SET NAMES 'utf8mb4'");
$conn->query("SET CHARACTER SET utf8mb4");
$conn->query("SET COLLATION_CONNECTION = 'utf8mb4_unicode_ci'");

// Obtener parámetros de búsqueda
$search = isset($_GET['search']) ? trim($_GET['search']) : '';
$promocion_id = isset($_GET['promocion_id']) ? intval($_GET['promocion_id']) : 0;
$postgrado_id = isset($_GET['postgrado_id']) ? intval($_GET['postgrado_id']) : 0;

// Configuración de paginación
$registros_por_pagina = 10;
$pagina_actual = isset($_GET['pagina']) ? (int)$_GET['pagina'] : 1;
$offset = ($pagina_actual - 1) * $registros_por_pagina;

// Obtener listas para filtros
$sql_promociones = "SELECT id, nombre FROM promociones ORDER BY nombre";
$promociones = $conn->query($sql_promociones);

$sql_postgrados = "SELECT id, nombre FROM postgrados ORDER BY nombre";
$postgrados = $conn->query($sql_postgrados);

// Consulta para obtener estudiantes (SOLO GRADUADOS)
$sql_estudiantes = "SELECT 
                    e.id, e.nombres, e.apellidos, e.tipo_documento,
                    e.cedula_venezolana, e.cedula_extranjera, e.pasaporte
                   FROM estudiantes e
                   JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                   WHERE ep.estado = 'graduado'
                   AND (e.cedula_venezolana LIKE ? OR e.cedula_extranjera LIKE ? OR e.pasaporte LIKE ? OR e.nombres LIKE ? OR e.apellidos LIKE ? OR e.correo LIKE ?)";

$params = ["%$search%", "%$search%", "%$search%", "%$search%", "%$search%", "%$search%"];
$types = "ssssss";

if ($promocion_id > 0) {
    $sql_estudiantes .= " AND ep.promocion_id = ?";
    $params[] = $promocion_id;
    $types .= "i";
}

if ($postgrado_id > 0) {
    $sql_estudiantes .= " AND ep.postgrado_id = ?";
    $params[] = $postgrado_id;
    $types .= "i";
}

// Consulta para contar total de registros (para paginación) - SOLO GRADUADOS
$sql_count = "SELECT COUNT(DISTINCT e.id) as total FROM estudiantes e
              JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
              WHERE ep.estado = 'graduado'
              AND (e.cedula_venezolana LIKE ? OR e.cedula_extranjera LIKE ? OR e.pasaporte LIKE ? OR e.nombres LIKE ? OR e.apellidos LIKE ? OR e.correo LIKE ?)";

$params_count = ["%$search%", "%$search%", "%$search%", "%$search%", "%$search%", "%$search%"];
$types_count = "ssssss";

if ($promocion_id > 0) {
    $sql_count .= " AND ep.promocion_id = ?";
    $params_count[] = $promocion_id;
    $types_count .= "i";
}

if ($postgrado_id > 0) {
    $sql_count .= " AND ep.postgrado_id = ?";
    $params_count[] = $postgrado_id;
    $types_count .= "i";
}

$stmt_count = $conn->prepare($sql_count);
if ($stmt_count) {
    $stmt_count->bind_param($types_count, ...$params_count);
    $stmt_count->execute();
    $result_count = $stmt_count->get_result();
    $count_data = $result_count->fetch_assoc();
    $total_registros = $count_data ? $count_data['total'] : 0;
    $stmt_count->close();
} else {
    $total_registros = 0;
}

$total_paginas = ceil($total_registros / $registros_por_pagina);

// Consulta para obtener datos con paginación
$sql_estudiantes .= " GROUP BY e.id ORDER BY e.apellidos LIMIT ?, ?";
$params[] = $offset;
$params[] = $registros_por_pagina;
$types .= "ii";

$stmt_estudiantes = $conn->prepare($sql_estudiantes);
if (!$stmt_estudiantes) {
    die("Error en la preparación de la consulta: " . $conn->error);
}

$stmt_estudiantes->bind_param($types, ...$params);
$stmt_estudiantes->execute();
$estudiantes = $stmt_estudiantes->get_result()->fetch_all(MYSQLI_ASSOC);

// Para cada estudiante, obtener sus postgrados (SOLO GRADUADOS)
foreach ($estudiantes as &$estudiante) {
    $sql_postgrados = "SELECT 
                        p.id AS postgrado_id, 
                        p.nombre AS postgrado_nombre,
                        pr.nombre AS promocion,
                        ep.estado AS estado_postgrado,
                        ep.fecha_graduacion,
                        ep.promocion_id
                       FROM estudiante_postgrado ep
                       JOIN postgrados p ON ep.postgrado_id = p.id
                       LEFT JOIN promociones pr ON ep.promocion_id = pr.id
                       WHERE ep.estudiante_id = ?
                       AND ep.estado = 'graduado'
                       ORDER BY ep.fecha_inscripcion DESC";
    
    $stmt_postgrados = $conn->prepare($sql_postgrados);
    $stmt_postgrados->bind_param("i", $estudiante['id']);
    $stmt_postgrados->execute();
    $estudiante['postgrados'] = $stmt_postgrados->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt_postgrados->close();
}
unset($estudiante);

$page_title = "Buscar Estudiante Graduado";
include '../includes/header.php';
?>

<div class="container">
    <!-- Botón para volver al panel -->
    <div class="mb-4">
        <a href="index.php" class="btn btn-secondary">
            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-arrow-left" viewBox="0 0 16 16">
                <path fill-rule="evenodd" d="M15 8a.5.5 0 0 0-.5-.5H2.707l3.147-3.146a.5.5 0 1 0-.708-.708l-4 4a.5.5 0 0 0 0 .708l4 4a.5.5 0 0 0 .708-.708L2.707 8.5H14.5A.5.5 0 0 0 15 8z"/>
            </svg>
            Volver al Panel
        </a>
    </div>

    <div class="card">
        <div class="card-header bg-primary text-white">
            <h3 class="mb-0">Filtros de Búsqueda - Estudiantes Graduados</h3>
        </div>
        <div class="card-body">
            <form method="get" class="row g-3 search-form">
                <input type="hidden" name="pagina" value="1">
                <div class="col-md-3">
                    <label for="search" class="form-label">Buscar Estudiante:</label>
                    <input type="text" class="form-control" name="search" id="search" 
                           placeholder="Cédula, Pasaporte, Nombre, Apellido o Correo" 
                           value="<?= htmlspecialchars($search) ?>">
                </div>
                
                <div class="col-md-3">
                    <label for="promocion_id" class="form-label">Promoción:</label>
                    <select class="form-select" id="promocion_id" name="promocion_id">
                        <option value="0">Todas las promociones</option>
                        <?php while ($promocion = $promociones->fetch_assoc()): ?>
                            <option value="<?= $promocion['id'] ?>" 
                                <?= $promocion_id == $promocion['id'] ? 'selected' : '' ?>>
                                <?= htmlspecialchars($promocion['nombre']) ?>
                            </option>
                        <?php endwhile; ?>
                    </select>
                </div>
                
                <div class="col-md-3">
                    <label for="postgrado_id" class="form-label">Postgrado:</label>
                    <select class="form-select" id="postgrado_id" name="postgrado_id">
                        <option value="0">Todos los postgrados</option>
                        <?php while ($postgrado = $postgrados->fetch_assoc()): ?>
                            <option value="<?= $postgrado['id'] ?>" 
                                <?= $postgrado_id == $postgrado['id'] ? 'selected' : '' ?>>
                                <?= htmlspecialchars($postgrado['nombre']) ?>
                            </option>
                        <?php endwhile; ?>
                    </select>
                </div>
                
                <div class="col-md-3 d-flex align-items-end">
                    <button type="submit" class="btn btn-primary w-100">Buscar</button>
                </div>
            </form>
        </div>
    </div>

    <?php if ($promocion_id > 0): ?>
        <div class="card mt-4">
            <div class="card-header bg-success text-white">
                <h3 class="mb-0">Generar Constancias Masivas (Notas)</h3>
            </div>
            <div class="card-body">
                <form id="formConstanciasMasivasNotas" onsubmit="generarConstanciasMasivasNotas(event)">
                    <input type="hidden" name="promocion_id" value="<?= $promocion_id ?>">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label for="postgrado_id_masivo_notas" class="form-label">Postgrado (opcional):</label>
                            <select class="form-select" id="postgrado_id_masivo_notas" name="postgrado_id">
                                <option value="0">Todos los postgrados de esta promoción</option>
                                <?php 
                                $postgrados_promocion = [];
                                foreach ($estudiantes as $est) {
                                    foreach ($est['postgrados'] as $postgrado) {
                                        if ($postgrado['promocion_id'] == $promocion_id) {
                                            $postgrados_promocion[$postgrado['postgrado_id']] = $postgrado['postgrado_nombre'];
                                        }
                                    }
                                }
                                foreach ($postgrados_promocion as $id => $nombre): ?>
                                    <option value="<?= $id ?>"><?= htmlspecialchars($nombre) ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-6 d-flex align-items-end">
                            <button type="submit" class="btn btn-success w-100">
                                <i class="bi bi-file-earmark-pdf"></i> Generar Constancias de Notas
                            </button>
                        </div>
                    </div>
                </form>
                <div id="mensajeConstanciasNotas" class="mt-3"></div>
            </div>
        </div>

        <script>
        function generarConstanciasMasivasNotas(event) {
            event.preventDefault();
            
            const form = event.target;
            const formData = new FormData(form);
            const mensajeDiv = document.getElementById('mensajeConstanciasNotas');
            
            mensajeDiv.innerHTML = `
                <div class="alert alert-info d-flex align-items-center">
                    <div class="spinner-border spinner-border-sm me-2" role="status"></div>
                    <span>Generando constancias, por favor espere...</span>
                </div>
            `;
            
            const submitBtn = form.querySelector('button[type="submit"]');
            submitBtn.disabled = true;
            
            fetch('generar_constancias_masivas_notas.php', {
                method: 'POST',
                body: formData
            })
            .then(response => {
                if (!response.ok) {
                    return response.text().then(text => {
                        throw new Error(text || 'Error en la respuesta del servidor');
                    });
                }
                return response.blob();
            })
            .then(blob => {
                const url = window.URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.href = url;
                a.download = `constancias_notas_promocion_${formData.get('promocion_id')}_${new Date().toISOString().slice(0,10)}.pdf`;
                document.body.appendChild(a);
                a.click();
                window.URL.revokeObjectURL(url);
                a.remove();
                
                mensajeDiv.innerHTML = '<div class="alert alert-success">Constancias generadas y descargadas correctamente.</div>';
            })
            .catch(error => {
                console.error('Error:', error);
                mensajeDiv.innerHTML = `
                    <div class="alert alert-danger">
                        <strong>Error al generar constancias:</strong><br>
                        ${error.message}
                    </div>
                `;
            })
            .finally(() => {
                submitBtn.disabled = false;
            });
        }
        </script>
    <?php endif; ?>

    <!-- Lista de estudiantes -->
    <?php if (!empty($estudiantes)): ?>
        <div class="card mt-4">
            <div class="card-header bg-info text-white">
                <h3 class="mb-0">Resultados de la Búsqueda - Estudiantes Graduados</h3>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-striped table-hover">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Nombre</th>
                                <th>Apellidos</th>
                                <th>Tipo Documento</th>
                                <th>Documento</th>
                                <th>Postgrados</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php $contador = ($pagina_actual - 1) * $registros_por_pagina + 1; ?>
                            <?php foreach ($estudiantes as $estudiante): 
                                $tipo_doc = '';
                                $documento = '';
                                switch ($estudiante['tipo_documento']) {
                                    case 'V':
                                        $tipo_doc = 'Cédula Venezolana';
                                        $documento = $estudiante['cedula_venezolana'];
                                        break;
                                    case 'E':
                                        $tipo_doc = 'Cédula Extranjera';
                                        $documento = $estudiante['cedula_extranjera'];
                                        break;
                                    case 'P':
                                        $tipo_doc = 'Pasaporte';
                                        $documento = $estudiante['pasaporte'];
                                        break;
                                    default:
                                        $tipo_doc = 'Cédula Venezolana';
                                        $documento = $estudiante['cedula_venezolana'];
                                        break;
                                }
                            ?>
                                <tr>
                                    <td><?= $contador++ ?></td>
                                    <td><?= htmlspecialchars($estudiante['nombres']) ?></td>
                                    <td><?= htmlspecialchars($estudiante['apellidos']) ?></td>
                                    <td><?= htmlspecialchars($tipo_doc) ?></td>
                                    <td><?= htmlspecialchars($documento) ?></td>
                                    <td>
                                        <?php if (!empty($estudiante['postgrados'])): ?>
                                            <?php foreach ($estudiante['postgrados'] as $postgrado): ?>
                                                <div class="postgrado-item">
                                                    <strong><?= htmlspecialchars($postgrado['postgrado_nombre']) ?></strong><br>
                                                    <?php if ($postgrado['promocion']): ?>
                                                        Promoción: <?= htmlspecialchars($postgrado['promocion']) ?><br>
                                                    <?php endif; ?>
                                                    Estado: <span class="badge bg-success"><?= htmlspecialchars($postgrado['estado_postgrado']) ?></span>
                                                    <?php if ($postgrado['estado_postgrado'] == 'graduado' && !empty($postgrado['fecha_graduacion'])): ?>
                                                        <br>Graduación: <?= htmlspecialchars($postgrado['fecha_graduacion']) ?>
                                                    <?php endif; ?>
                                                </div>
                                            <?php endforeach; ?>
                                        <?php else: ?>
                                            <span class="no-postgrado">No asignado</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php if (!empty($estudiante['postgrados'])): ?>
                                            <div class="d-flex flex-column">
                                                <?php foreach ($estudiante['postgrados'] as $postgrado): ?>
                                                    <a href="constancia_notas.php?estudiante_id=<?= $estudiante['id'] ?>&postgrado_id=<?= $postgrado['postgrado_id'] ?>&fecha_constancia=<?= date('Y-m-d') ?>" 
                                                       class="btn btn-sm btn-primary mb-1" 
                                                       target="_blank"
                                                       title="Generar constancia para <?= htmlspecialchars($postgrado['postgrado_nombre']) ?>">
                                                        Constancia
                                                    </a>
                                                <?php endforeach; ?>
                                            </div>
                                        <?php else: ?>
                                            <span class="no-postgrado">No disponible</span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Paginación -->
                <?php if ($total_paginas > 1): ?>
                    <nav aria-label="Page navigation">
                        <ul class="pagination justify-content-center">
                            <?php if ($pagina_actual > 1): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?<?php 
                                        echo http_build_query(array_merge($_GET, ['pagina' => $pagina_actual - 1]));
                                    ?>" aria-label="Previous">
                                        <span aria-hidden="true">&laquo;</span>
                                    </a>
                                </li>
                            <?php endif; ?>

                            <?php for ($i = 1; $i <= $total_paginas; $i++): ?>
                                <li class="page-item <?= $i == $pagina_actual ? 'active' : '' ?>">
                                    <a class="page-link" href="?<?php 
                                        echo http_build_query(array_merge($_GET, ['pagina' => $i]));
                                    ?>"><?= $i ?></a>
                                </li>
                            <?php endfor; ?>

                            <?php if ($pagina_actual < $total_paginas): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?<?php 
                                        echo http_build_query(array_merge($_GET, ['pagina' => $pagina_actual + 1]));
                                    ?>" aria-label="Next">
                                        <span aria-hidden="true">&raquo;</span>
                                    </a>
                                </li>
                            <?php endif; ?>
                        </ul>
                    </nav>
                <?php endif; ?>
            </div>
        </div>
    <?php elseif (!empty($search) || $promocion_id > 0 || $postgrado_id > 0): ?>
        <div class="alert alert-info mt-4">
            No se encontraron estudiantes graduados con los criterios de búsqueda especificados.
        </div>
    <?php endif; ?>
</div>

<?php
include '../includes/footer.php';
$conn->close();
?>